'''
This module reads xwand and valodatemod excel files
and generates new file with difference
'''

import hashlib
from validatemod.plugin.ExcelWorker import ExcelWorker
from enum import Enum


# Execution results
class exit_result(Enum):
    OK = 0,
    OK_FILES_ARE_EQUAL = 1,
    WRONG_FILE_XWAND = 20,
    WRONG_FILE_VALIDATEMOD = 25,
    WRONG_INPUT_FILES_ARE_EQUAL = 26,
    WRONG_FILES_STRUCTURE_IS_EQUAL = 27,
    CALCULATION_ERROR = 30,
    EXCEL_WRITE_ERROR = 40


# xWand excel file header to compare it to validatemod's header
# Global variable
XWAND_HEADER = []


# Generate hash from string
def get_hash(str_in):
    return hashlib.sha256(str_in.encode('cp1251')).hexdigest()


# Returns cell value or '' if cell value is None
def get_item_from_cell(item):
    return str(item) if item is not None else ''


# Returns False/True for boolean values
# else returns ""
def get_if_bool(str_in):
    if str_in.upper() == "FALSE":
        return False
    elif str_in.upper() == "TRUE":
        return True
    return ""


# Get full key for True/False
def get_key(arr):
    list_key = []
    for item in arr:
        state = get_item_from_cell(item[4])
        if state not in ("(Not Evaluated)", "(Skipped)"):
            key = f'{get_item_from_cell(item[0])}_{get_item_from_cell(item[2])}_{get_item_from_cell(item[6])}_{"" if get_item_from_cell(item[7])=="(fallback)" else get_item_from_cell(item[7])}_{get_item_from_cell(item[8])}_{get_item_from_cell(item[9])}_{get_item_from_cell(item[10])}'
        else:
            # (Not Evaluated), (Skipped)
            key = f'{get_item_from_cell(item[0])}_{get_item_from_cell(item[2])}'

        list_key.append(key)
    list_key.sort()
    
    str_result = "_".join(list_key)
    return str_result


# Read and calc hash common function
def read_excel_common(list_excel):
    dict_formulas = dict()
    num_valid_res = ""
    # List for generate hash: all records instead of fallback
    list_formulas_for_key = []
    # List for append to dictionary: all records
    list_formulas_for_append = []
    for row in list_excel:
        if num_valid_res == "":
            num_valid_res = row[3]
        if row[3] == num_valid_res:
            # Collect formulas and keys for the same counters
            list_formulas_for_append.append(row)

            # clear list_formulas_for_key
            if '(Not Evaluated)' in get_item_from_cell(row[4]):
                # Reset list
                list_formulas_for_key = [row]
            else:
                # Push current row
                list_formulas_for_key.append(row)
        else:
            # Save previous results, it's next counter
            key = get_key(list_formulas_for_key)
            hash_key = get_hash(key)
            dict_formulas[hash_key] = list_formulas_for_append
            num_valid_res = row[3]
            list_formulas_for_key = [row]
            list_formulas_for_append = [row]

    # Save the last data
    key = get_key(list_formulas_for_key)
    hash_key = get_hash(key)
    dict_formulas[hash_key] = list_formulas_for_append

    return dict_formulas


# Reads xWand excel file
def read_xwand(fname):
    global XWAND_HEADER
    excel = ExcelWorker(fname)

    # Exclude table header
    list_excel = excel.get_all_lists(skip_empty_rows=True)
    if len(list_excel) == 0:
        return list_excel

    # ===================== Check file structure =====================
    # Check columns count
    if len(list_excel[0]) < 11:
        raise Exception('Wrong xWand file structure', exit_result.WRONG_FILE_XWAND)
    # ================================================================

    # Read xWand excel file header
    XWAND_HEADER = list_excel[0]
    list_excel = list_excel[1:]

    # Exclude first row, extra rows
    for i in range(0, len(list_excel)):
        list_excel[i] = list_excel[i][1:12]

    dict_formulas = read_excel_common(list_excel)
    del excel
    return dict_formulas


# Reads validatemod excel file
def read_validatemod(fname):
    global XWAND_HEADER
    excel = ExcelWorker(fname)

    # Exclude table header and columns numbers
    list_excel = excel.get_all_lists(skip_empty_rows=True)
    if len(list_excel) == 0:
        return list_excel

    # Compare xWand and validatemod excel files headers
    if XWAND_HEADER == list_excel[0]:
        raise Exception('xWand file structure is the same as validatemod', exit_result.WRONG_FILES_STRUCTURE_IS_EQUAL)


    # Exclude extra rows
    for i in range(0, len(list_excel)):
        list_excel[i] = list_excel[i][0:11]


    # ===================== Check file structure =====================
    # Check sheet header for validatemod
    str_row = [
        'Наименование файла отчетности',
        'Точка входа',
        'Идентификатор контрольного соотношения',
        'Порядковый номер запуска контроля',
        'Статус отработки запуска контрольного соотношения',
        'Сообщение',
        'Имя переменной в формуле',
        'QName показателя',
        'Значение переменной',
        'Идентификатор контекста',
        'Период в контексте'
    ]
    if list_excel[0] != str_row:
        raise Exception('Wrong validatemod file structure', exit_result.WRONG_FILE_VALIDATEMOD)
    # ================================================================

    list_excel = list_excel[1:]
    dict_formulas = read_excel_common(list_excel)

    del excel
    return dict_formulas


# Returns key's values (and key at the and of the list) as list to add to excel
def get_diff_row(keys_xwand, dict_xwand):
    diff = []
    for key in keys_xwand:
        for item in dict_xwand[key]:
            item.append(key)
            # Append hash
            diff.append(item)

    return diff


# Returns list of difference between xwand and validatemod lists to add to excel
def get_validation_results_diff(dict_formulas_xwand, dict_formulas_validatemod):
    list_result = []
    
    for key in dict_formulas_xwand:
        if key in dict_formulas_validatemod:
            for item1, item2 in zip(dict_formulas_xwand[key], dict_formulas_validatemod[key]):
                validation_xwand = get_item_from_cell(item1[4])
                validation_validatemod = get_item_from_cell(item2[4])
                if validation_xwand != validation_validatemod:
                    # Get boolean values of validation results in case of False/True
                    bool_validation_validatemod = get_if_bool(validation_validatemod)
                    bool_validation_xwand = get_if_bool(validation_xwand)
                    # Get message from xwand (if false) or validatemod (else)
                    if bool_validation_xwand == False or bool_validation_validatemod == False:
                        message = get_item_from_cell(item1[5]) if bool_validation_xwand == False else get_item_from_cell(item2[5])
                    else:
                        message = ""
                    record = [item1[0], item1[1], item1[2], item1[3], str(validation_validatemod), str(validation_xwand), message, item1[6], item1[7], item1[8], item1[9], item1[10]]
                    list_result.append(record)
    return list_result


# Main function
#================================================================================
def generate_report_excel_xwand(xlsx_file_xwand, xlsx_file_excel, xlsx_file_out):
    if xlsx_file_xwand == xlsx_file_excel:
        return exit_result.WRONG_INPUT_FILES_ARE_EQUAL.value[0]

    # Read excels
    try:
        # Read xwand excel file
        dict_formulas_xwand = read_xwand(xlsx_file_xwand)
    except Exception as e:
        print(e)
        return exit_result.WRONG_FILE_XWAND.value[0]
    try:
        # Read validatemod excel file
        dict_formulas_validatemod = read_validatemod(xlsx_file_excel)
    except Exception as e:
        print(e)
        if len(e.args) > 0 and e.args[1] == exit_result.WRONG_FILES_STRUCTURE_IS_EQUAL:
            return exit_result.WRONG_FILES_STRUCTURE_IS_EQUAL.value[0]
        else:
            return exit_result.WRONG_FILE_VALIDATEMOD.value[0]

    try:
        # Look for absent keys
        keys_xwand = [x for x in dict_formulas_xwand if x not in dict_formulas_validatemod]
        keys_validatemod = [x for x in dict_formulas_validatemod if x not in dict_formulas_xwand]

        # Get difference in validation results
        diff_xwand = get_diff_row(keys_xwand, dict_formulas_xwand)
        diff_validatemod = get_diff_row(keys_validatemod, dict_formulas_validatemod)
        diff_validation_results = get_validation_results_diff(dict_formulas_xwand, dict_formulas_validatemod)
    except Exception as e:
        print(e)
        return exit_result.CALCULATION_ERROR.value[0]

    if len(diff_xwand) == 0 and len(diff_validatemod) == 0 and len(diff_validation_results) == 0:
        return exit_result.OK_FILES_ARE_EQUAL.value[0]

    try:
        # Generate ex excel file
        excel_writer = ExcelWorker(xlsx_file_out, "xWand")
        # Report # 1, hash in Validatemod but not in xWand
        excel_writer.set_table_header([
                        'Наименование файла отчетности',
                        'Точка входа',
                        'Идентификатор контрольного соотношения',
                        'Порядковый номер запуска контроля',
                        'Статус отработки запуска контрольного соотношения xWand',
                        'Сообщение',
                        'Имя переменной в формуле',
                        'QName показателя',
                        'Значение переменной',
                        'Идентификатор контекста',
                        'Период в контексте',
                        'Идентификационный ключ срабатывания КС'],
            show_columns_numbers=False)

        for item in diff_xwand:
            excel_writer.append_row([get_item_from_cell(x) for x in item])


        # Report # 2, hash in xWand  but not in Validatemod
        excel_writer.append_sheet("Anketa")
        excel_writer.set_table_header([
                        'Наименование файла отчетности',
                        'Точка входа',
                        'Идентификатор контрольного соотношения',
                        'Порядковый номер запуска контроля',
                        'Статус отработки запуска контрольного соотношения ПО "Анкета-редактор"',
                        'Сообщение',
                        'Имя переменной в формуле',
                        'QName показателя',
                        'Значение переменной',
                        'Идентификатор контекста',
                        'Период в контексте',
                        'Идентификационный ключ срабатывания КС'],
            show_columns_numbers=False)
        for item in diff_validatemod:
            excel_writer.append_row([get_item_from_cell(x) for x in item])

        # Report # 3, compare validation results (statuses)
        excel_writer.append_sheet("Compare")
        excel_writer.set_table_header([
            'Наименование файла отчетности',
            'Точка входа',
            'Идентификатор контрольного соотношения',
            'Порядковый номер запуска контроля',
            'Статус отработки запуска контрольного соотношения ПО "Анкета-редактор"',
            'Статус отработки запуска контрольного соотношения xWand',
            'Сообщение',
            'Имя переменной в формуле',
            'QName показателя',
            'Значение переменной',
            'Идентификатор контекста',
            'Период в контексте'],
            show_columns_numbers=False)
        for item in diff_validation_results:
            excel_writer.append_row([get_item_from_cell(x) for x in item])

        excel_writer.save_excel()
    except Exception as e:
        print(e)
        return exit_result.EXCEL_WRITE_ERROR.value[0]
    # Return OK
    return exit_result.OK.value[0]
#================================================================================