import openpyxl
from openpyxl.styles import Alignment, Font, Color, PatternFill, Border, Side
from openpyxl.utils import get_column_letter
import os


class ExcelWorker:
    # Columns indexes list, text in which should be wrapped
    __wrap_text_columns = None
    __book = None
    __int_columns_type_list = []
    __float_columns_type_list = []

    # Max with of each column
    __max_column_width = 100
    __fl_show_columns_numbers = False
    __sheet_row_index = 0

    def __init__(self, excel_file, default_sheet_name="Лист", max_row_count = 1000000):
        self.__excel_file = excel_file
        # Default sheet base name: sheet, sheet1, sheet2, sheet3...
        self.__data_sheet_name = default_sheet_name
        # Max row count per list
        self.__max_row_per_list = max_row_count

        self.__fl_file_exists = os.path.isfile(self.__excel_file)

        if self.__fl_file_exists:
            # Append to exists file
            self.__book = openpyxl.load_workbook(self.__excel_file)
        else:
            # Create new file
            self.__book = openpyxl.Workbook()
            self.__book.active.title = self.__data_sheet_name
        self.__sheet = self.__book.active
        self.__sheet_row_index = self.__sheet.max_row
        #self.__max_column = self.__sheet.max_column

    # Append table header to excel if it is a new document
    def set_table_header(self, record, show_columns_numbers=True):
        self.__fl_show_columns_numbers = show_columns_numbers
        if self.__sheet.max_row > 1:
            # Unable to append header if document is not empty
            return
        # Append header
        self.append_row(record)
        self.__draw_header(0, True)
        # Column numbers
        if self.__fl_show_columns_numbers:
            # Add column numbers
            self.append_row([x for x in range(1, len(record) + 1)])
            self.__draw_header(1, False)

    # Add new sheet to Excel and set it as active
    def append_sheet(self, sheet_name=None):
        worksheet = self.__book.create_sheet()
        if sheet_name is not None:
            worksheet.title = sheet_name
        self.__sheet = worksheet

    # Switch to sheet by index or name
    def set_active_sheet(self, *args):
        if args is None or len(args) == 0 or args[0] is None:
            return

        if isinstance(args[0], int) and args[0] in range(0, len(self.__book.sheetnames)):
            # Set active sheet by index
            self.__sheet = self.__book[self.__book.sheetnames[args[0]]]
        else:
            # Set active sheet by name
            name = str(args[0]).strip()
            for i in range(len(self.__book.sheetnames)):
                if self.__book.sheetnames[i].strip() == name:
                    self.__sheet = self.__book[self.__book.sheetnames[i]]
                    break
            else:
                self.append_sheet(name)

    # Looking for not filled sheet with name started from the __data_sheet_name: sheet1, sheet2, sheet3 etc
    # or append new sheet with name started from the __data_sheet_name
    # New sheet will be active
    def find_or_append_sheet(self):
        # Save current sheet
        sheet = self.__sheet
        for item in [x for x in self.__book.sheetnames if self.__data_sheet_name in x]:
            self.set_active_sheet(item)
            if self.__sheet.max_row >= self.__max_row_per_list:
                continue
            else:
                # We have found not filled sheet
                # Active sheet is item
                self.__sheet_row_index = self.__sheet.max_row + 1
                break
        else:
            # Append new sheet
            self.append_sheet(self.__data_sheet_name)
            self.__sheet_row_index = 1

    # Append current record to excel
    def append_row(self, record):
        def set_variable_type(value):
            try:
                if type(value) is int:
                    return int(value)
            except:
                return str(value)
            return str(value)

        if record is None or len(record) == 0:
            return

        if self.__sheet_row_index >= self.__max_row_per_list:
            self.find_or_append_sheet()

        self.__sheet.append([set_variable_type(x) for x in record])
        self.__sheet_row_index += 1

    def append_list(self, my_list):
        for item in my_list:
            self.append_row(item)

    # Set columns indexes, text in which should be wrapped (long strings)
    def set_wrap_text_columns(self, columns):
        self.__wrap_text_columns = columns

    def set_int_columns_list(self, columns):
        self.__int_columns_type_list = columns

    def set_float_columns_list(self, columns):
        self.__float_columns_type_list = columns

    # Read and return all excel list
    def get_list(self, skip_empty_rows=False):
        my_list = []
        for row in self.__sheet.rows:
            current_row = [x.value if x.value is not None else "" for x in row]
            if skip_empty_rows:
                # Skip row with empty cells
                if len([x for x in current_row if x is not None and str(x).strip() != ""]) == 0:
                    # Skip current row
                    continue
            my_list.append(current_row)
        return my_list

    # Return all sheets with name started from sheet_name
    def get_sheets_list(self, sheet_name=""):
        return [x for x in self.__book.sheetnames] if sheet_name.strip() == "" \
            else [x for x in self.__book.sheetnames if sheet_name in x]

    #
    def get_all_lists(self, sheet_name="", skip_empty_rows = False):
        list = []
        for item in self.get_sheets_list(sheet_name):
            self.set_active_sheet(item)
            list_current = self.get_list(skip_empty_rows=skip_empty_rows)
            for item in list_current:
                list.append(item)
        return list

    # Save changes to excel
    def save_excel(self):
        self.__set_columns_format()
        self.__book.save(self.__excel_file)

    # Private methods
    # Draws table header
    def __draw_header(self, row_index, fl_draw = True):
        for row in self.__sheet.iter_rows(min_row=row_index + 1, max_row=row_index + 1):
            for cell in row:
                cell.alignment = Alignment(horizontal = 'center', vertical = 'center')
                cell.font = Font(bold = True)
                if fl_draw:
                    cell.fill = PatternFill(fgColor=Color('C4C4C4'), fill_type='solid')
                    cell.border = Border(left=Side(style='thin'),
                                         right=Side(style='thin'),
                                         top=Side(style='thin'),
                                         bottom=Side(style='thin')
                                         )
    
    # Set columns width and alignment
    def __set_columns_format(self):
        # Format columns for all sheets in the book
        for i in range(len(self.__book.sheetnames)):
            self.set_active_sheet(i)
            # Start row for vertical alignment format
            start_row = 1
            if self.__fl_show_columns_numbers:
                start_row += 1

            # Set columns width by hand
            for x in range(0, self.__sheet.max_column):
                s = get_column_letter(x + 1)
                max_length = 0
                for y in range(0, self.__sheet.max_row):
                    # Calculate column width
                    cell = self.__sheet.cell(row = y + 1, column = x + 1)
                    try:
                        # It may be exception in case of an empty cell
                        if len(str(cell.value)) > max_length:
                            max_length = len(cell.value)
                    except:
                        pass

                    # Vertical align for all cells exclude header and column numbers
                    if cell.row > start_row:
                        cell.alignment = Alignment(vertical='center')

                    if x + 1 in self.__int_columns_type_list and y + 1 > start_row:
                        # Columns with integer values -> int
                        try:
                            cell.value = int(cell.value)
                        except:
                            cell.value = str(cell.value)
                    elif x + 1 in self.__float_columns_type_list and y + 1 > start_row:
                        # Align columns with float values to right
                        # We should use 'copy' in order tyo store old alignment format (center)
                        cell.alignment = cell.alignment.copy(horizontal='right')

                    # Wrap text for some columns
                    if self.__wrap_text_columns is not None and x + 1 in self.__wrap_text_columns:
                        cell.alignment = cell.alignment.copy(wrapText = True)
                column_width = max_length + 4
                # Max column width limitation
                if column_width > self.__max_column_width:
                    column_width = self.__max_column_width
                self.__sheet.column_dimensions[s].width = column_width
