-- This SQL script initializes a database for the XBRL Abstract Model using Oracle SQL

-- (c) Copyright 2013 Mark V Systems Limited, California US, All rights reserved.  
-- Mark V copyright applies to this software, which is licensed according to the terms of Validatemod(r).

-- note that any "&" must be quoted with double-quotes
set define off;

-- drop triggers
begin
   for t in (select trigger_name from user_triggers)
   loop
       execute immediate 'drop trigger "' || t.trigger_name || '"';
   end loop;
end;
/
-- drop tables
begin
   for t in (select table_name from user_tables)
   loop
       execute immediate 'drop table "' || t.table_name || '" CASCADE CONSTRAINTS PURGE ';
   end loop;
end;
/
-- drop sequences
begin
   for s in (select sequence_name from user_sequences)
   loop
       execute immediate 'drop sequence "' || s.sequence_name || '"';
   end loop;
end;
/

CREATE TABLE "entity" (
    entity_id number(19) NOT NULL,
    legal_entity_number varchar2(30), -- LEI
    file_number varchar2(30), -- authority internal number
    reference_number varchar2(30), -- external code, e.g. CIK
    tax_number varchar2(30),
    standard_industry_code integer DEFAULT -1,
    name varchar2(1024),
    legal_state varchar2(32),
    phone varchar2(32),
    phys_addr1 varchar2(128), -- physical (real) address
    phys_addr2 varchar2(128),
    phys_city varchar2(128),
    phys_state varchar2(128),
    phys_zip varchar2(32),
    phys_country varchar2(16),
    mail_addr1 varchar2(128), -- mailing (postal) address
    mail_addr2 varchar2(128),
    mail_city varchar2(128),
    mail_state varchar2(128),
    mail_zip varchar2(32),
    mail_country varchar2(16),
    fiscal_year_end varchar2(6),
    filer_category varchar2(128),
    public_float binary_double,
    trading_symbol varchar2(32)
);
CREATE INDEX entity_index01 ON "entity" (entity_id);
CREATE INDEX entity_index02 ON "entity" (file_number);
CREATE INDEX entity_index03 ON "entity" (reference_number);
CREATE INDEX entity_index04 ON "entity" (legal_entity_number);
CREATE INDEX entity_index05 ON "entity" (legal_entity_number, file_number);

CREATE TRIGGER entity_insert_trigger BEFORE INSERT ON "entity" 
  FOR EACH ROW
    BEGIN
       SELECT seq_entity.NEXTVAL INTO :NEW.entity_id from dual;
    END;
/

CREATE TABLE "former_entity" (
    entity_id number(19) NOT NULL,
    date_changed date,
    former_name varchar2(1024)
);
CREATE INDEX former_entity_index02 ON "former_entity" (entity_id);

CREATE SEQUENCE seq_filing;

CREATE TABLE "filing" (
    filing_id number(19) NOT NULL,
    filing_number varchar2(30) NOT NULL,
    form_type varchar2(30),
    entity_id number(19) NOT NULL,
    reference_number varchar2(30), -- external code, e.g. CIK
    accepted_timestamp date DEFAULT sysdate,
    is_most_current number(1) DEFAULT 0,
    filing_date date NOT NULL,
    creation_software nclob,
    authority_html_url nclob,
    entry_url nclob
);
CREATE INDEX filing_index01 ON "filing" (filing_id);
CREATE UNIQUE INDEX filing_index02 ON "filing" (filing_number);

CREATE TRIGGER filing_insert_trigger BEFORE INSERT ON "filing" 
  FOR EACH ROW
    BEGIN
       SELECT seq_filing.NEXTVAL INTO :NEW.filing_id from dual;
    END;
/

-- object sequence can be any element that can terminate a relationship (aspect, type, resource, data point, document, role type, ...)
-- or be a reference of a message (report or any of above)
CREATE SEQUENCE seq_object;

CREATE TABLE "report" (
    report_id number(19) NOT NULL,
    filing_id number(19) NOT NULL,
    report_data_doc_id number(19),  -- instance or primary inline document
    report_schema_doc_id number(19),  -- extension schema of the report (primary)
    agency_schema_doc_id number(19),  -- agency schema (receiving authority)
    standard_schema_doc_id number(19)  -- e.g., IFRS, XBRL-US, or EDInet schema
);
CREATE INDEX report_index01 ON "report" (report_id);
CREATE INDEX report_index02 ON "report" (filing_id);

CREATE TRIGGER report_insert_trigger BEFORE INSERT ON "report"
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.report_id from dual;
    END;
/
CREATE TABLE "document" (
    document_id number(19) NOT NULL,
    document_url varchar2(2048) NOT NULL,
    document_type varchar2(32),  -- ModelDocument.Type string value
    namespace varchar2(1024),  -- targetNamespace if schema else NULL
    PRIMARY KEY (document_id)
);
CREATE INDEX document_index02 ON "document" (document_url) COMPRESS;

CREATE TRIGGER document_insert_trigger BEFORE INSERT ON "document" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.document_id from dual;
    END;
/
-- documents referenced by report or document

CREATE TABLE "referenced_documents" (
    object_id number(19) NOT NULL,
    document_id number(19) NOT NULL
);
CREATE INDEX referenced_documents_index01 ON "referenced_documents" (object_id);
CREATE UNIQUE INDEX referenced_documents_index02 ON "referenced_documents" (object_id, document_id);

CREATE TABLE "aspect" (
    aspect_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    qname varchar2(1024) NOT NULL,  -- clark notation qname (do we need this?)
    name varchar2(1024) NOT NULL,  -- local qname
    datatype_id number(19),
    base_type varchar2(128), -- xml base type if any
    substitution_group_aspect_id number(19),
    balance varchar2(16),
    period_type varchar2(16),
    abstract number(1) NOT NULL,
    nillable number(1) NOT NULL,
    is_numeric number(1) NOT NULL,
    is_monetary number(1) NOT NULL,
    is_text_block number(1) NOT NULL,
    PRIMARY KEY (aspect_id)
);
CREATE INDEX aspect_index02 ON "aspect" (document_id);
CREATE INDEX aspect_index03 ON "aspect" (qname) COMPRESS;

CREATE TRIGGER aspect_insert_trigger BEFORE INSERT ON "aspect" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.aspect_id from dual;
    END;
/
CREATE TABLE "data_type" (
    data_type_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    qname varchar2(1024) NOT NULL,  -- clark notation qname (do we need this?)
    name varchar2(1024) NOT NULL,  -- local qname
    base_type varchar2(128), -- xml base type if any
    derived_from_type_id number(19),
    PRIMARY KEY (data_type_id)
);
CREATE INDEX data_type_index02 ON "data_type" (document_id);
CREATE INDEX data_type_index03 ON "data_type" (qname) COMPRESS;

CREATE TRIGGER data_type_insert_trigger BEFORE INSERT ON "data_type" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.data_type_id from dual;
    END;
/
CREATE TABLE "role_type" (
    role_type_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    role_uri varchar2(1024) NOT NULL,
    definition nclob,
    PRIMARY KEY (role_type_id)
);
CREATE INDEX role_type_index02 ON "role_type" (document_id);
CREATE INDEX role_type_index03 ON "role_type" (role_uri) COMPRESS;

CREATE TRIGGER role_type_insert_trigger BEFORE INSERT ON "role_type" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.role_type_id from dual;
    END;
/
CREATE TABLE "arcrole_type" (
    arcrole_type_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    arcrole_uri varchar2(1024) NOT NULL,
    cycles_allowed varchar2(10) NOT NULL,
    definition nclob,
    PRIMARY KEY (arcrole_type_id)
);
CREATE INDEX arcrole_type_index02 ON "arcrole_type" (document_id);
CREATE INDEX arcrole_type_index03 ON "arcrole_type" (arcrole_uri) COMPRESS;

CREATE TRIGGER arcrole_type_insert_trigger BEFORE INSERT ON "arcrole_type" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.arcrole_type_id from dual;
    END;
/
CREATE TABLE "used_on" (
    object_id number(19) NOT NULL,
    aspect_id number(19) NOT NULL
);
CREATE INDEX used_on_index01 ON "used_on" (object_id);
CREATE UNIQUE INDEX used_on_index02 ON "used_on" (object_id, aspect_id);

CREATE TABLE "resource" (
    resource_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    qname varchar2(1024) NOT NULL,  -- clark notation qname (do we need this?)
    role varchar2(1024),
    value nclob,
    xml_lang varchar2(16),
    PRIMARY KEY (resource_id)
);
CREATE UNIQUE INDEX resource_index02 ON "resource" (document_id, xml_child_seq);

CREATE TRIGGER resource_insert_trigger BEFORE INSERT ON "resource" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.resource_id from dual;
    END;
/
CREATE SEQUENCE seq_relationship_set;

CREATE TABLE "relationship_set" (
    relationship_set_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    arc_qname varchar2(1024) NOT NULL,  -- clark notation qname (do we need this?)
    link_qname varchar2(1024) NOT NULL,  -- clark notation qname (do we need this?)
    arc_role varchar2(1024) NOT NULL,
    link_role varchar2(1024) NOT NULL,
    PRIMARY KEY (relationship_set_id)
);
CREATE INDEX relationship_set_index02 ON "relationship_set" (document_id);
CREATE INDEX relationship_set_index03 ON "relationship_set" (link_role) COMPRESS;
CREATE INDEX relationship_set_index04 ON "relationship_set" (arc_role) COMPRESS;

CREATE TRIGGER rel_set_insert_trigger BEFORE INSERT ON "relationship_set" 
  FOR EACH ROW
    BEGIN
       SELECT seq_relationship_set.NEXTVAL INTO :NEW.relationship_set_id from dual;
    END;
/

CREATE TABLE "root" (
    relationship_set_id number(19) NOT NULL,
    relationship_id number(19) NOT NULL
);
CREATE INDEX root_index02 ON "root" (relationship_set_id);
/
CREATE TABLE "relationship" (
    relationship_id number(19) NOT NULL,
    document_id number(19) NOT NULL,
    xml_id nclob,
    xml_child_seq varchar2(512),
    relationship_set_id number(19) NOT NULL,
    reln_order binary_double,
    from_id number(19),
    to_id number(19),
    calculation_weight binary_double,
    tree_sequence integer NOT NULL,
    tree_depth integer NOT NULL,
    preferred_label_role varchar2(1024),
    PRIMARY KEY (relationship_id)
);
CREATE INDEX relationship_index02 ON "relationship" (relationship_set_id);
CREATE INDEX relationship_index03 ON "relationship" (relationship_set_id, tree_depth);
CREATE INDEX relationship_index04 ON "relationship" (relationship_set_id, document_id, xml_child_seq);
CREATE INDEX relationship_index05 ON "relationship" (from_id);

CREATE TRIGGER relationship_insert_trigger BEFORE INSERT ON "relationship" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.relationship_id from dual;
    END;
/
CREATE TABLE "data_point" (
    datapoint_id number(19) NOT NULL,
    report_id number(19),
    document_id number(19) NOT NULL,  -- multiple inline documents are sources of data points
    xml_id nclob,
    xml_child_seq varchar2(512),
    source_line integer,
    parent_datapoint_id number(19), -- id of tuple parent
    aspect_id number(19),
    context_xml_id varchar2(1024), -- (max observed length 693 in SEC 2012-2014)
    entity_identifier_id number(19),
    period_id number(19),
    aspect_value_selection_id number(19),
    unit_id number(19),
    is_nil number(1) DEFAULT 0,
    precision_value varchar2(16),
    decimals_value varchar2(16),
    effective_value binary_double,
    value nclob,
    PRIMARY KEY (datapoint_id)
);
CREATE INDEX data_point_index02 ON "data_point" (document_id, xml_child_seq);
CREATE INDEX data_point_index03 ON "data_point" (report_id);
CREATE INDEX data_point_index04 ON "data_point" (aspect_id);

CREATE TRIGGER data_point_insert_trigger BEFORE INSERT ON "data_point" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.datapoint_id from dual;
    END;
/
CREATE TABLE "entity_identifier" (
    entity_identifier_id number(19) NOT NULL,
    report_id number(19),
    scheme varchar2(1024) NOT NULL,
    identifier varchar2(1024) NOT NULL,
    PRIMARY KEY (entity_id)
);
CREATE INDEX entity_identifier_index02 ON "entity_identifier" (report_id, scheme, identifier);

CREATE TRIGGER entity_insert_trigger BEFORE INSERT ON "entity" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.entity_identifier_id from dual;
    END;
/
CREATE TABLE "period" (
    period_id number(19) NOT NULL,
    report_id number(19),
    start_date date,
    end_date date,
    is_instant number(1) NOT NULL,
    is_forever number(1) NOT NULL,
    PRIMARY KEY (period_id)
);
CREATE INDEX period_index02 ON "period" (report_id, start_date, end_date, is_instant, is_forever);

CREATE TRIGGER period_insert_trigger BEFORE INSERT ON "period" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.period_id from dual;
    END;
/
CREATE TABLE "unit" (
    unit_id number(19) NOT NULL,
    report_id number(19),
    xml_id nclob,
    xml_child_seq varchar2(512),
    measures_hash varchar2(36),
    PRIMARY KEY (unit_id)
);
CREATE INDEX unit_index02 ON "unit" (report_id, measures_hash);

CREATE TRIGGER unit_insert_trigger BEFORE INSERT ON "unit" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.unit_id from dual;
    END;
/
CREATE TABLE "unit_measure" (
    unit_id number(19) NOT NULL,
    qname varchar2(1024) NOT NULL,  -- clark notation qname (first if multiple)
    measures_hash char(32),
    is_multiplicand number(1) NOT NULL
);
CREATE INDEX unit_measure_index01 ON "unit_measure" (unit_id);
CREATE INDEX unit_measure_index02 ON "unit_measure" (unit_id, qname, is_multiplicand);

CREATE TABLE "aspect_value_selection_set" (
    aspect_value_selection_id number(19) NOT NULL,
    report_id number(19)
);
CREATE UNIQUE INDEX aspect_value_sel_set_index01 ON "aspect_value_selection_set" (aspect_value_selection_id);
CREATE INDEX aspect_value_sel_set_index02 ON "aspect_value_selection_set" (report_id);

CREATE TRIGGER aspect_value_sel_ins_trigger BEFORE INSERT ON "aspect_value_selection_set" 
  FOR EACH ROW
    BEGIN
       SELECT seq_object.NEXTVAL INTO :NEW.aspect_value_selection_id from dual;
    END;
/
CREATE TABLE "aspect_value_selection" (
    aspect_value_selection_id number(19) NOT NULL,
    aspect_id number(19) NOT NULL,
    aspect_value_id number(19),
    is_typed_value number(1) NOT NULL,
    typed_value nclob
);
CREATE INDEX aspect_value_selection_index01 ON "aspect_value_selection" (aspect_value_selection_id);
CREATE INDEX aspect_value_selection_index02 ON "aspect_value_selection" (aspect_id);

CREATE TABLE "table_data_points"(
    report_id bigint,
    object_id bigint NOT NULL, -- may be any role_type or aspect defining a table table with 'seq_object' id
    table_code varchar2(16),  -- short code of table, like BS, PL, or 4.15.221
    datapoint_id bigint -- id of data_point in this table (according to its aspects)
);
CREATE INDEX table_data_points_index01 ON "table_data_points" (report_id);
CREATE INDEX table_data_points_index02 ON "table_data_points" (table_code);
CREATE INDEX table_data_points_index03 ON "table_data_points" (datapoint_id);

CREATE SEQUENCE seq_message;

CREATE TABLE "message" (
    message_id number(19) NOT NULL,
    report_id number(19),
    sequence_in_report int,
    message_code varchar2(256),
    message_level varchar2(256),
    value nclob,
    PRIMARY KEY (message_id)
);
CREATE INDEX message_index02 ON "message" (report_id, sequence_in_report);

CREATE TRIGGER message_insert_trigger BEFORE INSERT ON "message" 
  FOR EACH ROW
    BEGIN
       SELECT seq_message.NEXTVAL INTO :NEW.message_id from dual;
    END;
/
CREATE TABLE "message_reference" (
    message_id number(19) NOT NULL,
    object_id number(19) NOT NULL -- may be any table with 'seq_object' id
);
CREATE INDEX message_reference_index01 ON "message_reference" (message_id);
CREATE UNIQUE INDEX message_reference_index02 ON "message_reference" (message_id, object_id);

CREATE SEQUENCE seq_industry;

CREATE TABLE "industry" (
    industry_id number(19) NOT NULL,
    industry_classification varchar2(16),
    industry_code integer,
    industry_description varchar2(512),
    depth integer,
    parent_id number(19),
    PRIMARY KEY (industry_id)
);

INSERT INTO "industry" (industry_id, industry_classification, industry_code, industry_description, depth, parent_id) 
          SELECT 4315, 'SEC', 3576, 'Computer Communications Equipment', 4, 2424 FROM dual
UNION ALL SELECT 4316, 'SEC', 4955, 'Hazardous Waste Management', 4, 2552 FROM dual
UNION ALL SELECT 4317, 'SEC', 4990, 'Hazardous Waste Management', 3, 2792 FROM dual
UNION ALL SELECT 4318, 'SEC', 4991, 'Hazardous Waste Management', 4, 4317 FROM dual
UNION ALL SELECT 4319, 'SEC', 5412, 'Convenience Stores', 4, 2618 FROM dual
UNION ALL SELECT 4320, 'SEC', 6170, 'Finance Lessors', 3, 2659 FROM dual
UNION ALL SELECT 4322, 'SEC', 6180, 'Asset Backed Securities', 3, 2659 FROM dual
UNION ALL SELECT 4323, 'SEC', 6189, 'Asset Backed Securities', 4, 4322 FROM dual
UNION ALL SELECT 4324, 'SEC', 6190, 'Financial Services', 3, 2659 FROM dual
UNION ALL SELECT 4325, 'SEC', 6199, 'Financial Services', 4, 4324 FROM dual
UNION ALL SELECT 4326, 'SEC', 6532, 'Real Estate Dealers (for their own account)', 4, 2694 FROM dual
UNION ALL SELECT 4327, 'SEC', 6770, 'Blank Checks', 3, 2703 FROM dual
UNION ALL SELECT 4328, 'SEC', 6795, 'Mineral Royalty Traders', 4, 2704 FROM dual
UNION ALL SELECT 4329, 'SEC', 7385, 'Telephone Interconnect Systems', 4, 2731 FROM dual
UNION ALL SELECT 4337, 'SEC', 8880, 'American Depositary Receipts', 2, 4336 FROM dual
UNION ALL SELECT 4338, 'SEC', 8888, 'Foreign Governments', 2, 4336 FROM dual
UNION ALL SELECT 4339, 'SEC', 9995, 'Non-operating Establishments', 2, 4336 FROM dual
UNION ALL SELECT 4340, 'SEC', 6172, 'Finance Lessors', 4, 4320 FROM dual
UNION ALL SELECT 4336, 'SEC', 99, 'Miscellaneous', 1, NULL FROM dual
UNION ALL SELECT 1, 'NAICS', 11, 'Agriculture, Forestry, Fishing and Hunting', 1, NULL FROM dual
UNION ALL SELECT 132, 'NAICS', 21, 'Mining, Quarrying, and Oil and Gas Extraction', 1, NULL FROM dual
UNION ALL SELECT 98, 'NAICS', 113, 'Forestry and Logging', 2, 1 FROM dual
UNION ALL SELECT 180, 'NAICS', 22, 'Utilities', 1, NULL FROM dual
UNION ALL SELECT 205, 'NAICS', 23, 'Construction', 1, NULL FROM dual
UNION ALL SELECT 278, 'NAICS', 31, 'Manufacturing', 4, NULL FROM dual
UNION ALL SELECT 572, 'NAICS', 327, 'Nonmetallic Mineral Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 700, 'NAICS', 333, 'Machinery Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 930, 'NAICS', 42, 'Wholesale Trade', 1, NULL FROM dual
UNION ALL SELECT 1095, 'NAICS', 44, 'Retail Trade', 4, NULL FROM dual
UNION ALL SELECT 1194, 'NAICS', 4482, 'Shoe Stores', 3, 1180 FROM dual
UNION ALL SELECT 1262, 'NAICS', 48, 'Transportation and Warehousing', 4, NULL FROM dual
UNION ALL SELECT 1273, 'NAICS', 482, 'Rail Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1402, 'NAICS', 51, 'Information', 1, NULL FROM dual
UNION ALL SELECT 1480, 'NAICS', 52, 'Finance and Insurance', 1, NULL FROM dual
UNION ALL SELECT 1569, 'NAICS', 53, 'Real Estate and Rental and Leasing', 1, NULL FROM dual
UNION ALL SELECT 1624, 'NAICS', 54, 'Professional, Scientific, and Technical Services', 1, NULL FROM dual
UNION ALL SELECT 1591, 'NAICS', 532, 'Rental and Leasing Services', 2, 1569 FROM dual
UNION ALL SELECT 1718, 'NAICS', 55, 'Management of Companies and Enterprises', 1, NULL FROM dual
UNION ALL SELECT 1725, 'NAICS', 56, 'Administrative and Support and Waste Management and Remediation Services', 1, NULL FROM dual
UNION ALL SELECT 1812, 'NAICS', 61, 'Educational Services', 1, NULL FROM dual
UNION ALL SELECT 1850, 'NAICS', 62, 'Health Care and Social Assistance', 1, NULL FROM dual
UNION ALL SELECT 1942, 'NAICS', 71, 'Arts, Entertainment, and Recreation', 1, NULL FROM dual
UNION ALL SELECT 2003, 'NAICS', 72, 'Accommodation and Food Services', 1, NULL FROM dual
UNION ALL SELECT 2037, 'NAICS', 81, 'Other Services (except Public Administration)', 1, NULL FROM dual
UNION ALL SELECT 2071, 'NAICS', 812, 'Personal and Laundry Services', 2, 2037 FROM dual
UNION ALL SELECT 2135, 'NAICS', 92, 'Public Administration', 1, NULL FROM dual
UNION ALL SELECT 2900, 'SIC', 1099, 'Metal Ores, nec', 4, 2898 FROM dual
UNION ALL SELECT 3040, 'SIC', 2099, 'Food Preparations, nec', 4, 3033 FROM dual
UNION ALL SELECT 3162, 'SIC', 2541, 'Wood Partitions & Fixtures', 4, 3161 FROM dual
UNION ALL SELECT 3295, 'SIC', 3161, 'Luggage', 4, 3294 FROM dual
UNION ALL SELECT 2797, 'SEC', 90, 'Public Administration', 1, NULL FROM dual
UNION ALL SELECT 2789, 'SEC', 10, 'Mining', 1, NULL FROM dual
UNION ALL SELECT 2790, 'SEC', 15, 'Construction', 1, NULL FROM dual
UNION ALL SELECT 2791, 'SEC', 20, 'Manufacturing', 1, NULL FROM dual
UNION ALL SELECT 2792, 'SEC', 40, 'Transportation, Communications, Eletric, Gas and Sanitary Services', 1, NULL FROM dual
UNION ALL SELECT 2793, 'SEC', 50, 'Wholesale Trade', 1, NULL FROM dual
UNION ALL SELECT 2794, 'SEC', 52, 'Retail Trade', 1, NULL FROM dual
UNION ALL SELECT 2795, 'SEC', 60, 'Finance, Insurance, and Real Estate', 1, NULL FROM dual
UNION ALL SELECT 2796, 'SEC', 70, 'Services', 1, NULL FROM dual
UNION ALL SELECT 4314, 'SIC', 90, 'Public Administration', 1, NULL FROM dual
UNION ALL SELECT 4306, 'SIC', 10, 'Mining', 1, NULL FROM dual
UNION ALL SELECT 4307, 'SIC', 15, 'Construction', 1, NULL FROM dual
UNION ALL SELECT 4308, 'SIC', 20, 'Manufacturing', 1, NULL FROM dual
UNION ALL SELECT 4309, 'SIC', 40, 'Transportation, Communications, Eletric, Gas and Sanitary Services', 1, NULL FROM dual
UNION ALL SELECT 4310, 'SIC', 50, 'Wholesale Trade', 1, NULL FROM dual
UNION ALL SELECT 4311, 'SIC', 52, 'Retail Trade', 1, NULL FROM dual
UNION ALL SELECT 4312, 'SIC', 60, 'Finance, Insurance, and Real Estate', 1, NULL FROM dual
UNION ALL SELECT 4313, 'SIC', 70, 'Services', 1, NULL FROM dual
UNION ALL SELECT 2788, 'SEC', 1, 'Agriculture, Forestry, And Fishing', 1, NULL FROM dual
UNION ALL SELECT 4305, 'SIC', 1, 'Agriculture, Forestry, And Fishing', 1, NULL FROM dual
UNION ALL SELECT 2513, 'SEC', 4010, 'Railroads', 3, 2512 FROM dual
UNION ALL SELECT 2799, 'SIC', 110, 'Cash Grains', 3, 2798 FROM dual
UNION ALL SELECT 3420, 'SIC', 3510, 'Engines & Turbines', 3, 3419 FROM dual
UNION ALL SELECT 2868, 'SIC', 800, 'FORESTRY', 2, 4305 FROM dual
UNION ALL SELECT 2804, 'SIC', 119, 'Cash Grains, nec', 4, 2799 FROM dual
UNION ALL SELECT 2800, 'SIC', 111, 'Wheat', 4, 2799 FROM dual
UNION ALL SELECT 2801, 'SIC', 112, 'Rice', 4, 2799 FROM dual
UNION ALL SELECT 2802, 'SIC', 115, 'Corn', 4, 2799 FROM dual
UNION ALL SELECT 2803, 'SIC', 116, 'Soybeans', 4, 2799 FROM dual
UNION ALL SELECT 2810, 'SIC', 139, 'Field Crops, Except Cash Grains, nec', 4, 2805 FROM dual
UNION ALL SELECT 2809, 'SIC', 134, 'Potato Growers', 4, 2805 FROM dual
UNION ALL SELECT 2808, 'SIC', 133, 'Sugarcane And Sugar Beets', 4, 2805 FROM dual
UNION ALL SELECT 2807, 'SIC', 132, 'Tobacco Growers', 4, 2805 FROM dual
UNION ALL SELECT 2806, 'SIC', 131, 'Cotton', 4, 2805 FROM dual
UNION ALL SELECT 2812, 'SIC', 161, 'Vegetables And Melons', 4, 2811 FROM dual
UNION ALL SELECT 2814, 'SIC', 171, 'Berry Crops', 4, 2813 FROM dual
UNION ALL SELECT 2815, 'SIC', 172, 'Grapes/Vineyards', 4, 2813 FROM dual
UNION ALL SELECT 2816, 'SIC', 173, 'Tree Nuts', 4, 2813 FROM dual
UNION ALL SELECT 2214, 'SEC', 900, 'FISHING, HUNTING & TRAPPING', 2, 2788 FROM dual
UNION ALL SELECT 2213, 'SEC', 800, 'FORESTRY', 2, 2788 FROM dual
UNION ALL SELECT 105, 'NAICS', 1133, 'Logging', 3, 98 FROM dual
UNION ALL SELECT 368, 'NAICS', 313, 'Textile Mills', 2, 278 FROM dual
UNION ALL SELECT 2817, 'SIC', 174, 'Citrus Fruits', 4, 2813 FROM dual
UNION ALL SELECT 2818, 'SIC', 175, 'Deciduous Tree Fruits/Orchards', 4, 2813 FROM dual
UNION ALL SELECT 2819, 'SIC', 179, 'Fruits & Tree Nuts, nec', 4, 2813 FROM dual
UNION ALL SELECT 2821, 'SIC', 181, 'Ornamental Nursery Products', 4, 2820 FROM dual
UNION ALL SELECT 2822, 'SIC', 182, 'Food Crops Grown Under Cover', 4, 2820 FROM dual
UNION ALL SELECT 2824, 'SIC', 191, 'General Farms, Primarily Crop', 4, 2823 FROM dual
UNION ALL SELECT 2829, 'SIC', 213, 'Hogs', 4, 2826 FROM dual
UNION ALL SELECT 2877, 'SIC', 912, 'Finfish', 4, 2876 FROM dual
UNION ALL SELECT 2830, 'SIC', 214, 'Sheep & Goats', 4, 2826 FROM dual
UNION ALL SELECT 2828, 'SIC', 212, 'Beef Cattle, Except Feedlots', 4, 2826 FROM dual
UNION ALL SELECT 2827, 'SIC', 211, 'Beef Cattle Feedlots', 4, 2826 FROM dual
UNION ALL SELECT 2831, 'SIC', 219, 'General Livestock, Except Dairy/Poultry', 4, 2826 FROM dual
UNION ALL SELECT 2833, 'SIC', 241, 'Dairy Farms', 4, 2832 FROM dual
UNION ALL SELECT 2978, 'SIC', 1794, 'Excavation Work', 4, 2975 FROM dual
UNION ALL SELECT 2979, 'SIC', 1795, 'Wrecking & Demolition Work', 4, 2975 FROM dual
UNION ALL SELECT 2980, 'SIC', 1796, 'Installing Building Equipment, nec', 4, 2975 FROM dual
UNION ALL SELECT 2981, 'SIC', 1799, 'Special Trade Contractors, nec', 4, 2975 FROM dual
UNION ALL SELECT 2976, 'SIC', 1791, 'Structural Steel Erection', 4, 2975 FROM dual
UNION ALL SELECT 2242, 'SEC', 2011, 'Meat Packing Plants', 4, 2241 FROM dual
UNION ALL SELECT 2244, 'SEC', 2015, 'Poultry Slaughtering & Processing', 4, 2241 FROM dual
UNION ALL SELECT 2243, 'SEC', 2013, 'Sausages & Other Prepared Meats', 4, 2241 FROM dual
UNION ALL SELECT 2984, 'SIC', 2011, 'Meat Packing Plants', 4, 2983 FROM dual
UNION ALL SELECT 2986, 'SIC', 2015, 'Poultry Slaughtering & Processing', 4, 2983 FROM dual
UNION ALL SELECT 2985, 'SIC', 2013, 'Sausages & Other Prepared Meats', 4, 2983 FROM dual
UNION ALL SELECT 2246, 'SEC', 2024, 'Ice Cream & Frozen Desserts', 4, 2245 FROM dual
UNION ALL SELECT 2990, 'SIC', 2023, 'Dry, Condensed & Evaporated Products', 4, 2987 FROM dual
UNION ALL SELECT 2991, 'SIC', 2024, 'Ice Cream & Frozen Desserts', 4, 2987 FROM dual
UNION ALL SELECT 2989, 'SIC', 2022, 'Cheese, Natural & Processed', 4, 2987 FROM dual
UNION ALL SELECT 2988, 'SIC', 2021, 'Creamery Butter', 4, 2987 FROM dual
UNION ALL SELECT 2992, 'SIC', 2026, 'Fluid Milk', 4, 2987 FROM dual
UNION ALL SELECT 2248, 'SEC', 2033, 'Canned Fruits & Vegetables', 4, 2247 FROM dual
UNION ALL SELECT 2999, 'SIC', 2038, 'Frozen Specialties, nec', 4, 2993 FROM dual
UNION ALL SELECT 2994, 'SIC', 2032, 'Canned Specialties', 4, 2993 FROM dual
UNION ALL SELECT 2998, 'SIC', 2037, 'Frozen Fruits & Vegetables', 4, 2993 FROM dual
UNION ALL SELECT 2997, 'SIC', 2035, 'Pickles, Sauces & Salad Dressings', 4, 2993 FROM dual
UNION ALL SELECT 2607, 'SEC', 5211, 'Lumber & Other Building Materials', 4, 2606 FROM dual
UNION ALL SELECT 3806, 'SIC', 5211, 'Lumber & Other Building Materials', 4, 3805 FROM dual
UNION ALL SELECT 3808, 'SIC', 5231, 'Paint, Glass & Wallpaper Stores', 4, 3807 FROM dual
UNION ALL SELECT 3810, 'SIC', 5251, 'Hardware Stores', 4, 3809 FROM dual
UNION ALL SELECT 3812, 'SIC', 5261, 'Retail Nurseries & Garden Stores', 4, 3811 FROM dual
UNION ALL SELECT 2609, 'SEC', 5271, 'Mobile Homes Dealers', 4, 2608 FROM dual
UNION ALL SELECT 3814, 'SIC', 5271, 'Mobile Homes Dealers', 4, 3813 FROM dual
UNION ALL SELECT 2612, 'SEC', 5311, 'Department Stores', 4, 2611 FROM dual
UNION ALL SELECT 3817, 'SIC', 5311, 'Department Stores', 4, 3816 FROM dual
UNION ALL SELECT 2614, 'SEC', 5331, 'Variety Stores', 4, 2613 FROM dual
UNION ALL SELECT 3819, 'SIC', 5331, 'Variety Stores', 4, 3818 FROM dual
UNION ALL SELECT 2616, 'SEC', 5399, 'Miscellaneous General Merchandise Stores', 4, 2615 FROM dual
UNION ALL SELECT 3821, 'SIC', 5399, 'Miscellaneous General Merchandise Stores', 4, 3820 FROM dual
UNION ALL SELECT 2619, 'SEC', 5411, 'Grocery Stores', 4, 2618 FROM dual
UNION ALL SELECT 3824, 'SIC', 5411, 'Grocery Stores', 4, 3823 FROM dual
UNION ALL SELECT 3826, 'SIC', 5421, 'Meat & Fish Markets', 4, 3825 FROM dual
UNION ALL SELECT 3828, 'SIC', 5431, 'Fruit & Vegetable Markets', 4, 3827 FROM dual
UNION ALL SELECT 3830, 'SIC', 5441, 'Candy, Nut & Confectionery Stores', 4, 3829 FROM dual
UNION ALL SELECT 3832, 'SIC', 5451, 'Dairy Products Stores', 4, 3831 FROM dual
UNION ALL SELECT 3834, 'SIC', 5461, 'Retail Bakeries', 4, 3833 FROM dual
UNION ALL SELECT 3836, 'SIC', 5499, 'Miscellaneous Food Stores', 4, 3835 FROM dual
UNION ALL SELECT 3839, 'SIC', 5511, 'New & Used Car Dealers', 4, 3838 FROM dual
UNION ALL SELECT 3841, 'SIC', 5521, 'Used Car Dealers', 4, 3840 FROM dual
UNION ALL SELECT 2622, 'SEC', 5531, 'Auto & Home Supply Stores', 4, 2621 FROM dual
UNION ALL SELECT 3919, 'SIC', 6011, 'Federal Reserve Banks', 4, 3918 FROM dual
UNION ALL SELECT 2711, 'SEC', 7011, 'Hotels & Motels', 4, 2710 FROM dual
UNION ALL SELECT 4009, 'SIC', 7011, 'Hotels & Motels', 4, 4008 FROM dual
UNION ALL SELECT 4011, 'SIC', 7021, 'Rooming & Boarding Houses', 4, 4010 FROM dual
UNION ALL SELECT 4013, 'SIC', 7032, 'Sporting & Recreational Camps', 4, 4012 FROM dual
UNION ALL SELECT 4014, 'SIC', 7033, 'Trailer Parks & Campsites', 4, 4012 FROM dual
UNION ALL SELECT 4016, 'SIC', 7041, 'Membership Basis Organization Hotels', 4, 4015 FROM dual
UNION ALL SELECT 4024, 'SIC', 7217, 'Carpet & Upholstery Cleaners', 4, 4018 FROM dual
UNION ALL SELECT 4020, 'SIC', 7212, 'Garment Pressing & Cleaners'' Agents', 4, 4018 FROM dual
UNION ALL SELECT 4021, 'SIC', 7213, 'Linen Supply', 4, 4018 FROM dual
UNION ALL SELECT 4022, 'SIC', 7215, 'Coin-Operated Laundries & Cleaning', 4, 4018 FROM dual
UNION ALL SELECT 4023, 'SIC', 7216, 'Drycleaning Plants, Except Rug', 4, 4018 FROM dual
UNION ALL SELECT 12, 'NAICS', 111150, 'Corn Farming', 5, 13 FROM dual
UNION ALL SELECT 107, 'NAICS', 11331, 'Logging', 4, 105 FROM dual
UNION ALL SELECT 1151, 'NAICS', 44521, 'Meat Markets', 4, 1149 FROM dual
UNION ALL SELECT 1773, 'NAICS', 561622, 'Locksmiths', 5, 1771 FROM dual
UNION ALL SELECT 2315, 'SEC', 2770, 'Greeting Cards', 3, 2302 FROM dual
UNION ALL SELECT 2860, 'SIC', 752, 'Animal Specialty Services', 4, 2858 FROM dual
UNION ALL SELECT 2837, 'SIC', 253, 'Turkey Farms', 4, 2834 FROM dual
UNION ALL SELECT 2839, 'SIC', 259, 'Poultry & Eggs, nec', 4, 2834 FROM dual
UNION ALL SELECT 2838, 'SIC', 254, 'Poultry Farms', 4, 2834 FROM dual
UNION ALL SELECT 2836, 'SIC', 252, 'Chicken Eggs', 4, 2834 FROM dual
UNION ALL SELECT 2835, 'SIC', 251, 'Broiler, Fryer & Roaster Chickens', 4, 2834 FROM dual
UNION ALL SELECT 2843, 'SIC', 273, 'Aquaculture', 4, 2840 FROM dual
UNION ALL SELECT 2844, 'SIC', 279, 'Animal Specialties, nec', 4, 2840 FROM dual
UNION ALL SELECT 2841, 'SIC', 271, 'Fur Bearing Animals & Rabbits', 4, 2840 FROM dual
UNION ALL SELECT 2842, 'SIC', 272, 'Horses & Other Equines', 4, 2840 FROM dual
UNION ALL SELECT 2846, 'SIC', 291, 'General Farms, Primarily Animal', 4, 2845 FROM dual
UNION ALL SELECT 2849, 'SIC', 711, 'Soil Preparation Services', 4, 2848 FROM dual
UNION ALL SELECT 2854, 'SIC', 724, 'Cotton Ginning', 4, 2850 FROM dual
UNION ALL SELECT 2853, 'SIC', 723, 'Crop Preparation Services For Market', 4, 2850 FROM dual
UNION ALL SELECT 2851, 'SIC', 721, 'Crop Planting & Protecting', 4, 2850 FROM dual
UNION ALL SELECT 2852, 'SIC', 722, 'Crop Harvesting', 4, 2850 FROM dual
UNION ALL SELECT 2856, 'SIC', 741, 'Veterinary Services for Livestock', 4, 2855 FROM dual
UNION ALL SELECT 2857, 'SIC', 742, 'Veterinary Services, Specialties', 4, 2855 FROM dual
UNION ALL SELECT 2859, 'SIC', 751, 'Livestock Services, Except Veterinary', 4, 2858 FROM dual
UNION ALL SELECT 2862, 'SIC', 761, 'Farm Labor Contractors', 4, 2861 FROM dual
UNION ALL SELECT 2863, 'SIC', 762, 'Farm Management Services', 4, 2861 FROM dual
UNION ALL SELECT 2866, 'SIC', 782, 'Lawn & Garden Services', 4, 2864 FROM dual
UNION ALL SELECT 2867, 'SIC', 783, 'Ornamental Shrub & Tree Services', 4, 2864 FROM dual
UNION ALL SELECT 2865, 'SIC', 781, 'Landscape Counseling & Planning', 4, 2864 FROM dual
UNION ALL SELECT 2870, 'SIC', 811, 'Timber Tracts', 4, 2869 FROM dual
UNION ALL SELECT 2872, 'SIC', 831, 'Forest Products', 4, 2871 FROM dual
UNION ALL SELECT 2874, 'SIC', 851, 'Forestry Services', 4, 2873 FROM dual
UNION ALL SELECT 2879, 'SIC', 919, 'Miscellaneous Marine Products', 4, 2876 FROM dual
UNION ALL SELECT 2878, 'SIC', 913, 'Shellfish', 4, 2876 FROM dual
UNION ALL SELECT 2881, 'SIC', 921, 'Fish Hatcheries & Preserves', 4, 2880 FROM dual
UNION ALL SELECT 2883, 'SIC', 971, 'Hunting, Trapping & Game Propagation', 4, 2882 FROM dual
UNION ALL SELECT 2886, 'SIC', 1011, 'Iron Ores', 4, 2885 FROM dual
UNION ALL SELECT 2888, 'SIC', 1021, 'Copper Ores', 4, 2887 FROM dual
UNION ALL SELECT 2890, 'SIC', 1031, 'Lead & Zinc Ores', 4, 2889 FROM dual
UNION ALL SELECT 2893, 'SIC', 1044, 'Silver Ores', 4, 2891 FROM dual
UNION ALL SELECT 2892, 'SIC', 1041, 'Gold Ores', 4, 2891 FROM dual
UNION ALL SELECT 2895, 'SIC', 1061, 'Ferroalloy Ores, Except Vanadium', 4, 2894 FROM dual
UNION ALL SELECT 2223, 'SEC', 1311, 'Crude Petroleum & Natural Gas', 4, 2222 FROM dual
UNION ALL SELECT 2911, 'SIC', 1311, 'Crude Petroleum & Natural Gas', 4, 2910 FROM dual
UNION ALL SELECT 2913, 'SIC', 1321, 'Natural Gas Liquids', 4, 2912 FROM dual
UNION ALL SELECT 2226, 'SEC', 1382, 'Oil & Gas Exploration Services', 4, 2224 FROM dual
UNION ALL SELECT 2227, 'SEC', 1389, 'Oil & Gas Field Services, nec', 4, 2224 FROM dual
UNION ALL SELECT 2225, 'SEC', 1381, 'Drilling Oil & Gas Wells', 4, 2224 FROM dual
UNION ALL SELECT 2917, 'SIC', 1389, 'Oil & Gas Field Services, nec', 4, 2914 FROM dual
UNION ALL SELECT 2915, 'SIC', 1381, 'Drilling Oil & Gas Wells', 4, 2914 FROM dual
UNION ALL SELECT 2916, 'SIC', 1382, 'Oil & Gas Exploration Services', 4, 2914 FROM dual
UNION ALL SELECT 2920, 'SIC', 1411, 'Dimension Stone', 4, 2919 FROM dual
UNION ALL SELECT 2924, 'SIC', 1429, 'Crushed & Broken Stone, nec', 4, 2921 FROM dual
UNION ALL SELECT 2922, 'SIC', 1422, 'Crushed & Broken Limestone', 4, 2921 FROM dual
UNION ALL SELECT 2923, 'SIC', 1423, 'Crushed & Broken Granite', 4, 2921 FROM dual
UNION ALL SELECT 2926, 'SIC', 1442, 'Construction Sand & Gravel', 4, 2925 FROM dual
UNION ALL SELECT 2927, 'SIC', 1446, 'Industrial Sand', 4, 2925 FROM dual
UNION ALL SELECT 2929, 'SIC', 1455, 'Kaolin & Ball Clay', 4, 2928 FROM dual
UNION ALL SELECT 2930, 'SIC', 1459, 'Clay & Related Minerals, nec', 4, 2928 FROM dual
UNION ALL SELECT 2934, 'SIC', 1479, 'Chemical & Fertilizer Mining nec', 4, 2931 FROM dual
UNION ALL SELECT 2932, 'SIC', 1474, 'Potash, Soda & Borate Minerals', 4, 2931 FROM dual
UNION ALL SELECT 2933, 'SIC', 1475, 'Phosphate Rock', 4, 2931 FROM dual
UNION ALL SELECT 2936, 'SIC', 1481, 'Nonmetallic Minerals Services', 4, 2935 FROM dual
UNION ALL SELECT 2938, 'SIC', 1499, 'Miscellaneous Nonmetallic Minerals', 4, 2937 FROM dual
UNION ALL SELECT 2942, 'SIC', 1522, 'Residential Construction, nec', 4, 2940 FROM dual
UNION ALL SELECT 2941, 'SIC', 1521, 'Single-Family Housing Construction', 4, 2940 FROM dual
UNION ALL SELECT 2232, 'SEC', 1531, 'Operative Builders', 4, 2231 FROM dual
UNION ALL SELECT 2944, 'SIC', 1531, 'Operative Builders', 4, 2943 FROM dual
UNION ALL SELECT 2946, 'SIC', 1541, 'Industrial Buildings & Warehouses', 4, 2945 FROM dual
UNION ALL SELECT 2947, 'SIC', 1542, 'Nonresidential Construction, nec', 4, 2945 FROM dual
UNION ALL SELECT 2950, 'SIC', 1611, 'Highway & Street Construction', 4, 2949 FROM dual
UNION ALL SELECT 2236, 'SEC', 1623, 'Water, Sewer & Utility Lines', 4, 2235 FROM dual
UNION ALL SELECT 2954, 'SIC', 1629, 'Heavy Construction, nec', 4, 2951 FROM dual
UNION ALL SELECT 2953, 'SIC', 1623, 'Water, Sewer & Utility Lines', 4, 2951 FROM dual
UNION ALL SELECT 2952, 'SIC', 1622, 'Bridge, Tunnel & Elevated Highway', 4, 2951 FROM dual
UNION ALL SELECT 2957, 'SIC', 1711, 'Plumbing, Heating, Air-Conditioning', 4, 2956 FROM dual
UNION ALL SELECT 2996, 'SIC', 2034, 'Dehydrated Fruits, Vegetables & Soups', 4, 2993 FROM dual
UNION ALL SELECT 2995, 'SIC', 2033, 'Canned Fruits & Vegetables', 4, 2993 FROM dual
UNION ALL SELECT 3005, 'SIC', 2046, 'Wet Corn Milling', 4, 3000 FROM dual
UNION ALL SELECT 3007, 'SIC', 2048, 'Prepared Feeds, nec', 4, 3000 FROM dual
UNION ALL SELECT 3006, 'SIC', 2047, 'Dog & Cat Food', 4, 3000 FROM dual
UNION ALL SELECT 3004, 'SIC', 2045, 'Prepared Flour Mixes & Doughs', 4, 3000 FROM dual
UNION ALL SELECT 3003, 'SIC', 2044, 'Rice Milling', 4, 3000 FROM dual
UNION ALL SELECT 3002, 'SIC', 2043, 'Cereal Breakfast Foods', 4, 3000 FROM dual
UNION ALL SELECT 3001, 'SIC', 2041, 'Flour & Other Grain Mill Products', 4, 3000 FROM dual
UNION ALL SELECT 2251, 'SEC', 2052, 'Cookies & Crackers', 4, 2250 FROM dual
UNION ALL SELECT 3009, 'SIC', 2051, 'Bread, Cake & Related Products', 4, 3008 FROM dual
UNION ALL SELECT 3011, 'SIC', 2053, 'Frozen Bakery Products, Except Bread', 4, 3008 FROM dual
UNION ALL SELECT 3010, 'SIC', 2052, 'Cookies & Crackers', 4, 3008 FROM dual
UNION ALL SELECT 3015, 'SIC', 2063, 'Beet Sugar', 4, 3012 FROM dual
UNION ALL SELECT 3019, 'SIC', 2068, 'Salted & Roasted Nuts & Seeds', 4, 3012 FROM dual
UNION ALL SELECT 3018, 'SIC', 2067, 'Chewing Gum', 4, 3012 FROM dual
UNION ALL SELECT 3017, 'SIC', 2066, 'Chocolate & Cocoa Products', 4, 3012 FROM dual
UNION ALL SELECT 3016, 'SIC', 2064, 'Candy & Other Confectionery Products', 4, 3012 FROM dual
UNION ALL SELECT 3072, 'SIC', 2273, 'Carpets & Rugs', 4, 3071 FROM dual
UNION ALL SELECT 3075, 'SIC', 2282, 'Throwing & Winding Mills', 4, 3073 FROM dual
UNION ALL SELECT 3076, 'SIC', 2284, 'Thread Mills', 4, 3073 FROM dual
UNION ALL SELECT 3074, 'SIC', 2281, 'Yarn Spinning Mills', 4, 3073 FROM dual
UNION ALL SELECT 3079, 'SIC', 2296, 'Tire Cord & Fabrics', 4, 3077 FROM dual
UNION ALL SELECT 3082, 'SIC', 2299, 'Textile Goods, nec', 4, 3077 FROM dual
UNION ALL SELECT 3081, 'SIC', 2298, 'Cordage & Twine', 4, 3077 FROM dual
UNION ALL SELECT 3080, 'SIC', 2297, 'Nonwoven Fabrics', 4, 3077 FROM dual
UNION ALL SELECT 3078, 'SIC', 2295, 'Coated Fabrics, Not Rubberized', 4, 3077 FROM dual
UNION ALL SELECT 3085, 'SIC', 2311, 'Men''s & Boys'' Suits & Coats', 4, 3084 FROM dual
UNION ALL SELECT 3087, 'SIC', 2321, 'Men''s & Boys'' Shirts', 4, 3086 FROM dual
UNION ALL SELECT 3091, 'SIC', 2326, 'Men''s & Boys'' Work Clothing', 4, 3086 FROM dual
UNION ALL SELECT 3092, 'SIC', 2329, 'Men''s & Boys'' Clothing, nec', 4, 3086 FROM dual
UNION ALL SELECT 3088, 'SIC', 2322, 'Men''s & Boys'' Underwear & Nightwear', 4, 3086 FROM dual
UNION ALL SELECT 3089, 'SIC', 2323, 'Men''s & Boys'' Neckwear', 4, 3086 FROM dual
UNION ALL SELECT 3090, 'SIC', 2325, 'Men''s & Boys'' Trousers & Slacks', 4, 3086 FROM dual
UNION ALL SELECT 3097, 'SIC', 2339, 'Women''s & Misses'' Outerwear, nec', 4, 3093 FROM dual
UNION ALL SELECT 3094, 'SIC', 2331, 'Women''s & Misses'' Blouses & Shirts', 4, 3093 FROM dual
UNION ALL SELECT 3095, 'SIC', 2335, 'Women''s & Misses'' & Junior''s Dresses', 4, 3093 FROM dual
UNION ALL SELECT 3096, 'SIC', 2337, 'Women''s & Misses'' Suits & Coats', 4, 3093 FROM dual
UNION ALL SELECT 3100, 'SIC', 2342, 'Bras, Girdles & Allied Garments', 4, 3098 FROM dual
UNION ALL SELECT 3099, 'SIC', 2341, 'Women''s & Children''s Underwear', 4, 3098 FROM dual
UNION ALL SELECT 3107, 'SIC', 2371, 'Fur Goods', 4, 3106 FROM dual
UNION ALL SELECT 3147, 'SIC', 2499, 'Wood Products, nec', 4, 3144 FROM dual
UNION ALL SELECT 3145, 'SIC', 2491, 'Wood Preserving', 4, 3144 FROM dual
UNION ALL SELECT 3146, 'SIC', 2493, 'Reconstituted Wood Products', 4, 3144 FROM dual
UNION ALL SELECT 2285, 'SEC', 2511, 'Wood Household Furniture', 4, 2284 FROM dual
UNION ALL SELECT 3155, 'SIC', 2519, 'Household Furniture, nec', 4, 3149 FROM dual
UNION ALL SELECT 3154, 'SIC', 2517, 'Wood TV & Radio Cabinets', 4, 3149 FROM dual
UNION ALL SELECT 3153, 'SIC', 2515, 'Mattresses & Bedsprings', 4, 3149 FROM dual
UNION ALL SELECT 3152, 'SIC', 2514, 'Metal Household Furniture', 4, 3149 FROM dual
UNION ALL SELECT 3151, 'SIC', 2512, 'Upholstered Household Furniture', 4, 3149 FROM dual
UNION ALL SELECT 3150, 'SIC', 2511, 'Wood Household Furniture', 4, 3149 FROM dual
UNION ALL SELECT 2287, 'SEC', 2522, 'Office Furniture, Except Wood', 4, 2286 FROM dual
UNION ALL SELECT 3158, 'SIC', 2522, 'Office Furniture, Except Wood', 4, 3156 FROM dual
UNION ALL SELECT 3157, 'SIC', 2521, 'Wood Office Furniture', 4, 3156 FROM dual
UNION ALL SELECT 2289, 'SEC', 2531, 'Public Building & Related Furniture', 4, 2288 FROM dual
UNION ALL SELECT 3160, 'SIC', 2531, 'Public Building & Related Furniture', 4, 3159 FROM dual
UNION ALL SELECT 3163, 'SIC', 2542, 'Partitions & Fixtures, Except Wood', 4, 3161 FROM dual
UNION ALL SELECT 3165, 'SIC', 2591, 'Drapery Hardware & Blinds & Shades', 4, 3164 FROM dual
UNION ALL SELECT 3166, 'SIC', 2599, 'Furniture & Fixtures, nec', 4, 3164 FROM dual
UNION ALL SELECT 2294, 'SEC', 2611, 'Pulp Mills', 4, 2293 FROM dual
UNION ALL SELECT 3169, 'SIC', 2611, 'Pulp Mills', 4, 3168 FROM dual
UNION ALL SELECT 2296, 'SEC', 2621, 'Paper Mills', 4, 2295 FROM dual
UNION ALL SELECT 3171, 'SIC', 2621, 'Paper Mills', 4, 3170 FROM dual
UNION ALL SELECT 2298, 'SEC', 2631, 'Paperboard Mills', 4, 2297 FROM dual
UNION ALL SELECT 3173, 'SIC', 2631, 'Paperboard Mills', 4, 3172 FROM dual
UNION ALL SELECT 3175, 'SIC', 2652, 'Setup Paperboard Boxes', 4, 3174 FROM dual
UNION ALL SELECT 3179, 'SIC', 2657, 'Folding Paperboard Boxes', 4, 3174 FROM dual
UNION ALL SELECT 3177, 'SIC', 2655, 'Fiber Cans, Drums & Similar Products', 4, 3174 FROM dual
UNION ALL SELECT 3176, 'SIC', 2653, 'Corrugated & Solid Fiber Boxes', 4, 3174 FROM dual
UNION ALL SELECT 3178, 'SIC', 2656, 'Sanitary Food Containers', 4, 3174 FROM dual
UNION ALL SELECT 2301, 'SEC', 2673, 'Bags: Plastic, Laminated & Coated', 4, 2300 FROM dual
UNION ALL SELECT 3188, 'SIC', 2678, 'Stationery Products', 4, 3180 FROM dual
UNION ALL SELECT 3189, 'SIC', 2679, 'Converted Paper Products, nec', 4, 3180 FROM dual
UNION ALL SELECT 3183, 'SIC', 2673, 'Bags: Plastic, Laminated & Coated', 4, 3180 FROM dual
UNION ALL SELECT 3184, 'SIC', 2674, 'Bags: Uncoated Paper & Multiwall', 4, 3180 FROM dual
UNION ALL SELECT 3185, 'SIC', 2675, 'Die-Cut Paper & Board', 4, 3180 FROM dual
UNION ALL SELECT 3186, 'SIC', 2676, 'Sanitary Paper Products', 4, 3180 FROM dual
UNION ALL SELECT 3187, 'SIC', 2677, 'Envelopes', 4, 3180 FROM dual
UNION ALL SELECT 3197, 'SIC', 2732, 'Book Printing', 4, 3195 FROM dual
UNION ALL SELECT 3196, 'SIC', 2731, 'Book Publishing', 4, 3195 FROM dual
UNION ALL SELECT 2311, 'SEC', 2741, 'Miscellaneous Publishing', 4, 2310 FROM dual
UNION ALL SELECT 3199, 'SIC', 2741, 'Miscellaneous Publishing', 4, 3198 FROM dual
UNION ALL SELECT 3202, 'SIC', 2754, 'Commercial Printing, Gravure', 4, 3200 FROM dual
UNION ALL SELECT 3203, 'SIC', 2759, 'Commercial Printing, nec', 4, 3200 FROM dual
UNION ALL SELECT 3201, 'SIC', 2752, 'Commercial Printing, Lithographic', 4, 3200 FROM dual
UNION ALL SELECT 2314, 'SEC', 2761, 'Manifold Business Forms', 4, 2313 FROM dual
UNION ALL SELECT 3205, 'SIC', 2761, 'Manifold Business Forms', 4, 3204 FROM dual
UNION ALL SELECT 2316, 'SEC', 2771, 'Greeting Cards', 4, 2315 FROM dual
UNION ALL SELECT 3207, 'SIC', 2771, 'Greeting Cards', 4, 3206 FROM dual
UNION ALL SELECT 3209, 'SIC', 2782, 'Blankbooks & Looseleaf Binders', 4, 3208 FROM dual
UNION ALL SELECT 3210, 'SIC', 2789, 'Bookbinding & Related Work', 4, 3208 FROM dual
UNION ALL SELECT 3212, 'SIC', 2791, 'Typesetting', 4, 3211 FROM dual
UNION ALL SELECT 3213, 'SIC', 2796, 'Platemaking Services', 4, 3211 FROM dual
UNION ALL SELECT 3216, 'SIC', 2812, 'Alkalies & Chlorine', 4, 3215 FROM dual
UNION ALL SELECT 3217, 'SIC', 2813, 'Industrial Gases', 4, 3215 FROM dual
UNION ALL SELECT 3218, 'SIC', 2816, 'Inorganic Pigments', 4, 3215 FROM dual
UNION ALL SELECT 3219, 'SIC', 2819, 'Industrial Inorganic Chemicals, nec', 4, 3215 FROM dual
UNION ALL SELECT 2322, 'SEC', 2821, 'Plastics Materials & Resins', 4, 2321 FROM dual
UNION ALL SELECT 3222, 'SIC', 2822, 'Synthetic Rubber', 4, 3220 FROM dual
UNION ALL SELECT 3221, 'SIC', 2821, 'Plastics Materials & Resins', 4, 3220 FROM dual
UNION ALL SELECT 3223, 'SIC', 2823, 'Cellulosic Manmade Fibers', 4, 3220 FROM dual
UNION ALL SELECT 3248, 'SIC', 2892, 'Explosives', 4, 3246 FROM dual
UNION ALL SELECT 3259, 'SIC', 2992, 'Lubricating Oils & Greases', 4, 3258 FROM dual
UNION ALL SELECT 2344, 'SEC', 3011, 'Tires & Inner Tubes', 4, 2343 FROM dual
UNION ALL SELECT 3263, 'SIC', 3011, 'Tires & Inner Tubes', 4, 3262 FROM dual
UNION ALL SELECT 2346, 'SEC', 3021, 'Rubber & Plastics Footwear', 4, 2345 FROM dual
UNION ALL SELECT 3265, 'SIC', 3021, 'Rubber & Plastics Footwear', 4, 3264 FROM dual
UNION ALL SELECT 3267, 'SIC', 3052, 'Rubber & Plastics Hose & Belting', 4, 3266 FROM dual
UNION ALL SELECT 3268, 'SIC', 3053, 'Gaskets, Packing & Sealing Devices', 4, 3266 FROM dual
UNION ALL SELECT 3271, 'SIC', 3069, 'Fabricated Rubber Products, nec', 4, 3269 FROM dual
UNION ALL SELECT 3270, 'SIC', 3061, 'Mechanical Rubber Goods', 4, 3269 FROM dual
UNION ALL SELECT 2352, 'SEC', 3089, 'Plastics Products, nec', 4, 2349 FROM dual
UNION ALL SELECT 2350, 'SEC', 3081, 'Unsupported Plastics Film & Sheet', 4, 2349 FROM dual
UNION ALL SELECT 2351, 'SEC', 3086, 'Plastics Foam Products', 4, 2349 FROM dual
UNION ALL SELECT 3276, 'SIC', 3084, 'Plastics Pipe', 4, 3272 FROM dual
UNION ALL SELECT 3278, 'SIC', 3086, 'Plastics Foam Products', 4, 3272 FROM dual
UNION ALL SELECT 3279, 'SIC', 3087, 'Custom Compound Purchased Resins', 4, 3272 FROM dual
UNION ALL SELECT 3280, 'SIC', 3088, 'Plastics Plumbing Fixtures', 4, 3272 FROM dual
UNION ALL SELECT 3281, 'SIC', 3089, 'Plastics Products, nec', 4, 3272 FROM dual
UNION ALL SELECT 3273, 'SIC', 3081, 'Unsupported Plastics Film & Sheet', 4, 3272 FROM dual
UNION ALL SELECT 3277, 'SIC', 3085, 'Plastics Bottles', 4, 3272 FROM dual
UNION ALL SELECT 3274, 'SIC', 3082, 'Unsupported Plastics Profile Shapes', 4, 3272 FROM dual
UNION ALL SELECT 3275, 'SIC', 3083, 'Laminated Plastics Plate, Sheet, and Profile Shapes', 4, 3272 FROM dual
UNION ALL SELECT 3284, 'SIC', 3111, 'Leather Tanning & Finishing', 4, 3283 FROM dual
UNION ALL SELECT 3286, 'SIC', 3131, 'Footwear Cut Stock', 4, 3285 FROM dual
UNION ALL SELECT 3291, 'SIC', 3149, 'Footwear, Except Rubber, nec', 4, 3287 FROM dual
UNION ALL SELECT 3289, 'SIC', 3143, 'Men''s Footwear, Except Athletic', 4, 3287 FROM dual
UNION ALL SELECT 3290, 'SIC', 3144, 'Women''s Footwear, Except Athletic', 4, 3287 FROM dual
UNION ALL SELECT 3288, 'SIC', 3142, 'House Slippers', 4, 3287 FROM dual
UNION ALL SELECT 3293, 'SIC', 3151, 'Leather Gloves & Mittens', 4, 3292 FROM dual
UNION ALL SELECT 3298, 'SIC', 3172, 'Personal Leather Goods, nec', 4, 3296 FROM dual
UNION ALL SELECT 3297, 'SIC', 3171, 'Women''s Handbags & Purses', 4, 3296 FROM dual
UNION ALL SELECT 3300, 'SIC', 3199, 'Leather Goods, nec', 4, 3299 FROM dual
UNION ALL SELECT 2357, 'SEC', 3211, 'Flat Glass', 4, 2356 FROM dual
UNION ALL SELECT 3303, 'SIC', 3211, 'Flat Glass', 4, 3302 FROM dual
UNION ALL SELECT 2359, 'SEC', 3221, 'Glass Containers', 4, 2358 FROM dual
UNION ALL SELECT 3305, 'SIC', 3221, 'Glass Containers', 4, 3304 FROM dual
UNION ALL SELECT 3306, 'SIC', 3229, 'Pressed & Blown Glass, nec', 4, 3304 FROM dual
UNION ALL SELECT 2361, 'SEC', 3231, 'Products of Purchased Glass', 4, 2360 FROM dual
UNION ALL SELECT 3308, 'SIC', 3231, 'Products of Purchased Glass', 4, 3307 FROM dual
UNION ALL SELECT 3321, 'SIC', 3269, 'Pottery Products, nec', 4, 3316 FROM dual
UNION ALL SELECT 3319, 'SIC', 3263, 'Semivitreous Table & Kitchenware', 4, 3316 FROM dual
UNION ALL SELECT 3317, 'SIC', 3261, 'Vitreous Plumbing Fixtures', 4, 3316 FROM dual
UNION ALL SELECT 3318, 'SIC', 3262, 'Vitreous China Table & Kitchenware', 4, 3316 FROM dual
UNION ALL SELECT 3320, 'SIC', 3264, 'Porcelain Electrical Supplies', 4, 3316 FROM dual
UNION ALL SELECT 2367, 'SEC', 3272, 'Concrete Products, nec', 4, 2366 FROM dual
UNION ALL SELECT 3325, 'SIC', 3273, 'Ready-Mixed Concrete', 4, 3322 FROM dual
UNION ALL SELECT 3324, 'SIC', 3272, 'Concrete Products, nec', 4, 3322 FROM dual
UNION ALL SELECT 3326, 'SIC', 3274, 'Lime', 4, 3322 FROM dual
UNION ALL SELECT 3327, 'SIC', 3275, 'Gypsum Products', 4, 3322 FROM dual
UNION ALL SELECT 3323, 'SIC', 3271, 'Concrete Block & Brick', 4, 3322 FROM dual
UNION ALL SELECT 2369, 'SEC', 3281, 'Cut Stone & Stone Products', 4, 2368 FROM dual
UNION ALL SELECT 3329, 'SIC', 3281, 'Cut Stone & Stone Products', 4, 3328 FROM dual
UNION ALL SELECT 3336, 'SIC', 3299, 'Nonmetallic Mineral Products, nec', 4, 3330 FROM dual
UNION ALL SELECT 3331, 'SIC', 3291, 'Abrasive Products', 4, 3330 FROM dual
UNION ALL SELECT 3332, 'SIC', 3292, 'Asbestos Products', 4, 3330 FROM dual
UNION ALL SELECT 3333, 'SIC', 3295, 'Minerals, Ground or Treated', 4, 3330 FROM dual
UNION ALL SELECT 3334, 'SIC', 3296, 'Mineral Wool', 4, 3330 FROM dual
UNION ALL SELECT 3335, 'SIC', 3297, 'Nonclay Refractories', 4, 3330 FROM dual
UNION ALL SELECT 2373, 'SEC', 3312, 'Blast Furnaces & Steel Mills', 4, 2372 FROM dual
UNION ALL SELECT 2374, 'SEC', 3317, 'Steel Pipe & Tubes', 4, 2372 FROM dual
UNION ALL SELECT 3341, 'SIC', 3315, 'Steel Wire & Related Products', 4, 3338 FROM dual
UNION ALL SELECT 3342, 'SIC', 3316, 'Cold Finishing of Steel Shapes', 4, 3338 FROM dual
UNION ALL SELECT 3343, 'SIC', 3317, 'Steel Pipe & Tubes', 4, 3338 FROM dual
UNION ALL SELECT 3339, 'SIC', 3312, 'Blast Furnaces & Steel Mills', 4, 3338 FROM dual
UNION ALL SELECT 3340, 'SIC', 3313, 'Electrometallurgical Products', 4, 3338 FROM dual
UNION ALL SELECT 3347, 'SIC', 3324, 'Steel Investment Foundries', 4, 3344 FROM dual
UNION ALL SELECT 3348, 'SIC', 3325, 'Steel Foundries, nec', 4, 3344 FROM dual
UNION ALL SELECT 3346, 'SIC', 3322, 'Malleable Iron Foundries', 4, 3344 FROM dual
UNION ALL SELECT 3345, 'SIC', 3321, 'Gray & Ductile Iron Foundries', 4, 3344 FROM dual
UNION ALL SELECT 2377, 'SEC', 3334, 'Primary Aluminum', 4, 2376 FROM dual
UNION ALL SELECT 2387, 'SEC', 3412, 'Metal Barrels, Drums & Pails', 4, 2385 FROM dual
UNION ALL SELECT 2386, 'SEC', 3411, 'Metal Cans', 4, 2385 FROM dual
UNION ALL SELECT 3374, 'SIC', 3412, 'Metal Barrels, Drums & Pails', 4, 3372 FROM dual
UNION ALL SELECT 3373, 'SIC', 3411, 'Metal Cans', 4, 3372 FROM dual
UNION ALL SELECT 3377, 'SIC', 3423, 'Hand & Edge Tools, nec', 4, 3375 FROM dual
UNION ALL SELECT 3379, 'SIC', 3429, 'Hardware, nec', 4, 3375 FROM dual
UNION ALL SELECT 3378, 'SIC', 3425, 'Saw Blades & Handsaws', 4, 3375 FROM dual
UNION ALL SELECT 3376, 'SIC', 3421, 'Cutlery', 4, 3375 FROM dual
UNION ALL SELECT 2390, 'SEC', 3433, 'Heating Equipment, Except Electric', 4, 2389 FROM dual
UNION ALL SELECT 3382, 'SIC', 3432, 'Plumbing Fixture Fittings & Trim', 4, 3380 FROM dual
UNION ALL SELECT 3383, 'SIC', 3433, 'Heating Equipment, Except Electric', 4, 3380 FROM dual
UNION ALL SELECT 3381, 'SIC', 3431, 'Metal Sanitary Ware', 4, 3380 FROM dual
UNION ALL SELECT 2392, 'SEC', 3442, 'Metal Doors, Sash & Trim', 4, 2391 FROM dual
UNION ALL SELECT 2393, 'SEC', 3443, 'Fabricated Plate Work (Boiler Shops)', 4, 2391 FROM dual
UNION ALL SELECT 2395, 'SEC', 3448, 'Prefabricated Metal Buildings', 4, 2391 FROM dual
UNION ALL SELECT 2394, 'SEC', 3444, 'Sheet Metal Work', 4, 2391 FROM dual
UNION ALL SELECT 3391, 'SIC', 3449, 'Miscellaneous Metal Work', 4, 3384 FROM dual
UNION ALL SELECT 3390, 'SIC', 3448, 'Prefabricated Metal Buildings', 4, 3384 FROM dual
UNION ALL SELECT 3389, 'SIC', 3446, 'Architectural Metal Work', 4, 3384 FROM dual
UNION ALL SELECT 3388, 'SIC', 3444, 'Sheet Metal Work', 4, 3384 FROM dual
UNION ALL SELECT 3387, 'SIC', 3443, 'Fabricated Plate Work (Boiler Shops)', 4, 3384 FROM dual
UNION ALL SELECT 3386, 'SIC', 3442, 'Metal Doors, Sash & Trim', 4, 3384 FROM dual
UNION ALL SELECT 3432, 'SIC', 3536, 'Hoists, Cranes & Monorails', 4, 3426 FROM dual
UNION ALL SELECT 3431, 'SIC', 3535, 'Conveyors & Conveying Equipment', 4, 3426 FROM dual
UNION ALL SELECT 3430, 'SIC', 3534, 'Elevators & Moving Stairways', 4, 3426 FROM dual
UNION ALL SELECT 3429, 'SIC', 3533, 'Oil & Gas Field Machinery', 4, 3426 FROM dual
UNION ALL SELECT 3428, 'SIC', 3532, 'Mining Machinery', 4, 3426 FROM dual
UNION ALL SELECT 2414, 'SEC', 3541, 'Machine Tools, Metal Cutting Types', 4, 2413 FROM dual
UNION ALL SELECT 3443, 'SIC', 3549, 'Metalworking Machinery, nec', 4, 3434 FROM dual
UNION ALL SELECT 3435, 'SIC', 3541, 'Machine Tools, Metal Cutting Types', 4, 3434 FROM dual
UNION ALL SELECT 3436, 'SIC', 3542, 'Machine Tools, Metal Forming Types', 4, 3434 FROM dual
UNION ALL SELECT 3437, 'SIC', 3543, 'Industrial Patterns', 4, 3434 FROM dual
UNION ALL SELECT 3438, 'SIC', 3544, 'Special Dies, Tools, Jigs & Fixtures', 4, 3434 FROM dual
UNION ALL SELECT 3439, 'SIC', 3545, 'Machine Tool Accessories', 4, 3434 FROM dual
UNION ALL SELECT 3440, 'SIC', 3546, 'Power-Driven Handtools', 4, 3434 FROM dual
UNION ALL SELECT 3441, 'SIC', 3547, 'Rolling Mill Machinery', 4, 3434 FROM dual
UNION ALL SELECT 3442, 'SIC', 3548, 'Welding Apparatus', 4, 3434 FROM dual
UNION ALL SELECT 2416, 'SEC', 3555, 'Printing Trades Machinery', 4, 2415 FROM dual
UNION ALL SELECT 2417, 'SEC', 3559, 'Special Industry Machinery, nec', 4, 2415 FROM dual
UNION ALL SELECT 3449, 'SIC', 3556, 'Food Products Machinery', 4, 3444 FROM dual
UNION ALL SELECT 3483, 'SIC', 3613, 'Switchgear & Switchboard Apparatus', 4, 3481 FROM dual
UNION ALL SELECT 3482, 'SIC', 3612, 'Transformers, Except Electronic', 4, 3481 FROM dual
UNION ALL SELECT 2439, 'SEC', 3621, 'Motors & Generators', 4, 2438 FROM dual
UNION ALL SELECT 3488, 'SIC', 3629, 'Electrical Industrial Apparatus, nec', 4, 3484 FROM dual
UNION ALL SELECT 3486, 'SIC', 3624, 'Carbon & Graphite Products', 4, 3484 FROM dual
UNION ALL SELECT 3485, 'SIC', 3621, 'Motors & Generators', 4, 3484 FROM dual
UNION ALL SELECT 3487, 'SIC', 3625, 'Relays & Industrial Controls', 4, 3484 FROM dual
UNION ALL SELECT 2441, 'SEC', 3634, 'Electric Housewares & Fans', 4, 2440 FROM dual
UNION ALL SELECT 3490, 'SIC', 3631, 'Household Cooking Equipment', 4, 3489 FROM dual
UNION ALL SELECT 3491, 'SIC', 3632, 'Household Refrigerators & Freezers', 4, 3489 FROM dual
UNION ALL SELECT 3492, 'SIC', 3633, 'Household Laundry Equipment', 4, 3489 FROM dual
UNION ALL SELECT 3493, 'SIC', 3634, 'Electric Housewares & Fans', 4, 3489 FROM dual
UNION ALL SELECT 3494, 'SIC', 3635, 'Household Vacuum Cleaners', 4, 3489 FROM dual
UNION ALL SELECT 3495, 'SIC', 3639, 'Household Appliances, nec', 4, 3489 FROM dual
UNION ALL SELECT 3502, 'SIC', 3647, 'Vehicular Lighting Equipment', 4, 3496 FROM dual
UNION ALL SELECT 3536, 'SIC', 3728, 'Aircraft Parts & Equipment, nec', 4, 3533 FROM dual
UNION ALL SELECT 3535, 'SIC', 3724, 'Aircraft Engines & Engine Parts', 4, 3533 FROM dual
UNION ALL SELECT 3539, 'SIC', 3732, 'Boat Building & Repairing', 4, 3537 FROM dual
UNION ALL SELECT 3538, 'SIC', 3731, 'Ship Building & Repairing', 4, 3537 FROM dual
UNION ALL SELECT 2471, 'SEC', 3743, 'Railroad Equipment', 4, 2470 FROM dual
UNION ALL SELECT 3541, 'SIC', 3743, 'Railroad Equipment', 4, 3540 FROM dual
UNION ALL SELECT 2473, 'SEC', 3751, 'Motorcycles, Bicycles & Parts', 4, 2472 FROM dual
UNION ALL SELECT 3543, 'SIC', 3751, 'Motorcycles, Bicycles & Parts', 4, 3542 FROM dual
UNION ALL SELECT 3546, 'SIC', 3764, 'Space Propulsion Units & Parts', 4, 3544 FROM dual
UNION ALL SELECT 3547, 'SIC', 3769, 'Space Vehicle Equipment, nec', 4, 3544 FROM dual
UNION ALL SELECT 3545, 'SIC', 3761, 'Guided Missiles & Space Vehicles', 4, 3544 FROM dual
UNION ALL SELECT 3549, 'SIC', 3792, 'Travel Trailers & Campers', 4, 3548 FROM dual
UNION ALL SELECT 3550, 'SIC', 3795, 'Tanks & Tank Components', 4, 3548 FROM dual
UNION ALL SELECT 3551, 'SIC', 3799, 'Transportation Equipment, nec', 4, 3548 FROM dual
UNION ALL SELECT 2478, 'SEC', 3812, 'Search & Navigation Equipment', 4, 2477 FROM dual
UNION ALL SELECT 3554, 'SIC', 3812, 'Search & Navigation Equipment', 4, 3553 FROM dual
UNION ALL SELECT 2482, 'SEC', 3823, 'Process Control Instruments', 4, 2479 FROM dual
UNION ALL SELECT 2480, 'SEC', 3821, 'Laboratory Apparatus & Furniture', 4, 2479 FROM dual
UNION ALL SELECT 2481, 'SEC', 3822, 'Environmental Controls', 4, 2479 FROM dual
UNION ALL SELECT 2483, 'SEC', 3824, 'Fluid Meters & Counting Devices', 4, 2479 FROM dual
UNION ALL SELECT 2484, 'SEC', 3825, 'Instruments to Measure Electricity', 4, 2479 FROM dual
UNION ALL SELECT 2485, 'SEC', 3826, 'Analytical Instruments', 4, 2479 FROM dual
UNION ALL SELECT 2486, 'SEC', 3827, 'Optical Instruments & Lenses', 4, 2479 FROM dual
UNION ALL SELECT 2487, 'SEC', 3829, 'Measuring & Controlling Devices, nec', 4, 2479 FROM dual
UNION ALL SELECT 2506, 'SEC', 3942, 'Dolls & Stuffed Toys', 4, 2505 FROM dual
UNION ALL SELECT 2508, 'SEC', 3949, 'Sporting & Athletic Goods, nec', 4, 2505 FROM dual
UNION ALL SELECT 2507, 'SEC', 3944, 'Games, Toys & Children''s Vehicles', 4, 2505 FROM dual
UNION ALL SELECT 3586, 'SIC', 3949, 'Sporting & Athletic Goods, nec', 4, 3583 FROM dual
UNION ALL SELECT 3584, 'SIC', 3942, 'Dolls & Stuffed Toys', 4, 3583 FROM dual
UNION ALL SELECT 3585, 'SIC', 3944, 'Games, Toys & Children''s Vehicles', 4, 3583 FROM dual
UNION ALL SELECT 3591, 'SIC', 3955, 'Carbon Paper & Inked Ribbons', 4, 3587 FROM dual
UNION ALL SELECT 3590, 'SIC', 3953, 'Marking Devices', 4, 3587 FROM dual
UNION ALL SELECT 3589, 'SIC', 3952, 'Lead Pencils & Art Goods', 4, 3587 FROM dual
UNION ALL SELECT 3588, 'SIC', 3951, 'Pens & Mechanical Pencils', 4, 3587 FROM dual
UNION ALL SELECT 3593, 'SIC', 3961, 'Costume Jewelry', 4, 3592 FROM dual
UNION ALL SELECT 3594, 'SIC', 3965, 'Fasteners, Buttons, Needles & Pins', 4, 3592 FROM dual
UNION ALL SELECT 3597, 'SIC', 3993, 'Signs & Advertising Specialties', 4, 3595 FROM dual
UNION ALL SELECT 3600, 'SIC', 3999, 'Manufacturing Industries, nec', 4, 3595 FROM dual
UNION ALL SELECT 3599, 'SIC', 3996, 'Hard Surface Floor Coverings, nec', 4, 3595 FROM dual
UNION ALL SELECT 3598, 'SIC', 3995, 'Burial Caskets', 4, 3595 FROM dual
UNION ALL SELECT 3596, 'SIC', 3991, 'Brooms & Brushes', 4, 3595 FROM dual
UNION ALL SELECT 2514, 'SEC', 4011, 'Railroads Line-Haul Operating', 4, 2513 FROM dual
UNION ALL SELECT 2515, 'SEC', 4013, 'Switching & Terminal Devices', 4, 2513 FROM dual
UNION ALL SELECT 3604, 'SIC', 4013, 'Switching & Terminal Devices', 4, 3602 FROM dual
UNION ALL SELECT 3603, 'SIC', 4011, 'Railroads Line-Haul Operating', 4, 3602 FROM dual
UNION ALL SELECT 3608, 'SIC', 4119, 'Local Passenger Transportation, nec', 4, 3606 FROM dual
UNION ALL SELECT 3607, 'SIC', 4111, 'Local & Suburban Transit', 4, 3606 FROM dual
UNION ALL SELECT 3610, 'SIC', 4121, 'Taxicabs', 4, 3609 FROM dual
UNION ALL SELECT 3612, 'SIC', 4131, 'Intercity & Rural Bus Transportation', 4, 3611 FROM dual
UNION ALL SELECT 3615, 'SIC', 4142, 'Bus Charter Service, Except Local', 4, 3613 FROM dual
UNION ALL SELECT 3614, 'SIC', 4141, 'Local Bus Charter Service', 4, 3613 FROM dual
UNION ALL SELECT 3617, 'SIC', 4151, 'School Buses', 4, 3616 FROM dual
UNION ALL SELECT 3619, 'SIC', 4173, 'Bus Terminal & Service Facilities', 4, 3618 FROM dual
UNION ALL SELECT 2519, 'SEC', 4213, 'Trucking, Except Local', 4, 2518 FROM dual
UNION ALL SELECT 3622, 'SIC', 4212, 'Local Trucking, Without Storage', 4, 3621 FROM dual
UNION ALL SELECT 3624, 'SIC', 4214, 'Local Trucking With Storage', 4, 3621 FROM dual
UNION ALL SELECT 3625, 'SIC', 4215, 'Courier Service, Except by Air', 4, 3621 FROM dual
UNION ALL SELECT 3623, 'SIC', 4213, 'Trucking, Except Local', 4, 3621 FROM dual
UNION ALL SELECT 3627, 'SIC', 4221, 'Farm Product Warehousing & Storage', 4, 3626 FROM dual
UNION ALL SELECT 3630, 'SIC', 4226, 'Special Warehousing & Storage, nec', 4, 3626 FROM dual
UNION ALL SELECT 3629, 'SIC', 4225, 'General Warehousing & Storage', 4, 3626 FROM dual
UNION ALL SELECT 3628, 'SIC', 4222, 'Refrigerated Warehousing & Storage', 4, 3626 FROM dual
UNION ALL SELECT 2522, 'SEC', 4231, 'Trucking Terminal Facilities', 4, 2521 FROM dual
UNION ALL SELECT 2525, 'SEC', 4412, 'Deep Sea Foreign Transport of Freight', 4, 2524 FROM dual
UNION ALL SELECT 3638, 'SIC', 4412, 'Deep Sea Foreign Transport of Freight', 4, 3637 FROM dual
UNION ALL SELECT 3640, 'SIC', 4424, 'Deep Sea Domestic Transport of Freight', 4, 3639 FROM dual
UNION ALL SELECT 3642, 'SIC', 4432, 'Freight Transport on The Great Lakes', 4, 3641 FROM dual
UNION ALL SELECT 3644, 'SIC', 4449, 'Water Transportation of Freight, nec', 4, 3643 FROM dual
UNION ALL SELECT 3648, 'SIC', 4489, 'Water Passenger Transportation, nec', 4, 3645 FROM dual
UNION ALL SELECT 3646, 'SIC', 4481, 'Deep Sea Passenger Transportation, Except Ferry', 4, 3645 FROM dual
UNION ALL SELECT 3647, 'SIC', 4482, 'Ferries', 4, 3645 FROM dual
UNION ALL SELECT 3650, 'SIC', 4491, 'Marine Cargo Handling', 4, 3649 FROM dual
UNION ALL SELECT 3653, 'SIC', 4499, 'Water Transportation Services, nec', 4, 3649 FROM dual
UNION ALL SELECT 3652, 'SIC', 4493, 'Marinas', 4, 3649 FROM dual
UNION ALL SELECT 3651, 'SIC', 4492, 'Towing & Tugboat Service', 4, 3649 FROM dual
UNION ALL SELECT 2528, 'SEC', 4512, 'Air Transportation, Scheduled', 4, 2527 FROM dual
UNION ALL SELECT 2529, 'SEC', 4513, 'Air Courier Services', 4, 2527 FROM dual
UNION ALL SELECT 3656, 'SIC', 4512, 'Air Transportation, Scheduled', 4, 3655 FROM dual
UNION ALL SELECT 3657, 'SIC', 4513, 'Air Courier Services', 4, 3655 FROM dual
UNION ALL SELECT 2531, 'SEC', 4522, 'Air Transportation, Nonscheduled', 4, 2530 FROM dual
UNION ALL SELECT 3659, 'SIC', 4522, 'Air Transportation, Nonscheduled', 4, 3658 FROM dual
UNION ALL SELECT 2533, 'SEC', 4581, 'Airports, Flying Fields & Services', 4, 2532 FROM dual
UNION ALL SELECT 3661, 'SIC', 4581, 'Airports, Flying Fields & Services', 4, 3660 FROM dual
UNION ALL SELECT 3664, 'SIC', 4612, 'Crude Petroleum Pipelines', 4, 3663 FROM dual
UNION ALL SELECT 3666, 'SIC', 4619, 'Pipelines, nec', 4, 3663 FROM dual
UNION ALL SELECT 3665, 'SIC', 4613, 'Refined Petroleum Pipelines', 4, 3663 FROM dual
UNION ALL SELECT 3670, 'SIC', 4725, 'Tour Operators', 4, 3668 FROM dual
UNION ALL SELECT 3669, 'SIC', 4724, 'Travel Agencies', 4, 3668 FROM dual
UNION ALL SELECT 3671, 'SIC', 4729, 'Passenger Transport Arrangement, nec', 4, 3668 FROM dual
UNION ALL SELECT 2538, 'SEC', 4731, 'Freight Transportation Arrangement', 4, 2537 FROM dual
UNION ALL SELECT 3673, 'SIC', 4731, 'Freight Transportation Arrangement', 4, 3672 FROM dual
UNION ALL SELECT 3675, 'SIC', 4741, 'Rental of Railroad Cars', 4, 3674 FROM dual
UNION ALL SELECT 3679, 'SIC', 4789, 'Transportation Services, nec', 4, 3676 FROM dual
UNION ALL SELECT 3678, 'SIC', 4785, 'Inspection & Fixed Facilities', 4, 3676 FROM dual
UNION ALL SELECT 3677, 'SIC', 4783, 'Packing & Crating', 4, 3676 FROM dual
UNION ALL SELECT 2554, 'SEC', 4911, 'Electric Services', 4, 2553 FROM dual
UNION ALL SELECT 3695, 'SIC', 4911, 'Electric Services', 4, 3694 FROM dual
UNION ALL SELECT 2558, 'SEC', 4924, 'Natural Gas Distribution', 4, 2555 FROM dual
UNION ALL SELECT 2556, 'SEC', 4922, 'Natural Gas Transmission', 4, 2555 FROM dual
UNION ALL SELECT 2557, 'SEC', 4923, 'Gas Transmission & Distribution', 4, 2555 FROM dual
UNION ALL SELECT 3698, 'SIC', 4923, 'Gas Transmission & Distribution', 4, 3696 FROM dual
UNION ALL SELECT 3697, 'SIC', 4922, 'Natural Gas Transmission', 4, 3696 FROM dual
UNION ALL SELECT 3700, 'SIC', 4925, 'Gas Production and/or Distribution', 4, 3696 FROM dual
UNION ALL SELECT 3699, 'SIC', 4924, 'Natural Gas Distribution', 4, 3696 FROM dual
UNION ALL SELECT 2561, 'SEC', 4932, 'Gas & Other Services Combined', 4, 2559 FROM dual
UNION ALL SELECT 2560, 'SEC', 4931, 'Electric & Other Services Combined', 4, 2559 FROM dual
UNION ALL SELECT 3704, 'SIC', 4939, 'Combination Utilities, nec', 4, 3701 FROM dual
UNION ALL SELECT 3703, 'SIC', 4932, 'Gas & Other Services Combined', 4, 3701 FROM dual
UNION ALL SELECT 3702, 'SIC', 4931, 'Electric & Other Services Combined', 4, 3701 FROM dual
UNION ALL SELECT 2563, 'SEC', 4941, 'Water Supply', 4, 2562 FROM dual
UNION ALL SELECT 3706, 'SIC', 4941, 'Water Supply', 4, 3705 FROM dual
UNION ALL SELECT 2565, 'SEC', 4953, 'Refuse Systems', 4, 2564 FROM dual
UNION ALL SELECT 3708, 'SIC', 4952, 'Sewerage Systems', 4, 3707 FROM dual
UNION ALL SELECT 3709, 'SIC', 4953, 'Refuse Systems', 4, 3707 FROM dual
UNION ALL SELECT 3710, 'SIC', 4959, 'Sanitary Services, nec', 4, 3707 FROM dual
UNION ALL SELECT 2567, 'SEC', 4961, 'Steam & Air Conditioning Supply', 4, 2566 FROM dual
UNION ALL SELECT 3712, 'SIC', 4961, 'Steam & Air Conditioning Supply', 4, 3711 FROM dual
UNION ALL SELECT 3714, 'SIC', 4971, 'Irrigation Systems', 4, 3713 FROM dual
UNION ALL SELECT 2570, 'SEC', 5013, 'Motor Vehicle Supplies "&" New Parts', 4, 2569 FROM dual
UNION ALL SELECT 3719, 'SIC', 5014, 'Tires & Tubes', 4, 3716 FROM dual
UNION ALL SELECT 3717, 'SIC', 5012, 'Automobiles & Other Motor Vehicles', 4, 3716 FROM dual
UNION ALL SELECT 3718, 'SIC', 5013, 'Motor Vehicle Supplies "&" New Parts', 4, 3716 FROM dual
UNION ALL SELECT 3720, 'SIC', 5015, 'Motor Vehicle Parts, Used', 4, 3716 FROM dual
UNION ALL SELECT 3723, 'SIC', 5023, 'Homefurnishings', 4, 3721 FROM dual
UNION ALL SELECT 3722, 'SIC', 5021, 'Furniture', 4, 3721 FROM dual
UNION ALL SELECT 2573, 'SEC', 5031, 'Lumber, Plywood & Millwork', 4, 2572 FROM dual
UNION ALL SELECT 3728, 'SIC', 5039, 'Construction Materials, nec', 4, 3724 FROM dual
UNION ALL SELECT 3726, 'SIC', 5032, 'Brick, Stone & Related Materials', 4, 3724 FROM dual
UNION ALL SELECT 2582, 'SEC', 5065, 'Electronic Parts & Equipment', 4, 2579 FROM dual
UNION ALL SELECT 3743, 'SIC', 5065, 'Electronic Parts & Equipment', 4, 3740 FROM dual
UNION ALL SELECT 3741, 'SIC', 5063, 'Electrical Apparatus & Equipment', 4, 3740 FROM dual
UNION ALL SELECT 3742, 'SIC', 5064, 'Electrical Appliances, TV & Radios', 4, 3740 FROM dual
UNION ALL SELECT 2584, 'SEC', 5072, 'Hardware', 4, 2583 FROM dual
UNION ALL SELECT 3745, 'SIC', 5072, 'Hardware', 4, 3744 FROM dual
UNION ALL SELECT 3747, 'SIC', 5075, 'Warm Air Heating & Air-Conditioning', 4, 3744 FROM dual
UNION ALL SELECT 3746, 'SIC', 5074, 'Plumbing & Hydronic Heating Supplies', 4, 3744 FROM dual
UNION ALL SELECT 3748, 'SIC', 5078, 'Refrigeration Equipment & Supplies', 4, 3744 FROM dual
UNION ALL SELECT 2586, 'SEC', 5082, 'Construction & Mining Machinery', 4, 2585 FROM dual
UNION ALL SELECT 2587, 'SEC', 5084, 'Industrial Machinery & Equipment', 4, 2585 FROM dual
UNION ALL SELECT 3751, 'SIC', 5083, 'Farm & Garden Machinery', 4, 3749 FROM dual
UNION ALL SELECT 3752, 'SIC', 5084, 'Industrial Machinery & Equipment', 4, 3749 FROM dual
UNION ALL SELECT 3750, 'SIC', 5082, 'Construction & Mining Machinery', 4, 3749 FROM dual
UNION ALL SELECT 3755, 'SIC', 5088, 'Transportation Equipment & Supplies', 4, 3749 FROM dual
UNION ALL SELECT 3754, 'SIC', 5087, 'Service Establishment Equipment', 4, 3749 FROM dual
UNION ALL SELECT 3753, 'SIC', 5085, 'Industrial Supplies', 4, 3749 FROM dual
UNION ALL SELECT 2590, 'SEC', 5099, 'Durable Goods, nec', 4, 2588 FROM dual
UNION ALL SELECT 2589, 'SEC', 5094, 'Jewelry & Precious Stones', 4, 2588 FROM dual
UNION ALL SELECT 3761, 'SIC', 5099, 'Durable Goods, nec', 4, 3756 FROM dual
UNION ALL SELECT 3843, 'SIC', 5531, 'Auto & Home Supply Stores', 4, 3842 FROM dual
UNION ALL SELECT 3845, 'SIC', 5541, 'Gasoline Service Stations', 4, 3844 FROM dual
UNION ALL SELECT 3847, 'SIC', 5551, 'Boat Dealers', 4, 3846 FROM dual
UNION ALL SELECT 3849, 'SIC', 5561, 'Recreational Vehicle Dealers', 4, 3848 FROM dual
UNION ALL SELECT 3851, 'SIC', 5571, 'Motorcycle Dealers', 4, 3850 FROM dual
UNION ALL SELECT 3853, 'SIC', 5599, 'Automotive Dealers, nec', 4, 3852 FROM dual
UNION ALL SELECT 3856, 'SIC', 5611, 'Men''s & Boys'' Clothing Stores', 4, 3855 FROM dual
UNION ALL SELECT 2625, 'SEC', 5621, 'Women''s Clothing Stores', 4, 2624 FROM dual
UNION ALL SELECT 3858, 'SIC', 5621, 'Women''s Clothing Stores', 4, 3857 FROM dual
UNION ALL SELECT 3860, 'SIC', 5632, 'Women''s Accessory & Specialty Stores', 4, 3859 FROM dual
UNION ALL SELECT 3862, 'SIC', 5641, 'Children''s & Infants'' Wear Stores', 4, 3861 FROM dual
UNION ALL SELECT 2627, 'SEC', 5651, 'Family Clothing Stores', 4, 2626 FROM dual
UNION ALL SELECT 3864, 'SIC', 5651, 'Family Clothing Stores', 4, 3863 FROM dual
UNION ALL SELECT 2629, 'SEC', 5661, 'Shoe Stores', 4, 2628 FROM dual
UNION ALL SELECT 3866, 'SIC', 5661, 'Shoe Stores', 4, 3865 FROM dual
UNION ALL SELECT 3868, 'SIC', 5699, 'Miscellaneous Apparel & Accessory Stores', 4, 3867 FROM dual
UNION ALL SELECT 2632, 'SEC', 5712, 'Furniture Stores', 4, 2631 FROM dual
UNION ALL SELECT 3871, 'SIC', 5712, 'Furniture Stores', 4, 3870 FROM dual
UNION ALL SELECT 3874, 'SIC', 5719, 'Miscellaneous Homefurnishings Stores', 4, 3870 FROM dual
UNION ALL SELECT 3873, 'SIC', 5714, 'Drapery & Upholstery Stores', 4, 3870 FROM dual
UNION ALL SELECT 3872, 'SIC', 5713, 'Floor Covering Stores', 4, 3870 FROM dual
UNION ALL SELECT 3876, 'SIC', 5722, 'Household Appliance Stores', 4, 3875 FROM dual
UNION ALL SELECT 2634, 'SEC', 5731, 'Radio, TV & Electronic Stores', 4, 2633 FROM dual
UNION ALL SELECT 2636, 'SEC', 5735, 'Record & Prerecorded Tape Stores', 4, 2633 FROM dual
UNION ALL SELECT 2635, 'SEC', 5734, 'Computer & Software Stores', 4, 2633 FROM dual
UNION ALL SELECT 3880, 'SIC', 5735, 'Record & Prerecorded Tape Stores', 4, 3877 FROM dual
UNION ALL SELECT 3878, 'SIC', 5731, 'Radio, TV & Electronic Stores', 4, 3877 FROM dual
UNION ALL SELECT 3881, 'SIC', 5736, 'Musical Instrument Stores', 4, 3877 FROM dual
UNION ALL SELECT 3879, 'SIC', 5734, 'Computer & Software Stores', 4, 3877 FROM dual
UNION ALL SELECT 2639, 'SEC', 5812, 'Caterers & Banquet Halls', 4, 2638 FROM dual
UNION ALL SELECT 3885, 'SIC', 5813, 'Drinking Places', 4, 3883 FROM dual
UNION ALL SELECT 3884, 'SIC', 5812, 'Caterers & Banquet Halls', 4, 3883 FROM dual
UNION ALL SELECT 2642, 'SEC', 5912, 'Drug Stores & Proprietary Stores', 4, 2641 FROM dual
UNION ALL SELECT 3888, 'SIC', 5912, 'Drug Stores & Proprietary Stores', 4, 3887 FROM dual
UNION ALL SELECT 3920, 'SIC', 6019, 'Central Reserve Depository, nec', 4, 3918 FROM dual
UNION ALL SELECT 2651, 'SEC', 6021, 'National Commercial Banks', 4, 2650 FROM dual
UNION ALL SELECT 2652, 'SEC', 6022, 'State Commercial Banks', 4, 2650 FROM dual
UNION ALL SELECT 2653, 'SEC', 6029, 'Commercial Banks, nec', 4, 2650 FROM dual
UNION ALL SELECT 3923, 'SIC', 6022, 'State Commercial Banks', 4, 3921 FROM dual
UNION ALL SELECT 3922, 'SIC', 6021, 'National Commercial Banks', 4, 3921 FROM dual
UNION ALL SELECT 3924, 'SIC', 6029, 'Commercial Banks, nec', 4, 3921 FROM dual
UNION ALL SELECT 2656, 'SEC', 6036, 'Savings Institutions, Except Federal', 4, 2654 FROM dual
UNION ALL SELECT 2655, 'SEC', 6035, 'Federal Savings Institutions', 4, 2654 FROM dual
UNION ALL SELECT 3927, 'SIC', 6036, 'Savings Institutions, Except Federal', 4, 3925 FROM dual
UNION ALL SELECT 3926, 'SIC', 6035, 'Federal Savings Institutions', 4, 3925 FROM dual
UNION ALL SELECT 3930, 'SIC', 6062, 'State Credit Unions', 4, 3928 FROM dual
UNION ALL SELECT 3929, 'SIC', 6061, 'Federal Credit Unions', 4, 3928 FROM dual
UNION ALL SELECT 3932, 'SIC', 6081, 'Foreign Bank & Branches & Agencies', 4, 3931 FROM dual
UNION ALL SELECT 3933, 'SIC', 6082, 'Foreign Trade & International Banks', 4, 3931 FROM dual
UNION ALL SELECT 2658, 'SEC', 6099, 'Functions Related to Deposit Banking', 4, 2657 FROM dual
UNION ALL SELECT 3936, 'SIC', 6099, 'Functions Related to Deposit Banking', 4, 3934 FROM dual
UNION ALL SELECT 3935, 'SIC', 6091, 'Nondeposit Trust Facilities', 4, 3934 FROM dual
UNION ALL SELECT 2661, 'SEC', 6111, 'Federal & Federally Sponsored Credit Agencies', 4, 2660 FROM dual
UNION ALL SELECT 3939, 'SIC', 6111, 'Federal & Federally Sponsored Credit Agencies', 4, 3938 FROM dual
UNION ALL SELECT 2663, 'SEC', 6141, 'Personal Credit Unions', 4, 2662 FROM dual
UNION ALL SELECT 3941, 'SIC', 6141, 'Personal Credit Unions', 4, 3940 FROM dual
UNION ALL SELECT 2665, 'SEC', 6153, 'Short Term Business Credit', 4, 2664 FROM dual
UNION ALL SELECT 2666, 'SEC', 6159, 'Misc. Business Credit Institutions', 4, 2664 FROM dual
UNION ALL SELECT 3944, 'SIC', 6159, 'Misc. Business Credit Institutions', 4, 3942 FROM dual
UNION ALL SELECT 3943, 'SIC', 6153, 'Short Term Business Credit', 4, 3942 FROM dual
UNION ALL SELECT 2669, 'SEC', 6163, 'Loan Brokers', 4, 2667 FROM dual
UNION ALL SELECT 2668, 'SEC', 6162, 'Mortgage Bankers & Correspondents', 4, 2667 FROM dual
UNION ALL SELECT 3946, 'SIC', 6162, 'Mortgage Bankers & Correspondents', 4, 3945 FROM dual
UNION ALL SELECT 3947, 'SIC', 6163, 'Loan Brokers', 4, 3945 FROM dual
UNION ALL SELECT 2672, 'SEC', 6211, 'Security Brokers & Dealers', 4, 2671 FROM dual
UNION ALL SELECT 3950, 'SIC', 6211, 'Security Brokers & Dealers', 4, 3949 FROM dual
UNION ALL SELECT 2674, 'SEC', 6221, 'Commodity Contracts Brokers, Dealers', 4, 2673 FROM dual
UNION ALL SELECT 3952, 'SIC', 6221, 'Commodity Contracts Brokers, Dealers', 4, 3951 FROM dual
UNION ALL SELECT 3954, 'SIC', 6231, 'Security & Commodity Exchanges', 4, 3953 FROM dual
UNION ALL SELECT 2676, 'SEC', 6282, 'Investment Advice', 4, 2675 FROM dual
UNION ALL SELECT 3957, 'SIC', 6289, 'Security & Commodity Services, nec', 4, 3955 FROM dual
UNION ALL SELECT 3956, 'SIC', 6282, 'Investment Advice', 4, 3955 FROM dual
UNION ALL SELECT 2679, 'SEC', 6311, 'Life Insurance', 4, 2678 FROM dual
UNION ALL SELECT 3960, 'SIC', 6311, 'Life Insurance', 4, 3959 FROM dual
UNION ALL SELECT 2682, 'SEC', 6324, 'Hospital & Medical Service Plans', 4, 2680 FROM dual
UNION ALL SELECT 2681, 'SEC', 6321, 'Accident & Health Insurance', 4, 2680 FROM dual
UNION ALL SELECT 3962, 'SIC', 6321, 'Accident & Health Insurance', 4, 3961 FROM dual
UNION ALL SELECT 3963, 'SIC', 6324, 'Hospital & Medical Service Plans', 4, 3961 FROM dual
UNION ALL SELECT 2684, 'SEC', 6331, 'Fire, Marine & Casualty Insurance', 4, 2683 FROM dual
UNION ALL SELECT 3965, 'SIC', 6331, 'Fire, Marine & Casualty Insurance', 4, 3964 FROM dual
UNION ALL SELECT 2686, 'SEC', 6351, 'Surety Insurance', 4, 2685 FROM dual
UNION ALL SELECT 3967, 'SIC', 6351, 'Surety Insurance', 4, 3966 FROM dual
UNION ALL SELECT 2688, 'SEC', 6361, 'Title Insurance', 4, 2687 FROM dual
UNION ALL SELECT 3969, 'SIC', 6361, 'Title Insurance', 4, 3968 FROM dual
UNION ALL SELECT 3971, 'SIC', 6371, 'Pension, Health & Welfare Funds', 4, 3970 FROM dual
UNION ALL SELECT 2690, 'SEC', 6399, 'Insurance Carriers, nec', 4, 2689 FROM dual
UNION ALL SELECT 3973, 'SIC', 6399, 'Insurance Carriers, nec', 4, 3972 FROM dual
UNION ALL SELECT 2693, 'SEC', 6411, 'Insurance Agents, Brokers & Service', 4, 2692 FROM dual
UNION ALL SELECT 3976, 'SIC', 6411, 'Insurance Agents, Brokers & Service', 4, 3975 FROM dual
UNION ALL SELECT 2696, 'SEC', 6512, 'Nonresidential Building Operators', 4, 2695 FROM dual
UNION ALL SELECT 2698, 'SEC', 6519, 'Real Property Lessors, nec', 4, 2695 FROM dual
UNION ALL SELECT 2697, 'SEC', 6513, 'Apartment Building Operators', 4, 2695 FROM dual
UNION ALL SELECT 3983, 'SIC', 6517, 'Railroad Property Lessors', 4, 3978 FROM dual
UNION ALL SELECT 3984, 'SIC', 6519, 'Real Property Lessors, nec', 4, 3978 FROM dual
UNION ALL SELECT 3979, 'SIC', 6512, 'Nonresidential Building Operators', 4, 3978 FROM dual
UNION ALL SELECT 3980, 'SIC', 6513, 'Apartment Building Operators', 4, 3978 FROM dual
UNION ALL SELECT 3981, 'SIC', 6514, 'Dwelling Operators, Except Apartments', 4, 3978 FROM dual
UNION ALL SELECT 3982, 'SIC', 6515, 'Mobile Home Site Operators', 4, 3978 FROM dual
UNION ALL SELECT 2700, 'SEC', 6531, 'Real Estate Agents & Managers', 4, 2699 FROM dual
UNION ALL SELECT 3986, 'SIC', 6531, 'Real Estate Agents & Managers', 4, 3985 FROM dual
UNION ALL SELECT 3988, 'SIC', 6541, 'Title Abstract Offices', 4, 3987 FROM dual
UNION ALL SELECT 2702, 'SEC', 6552, 'Subdividers & Developers, nec', 4, 2701 FROM dual
UNION ALL SELECT 3990, 'SIC', 6552, 'Subdividers & Developers, nec', 4, 3989 FROM dual
UNION ALL SELECT 3991, 'SIC', 6553, 'Cemetery Subdividers & Developers', 4, 3989 FROM dual
UNION ALL SELECT 3994, 'SIC', 6712, 'Bank Holding Companies', 4, 3993 FROM dual
UNION ALL SELECT 3995, 'SIC', 6719, 'Holding Companies, nec', 4, 3993 FROM dual
UNION ALL SELECT 4025, 'SIC', 7218, 'Industrial Launderers', 4, 4018 FROM dual
UNION ALL SELECT 4026, 'SIC', 7219, 'Laundry & Garment Services, nec', 4, 4018 FROM dual
UNION ALL SELECT 4019, 'SIC', 7211, 'Power Laundries, Family & Commercial', 4, 4018 FROM dual
UNION ALL SELECT 4028, 'SIC', 7221, 'Photographic Studios, Portrait', 4, 4027 FROM dual
UNION ALL SELECT 4030, 'SIC', 7231, 'Beauty Shops', 4, 4029 FROM dual
UNION ALL SELECT 4032, 'SIC', 7241, 'Barber Shops', 4, 4031 FROM dual
UNION ALL SELECT 4034, 'SIC', 7251, 'Shoe Repair & Shoeshine Parlors', 4, 4033 FROM dual
UNION ALL SELECT 4036, 'SIC', 7261, 'Funeral Service & Crematories', 4, 4035 FROM dual
UNION ALL SELECT 4039, 'SIC', 7299, 'Miscellaneous Personal Services, nec', 4, 4037 FROM dual
UNION ALL SELECT 4038, 'SIC', 7291, 'Tax Return Preparation Services', 4, 4037 FROM dual
UNION ALL SELECT 2715, 'SEC', 7311, 'Advertising Agencies', 4, 2714 FROM dual
UNION ALL SELECT 4042, 'SIC', 7311, 'Advertising Agencies', 4, 4041 FROM dual
UNION ALL SELECT 4044, 'SIC', 7313, 'Radio, TV Publisher Representatives', 4, 4041 FROM dual
UNION ALL SELECT 4045, 'SIC', 7319, 'Advertising, nec', 4, 4041 FROM dual
UNION ALL SELECT 4043, 'SIC', 7312, 'Outdoor Advertising Services', 4, 4041 FROM dual
UNION ALL SELECT 4047, 'SIC', 7322, 'Adjustment & Collection Services', 4, 4046 FROM dual
UNION ALL SELECT 4048, 'SIC', 7323, 'Credit Reporting Services', 4, 4046 FROM dual
UNION ALL SELECT 2718, 'SEC', 7331, 'Direct Mail Advertising Services', 4, 2717 FROM dual
UNION ALL SELECT 4050, 'SIC', 7331, 'Direct Mail Advertising Services', 4, 4049 FROM dual
UNION ALL SELECT 4051, 'SIC', 7334, 'Photocopying & Duplicating Services', 4, 4049 FROM dual
UNION ALL SELECT 4052, 'SIC', 7335, 'Commercial Photography', 4, 4049 FROM dual
UNION ALL SELECT 4053, 'SIC', 7336, 'Commercial Art & Graphic Design', 4, 4049 FROM dual
UNION ALL SELECT 4054, 'SIC', 7338, 'Secretarial & Court Reporting', 4, 4049 FROM dual
UNION ALL SELECT 4057, 'SIC', 7349, 'Building Maintenance Services, nec', 4, 4055 FROM dual
UNION ALL SELECT 4056, 'SIC', 7342, 'Disinfecting & Pest Control Services', 4, 4055 FROM dual
UNION ALL SELECT 2721, 'SEC', 7359, 'Misc. Equipment Rental & Leasing, nec', 4, 2720 FROM dual
UNION ALL SELECT 4061, 'SIC', 7359, 'Misc. Equipment Rental & Leasing, nec', 4, 4058 FROM dual
UNION ALL SELECT 4059, 'SIC', 7352, 'Medical Equipment Rental', 4, 4058 FROM dual
UNION ALL SELECT 4060, 'SIC', 7353, 'Heavy Construction Equipment Rental', 4, 4058 FROM dual
UNION ALL SELECT 2723, 'SEC', 7361, 'Employment Agencies', 4, 2722 FROM dual
UNION ALL SELECT 2724, 'SEC', 7363, 'Help Supply Services', 4, 2722 FROM dual
UNION ALL SELECT 4063, 'SIC', 7361, 'Employment Agencies', 4, 4062 FROM dual
UNION ALL SELECT 4064, 'SIC', 7363, 'Help Supply Services', 4, 4062 FROM dual
UNION ALL SELECT 2727, 'SEC', 7372, 'Prepackaged Software', 4, 2725 FROM dual
UNION ALL SELECT 2730, 'SEC', 7377, 'Computer Rental & Leasing', 4, 2725 FROM dual
UNION ALL SELECT 2729, 'SEC', 7374, 'Data Processing & Preparation', 4, 2725 FROM dual
UNION ALL SELECT 2728, 'SEC', 7373, 'Computer Integrated Systems Design', 4, 2725 FROM dual
UNION ALL SELECT 2726, 'SEC', 7371, 'Computer Programming Services', 4, 2725 FROM dual
UNION ALL SELECT 4071, 'SIC', 7376, 'Computer Facilities Management', 4, 4065 FROM dual
UNION ALL SELECT 4069, 'SIC', 7374, 'Data Processing & Preparation', 4, 4065 FROM dual
UNION ALL SELECT 4068, 'SIC', 7373, 'Computer Integrated Systems Design', 4, 4065 FROM dual
UNION ALL SELECT 4119, 'SIC', 7829, 'Motion Picture Distribution Services', 4, 4117 FROM dual
UNION ALL SELECT 4118, 'SIC', 7822, 'Motion Picture & Tape Distribution', 4, 4117 FROM dual
UNION ALL SELECT 4121, 'SIC', 7832, 'Motion Picture Theaters, Except Drive-In', 4, 4120 FROM dual
UNION ALL SELECT 4122, 'SIC', 7833, 'Drive-In Motion Picture Theaters', 4, 4120 FROM dual
UNION ALL SELECT 2747, 'SEC', 7841, 'Video Tape Rental', 4, 2746 FROM dual
UNION ALL SELECT 4124, 'SIC', 7841, 'Video Tape Rental', 4, 4123 FROM dual
UNION ALL SELECT 4127, 'SIC', 7911, 'Dance Studios, Schools & Halls', 4, 4126 FROM dual
UNION ALL SELECT 4129, 'SIC', 7922, 'Theatrical Producers & Services', 4, 4128 FROM dual
UNION ALL SELECT 4130, 'SIC', 7929, 'Entertainers & Entertainment Groups', 4, 4128 FROM dual
UNION ALL SELECT 4132, 'SIC', 7933, 'Bowling Centers', 4, 4131 FROM dual
UNION ALL SELECT 2750, 'SEC', 7948, 'Racing, Including Track Operation', 4, 2749 FROM dual
UNION ALL SELECT 4134, 'SIC', 7941, 'Sports Clubs, Managers & Promoters', 4, 4133 FROM dual
UNION ALL SELECT 4135, 'SIC', 7948, 'Racing, Including Track Operation', 4, 4133 FROM dual
UNION ALL SELECT 4137, 'SIC', 7951, 'Ski Facilities, Down Hill', 4, 4136 FROM dual
UNION ALL SELECT 4138, 'SIC', 7952, 'Ski Facilities, Cross Country', 4, 4136 FROM dual
UNION ALL SELECT 2752, 'SEC', 7997, 'Membership Sports & Recreation Clubs', 4, 2751 FROM dual
UNION ALL SELECT 4140, 'SIC', 7991, 'Physical Fitness Facilities', 4, 4139 FROM dual
UNION ALL SELECT 4145, 'SIC', 7999, 'Amusement & Recreation, nec', 4, 4139 FROM dual
UNION ALL SELECT 4144, 'SIC', 7997, 'Membership Sports & Recreation Clubs', 4, 4139 FROM dual
UNION ALL SELECT 4143, 'SIC', 7996, 'Amusement Parks', 4, 4139 FROM dual
UNION ALL SELECT 4142, 'SIC', 7993, 'Coin-Operated Amusement Devices', 4, 4139 FROM dual
UNION ALL SELECT 4141, 'SIC', 7992, 'Public Golf Courses', 4, 4139 FROM dual
UNION ALL SELECT 2755, 'SEC', 8011, 'Offices & Clinics of Medical Doctors', 4, 2754 FROM dual
UNION ALL SELECT 4148, 'SIC', 8011, 'Offices & Clinics of Medical Doctors', 4, 4147 FROM dual
UNION ALL SELECT 4150, 'SIC', 8021, 'Offices & Clinics of Dentists', 4, 4149 FROM dual
UNION ALL SELECT 4152, 'SIC', 8031, 'Offices of Osteopathic Physicians', 4, 4151 FROM dual
UNION ALL SELECT 4157, 'SIC', 8049, 'Offices of Health Practitioners, nec', 4, 4153 FROM dual
UNION ALL SELECT 4156, 'SIC', 8043, 'Offices & Clinics of Podiatrists', 4, 4153 FROM dual
UNION ALL SELECT 4180, 'SIC', 8211, 'Elementary & Secondary Schools', 4, 4179 FROM dual
UNION ALL SELECT 4182, 'SIC', 8221, 'Colleges & Universities', 4, 4181 FROM dual
UNION ALL SELECT 4183, 'SIC', 8222, 'Junior Colleges', 4, 4181 FROM dual
UNION ALL SELECT 4185, 'SIC', 8231, 'Libraries', 4, 4184 FROM dual
UNION ALL SELECT 4187, 'SIC', 8243, 'Data Processing Schools', 4, 4186 FROM dual
UNION ALL SELECT 4189, 'SIC', 8249, 'Vocational Schools, nec', 4, 4186 FROM dual
UNION ALL SELECT 4188, 'SIC', 8244, 'Business & Secretarial Schools', 4, 4186 FROM dual
UNION ALL SELECT 4191, 'SIC', 8299, 'Schools & Educational Services, nec', 4, 4190 FROM dual
UNION ALL SELECT 4194, 'SIC', 8322, 'Individual & Family Services', 4, 4193 FROM dual
UNION ALL SELECT 4196, 'SIC', 8331, 'Job Training & Related Services', 4, 4195 FROM dual
UNION ALL SELECT 2772, 'SEC', 8351, 'Child Day Care Services', 4, 2771 FROM dual
UNION ALL SELECT 4198, 'SIC', 8351, 'Child Day Care Services', 4, 4197 FROM dual
UNION ALL SELECT 4200, 'SIC', 8361, 'Residential Care', 4, 4199 FROM dual
UNION ALL SELECT 4202, 'SIC', 8399, 'Social Services, nec', 4, 4201 FROM dual
UNION ALL SELECT 4205, 'SIC', 8412, 'Museums & Art Galleries', 4, 4204 FROM dual
UNION ALL SELECT 4207, 'SIC', 8422, 'Botanical & Zoological Gardens', 4, 4206 FROM dual
UNION ALL SELECT 4210, 'SIC', 8611, 'Business Associations', 4, 4209 FROM dual
UNION ALL SELECT 4212, 'SIC', 8621, 'Professional Organizations', 4, 4211 FROM dual
UNION ALL SELECT 4214, 'SIC', 8631, 'Labor Organizations', 4, 4213 FROM dual
UNION ALL SELECT 4216, 'SIC', 8641, 'Civic & Social Organizations', 4, 4215 FROM dual
UNION ALL SELECT 4218, 'SIC', 8651, 'Political Organizations', 4, 4217 FROM dual
UNION ALL SELECT 4220, 'SIC', 8661, 'Religious Organizations', 4, 4219 FROM dual
UNION ALL SELECT 4222, 'SIC', 8699, 'Membership Organizations, nec', 4, 4221 FROM dual
UNION ALL SELECT 2776, 'SEC', 8711, 'Engineering Services', 4, 2775 FROM dual
UNION ALL SELECT 4226, 'SIC', 8712, 'Architectural Services', 4, 4224 FROM dual
UNION ALL SELECT 4227, 'SIC', 8713, 'Surveying Services', 4, 4224 FROM dual
UNION ALL SELECT 4225, 'SIC', 8711, 'Engineering Services', 4, 4224 FROM dual
UNION ALL SELECT 4229, 'SIC', 8721, 'Accounting, Auditing & Bookkeeping', 4, 4228 FROM dual
UNION ALL SELECT 2778, 'SEC', 8731, 'Commercial Physical Research', 4, 2777 FROM dual
UNION ALL SELECT 2779, 'SEC', 8734, 'Testing Laboratories', 4, 2777 FROM dual
UNION ALL SELECT 4232, 'SIC', 8732, 'Commercial Nonphysical Research', 4, 4230 FROM dual
UNION ALL SELECT 4234, 'SIC', 8734, 'Testing Laboratories', 4, 4230 FROM dual
UNION ALL SELECT 4231, 'SIC', 8731, 'Commercial Physical Research', 4, 4230 FROM dual
UNION ALL SELECT 4233, 'SIC', 8733, 'Noncommercial Research Organizations', 4, 4230 FROM dual
UNION ALL SELECT 2781, 'SEC', 8741, 'Management Services', 4, 2780 FROM dual
UNION ALL SELECT 2783, 'SEC', 8744, 'Facilities Support Services', 4, 2780 FROM dual
UNION ALL SELECT 2782, 'SEC', 8742, 'Management Consulting Services', 4, 2780 FROM dual
UNION ALL SELECT 4249, 'SIC', 9111, 'Executive Offices', 4, 4248 FROM dual
UNION ALL SELECT 4251, 'SIC', 9121, 'Legislative Bodies', 4, 4250 FROM dual
UNION ALL SELECT 4253, 'SIC', 9131, 'Executive & Legislative Combined', 4, 4252 FROM dual
UNION ALL SELECT 4255, 'SIC', 9191, 'General Government, nec', 4, 4254 FROM dual
UNION ALL SELECT 4258, 'SIC', 9211, 'Courts', 4, 4257 FROM dual
UNION ALL SELECT 4261, 'SIC', 9222, 'Legal Counsel & Prosecution', 4, 4259 FROM dual
UNION ALL SELECT 4262, 'SIC', 9223, 'Correctional Institutions', 4, 4259 FROM dual
UNION ALL SELECT 4263, 'SIC', 9224, 'Fire Protection', 4, 4259 FROM dual
UNION ALL SELECT 4264, 'SIC', 9229, 'Public Order & Safety, nec', 4, 4259 FROM dual
UNION ALL SELECT 4260, 'SIC', 9221, 'Police Protection', 4, 4259 FROM dual
UNION ALL SELECT 4267, 'SIC', 9311, 'Finance, Taxation & Monetary Policy', 4, 4266 FROM dual
UNION ALL SELECT 4270, 'SIC', 9411, 'Admin. of Educational Programs', 4, 4269 FROM dual
UNION ALL SELECT 4272, 'SIC', 9431, 'Admin. of Public Health Programs', 4, 4271 FROM dual
UNION ALL SELECT 4274, 'SIC', 9441, 'Admin. of Social & Manpower Programs', 4, 4273 FROM dual
UNION ALL SELECT 4276, 'SIC', 9451, 'Administration of Veteran''s Affairs', 4, 4275 FROM dual
UNION ALL SELECT 4280, 'SIC', 9512, 'Land, Mineral, Wildlife Conservation', 4, 4278 FROM dual
UNION ALL SELECT 4279, 'SIC', 9511, 'Air, Water & Solid Waste Management', 4, 4278 FROM dual
UNION ALL SELECT 4283, 'SIC', 9532, 'Urban & Community Development', 4, 4281 FROM dual
UNION ALL SELECT 4282, 'SIC', 9531, 'Housing Programs', 4, 4281 FROM dual
UNION ALL SELECT 4286, 'SIC', 9611, 'Administration of General Economic Programs', 4, 4285 FROM dual
UNION ALL SELECT 4288, 'SIC', 9621, 'Regulation, Admin. of Transportation', 4, 4287 FROM dual
UNION ALL SELECT 4290, 'SIC', 9631, 'Regulation, Admin. of Utilities', 4, 4289 FROM dual
UNION ALL SELECT 1321, 'NAICS', 4855, 'Charter Bus Industry', 3, 1303 FROM dual
UNION ALL SELECT 2163, 'NAICS', 92216, 'Fire Protection', 4, 2151 FROM dual
UNION ALL SELECT 3127, 'SIC', 2420, 'Sawmills & Planing Mills', 3, 3124 FROM dual
UNION ALL SELECT 3840, 'SIC', 5520, 'Used Car Dealers', 3, 3837 FROM dual
UNION ALL SELECT 2210, 'SEC', 100, 'AGRICULTURAL PRODUCTION - CROPS', 2, 2788 FROM dual
UNION ALL SELECT 2212, 'SEC', 700, 'AGRICULTURAL SERVICES', 2, 2788 FROM dual
UNION ALL SELECT 2211, 'SEC', 200, 'AGRICULTURAL PRODUCTION - LIVESTOCK', 2, 2788 FROM dual
UNION ALL SELECT 2875, 'SIC', 900, 'FISHING, HUNTING & TRAPPING', 2, 4305 FROM dual
UNION ALL SELECT 2798, 'SIC', 100, 'AGRICULTURAL PRODUCTION - CROPS', 2, 4305 FROM dual
UNION ALL SELECT 2825, 'SIC', 200, 'AGRICULTURAL PRODUCTION - LIVESTOCK', 2, 4305 FROM dual
UNION ALL SELECT 2847, 'SIC', 700, 'AGRICULTURAL SERVICES', 2, 4305 FROM dual
UNION ALL SELECT 2820, 'SIC', 180, 'Horticultural Specialties', 3, 2798 FROM dual
UNION ALL SELECT 2805, 'SIC', 130, 'Field Crops, Except Cash Grains', 3, 2798 FROM dual
UNION ALL SELECT 2811, 'SIC', 160, 'Vegetables & Melons', 3, 2798 FROM dual
UNION ALL SELECT 2823, 'SIC', 190, 'General Farms, Primarily Crop', 3, 2798 FROM dual
UNION ALL SELECT 2813, 'SIC', 170, 'Fruits & Tree Nuts', 3, 2798 FROM dual
UNION ALL SELECT 2897, 'SIC', 1081, 'Metal Mining Services', 4, 2896 FROM dual
UNION ALL SELECT 2899, 'SIC', 1094, 'Uranium Radium Vanadium Ores', 4, 2898 FROM dual
UNION ALL SELECT 2220, 'SEC', 1221, 'Bituminous Coal & Lignite - Surface', 4, 2219 FROM dual
UNION ALL SELECT 2903, 'SIC', 1221, 'Bituminous Coal & Lignite - Surface', 4, 2902 FROM dual
UNION ALL SELECT 2904, 'SIC', 1222, 'Bituminous Coal - Underground', 4, 2902 FROM dual
UNION ALL SELECT 2906, 'SIC', 1231, 'Anthracite Mining', 4, 2905 FROM dual
UNION ALL SELECT 2908, 'SIC', 1241, 'Coal Mining Services', 4, 2907 FROM dual
UNION ALL SELECT 2218, 'SEC', 1200, 'COAL MINING', 2, 2789 FROM dual
UNION ALL SELECT 2959, 'SIC', 1721, 'Painting & Paper Hanging', 4, 2958 FROM dual
UNION ALL SELECT 2239, 'SEC', 1731, 'Electrical Work', 4, 2238 FROM dual
UNION ALL SELECT 2961, 'SIC', 1731, 'Electrical Work', 4, 2960 FROM dual
UNION ALL SELECT 2965, 'SIC', 1743, 'Terrazzo, Tile, Marble, Mosaic Work', 4, 2962 FROM dual
UNION ALL SELECT 2964, 'SIC', 1742, 'Plastering, Drywall & Insulation', 4, 2962 FROM dual
UNION ALL SELECT 2963, 'SIC', 1741, 'Masonry & Other Stonework', 4, 2962 FROM dual
UNION ALL SELECT 2968, 'SIC', 1752, 'Floor Laying & Floor Work, nec', 4, 2966 FROM dual
UNION ALL SELECT 2967, 'SIC', 1751, 'Carpentry Work', 4, 2966 FROM dual
UNION ALL SELECT 2970, 'SIC', 1761, 'Roofing, Siding & Sheet Metal Work', 4, 2969 FROM dual
UNION ALL SELECT 2972, 'SIC', 1771, 'Concrete Work', 4, 2971 FROM dual
UNION ALL SELECT 2974, 'SIC', 1781, 'Water Well Drilling', 4, 2973 FROM dual
UNION ALL SELECT 2977, 'SIC', 1793, 'Glass & Glazing Work', 4, 2975 FROM dual
UNION ALL SELECT 3013, 'SIC', 2061, 'Raw Cane Sugar', 4, 3012 FROM dual
UNION ALL SELECT 3014, 'SIC', 2062, 'Cane Sugar Refining', 4, 3012 FROM dual
UNION ALL SELECT 3021, 'SIC', 2074, 'Cottonseed Oil Mills', 4, 3020 FROM dual
UNION ALL SELECT 3025, 'SIC', 2079, 'Edible Fats & Oils, nec', 4, 3020 FROM dual
UNION ALL SELECT 3024, 'SIC', 2077, 'Animal & Marine Fats & Oils', 4, 3020 FROM dual
UNION ALL SELECT 3023, 'SIC', 2076, 'Vegetable Oil Mills, nec', 4, 3020 FROM dual
UNION ALL SELECT 3022, 'SIC', 2075, 'Soybean Oil Mills', 4, 3020 FROM dual
UNION ALL SELECT 2256, 'SEC', 2086, 'Bottled & Canned Soft Drinks', 4, 2254 FROM dual
UNION ALL SELECT 2255, 'SEC', 2082, 'Malt Beverages', 4, 2254 FROM dual
UNION ALL SELECT 3029, 'SIC', 2084, 'Wines, Brandy & Brandy Spirits', 4, 3026 FROM dual
UNION ALL SELECT 3028, 'SIC', 2083, 'Malt', 4, 3026 FROM dual
UNION ALL SELECT 3027, 'SIC', 2082, 'Malt Beverages', 4, 3026 FROM dual
UNION ALL SELECT 3031, 'SIC', 2086, 'Bottled & Canned Soft Drinks', 4, 3026 FROM dual
UNION ALL SELECT 3030, 'SIC', 2085, 'Distilled & Blended Liquors', 4, 3026 FROM dual
UNION ALL SELECT 3032, 'SIC', 2087, 'Flavoring Extracts & Syrups, nec', 4, 3026 FROM dual
UNION ALL SELECT 2258, 'SEC', 2092, 'Fresh or Frozen Prepared Fish', 4, 2257 FROM dual
UNION ALL SELECT 3034, 'SIC', 2091, 'Canned & Cured Fish & Seafoods', 4, 3033 FROM dual
UNION ALL SELECT 3039, 'SIC', 2098, 'Macaroni & Spaghetti', 4, 3033 FROM dual
UNION ALL SELECT 3038, 'SIC', 2097, 'Manufactured Ice', 4, 3033 FROM dual
UNION ALL SELECT 3037, 'SIC', 2096, 'Potato Chips & Similar Snacks', 4, 3033 FROM dual
UNION ALL SELECT 3035, 'SIC', 2092, 'Fresh or Frozen Prepared Fish', 4, 3033 FROM dual
UNION ALL SELECT 3036, 'SIC', 2095, 'Roasted Coffee', 4, 3033 FROM dual
UNION ALL SELECT 2261, 'SEC', 2111, 'Cigarettes', 4, 2260 FROM dual
UNION ALL SELECT 3043, 'SIC', 2111, 'Cigarettes', 4, 3042 FROM dual
UNION ALL SELECT 3045, 'SIC', 2121, 'Cigars', 4, 3044 FROM dual
UNION ALL SELECT 3047, 'SIC', 2131, 'Chewing & Smoking Tobacco', 4, 3046 FROM dual
UNION ALL SELECT 3049, 'SIC', 2141, 'Tobacco Stemming & Redrying', 4, 3048 FROM dual
UNION ALL SELECT 2264, 'SEC', 2211, 'Broadwoven Fabric Mills, Cotton', 4, 2263 FROM dual
UNION ALL SELECT 3052, 'SIC', 2211, 'Broadwoven Fabric Mills, Cotton', 4, 3051 FROM dual
UNION ALL SELECT 2266, 'SEC', 2221, 'Broadwoven Fabric Mills, Manmade', 4, 2265 FROM dual
UNION ALL SELECT 3054, 'SIC', 2221, 'Broadwoven Fabric Mills, Manmade', 4, 3053 FROM dual
UNION ALL SELECT 3056, 'SIC', 2231, 'Broadwoven Fabric Mills, Wool', 4, 3055 FROM dual
UNION ALL SELECT 3058, 'SIC', 2241, 'Narrow Fabric Mills', 4, 3057 FROM dual
UNION ALL SELECT 2268, 'SEC', 2253, 'Knit Outerwear Mills', 4, 2267 FROM dual
UNION ALL SELECT 3062, 'SIC', 2253, 'Knit Outerwear Mills', 4, 3059 FROM dual
UNION ALL SELECT 3066, 'SIC', 2259, 'Knitting Mills, nec', 4, 3059 FROM dual
UNION ALL SELECT 3065, 'SIC', 2258, 'Lace & Warp Knit Fabric Mills', 4, 3059 FROM dual
UNION ALL SELECT 3064, 'SIC', 2257, 'Weft Knit Fabric Mills', 4, 3059 FROM dual
UNION ALL SELECT 3063, 'SIC', 2254, 'Knit Underwear Mills', 4, 3059 FROM dual
UNION ALL SELECT 3061, 'SIC', 2252, 'Hosiery, nec', 4, 3059 FROM dual
UNION ALL SELECT 3060, 'SIC', 2251, 'Women''s Hosiery, Except Socks', 4, 3059 FROM dual
UNION ALL SELECT 3068, 'SIC', 2261, 'Finishing Plants, Cotton', 4, 3067 FROM dual
UNION ALL SELECT 3069, 'SIC', 2262, 'Finishing Plants, Manmade', 4, 3067 FROM dual
UNION ALL SELECT 3070, 'SIC', 2269, 'Finishing Plants, nec', 4, 3067 FROM dual
UNION ALL SELECT 2270, 'SEC', 2273, 'Carpets & Rugs', 4, 2269 FROM dual
UNION ALL SELECT 2260, 'SEC', 2110, 'Cigarettes', 3, 2259 FROM dual
UNION ALL SELECT 3102, 'SIC', 2353, 'Hats, Caps & Millinery', 4, 3101 FROM dual
UNION ALL SELECT 3104, 'SIC', 2361, 'Girls & Children''s Dresses & Blouses', 4, 3103 FROM dual
UNION ALL SELECT 3105, 'SIC', 2369, 'Girls & Children''s Outerwear, nec', 4, 3103 FROM dual
UNION ALL SELECT 3109, 'SIC', 2381, 'Fabric Dress & Work Gloves', 4, 3108 FROM dual
UNION ALL SELECT 3113, 'SIC', 2387, 'Apparel Belts', 4, 3108 FROM dual
UNION ALL SELECT 3114, 'SIC', 2389, 'Apparel & Accessories, nec', 4, 3108 FROM dual
UNION ALL SELECT 3112, 'SIC', 2386, 'Leather & Sheep-Lined Clothing', 4, 3108 FROM dual
UNION ALL SELECT 3111, 'SIC', 2385, 'Waterproof Outerwear', 4, 3108 FROM dual
UNION ALL SELECT 3110, 'SIC', 2384, 'Robes & Dressing Gowns', 4, 3108 FROM dual
UNION ALL SELECT 3120, 'SIC', 2395, 'Pleating & Stitching', 4, 3115 FROM dual
UNION ALL SELECT 3122, 'SIC', 2397, 'Schiffli Machine Embroideries', 4, 3115 FROM dual
UNION ALL SELECT 3116, 'SIC', 2391, 'Curtains & Draperies', 4, 3115 FROM dual
UNION ALL SELECT 3123, 'SIC', 2399, 'Fabricated Textile Products, nec', 4, 3115 FROM dual
UNION ALL SELECT 3117, 'SIC', 2392, 'House Furnishings, nec', 4, 3115 FROM dual
UNION ALL SELECT 3118, 'SIC', 2393, 'Textile Bags', 4, 3115 FROM dual
UNION ALL SELECT 3119, 'SIC', 2394, 'Canvas & Related Products', 4, 3115 FROM dual
UNION ALL SELECT 3121, 'SIC', 2396, 'Automotive & Apparel Trimmings', 4, 3115 FROM dual
UNION ALL SELECT 3126, 'SIC', 2411, 'Logging', 4, 3125 FROM dual
UNION ALL SELECT 2278, 'SEC', 2421, 'Sawmills & Planing Mills, General', 4, 2277 FROM dual
UNION ALL SELECT 3128, 'SIC', 2421, 'Sawmills & Planing Mills, General', 4, 3127 FROM dual
UNION ALL SELECT 3130, 'SIC', 2429, 'Special Product Sawmills, nec', 4, 3127 FROM dual
UNION ALL SELECT 3129, 'SIC', 2426, 'Hardwood Dimension & Flooring Mills', 4, 3127 FROM dual
UNION ALL SELECT 3134, 'SIC', 2435, 'Hardwood Veneer & Plywood', 4, 3131 FROM dual
UNION ALL SELECT 3133, 'SIC', 2434, 'Wood Kitchen Cabinets', 4, 3131 FROM dual
UNION ALL SELECT 3132, 'SIC', 2431, 'Millwork', 4, 3131 FROM dual
UNION ALL SELECT 3136, 'SIC', 2439, 'Structural Wood Members, nec', 4, 3131 FROM dual
UNION ALL SELECT 3135, 'SIC', 2436, 'Softwood Veneer & Plywood', 4, 3131 FROM dual
UNION ALL SELECT 3139, 'SIC', 2448, 'Wood Pallets & Skids', 4, 3137 FROM dual
UNION ALL SELECT 3138, 'SIC', 2441, 'Nailed Wood Boxes & Shook', 4, 3137 FROM dual
UNION ALL SELECT 3140, 'SIC', 2449, 'Wood Containers, nec', 4, 3137 FROM dual
UNION ALL SELECT 2281, 'SEC', 2451, 'Mobile Homes', 4, 2280 FROM dual
UNION ALL SELECT 2282, 'SEC', 2452, 'Prefabricated Wood Buildings', 4, 2280 FROM dual
UNION ALL SELECT 3142, 'SIC', 2451, 'Mobile Homes', 4, 3141 FROM dual
UNION ALL SELECT 3143, 'SIC', 2452, 'Prefabricated Wood Buildings', 4, 3141 FROM dual
UNION ALL SELECT 3106, 'SIC', 2370, 'Fur Goods', 3, 3083 FROM dual
UNION ALL SELECT 3181, 'SIC', 2671, 'Paper Coated & Laminated, Packaging', 4, 3180 FROM dual
UNION ALL SELECT 3182, 'SIC', 2672, 'Paper Coated & Laminated, nec', 4, 3180 FROM dual
UNION ALL SELECT 2304, 'SEC', 2711, 'Newspapers', 4, 2303 FROM dual
UNION ALL SELECT 3192, 'SIC', 2711, 'Newspapers', 4, 3191 FROM dual
UNION ALL SELECT 2306, 'SEC', 2721, 'Periodicals', 4, 2305 FROM dual
UNION ALL SELECT 3194, 'SIC', 2721, 'Periodicals', 4, 3193 FROM dual
UNION ALL SELECT 2309, 'SEC', 2732, 'Book Printing', 4, 2307 FROM dual
UNION ALL SELECT 2308, 'SEC', 2731, 'Book Publishing', 4, 2307 FROM dual
UNION ALL SELECT 3224, 'SIC', 2824, 'Organic Fibers, Noncellulosic', 4, 3220 FROM dual
UNION ALL SELECT 2326, 'SEC', 2835, 'Diagnostic Substances', 4, 2323 FROM dual
UNION ALL SELECT 2327, 'SEC', 2836, 'Biological Products, Except Diagnostic', 4, 2323 FROM dual
UNION ALL SELECT 2324, 'SEC', 2833, 'Medicinals & Botanicals', 4, 2323 FROM dual
UNION ALL SELECT 2325, 'SEC', 2834, 'Pharmaceutical Preparations', 4, 2323 FROM dual
UNION ALL SELECT 3229, 'SIC', 2836, 'Biological Products, Except Diagnostic', 4, 3225 FROM dual
UNION ALL SELECT 3226, 'SIC', 2833, 'Medicinals & Botanicals', 4, 3225 FROM dual
UNION ALL SELECT 3228, 'SIC', 2835, 'Diagnostic Substances', 4, 3225 FROM dual
UNION ALL SELECT 3227, 'SIC', 2834, 'Pharmaceutical Preparations', 4, 3225 FROM dual
UNION ALL SELECT 2329, 'SEC', 2842, 'Polishes & Sanitation Goods', 4, 2328 FROM dual
UNION ALL SELECT 2330, 'SEC', 2844, 'Perfumes & Cosmetics -Toilet Preparations', 4, 2328 FROM dual
UNION ALL SELECT 3231, 'SIC', 2841, 'Soap & Other Detergents', 4, 3230 FROM dual
UNION ALL SELECT 3232, 'SIC', 2842, 'Polishes & Sanitation Goods', 4, 3230 FROM dual
UNION ALL SELECT 3233, 'SIC', 2843, 'Surface Active Agents', 4, 3230 FROM dual
UNION ALL SELECT 3234, 'SIC', 2844, 'Perfumes & Cosmetics -Toilet Preparations', 4, 3230 FROM dual
UNION ALL SELECT 2332, 'SEC', 2851, 'Paints & Allied Products', 4, 2331 FROM dual
UNION ALL SELECT 3236, 'SIC', 2851, 'Paints & Allied Products', 4, 3235 FROM dual
UNION ALL SELECT 3240, 'SIC', 2869, 'Industrial Organic Chemicals, nec', 4, 3237 FROM dual
UNION ALL SELECT 3238, 'SIC', 2861, 'Gum & Wood Chemicals', 4, 3237 FROM dual
UNION ALL SELECT 3239, 'SIC', 2865, 'Cyclic Crudes & Intermediates', 4, 3237 FROM dual
UNION ALL SELECT 3244, 'SIC', 2875, 'Fertilizers, Mixing Only', 4, 3241 FROM dual
UNION ALL SELECT 3245, 'SIC', 2879, 'Agricultural Chemicals, nec', 4, 3241 FROM dual
UNION ALL SELECT 3243, 'SIC', 2874, 'Phosphatic Fertilizers', 4, 3241 FROM dual
UNION ALL SELECT 3242, 'SIC', 2873, 'Nitrogenous Fertilizers', 4, 3241 FROM dual
UNION ALL SELECT 2336, 'SEC', 2891, 'Adhesives & Sealants', 4, 2335 FROM dual
UNION ALL SELECT 3251, 'SIC', 2899, 'Chemical Preparations, nec', 4, 3246 FROM dual
UNION ALL SELECT 3247, 'SIC', 2891, 'Adhesives & Sealants', 4, 3246 FROM dual
UNION ALL SELECT 3249, 'SIC', 2893, 'Printing Ink', 4, 3246 FROM dual;

INSERT INTO "industry" (industry_id, industry_classification, industry_code, industry_description, depth, parent_id) 
          SELECT 3250, 'SIC', 2895, 'Carbon Black', 4, 3246 FROM dual
UNION ALL SELECT 2339, 'SEC', 2911, 'Petroleum Refining', 4, 2338 FROM dual
UNION ALL SELECT 3254, 'SIC', 2911, 'Petroleum Refining', 4, 3253 FROM dual
UNION ALL SELECT 3256, 'SIC', 2951, 'Asphalt Paving Mixtures & Blocks', 4, 3255 FROM dual
UNION ALL SELECT 3257, 'SIC', 2952, 'Asphalt Felts & Coatings', 4, 3255 FROM dual
UNION ALL SELECT 3260, 'SIC', 2999, 'Petroleum & Coal Products, nec', 4, 3258 FROM dual
UNION ALL SELECT 2363, 'SEC', 3241, 'Cement, Hydraulic', 4, 2362 FROM dual
UNION ALL SELECT 3310, 'SIC', 3241, 'Cement, Hydraulic', 4, 3309 FROM dual
UNION ALL SELECT 3313, 'SIC', 3253, 'Ceramic Wall & Floor Tile', 4, 3311 FROM dual
UNION ALL SELECT 3314, 'SIC', 3255, 'Clay Refectories', 4, 3311 FROM dual
UNION ALL SELECT 3315, 'SIC', 3259, 'Structural Clay Products, nec', 4, 3311 FROM dual
UNION ALL SELECT 3312, 'SIC', 3251, 'Brick & Structural Clay Tile', 4, 3311 FROM dual
UNION ALL SELECT 3351, 'SIC', 3334, 'Primary Aluminum', 4, 3349 FROM dual
UNION ALL SELECT 3352, 'SIC', 3339, 'Primary Nonferrous Metals, nec', 4, 3349 FROM dual
UNION ALL SELECT 3350, 'SIC', 3331, 'Primary Copper', 4, 3349 FROM dual
UNION ALL SELECT 2379, 'SEC', 3341, 'Secondary Nonferrous Metals', 4, 2378 FROM dual
UNION ALL SELECT 3354, 'SIC', 3341, 'Secondary Nonferrous Metals', 4, 3353 FROM dual
UNION ALL SELECT 2381, 'SEC', 3357, 'Nonferrous Wiredrawing & Insulating', 4, 2380 FROM dual
UNION ALL SELECT 3357, 'SIC', 3353, 'Aluminum Sheet, Plate & Foil', 4, 3355 FROM dual
UNION ALL SELECT 3359, 'SIC', 3355, 'Aluminum Rolling & Drawing, nec', 4, 3355 FROM dual
UNION ALL SELECT 3361, 'SIC', 3357, 'Nonferrous Wiredrawing & Insulating', 4, 3355 FROM dual
UNION ALL SELECT 3360, 'SIC', 3356, 'Nonferrous Rolling & Drawing, nec', 4, 3355 FROM dual
UNION ALL SELECT 3356, 'SIC', 3351, 'Copper Rolling & Drawing', 4, 3355 FROM dual
UNION ALL SELECT 3358, 'SIC', 3354, 'Aluminum Extruded Products', 4, 3355 FROM dual
UNION ALL SELECT 3365, 'SIC', 3365, 'Aluminum Foundries', 4, 3362 FROM dual
UNION ALL SELECT 3367, 'SIC', 3369, 'Nonferrous Foundries, nec', 4, 3362 FROM dual
UNION ALL SELECT 3364, 'SIC', 3364, 'Nonferrous Die Castings, Except Aluminum', 4, 3362 FROM dual
UNION ALL SELECT 3366, 'SIC', 3366, 'Copper Foundries', 4, 3362 FROM dual
UNION ALL SELECT 3363, 'SIC', 3363, 'Aluminum Die Castings', 4, 3362 FROM dual
UNION ALL SELECT 3370, 'SIC', 3399, 'Primary Metal Products, nec', 4, 3368 FROM dual
UNION ALL SELECT 3369, 'SIC', 3398, 'Metal Heat Treating', 4, 3368 FROM dual
UNION ALL SELECT 2400, 'SEC', 3470, 'Metal Services, nec', 3, 2384 FROM dual
UNION ALL SELECT 3385, 'SIC', 3441, 'Fabricated Structural Metal', 4, 3384 FROM dual
UNION ALL SELECT 2398, 'SEC', 3452, 'Bolts, Nuts, Rivets & Washers', 4, 2396 FROM dual
UNION ALL SELECT 2397, 'SEC', 3451, 'Screw Machine Products', 4, 2396 FROM dual
UNION ALL SELECT 3394, 'SIC', 3452, 'Bolts, Nuts, Rivets & Washers', 4, 3392 FROM dual
UNION ALL SELECT 3393, 'SIC', 3451, 'Screw Machine Products', 4, 3392 FROM dual
UNION ALL SELECT 3399, 'SIC', 3466, 'Crowns & Closures', 4, 3395 FROM dual
UNION ALL SELECT 3400, 'SIC', 3469, 'Metal Stampings, nec', 4, 3395 FROM dual
UNION ALL SELECT 3396, 'SIC', 3462, 'Iron & Steel Forgings', 4, 3395 FROM dual
UNION ALL SELECT 3397, 'SIC', 3463, 'Nonferrous Forgings', 4, 3395 FROM dual
UNION ALL SELECT 3398, 'SIC', 3465, 'Automotive Stampings', 4, 3395 FROM dual
UNION ALL SELECT 3402, 'SIC', 3471, 'Plating & Polishing', 4, 3401 FROM dual
UNION ALL SELECT 3403, 'SIC', 3479, 'Metal Coating & Allied Services', 4, 3401 FROM dual
UNION ALL SELECT 3406, 'SIC', 3483, 'Ammunition, Except for Small Arms, nec', 4, 3404 FROM dual
UNION ALL SELECT 3405, 'SIC', 3482, 'Small Arms Ammunition', 4, 3404 FROM dual
UNION ALL SELECT 3408, 'SIC', 3489, 'Ordnance & Accessories, nec', 4, 3404 FROM dual
UNION ALL SELECT 3407, 'SIC', 3484, 'Small Arms', 4, 3404 FROM dual
UNION ALL SELECT 3412, 'SIC', 3493, 'Steel Spring, Except Wire', 4, 3409 FROM dual
UNION ALL SELECT 3418, 'SIC', 3499, 'Fabricated Metal Products, nec', 4, 3409 FROM dual
UNION ALL SELECT 3417, 'SIC', 3498, 'Fabricated Pipe & Fittings', 4, 3409 FROM dual
UNION ALL SELECT 3416, 'SIC', 3497, 'Metal Foil & Leaf', 4, 3409 FROM dual
UNION ALL SELECT 3415, 'SIC', 3496, 'Miscellaneous Fabricated Wire Products', 4, 3409 FROM dual
UNION ALL SELECT 3414, 'SIC', 3495, 'Wire Springs', 4, 3409 FROM dual
UNION ALL SELECT 3413, 'SIC', 3494, 'Valves & Pipe Fittings, nec', 4, 3409 FROM dual
UNION ALL SELECT 3411, 'SIC', 3492, 'Fluid Power Valves & Hose Fittings', 4, 3409 FROM dual
UNION ALL SELECT 3410, 'SIC', 3491, 'Industrial Valves', 4, 3409 FROM dual
UNION ALL SELECT 3422, 'SIC', 3519, 'Internal Combustion Engines, nec', 4, 3420 FROM dual
UNION ALL SELECT 3421, 'SIC', 3511, 'Turbines & Turbine Generator Sets', 4, 3420 FROM dual
UNION ALL SELECT 2407, 'SEC', 3524, 'Lawn & Garden Equipment', 4, 2405 FROM dual
UNION ALL SELECT 2406, 'SEC', 3523, 'Farm Machinery & Equipment', 4, 2405 FROM dual
UNION ALL SELECT 3425, 'SIC', 3524, 'Lawn & Garden Equipment', 4, 3423 FROM dual
UNION ALL SELECT 3424, 'SIC', 3523, 'Farm Machinery & Equipment', 4, 3423 FROM dual
UNION ALL SELECT 2412, 'SEC', 3537, 'Industrial Trucks & Tractors', 4, 2408 FROM dual
UNION ALL SELECT 2409, 'SEC', 3531, 'Construction Machinery', 4, 2408 FROM dual
UNION ALL SELECT 2410, 'SEC', 3532, 'Mining Machinery', 4, 2408 FROM dual
UNION ALL SELECT 2411, 'SEC', 3533, 'Oil & Gas Field Machinery', 4, 2408 FROM dual
UNION ALL SELECT 3427, 'SIC', 3531, 'Construction Machinery', 4, 3426 FROM dual
UNION ALL SELECT 3433, 'SIC', 3537, 'Industrial Trucks & Tractors', 4, 3426 FROM dual
UNION ALL SELECT 3450, 'SIC', 3559, 'Special Industry Machinery, nec', 4, 3444 FROM dual
UNION ALL SELECT 3445, 'SIC', 3552, 'Textile Machinery', 4, 3444 FROM dual
UNION ALL SELECT 3447, 'SIC', 3554, 'Paper Industries Machinery', 4, 3444 FROM dual
UNION ALL SELECT 3446, 'SIC', 3553, 'Woodworking Machinery', 4, 3444 FROM dual
UNION ALL SELECT 3448, 'SIC', 3555, 'Printing Trades Machinery', 4, 3444 FROM dual
UNION ALL SELECT 2420, 'SEC', 3562, 'Ball & Roller Bearings', 4, 2418 FROM dual
UNION ALL SELECT 2421, 'SEC', 3564, 'Blowers & Fans', 4, 2418 FROM dual
UNION ALL SELECT 2422, 'SEC', 3567, 'Industrial Furnaces & Ovens', 4, 2418 FROM dual
UNION ALL SELECT 2423, 'SEC', 3569, 'General Industrial Machinery, nec', 4, 2418 FROM dual
UNION ALL SELECT 2419, 'SEC', 3561, 'Pumps & Pumping Equipment', 4, 2418 FROM dual
UNION ALL SELECT 3452, 'SIC', 3561, 'Pumps & Pumping Equipment', 4, 3451 FROM dual
UNION ALL SELECT 3454, 'SIC', 3563, 'Air & Gas Compressors', 4, 3451 FROM dual
UNION ALL SELECT 3455, 'SIC', 3564, 'Blowers & Fans', 4, 3451 FROM dual
UNION ALL SELECT 3456, 'SIC', 3565, 'Packaging Machinery', 4, 3451 FROM dual
UNION ALL SELECT 3457, 'SIC', 3566, 'Speed Changers, Drives & Gears', 4, 3451 FROM dual
UNION ALL SELECT 3458, 'SIC', 3567, 'Industrial Furnaces & Ovens', 4, 3451 FROM dual
UNION ALL SELECT 3459, 'SIC', 3568, 'Power Transmission Equipment, nec', 4, 3451 FROM dual
UNION ALL SELECT 3460, 'SIC', 3569, 'General Industrial Machinery, nec', 4, 3451 FROM dual
UNION ALL SELECT 3453, 'SIC', 3562, 'Ball & Roller Bearings', 4, 3451 FROM dual
UNION ALL SELECT 2426, 'SEC', 3572, 'Computer Storage Devices', 4, 2424 FROM dual
UNION ALL SELECT 2428, 'SEC', 3577, 'Computer Peripheral Equipment, nec', 4, 2424 FROM dual
UNION ALL SELECT 2429, 'SEC', 3578, 'Calculating & Accounting Equipment', 4, 2424 FROM dual
UNION ALL SELECT 2430, 'SEC', 3579, 'Office Machines, nec', 4, 2424 FROM dual
UNION ALL SELECT 2425, 'SEC', 3571, 'Electronic Computers', 4, 2424 FROM dual
UNION ALL SELECT 2427, 'SEC', 3575, 'Computer Terminals', 4, 2424 FROM dual
UNION ALL SELECT 3462, 'SIC', 3571, 'Electronic Computers', 4, 3461 FROM dual
UNION ALL SELECT 3463, 'SIC', 3572, 'Computer Storage Devices', 4, 3461 FROM dual
UNION ALL SELECT 3464, 'SIC', 3575, 'Computer Terminals', 4, 3461 FROM dual
UNION ALL SELECT 3465, 'SIC', 3577, 'Computer Peripheral Equipment, nec', 4, 3461 FROM dual
UNION ALL SELECT 3466, 'SIC', 3578, 'Calculating & Accounting Equipment', 4, 3461 FROM dual
UNION ALL SELECT 3467, 'SIC', 3579, 'Office Machines, nec', 4, 3461 FROM dual
UNION ALL SELECT 2432, 'SEC', 3585, 'Refrigeration & Heating Equipment', 4, 2431 FROM dual
UNION ALL SELECT 3473, 'SIC', 3589, 'Service Industry Machinery, nec', 4, 3468 FROM dual
UNION ALL SELECT 3472, 'SIC', 3586, 'Measuring & Dispensing Pumps', 4, 3468 FROM dual
UNION ALL SELECT 3469, 'SIC', 3581, 'Automatic Vending Machines', 4, 3468 FROM dual
UNION ALL SELECT 3470, 'SIC', 3582, 'Commercial Laundry Equipment', 4, 3468 FROM dual
UNION ALL SELECT 3471, 'SIC', 3585, 'Refrigeration & Heating Equipment', 4, 3468 FROM dual
UNION ALL SELECT 3475, 'SIC', 3592, 'Carburetors, Piston Rings & Valves', 4, 3474 FROM dual
UNION ALL SELECT 3479, 'SIC', 3599, 'Industrial Machinery, nec', 4, 3474 FROM dual
UNION ALL SELECT 3476, 'SIC', 3593, 'Fluid Power Cylinders & Actuators', 4, 3474 FROM dual
UNION ALL SELECT 3477, 'SIC', 3594, 'Fluid Power Pumps & Motors', 4, 3474 FROM dual
UNION ALL SELECT 3478, 'SIC', 3596, 'Scales & Balances, Except Laboratory', 4, 3474 FROM dual
UNION ALL SELECT 2437, 'SEC', 3613, 'Switchgear & Switchboard Apparatus', 4, 2435 FROM dual
UNION ALL SELECT 2436, 'SEC', 3612, 'Transformers, Except Electronic', 4, 2435 FROM dual
UNION ALL SELECT 3503, 'SIC', 3648, 'Lighting Equipment, nec', 4, 3496 FROM dual
UNION ALL SELECT 3501, 'SIC', 3646, 'Commercial Lighting Fixtures', 4, 3496 FROM dual
UNION ALL SELECT 3500, 'SIC', 3645, 'Residential Lighting Fixtures', 4, 3496 FROM dual
UNION ALL SELECT 3499, 'SIC', 3644, 'Noncurrent-Carrying Wiring Devices', 4, 3496 FROM dual
UNION ALL SELECT 3498, 'SIC', 3643, 'Current-Carrying Wiring Devices', 4, 3496 FROM dual
UNION ALL SELECT 3497, 'SIC', 3641, 'Electric Lamps', 4, 3496 FROM dual
UNION ALL SELECT 2444, 'SEC', 3651, 'Household Audio & Video Equipment', 4, 2443 FROM dual
UNION ALL SELECT 2445, 'SEC', 3652, 'Prerecorded Records & Tapes', 4, 2443 FROM dual
UNION ALL SELECT 3506, 'SIC', 3652, 'Prerecorded Records & Tapes', 4, 3504 FROM dual
UNION ALL SELECT 3505, 'SIC', 3651, 'Household Audio & Video Equipment', 4, 3504 FROM dual
UNION ALL SELECT 2449, 'SEC', 3669, 'Communications Equipment, nec', 4, 2446 FROM dual
UNION ALL SELECT 2447, 'SEC', 3661, 'Telephone & Telegraph Apparatus', 4, 2446 FROM dual
UNION ALL SELECT 2448, 'SEC', 3663, 'Radio & TV Communications Equipment', 4, 2446 FROM dual
UNION ALL SELECT 3508, 'SIC', 3661, 'Telephone & Telegraph Apparatus', 4, 3507 FROM dual
UNION ALL SELECT 3509, 'SIC', 3663, 'Radio & TV Communications Equipment', 4, 3507 FROM dual
UNION ALL SELECT 3510, 'SIC', 3669, 'Communications Equipment, nec', 4, 3507 FROM dual
UNION ALL SELECT 2451, 'SEC', 3672, 'Printed Circuit Boards', 4, 2450 FROM dual
UNION ALL SELECT 2453, 'SEC', 3677, 'Electronic Coils & Transformers', 4, 2450 FROM dual
UNION ALL SELECT 2452, 'SEC', 3674, 'Semiconductors & Related Devices', 4, 2450 FROM dual
UNION ALL SELECT 2454, 'SEC', 3678, 'Electronic Connectors', 4, 2450 FROM dual
UNION ALL SELECT 2455, 'SEC', 3679, 'Electronic Components, nec', 4, 2450 FROM dual
UNION ALL SELECT 3516, 'SIC', 3676, 'Electronic Resistors', 4, 3511 FROM dual
UNION ALL SELECT 3519, 'SIC', 3679, 'Electronic Components, nec', 4, 3511 FROM dual
UNION ALL SELECT 3518, 'SIC', 3678, 'Electronic Connectors', 4, 3511 FROM dual
UNION ALL SELECT 3517, 'SIC', 3677, 'Electronic Coils & Transformers', 4, 3511 FROM dual
UNION ALL SELECT 3515, 'SIC', 3675, 'Electronic Capacitors', 4, 3511 FROM dual
UNION ALL SELECT 3514, 'SIC', 3674, 'Semiconductors & Related Devices', 4, 3511 FROM dual
UNION ALL SELECT 3513, 'SIC', 3672, 'Printed Circuit Boards', 4, 3511 FROM dual
UNION ALL SELECT 3512, 'SIC', 3671, 'Electron Tubes', 4, 3511 FROM dual
UNION ALL SELECT 2457, 'SEC', 3695, 'Magnetic & Optical Recording Media', 4, 2456 FROM dual
UNION ALL SELECT 3521, 'SIC', 3691, 'Storage Batteries', 4, 3520 FROM dual
UNION ALL SELECT 3522, 'SIC', 3692, 'Primary Batteries, Dry & Wet', 4, 3520 FROM dual
UNION ALL SELECT 3523, 'SIC', 3694, 'Engine Electrical Equipment', 4, 3520 FROM dual
UNION ALL SELECT 3524, 'SIC', 3695, 'Magnetic & Optical Recording Media', 4, 3520 FROM dual
UNION ALL SELECT 3525, 'SIC', 3699, 'Electrical Equipment & Supplies, nec', 4, 3520 FROM dual
UNION ALL SELECT 2460, 'SEC', 3711, 'Motor Vehicles & Car Bodies', 4, 2459 FROM dual
UNION ALL SELECT 2464, 'SEC', 3716, 'Motor Homes', 4, 2459 FROM dual
UNION ALL SELECT 2463, 'SEC', 3715, 'Truck Trailers', 4, 2459 FROM dual
UNION ALL SELECT 2462, 'SEC', 3714, 'Motor Vehicle Parts & Accessories', 4, 2459 FROM dual
UNION ALL SELECT 2461, 'SEC', 3713, 'Truck & Bus Bodies', 4, 2459 FROM dual
UNION ALL SELECT 3532, 'SIC', 3716, 'Motor Homes', 4, 3527 FROM dual
UNION ALL SELECT 3528, 'SIC', 3711, 'Motor Vehicles & Car Bodies', 4, 3527 FROM dual
UNION ALL SELECT 3530, 'SIC', 3714, 'Motor Vehicle Parts & Accessories', 4, 3527 FROM dual
UNION ALL SELECT 3531, 'SIC', 3715, 'Truck Trailers', 4, 3527 FROM dual
UNION ALL SELECT 3529, 'SIC', 3713, 'Truck & Bus Bodies', 4, 3527 FROM dual
UNION ALL SELECT 2467, 'SEC', 3724, 'Aircraft Engines & Engine Parts', 4, 2465 FROM dual
UNION ALL SELECT 2468, 'SEC', 3728, 'Aircraft Parts & Equipment, nec', 4, 2465 FROM dual
UNION ALL SELECT 2466, 'SEC', 3721, 'Aircraft', 4, 2465 FROM dual
UNION ALL SELECT 3534, 'SIC', 3721, 'Aircraft', 4, 3533 FROM dual
UNION ALL SELECT 3559, 'SIC', 3824, 'Fluid Meters & Counting Devices', 4, 3555 FROM dual
UNION ALL SELECT 3560, 'SIC', 3825, 'Instruments to Measure Electricity', 4, 3555 FROM dual
UNION ALL SELECT 3561, 'SIC', 3826, 'Analytical Instruments', 4, 3555 FROM dual
UNION ALL SELECT 3563, 'SIC', 3829, 'Measuring & Controlling Devices, nec', 4, 3555 FROM dual
UNION ALL SELECT 3562, 'SIC', 3827, 'Optical Instruments & Lenses', 4, 3555 FROM dual
UNION ALL SELECT 3556, 'SIC', 3821, 'Laboratory Apparatus & Furniture', 4, 3555 FROM dual
UNION ALL SELECT 3557, 'SIC', 3822, 'Environmental Controls', 4, 3555 FROM dual
UNION ALL SELECT 3558, 'SIC', 3823, 'Process Control Instruments', 4, 3555 FROM dual
UNION ALL SELECT 2490, 'SEC', 3842, 'Surgical Appliances & Supplies', 4, 2488 FROM dual
UNION ALL SELECT 2491, 'SEC', 3843, 'Dental Equipment & Supplies', 4, 2488 FROM dual
UNION ALL SELECT 2492, 'SEC', 3844, 'X-Ray Apparatus & Tubes', 4, 2488 FROM dual
UNION ALL SELECT 2493, 'SEC', 3845, 'Electromedical Equipment', 4, 2488 FROM dual
UNION ALL SELECT 2489, 'SEC', 3841, 'Surgical & Medical Instruments', 4, 2488 FROM dual
UNION ALL SELECT 3565, 'SIC', 3841, 'Surgical & Medical Instruments', 4, 3564 FROM dual
UNION ALL SELECT 3566, 'SIC', 3842, 'Surgical Appliances & Supplies', 4, 3564 FROM dual
UNION ALL SELECT 3567, 'SIC', 3843, 'Dental Equipment & Supplies', 4, 3564 FROM dual
UNION ALL SELECT 3568, 'SIC', 3844, 'X-Ray Apparatus & Tubes', 4, 3564 FROM dual
UNION ALL SELECT 3569, 'SIC', 3845, 'Electromedical Equipment', 4, 3564 FROM dual
UNION ALL SELECT 2495, 'SEC', 3851, 'Ophthalmic Goods', 4, 2494 FROM dual
UNION ALL SELECT 3571, 'SIC', 3851, 'Ophthalmic Goods', 4, 3570 FROM dual
UNION ALL SELECT 2497, 'SEC', 3861, 'Photographic Equipment & Supplies', 4, 2496 FROM dual
UNION ALL SELECT 3573, 'SIC', 3861, 'Photographic Equipment & Supplies', 4, 3572 FROM dual
UNION ALL SELECT 2499, 'SEC', 3873, 'Watches, Clocks, Watchcases & Parts', 4, 2498 FROM dual
UNION ALL SELECT 3575, 'SIC', 3873, 'Watches, Clocks, Watchcases & Parts', 4, 3574 FROM dual
UNION ALL SELECT 2502, 'SEC', 3911, 'Jewelry, Precious Metal', 4, 2501 FROM dual
UNION ALL SELECT 3580, 'SIC', 3915, 'Jewelers'' Materials & Lapidary Work', 4, 3577 FROM dual
UNION ALL SELECT 3579, 'SIC', 3914, 'Silverware & Plated Ware', 4, 3577 FROM dual
UNION ALL SELECT 3578, 'SIC', 3911, 'Jewelry, Precious Metal', 4, 3577 FROM dual
UNION ALL SELECT 2504, 'SEC', 3931, 'Musical Instruments', 4, 2503 FROM dual
UNION ALL SELECT 3582, 'SIC', 3931, 'Musical Instruments', 4, 3581 FROM dual
UNION ALL SELECT 3632, 'SIC', 4231, 'Trucking Terminal Facilities', 4, 3631 FROM dual
UNION ALL SELECT 3635, 'SIC', 4311, 'US Postal Service', 4, 3634 FROM dual
UNION ALL SELECT 2541, 'SEC', 4812, 'Radiotelephone Communications', 4, 2540 FROM dual
UNION ALL SELECT 2542, 'SEC', 4813, 'Telephone Communications, Except Radio', 4, 2540 FROM dual
UNION ALL SELECT 3682, 'SIC', 4812, 'Radiotelephone Communications', 4, 3681 FROM dual
UNION ALL SELECT 3683, 'SIC', 4813, 'Telephone Communications, Except Radio', 4, 3681 FROM dual
UNION ALL SELECT 2544, 'SEC', 4822, 'Telegraph & Other Communications', 4, 2543 FROM dual
UNION ALL SELECT 3685, 'SIC', 4822, 'Telegraph & Other Communications', 4, 3684 FROM dual
UNION ALL SELECT 2546, 'SEC', 4832, 'Radio Broadcasting Stations', 4, 2545 FROM dual
UNION ALL SELECT 2547, 'SEC', 4833, 'Television Broadcasting Stations', 4, 2545 FROM dual
UNION ALL SELECT 3688, 'SIC', 4833, 'Television Broadcasting Stations', 4, 3686 FROM dual
UNION ALL SELECT 3687, 'SIC', 4832, 'Radio Broadcasting Stations', 4, 3686 FROM dual
UNION ALL SELECT 2549, 'SEC', 4841, 'Cable & Other Pay TV Services', 4, 2548 FROM dual
UNION ALL SELECT 3690, 'SIC', 4841, 'Cable & Other Pay TV Services', 4, 3689 FROM dual
UNION ALL SELECT 2551, 'SEC', 4899, 'Communications Services, nec', 4, 2550 FROM dual
UNION ALL SELECT 3692, 'SIC', 4899, 'Communications Services, nec', 4, 3691 FROM dual
UNION ALL SELECT 3725, 'SIC', 5031, 'Lumber, Plywood & Millwork', 4, 3724 FROM dual
UNION ALL SELECT 3727, 'SIC', 5033, 'Roofing, Siding & Insulation', 4, 3724 FROM dual
UNION ALL SELECT 2576, 'SEC', 5047, 'Medical & Hospital Equipment', 4, 2574 FROM dual
UNION ALL SELECT 2575, 'SEC', 5045, 'Computers, Peripherals & Software', 4, 2574 FROM dual
UNION ALL SELECT 3732, 'SIC', 5045, 'Computers, Peripherals & Software', 4, 3729 FROM dual
UNION ALL SELECT 3736, 'SIC', 5049, 'Professional Equipment, nec', 4, 3729 FROM dual
UNION ALL SELECT 3735, 'SIC', 5048, 'Ophthalmic Goods', 4, 3729 FROM dual
UNION ALL SELECT 3734, 'SIC', 5047, 'Medical & Hospital Equipment', 4, 3729 FROM dual
UNION ALL SELECT 3733, 'SIC', 5046, 'Commercial Equipment, nec', 4, 3729 FROM dual
UNION ALL SELECT 3731, 'SIC', 5044, 'Office Equipment', 4, 3729 FROM dual
UNION ALL SELECT 3730, 'SIC', 5043, 'Photographic Equipment & Supplies', 4, 3729 FROM dual
UNION ALL SELECT 2578, 'SEC', 5051, 'Metals Service Centers & Offices', 4, 2577 FROM dual
UNION ALL SELECT 3738, 'SIC', 5051, 'Metals Service Centers & Offices', 4, 3737 FROM dual
UNION ALL SELECT 3739, 'SIC', 5052, 'Coal & Other Minerals & Ores', 4, 3737 FROM dual
UNION ALL SELECT 2580, 'SEC', 5063, 'Electrical Apparatus & Equipment', 4, 2579 FROM dual
UNION ALL SELECT 2581, 'SEC', 5064, 'Electrical Appliances, TV & Radios', 4, 2579 FROM dual
UNION ALL SELECT 3758, 'SIC', 5092, 'Toys & Hobby Goods & Supplies', 4, 3756 FROM dual
UNION ALL SELECT 3759, 'SIC', 5093, 'Scrap & Waste Materials', 4, 3756 FROM dual
UNION ALL SELECT 3760, 'SIC', 5094, 'Jewelry & Precious Stones', 4, 3756 FROM dual
UNION ALL SELECT 3757, 'SIC', 5091, 'Sporting & Recreational Goods', 4, 3756 FROM dual
UNION ALL SELECT 3766, 'SIC', 5113, 'Industrial & Personal Service Paper', 4, 3763 FROM dual
UNION ALL SELECT 3764, 'SIC', 5111, 'Printing & Writing Paper', 4, 3763 FROM dual
UNION ALL SELECT 3765, 'SIC', 5112, 'Stationery & Office Supplies', 4, 3763 FROM dual
UNION ALL SELECT 2594, 'SEC', 5122, 'Drugs, Proprietaries & Sundries', 4, 2593 FROM dual
UNION ALL SELECT 3768, 'SIC', 5122, 'Drugs, Proprietaries & Sundries', 4, 3767 FROM dual
UNION ALL SELECT 3770, 'SIC', 5131, 'Piece Goods & Notations', 4, 3769 FROM dual
UNION ALL SELECT 3773, 'SIC', 5139, 'Footwear', 4, 3769 FROM dual
UNION ALL SELECT 3772, 'SIC', 5137, 'Women''s & Children''s Clothing', 4, 3769 FROM dual
UNION ALL SELECT 3771, 'SIC', 5136, 'Men''s & Boys'' Clothing', 4, 3769 FROM dual
UNION ALL SELECT 2597, 'SEC', 5141, 'Groceries, General Line', 4, 2596 FROM dual
UNION ALL SELECT 3775, 'SIC', 5141, 'Groceries, General Line', 4, 3774 FROM dual
UNION ALL SELECT 3776, 'SIC', 5142, 'Packaged Frozen Foods', 4, 3774 FROM dual
UNION ALL SELECT 3777, 'SIC', 5143, 'Dairy Products, Except Dried or Canned', 4, 3774 FROM dual
UNION ALL SELECT 3778, 'SIC', 5144, 'Poultry & Poultry Products', 4, 3774 FROM dual
UNION ALL SELECT 3779, 'SIC', 5145, 'Confectionery', 4, 3774 FROM dual
UNION ALL SELECT 3780, 'SIC', 5146, 'Fish & Seafoods', 4, 3774 FROM dual
UNION ALL SELECT 3781, 'SIC', 5147, 'Meat & Meat Products', 4, 3774 FROM dual
UNION ALL SELECT 3783, 'SIC', 5149, 'Groceries & Related Products, nec', 4, 3774 FROM dual
UNION ALL SELECT 3782, 'SIC', 5148, 'Fresh Fruits & Vegetables', 4, 3774 FROM dual
UNION ALL SELECT 3786, 'SIC', 5154, 'Livestock', 4, 3784 FROM dual
UNION ALL SELECT 3785, 'SIC', 5153, 'Grain & Field Beans', 4, 3784 FROM dual
UNION ALL SELECT 3787, 'SIC', 5159, 'Farm-Product Raw Materials, nec', 4, 3784 FROM dual
UNION ALL SELECT 3789, 'SIC', 5162, 'Plastics Materials & Basic Shapes', 4, 3788 FROM dual
UNION ALL SELECT 3790, 'SIC', 5169, 'Chemicals & Allied Products, nec', 4, 3788 FROM dual
UNION ALL SELECT 2601, 'SEC', 5171, 'Petroleum Bulk Stations &Terminals', 4, 2600 FROM dual
UNION ALL SELECT 2602, 'SEC', 5172, 'Petroleum Products, nec', 4, 2600 FROM dual
UNION ALL SELECT 3793, 'SIC', 5172, 'Petroleum Products, nec', 4, 3791 FROM dual
UNION ALL SELECT 3792, 'SIC', 5171, 'Petroleum Bulk Stations &Terminals', 4, 3791 FROM dual
UNION ALL SELECT 3796, 'SIC', 5182, 'Wines & Distilled Beverages', 4, 3794 FROM dual
UNION ALL SELECT 3795, 'SIC', 5181, 'Beer & Ale', 4, 3794 FROM dual
UNION ALL SELECT 3800, 'SIC', 5193, 'Flowers & Florists'' Supplies', 4, 3797 FROM dual
UNION ALL SELECT 3798, 'SIC', 5191, 'Farm Supplies', 4, 3797 FROM dual
UNION ALL SELECT 3799, 'SIC', 5192, 'Books, Periodicals & Newspapers', 4, 3797 FROM dual
UNION ALL SELECT 3803, 'SIC', 5199, 'Nondurable Goods, nec', 4, 3797 FROM dual
UNION ALL SELECT 3802, 'SIC', 5198, 'Paints, Varnishes & Supplies', 4, 3797 FROM dual
UNION ALL SELECT 3801, 'SIC', 5194, 'Tobacco & Tobacco Products', 4, 3797 FROM dual
UNION ALL SELECT 3890, 'SIC', 5921, 'Liquor Stores', 4, 3889 FROM dual
UNION ALL SELECT 3892, 'SIC', 5932, 'Used Merchandise Stores', 4, 3891 FROM dual
UNION ALL SELECT 2644, 'SEC', 5944, 'Jewelers Stores', 4, 2643 FROM dual
UNION ALL SELECT 2645, 'SEC', 5945, 'Hobby, Toy & Game Shops', 4, 2643 FROM dual
UNION ALL SELECT 3896, 'SIC', 5943, 'Stationary Stores', 4, 3893 FROM dual
UNION ALL SELECT 3895, 'SIC', 5942, 'Book Stores', 4, 3893 FROM dual
UNION ALL SELECT 3894, 'SIC', 5941, 'Sporting Goods & Bicycle Shops', 4, 3893 FROM dual
UNION ALL SELECT 3897, 'SIC', 5944, 'Jewelers Stores', 4, 3893 FROM dual
UNION ALL SELECT 3902, 'SIC', 5949, 'Sewing, Needlework & Piece Goods', 4, 3893 FROM dual
UNION ALL SELECT 3901, 'SIC', 5948, 'Luggage & Leather Goods Stores', 4, 3893 FROM dual
UNION ALL SELECT 3900, 'SIC', 5947, 'Gift, Novelty & Souvenir Shops', 4, 3893 FROM dual
UNION ALL SELECT 3899, 'SIC', 5946, 'Camera & Photographic Supply Stores', 4, 3893 FROM dual
UNION ALL SELECT 3898, 'SIC', 5945, 'Hobby, Toy & Game Shops', 4, 3893 FROM dual
UNION ALL SELECT 2647, 'SEC', 5961, 'Catalog & Mail Order Houses', 4, 2646 FROM dual
UNION ALL SELECT 3906, 'SIC', 5963, 'Direct Selling Establishments', 4, 3903 FROM dual
UNION ALL SELECT 3905, 'SIC', 5962, 'Merchandising Machine Operators', 4, 3903 FROM dual
UNION ALL SELECT 3904, 'SIC', 5961, 'Catalog & Mail Order Houses', 4, 3903 FROM dual
UNION ALL SELECT 3910, 'SIC', 5989, 'Fuel Dealers, nec', 4, 3907 FROM dual
UNION ALL SELECT 3909, 'SIC', 5984, 'Liquefied Petroleum Gas Dealers', 4, 3907 FROM dual
UNION ALL SELECT 3908, 'SIC', 5983, 'Fuel Oil Dealers', 4, 3907 FROM dual
UNION ALL SELECT 3915, 'SIC', 5995, 'Optical Goods Stores', 4, 3911 FROM dual
UNION ALL SELECT 3912, 'SIC', 5992, 'Florists', 4, 3911 FROM dual
UNION ALL SELECT 3913, 'SIC', 5993, 'Tobacco Stores & Stands', 4, 3911 FROM dual
UNION ALL SELECT 3914, 'SIC', 5994, 'News Dealers & News Stands', 4, 3911 FROM dual
UNION ALL SELECT 3916, 'SIC', 5999, 'Miscellaneous Retail Stores, nec', 4, 3911 FROM dual
UNION ALL SELECT 3998, 'SIC', 6726, 'Investment Offices, nec', 4, 3996 FROM dual
UNION ALL SELECT 3997, 'SIC', 6722, 'Management Investment Open-End', 4, 3996 FROM dual
UNION ALL SELECT 4000, 'SIC', 6732, 'Education, Religious, Etc. Trusts', 4, 3999 FROM dual
UNION ALL SELECT 4001, 'SIC', 6733, 'Trusts, nec', 4, 3999 FROM dual
UNION ALL SELECT 2706, 'SEC', 6794, 'Patent Owners & Lessors', 4, 2704 FROM dual
UNION ALL SELECT 2705, 'SEC', 6792, 'Oil Royalty Traders', 4, 2704 FROM dual
UNION ALL SELECT 2707, 'SEC', 6798, 'Real Estate Investment Trusts', 4, 2704 FROM dual
UNION ALL SELECT 2708, 'SEC', 6799, 'Investors, nec', 4, 2704 FROM dual
UNION ALL SELECT 4004, 'SIC', 6794, 'Patent Owners & Lessors', 4, 4002 FROM dual
UNION ALL SELECT 4003, 'SIC', 6792, 'Oil Royalty Traders', 4, 4002 FROM dual
UNION ALL SELECT 4005, 'SIC', 6798, 'Real Estate Investment Trusts', 4, 4002 FROM dual
UNION ALL SELECT 4006, 'SIC', 6799, 'Investors, nec', 4, 4002 FROM dual
UNION ALL SELECT 4067, 'SIC', 7372, 'Prepackaged Software', 4, 4065 FROM dual
UNION ALL SELECT 4074, 'SIC', 7379, 'Computer Related Services, nec', 4, 4065 FROM dual
UNION ALL SELECT 4066, 'SIC', 7371, 'Computer Programming Services', 4, 4065 FROM dual
UNION ALL SELECT 4073, 'SIC', 7378, 'Computer Maintenance & Repair', 4, 4065 FROM dual
UNION ALL SELECT 4072, 'SIC', 7377, 'Computer Rental & Leasing', 4, 4065 FROM dual
UNION ALL SELECT 4070, 'SIC', 7375, 'Information Retrieval Services', 4, 4065 FROM dual
UNION ALL SELECT 2732, 'SEC', 7381, 'Detective & Armored Car Services', 4, 2731 FROM dual
UNION ALL SELECT 2733, 'SEC', 7384, 'Photo Finishing Laboratories', 4, 2731 FROM dual
UNION ALL SELECT 2734, 'SEC', 7389, 'Trade Shows & Fairs', 4, 2731 FROM dual
UNION ALL SELECT 4077, 'SIC', 7382, 'Security Systems Services', 4, 4075 FROM dual
UNION ALL SELECT 4080, 'SIC', 7389, 'Trade Shows & Fairs', 4, 4075 FROM dual
UNION ALL SELECT 4078, 'SIC', 7383, 'News Syndicates', 4, 4075 FROM dual
UNION ALL SELECT 4076, 'SIC', 7381, 'Detective & Armored Car Services', 4, 4075 FROM dual
UNION ALL SELECT 4079, 'SIC', 7384, 'Photo Finishing Laboratories', 4, 4075 FROM dual
UNION ALL SELECT 4083, 'SIC', 7513, 'Truck Rental & Leasing, No Drivers', 4, 4082 FROM dual
UNION ALL SELECT 4086, 'SIC', 7519, 'Utility Trailer Rental', 4, 4082 FROM dual
UNION ALL SELECT 4085, 'SIC', 7515, 'Passenger Car Leasing', 4, 4082 FROM dual
UNION ALL SELECT 4084, 'SIC', 7514, 'Passenger Car Rental', 4, 4082 FROM dual
UNION ALL SELECT 4088, 'SIC', 7521, 'Automobile Parking', 4, 4087 FROM dual
UNION ALL SELECT 4091, 'SIC', 7533, 'Auto Exhaust System Repair Shops', 4, 4089 FROM dual
UNION ALL SELECT 4093, 'SIC', 7536, 'Automotive Glass Replacement Shops', 4, 4089 FROM dual
UNION ALL SELECT 4092, 'SIC', 7534, 'Tire Retreading & Repair Shops', 4, 4089 FROM dual
UNION ALL SELECT 4096, 'SIC', 7539, 'Automotive Repair Shops, nec', 4, 4089 FROM dual
UNION ALL SELECT 4095, 'SIC', 7538, 'General Automotive Repair Shops', 4, 4089 FROM dual
UNION ALL SELECT 4094, 'SIC', 7537, 'Automotive Transmission Repair Shops', 4, 4089 FROM dual
UNION ALL SELECT 4090, 'SIC', 7532, 'Top & Body Repair & Paint Shops', 4, 4089 FROM dual
UNION ALL SELECT 4099, 'SIC', 7549, 'Automotive Services, nec', 4, 4097 FROM dual
UNION ALL SELECT 4098, 'SIC', 7542, 'Carwashes', 4, 4097 FROM dual
UNION ALL SELECT 4102, 'SIC', 7622, 'Radio & Television Repair', 4, 4101 FROM dual
UNION ALL SELECT 4103, 'SIC', 7623, 'Refrigeration Service & Repair', 4, 4101 FROM dual
UNION ALL SELECT 4104, 'SIC', 7629, 'Electrical Repair Shops, nec', 4, 4101 FROM dual
UNION ALL SELECT 4106, 'SIC', 7631, 'Watch, Clock & Jewelry Repair', 4, 4105 FROM dual
UNION ALL SELECT 4108, 'SIC', 7641, 'Reupholstery & Furniture Repair', 4, 4107 FROM dual
UNION ALL SELECT 4112, 'SIC', 7699, 'Repair Services, nec', 4, 4109 FROM dual
UNION ALL SELECT 4111, 'SIC', 7694, 'Armatures Rewinding Shops', 4, 4109 FROM dual
UNION ALL SELECT 4110, 'SIC', 7692, 'Welding Repair', 4, 4109 FROM dual
UNION ALL SELECT 2740, 'SEC', 7812, 'Motion Picture & Video Production', 4, 2739 FROM dual
UNION ALL SELECT 2741, 'SEC', 7819, 'Services Allied to Motion Pictures', 4, 2739 FROM dual
UNION ALL SELECT 4115, 'SIC', 7812, 'Motion Picture & Video Production', 4, 4114 FROM dual
UNION ALL SELECT 4116, 'SIC', 7819, 'Services Allied to Motion Pictures', 4, 4114 FROM dual
UNION ALL SELECT 2743, 'SEC', 7822, 'Motion Picture & Tape Distribution', 4, 2742 FROM dual
UNION ALL SELECT 2744, 'SEC', 7829, 'Motion Picture Distribution Services', 4, 2742 FROM dual
UNION ALL SELECT 4155, 'SIC', 8042, 'Offices & Clinics of Optometrists', 4, 4153 FROM dual
UNION ALL SELECT 4154, 'SIC', 8041, 'Offices & Clinics of Chiropractors', 4, 4153 FROM dual
UNION ALL SELECT 2757, 'SEC', 8051, 'Skilled Nursing Care Facilities', 4, 2756 FROM dual
UNION ALL SELECT 4159, 'SIC', 8051, 'Skilled Nursing Care Facilities', 4, 4158 FROM dual
UNION ALL SELECT 4161, 'SIC', 8059, 'Nursing & Personal Care, nec', 4, 4158 FROM dual
UNION ALL SELECT 4160, 'SIC', 8052, 'Intermediate Care Facilities', 4, 4158 FROM dual
UNION ALL SELECT 2759, 'SEC', 8062, 'General Medical & Surgical Hospitals', 4, 2758 FROM dual
UNION ALL SELECT 4163, 'SIC', 8062, 'General Medical & Surgical Hospitals', 4, 4162 FROM dual
UNION ALL SELECT 4164, 'SIC', 8063, 'Psychiatric Hospitals', 4, 4162 FROM dual
UNION ALL SELECT 4165, 'SIC', 8069, 'Specialty Hospitals, Except Psychiatric', 4, 4162 FROM dual
UNION ALL SELECT 2761, 'SEC', 8071, 'Medical Laboratories', 4, 2760 FROM dual
UNION ALL SELECT 4168, 'SIC', 8072, 'Dental Laboratories', 4, 4166 FROM dual
UNION ALL SELECT 4167, 'SIC', 8071, 'Medical Laboratories', 4, 4166 FROM dual
UNION ALL SELECT 2763, 'SEC', 8082, 'Home Health Care Services', 4, 2762 FROM dual
UNION ALL SELECT 4170, 'SIC', 8082, 'Home Health Care Services', 4, 4169 FROM dual
UNION ALL SELECT 2765, 'SEC', 8093, 'Specialty Outpatient Clinics, nec', 4, 2764 FROM dual
UNION ALL SELECT 4173, 'SIC', 8093, 'Specialty Outpatient Clinics, nec', 4, 4171 FROM dual
UNION ALL SELECT 4172, 'SIC', 8092, 'Kidney Dialysis Centers', 4, 4171 FROM dual
UNION ALL SELECT 4174, 'SIC', 8099, 'Health & Allied Services, nec', 4, 4171 FROM dual
UNION ALL SELECT 2768, 'SEC', 8111, 'Legal Services', 4, 2767 FROM dual
UNION ALL SELECT 4177, 'SIC', 8111, 'Legal Services', 4, 4176 FROM dual
UNION ALL SELECT 4238, 'SIC', 8743, 'Public Relations Services', 4, 4235 FROM dual
UNION ALL SELECT 4236, 'SIC', 8741, 'Management Services', 4, 4235 FROM dual
UNION ALL SELECT 4237, 'SIC', 8742, 'Management Consulting Services', 4, 4235 FROM dual
UNION ALL SELECT 4239, 'SIC', 8744, 'Facilities Support Services', 4, 4235 FROM dual
UNION ALL SELECT 4240, 'SIC', 8748, 'Business Consulting, nec', 4, 4235 FROM dual
UNION ALL SELECT 4243, 'SIC', 8811, 'Private Households', 4, 4242 FROM dual
UNION ALL SELECT 4246, 'SIC', 8999, 'Services, nec', 4, 4245 FROM dual
UNION ALL SELECT 4292, 'SIC', 9641, 'Regulation of Agricultural Marketing', 4, 4291 FROM dual
UNION ALL SELECT 4294, 'SIC', 9651, 'Regulation Misc. Commercial Sectors', 4, 4293 FROM dual
UNION ALL SELECT 4296, 'SIC', 9661, 'Space Research & Technology', 4, 4295 FROM dual
UNION ALL SELECT 4299, 'SIC', 9711, 'National Security', 4, 4298 FROM dual
UNION ALL SELECT 2787, 'SEC', 9721, 'International Affairs', 4, 2786 FROM dual
UNION ALL SELECT 4301, 'SIC', 9721, 'International Affairs', 4, 4300 FROM dual
UNION ALL SELECT 4304, 'SIC', 9999, 'Nonclassifiable Establishments', 4, 4303 FROM dual
UNION ALL SELECT 56, 'NAICS', 112, 'Animal Production and Aquaculture', 2, 1 FROM dual
UNION ALL SELECT 108, 'NAICS', 114, 'Fishing, Hunting and Trapping', 2, 1 FROM dual
UNION ALL SELECT 2, 'NAICS', 111, 'Crop Production', 2, 1 FROM dual
UNION ALL SELECT 117, 'NAICS', 115, 'Support Activities for Agriculture and Forestry', 2, 1 FROM dual
UNION ALL SELECT 3, 'NAICS', 1111, 'Oilseed and Grain Farming', 3, 2 FROM dual
UNION ALL SELECT 23, 'NAICS', 1113, 'Fruit and Tree Nut Farming', 3, 2 FROM dual
UNION ALL SELECT 43, 'NAICS', 1119, 'Other Crop Farming', 3, 2 FROM dual
UNION ALL SELECT 19, 'NAICS', 1112, 'Vegetable and Melon Farming', 3, 2 FROM dual
UNION ALL SELECT 36, 'NAICS', 1114, 'Greenhouse, Nursery, and Floriculture Production', 3, 2 FROM dual
UNION ALL SELECT 11, 'NAICS', 11114, 'Wheat Farming', 4, 3 FROM dual
UNION ALL SELECT 5, 'NAICS', 11111, 'Soybean Farming', 4, 3 FROM dual
UNION ALL SELECT 13, 'NAICS', 11115, 'Corn Farming', 4, 3 FROM dual
UNION ALL SELECT 15, 'NAICS', 11116, 'Rice Farming', 4, 3 FROM dual
UNION ALL SELECT 16, 'NAICS', 11119, 'Other Grain Farming', 4, 3 FROM dual
UNION ALL SELECT 7, 'NAICS', 11112, 'Oilseed (except Soybean) Farming', 4, 3 FROM dual
UNION ALL SELECT 9, 'NAICS', 11113, 'Dry Pea and Bean Farming', 4, 3 FROM dual
UNION ALL SELECT 4, 'NAICS', 111110, 'Soybean Farming', 5, 5 FROM dual
UNION ALL SELECT 6, 'NAICS', 111120, 'Oilseed (except Soybean) Farming', 5, 7 FROM dual
UNION ALL SELECT 8, 'NAICS', 111130, 'Dry Pea and Bean Farming', 5, 9 FROM dual
UNION ALL SELECT 10, 'NAICS', 111140, 'Wheat Farming', 5, 11 FROM dual
UNION ALL SELECT 14, 'NAICS', 111160, 'Rice Farming', 5, 15 FROM dual
UNION ALL SELECT 17, 'NAICS', 111191, 'Oilseed and Grain Combination Farming', 5, 16 FROM dual
UNION ALL SELECT 18, 'NAICS', 111199, 'All Other Grain Farming', 5, 16 FROM dual
UNION ALL SELECT 20, 'NAICS', 11121, 'Vegetable and Melon Farming', 4, 19 FROM dual
UNION ALL SELECT 21, 'NAICS', 111211, 'Potato Farming', 5, 20 FROM dual
UNION ALL SELECT 4257, 'SIC', 9210, 'Courts', 3, 4256 FROM dual
UNION ALL SELECT 22, 'NAICS', 111219, 'Other Vegetable (except Potato) and Melon Farming', 5, 20 FROM dual
UNION ALL SELECT 28, 'NAICS', 11133, 'Noncitrus Fruit and Tree Nut Farming', 4, 23 FROM dual
UNION ALL SELECT 27, 'NAICS', 11132, 'Citrus (except Orange) Groves', 4, 23 FROM dual
UNION ALL SELECT 25, 'NAICS', 11131, 'Orange Groves', 4, 23 FROM dual
UNION ALL SELECT 24, 'NAICS', 111310, 'Orange Groves', 5, 25 FROM dual
UNION ALL SELECT 26, 'NAICS', 111320, 'Citrus (except Orange) Groves', 5, 27 FROM dual
UNION ALL SELECT 32, 'NAICS', 111334, 'Berry (except Strawberry) Farming', 5, 28 FROM dual
UNION ALL SELECT 33, 'NAICS', 111335, 'Tree Nut Farming', 5, 28 FROM dual
UNION ALL SELECT 34, 'NAICS', 111336, 'Fruit and Tree Nut Combination Farming', 5, 28 FROM dual
UNION ALL SELECT 30, 'NAICS', 111332, 'Grape Vineyards', 5, 28 FROM dual
UNION ALL SELECT 29, 'NAICS', 111331, 'Apple Orchards', 5, 28 FROM dual
UNION ALL SELECT 35, 'NAICS', 111339, 'Other Noncitrus Fruit Farming', 5, 28 FROM dual
UNION ALL SELECT 31, 'NAICS', 111333, 'Strawberry Farming', 5, 28 FROM dual
UNION ALL SELECT 37, 'NAICS', 11141, 'Food Crops Grown Under Cover', 4, 36 FROM dual
UNION ALL SELECT 40, 'NAICS', 11142, 'Nursery and Floriculture Production', 4, 36 FROM dual
UNION ALL SELECT 39, 'NAICS', 111419, 'Other Food Crops Grown Under Cover', 5, 37 FROM dual
UNION ALL SELECT 38, 'NAICS', 111411, 'Mushroom Production', 5, 37 FROM dual
UNION ALL SELECT 42, 'NAICS', 111422, 'Floriculture Production', 5, 40 FROM dual
UNION ALL SELECT 41, 'NAICS', 111421, 'Nursery and Tree Production', 5, 40 FROM dual
UNION ALL SELECT 51, 'NAICS', 11194, 'Hay Farming', 4, 43 FROM dual
UNION ALL SELECT 45, 'NAICS', 11191, 'Tobacco Farming', 4, 43 FROM dual
UNION ALL SELECT 47, 'NAICS', 11192, 'Cotton Farming', 4, 43 FROM dual
UNION ALL SELECT 49, 'NAICS', 11193, 'Sugarcane Farming', 4, 43 FROM dual
UNION ALL SELECT 52, 'NAICS', 11199, 'All Other Crop Farming', 4, 43 FROM dual
UNION ALL SELECT 44, 'NAICS', 111910, 'Tobacco Farming', 5, 45 FROM dual
UNION ALL SELECT 46, 'NAICS', 111920, 'Cotton Farming', 5, 47 FROM dual
UNION ALL SELECT 48, 'NAICS', 111930, 'Sugarcane Farming', 5, 49 FROM dual
UNION ALL SELECT 50, 'NAICS', 111940, 'Hay Farming', 5, 51 FROM dual
UNION ALL SELECT 53, 'NAICS', 111991, 'Sugar Beet Farming', 5, 52 FROM dual
UNION ALL SELECT 55, 'NAICS', 111998, 'All Other Miscellaneous Crop Farming', 5, 52 FROM dual
UNION ALL SELECT 54, 'NAICS', 111992, 'Peanut Farming', 5, 52 FROM dual
UNION ALL SELECT 68, 'NAICS', 1123, 'Poultry and Egg Production', 3, 56 FROM dual
UNION ALL SELECT 65, 'NAICS', 1122, 'Hog and Pig Farming', 3, 56 FROM dual
UNION ALL SELECT 89, 'NAICS', 1129, 'Other Animal Production', 3, 56 FROM dual
UNION ALL SELECT 84, 'NAICS', 1125, 'Aquaculture', 3, 56 FROM dual
UNION ALL SELECT 79, 'NAICS', 1124, 'Sheep and Goat Farming', 3, 56 FROM dual
UNION ALL SELECT 57, 'NAICS', 1121, 'Cattle Ranching and Farming', 3, 56 FROM dual
UNION ALL SELECT 62, 'NAICS', 11212, 'Dairy Cattle and Milk Production', 4, 57 FROM dual
UNION ALL SELECT 64, 'NAICS', 11213, 'Dual-Purpose Cattle Ranching and Farming', 4, 57 FROM dual
UNION ALL SELECT 58, 'NAICS', 11211, 'Beef Cattle Ranching and Farming, including Feedlots', 4, 57 FROM dual
UNION ALL SELECT 59, 'NAICS', 112111, 'Beef Cattle Ranching and Farming', 5, 58 FROM dual
UNION ALL SELECT 60, 'NAICS', 112112, 'Cattle Feedlots', 5, 58 FROM dual
UNION ALL SELECT 61, 'NAICS', 112120, 'Dairy Cattle and Milk Production', 5, 62 FROM dual
UNION ALL SELECT 63, 'NAICS', 112130, 'Dual-Purpose Cattle Ranching and Farming', 5, 64 FROM dual
UNION ALL SELECT 67, 'NAICS', 11221, 'Hog and Pig Farming', 4, 65 FROM dual
UNION ALL SELECT 66, 'NAICS', 112210, 'Hog and Pig Farming', 5, 67 FROM dual
UNION ALL SELECT 70, 'NAICS', 11231, 'Chicken Egg Production', 4, 68 FROM dual
UNION ALL SELECT 72, 'NAICS', 11232, 'Broilers and Other Meat Type Chicken Production', 4, 68 FROM dual
UNION ALL SELECT 76, 'NAICS', 11234, 'Poultry Hatcheries', 4, 68 FROM dual
UNION ALL SELECT 78, 'NAICS', 11239, 'Other Poultry Production', 4, 68 FROM dual
UNION ALL SELECT 74, 'NAICS', 11233, 'Turkey Production', 4, 68 FROM dual
UNION ALL SELECT 69, 'NAICS', 112310, 'Chicken Egg Production', 5, 70 FROM dual
UNION ALL SELECT 71, 'NAICS', 112320, 'Broilers and Other Meat Type Chicken Production', 5, 72 FROM dual
UNION ALL SELECT 73, 'NAICS', 112330, 'Turkey Production', 5, 74 FROM dual
UNION ALL SELECT 75, 'NAICS', 112340, 'Poultry Hatcheries', 5, 76 FROM dual
UNION ALL SELECT 77, 'NAICS', 112390, 'Other Poultry Production', 5, 78 FROM dual
UNION ALL SELECT 81, 'NAICS', 11241, 'Sheep Farming', 4, 79 FROM dual
UNION ALL SELECT 83, 'NAICS', 11242, 'Goat Farming', 4, 79 FROM dual
UNION ALL SELECT 80, 'NAICS', 112410, 'Sheep Farming', 5, 81 FROM dual
UNION ALL SELECT 82, 'NAICS', 112420, 'Goat Farming', 5, 83 FROM dual
UNION ALL SELECT 85, 'NAICS', 11251, 'Aquaculture', 4, 84 FROM dual
UNION ALL SELECT 88, 'NAICS', 112519, 'Other Aquaculture', 5, 85 FROM dual
UNION ALL SELECT 86, 'NAICS', 112511, 'Finfish Farming and Fish Hatcheries', 5, 85 FROM dual
UNION ALL SELECT 87, 'NAICS', 112512, 'Shellfish Farming', 5, 85 FROM dual
UNION ALL SELECT 95, 'NAICS', 11293, 'Fur-Bearing Animal and Rabbit Production', 4, 89 FROM dual
UNION ALL SELECT 97, 'NAICS', 11299, 'All Other Animal Production', 4, 89 FROM dual
UNION ALL SELECT 93, 'NAICS', 11292, 'Horses and Other Equine Production', 4, 89 FROM dual
UNION ALL SELECT 91, 'NAICS', 11291, 'Apiculture', 4, 89 FROM dual
UNION ALL SELECT 90, 'NAICS', 112910, 'Apiculture', 5, 91 FROM dual
UNION ALL SELECT 92, 'NAICS', 112920, 'Horses and Other Equine Production', 5, 93 FROM dual
UNION ALL SELECT 94, 'NAICS', 112930, 'Fur-Bearing Animal and Rabbit Production', 5, 95 FROM dual
UNION ALL SELECT 96, 'NAICS', 112990, 'All Other Animal Production', 5, 97 FROM dual
UNION ALL SELECT 102, 'NAICS', 1132, 'Forest Nurseries and Gathering of Forest Products', 3, 98 FROM dual
UNION ALL SELECT 99, 'NAICS', 1131, 'Timber Tract Operations', 3, 98 FROM dual
UNION ALL SELECT 101, 'NAICS', 11311, 'Timber Tract Operations', 4, 99 FROM dual
UNION ALL SELECT 100, 'NAICS', 113110, 'Timber Tract Operations', 5, 101 FROM dual
UNION ALL SELECT 104, 'NAICS', 11321, 'Forest Nurseries and Gathering of Forest Products', 4, 102 FROM dual
UNION ALL SELECT 103, 'NAICS', 113210, 'Forest Nurseries and Gathering of Forest Products', 5, 104 FROM dual
UNION ALL SELECT 106, 'NAICS', 113310, 'Logging', 5, 107 FROM dual
UNION ALL SELECT 109, 'NAICS', 1141, 'Fishing', 3, 108 FROM dual
UNION ALL SELECT 114, 'NAICS', 1142, 'Hunting and Trapping', 3, 108 FROM dual
UNION ALL SELECT 110, 'NAICS', 11411, 'Fishing', 4, 109 FROM dual
UNION ALL SELECT 113, 'NAICS', 114119, 'Other Marine Fishing', 5, 110 FROM dual
UNION ALL SELECT 111, 'NAICS', 114111, 'Finfish Fishing', 5, 110 FROM dual
UNION ALL SELECT 112, 'NAICS', 114112, 'Shellfish Fishing', 5, 110 FROM dual
UNION ALL SELECT 116, 'NAICS', 11421, 'Hunting and Trapping', 4, 114 FROM dual
UNION ALL SELECT 115, 'NAICS', 114210, 'Hunting and Trapping', 5, 116 FROM dual
UNION ALL SELECT 129, 'NAICS', 1153, 'Support Activities for Forestry', 3, 117 FROM dual
UNION ALL SELECT 126, 'NAICS', 1152, 'Support Activities for Animal Production', 3, 117 FROM dual
UNION ALL SELECT 118, 'NAICS', 1151, 'Support Activities for Crop Production', 3, 117 FROM dual
UNION ALL SELECT 119, 'NAICS', 11511, 'Support Activities for Crop Production', 4, 118 FROM dual
UNION ALL SELECT 120, 'NAICS', 115111, 'Cotton Ginning', 5, 119 FROM dual
UNION ALL SELECT 125, 'NAICS', 115116, 'Farm Management Services', 5, 119 FROM dual
UNION ALL SELECT 124, 'NAICS', 115115, 'Farm Labor Contractors and Crew Leaders', 5, 119 FROM dual
UNION ALL SELECT 123, 'NAICS', 115114, 'Postharvest Crop Activities (except Cotton Ginning)', 5, 119 FROM dual
UNION ALL SELECT 122, 'NAICS', 115113, 'Crop Harvesting, Primarily by Machine', 5, 119 FROM dual
UNION ALL SELECT 121, 'NAICS', 115112, 'Soil Preparation, Planting, and Cultivating', 5, 119 FROM dual
UNION ALL SELECT 128, 'NAICS', 11521, 'Support Activities for Animal Production', 4, 126 FROM dual
UNION ALL SELECT 127, 'NAICS', 115210, 'Support Activities for Animal Production', 5, 128 FROM dual
UNION ALL SELECT 131, 'NAICS', 11531, 'Support Activities for Forestry', 4, 129 FROM dual
UNION ALL SELECT 130, 'NAICS', 115310, 'Support Activities for Forestry', 5, 131 FROM dual
UNION ALL SELECT 172, 'NAICS', 213, 'Support Activities for Mining', 2, 132 FROM dual
UNION ALL SELECT 133, 'NAICS', 211, 'Oil and Gas Extraction', 2, 132 FROM dual
UNION ALL SELECT 138, 'NAICS', 212, 'Mining (except Oil and Gas)', 2, 132 FROM dual
UNION ALL SELECT 134, 'NAICS', 2111, 'Oil and Gas Extraction', 3, 133 FROM dual
UNION ALL SELECT 135, 'NAICS', 21111, 'Oil and Gas Extraction', 4, 134 FROM dual
UNION ALL SELECT 137, 'NAICS', 211112, 'Natural Gas Liquid Extraction', 5, 135 FROM dual
UNION ALL SELECT 136, 'NAICS', 211111, 'Crude Petroleum and Natural Gas Extraction', 5, 135 FROM dual
UNION ALL SELECT 156, 'NAICS', 2123, 'Nonmetallic Mineral Mining and Quarrying', 3, 138 FROM dual
UNION ALL SELECT 144, 'NAICS', 2122, 'Metal Ore Mining', 3, 138 FROM dual
UNION ALL SELECT 139, 'NAICS', 2121, 'Coal Mining', 3, 138 FROM dual
UNION ALL SELECT 140, 'NAICS', 21211, 'Coal Mining', 4, 139 FROM dual
UNION ALL SELECT 143, 'NAICS', 212113, 'Anthracite Mining', 5, 140 FROM dual
UNION ALL SELECT 142, 'NAICS', 212112, 'Bituminous Coal Underground Mining', 5, 140 FROM dual
UNION ALL SELECT 141, 'NAICS', 212111, 'Bituminous Coal and Lignite Surface Mining', 5, 140 FROM dual
UNION ALL SELECT 153, 'NAICS', 21229, 'Other Metal Ore Mining', 4, 144 FROM dual
UNION ALL SELECT 150, 'NAICS', 21223, 'Copper, Nickel, Lead, and Zinc Mining', 4, 144 FROM dual
UNION ALL SELECT 147, 'NAICS', 21222, 'Gold Ore and Silver Ore Mining', 4, 144 FROM dual
UNION ALL SELECT 146, 'NAICS', 21221, 'Iron Ore Mining', 4, 144 FROM dual
UNION ALL SELECT 145, 'NAICS', 212210, 'Iron Ore Mining', 5, 146 FROM dual
UNION ALL SELECT 148, 'NAICS', 212221, 'Gold Ore Mining', 5, 147 FROM dual
UNION ALL SELECT 149, 'NAICS', 212222, 'Silver Ore Mining', 5, 147 FROM dual
UNION ALL SELECT 151, 'NAICS', 212231, 'Lead Ore and Zinc Ore Mining', 5, 150 FROM dual
UNION ALL SELECT 152, 'NAICS', 212234, 'Copper Ore and Nickel Ore Mining', 5, 150 FROM dual
UNION ALL SELECT 155, 'NAICS', 212299, 'All Other Metal Ore Mining', 5, 153 FROM dual
UNION ALL SELECT 154, 'NAICS', 212291, 'Uranium-Radium-Vanadium Ore Mining', 5, 153 FROM dual
UNION ALL SELECT 162, 'NAICS', 21232, 'Sand, Gravel, Clay, and Ceramic and Refractory Minerals Mining and Quarrying', 4, 156 FROM dual
UNION ALL SELECT 167, 'NAICS', 21239, 'Other Nonmetallic Mineral Mining and Quarrying', 4, 156 FROM dual
UNION ALL SELECT 157, 'NAICS', 21231, 'Stone Mining and Quarrying', 4, 156 FROM dual
UNION ALL SELECT 159, 'NAICS', 212312, 'Crushed and Broken Limestone Mining and Quarrying', 5, 157 FROM dual
UNION ALL SELECT 160, 'NAICS', 212313, 'Crushed and Broken Granite Mining and Quarrying', 5, 157 FROM dual
UNION ALL SELECT 161, 'NAICS', 212319, 'Other Crushed and Broken Stone Mining and Quarrying', 5, 157 FROM dual
UNION ALL SELECT 158, 'NAICS', 212311, 'Dimension Stone Mining and Quarrying', 5, 157 FROM dual
UNION ALL SELECT 163, 'NAICS', 212321, 'Construction Sand and Gravel Mining', 5, 162 FROM dual
UNION ALL SELECT 166, 'NAICS', 212325, 'Clay and Ceramic and Refractory Minerals Mining', 5, 162 FROM dual
UNION ALL SELECT 165, 'NAICS', 212324, 'Kaolin and Ball Clay Mining', 5, 162 FROM dual
UNION ALL SELECT 164, 'NAICS', 212322, 'Industrial Sand Mining', 5, 162 FROM dual
UNION ALL SELECT 170, 'NAICS', 212393, 'Other Chemical and Fertilizer Mineral Mining', 5, 167 FROM dual
UNION ALL SELECT 169, 'NAICS', 212392, 'Phosphate Rock Mining', 5, 167 FROM dual
UNION ALL SELECT 171, 'NAICS', 212399, 'All Other Nonmetallic Mineral Mining', 5, 167 FROM dual
UNION ALL SELECT 168, 'NAICS', 212391, 'Potash, Soda, and Borate Mineral Mining', 5, 167 FROM dual
UNION ALL SELECT 173, 'NAICS', 2131, 'Support Activities for Mining', 3, 172 FROM dual
UNION ALL SELECT 174, 'NAICS', 21311, 'Support Activities for Mining', 4, 173 FROM dual
UNION ALL SELECT 176, 'NAICS', 213112, 'Support Activities for Oil and Gas Operations', 5, 174 FROM dual
UNION ALL SELECT 178, 'NAICS', 213114, 'Support Activities for Metal Mining', 5, 174 FROM dual
UNION ALL SELECT 179, 'NAICS', 213115, 'Support Activities for Nonmetallic Minerals (except Fuels) Mining', 5, 174 FROM dual
UNION ALL SELECT 296, 'NAICS', 31131, 'Sugar Manufacturing', 4, 295 FROM dual
UNION ALL SELECT 177, 'NAICS', 213113, 'Support Activities for Coal Mining', 5, 174 FROM dual
UNION ALL SELECT 175, 'NAICS', 213111, 'Drilling Oil and Gas Wells', 5, 174 FROM dual
UNION ALL SELECT 181, 'NAICS', 221, 'Utilities', 2, 180 FROM dual
UNION ALL SELECT 182, 'NAICS', 2211, 'Electric Power Generation, Transmission and Distribution', 3, 181 FROM dual
UNION ALL SELECT 195, 'NAICS', 2212, 'Natural Gas Distribution', 3, 181 FROM dual
UNION ALL SELECT 198, 'NAICS', 2213, 'Water, Sewage and Other Systems', 3, 181 FROM dual
UNION ALL SELECT 192, 'NAICS', 22112, 'Electric Power Transmission, Control, and Distribution', 4, 182 FROM dual
UNION ALL SELECT 183, 'NAICS', 22111, 'Electric Power Generation', 4, 182 FROM dual
UNION ALL SELECT 190, 'NAICS', 221117, 'Biomass Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 184, 'NAICS', 221111, 'Hydroelectric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 185, 'NAICS', 221112, 'Fossil Fuel Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 186, 'NAICS', 221113, 'Nuclear Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 187, 'NAICS', 221114, 'Solar Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 188, 'NAICS', 221115, 'Wind Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 189, 'NAICS', 221116, 'Geothermal Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 191, 'NAICS', 221118, 'Other Electric Power Generation', 5, 183 FROM dual
UNION ALL SELECT 193, 'NAICS', 221121, 'Electric Bulk Power Transmission and Control', 5, 192 FROM dual
UNION ALL SELECT 194, 'NAICS', 221122, 'Electric Power Distribution', 5, 192 FROM dual
UNION ALL SELECT 197, 'NAICS', 22121, 'Natural Gas Distribution', 4, 195 FROM dual
UNION ALL SELECT 196, 'NAICS', 221210, 'Natural Gas Distribution', 5, 197 FROM dual
UNION ALL SELECT 204, 'NAICS', 22133, 'Steam and Air-Conditioning Supply', 4, 198 FROM dual
UNION ALL SELECT 200, 'NAICS', 22131, 'Water Supply and Irrigation Systems', 4, 198 FROM dual
UNION ALL SELECT 202, 'NAICS', 22132, 'Sewage Treatment Facilities', 4, 198 FROM dual
UNION ALL SELECT 199, 'NAICS', 221310, 'Water Supply and Irrigation Systems', 5, 200 FROM dual
UNION ALL SELECT 201, 'NAICS', 221320, 'Sewage Treatment Facilities', 5, 202 FROM dual
UNION ALL SELECT 203, 'NAICS', 221330, 'Steam and Air-Conditioning Supply', 5, 204 FROM dual
UNION ALL SELECT 206, 'NAICS', 236, 'Construction of Buildings', 2, 205 FROM dual
UNION ALL SELECT 235, 'NAICS', 238, 'Specialty Trade Contractors', 2, 205 FROM dual
UNION ALL SELECT 218, 'NAICS', 237, 'Heavy and Civil Engineering Construction', 2, 205 FROM dual
UNION ALL SELECT 213, 'NAICS', 2362, 'Nonresidential Building Construction', 3, 206 FROM dual
UNION ALL SELECT 207, 'NAICS', 2361, 'Residential Building Construction', 3, 206 FROM dual
UNION ALL SELECT 208, 'NAICS', 23611, 'Residential Building Construction', 4, 207 FROM dual
UNION ALL SELECT 209, 'NAICS', 236115, 'New Single-Family Housing Construction (except For-Sale Builders)', 5, 208 FROM dual
UNION ALL SELECT 210, 'NAICS', 236116, 'New Multifamily Housing Construction (except For-Sale Builders)', 5, 208 FROM dual
UNION ALL SELECT 211, 'NAICS', 236117, 'New Housing For-Sale Builders', 5, 208 FROM dual
UNION ALL SELECT 212, 'NAICS', 236118, 'Residential Remodelers', 5, 208 FROM dual
UNION ALL SELECT 217, 'NAICS', 23622, 'Commercial and Institutional Building Construction', 4, 213 FROM dual
UNION ALL SELECT 215, 'NAICS', 23621, 'Industrial Building Construction', 4, 213 FROM dual
UNION ALL SELECT 214, 'NAICS', 236210, 'Industrial Building Construction', 5, 215 FROM dual
UNION ALL SELECT 216, 'NAICS', 236220, 'Commercial and Institutional Building Construction', 5, 217 FROM dual
UNION ALL SELECT 229, 'NAICS', 2373, 'Highway, Street, and Bridge Construction', 3, 218 FROM dual
UNION ALL SELECT 232, 'NAICS', 2379, 'Other Heavy and Civil Engineering Construction', 3, 218 FROM dual
UNION ALL SELECT 219, 'NAICS', 2371, 'Utility System Construction', 3, 218 FROM dual
UNION ALL SELECT 226, 'NAICS', 2372, 'Land Subdivision', 3, 218 FROM dual
UNION ALL SELECT 225, 'NAICS', 23713, 'Power and Communication Line and Related Structures Construction', 4, 219 FROM dual
UNION ALL SELECT 221, 'NAICS', 23711, 'Water and Sewer Line and Related Structures Construction', 4, 219 FROM dual
UNION ALL SELECT 223, 'NAICS', 23712, 'Oil and Gas Pipeline and Related Structures Construction', 4, 219 FROM dual
UNION ALL SELECT 220, 'NAICS', 237110, 'Water and Sewer Line and Related Structures Construction', 5, 221 FROM dual
UNION ALL SELECT 222, 'NAICS', 237120, 'Oil and Gas Pipeline and Related Structures Construction', 5, 223 FROM dual
UNION ALL SELECT 224, 'NAICS', 237130, 'Power and Communication Line and Related Structures Construction', 5, 225 FROM dual
UNION ALL SELECT 228, 'NAICS', 23721, 'Land Subdivision', 4, 226 FROM dual
UNION ALL SELECT 227, 'NAICS', 237210, 'Land Subdivision', 5, 228 FROM dual
UNION ALL SELECT 231, 'NAICS', 23731, 'Highway, Street, and Bridge Construction', 4, 229 FROM dual
UNION ALL SELECT 230, 'NAICS', 237310, 'Highway, Street, and Bridge Construction', 5, 231 FROM dual
UNION ALL SELECT 234, 'NAICS', 23799, 'Other Heavy and Civil Engineering Construction', 4, 232 FROM dual
UNION ALL SELECT 233, 'NAICS', 237990, 'Other Heavy and Civil Engineering Construction', 5, 234 FROM dual
UNION ALL SELECT 253, 'NAICS', 2382, 'Building Equipment Contractors', 3, 235 FROM dual
UNION ALL SELECT 273, 'NAICS', 2389, 'Other Specialty Trade Contractors', 3, 235 FROM dual
UNION ALL SELECT 260, 'NAICS', 2383, 'Building Finishing Contractors', 3, 235 FROM dual
UNION ALL SELECT 236, 'NAICS', 2381, 'Foundation, Structure, and Building Exterior Contractors', 3, 235 FROM dual
UNION ALL SELECT 246, 'NAICS', 23815, 'Glass and Glazing Contractors', 4, 236 FROM dual
UNION ALL SELECT 252, 'NAICS', 23819, 'Other Foundation, Structure, and Building Exterior Contractors', 4, 236 FROM dual
UNION ALL SELECT 238, 'NAICS', 23811, 'Poured Concrete Foundation and Structure Contractors', 4, 236 FROM dual
UNION ALL SELECT 250, 'NAICS', 23817, 'Siding Contractors', 4, 236 FROM dual
UNION ALL SELECT 240, 'NAICS', 23812, 'Structural Steel and Precast Concrete Contractors', 4, 236 FROM dual
UNION ALL SELECT 248, 'NAICS', 23816, 'Roofing Contractors', 4, 236 FROM dual
UNION ALL SELECT 242, 'NAICS', 23813, 'Framing Contractors', 4, 236 FROM dual
UNION ALL SELECT 244, 'NAICS', 23814, 'Masonry Contractors', 4, 236 FROM dual
UNION ALL SELECT 237, 'NAICS', 238110, 'Poured Concrete Foundation and Structure Contractors', 5, 238 FROM dual
UNION ALL SELECT 239, 'NAICS', 238120, 'Structural Steel and Precast Concrete Contractors', 5, 240 FROM dual
UNION ALL SELECT 241, 'NAICS', 238130, 'Framing Contractors', 5, 242 FROM dual
UNION ALL SELECT 243, 'NAICS', 238140, 'Masonry Contractors', 5, 244 FROM dual
UNION ALL SELECT 245, 'NAICS', 238150, 'Glass and Glazing Contractors', 5, 246 FROM dual
UNION ALL SELECT 247, 'NAICS', 238160, 'Roofing Contractors', 5, 248 FROM dual
UNION ALL SELECT 249, 'NAICS', 238170, 'Siding Contractors', 5, 250 FROM dual
UNION ALL SELECT 251, 'NAICS', 238190, 'Other Foundation, Structure, and Building Exterior Contractors', 5, 252 FROM dual
UNION ALL SELECT 259, 'NAICS', 23829, 'Other Building Equipment Contractors', 4, 253 FROM dual
UNION ALL SELECT 255, 'NAICS', 23821, 'Electrical Contractors and Other Wiring Installation Contractors', 4, 253 FROM dual
UNION ALL SELECT 257, 'NAICS', 23822, 'Plumbing, Heating, and Air-Conditioning Contractors', 4, 253 FROM dual
UNION ALL SELECT 254, 'NAICS', 238210, 'Electrical Contractors and Other Wiring Installation Contractors', 5, 255 FROM dual
UNION ALL SELECT 256, 'NAICS', 238220, 'Plumbing, Heating, and Air-Conditioning Contractors', 5, 257 FROM dual
UNION ALL SELECT 258, 'NAICS', 238290, 'Other Building Equipment Contractors', 5, 259 FROM dual
UNION ALL SELECT 268, 'NAICS', 23834, 'Tile and Terrazzo Contractors', 4, 260 FROM dual
UNION ALL SELECT 266, 'NAICS', 23833, 'Flooring Contractors', 4, 260 FROM dual
UNION ALL SELECT 272, 'NAICS', 23839, 'Other Building Finishing Contractors', 4, 260 FROM dual
UNION ALL SELECT 270, 'NAICS', 23835, 'Finish Carpentry Contractors', 4, 260 FROM dual
UNION ALL SELECT 264, 'NAICS', 23832, 'Painting and Wall Covering Contractors', 4, 260 FROM dual
UNION ALL SELECT 262, 'NAICS', 23831, 'Drywall and Insulation Contractors', 4, 260 FROM dual
UNION ALL SELECT 261, 'NAICS', 238310, 'Drywall and Insulation Contractors', 5, 262 FROM dual
UNION ALL SELECT 263, 'NAICS', 238320, 'Painting and Wall Covering Contractors', 5, 264 FROM dual
UNION ALL SELECT 265, 'NAICS', 238330, 'Flooring Contractors', 5, 266 FROM dual
UNION ALL SELECT 267, 'NAICS', 238340, 'Tile and Terrazzo Contractors', 5, 268 FROM dual
UNION ALL SELECT 269, 'NAICS', 238350, 'Finish Carpentry Contractors', 5, 270 FROM dual
UNION ALL SELECT 271, 'NAICS', 238390, 'Other Building Finishing Contractors', 5, 272 FROM dual
UNION ALL SELECT 275, 'NAICS', 23891, 'Site Preparation Contractors', 4, 273 FROM dual
UNION ALL SELECT 277, 'NAICS', 23899, 'All Other Specialty Trade Contractors', 4, 273 FROM dual
UNION ALL SELECT 274, 'NAICS', 238910, 'Site Preparation Contractors', 5, 275 FROM dual
UNION ALL SELECT 276, 'NAICS', 238990, 'All Other Specialty Trade Contractors', 5, 277 FROM dual
UNION ALL SELECT 353, 'NAICS', 312, 'Beverage and Tobacco Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 904, 'NAICS', 339, 'Miscellaneous Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 883, 'NAICS', 337, 'Furniture and Related Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 833, 'NAICS', 336, 'Transportation Equipment Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 798, 'NAICS', 335, 'Electrical Equipment, Appliance, and Component Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 279, 'NAICS', 311, 'Food Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 759, 'NAICS', 334, 'Computer and Electronic Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 386, 'NAICS', 314, 'Textile Product Mills', 2, 278 FROM dual
UNION ALL SELECT 398, 'NAICS', 315, 'Apparel Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 416, 'NAICS', 316, 'Leather and Allied Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 427, 'NAICS', 321, 'Wood Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 640, 'NAICS', 332, 'Fabricated Metal Product Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 450, 'NAICS', 322, 'Paper Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 471, 'NAICS', 323, 'Printing and Related Support Activities', 2, 278 FROM dual
UNION ALL SELECT 479, 'NAICS', 324, 'Petroleum and Coal Products Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 607, 'NAICS', 331, 'Primary Metal Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 489, 'NAICS', 325, 'Chemical Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 543, 'NAICS', 326, 'Plastics and Rubber Products Manufacturing', 2, 278 FROM dual
UNION ALL SELECT 329, 'NAICS', 3118, 'Bakeries and Tortilla Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 326, 'NAICS', 3117, 'Seafood Product Preparation and Packaging', 3, 279 FROM dual
UNION ALL SELECT 320, 'NAICS', 3116, 'Animal Slaughtering and Processing', 3, 279 FROM dual
UNION ALL SELECT 295, 'NAICS', 3113, 'Sugar and Confectionery Product Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 280, 'NAICS', 3111, 'Animal Food Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 284, 'NAICS', 3112, 'Grain and Oilseed Milling', 3, 279 FROM dual
UNION ALL SELECT 304, 'NAICS', 3114, 'Fruit and Vegetable Preserving and Specialty Food Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 312, 'NAICS', 3115, 'Dairy Product Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 339, 'NAICS', 3119, 'Other Food Manufacturing', 3, 279 FROM dual
UNION ALL SELECT 281, 'NAICS', 31111, 'Animal Food Manufacturing', 4, 280 FROM dual
UNION ALL SELECT 283, 'NAICS', 311119, 'Other Animal Food Manufacturing', 5, 281 FROM dual
UNION ALL SELECT 282, 'NAICS', 311111, 'Dog and Cat Food Manufacturing', 5, 281 FROM dual
UNION ALL SELECT 289, 'NAICS', 31122, 'Starch and Vegetable Fats and Oils Manufacturing', 4, 284 FROM dual
UNION ALL SELECT 285, 'NAICS', 31121, 'Flour Milling and Malt Manufacturing', 4, 284 FROM dual
UNION ALL SELECT 294, 'NAICS', 31123, 'Breakfast Cereal Manufacturing', 4, 284 FROM dual
UNION ALL SELECT 287, 'NAICS', 311212, 'Rice Milling', 5, 285 FROM dual
UNION ALL SELECT 288, 'NAICS', 311213, 'Malt Manufacturing', 5, 285 FROM dual
UNION ALL SELECT 286, 'NAICS', 311211, 'Flour Milling', 5, 285 FROM dual
UNION ALL SELECT 290, 'NAICS', 311221, 'Wet Corn Milling', 5, 289 FROM dual
UNION ALL SELECT 292, 'NAICS', 311225, 'Fats and Oils Refining and Blending', 5, 289 FROM dual
UNION ALL SELECT 291, 'NAICS', 311224, 'Soybean and Other Oilseed Processing', 5, 289 FROM dual
UNION ALL SELECT 293, 'NAICS', 311230, 'Breakfast Cereal Manufacturing', 5, 294 FROM dual
UNION ALL SELECT 300, 'NAICS', 31134, 'Nonchocolate Confectionery Manufacturing', 4, 295 FROM dual
UNION ALL SELECT 301, 'NAICS', 31135, 'Chocolate and Confectionery Manufacturing', 4, 295 FROM dual
UNION ALL SELECT 298, 'NAICS', 311314, 'Cane Sugar Manufacturing', 5, 296 FROM dual
UNION ALL SELECT 297, 'NAICS', 311313, 'Beet Sugar Manufacturing', 5, 296 FROM dual
UNION ALL SELECT 299, 'NAICS', 311340, 'Nonchocolate Confectionery Manufacturing', 5, 300 FROM dual
UNION ALL SELECT 303, 'NAICS', 311352, 'Confectionery Manufacturing from Purchased Chocolate', 5, 301 FROM dual
UNION ALL SELECT 302, 'NAICS', 311351, 'Chocolate and Confectionery Manufacturing from Cacao Beans', 5, 301 FROM dual
UNION ALL SELECT 305, 'NAICS', 31141, 'Frozen Food Manufacturing', 4, 304 FROM dual
UNION ALL SELECT 308, 'NAICS', 31142, 'Fruit and Vegetable Canning, Pickling, and Drying', 4, 304 FROM dual
UNION ALL SELECT 306, 'NAICS', 311411, 'Frozen Fruit, Juice, and Vegetable Manufacturing', 5, 305 FROM dual
UNION ALL SELECT 307, 'NAICS', 311412, 'Frozen Specialty Food Manufacturing', 5, 305 FROM dual
UNION ALL SELECT 311, 'NAICS', 311423, 'Dried and Dehydrated Food Manufacturing', 5, 308 FROM dual
UNION ALL SELECT 309, 'NAICS', 311421, 'Fruit and Vegetable Canning', 5, 308 FROM dual
UNION ALL SELECT 310, 'NAICS', 311422, 'Specialty Canning', 5, 308 FROM dual
UNION ALL SELECT 319, 'NAICS', 31152, 'Ice Cream and Frozen Dessert Manufacturing', 4, 312 FROM dual
UNION ALL SELECT 313, 'NAICS', 31151, 'Dairy Product (except Frozen) Manufacturing', 4, 312 FROM dual
UNION ALL SELECT 317, 'NAICS', 311514, 'Dry, Condensed, and Evaporated Dairy Product Manufacturing', 5, 313 FROM dual
UNION ALL SELECT 316, 'NAICS', 311513, 'Cheese Manufacturing', 5, 313 FROM dual
UNION ALL SELECT 315, 'NAICS', 311512, 'Creamery Butter Manufacturing', 5, 313 FROM dual
UNION ALL SELECT 314, 'NAICS', 311511, 'Fluid Milk Manufacturing', 5, 313 FROM dual
UNION ALL SELECT 318, 'NAICS', 311520, 'Ice Cream and Frozen Dessert Manufacturing', 5, 319 FROM dual
UNION ALL SELECT 321, 'NAICS', 31161, 'Animal Slaughtering and Processing', 4, 320 FROM dual
UNION ALL SELECT 323, 'NAICS', 311612, 'Meat Processed from Carcasses', 5, 321 FROM dual
UNION ALL SELECT 322, 'NAICS', 311611, 'Animal (except Poultry) Slaughtering', 5, 321 FROM dual
UNION ALL SELECT 324, 'NAICS', 311613, 'Rendering and Meat Byproduct Processing', 5, 321 FROM dual
UNION ALL SELECT 325, 'NAICS', 311615, 'Poultry Processing', 5, 321 FROM dual
UNION ALL SELECT 328, 'NAICS', 31171, 'Seafood Product Preparation and Packaging', 4, 326 FROM dual
UNION ALL SELECT 327, 'NAICS', 311710, 'Seafood Product Preparation and Packaging', 5, 328 FROM dual
UNION ALL SELECT 338, 'NAICS', 31183, 'Tortilla Manufacturing', 4, 329 FROM dual
UNION ALL SELECT 330, 'NAICS', 31181, 'Bread and Bakery Product Manufacturing', 4, 329 FROM dual
UNION ALL SELECT 334, 'NAICS', 31182, 'Cookie, Cracker, and Pasta Manufacturing', 4, 329 FROM dual
UNION ALL SELECT 331, 'NAICS', 311811, 'Retail Bakeries', 5, 330 FROM dual
UNION ALL SELECT 332, 'NAICS', 311812, 'Commercial Bakeries', 5, 330 FROM dual
UNION ALL SELECT 333, 'NAICS', 311813, 'Frozen Cakes, Pies, and Other Pastries Manufacturing', 5, 330 FROM dual
UNION ALL SELECT 335, 'NAICS', 311821, 'Cookie and Cracker Manufacturing', 5, 334 FROM dual
UNION ALL SELECT 336, 'NAICS', 311824, 'Dry Pasta, Dough, and Flour Mixes Manufacturing from Purchased Flour', 5, 334 FROM dual
UNION ALL SELECT 337, 'NAICS', 311830, 'Tortilla Manufacturing', 5, 338 FROM dual
UNION ALL SELECT 347, 'NAICS', 31194, 'Seasoning and Dressing Manufacturing', 4, 339 FROM dual
UNION ALL SELECT 340, 'NAICS', 31191, 'Snack Food Manufacturing', 4, 339 FROM dual
UNION ALL SELECT 344, 'NAICS', 31192, 'Coffee and Tea Manufacturing', 4, 339 FROM dual
UNION ALL SELECT 346, 'NAICS', 31193, 'Flavoring Syrup and Concentrate Manufacturing', 4, 339 FROM dual
UNION ALL SELECT 350, 'NAICS', 31199, 'All Other Food Manufacturing', 4, 339 FROM dual
UNION ALL SELECT 341, 'NAICS', 311911, 'Roasted Nuts and Peanut Butter Manufacturing', 5, 340 FROM dual
UNION ALL SELECT 342, 'NAICS', 311919, 'Other Snack Food Manufacturing', 5, 340 FROM dual
UNION ALL SELECT 343, 'NAICS', 311920, 'Coffee and Tea Manufacturing', 5, 344 FROM dual
UNION ALL SELECT 345, 'NAICS', 311930, 'Flavoring Syrup and Concentrate Manufacturing', 5, 346 FROM dual
UNION ALL SELECT 349, 'NAICS', 311942, 'Spice and Extract Manufacturing', 5, 347 FROM dual
UNION ALL SELECT 348, 'NAICS', 311941, 'Mayonnaise, Dressing, and Other Prepared Sauce Manufacturing', 5, 347 FROM dual
UNION ALL SELECT 351, 'NAICS', 311991, 'Perishable Prepared Food Manufacturing', 5, 350 FROM dual
UNION ALL SELECT 352, 'NAICS', 311999, 'All Other Miscellaneous Food Manufacturing', 5, 350 FROM dual
UNION ALL SELECT 354, 'NAICS', 3121, 'Beverage Manufacturing', 3, 353 FROM dual
UNION ALL SELECT 365, 'NAICS', 3122, 'Tobacco Manufacturing', 3, 353 FROM dual
UNION ALL SELECT 364, 'NAICS', 31214, 'Distilleries', 4, 354 FROM dual
UNION ALL SELECT 360, 'NAICS', 31212, 'Breweries', 4, 354 FROM dual
UNION ALL SELECT 355, 'NAICS', 31211, 'Soft Drink and Ice Manufacturing', 4, 354 FROM dual
UNION ALL SELECT 362, 'NAICS', 31213, 'Wineries', 4, 354 FROM dual
UNION ALL SELECT 356, 'NAICS', 312111, 'Soft Drink Manufacturing', 5, 355 FROM dual
UNION ALL SELECT 357, 'NAICS', 312112, 'Bottled Water Manufacturing', 5, 355 FROM dual
UNION ALL SELECT 358, 'NAICS', 312113, 'Ice Manufacturing', 5, 355 FROM dual
UNION ALL SELECT 359, 'NAICS', 312120, 'Breweries', 5, 360 FROM dual
UNION ALL SELECT 361, 'NAICS', 312130, 'Wineries', 5, 362 FROM dual
UNION ALL SELECT 363, 'NAICS', 312140, 'Distilleries', 5, 364 FROM dual
UNION ALL SELECT 367, 'NAICS', 31223, 'Tobacco Manufacturing', 4, 365 FROM dual
UNION ALL SELECT 366, 'NAICS', 312230, 'Tobacco Manufacturing', 5, 367 FROM dual
UNION ALL SELECT 381, 'NAICS', 3133, 'Textile and Fabric Finishing and Fabric Coating Mills', 3, 368 FROM dual
UNION ALL SELECT 369, 'NAICS', 3131, 'Fiber, Yarn, and Thread Mills', 3, 368 FROM dual
UNION ALL SELECT 372, 'NAICS', 3132, 'Fabric Mills', 3, 368 FROM dual
UNION ALL SELECT 371, 'NAICS', 31311, 'Fiber, Yarn, and Thread Mills', 4, 369 FROM dual
UNION ALL SELECT 370, 'NAICS', 313110, 'Fiber, Yarn, and Thread Mills', 5, 371 FROM dual
UNION ALL SELECT 376, 'NAICS', 31322, 'Narrow Fabric Mills and Schiffli Machine Embroidery', 4, 372 FROM dual
UNION ALL SELECT 378, 'NAICS', 31323, 'Nonwoven Fabric Mills', 4, 372 FROM dual
UNION ALL SELECT 374, 'NAICS', 31321, 'Broadwoven Fabric Mills', 4, 372 FROM dual
UNION ALL SELECT 380, 'NAICS', 31324, 'Knit Fabric Mills', 4, 372 FROM dual
UNION ALL SELECT 373, 'NAICS', 313210, 'Broadwoven Fabric Mills', 5, 374 FROM dual
UNION ALL SELECT 375, 'NAICS', 313220, 'Narrow Fabric Mills and Schiffli Machine Embroidery', 5, 376 FROM dual
UNION ALL SELECT 377, 'NAICS', 313230, 'Nonwoven Fabric Mills', 5, 378 FROM dual
UNION ALL SELECT 379, 'NAICS', 313240, 'Knit Fabric Mills', 5, 380 FROM dual
UNION ALL SELECT 385, 'NAICS', 31332, 'Fabric Coating Mills', 4, 381 FROM dual
UNION ALL SELECT 383, 'NAICS', 31331, 'Textile and Fabric Finishing Mills', 4, 381 FROM dual
UNION ALL SELECT 382, 'NAICS', 313310, 'Textile and Fabric Finishing Mills', 5, 383 FROM dual
UNION ALL SELECT 384, 'NAICS', 313320, 'Fabric Coating Mills', 5, 385 FROM dual
UNION ALL SELECT 387, 'NAICS', 3141, 'Textile Furnishings Mills', 3, 386 FROM dual
UNION ALL SELECT 392, 'NAICS', 3149, 'Other Textile Product Mills', 3, 386 FROM dual
UNION ALL SELECT 391, 'NAICS', 31412, 'Curtain and Linen Mills', 4, 387 FROM dual
UNION ALL SELECT 389, 'NAICS', 31411, 'Carpet and Rug Mills', 4, 387 FROM dual
UNION ALL SELECT 388, 'NAICS', 314110, 'Carpet and Rug Mills', 5, 389 FROM dual
UNION ALL SELECT 390, 'NAICS', 314120, 'Curtain and Linen Mills', 5, 391 FROM dual
UNION ALL SELECT 394, 'NAICS', 31491, 'Textile Bag and Canvas Mills', 4, 392 FROM dual
UNION ALL SELECT 395, 'NAICS', 31499, 'All Other Textile Product Mills', 4, 392 FROM dual
UNION ALL SELECT 393, 'NAICS', 314910, 'Textile Bag and Canvas Mills', 5, 394 FROM dual
UNION ALL SELECT 396, 'NAICS', 314994, 'Rope, Cordage, Twine, Tire Cord, and Tire Fabric Mills', 5, 395 FROM dual
UNION ALL SELECT 397, 'NAICS', 314999, 'All Other Miscellaneous Textile Product Mills', 5, 395 FROM dual
UNION ALL SELECT 399, 'NAICS', 3151, 'Apparel Knitting Mills', 3, 398 FROM dual
UNION ALL SELECT 404, 'NAICS', 3152, 'Cut and Sew Apparel Manufacturing', 3, 398 FROM dual
UNION ALL SELECT 413, 'NAICS', 3159, 'Apparel Accessories and Other Apparel Manufacturing', 3, 398 FROM dual
UNION ALL SELECT 403, 'NAICS', 31519, 'Other Apparel Knitting Mills', 4, 399 FROM dual
UNION ALL SELECT 401, 'NAICS', 31511, 'Hosiery and Sock Mills', 4, 399 FROM dual
UNION ALL SELECT 400, 'NAICS', 315110, 'Hosiery and Sock Mills', 5, 401 FROM dual
UNION ALL SELECT 402, 'NAICS', 315190, 'Other Apparel Knitting Mills', 5, 403 FROM dual
UNION ALL SELECT 412, 'NAICS', 31528, 'Other Cut and Sew Apparel Manufacturing', 4, 404 FROM dual
UNION ALL SELECT 406, 'NAICS', 31521, 'Cut and Sew Apparel Contractors', 4, 404 FROM dual
UNION ALL SELECT 408, 'NAICS', 31522, 'Men.s and Boys. Cut and Sew Apparel Manufacturing', 4, 404 FROM dual
UNION ALL SELECT 410, 'NAICS', 31524, 'Women.s, Girls., and Infants. Cut and Sew Apparel Manufacturing', 4, 404 FROM dual
UNION ALL SELECT 405, 'NAICS', 315210, 'Cut and Sew Apparel Contractors', 5, 406 FROM dual
UNION ALL SELECT 407, 'NAICS', 315220, 'Men.s and Boys. Cut and Sew Apparel Manufacturing', 5, 408 FROM dual
UNION ALL SELECT 409, 'NAICS', 315240, 'Women.s, Girls., and Infants. Cut and Sew Apparel Manufacturing', 5, 410 FROM dual
UNION ALL SELECT 411, 'NAICS', 315280, 'Other Cut and Sew Apparel Manufacturing', 5, 412 FROM dual
UNION ALL SELECT 415, 'NAICS', 31599, 'Apparel Accessories and Other Apparel Manufacturing', 4, 413 FROM dual
UNION ALL SELECT 414, 'NAICS', 315990, 'Apparel Accessories and Other Apparel Manufacturing', 5, 415 FROM dual
UNION ALL SELECT 420, 'NAICS', 3162, 'Footwear Manufacturing', 3, 416 FROM dual
UNION ALL SELECT 423, 'NAICS', 3169, 'Other Leather and Allied Product Manufacturing', 3, 416 FROM dual
UNION ALL SELECT 417, 'NAICS', 3161, 'Leather and Hide Tanning and Finishing', 3, 416 FROM dual
UNION ALL SELECT 419, 'NAICS', 31611, 'Leather and Hide Tanning and Finishing', 4, 417 FROM dual
UNION ALL SELECT 418, 'NAICS', 316110, 'Leather and Hide Tanning and Finishing', 5, 419 FROM dual
UNION ALL SELECT 422, 'NAICS', 31621, 'Footwear Manufacturing', 4, 420 FROM dual
UNION ALL SELECT 421, 'NAICS', 316210, 'Footwear Manufacturing', 5, 422 FROM dual
UNION ALL SELECT 424, 'NAICS', 31699, 'Other Leather and Allied Product Manufacturing', 4, 423 FROM dual
UNION ALL SELECT 426, 'NAICS', 316998, 'All Other Leather Good and Allied Product Manufacturing', 5, 424 FROM dual
UNION ALL SELECT 425, 'NAICS', 316992, 'Women''s Handbag and Purse Manufacturing', 5, 424 FROM dual
UNION ALL SELECT 439, 'NAICS', 3219, 'Other Wood Product Manufacturing', 3, 427 FROM dual
UNION ALL SELECT 428, 'NAICS', 3211, 'Sawmills and Wood Preservation', 3, 427 FROM dual
UNION ALL SELECT 432, 'NAICS', 3212, 'Veneer, Plywood, and Engineered Wood Product Manufacturing', 3, 427 FROM dual
UNION ALL SELECT 429, 'NAICS', 32111, 'Sawmills and Wood Preservation', 4, 428 FROM dual
UNION ALL SELECT 430, 'NAICS', 321113, 'Sawmills', 5, 429 FROM dual
UNION ALL SELECT 431, 'NAICS', 321114, 'Wood Preservation', 5, 429 FROM dual
UNION ALL SELECT 433, 'NAICS', 32121, 'Veneer, Plywood, and Engineered Wood Product Manufacturing', 4, 432 FROM dual
UNION ALL SELECT 435, 'NAICS', 321212, 'Softwood Veneer and Plywood Manufacturing', 5, 433 FROM dual
UNION ALL SELECT 434, 'NAICS', 321211, 'Hardwood Veneer and Plywood Manufacturing', 5, 433 FROM dual
UNION ALL SELECT 436, 'NAICS', 321213, 'Engineered Wood Member (except Truss) Manufacturing', 5, 433 FROM dual
UNION ALL SELECT 437, 'NAICS', 321214, 'Truss Manufacturing', 5, 433 FROM dual
UNION ALL SELECT 438, 'NAICS', 321219, 'Reconstituted Wood Product Manufacturing', 5, 433 FROM dual
UNION ALL SELECT 445, 'NAICS', 32192, 'Wood Container and Pallet Manufacturing', 4, 439 FROM dual
UNION ALL SELECT 440, 'NAICS', 32191, 'Millwork', 4, 439 FROM dual
UNION ALL SELECT 446, 'NAICS', 32199, 'All Other Wood Product Manufacturing', 4, 439 FROM dual
UNION ALL SELECT 442, 'NAICS', 321912, 'Cut Stock, Resawing Lumber, and Planing', 5, 440 FROM dual
UNION ALL SELECT 443, 'NAICS', 321918, 'Other Millwork (including Flooring)', 5, 440 FROM dual
UNION ALL SELECT 441, 'NAICS', 321911, 'Wood Window and Door Manufacturing', 5, 440 FROM dual
UNION ALL SELECT 444, 'NAICS', 321920, 'Wood Container and Pallet Manufacturing', 5, 445 FROM dual
UNION ALL SELECT 447, 'NAICS', 321991, 'Manufactured Home (Mobile Home) Manufacturing', 5, 446 FROM dual
UNION ALL SELECT 448, 'NAICS', 321992, 'Prefabricated Wood Building Manufacturing', 5, 446 FROM dual
UNION ALL SELECT 449, 'NAICS', 321999, 'All Other Miscellaneous Wood Product Manufacturing', 5, 446 FROM dual
UNION ALL SELECT 459, 'NAICS', 3222, 'Converted Paper Product Manufacturing', 3, 450 FROM dual
UNION ALL SELECT 451, 'NAICS', 3221, 'Pulp, Paper, and Paperboard Mills', 3, 450 FROM dual
UNION ALL SELECT 454, 'NAICS', 32212, 'Paper Mills', 4, 451 FROM dual
UNION ALL SELECT 453, 'NAICS', 32211, 'Pulp Mills', 4, 451 FROM dual
UNION ALL SELECT 458, 'NAICS', 32213, 'Paperboard Mills', 4, 451 FROM dual
UNION ALL SELECT 452, 'NAICS', 322110, 'Pulp Mills', 5, 453 FROM dual
UNION ALL SELECT 456, 'NAICS', 322122, 'Newsprint Mills', 5, 454 FROM dual
UNION ALL SELECT 455, 'NAICS', 322121, 'Paper (except Newsprint) Mills', 5, 454 FROM dual
UNION ALL SELECT 457, 'NAICS', 322130, 'Paperboard Mills', 5, 458 FROM dual
UNION ALL SELECT 460, 'NAICS', 32221, 'Paperboard Container Manufacturing', 4, 459 FROM dual
UNION ALL SELECT 467, 'NAICS', 32223, 'Stationery Product Manufacturing', 4, 459 FROM dual
UNION ALL SELECT 468, 'NAICS', 32229, 'Other Converted Paper Product Manufacturing', 4, 459 FROM dual
UNION ALL SELECT 465, 'NAICS', 32222, 'Paper Bag and Coated and Treated Paper Manufacturing', 4, 459 FROM dual
UNION ALL SELECT 463, 'NAICS', 322219, 'Other Paperboard Container Manufacturing', 5, 460 FROM dual
UNION ALL SELECT 462, 'NAICS', 322212, 'Folding Paperboard Box Manufacturing', 5, 460 FROM dual
UNION ALL SELECT 461, 'NAICS', 322211, 'Corrugated and Solid Fiber Box Manufacturing', 5, 460 FROM dual
UNION ALL SELECT 464, 'NAICS', 322220, 'Paper Bag and Coated and Treated Paper Manufacturing', 5, 465 FROM dual
UNION ALL SELECT 466, 'NAICS', 322230, 'Stationery Product Manufacturing', 5, 467 FROM dual
UNION ALL SELECT 469, 'NAICS', 322291, 'Sanitary Paper Product Manufacturing', 5, 468 FROM dual
UNION ALL SELECT 470, 'NAICS', 322299, 'All Other Converted Paper Product Manufacturing', 5, 468 FROM dual
UNION ALL SELECT 472, 'NAICS', 3231, 'Printing and Related Support Activities', 3, 471 FROM dual
UNION ALL SELECT 478, 'NAICS', 32312, 'Support Activities for Printing', 4, 472 FROM dual
UNION ALL SELECT 473, 'NAICS', 32311, 'Printing', 4, 472 FROM dual
UNION ALL SELECT 827, 'NAICS', 33593, 'Wiring Device Manufacturing', 4, 820 FROM dual
UNION ALL SELECT 474, 'NAICS', 323111, 'Commercial Printing (except Screen and Books)', 5, 473 FROM dual
UNION ALL SELECT 475, 'NAICS', 323113, 'Commercial Screen Printing', 5, 473 FROM dual
UNION ALL SELECT 476, 'NAICS', 323117, 'Books Printing', 5, 473 FROM dual
UNION ALL SELECT 477, 'NAICS', 323120, 'Support Activities for Printing', 5, 478 FROM dual
UNION ALL SELECT 480, 'NAICS', 3241, 'Petroleum and Coal Products Manufacturing', 3, 479 FROM dual
UNION ALL SELECT 482, 'NAICS', 32411, 'Petroleum Refineries', 4, 480 FROM dual
UNION ALL SELECT 486, 'NAICS', 32419, 'Other Petroleum and Coal Products Manufacturing', 4, 480 FROM dual
UNION ALL SELECT 483, 'NAICS', 32412, 'Asphalt Paving, Roofing, and Saturated Materials Manufacturing', 4, 480 FROM dual
UNION ALL SELECT 481, 'NAICS', 324110, 'Petroleum Refineries', 5, 482 FROM dual
UNION ALL SELECT 485, 'NAICS', 324122, 'Asphalt Shingle and Coating Materials Manufacturing', 5, 483 FROM dual
UNION ALL SELECT 484, 'NAICS', 324121, 'Asphalt Paving Mixture and Block Manufacturing', 5, 483 FROM dual
UNION ALL SELECT 488, 'NAICS', 324199, 'All Other Petroleum and Coal Products Manufacturing', 5, 486 FROM dual
UNION ALL SELECT 487, 'NAICS', 324191, 'Petroleum Lubricating Oil and Grease Manufacturing', 5, 486 FROM dual
UNION ALL SELECT 534, 'NAICS', 3259, 'Other Chemical Product and Preparation Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 522, 'NAICS', 3255, 'Paint, Coating, and Adhesive Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 509, 'NAICS', 3253, 'Pesticide, Fertilizer, and Other Agricultural Chemical Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 516, 'NAICS', 3254, 'Pharmaceutical and Medicine Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 503, 'NAICS', 3252, 'Resin, Synthetic Rubber, and Artificial Synthetic Fibers and Filaments Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 490, 'NAICS', 3251, 'Basic Chemical Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 527, 'NAICS', 3256, 'Soap, Cleaning Compound, and Toilet Preparation Manufacturing', 3, 489 FROM dual
UNION ALL SELECT 492, 'NAICS', 32511, 'Petrochemical Manufacturing', 4, 490 FROM dual
UNION ALL SELECT 494, 'NAICS', 32512, 'Industrial Gas Manufacturing', 4, 490 FROM dual
UNION ALL SELECT 496, 'NAICS', 32513, 'Synthetic Dye and Pigment Manufacturing', 4, 490 FROM dual
UNION ALL SELECT 498, 'NAICS', 32518, 'Other Basic Inorganic Chemical Manufacturing', 4, 490 FROM dual
UNION ALL SELECT 499, 'NAICS', 32519, 'Other Basic Organic Chemical Manufacturing', 4, 490 FROM dual
UNION ALL SELECT 491, 'NAICS', 325110, 'Petrochemical Manufacturing', 5, 492 FROM dual
UNION ALL SELECT 493, 'NAICS', 325120, 'Industrial Gas Manufacturing', 5, 494 FROM dual
UNION ALL SELECT 495, 'NAICS', 325130, 'Synthetic Dye and Pigment Manufacturing', 5, 496 FROM dual
UNION ALL SELECT 497, 'NAICS', 325180, 'Other Basic Inorganic Chemical Manufacturing', 5, 498 FROM dual
UNION ALL SELECT 501, 'NAICS', 325194, 'Cyclic Crude, Intermediate, and Gum and Wood Chemical Manufacturing', 5, 499 FROM dual
UNION ALL SELECT 500, 'NAICS', 325193, 'Ethyl Alcohol Manufacturing', 5, 499 FROM dual
UNION ALL SELECT 502, 'NAICS', 325199, 'All Other Basic Organic Chemical Manufacturing', 5, 499 FROM dual
UNION ALL SELECT 508, 'NAICS', 32522, 'Artificial and Synthetic Fibers and Filaments Manufacturing', 4, 503 FROM dual
UNION ALL SELECT 504, 'NAICS', 32521, 'Resin and Synthetic Rubber Manufacturing', 4, 503 FROM dual
UNION ALL SELECT 506, 'NAICS', 325212, 'Synthetic Rubber Manufacturing', 5, 504 FROM dual
UNION ALL SELECT 505, 'NAICS', 325211, 'Plastics Material and Resin Manufacturing', 5, 504 FROM dual
UNION ALL SELECT 507, 'NAICS', 325220, 'Artificial and Synthetic Fibers and Filaments Manufacturing', 5, 508 FROM dual
UNION ALL SELECT 515, 'NAICS', 32532, 'Pesticide and Other Agricultural Chemical Manufacturing', 4, 509 FROM dual
UNION ALL SELECT 510, 'NAICS', 32531, 'Fertilizer Manufacturing', 4, 509 FROM dual
UNION ALL SELECT 512, 'NAICS', 325312, 'Phosphatic Fertilizer Manufacturing', 5, 510 FROM dual
UNION ALL SELECT 513, 'NAICS', 325314, 'Fertilizer (Mixing Only) Manufacturing', 5, 510 FROM dual
UNION ALL SELECT 511, 'NAICS', 325311, 'Nitrogenous Fertilizer Manufacturing', 5, 510 FROM dual
UNION ALL SELECT 514, 'NAICS', 325320, 'Pesticide and Other Agricultural Chemical Manufacturing', 5, 515 FROM dual
UNION ALL SELECT 517, 'NAICS', 32541, 'Pharmaceutical and Medicine Manufacturing', 4, 516 FROM dual
UNION ALL SELECT 519, 'NAICS', 325412, 'Pharmaceutical Preparation Manufacturing', 5, 517 FROM dual
UNION ALL SELECT 520, 'NAICS', 325413, 'In-Vitro Diagnostic Substance Manufacturing', 5, 517 FROM dual
UNION ALL SELECT 521, 'NAICS', 325414, 'Biological Product (except Diagnostic) Manufacturing', 5, 517 FROM dual
UNION ALL SELECT 518, 'NAICS', 325411, 'Medicinal and Botanical Manufacturing', 5, 517 FROM dual
UNION ALL SELECT 524, 'NAICS', 32551, 'Paint and Coating Manufacturing', 4, 522 FROM dual
UNION ALL SELECT 526, 'NAICS', 32552, 'Adhesive Manufacturing', 4, 522 FROM dual
UNION ALL SELECT 523, 'NAICS', 325510, 'Paint and Coating Manufacturing', 5, 524 FROM dual
UNION ALL SELECT 525, 'NAICS', 325520, 'Adhesive Manufacturing', 5, 526 FROM dual
UNION ALL SELECT 533, 'NAICS', 32562, 'Toilet Preparation Manufacturing', 4, 527 FROM dual
UNION ALL SELECT 528, 'NAICS', 32561, 'Soap and Cleaning Compound Manufacturing', 4, 527 FROM dual
UNION ALL SELECT 530, 'NAICS', 325612, 'Polish and Other Sanitation Good Manufacturing', 5, 528 FROM dual
UNION ALL SELECT 531, 'NAICS', 325613, 'Surface Active Agent Manufacturing', 5, 528 FROM dual
UNION ALL SELECT 529, 'NAICS', 325611, 'Soap and Other Detergent Manufacturing', 5, 528 FROM dual
UNION ALL SELECT 532, 'NAICS', 325620, 'Toilet Preparation Manufacturing', 5, 533 FROM dual
UNION ALL SELECT 539, 'NAICS', 32599, 'All Other Chemical Product and Preparation Manufacturing', 4, 534 FROM dual
UNION ALL SELECT 538, 'NAICS', 32592, 'Explosives Manufacturing', 4, 534 FROM dual
UNION ALL SELECT 536, 'NAICS', 32591, 'Printing Ink Manufacturing', 4, 534 FROM dual
UNION ALL SELECT 535, 'NAICS', 325910, 'Printing Ink Manufacturing', 5, 536 FROM dual
UNION ALL SELECT 537, 'NAICS', 325920, 'Explosives Manufacturing', 5, 538 FROM dual
UNION ALL SELECT 540, 'NAICS', 325991, 'Custom Compounding of Purchased Resins', 5, 539 FROM dual
UNION ALL SELECT 541, 'NAICS', 325992, 'Photographic Film, Paper, Plate, and Chemical Manufacturing', 5, 539 FROM dual
UNION ALL SELECT 542, 'NAICS', 325998, 'All Other Miscellaneous Chemical Product and Preparation Manufacturing', 5, 539 FROM dual
UNION ALL SELECT 563, 'NAICS', 3262, 'Rubber Product Manufacturing', 3, 543 FROM dual
UNION ALL SELECT 544, 'NAICS', 3261, 'Plastics Product Manufacturing', 3, 543 FROM dual
UNION ALL SELECT 549, 'NAICS', 32612, 'Plastics Pipe, Pipe Fitting, and Unlaminated Profile Shape Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 545, 'NAICS', 32611, 'Plastics Packaging Materials and Unlaminated Film and Sheet Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 553, 'NAICS', 32613, 'Laminated Plastics Plate, Sheet (except Packaging), and Shape Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 555, 'NAICS', 32614, 'Polystyrene Foam Product Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 557, 'NAICS', 32615, 'Urethane and Other Foam Product (except Polystyrene) Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 559, 'NAICS', 32616, 'Plastics Bottle Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 560, 'NAICS', 32619, 'Other Plastics Product Manufacturing', 4, 544 FROM dual
UNION ALL SELECT 546, 'NAICS', 326111, 'Plastics Bag and Pouch Manufacturing', 5, 545 FROM dual
UNION ALL SELECT 548, 'NAICS', 326113, 'Unlaminated Plastics Film and Sheet (except Packaging) Manufacturing', 5, 545 FROM dual
UNION ALL SELECT 547, 'NAICS', 326112, 'Plastics Packaging Film and Sheet (including Laminated) Manufacturing', 5, 545 FROM dual
UNION ALL SELECT 551, 'NAICS', 326122, 'Plastics Pipe and Pipe Fitting Manufacturing', 5, 549 FROM dual
UNION ALL SELECT 550, 'NAICS', 326121, 'Unlaminated Plastics Profile Shape Manufacturing', 5, 549 FROM dual
UNION ALL SELECT 552, 'NAICS', 326130, 'Laminated Plastics Plate, Sheet (except Packaging), and Shape Manufacturing', 5, 553 FROM dual
UNION ALL SELECT 554, 'NAICS', 326140, 'Polystyrene Foam Product Manufacturing', 5, 555 FROM dual
UNION ALL SELECT 556, 'NAICS', 326150, 'Urethane and Other Foam Product (except Polystyrene) Manufacturing', 5, 557 FROM dual
UNION ALL SELECT 558, 'NAICS', 326160, 'Plastics Bottle Manufacturing', 5, 559 FROM dual
UNION ALL SELECT 562, 'NAICS', 326199, 'All Other Plastics Product Manufacturing', 5, 560 FROM dual
UNION ALL SELECT 561, 'NAICS', 326191, 'Plastics Plumbing Fixture Manufacturing', 5, 560 FROM dual
UNION ALL SELECT 568, 'NAICS', 32622, 'Rubber and Plastics Hoses and Belting Manufacturing', 4, 563 FROM dual
UNION ALL SELECT 569, 'NAICS', 32629, 'Other Rubber Product Manufacturing', 4, 563 FROM dual
UNION ALL SELECT 564, 'NAICS', 32621, 'Tire Manufacturing', 4, 563 FROM dual
UNION ALL SELECT 565, 'NAICS', 326211, 'Tire Manufacturing (except Retreading)', 5, 564 FROM dual
UNION ALL SELECT 566, 'NAICS', 326212, 'Tire Retreading', 5, 564 FROM dual
UNION ALL SELECT 567, 'NAICS', 326220, 'Rubber and Plastics Hoses and Belting Manufacturing', 5, 568 FROM dual
UNION ALL SELECT 571, 'NAICS', 326299, 'All Other Rubber Product Manufacturing', 5, 569 FROM dual
UNION ALL SELECT 570, 'NAICS', 326291, 'Rubber Product Manufacturing for Mechanical Use', 5, 569 FROM dual
UNION ALL SELECT 578, 'NAICS', 3272, 'Glass and Glass Product Manufacturing', 3, 572 FROM dual
UNION ALL SELECT 594, 'NAICS', 3274, 'Lime and Gypsum Product Manufacturing', 3, 572 FROM dual
UNION ALL SELECT 584, 'NAICS', 3273, 'Cement and Concrete Product Manufacturing', 3, 572 FROM dual
UNION ALL SELECT 599, 'NAICS', 3279, 'Other Nonmetallic Mineral Product Manufacturing', 3, 572 FROM dual
UNION ALL SELECT 573, 'NAICS', 3271, 'Clay Product and Refractory Manufacturing', 3, 572 FROM dual
UNION ALL SELECT 577, 'NAICS', 32712, 'Clay Building Material and Refractories Manufacturing', 4, 573 FROM dual
UNION ALL SELECT 575, 'NAICS', 32711, 'Pottery, Ceramics, and Plumbing Fixture Manufacturing', 4, 573 FROM dual
UNION ALL SELECT 574, 'NAICS', 327110, 'Pottery, Ceramics, and Plumbing Fixture Manufacturing', 5, 575 FROM dual
UNION ALL SELECT 576, 'NAICS', 327120, 'Clay Building Material and Refractories Manufacturing', 5, 577 FROM dual
UNION ALL SELECT 579, 'NAICS', 32721, 'Glass and Glass Product Manufacturing', 4, 578 FROM dual
UNION ALL SELECT 582, 'NAICS', 327213, 'Glass Container Manufacturing', 5, 579 FROM dual
UNION ALL SELECT 583, 'NAICS', 327215, 'Glass Product Manufacturing Made of Purchased Glass', 5, 579 FROM dual
UNION ALL SELECT 580, 'NAICS', 327211, 'Flat Glass Manufacturing', 5, 579 FROM dual
UNION ALL SELECT 645, 'NAICS', 332114, 'Custom Roll Forming', 5, 642 FROM dual
UNION ALL SELECT 581, 'NAICS', 327212, 'Other Pressed and Blown Glass and Glassware Manufacturing', 5, 579 FROM dual
UNION ALL SELECT 588, 'NAICS', 32732, 'Ready-Mix Concrete Manufacturing', 4, 584 FROM dual
UNION ALL SELECT 586, 'NAICS', 32731, 'Cement Manufacturing', 4, 584 FROM dual
UNION ALL SELECT 589, 'NAICS', 32733, 'Concrete Pipe, Brick, and Block Manufacturing', 4, 584 FROM dual
UNION ALL SELECT 593, 'NAICS', 32739, 'Other Concrete Product Manufacturing', 4, 584 FROM dual
UNION ALL SELECT 585, 'NAICS', 327310, 'Cement Manufacturing', 5, 586 FROM dual
UNION ALL SELECT 587, 'NAICS', 327320, 'Ready-Mix Concrete Manufacturing', 5, 588 FROM dual
UNION ALL SELECT 590, 'NAICS', 327331, 'Concrete Block and Brick Manufacturing', 5, 589 FROM dual
UNION ALL SELECT 591, 'NAICS', 327332, 'Concrete Pipe Manufacturing', 5, 589 FROM dual
UNION ALL SELECT 592, 'NAICS', 327390, 'Other Concrete Product Manufacturing', 5, 593 FROM dual
UNION ALL SELECT 596, 'NAICS', 32741, 'Lime Manufacturing', 4, 594 FROM dual
UNION ALL SELECT 598, 'NAICS', 32742, 'Gypsum Product Manufacturing', 4, 594 FROM dual
UNION ALL SELECT 595, 'NAICS', 327410, 'Lime Manufacturing', 5, 596 FROM dual
UNION ALL SELECT 597, 'NAICS', 327420, 'Gypsum Product Manufacturing', 5, 598 FROM dual
UNION ALL SELECT 602, 'NAICS', 32799, 'All Other Nonmetallic Mineral Product Manufacturing', 4, 599 FROM dual
UNION ALL SELECT 601, 'NAICS', 32791, 'Abrasive Product Manufacturing', 4, 599 FROM dual
UNION ALL SELECT 600, 'NAICS', 327910, 'Abrasive Product Manufacturing', 5, 601 FROM dual
UNION ALL SELECT 603, 'NAICS', 327991, 'Cut Stone and Stone Product Manufacturing', 5, 602 FROM dual
UNION ALL SELECT 604, 'NAICS', 327992, 'Ground or Treated Mineral and Earth Manufacturing', 5, 602 FROM dual
UNION ALL SELECT 606, 'NAICS', 327999, 'All Other Miscellaneous Nonmetallic Mineral Product Manufacturing', 5, 602 FROM dual
UNION ALL SELECT 605, 'NAICS', 327993, 'Mineral Wool Manufacturing', 5, 602 FROM dual
UNION ALL SELECT 623, 'NAICS', 3314, 'Nonferrous Metal (except Aluminum) Production and Processing', 3, 607 FROM dual
UNION ALL SELECT 611, 'NAICS', 3312, 'Steel Product Manufacturing from Purchased Steel', 3, 607 FROM dual
UNION ALL SELECT 608, 'NAICS', 3311, 'Iron and Steel Mills and Ferroalloy Manufacturing', 3, 607 FROM dual
UNION ALL SELECT 631, 'NAICS', 3315, 'Foundries', 3, 607 FROM dual
UNION ALL SELECT 617, 'NAICS', 3313, 'Alumina and Aluminum Production and Processing', 3, 607 FROM dual
UNION ALL SELECT 610, 'NAICS', 33111, 'Iron and Steel Mills and Ferroalloy Manufacturing', 4, 608 FROM dual
UNION ALL SELECT 609, 'NAICS', 331110, 'Iron and Steel Mills and Ferroalloy Manufacturing', 5, 610 FROM dual
UNION ALL SELECT 614, 'NAICS', 33122, 'Rolling and Drawing of Purchased Steel', 4, 611 FROM dual
UNION ALL SELECT 613, 'NAICS', 33121, 'Iron and Steel Pipe and Tube Manufacturing from Purchased Steel', 4, 611 FROM dual
UNION ALL SELECT 612, 'NAICS', 331210, 'Iron and Steel Pipe and Tube Manufacturing from Purchased Steel', 5, 613 FROM dual
UNION ALL SELECT 616, 'NAICS', 331222, 'Steel Wire Drawing', 5, 614 FROM dual
UNION ALL SELECT 615, 'NAICS', 331221, 'Rolled Steel Shape Manufacturing', 5, 614 FROM dual
UNION ALL SELECT 618, 'NAICS', 33131, 'Alumina and Aluminum Production and Processing', 4, 617 FROM dual
UNION ALL SELECT 622, 'NAICS', 331318, 'Other Aluminum Rolling, Drawing, and Extruding', 5, 618 FROM dual
UNION ALL SELECT 619, 'NAICS', 331313, 'Alumina Refining and Primary Aluminum Production', 5, 618 FROM dual
UNION ALL SELECT 620, 'NAICS', 331314, 'Secondary Smelting and Alloying of Aluminum', 5, 618 FROM dual
UNION ALL SELECT 621, 'NAICS', 331315, 'Aluminum Sheet, Plate, and Foil Manufacturing', 5, 618 FROM dual;

INSERT INTO "industry" (industry_id, industry_classification, industry_code, industry_description, depth, parent_id) 
          SELECT 627, 'NAICS', 33142, 'Copper Rolling, Drawing, Extruding, and Alloying', 4, 623 FROM dual
UNION ALL SELECT 628, 'NAICS', 33149, 'Nonferrous Metal (except Copper and Aluminum) Rolling, Drawing, Extruding, and Alloying', 4, 623 FROM dual
UNION ALL SELECT 625, 'NAICS', 33141, 'Nonferrous Metal (except Aluminum) Smelting and Refining', 4, 623 FROM dual
UNION ALL SELECT 624, 'NAICS', 331410, 'Nonferrous Metal (except Aluminum) Smelting and Refining', 5, 625 FROM dual
UNION ALL SELECT 626, 'NAICS', 331420, 'Copper Rolling, Drawing, Extruding, and Alloying', 5, 627 FROM dual
UNION ALL SELECT 630, 'NAICS', 331492, 'Secondary Smelting, Refining, and Alloying of Nonferrous Metal (except Copper and Aluminum)', 5, 628 FROM dual
UNION ALL SELECT 629, 'NAICS', 331491, 'Nonferrous Metal (except Copper and Aluminum) Rolling, Drawing, and Extruding', 5, 628 FROM dual
UNION ALL SELECT 632, 'NAICS', 33151, 'Ferrous Metal Foundries', 4, 631 FROM dual
UNION ALL SELECT 636, 'NAICS', 33152, 'Nonferrous Metal Foundries', 4, 631 FROM dual
UNION ALL SELECT 633, 'NAICS', 331511, 'Iron Foundries', 5, 632 FROM dual
UNION ALL SELECT 635, 'NAICS', 331513, 'Steel Foundries (except Investment)', 5, 632 FROM dual
UNION ALL SELECT 634, 'NAICS', 331512, 'Steel Investment Foundries', 5, 632 FROM dual
UNION ALL SELECT 639, 'NAICS', 331529, 'Other Nonferrous Metal Foundries (except Die-Casting)', 5, 636 FROM dual
UNION ALL SELECT 637, 'NAICS', 331523, 'Nonferrous Metal Die-Casting Foundries', 5, 636 FROM dual
UNION ALL SELECT 638, 'NAICS', 331524, 'Aluminum Foundries (except Die-Casting)', 5, 636 FROM dual
UNION ALL SELECT 672, 'NAICS', 3326, 'Spring and Wire Product Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 687, 'NAICS', 3329, 'Other Fabricated Metal Product Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 669, 'NAICS', 3325, 'Hardware Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 641, 'NAICS', 3321, 'Forging and Stamping', 3, 640 FROM dual
UNION ALL SELECT 648, 'NAICS', 3322, 'Cutlery and Handtool Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 652, 'NAICS', 3323, 'Architectural and Structural Metals Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 661, 'NAICS', 3324, 'Boiler, Tank, and Shipping Container Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 676, 'NAICS', 3327, 'Machine Shops, Turned Product, and Screw, Nut, and Bolt Manufacturing', 3, 640 FROM dual
UNION ALL SELECT 682, 'NAICS', 3328, 'Coating, Engraving, Heat Treating, and Allied Activities', 3, 640 FROM dual
UNION ALL SELECT 642, 'NAICS', 33211, 'Forging and Stamping', 4, 641 FROM dual
UNION ALL SELECT 647, 'NAICS', 332119, 'Metal Crown, Closure, and Other Metal Stamping (except Automotive)', 5, 642 FROM dual
UNION ALL SELECT 643, 'NAICS', 332111, 'Iron and Steel Forging', 5, 642 FROM dual
UNION ALL SELECT 644, 'NAICS', 332112, 'Nonferrous Forging', 5, 642 FROM dual
UNION ALL SELECT 646, 'NAICS', 332117, 'Powder Metallurgy Part Manufacturing', 5, 642 FROM dual
UNION ALL SELECT 649, 'NAICS', 33221, 'Cutlery and Handtool Manufacturing', 4, 648 FROM dual
UNION ALL SELECT 651, 'NAICS', 332216, 'Saw Blade and Handtool Manufacturing', 5, 649 FROM dual
UNION ALL SELECT 650, 'NAICS', 332215, 'Metal Kitchen Cookware, Utensil, Cutlery, and Flatware (except Precious) Manufacturing', 5, 649 FROM dual
UNION ALL SELECT 657, 'NAICS', 33232, 'Ornamental and Architectural Metal Products Manufacturing', 4, 652 FROM dual
UNION ALL SELECT 653, 'NAICS', 33231, 'Plate Work and Fabricated Structural Product Manufacturing', 4, 652 FROM dual
UNION ALL SELECT 656, 'NAICS', 332313, 'Plate Work Manufacturing', 5, 653 FROM dual
UNION ALL SELECT 654, 'NAICS', 332311, 'Prefabricated Metal Building and Component Manufacturing', 5, 653 FROM dual
UNION ALL SELECT 655, 'NAICS', 332312, 'Fabricated Structural Metal Manufacturing', 5, 653 FROM dual
UNION ALL SELECT 660, 'NAICS', 332323, 'Ornamental and Architectural Metal Work Manufacturing', 5, 657 FROM dual
UNION ALL SELECT 658, 'NAICS', 332321, 'Metal Window and Door Manufacturing', 5, 657 FROM dual
UNION ALL SELECT 659, 'NAICS', 332322, 'Sheet Metal Work Manufacturing', 5, 657 FROM dual
UNION ALL SELECT 665, 'NAICS', 33242, 'Metal Tank (Heavy Gauge) Manufacturing', 4, 661 FROM dual
UNION ALL SELECT 663, 'NAICS', 33241, 'Power Boiler and Heat Exchanger Manufacturing', 4, 661 FROM dual
UNION ALL SELECT 666, 'NAICS', 33243, 'Metal Can, Box, and Other Metal Container (Light Gauge) Manufacturing', 4, 661 FROM dual
UNION ALL SELECT 662, 'NAICS', 332410, 'Power Boiler and Heat Exchanger Manufacturing', 5, 663 FROM dual
UNION ALL SELECT 664, 'NAICS', 332420, 'Metal Tank (Heavy Gauge) Manufacturing', 5, 665 FROM dual
UNION ALL SELECT 667, 'NAICS', 332431, 'Metal Can Manufacturing', 5, 666 FROM dual
UNION ALL SELECT 668, 'NAICS', 332439, 'Other Metal Container Manufacturing', 5, 666 FROM dual
UNION ALL SELECT 671, 'NAICS', 33251, 'Hardware Manufacturing', 4, 669 FROM dual
UNION ALL SELECT 670, 'NAICS', 332510, 'Hardware Manufacturing', 5, 671 FROM dual
UNION ALL SELECT 673, 'NAICS', 33261, 'Spring and Wire Product Manufacturing', 4, 672 FROM dual
UNION ALL SELECT 674, 'NAICS', 332613, 'Spring Manufacturing', 5, 673 FROM dual
UNION ALL SELECT 675, 'NAICS', 332618, 'Other Fabricated Wire Product Manufacturing', 5, 673 FROM dual
UNION ALL SELECT 678, 'NAICS', 33271, 'Machine Shops', 4, 676 FROM dual
UNION ALL SELECT 679, 'NAICS', 33272, 'Turned Product and Screw, Nut, and Bolt Manufacturing', 4, 676 FROM dual
UNION ALL SELECT 677, 'NAICS', 332710, 'Machine Shops', 5, 678 FROM dual
UNION ALL SELECT 680, 'NAICS', 332721, 'Precision Turned Product Manufacturing', 5, 679 FROM dual
UNION ALL SELECT 681, 'NAICS', 332722, 'Bolt, Nut, Screw, Rivet, and Washer Manufacturing', 5, 679 FROM dual
UNION ALL SELECT 683, 'NAICS', 33281, 'Coating, Engraving, Heat Treating, and Allied Activities', 4, 682 FROM dual
UNION ALL SELECT 685, 'NAICS', 332812, 'Metal Coating, Engraving (except Jewelry and Silverware), and Allied Services to Manufacturers', 5, 683 FROM dual
UNION ALL SELECT 684, 'NAICS', 332811, 'Metal Heat Treating', 5, 683 FROM dual
UNION ALL SELECT 686, 'NAICS', 332813, 'Electroplating, Plating, Polishing, Anodizing, and Coloring', 5, 683 FROM dual
UNION ALL SELECT 688, 'NAICS', 33291, 'Metal Valve Manufacturing', 4, 687 FROM dual
UNION ALL SELECT 693, 'NAICS', 33299, 'All Other Fabricated Metal Product Manufacturing', 4, 687 FROM dual
UNION ALL SELECT 689, 'NAICS', 332911, 'Industrial Valve Manufacturing', 5, 688 FROM dual
UNION ALL SELECT 692, 'NAICS', 332919, 'Other Metal Valve and Pipe Fitting Manufacturing', 5, 688 FROM dual
UNION ALL SELECT 691, 'NAICS', 332913, 'Plumbing Fixture Fitting and Trim Manufacturing', 5, 688 FROM dual
UNION ALL SELECT 690, 'NAICS', 332912, 'Fluid Power Valve and Hose Fitting Manufacturing', 5, 688 FROM dual
UNION ALL SELECT 694, 'NAICS', 332991, 'Ball and Roller Bearing Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 697, 'NAICS', 332994, 'Small Arms, Ordnance, and Ordnance Accessories Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 698, 'NAICS', 332996, 'Fabricated Pipe and Pipe Fitting Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 699, 'NAICS', 332999, 'All Other Miscellaneous Fabricated Metal Product Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 696, 'NAICS', 332993, 'Ammunition (except Small Arms) Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 695, 'NAICS', 332992, 'Small Arms Ammunition Manufacturing', 5, 693 FROM dual
UNION ALL SELECT 727, 'NAICS', 3335, 'Metalworking Machinery Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 701, 'NAICS', 3331, 'Agriculture, Construction, and Mining Machinery Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 710, 'NAICS', 3332, 'Industrial Machinery Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 722, 'NAICS', 3334, 'Ventilation, Heating, Air-Conditioning, and Commercial Refrigeration Equipment Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 734, 'NAICS', 3336, 'Engine, Turbine, and Power Transmission Equipment Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 717, 'NAICS', 3333, 'Commercial and Service Industry Machinery Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 740, 'NAICS', 3339, 'Other General Purpose Machinery Manufacturing', 3, 700 FROM dual
UNION ALL SELECT 702, 'NAICS', 33311, 'Agricultural Implement Manufacturing', 4, 701 FROM dual
UNION ALL SELECT 706, 'NAICS', 33312, 'Construction Machinery Manufacturing', 4, 701 FROM dual
UNION ALL SELECT 707, 'NAICS', 33313, 'Mining and Oil and Gas Field Machinery Manufacturing', 4, 701 FROM dual
UNION ALL SELECT 704, 'NAICS', 333112, 'Lawn and Garden Tractor and Home Lawn and Garden Equipment Manufacturing', 5, 702 FROM dual
UNION ALL SELECT 703, 'NAICS', 333111, 'Farm Machinery and Equipment Manufacturing', 5, 702 FROM dual
UNION ALL SELECT 705, 'NAICS', 333120, 'Construction Machinery Manufacturing', 5, 706 FROM dual
UNION ALL SELECT 708, 'NAICS', 333131, 'Mining Machinery and Equipment Manufacturing', 5, 707 FROM dual
UNION ALL SELECT 709, 'NAICS', 333132, 'Oil and Gas Field Machinery and Equipment Manufacturing', 5, 707 FROM dual
UNION ALL SELECT 711, 'NAICS', 33324, 'Industrial Machinery Manufacturing', 4, 710 FROM dual
UNION ALL SELECT 712, 'NAICS', 333241, 'Food Product Machinery Manufacturing', 5, 711 FROM dual
UNION ALL SELECT 713, 'NAICS', 333242, 'Semiconductor Machinery Manufacturing', 5, 711 FROM dual
UNION ALL SELECT 714, 'NAICS', 333243, 'Sawmill, Woodworking, and Paper Machinery Manufacturing', 5, 711 FROM dual
UNION ALL SELECT 715, 'NAICS', 333244, 'Printing Machinery and Equipment Manufacturing', 5, 711 FROM dual
UNION ALL SELECT 716, 'NAICS', 333249, 'Other Industrial Machinery Manufacturing', 5, 711 FROM dual
UNION ALL SELECT 718, 'NAICS', 33331, 'Commercial and Service Industry Machinery Manufacturing', 4, 717 FROM dual
UNION ALL SELECT 720, 'NAICS', 333316, 'Photographic and Photocopying Equipment Manufacturing', 5, 718 FROM dual
UNION ALL SELECT 721, 'NAICS', 333318, 'Other Commercial and Service Industry Machinery Manufacturing', 5, 718 FROM dual
UNION ALL SELECT 719, 'NAICS', 333314, 'Optical Instrument and Lens Manufacturing', 5, 718 FROM dual
UNION ALL SELECT 723, 'NAICS', 33341, 'Ventilation, Heating, Air-Conditioning, and Commercial Refrigeration Equipment Manufacturing', 4, 722 FROM dual
UNION ALL SELECT 725, 'NAICS', 333414, 'Heating Equipment (except Warm Air Furnaces) Manufacturing', 5, 723 FROM dual
UNION ALL SELECT 724, 'NAICS', 333413, 'Industrial and Commercial Fan and Blower and Air Purification Equipment Manufacturing', 5, 723 FROM dual
UNION ALL SELECT 726, 'NAICS', 333415, 'Air-Conditioning and Warm Air Heating Equipment and Commercial and Industrial Refrigeration Equipment Manufacturing', 5, 723 FROM dual
UNION ALL SELECT 728, 'NAICS', 33351, 'Metalworking Machinery Manufacturing', 4, 727 FROM dual
UNION ALL SELECT 732, 'NAICS', 333517, 'Machine Tool Manufacturing', 5, 728 FROM dual
UNION ALL SELECT 733, 'NAICS', 333519, 'Rolling Mill and Other Metalworking Machinery Manufacturing', 5, 728 FROM dual
UNION ALL SELECT 730, 'NAICS', 333514, 'Special Die and Tool, Die Set, Jig, and Fixture Manufacturing', 5, 728 FROM dual
UNION ALL SELECT 731, 'NAICS', 333515, 'Cutting Tool and Machine Tool Accessory Manufacturing', 5, 728 FROM dual
UNION ALL SELECT 729, 'NAICS', 333511, 'Industrial Mold Manufacturing', 5, 728 FROM dual
UNION ALL SELECT 735, 'NAICS', 33361, 'Engine, Turbine, and Power Transmission Equipment Manufacturing', 4, 734 FROM dual
UNION ALL SELECT 736, 'NAICS', 333611, 'Turbine and Turbine Generator Set Units Manufacturing', 5, 735 FROM dual
UNION ALL SELECT 739, 'NAICS', 333618, 'Other Engine Equipment Manufacturing', 5, 735 FROM dual
UNION ALL SELECT 737, 'NAICS', 333612, 'Speed Changer, Industrial High-Speed Drive, and Gear Manufacturing', 5, 735 FROM dual
UNION ALL SELECT 738, 'NAICS', 333613, 'Mechanical Power Transmission Equipment Manufacturing', 5, 735 FROM dual
UNION ALL SELECT 745, 'NAICS', 33392, 'Material Handling Equipment Manufacturing', 4, 740 FROM dual
UNION ALL SELECT 750, 'NAICS', 33399, 'All Other General Purpose Machinery Manufacturing', 4, 740 FROM dual
UNION ALL SELECT 741, 'NAICS', 33391, 'Pump and Compressor Manufacturing', 4, 740 FROM dual
UNION ALL SELECT 742, 'NAICS', 333911, 'Pump and Pumping Equipment Manufacturing', 5, 741 FROM dual
UNION ALL SELECT 743, 'NAICS', 333912, 'Air and Gas Compressor Manufacturing', 5, 741 FROM dual
UNION ALL SELECT 744, 'NAICS', 333913, 'Measuring and Dispensing Pump Manufacturing', 5, 741 FROM dual
UNION ALL SELECT 746, 'NAICS', 333921, 'Elevator and Moving Stairway Manufacturing', 5, 745 FROM dual
UNION ALL SELECT 748, 'NAICS', 333923, 'Overhead Traveling Crane, Hoist, and Monorail System Manufacturing', 5, 745 FROM dual
UNION ALL SELECT 749, 'NAICS', 333924, 'Industrial Truck, Tractor, Trailer, and Stacker Machinery Manufacturing', 5, 745 FROM dual
UNION ALL SELECT 747, 'NAICS', 333922, 'Conveyor and Conveying Equipment Manufacturing', 5, 745 FROM dual
UNION ALL SELECT 754, 'NAICS', 333994, 'Industrial Process Furnace and Oven Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 751, 'NAICS', 333991, 'Power-Driven Handtool Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 752, 'NAICS', 333992, 'Welding and Soldering Equipment Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 753, 'NAICS', 333993, 'Packaging Machinery Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 755, 'NAICS', 333995, 'Fluid Power Cylinder and Actuator Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 756, 'NAICS', 333996, 'Fluid Power Pump and Motor Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 757, 'NAICS', 333997, 'Scale and Balance Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 758, 'NAICS', 333999, 'All Other Miscellaneous General Purpose Machinery Manufacturing', 5, 750 FROM dual
UNION ALL SELECT 794, 'NAICS', 3346, 'Manufacturing and Reproducing Magnetic and Optical Media', 3, 759 FROM dual
UNION ALL SELECT 775, 'NAICS', 3344, 'Semiconductor and Other Electronic Component Manufacturing', 3, 759 FROM dual
UNION ALL SELECT 772, 'NAICS', 3343, 'Audio and Video Equipment Manufacturing', 3, 759 FROM dual
UNION ALL SELECT 783, 'NAICS', 3345, 'Navigational, Measuring, Electromedical, and Control Instruments Manufacturing', 3, 759 FROM dual
UNION ALL SELECT 765, 'NAICS', 3342, 'Communications Equipment Manufacturing', 3, 759 FROM dual
UNION ALL SELECT 760, 'NAICS', 3341, 'Computer and Peripheral Equipment Manufacturing', 3, 759 FROM dual
UNION ALL SELECT 761, 'NAICS', 33411, 'Computer and Peripheral Equipment Manufacturing', 4, 760 FROM dual
UNION ALL SELECT 763, 'NAICS', 334112, 'Computer Storage Device Manufacturing', 5, 761 FROM dual
UNION ALL SELECT 762, 'NAICS', 334111, 'Electronic Computer Manufacturing', 5, 761 FROM dual
UNION ALL SELECT 764, 'NAICS', 334118, 'Computer Terminal and Other Computer Peripheral Equipment Manufacturing', 5, 761 FROM dual
UNION ALL SELECT 767, 'NAICS', 33421, 'Telephone Apparatus Manufacturing', 4, 765 FROM dual
UNION ALL SELECT 769, 'NAICS', 33422, 'Radio and Television Broadcasting and Wireless Communications Equipment Manufacturing', 4, 765 FROM dual
UNION ALL SELECT 771, 'NAICS', 33429, 'Other Communications Equipment Manufacturing', 4, 765 FROM dual
UNION ALL SELECT 766, 'NAICS', 334210, 'Telephone Apparatus Manufacturing', 5, 767 FROM dual
UNION ALL SELECT 768, 'NAICS', 334220, 'Radio and Television Broadcasting and Wireless Communications Equipment Manufacturing', 5, 769 FROM dual
UNION ALL SELECT 770, 'NAICS', 334290, 'Other Communications Equipment Manufacturing', 5, 771 FROM dual
UNION ALL SELECT 774, 'NAICS', 33431, 'Audio and Video Equipment Manufacturing', 4, 772 FROM dual
UNION ALL SELECT 773, 'NAICS', 334310, 'Audio and Video Equipment Manufacturing', 5, 774 FROM dual
UNION ALL SELECT 776, 'NAICS', 33441, 'Semiconductor and Other Electronic Component Manufacturing', 4, 775 FROM dual
UNION ALL SELECT 780, 'NAICS', 334417, 'Electronic Connector Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 779, 'NAICS', 334416, 'Capacitor, Resistor, Coil, Transformer, and Other Inductor Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 781, 'NAICS', 334418, 'Printed Circuit Assembly (Electronic Assembly) Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 777, 'NAICS', 334412, 'Bare Printed Circuit Board Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 778, 'NAICS', 334413, 'Semiconductor and Related Device Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 782, 'NAICS', 334419, 'Other Electronic Component Manufacturing', 5, 776 FROM dual
UNION ALL SELECT 785, 'NAICS', 33451, 'Navigational, Measuring, Electromedical, and Control Instruments Manufacturing', 4, 783 FROM dual
UNION ALL SELECT 792, 'NAICS', 334517, 'Irradiation Apparatus Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 790, 'NAICS', 334515, 'Instrument Manufacturing for Measuring and Testing Electricity and Electrical Signals', 5, 785 FROM dual
UNION ALL SELECT 789, 'NAICS', 334514, 'Totalizing Fluid Meter and Counting Device Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 788, 'NAICS', 334513, 'Instruments and Related Products Manufacturing for Measuring, Displaying, and Controlling Industrial Process Variables', 5, 785 FROM dual
UNION ALL SELECT 787, 'NAICS', 334512, 'Automatic Environmental Control Manufacturing for Residential, Commercial, and Appliance Use', 5, 785 FROM dual
UNION ALL SELECT 786, 'NAICS', 334511, 'Search, Detection, Navigation, Guidance, Aeronautical, and Nautical System and Instrument Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 784, 'NAICS', 334510, 'Electromedical and Electrotherapeutic Apparatus Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 793, 'NAICS', 334519, 'Other Measuring and Controlling Device Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 791, 'NAICS', 334516, 'Analytical Laboratory Instrument Manufacturing', 5, 785 FROM dual
UNION ALL SELECT 795, 'NAICS', 33461, 'Manufacturing and Reproducing Magnetic and Optical Media', 4, 794 FROM dual
UNION ALL SELECT 797, 'NAICS', 334614, 'Software and Other Prerecorded Compact Disc, Tape, and Record Reproducing', 5, 795 FROM dual
UNION ALL SELECT 796, 'NAICS', 334613, 'Blank Magnetic and Optical Recording Media Manufacturing', 5, 795 FROM dual
UNION ALL SELECT 820, 'NAICS', 3359, 'Other Electrical Equipment and Component Manufacturing', 3, 798 FROM dual
UNION ALL SELECT 799, 'NAICS', 3351, 'Electric Lighting Equipment Manufacturing', 3, 798 FROM dual
UNION ALL SELECT 814, 'NAICS', 3353, 'Electrical Equipment Manufacturing', 3, 798 FROM dual
UNION ALL SELECT 806, 'NAICS', 3352, 'Household Appliance Manufacturing', 3, 798 FROM dual
UNION ALL SELECT 801, 'NAICS', 33511, 'Electric Lamp Bulb and Part Manufacturing', 4, 799 FROM dual
UNION ALL SELECT 802, 'NAICS', 33512, 'Lighting Fixture Manufacturing', 4, 799 FROM dual
UNION ALL SELECT 800, 'NAICS', 335110, 'Electric Lamp Bulb and Part Manufacturing', 5, 801 FROM dual
UNION ALL SELECT 804, 'NAICS', 335122, 'Commercial, Industrial, and Institutional Electric Lighting Fixture Manufacturing', 5, 802 FROM dual
UNION ALL SELECT 803, 'NAICS', 335121, 'Residential Electric Lighting Fixture Manufacturing', 5, 802 FROM dual
UNION ALL SELECT 805, 'NAICS', 335129, 'Other Lighting Equipment Manufacturing', 5, 802 FROM dual
UNION ALL SELECT 809, 'NAICS', 33522, 'Major Appliance Manufacturing', 4, 806 FROM dual
UNION ALL SELECT 808, 'NAICS', 33521, 'Small Electrical Appliance Manufacturing', 4, 806 FROM dual
UNION ALL SELECT 807, 'NAICS', 335210, 'Small Electrical Appliance Manufacturing', 5, 808 FROM dual
UNION ALL SELECT 810, 'NAICS', 335221, 'Household Cooking Appliance Manufacturing', 5, 809 FROM dual
UNION ALL SELECT 813, 'NAICS', 335228, 'Other Major Household Appliance Manufacturing', 5, 809 FROM dual
UNION ALL SELECT 812, 'NAICS', 335224, 'Household Laundry Equipment Manufacturing', 5, 809 FROM dual
UNION ALL SELECT 811, 'NAICS', 335222, 'Household Refrigerator and Home Freezer Manufacturing', 5, 809 FROM dual
UNION ALL SELECT 815, 'NAICS', 33531, 'Electrical Equipment Manufacturing', 4, 814 FROM dual
UNION ALL SELECT 816, 'NAICS', 335311, 'Power, Distribution, and Specialty Transformer Manufacturing', 5, 815 FROM dual
UNION ALL SELECT 817, 'NAICS', 335312, 'Motor and Generator Manufacturing', 5, 815 FROM dual
UNION ALL SELECT 819, 'NAICS', 335314, 'Relay and Industrial Control Manufacturing', 5, 815 FROM dual
UNION ALL SELECT 818, 'NAICS', 335313, 'Switchgear and Switchboard Apparatus Manufacturing', 5, 815 FROM dual
UNION ALL SELECT 824, 'NAICS', 33592, 'Communication and Energy Wire and Cable Manufacturing', 4, 820 FROM dual
UNION ALL SELECT 830, 'NAICS', 33599, 'All Other Electrical Equipment and Component Manufacturing', 4, 820 FROM dual
UNION ALL SELECT 821, 'NAICS', 33591, 'Battery Manufacturing', 4, 820 FROM dual
UNION ALL SELECT 823, 'NAICS', 335912, 'Primary Battery Manufacturing', 5, 821 FROM dual
UNION ALL SELECT 822, 'NAICS', 335911, 'Storage Battery Manufacturing', 5, 821 FROM dual
UNION ALL SELECT 825, 'NAICS', 335921, 'Fiber Optic Cable Manufacturing', 5, 824 FROM dual
UNION ALL SELECT 826, 'NAICS', 335929, 'Other Communication and Energy Wire Manufacturing', 5, 824 FROM dual
UNION ALL SELECT 828, 'NAICS', 335931, 'Current-Carrying Wiring Device Manufacturing', 5, 827 FROM dual
UNION ALL SELECT 829, 'NAICS', 335932, 'Noncurrent-Carrying Wiring Device Manufacturing', 5, 827 FROM dual
UNION ALL SELECT 831, 'NAICS', 335991, 'Carbon and Graphite Product Manufacturing', 5, 830 FROM dual
UNION ALL SELECT 832, 'NAICS', 335999, 'All Other Miscellaneous Electrical Equipment and Component Manufacturing', 5, 830 FROM dual
UNION ALL SELECT 846, 'NAICS', 3363, 'Motor Vehicle Parts Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 863, 'NAICS', 3364, 'Aerospace Product and Parts Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 874, 'NAICS', 3366, 'Ship and Boat Building', 3, 833 FROM dual
UNION ALL SELECT 834, 'NAICS', 3361, 'Motor Vehicle Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 878, 'NAICS', 3369, 'Other Transportation Equipment Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 840, 'NAICS', 3362, 'Motor Vehicle Body and Trailer Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 871, 'NAICS', 3365, 'Railroad Rolling Stock Manufacturing', 3, 833 FROM dual
UNION ALL SELECT 835, 'NAICS', 33611, 'Automobile and Light Duty Motor Vehicle Manufacturing', 4, 834 FROM dual
UNION ALL SELECT 839, 'NAICS', 33612, 'Heavy Duty Truck Manufacturing', 4, 834 FROM dual
UNION ALL SELECT 837, 'NAICS', 336112, 'Light Truck and Utility Vehicle Manufacturing', 5, 835 FROM dual
UNION ALL SELECT 836, 'NAICS', 336111, 'Automobile Manufacturing', 5, 835 FROM dual
UNION ALL SELECT 838, 'NAICS', 336120, 'Heavy Duty Truck Manufacturing', 5, 839 FROM dual
UNION ALL SELECT 841, 'NAICS', 33621, 'Motor Vehicle Body and Trailer Manufacturing', 4, 840 FROM dual
UNION ALL SELECT 843, 'NAICS', 336212, 'Truck Trailer Manufacturing', 5, 841 FROM dual
UNION ALL SELECT 845, 'NAICS', 336214, 'Travel Trailer and Camper Manufacturing', 5, 841 FROM dual
UNION ALL SELECT 844, 'NAICS', 336213, 'Motor Home Manufacturing', 5, 841 FROM dual
UNION ALL SELECT 842, 'NAICS', 336211, 'Motor Vehicle Body Manufacturing', 5, 841 FROM dual
UNION ALL SELECT 856, 'NAICS', 33635, 'Motor Vehicle Transmission and Power Train Parts Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 848, 'NAICS', 33631, 'Motor Vehicle Gasoline Engine and Engine Parts Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 862, 'NAICS', 33639, 'Other Motor Vehicle Parts Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 850, 'NAICS', 33632, 'Motor Vehicle Electrical and Electronic Equipment Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 860, 'NAICS', 33637, 'Motor Vehicle Metal Stamping', 4, 846 FROM dual
UNION ALL SELECT 852, 'NAICS', 33633, 'Motor Vehicle Steering and Suspension Components (except Spring) Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 858, 'NAICS', 33636, 'Motor Vehicle Seating and Interior Trim Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 854, 'NAICS', 33634, 'Motor Vehicle Brake System Manufacturing', 4, 846 FROM dual
UNION ALL SELECT 847, 'NAICS', 336310, 'Motor Vehicle Gasoline Engine and Engine Parts Manufacturing', 5, 848 FROM dual
UNION ALL SELECT 849, 'NAICS', 336320, 'Motor Vehicle Electrical and Electronic Equipment Manufacturing', 5, 850 FROM dual
UNION ALL SELECT 851, 'NAICS', 336330, 'Motor Vehicle Steering and Suspension Components (except Spring) Manufacturing', 5, 852 FROM dual
UNION ALL SELECT 853, 'NAICS', 336340, 'Motor Vehicle Brake System Manufacturing', 5, 854 FROM dual
UNION ALL SELECT 855, 'NAICS', 336350, 'Motor Vehicle Transmission and Power Train Parts Manufacturing', 5, 856 FROM dual
UNION ALL SELECT 857, 'NAICS', 336360, 'Motor Vehicle Seating and Interior Trim Manufacturing', 5, 858 FROM dual
UNION ALL SELECT 859, 'NAICS', 336370, 'Motor Vehicle Metal Stamping', 5, 860 FROM dual
UNION ALL SELECT 861, 'NAICS', 336390, 'Other Motor Vehicle Parts Manufacturing', 5, 862 FROM dual
UNION ALL SELECT 864, 'NAICS', 33641, 'Aerospace Product and Parts Manufacturing', 4, 863 FROM dual
UNION ALL SELECT 865, 'NAICS', 336411, 'Aircraft Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 869, 'NAICS', 336415, 'Guided Missile and Space Vehicle Propulsion Unit and Propulsion Unit Parts Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 870, 'NAICS', 336419, 'Other Guided Missile and Space Vehicle Parts and Auxiliary Equipment Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 867, 'NAICS', 336413, 'Other Aircraft Parts and Auxiliary Equipment Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 866, 'NAICS', 336412, 'Aircraft Engine and Engine Parts Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 868, 'NAICS', 336414, 'Guided Missile and Space Vehicle Manufacturing', 5, 864 FROM dual
UNION ALL SELECT 873, 'NAICS', 33651, 'Railroad Rolling Stock Manufacturing', 4, 871 FROM dual
UNION ALL SELECT 872, 'NAICS', 336510, 'Railroad Rolling Stock Manufacturing', 5, 873 FROM dual
UNION ALL SELECT 875, 'NAICS', 33661, 'Ship and Boat Building', 4, 874 FROM dual
UNION ALL SELECT 876, 'NAICS', 336611, 'Ship Building and Repairing', 5, 875 FROM dual
UNION ALL SELECT 877, 'NAICS', 336612, 'Boat Building', 5, 875 FROM dual
UNION ALL SELECT 879, 'NAICS', 33699, 'Other Transportation Equipment Manufacturing', 4, 878 FROM dual
UNION ALL SELECT 882, 'NAICS', 336999, 'All Other Transportation Equipment Manufacturing', 5, 879 FROM dual
UNION ALL SELECT 881, 'NAICS', 336992, 'Military Armored Vehicle, Tank, and Tank Component Manufacturing', 5, 879 FROM dual
UNION ALL SELECT 880, 'NAICS', 336991, 'Motorcycle, Bicycle, and Parts Manufacturing', 5, 879 FROM dual
UNION ALL SELECT 884, 'NAICS', 3371, 'Household and Institutional Furniture and Kitchen Cabinet Manufacturing', 3, 883 FROM dual
UNION ALL SELECT 893, 'NAICS', 3372, 'Office Furniture (including Fixtures) Manufacturing', 3, 883 FROM dual
UNION ALL SELECT 899, 'NAICS', 3379, 'Other Furniture Related Product Manufacturing', 3, 883 FROM dual
UNION ALL SELECT 886, 'NAICS', 33711, 'Wood Kitchen Cabinet and Countertop Manufacturing', 4, 884 FROM dual
UNION ALL SELECT 887, 'NAICS', 33712, 'Household and Institutional Furniture Manufacturing', 4, 884 FROM dual
UNION ALL SELECT 885, 'NAICS', 337110, 'Wood Kitchen Cabinet and Countertop Manufacturing', 5, 886 FROM dual
UNION ALL SELECT 892, 'NAICS', 337127, 'Institutional Furniture Manufacturing', 5, 887 FROM dual
UNION ALL SELECT 889, 'NAICS', 337122, 'Nonupholstered Wood Household Furniture Manufacturing', 5, 887 FROM dual
UNION ALL SELECT 888, 'NAICS', 337121, 'Upholstered Household Furniture Manufacturing', 5, 887 FROM dual
UNION ALL SELECT 890, 'NAICS', 337124, 'Metal Household Furniture Manufacturing', 5, 887 FROM dual
UNION ALL SELECT 891, 'NAICS', 337125, 'Household Furniture (except Wood and Metal) Manufacturing', 5, 887 FROM dual
UNION ALL SELECT 894, 'NAICS', 33721, 'Office Furniture (including Fixtures) Manufacturing', 4, 893 FROM dual
UNION ALL SELECT 898, 'NAICS', 337215, 'Showcase, Partition, Shelving, and Locker Manufacturing', 5, 894 FROM dual
UNION ALL SELECT 895, 'NAICS', 337211, 'Wood Office Furniture Manufacturing', 5, 894 FROM dual
UNION ALL SELECT 896, 'NAICS', 337212, 'Custom Architectural Woodwork and Millwork Manufacturing', 5, 894 FROM dual
UNION ALL SELECT 897, 'NAICS', 337214, 'Office Furniture (except Wood) Manufacturing', 5, 894 FROM dual
UNION ALL SELECT 903, 'NAICS', 33792, 'Blind and Shade Manufacturing', 4, 899 FROM dual
UNION ALL SELECT 901, 'NAICS', 33791, 'Mattress Manufacturing', 4, 899 FROM dual
UNION ALL SELECT 900, 'NAICS', 337910, 'Mattress Manufacturing', 5, 901 FROM dual
UNION ALL SELECT 902, 'NAICS', 337920, 'Blind and Shade Manufacturing', 5, 903 FROM dual
UNION ALL SELECT 912, 'NAICS', 3399, 'Other Miscellaneous Manufacturing', 3, 904 FROM dual
UNION ALL SELECT 905, 'NAICS', 3391, 'Medical Equipment and Supplies Manufacturing', 3, 904 FROM dual
UNION ALL SELECT 906, 'NAICS', 33911, 'Medical Equipment and Supplies Manufacturing', 4, 905 FROM dual
UNION ALL SELECT 908, 'NAICS', 339113, 'Surgical Appliance and Supplies Manufacturing', 5, 906 FROM dual
UNION ALL SELECT 911, 'NAICS', 339116, 'Dental Laboratories', 5, 906 FROM dual
UNION ALL SELECT 910, 'NAICS', 339115, 'Ophthalmic Goods Manufacturing', 5, 906 FROM dual
UNION ALL SELECT 909, 'NAICS', 339114, 'Dental Equipment and Supplies Manufacturing', 5, 906 FROM dual
UNION ALL SELECT 907, 'NAICS', 339112, 'Surgical and Medical Instrument Manufacturing', 5, 906 FROM dual
UNION ALL SELECT 918, 'NAICS', 33993, 'Doll, Toy, and Game Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 914, 'NAICS', 33991, 'Jewelry and Silverware Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 922, 'NAICS', 33995, 'Sign Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 923, 'NAICS', 33999, 'All Other Miscellaneous Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 920, 'NAICS', 33994, 'Office Supplies (except Paper) Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 916, 'NAICS', 33992, 'Sporting and Athletic Goods Manufacturing', 4, 912 FROM dual
UNION ALL SELECT 913, 'NAICS', 339910, 'Jewelry and Silverware Manufacturing', 5, 914 FROM dual
UNION ALL SELECT 915, 'NAICS', 339920, 'Sporting and Athletic Goods Manufacturing', 5, 916 FROM dual
UNION ALL SELECT 917, 'NAICS', 339930, 'Doll, Toy, and Game Manufacturing', 5, 918 FROM dual
UNION ALL SELECT 919, 'NAICS', 339940, 'Office Supplies (except Paper) Manufacturing', 5, 920 FROM dual
UNION ALL SELECT 921, 'NAICS', 339950, 'Sign Manufacturing', 5, 922 FROM dual
UNION ALL SELECT 925, 'NAICS', 339992, 'Musical Instrument Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 924, 'NAICS', 339991, 'Gasket, Packing, and Sealing Device Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 926, 'NAICS', 339993, 'Fastener, Button, Needle, and Pin Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 927, 'NAICS', 339994, 'Broom, Brush, and Mop Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 928, 'NAICS', 339995, 'Burial Casket Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 929, 'NAICS', 339999, 'All Other Miscellaneous Manufacturing', 5, 923 FROM dual
UNION ALL SELECT 1057, 'NAICS', 424520, 'Livestock Merchant Wholesalers', 5, 1058 FROM dual
UNION ALL SELECT 1089, 'NAICS', 425, 'Wholesale Electronic Markets and Agents and Brokers', 2, 930 FROM dual
UNION ALL SELECT 931, 'NAICS', 423, 'Merchant Wholesalers, Durable Goods', 2, 930 FROM dual
UNION ALL SELECT 1015, 'NAICS', 424, 'Merchant Wholesalers, Nondurable Goods', 2, 930 FROM dual
UNION ALL SELECT 1004, 'NAICS', 4239, 'Miscellaneous Durable Goods Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 991, 'NAICS', 4238, 'Machinery, Equipment, and Supplies Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 975, 'NAICS', 4236, 'Household Appliances and Electrical and Electronic Goods Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 982, 'NAICS', 4237, 'Hardware, and Plumbing and Heating Equipment and Supplies Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 941, 'NAICS', 4232, 'Furniture and Home Furnishing Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 970, 'NAICS', 4235, 'Metal and Mineral (except Petroleum) Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 946, 'NAICS', 4233, 'Lumber and Other Construction Materials Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 932, 'NAICS', 4231, 'Motor Vehicle and Motor Vehicle Parts and Supplies Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 955, 'NAICS', 4234, 'Professional and Commercial Equipment and Supplies Merchant Wholesalers', 3, 931 FROM dual
UNION ALL SELECT 938, 'NAICS', 42313, 'Tire and Tube Merchant Wholesalers', 4, 932 FROM dual
UNION ALL SELECT 940, 'NAICS', 42314, 'Motor Vehicle Parts (Used) Merchant Wholesalers', 4, 932 FROM dual
UNION ALL SELECT 934, 'NAICS', 42311, 'Automobile and Other Motor Vehicle Merchant Wholesalers', 4, 932 FROM dual
UNION ALL SELECT 936, 'NAICS', 42312, 'Motor Vehicle Supplies and New Parts Merchant Wholesalers', 4, 932 FROM dual
UNION ALL SELECT 933, 'NAICS', 423110, 'Automobile and Other Motor Vehicle Merchant Wholesalers', 5, 934 FROM dual
UNION ALL SELECT 935, 'NAICS', 423120, 'Motor Vehicle Supplies and New Parts Merchant Wholesalers', 5, 936 FROM dual
UNION ALL SELECT 937, 'NAICS', 423130, 'Tire and Tube Merchant Wholesalers', 5, 938 FROM dual
UNION ALL SELECT 939, 'NAICS', 423140, 'Motor Vehicle Parts (Used) Merchant Wholesalers', 5, 940 FROM dual
UNION ALL SELECT 943, 'NAICS', 42321, 'Furniture Merchant Wholesalers', 4, 941 FROM dual
UNION ALL SELECT 945, 'NAICS', 42322, 'Home Furnishing Merchant Wholesalers', 4, 941 FROM dual
UNION ALL SELECT 942, 'NAICS', 423210, 'Furniture Merchant Wholesalers', 5, 943 FROM dual
UNION ALL SELECT 1144, 'NAICS', 4451, 'Grocery Stores', 3, 1143 FROM dual
UNION ALL SELECT 944, 'NAICS', 423220, 'Home Furnishing Merchant Wholesalers', 5, 945 FROM dual
UNION ALL SELECT 948, 'NAICS', 42331, 'Lumber, Plywood, Millwork, and Wood Panel Merchant Wholesalers', 4, 946 FROM dual
UNION ALL SELECT 952, 'NAICS', 42333, 'Roofing, Siding, and Insulation Material Merchant Wholesalers', 4, 946 FROM dual
UNION ALL SELECT 954, 'NAICS', 42339, 'Other Construction Material Merchant Wholesalers', 4, 946 FROM dual
UNION ALL SELECT 950, 'NAICS', 42332, 'Brick, Stone, and Related Construction Material Merchant Wholesalers', 4, 946 FROM dual
UNION ALL SELECT 947, 'NAICS', 423310, 'Lumber, Plywood, Millwork, and Wood Panel Merchant Wholesalers', 5, 948 FROM dual
UNION ALL SELECT 949, 'NAICS', 423320, 'Brick, Stone, and Related Construction Material Merchant Wholesalers', 5, 950 FROM dual
UNION ALL SELECT 951, 'NAICS', 423330, 'Roofing, Siding, and Insulation Material Merchant Wholesalers', 5, 952 FROM dual
UNION ALL SELECT 953, 'NAICS', 423390, 'Other Construction Material Merchant Wholesalers', 5, 954 FROM dual
UNION ALL SELECT 961, 'NAICS', 42343, 'Computer and Computer Peripheral Equipment and Software Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 965, 'NAICS', 42345, 'Medical, Dental, and Hospital Equipment and Supplies Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 959, 'NAICS', 42342, 'Office Equipment Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 969, 'NAICS', 42349, 'Other Professional Equipment and Supplies Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 963, 'NAICS', 42344, 'Other Commercial Equipment Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 967, 'NAICS', 42346, 'Ophthalmic Goods Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 957, 'NAICS', 42341, 'Photographic Equipment and Supplies Merchant Wholesalers', 4, 955 FROM dual
UNION ALL SELECT 956, 'NAICS', 423410, 'Photographic Equipment and Supplies Merchant Wholesalers', 5, 957 FROM dual
UNION ALL SELECT 958, 'NAICS', 423420, 'Office Equipment Merchant Wholesalers', 5, 959 FROM dual
UNION ALL SELECT 1157, 'NAICS', 445291, 'Baked Goods Stores', 5, 1156 FROM dual
UNION ALL SELECT 960, 'NAICS', 423430, 'Computer and Computer Peripheral Equipment and Software Merchant Wholesalers', 5, 961 FROM dual
UNION ALL SELECT 962, 'NAICS', 423440, 'Other Commercial Equipment Merchant Wholesalers', 5, 963 FROM dual
UNION ALL SELECT 964, 'NAICS', 423450, 'Medical, Dental, and Hospital Equipment and Supplies Merchant Wholesalers', 5, 965 FROM dual
UNION ALL SELECT 966, 'NAICS', 423460, 'Ophthalmic Goods Merchant Wholesalers', 5, 967 FROM dual
UNION ALL SELECT 968, 'NAICS', 423490, 'Other Professional Equipment and Supplies Merchant Wholesalers', 5, 969 FROM dual
UNION ALL SELECT 974, 'NAICS', 42352, 'Coal and Other Mineral and Ore Merchant Wholesalers', 4, 970 FROM dual
UNION ALL SELECT 972, 'NAICS', 42351, 'Metal Service Centers and Other Metal Merchant Wholesalers', 4, 970 FROM dual
UNION ALL SELECT 971, 'NAICS', 423510, 'Metal Service Centers and Other Metal Merchant Wholesalers', 5, 972 FROM dual
UNION ALL SELECT 973, 'NAICS', 423520, 'Coal and Other Mineral and Ore Merchant Wholesalers', 5, 974 FROM dual
UNION ALL SELECT 981, 'NAICS', 42369, 'Other Electronic Parts and Equipment Merchant Wholesalers', 4, 975 FROM dual
UNION ALL SELECT 979, 'NAICS', 42362, 'Household Appliances, Electric Housewares, and Consumer Electronics Merchant Wholesalers', 4, 975 FROM dual
UNION ALL SELECT 977, 'NAICS', 42361, 'Electrical Apparatus and Equipment, Wiring Supplies, and Related Equipment Merchant Wholesalers', 4, 975 FROM dual
UNION ALL SELECT 976, 'NAICS', 423610, 'Electrical Apparatus and Equipment, Wiring Supplies, and Related Equipment Merchant Wholesalers', 5, 977 FROM dual
UNION ALL SELECT 978, 'NAICS', 423620, 'Household Appliances, Electric Housewares, and Consumer Electronics Merchant Wholesalers', 5, 979 FROM dual
UNION ALL SELECT 980, 'NAICS', 423690, 'Other Electronic Parts and Equipment Merchant Wholesalers', 5, 981 FROM dual
UNION ALL SELECT 986, 'NAICS', 42372, 'Plumbing and Heating Equipment and Supplies (Hydronics) Merchant Wholesalers', 4, 982 FROM dual
UNION ALL SELECT 984, 'NAICS', 42371, 'Hardware Merchant Wholesalers', 4, 982 FROM dual
UNION ALL SELECT 988, 'NAICS', 42373, 'Warm Air Heating and Air-Conditioning Equipment and Supplies Merchant Wholesalers', 4, 982 FROM dual
UNION ALL SELECT 990, 'NAICS', 42374, 'Refrigeration Equipment and Supplies Merchant Wholesalers', 4, 982 FROM dual
UNION ALL SELECT 983, 'NAICS', 423710, 'Hardware Merchant Wholesalers', 5, 984 FROM dual
UNION ALL SELECT 985, 'NAICS', 423720, 'Plumbing and Heating Equipment and Supplies (Hydronics) Merchant Wholesalers', 5, 986 FROM dual
UNION ALL SELECT 987, 'NAICS', 423730, 'Warm Air Heating and Air-Conditioning Equipment and Supplies Merchant Wholesalers', 5, 988 FROM dual
UNION ALL SELECT 989, 'NAICS', 423740, 'Refrigeration Equipment and Supplies Merchant Wholesalers', 5, 990 FROM dual
UNION ALL SELECT 999, 'NAICS', 42384, 'Industrial Supplies Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 1059, 'NAICS', 424590, 'Other Farm Product Raw Material Merchant Wholesalers', 5, 1060 FROM dual
UNION ALL SELECT 993, 'NAICS', 42381, 'Construction and Mining (except Oil Well) Machinery and Equipment Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 995, 'NAICS', 42382, 'Farm and Garden Machinery and Equipment Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 997, 'NAICS', 42383, 'Industrial Machinery and Equipment Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 1001, 'NAICS', 42385, 'Service Establishment Equipment and Supplies Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 1003, 'NAICS', 42386, 'Transportation Equipment and Supplies (except Motor Vehicle) Merchant Wholesalers', 4, 991 FROM dual
UNION ALL SELECT 992, 'NAICS', 423810, 'Construction and Mining (except Oil Well) Machinery and Equipment Merchant Wholesalers', 5, 993 FROM dual
UNION ALL SELECT 994, 'NAICS', 423820, 'Farm and Garden Machinery and Equipment Merchant Wholesalers', 5, 995 FROM dual
UNION ALL SELECT 996, 'NAICS', 423830, 'Industrial Machinery and Equipment Merchant Wholesalers', 5, 997 FROM dual
UNION ALL SELECT 998, 'NAICS', 423840, 'Industrial Supplies Merchant Wholesalers', 5, 999 FROM dual
UNION ALL SELECT 1000, 'NAICS', 423850, 'Service Establishment Equipment and Supplies Merchant Wholesalers', 5, 1001 FROM dual
UNION ALL SELECT 1002, 'NAICS', 423860, 'Transportation Equipment and Supplies (except Motor Vehicle) Merchant Wholesalers', 5, 1003 FROM dual
UNION ALL SELECT 1012, 'NAICS', 42394, 'Jewelry, Watch, Precious Stone, and Precious Metal Merchant Wholesalers', 4, 1004 FROM dual
UNION ALL SELECT 1006, 'NAICS', 42391, 'Sporting and Recreational Goods and Supplies Merchant Wholesalers', 4, 1004 FROM dual
UNION ALL SELECT 1008, 'NAICS', 42392, 'Toy and Hobby Goods and Supplies Merchant Wholesalers', 4, 1004 FROM dual
UNION ALL SELECT 1010, 'NAICS', 42393, 'Recyclable Material Merchant Wholesalers', 4, 1004 FROM dual
UNION ALL SELECT 1014, 'NAICS', 42399, 'Other Miscellaneous Durable Goods Merchant Wholesalers', 4, 1004 FROM dual
UNION ALL SELECT 1005, 'NAICS', 423910, 'Sporting and Recreational Goods and Supplies Merchant Wholesalers', 5, 1006 FROM dual
UNION ALL SELECT 1007, 'NAICS', 423920, 'Toy and Hobby Goods and Supplies Merchant Wholesalers', 5, 1008 FROM dual
UNION ALL SELECT 1009, 'NAICS', 423930, 'Recyclable Material Merchant Wholesalers', 5, 1010 FROM dual
UNION ALL SELECT 1011, 'NAICS', 423940, 'Jewelry, Watch, Precious Stone, and Precious Metal Merchant Wholesalers', 5, 1012 FROM dual
UNION ALL SELECT 1013, 'NAICS', 423990, 'Other Miscellaneous Durable Goods Merchant Wholesalers', 5, 1014 FROM dual
UNION ALL SELECT 1155, 'NAICS', 44523, 'Fruit and Vegetable Markets', 4, 1149 FROM dual
UNION ALL SELECT 1066, 'NAICS', 4247, 'Petroleum and Petroleum Products Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1016, 'NAICS', 4241, 'Paper and Paper Product Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1023, 'NAICS', 4242, 'Drugs and Druggists'' Sundries Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1246, 'NAICS', 453991, 'Tobacco Stores', 5, 1245 FROM dual
UNION ALL SELECT 1026, 'NAICS', 4243, 'Apparel, Piece Goods, and Notions Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1071, 'NAICS', 4248, 'Beer, Wine, and Distilled Alcoholic Beverage Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1054, 'NAICS', 4245, 'Farm Product Raw Material Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1061, 'NAICS', 4246, 'Chemical and Allied Products Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1076, 'NAICS', 4249, 'Miscellaneous Nondurable Goods Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1035, 'NAICS', 4244, 'Grocery and Related Product Merchant Wholesalers', 3, 1015 FROM dual
UNION ALL SELECT 1018, 'NAICS', 42411, 'Printing and Writing Paper Merchant Wholesalers', 4, 1016 FROM dual
UNION ALL SELECT 1020, 'NAICS', 42412, 'Stationery and Office Supplies Merchant Wholesalers', 4, 1016 FROM dual
UNION ALL SELECT 1022, 'NAICS', 42413, 'Industrial and Personal Service Paper Merchant Wholesalers', 4, 1016 FROM dual
UNION ALL SELECT 1017, 'NAICS', 424110, 'Printing and Writing Paper Merchant Wholesalers', 5, 1018 FROM dual
UNION ALL SELECT 1019, 'NAICS', 424120, 'Stationery and Office Supplies Merchant Wholesalers', 5, 1020 FROM dual
UNION ALL SELECT 1021, 'NAICS', 424130, 'Industrial and Personal Service Paper Merchant Wholesalers', 5, 1022 FROM dual
UNION ALL SELECT 1025, 'NAICS', 42421, 'Drugs and Druggists'' Sundries Merchant Wholesalers', 4, 1023 FROM dual
UNION ALL SELECT 1024, 'NAICS', 424210, 'Drugs and Druggists'' Sundries Merchant Wholesalers', 5, 1025 FROM dual
UNION ALL SELECT 1034, 'NAICS', 42434, 'Footwear Merchant Wholesalers', 4, 1026 FROM dual
UNION ALL SELECT 1028, 'NAICS', 42431, 'Piece Goods, Notions, and Other Dry Goods Merchant Wholesalers', 4, 1026 FROM dual
UNION ALL SELECT 1030, 'NAICS', 42432, 'Men''s and Boys'' Clothing and Furnishings Merchant Wholesalers', 4, 1026 FROM dual
UNION ALL SELECT 1032, 'NAICS', 42433, 'Women''s, Children''s, and Infants'' Clothing and Accessories Merchant Wholesalers', 4, 1026 FROM dual
UNION ALL SELECT 1027, 'NAICS', 424310, 'Piece Goods, Notions, and Other Dry Goods Merchant Wholesalers', 5, 1028 FROM dual
UNION ALL SELECT 1029, 'NAICS', 424320, 'Men''s and Boys'' Clothing and Furnishings Merchant Wholesalers', 5, 1030 FROM dual
UNION ALL SELECT 1031, 'NAICS', 424330, 'Women''s, Children''s, and Infants'' Clothing and Accessories Merchant Wholesalers', 5, 1032 FROM dual
UNION ALL SELECT 1033, 'NAICS', 424340, 'Footwear Merchant Wholesalers', 5, 1034 FROM dual
UNION ALL SELECT 1045, 'NAICS', 42445, 'Confectionery Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1047, 'NAICS', 42446, 'Fish and Seafood Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1051, 'NAICS', 42448, 'Fresh Fruit and Vegetable Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1039, 'NAICS', 42442, 'Packaged Frozen Food Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1041, 'NAICS', 42443, 'Dairy Product (except Dried or Canned) Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1043, 'NAICS', 42444, 'Poultry and Poultry Product Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1053, 'NAICS', 42449, 'Other Grocery and Related Products Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1049, 'NAICS', 42447, 'Meat and Meat Product Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1037, 'NAICS', 42441, 'General Line Grocery Merchant Wholesalers', 4, 1035 FROM dual
UNION ALL SELECT 1036, 'NAICS', 424410, 'General Line Grocery Merchant Wholesalers', 5, 1037 FROM dual
UNION ALL SELECT 1038, 'NAICS', 424420, 'Packaged Frozen Food Merchant Wholesalers', 5, 1039 FROM dual
UNION ALL SELECT 1040, 'NAICS', 424430, 'Dairy Product (except Dried or Canned) Merchant Wholesalers', 5, 1041 FROM dual
UNION ALL SELECT 1042, 'NAICS', 424440, 'Poultry and Poultry Product Merchant Wholesalers', 5, 1043 FROM dual
UNION ALL SELECT 1044, 'NAICS', 424450, 'Confectionery Merchant Wholesalers', 5, 1045 FROM dual
UNION ALL SELECT 1046, 'NAICS', 424460, 'Fish and Seafood Merchant Wholesalers', 5, 1047 FROM dual
UNION ALL SELECT 1048, 'NAICS', 424470, 'Meat and Meat Product Merchant Wholesalers', 5, 1049 FROM dual
UNION ALL SELECT 1050, 'NAICS', 424480, 'Fresh Fruit and Vegetable Merchant Wholesalers', 5, 1051 FROM dual
UNION ALL SELECT 1052, 'NAICS', 424490, 'Other Grocery and Related Products Merchant Wholesalers', 5, 1053 FROM dual
UNION ALL SELECT 1058, 'NAICS', 42452, 'Livestock Merchant Wholesalers', 4, 1054 FROM dual
UNION ALL SELECT 1060, 'NAICS', 42459, 'Other Farm Product Raw Material Merchant Wholesalers', 4, 1054 FROM dual
UNION ALL SELECT 1056, 'NAICS', 42451, 'Grain and Field Bean Merchant Wholesalers', 4, 1054 FROM dual
UNION ALL SELECT 1055, 'NAICS', 424510, 'Grain and Field Bean Merchant Wholesalers', 5, 1056 FROM dual
UNION ALL SELECT 1065, 'NAICS', 42469, 'Other Chemical and Allied Products Merchant Wholesalers', 4, 1061 FROM dual
UNION ALL SELECT 1063, 'NAICS', 42461, 'Plastics Materials and Basic Forms and Shapes Merchant Wholesalers', 4, 1061 FROM dual
UNION ALL SELECT 1062, 'NAICS', 424610, 'Plastics Materials and Basic Forms and Shapes Merchant Wholesalers', 5, 1063 FROM dual
UNION ALL SELECT 1064, 'NAICS', 424690, 'Other Chemical and Allied Products Merchant Wholesalers', 5, 1065 FROM dual
UNION ALL SELECT 1070, 'NAICS', 42472, 'Petroleum and Petroleum Products Merchant Wholesalers (except Bulk Stations and Terminals)', 4, 1066 FROM dual
UNION ALL SELECT 1068, 'NAICS', 42471, 'Petroleum Bulk Stations and Terminals', 4, 1066 FROM dual
UNION ALL SELECT 1067, 'NAICS', 424710, 'Petroleum Bulk Stations and Terminals', 5, 1068 FROM dual
UNION ALL SELECT 1069, 'NAICS', 424720, 'Petroleum and Petroleum Products Merchant Wholesalers (except Bulk Stations and Terminals)', 5, 1070 FROM dual
UNION ALL SELECT 1073, 'NAICS', 42481, 'Beer and Ale Merchant Wholesalers', 4, 1071 FROM dual
UNION ALL SELECT 1075, 'NAICS', 42482, 'Wine and Distilled Alcoholic Beverage Merchant Wholesalers', 4, 1071 FROM dual
UNION ALL SELECT 1072, 'NAICS', 424810, 'Beer and Ale Merchant Wholesalers', 5, 1073 FROM dual
UNION ALL SELECT 1074, 'NAICS', 424820, 'Wine and Distilled Alcoholic Beverage Merchant Wholesalers', 5, 1075 FROM dual
UNION ALL SELECT 1078, 'NAICS', 42491, 'Farm Supplies Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1080, 'NAICS', 42492, 'Book, Periodical, and Newspaper Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1082, 'NAICS', 42493, 'Flower, Nursery Stock, and Florists'' Supplies Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1084, 'NAICS', 42494, 'Tobacco and Tobacco Product Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1086, 'NAICS', 42495, 'Paint, Varnish, and Supplies Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1088, 'NAICS', 42499, 'Other Miscellaneous Nondurable Goods Merchant Wholesalers', 4, 1076 FROM dual
UNION ALL SELECT 1077, 'NAICS', 424910, 'Farm Supplies Merchant Wholesalers', 5, 1078 FROM dual
UNION ALL SELECT 1079, 'NAICS', 424920, 'Book, Periodical, and Newspaper Merchant Wholesalers', 5, 1080 FROM dual
UNION ALL SELECT 1081, 'NAICS', 424930, 'Flower, Nursery Stock, and Florists'' Supplies Merchant Wholesalers', 5, 1082 FROM dual
UNION ALL SELECT 1083, 'NAICS', 424940, 'Tobacco and Tobacco Product Merchant Wholesalers', 5, 1084 FROM dual
UNION ALL SELECT 1085, 'NAICS', 424950, 'Paint, Varnish, and Supplies Merchant Wholesalers', 5, 1086 FROM dual
UNION ALL SELECT 1087, 'NAICS', 424990, 'Other Miscellaneous Nondurable Goods Merchant Wholesalers', 5, 1088 FROM dual
UNION ALL SELECT 1090, 'NAICS', 4251, 'Wholesale Electronic Markets and Agents and Brokers', 3, 1089 FROM dual
UNION ALL SELECT 1092, 'NAICS', 42511, 'Business to Business Electronic Markets', 4, 1090 FROM dual
UNION ALL SELECT 1094, 'NAICS', 42512, 'Wholesale Trade Agents and Brokers', 4, 1090 FROM dual
UNION ALL SELECT 1091, 'NAICS', 425110, 'Business to Business Electronic Markets', 5, 1092 FROM dual
UNION ALL SELECT 1093, 'NAICS', 425120, 'Wholesale Trade Agents and Brokers', 5, 1094 FROM dual
UNION ALL SELECT 1163, 'NAICS', 446, 'Health and Personal Care Stores', 2, 1095 FROM dual
UNION ALL SELECT 1180, 'NAICS', 448, 'Clothing and Clothing Accessories Stores', 2, 1095 FROM dual
UNION ALL SELECT 1174, 'NAICS', 447, 'Gasoline Stations', 2, 1095 FROM dual
UNION ALL SELECT 1096, 'NAICS', 441, 'Motor Vehicle and Parts Dealers', 2, 1095 FROM dual
UNION ALL SELECT 1113, 'NAICS', 442, 'Furniture and Home Furnishings Stores', 2, 1095 FROM dual
UNION ALL SELECT 1123, 'NAICS', 443, 'Electronics and Appliance Stores', 2, 1095 FROM dual
UNION ALL SELECT 1128, 'NAICS', 444, 'Building Material and Garden Equipment and Supplies Dealers', 2, 1095 FROM dual
UNION ALL SELECT 1143, 'NAICS', 445, 'Food and Beverage Stores', 2, 1095 FROM dual
UNION ALL SELECT 1226, 'NAICS', 453, 'Miscellaneous Store Retailers', 2, 1095 FROM dual
UNION ALL SELECT 1248, 'NAICS', 454, 'Nonstore Retailers', 2, 1095 FROM dual
UNION ALL SELECT 1216, 'NAICS', 452, 'General Merchandise Stores', 2, 1095 FROM dual
UNION ALL SELECT 1202, 'NAICS', 451, 'Sporting Goods, Hobby, Musical Instrument, and Book Stores', 2, 1095 FROM dual
UNION ALL SELECT 1102, 'NAICS', 4412, 'Other Motor Vehicle Dealers', 3, 1096 FROM dual
UNION ALL SELECT 1097, 'NAICS', 4411, 'Automobile Dealers', 3, 1096 FROM dual
UNION ALL SELECT 1108, 'NAICS', 4413, 'Automotive Parts, Accessories, and Tire Stores', 3, 1096 FROM dual
UNION ALL SELECT 1099, 'NAICS', 44111, 'New Car Dealers', 4, 1097 FROM dual
UNION ALL SELECT 1101, 'NAICS', 44112, 'Used Car Dealers', 4, 1097 FROM dual
UNION ALL SELECT 1098, 'NAICS', 441110, 'New Car Dealers', 5, 1099 FROM dual
UNION ALL SELECT 1100, 'NAICS', 441120, 'Used Car Dealers', 5, 1101 FROM dual
UNION ALL SELECT 1104, 'NAICS', 44121, 'Recreational Vehicle Dealers', 4, 1102 FROM dual
UNION ALL SELECT 1105, 'NAICS', 44122, 'Motorcycle, Boat, and Other Motor Vehicle Dealers', 4, 1102 FROM dual
UNION ALL SELECT 1103, 'NAICS', 441210, 'Recreational Vehicle Dealers', 5, 1104 FROM dual
UNION ALL SELECT 1106, 'NAICS', 441222, 'Boat Dealers', 5, 1105 FROM dual
UNION ALL SELECT 1107, 'NAICS', 441228, 'Motorcycle, ATV, and All Other Motor Vehicle Dealers', 5, 1105 FROM dual
UNION ALL SELECT 1112, 'NAICS', 44132, 'Tire Dealers', 4, 1108 FROM dual
UNION ALL SELECT 1110, 'NAICS', 44131, 'Automotive Parts and Accessories Stores', 4, 1108 FROM dual
UNION ALL SELECT 1109, 'NAICS', 441310, 'Automotive Parts and Accessories Stores', 5, 1110 FROM dual
UNION ALL SELECT 1111, 'NAICS', 441320, 'Tire Dealers', 5, 1112 FROM dual
UNION ALL SELECT 1114, 'NAICS', 4421, 'Furniture Stores', 3, 1113 FROM dual
UNION ALL SELECT 1117, 'NAICS', 4422, 'Home Furnishings Stores', 3, 1113 FROM dual
UNION ALL SELECT 1116, 'NAICS', 44211, 'Furniture Stores', 4, 1114 FROM dual
UNION ALL SELECT 1115, 'NAICS', 442110, 'Furniture Stores', 5, 1116 FROM dual
UNION ALL SELECT 1120, 'NAICS', 44229, 'Other Home Furnishings Stores', 4, 1117 FROM dual
UNION ALL SELECT 1119, 'NAICS', 44221, 'Floor Covering Stores', 4, 1117 FROM dual
UNION ALL SELECT 1118, 'NAICS', 442210, 'Floor Covering Stores', 5, 1119 FROM dual
UNION ALL SELECT 1122, 'NAICS', 442299, 'All Other Home Furnishings Stores', 5, 1120 FROM dual
UNION ALL SELECT 1121, 'NAICS', 442291, 'Window Treatment Stores', 5, 1120 FROM dual
UNION ALL SELECT 1124, 'NAICS', 4431, 'Electronics and Appliance Stores', 3, 1123 FROM dual
UNION ALL SELECT 1125, 'NAICS', 44314, 'Electronics and Appliance Stores', 4, 1124 FROM dual
UNION ALL SELECT 1126, 'NAICS', 443141, 'Household Appliance Stores', 5, 1125 FROM dual
UNION ALL SELECT 1127, 'NAICS', 443142, 'Electronics Stores', 5, 1125 FROM dual
UNION ALL SELECT 1129, 'NAICS', 4441, 'Building Material and Supplies Dealers', 3, 1128 FROM dual
UNION ALL SELECT 1138, 'NAICS', 4442, 'Lawn and Garden Equipment and Supplies Stores', 3, 1128 FROM dual
UNION ALL SELECT 1137, 'NAICS', 44419, 'Other Building Material Dealers', 4, 1129 FROM dual
UNION ALL SELECT 1131, 'NAICS', 44411, 'Home Centers', 4, 1129 FROM dual
UNION ALL SELECT 1133, 'NAICS', 44412, 'Paint and Wallpaper Stores', 4, 1129 FROM dual
UNION ALL SELECT 1135, 'NAICS', 44413, 'Hardware Stores', 4, 1129 FROM dual
UNION ALL SELECT 1130, 'NAICS', 444110, 'Home Centers', 5, 1131 FROM dual
UNION ALL SELECT 1132, 'NAICS', 444120, 'Paint and Wallpaper Stores', 5, 1133 FROM dual
UNION ALL SELECT 1134, 'NAICS', 444130, 'Hardware Stores', 5, 1135 FROM dual
UNION ALL SELECT 1136, 'NAICS', 444190, 'Other Building Material Dealers', 5, 1137 FROM dual
UNION ALL SELECT 1140, 'NAICS', 44421, 'Outdoor Power Equipment Stores', 4, 1138 FROM dual
UNION ALL SELECT 1142, 'NAICS', 44422, 'Nursery, Garden Center, and Farm Supply Stores', 4, 1138 FROM dual
UNION ALL SELECT 1139, 'NAICS', 444210, 'Outdoor Power Equipment Stores', 5, 1140 FROM dual
UNION ALL SELECT 1141, 'NAICS', 444220, 'Nursery, Garden Center, and Farm Supply Stores', 5, 1142 FROM dual
UNION ALL SELECT 1149, 'NAICS', 4452, 'Specialty Food Stores', 3, 1143 FROM dual
UNION ALL SELECT 1160, 'NAICS', 4453, 'Beer, Wine, and Liquor Stores', 3, 1143 FROM dual
UNION ALL SELECT 1148, 'NAICS', 44512, 'Convenience Stores', 4, 1144 FROM dual
UNION ALL SELECT 1146, 'NAICS', 44511, 'Supermarkets and Other Grocery (except Convenience) Stores', 4, 1144 FROM dual
UNION ALL SELECT 1145, 'NAICS', 445110, 'Supermarkets and Other Grocery (except Convenience) Stores', 5, 1146 FROM dual
UNION ALL SELECT 1147, 'NAICS', 445120, 'Convenience Stores', 5, 1148 FROM dual
UNION ALL SELECT 1153, 'NAICS', 44522, 'Fish and Seafood Markets', 4, 1149 FROM dual
UNION ALL SELECT 1156, 'NAICS', 44529, 'Other Specialty Food Stores', 4, 1149 FROM dual
UNION ALL SELECT 1150, 'NAICS', 445210, 'Meat Markets', 5, 1151 FROM dual
UNION ALL SELECT 1152, 'NAICS', 445220, 'Fish and Seafood Markets', 5, 1153 FROM dual
UNION ALL SELECT 1154, 'NAICS', 445230, 'Fruit and Vegetable Markets', 5, 1155 FROM dual
UNION ALL SELECT 1158, 'NAICS', 445292, 'Confectionery and Nut Stores', 5, 1156 FROM dual
UNION ALL SELECT 1159, 'NAICS', 445299, 'All Other Specialty Food Stores', 5, 1156 FROM dual
UNION ALL SELECT 1162, 'NAICS', 44531, 'Beer, Wine, and Liquor Stores', 4, 1160 FROM dual
UNION ALL SELECT 1161, 'NAICS', 445310, 'Beer, Wine, and Liquor Stores', 5, 1162 FROM dual
UNION ALL SELECT 1164, 'NAICS', 4461, 'Health and Personal Care Stores', 3, 1163 FROM dual
UNION ALL SELECT 1171, 'NAICS', 44619, 'Other Health and Personal Care Stores', 4, 1164 FROM dual
UNION ALL SELECT 1166, 'NAICS', 44611, 'Pharmacies and Drug Stores', 4, 1164 FROM dual
UNION ALL SELECT 1168, 'NAICS', 44612, 'Cosmetics, Beauty Supplies, and Perfume Stores', 4, 1164 FROM dual
UNION ALL SELECT 1170, 'NAICS', 44613, 'Optical Goods Stores', 4, 1164 FROM dual
UNION ALL SELECT 1165, 'NAICS', 446110, 'Pharmacies and Drug Stores', 5, 1166 FROM dual
UNION ALL SELECT 1167, 'NAICS', 446120, 'Cosmetics, Beauty Supplies, and Perfume Stores', 5, 1168 FROM dual
UNION ALL SELECT 1169, 'NAICS', 446130, 'Optical Goods Stores', 5, 1170 FROM dual
UNION ALL SELECT 1173, 'NAICS', 446199, 'All Other Health and Personal Care Stores', 5, 1171 FROM dual
UNION ALL SELECT 1172, 'NAICS', 446191, 'Food (Health) Supplement Stores', 5, 1171 FROM dual
UNION ALL SELECT 1175, 'NAICS', 4471, 'Gasoline Stations', 3, 1174 FROM dual
UNION ALL SELECT 1179, 'NAICS', 44719, 'Other Gasoline Stations', 4, 1175 FROM dual
UNION ALL SELECT 1177, 'NAICS', 44711, 'Gasoline Stations with Convenience Stores', 4, 1175 FROM dual
UNION ALL SELECT 1176, 'NAICS', 447110, 'Gasoline Stations with Convenience Stores', 5, 1177 FROM dual
UNION ALL SELECT 1178, 'NAICS', 447190, 'Other Gasoline Stations', 5, 1179 FROM dual
UNION ALL SELECT 1181, 'NAICS', 4481, 'Clothing Stores', 3, 1180 FROM dual
UNION ALL SELECT 1197, 'NAICS', 4483, 'Jewelry, Luggage, and Leather Goods Stores', 3, 1180 FROM dual
UNION ALL SELECT 1189, 'NAICS', 44814, 'Family Clothing Stores', 4, 1181 FROM dual
UNION ALL SELECT 1187, 'NAICS', 44813, 'Children''s and Infants'' Clothing Stores', 4, 1181 FROM dual
UNION ALL SELECT 1185, 'NAICS', 44812, 'Women''s Clothing Stores', 4, 1181 FROM dual
UNION ALL SELECT 1183, 'NAICS', 44811, 'Men''s Clothing Stores', 4, 1181 FROM dual
UNION ALL SELECT 1193, 'NAICS', 44819, 'Other Clothing Stores', 4, 1181 FROM dual
UNION ALL SELECT 1191, 'NAICS', 44815, 'Clothing Accessories Stores', 4, 1181 FROM dual
UNION ALL SELECT 1182, 'NAICS', 448110, 'Men''s Clothing Stores', 5, 1183 FROM dual
UNION ALL SELECT 1184, 'NAICS', 448120, 'Women''s Clothing Stores', 5, 1185 FROM dual
UNION ALL SELECT 1186, 'NAICS', 448130, 'Children''s and Infants'' Clothing Stores', 5, 1187 FROM dual
UNION ALL SELECT 1188, 'NAICS', 448140, 'Family Clothing Stores', 5, 1189 FROM dual
UNION ALL SELECT 1190, 'NAICS', 448150, 'Clothing Accessories Stores', 5, 1191 FROM dual
UNION ALL SELECT 1192, 'NAICS', 448190, 'Other Clothing Stores', 5, 1193 FROM dual
UNION ALL SELECT 1196, 'NAICS', 44821, 'Shoe Stores', 4, 1194 FROM dual
UNION ALL SELECT 1195, 'NAICS', 448210, 'Shoe Stores', 5, 1196 FROM dual
UNION ALL SELECT 1199, 'NAICS', 44831, 'Jewelry Stores', 4, 1197 FROM dual
UNION ALL SELECT 1201, 'NAICS', 44832, 'Luggage and Leather Goods Stores', 4, 1197 FROM dual
UNION ALL SELECT 1198, 'NAICS', 448310, 'Jewelry Stores', 5, 1199 FROM dual
UNION ALL SELECT 1200, 'NAICS', 448320, 'Luggage and Leather Goods Stores', 5, 1201 FROM dual
UNION ALL SELECT 1203, 'NAICS', 4511, 'Sporting Goods, Hobby, and Musical Instrument Stores', 3, 1202 FROM dual
UNION ALL SELECT 1212, 'NAICS', 4512, 'Book Stores and News Dealers', 3, 1202 FROM dual
UNION ALL SELECT 1205, 'NAICS', 45111, 'Sporting Goods Stores', 4, 1203 FROM dual
UNION ALL SELECT 1207, 'NAICS', 45112, 'Hobby, Toy, and Game Stores', 4, 1203 FROM dual
UNION ALL SELECT 1209, 'NAICS', 45113, 'Sewing, Needlework, and Piece Goods Stores', 4, 1203 FROM dual
UNION ALL SELECT 1211, 'NAICS', 45114, 'Musical Instrument and Supplies Stores', 4, 1203 FROM dual
UNION ALL SELECT 1204, 'NAICS', 451110, 'Sporting Goods Stores', 5, 1205 FROM dual
UNION ALL SELECT 1206, 'NAICS', 451120, 'Hobby, Toy, and Game Stores', 5, 1207 FROM dual
UNION ALL SELECT 1208, 'NAICS', 451130, 'Sewing, Needlework, and Piece Goods Stores', 5, 1209 FROM dual
UNION ALL SELECT 1210, 'NAICS', 451140, 'Musical Instrument and Supplies Stores', 5, 1211 FROM dual
UNION ALL SELECT 1213, 'NAICS', 45121, 'Book Stores and News Dealers', 4, 1212 FROM dual
UNION ALL SELECT 1214, 'NAICS', 451211, 'Book Stores', 5, 1213 FROM dual
UNION ALL SELECT 1215, 'NAICS', 451212, 'News Dealers and Newsstands', 5, 1213 FROM dual
UNION ALL SELECT 1217, 'NAICS', 4521, 'Department Stores', 3, 1216 FROM dual
UNION ALL SELECT 1221, 'NAICS', 4529, 'Other General Merchandise Stores', 3, 1216 FROM dual
UNION ALL SELECT 1218, 'NAICS', 45211, 'Department Stores', 4, 1217 FROM dual
UNION ALL SELECT 1220, 'NAICS', 452112, 'Discount Department Stores', 5, 1218 FROM dual
UNION ALL SELECT 1219, 'NAICS', 452111, 'Department Stores (except Discount Department Stores)', 5, 1218 FROM dual
UNION ALL SELECT 1223, 'NAICS', 45291, 'Warehouse Clubs and Supercenters', 4, 1221 FROM dual
UNION ALL SELECT 1225, 'NAICS', 45299, 'All Other General Merchandise Stores', 4, 1221 FROM dual
UNION ALL SELECT 1222, 'NAICS', 452910, 'Warehouse Clubs and Supercenters', 5, 1223 FROM dual
UNION ALL SELECT 1224, 'NAICS', 452990, 'All Other General Merchandise Stores', 5, 1225 FROM dual
UNION ALL SELECT 1230, 'NAICS', 4532, 'Office Supplies, Stationery, and Gift Stores', 3, 1226 FROM dual
UNION ALL SELECT 1235, 'NAICS', 4533, 'Used Merchandise Stores', 3, 1226 FROM dual
UNION ALL SELECT 1227, 'NAICS', 4531, 'Florists', 3, 1226 FROM dual
UNION ALL SELECT 1238, 'NAICS', 4539, 'Other Miscellaneous Store Retailers', 3, 1226 FROM dual
UNION ALL SELECT 1229, 'NAICS', 45311, 'Florists', 4, 1227 FROM dual
UNION ALL SELECT 1228, 'NAICS', 453110, 'Florists', 5, 1229 FROM dual
UNION ALL SELECT 1232, 'NAICS', 45321, 'Office Supplies and Stationery Stores', 4, 1230 FROM dual
UNION ALL SELECT 1234, 'NAICS', 45322, 'Gift, Novelty, and Souvenir Stores', 4, 1230 FROM dual
UNION ALL SELECT 1231, 'NAICS', 453210, 'Office Supplies and Stationery Stores', 5, 1232 FROM dual
UNION ALL SELECT 1233, 'NAICS', 453220, 'Gift, Novelty, and Souvenir Stores', 5, 1234 FROM dual
UNION ALL SELECT 1237, 'NAICS', 45331, 'Used Merchandise Stores', 4, 1235 FROM dual
UNION ALL SELECT 1236, 'NAICS', 453310, 'Used Merchandise Stores', 5, 1237 FROM dual
UNION ALL SELECT 1244, 'NAICS', 45393, 'Manufactured (Mobile) Home Dealers', 4, 1238 FROM dual
UNION ALL SELECT 1240, 'NAICS', 45391, 'Pet and Pet Supplies Stores', 4, 1238 FROM dual
UNION ALL SELECT 1242, 'NAICS', 45392, 'Art Dealers', 4, 1238 FROM dual
UNION ALL SELECT 1245, 'NAICS', 45399, 'All Other Miscellaneous Store Retailers', 4, 1238 FROM dual
UNION ALL SELECT 1239, 'NAICS', 453910, 'Pet and Pet Supplies Stores', 5, 1240 FROM dual
UNION ALL SELECT 1241, 'NAICS', 453920, 'Art Dealers', 5, 1242 FROM dual
UNION ALL SELECT 1243, 'NAICS', 453930, 'Manufactured (Mobile) Home Dealers', 5, 1244 FROM dual
UNION ALL SELECT 1247, 'NAICS', 453998, 'All Other Miscellaneous Store Retailers (except Tobacco Stores)', 5, 1245 FROM dual
UNION ALL SELECT 1254, 'NAICS', 4542, 'Vending Machine Operators', 3, 1248 FROM dual
UNION ALL SELECT 1257, 'NAICS', 4543, 'Direct Selling Establishments', 3, 1248 FROM dual
UNION ALL SELECT 1249, 'NAICS', 4541, 'Electronic Shopping and Mail-Order Houses', 3, 1248 FROM dual
UNION ALL SELECT 1250, 'NAICS', 45411, 'Electronic Shopping and Mail-Order Houses', 4, 1249 FROM dual
UNION ALL SELECT 1251, 'NAICS', 454111, 'Electronic Shopping', 5, 1250 FROM dual
UNION ALL SELECT 1252, 'NAICS', 454112, 'Electronic Auctions', 5, 1250 FROM dual
UNION ALL SELECT 1253, 'NAICS', 454113, 'Mail-Order Houses', 5, 1250 FROM dual
UNION ALL SELECT 1256, 'NAICS', 45421, 'Vending Machine Operators', 4, 1254 FROM dual
UNION ALL SELECT 1255, 'NAICS', 454210, 'Vending Machine Operators', 5, 1256 FROM dual
UNION ALL SELECT 1259, 'NAICS', 45431, 'Fuel Dealers', 4, 1257 FROM dual
UNION ALL SELECT 1261, 'NAICS', 45439, 'Other Direct Selling Establishments', 4, 1257 FROM dual
UNION ALL SELECT 1258, 'NAICS', 454310, 'Fuel Dealers', 5, 1259 FROM dual
UNION ALL SELECT 1260, 'NAICS', 454390, 'Other Direct Selling Establishments', 5, 1261 FROM dual
UNION ALL SELECT 1303, 'NAICS', 485, 'Transit and Ground Passenger Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1381, 'NAICS', 491, 'Postal Service', 2, 1262 FROM dual
UNION ALL SELECT 1263, 'NAICS', 481, 'Air Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1328, 'NAICS', 486, 'Pipeline Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1340, 'NAICS', 487, 'Scenic and Sightseeing Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1350, 'NAICS', 488, 'Support Activities for Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1392, 'NAICS', 493, 'Warehousing and Storage', 2, 1262 FROM dual
UNION ALL SELECT 1278, 'NAICS', 483, 'Water Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1385, 'NAICS', 492, 'Couriers and Messengers', 2, 1262 FROM dual
UNION ALL SELECT 1289, 'NAICS', 484, 'Truck Transportation', 2, 1262 FROM dual
UNION ALL SELECT 1268, 'NAICS', 4812, 'Nonscheduled Air Transportation', 3, 1263 FROM dual
UNION ALL SELECT 1264, 'NAICS', 4811, 'Scheduled Air Transportation', 3, 1263 FROM dual
UNION ALL SELECT 1265, 'NAICS', 48111, 'Scheduled Air Transportation', 4, 1264 FROM dual
UNION ALL SELECT 1266, 'NAICS', 481111, 'Scheduled Passenger Air Transportation', 5, 1265 FROM dual
UNION ALL SELECT 1267, 'NAICS', 481112, 'Scheduled Freight Air Transportation', 5, 1265 FROM dual
UNION ALL SELECT 1269, 'NAICS', 48121, 'Nonscheduled Air Transportation', 4, 1268 FROM dual
UNION ALL SELECT 1270, 'NAICS', 481211, 'Nonscheduled Chartered Passenger Air Transportation', 5, 1269 FROM dual
UNION ALL SELECT 1271, 'NAICS', 481212, 'Nonscheduled Chartered Freight Air Transportation', 5, 1269 FROM dual
UNION ALL SELECT 1272, 'NAICS', 481219, 'Other Nonscheduled Air Transportation', 5, 1269 FROM dual
UNION ALL SELECT 1274, 'NAICS', 4821, 'Rail Transportation', 3, 1273 FROM dual
UNION ALL SELECT 1275, 'NAICS', 48211, 'Rail Transportation', 4, 1274 FROM dual
UNION ALL SELECT 1276, 'NAICS', 482111, 'Line-Haul Railroads', 5, 1275 FROM dual
UNION ALL SELECT 1277, 'NAICS', 482112, 'Short Line Railroads', 5, 1275 FROM dual
UNION ALL SELECT 1279, 'NAICS', 4831, 'Deep Sea, Coastal, and Great Lakes Water Transportation', 3, 1278 FROM dual
UNION ALL SELECT 1285, 'NAICS', 4832, 'Inland Water Transportation', 3, 1278 FROM dual
UNION ALL SELECT 1280, 'NAICS', 48311, 'Deep Sea, Coastal, and Great Lakes Water Transportation', 4, 1279 FROM dual
UNION ALL SELECT 1283, 'NAICS', 483113, 'Coastal and Great Lakes Freight Transportation', 5, 1280 FROM dual
UNION ALL SELECT 1282, 'NAICS', 483112, 'Deep Sea Passenger Transportation', 5, 1280 FROM dual
UNION ALL SELECT 1284, 'NAICS', 483114, 'Coastal and Great Lakes Passenger Transportation', 5, 1280 FROM dual
UNION ALL SELECT 1281, 'NAICS', 483111, 'Deep Sea Freight Transportation', 5, 1280 FROM dual
UNION ALL SELECT 1286, 'NAICS', 48321, 'Inland Water Transportation', 4, 1285 FROM dual
UNION ALL SELECT 1288, 'NAICS', 483212, 'Inland Water Passenger Transportation', 5, 1286 FROM dual
UNION ALL SELECT 1287, 'NAICS', 483211, 'Inland Water Freight Transportation', 5, 1286 FROM dual
UNION ALL SELECT 1296, 'NAICS', 4842, 'Specialized Freight Trucking', 3, 1289 FROM dual
UNION ALL SELECT 1290, 'NAICS', 4841, 'General Freight Trucking', 3, 1289 FROM dual
UNION ALL SELECT 1293, 'NAICS', 48412, 'General Freight Trucking, Long-Distance', 4, 1290 FROM dual
UNION ALL SELECT 1292, 'NAICS', 48411, 'General Freight Trucking, Local', 4, 1290 FROM dual
UNION ALL SELECT 1291, 'NAICS', 484110, 'General Freight Trucking, Local', 5, 1292 FROM dual
UNION ALL SELECT 1295, 'NAICS', 484122, 'General Freight Trucking, Long-Distance, Less Than Truckload', 5, 1293 FROM dual
UNION ALL SELECT 1294, 'NAICS', 484121, 'General Freight Trucking, Long-Distance, Truckload', 5, 1293 FROM dual
UNION ALL SELECT 1298, 'NAICS', 48421, 'Used Household and Office Goods Moving', 4, 1296 FROM dual
UNION ALL SELECT 1300, 'NAICS', 48422, 'Specialized Freight (except Used Goods) Trucking, Local', 4, 1296 FROM dual
UNION ALL SELECT 1302, 'NAICS', 48423, 'Specialized Freight (except Used Goods) Trucking, Long-Distance', 4, 1296 FROM dual
UNION ALL SELECT 1297, 'NAICS', 484210, 'Used Household and Office Goods Moving', 5, 1298 FROM dual
UNION ALL SELECT 1299, 'NAICS', 484220, 'Specialized Freight (except Used Goods) Trucking, Local', 5, 1300 FROM dual
UNION ALL SELECT 1301, 'NAICS', 484230, 'Specialized Freight (except Used Goods) Trucking, Long-Distance', 5, 1302 FROM dual
UNION ALL SELECT 1310, 'NAICS', 4852, 'Interurban and Rural Bus Transportation', 3, 1303 FROM dual
UNION ALL SELECT 1304, 'NAICS', 4851, 'Urban Transit Systems', 3, 1303 FROM dual
UNION ALL SELECT 1517, 'NAICS', 523120, 'Securities Brokerage', 5, 1518 FROM dual
UNION ALL SELECT 1324, 'NAICS', 4859, 'Other Transit and Ground Passenger Transportation', 3, 1303 FROM dual
UNION ALL SELECT 1313, 'NAICS', 4853, 'Taxi and Limousine Service', 3, 1303 FROM dual
UNION ALL SELECT 1318, 'NAICS', 4854, 'School and Employee Bus Transportation', 3, 1303 FROM dual
UNION ALL SELECT 1305, 'NAICS', 48511, 'Urban Transit Systems', 4, 1304 FROM dual
UNION ALL SELECT 1308, 'NAICS', 485113, 'Bus and Other Motor Vehicle Transit Systems', 5, 1305 FROM dual
UNION ALL SELECT 1309, 'NAICS', 485119, 'Other Urban Transit Systems', 5, 1305 FROM dual
UNION ALL SELECT 1306, 'NAICS', 485111, 'Mixed Mode Transit Systems', 5, 1305 FROM dual
UNION ALL SELECT 1307, 'NAICS', 485112, 'Commuter Rail Systems', 5, 1305 FROM dual
UNION ALL SELECT 1312, 'NAICS', 48521, 'Interurban and Rural Bus Transportation', 4, 1310 FROM dual
UNION ALL SELECT 1311, 'NAICS', 485210, 'Interurban and Rural Bus Transportation', 5, 1312 FROM dual
UNION ALL SELECT 1315, 'NAICS', 48531, 'Taxi Service', 4, 1313 FROM dual
UNION ALL SELECT 1317, 'NAICS', 48532, 'Limousine Service', 4, 1313 FROM dual
UNION ALL SELECT 1314, 'NAICS', 485310, 'Taxi Service', 5, 1315 FROM dual
UNION ALL SELECT 1316, 'NAICS', 485320, 'Limousine Service', 5, 1317 FROM dual
UNION ALL SELECT 1320, 'NAICS', 48541, 'School and Employee Bus Transportation', 4, 1318 FROM dual
UNION ALL SELECT 1319, 'NAICS', 485410, 'School and Employee Bus Transportation', 5, 1320 FROM dual
UNION ALL SELECT 1323, 'NAICS', 48551, 'Charter Bus Industry', 4, 1321 FROM dual
UNION ALL SELECT 1322, 'NAICS', 485510, 'Charter Bus Industry', 5, 1323 FROM dual
UNION ALL SELECT 1325, 'NAICS', 48599, 'Other Transit and Ground Passenger Transportation', 4, 1324 FROM dual
UNION ALL SELECT 1327, 'NAICS', 485999, 'All Other Transit and Ground Passenger Transportation', 5, 1325 FROM dual
UNION ALL SELECT 1326, 'NAICS', 485991, 'Special Needs Transportation', 5, 1325 FROM dual
UNION ALL SELECT 1329, 'NAICS', 4861, 'Pipeline Transportation of Crude Oil', 3, 1328 FROM dual
UNION ALL SELECT 1335, 'NAICS', 4869, 'Other Pipeline Transportation', 3, 1328 FROM dual
UNION ALL SELECT 1332, 'NAICS', 4862, 'Pipeline Transportation of Natural Gas', 3, 1328 FROM dual
UNION ALL SELECT 1331, 'NAICS', 48611, 'Pipeline Transportation of Crude Oil', 4, 1329 FROM dual
UNION ALL SELECT 1330, 'NAICS', 486110, 'Pipeline Transportation of Crude Oil', 5, 1331 FROM dual
UNION ALL SELECT 1334, 'NAICS', 48621, 'Pipeline Transportation of Natural Gas', 4, 1332 FROM dual
UNION ALL SELECT 1333, 'NAICS', 486210, 'Pipeline Transportation of Natural Gas', 5, 1334 FROM dual
UNION ALL SELECT 1337, 'NAICS', 48691, 'Pipeline Transportation of Refined Petroleum Products', 4, 1335 FROM dual
UNION ALL SELECT 1339, 'NAICS', 48699, 'All Other Pipeline Transportation', 4, 1335 FROM dual
UNION ALL SELECT 1336, 'NAICS', 486910, 'Pipeline Transportation of Refined Petroleum Products', 5, 1337 FROM dual
UNION ALL SELECT 1338, 'NAICS', 486990, 'All Other Pipeline Transportation', 5, 1339 FROM dual
UNION ALL SELECT 1344, 'NAICS', 4872, 'Scenic and Sightseeing Transportation, Water', 3, 1340 FROM dual
UNION ALL SELECT 1347, 'NAICS', 4879, 'Scenic and Sightseeing Transportation, Other', 3, 1340 FROM dual
UNION ALL SELECT 1341, 'NAICS', 4871, 'Scenic and Sightseeing Transportation, Land', 3, 1340 FROM dual
UNION ALL SELECT 1343, 'NAICS', 48711, 'Scenic and Sightseeing Transportation, Land', 4, 1341 FROM dual
UNION ALL SELECT 1342, 'NAICS', 487110, 'Scenic and Sightseeing Transportation, Land', 5, 1343 FROM dual
UNION ALL SELECT 1346, 'NAICS', 48721, 'Scenic and Sightseeing Transportation, Water', 4, 1344 FROM dual
UNION ALL SELECT 1345, 'NAICS', 487210, 'Scenic and Sightseeing Transportation, Water', 5, 1346 FROM dual
UNION ALL SELECT 1349, 'NAICS', 48799, 'Scenic and Sightseeing Transportation, Other', 4, 1347 FROM dual
UNION ALL SELECT 1348, 'NAICS', 487990, 'Scenic and Sightseeing Transportation, Other', 5, 1349 FROM dual
UNION ALL SELECT 1377, 'NAICS', 4889, 'Other Support Activities for Transportation', 3, 1350 FROM dual
UNION ALL SELECT 1351, 'NAICS', 4881, 'Support Activities for Air Transportation', 3, 1350 FROM dual
UNION ALL SELECT 1357, 'NAICS', 4882, 'Support Activities for Rail Transportation', 3, 1350 FROM dual
UNION ALL SELECT 1360, 'NAICS', 4883, 'Support Activities for Water Transportation', 3, 1350 FROM dual
UNION ALL SELECT 1369, 'NAICS', 4884, 'Support Activities for Road Transportation', 3, 1350 FROM dual
UNION ALL SELECT 1374, 'NAICS', 4885, 'Freight Transportation Arrangement', 3, 1350 FROM dual
UNION ALL SELECT 1352, 'NAICS', 48811, 'Airport Operations', 4, 1351 FROM dual
UNION ALL SELECT 1356, 'NAICS', 48819, 'Other Support Activities for Air Transportation', 4, 1351 FROM dual
UNION ALL SELECT 1353, 'NAICS', 488111, 'Air Traffic Control', 5, 1352 FROM dual
UNION ALL SELECT 1354, 'NAICS', 488119, 'Other Airport Operations', 5, 1352 FROM dual
UNION ALL SELECT 1355, 'NAICS', 488190, 'Other Support Activities for Air Transportation', 5, 1356 FROM dual
UNION ALL SELECT 1359, 'NAICS', 48821, 'Support Activities for Rail Transportation', 4, 1357 FROM dual
UNION ALL SELECT 1358, 'NAICS', 488210, 'Support Activities for Rail Transportation', 5, 1359 FROM dual
UNION ALL SELECT 1362, 'NAICS', 48831, 'Port and Harbor Operations', 4, 1360 FROM dual
UNION ALL SELECT 1366, 'NAICS', 48833, 'Navigational Services to Shipping', 4, 1360 FROM dual
UNION ALL SELECT 1368, 'NAICS', 48839, 'Other Support Activities for Water Transportation', 4, 1360 FROM dual
UNION ALL SELECT 1364, 'NAICS', 48832, 'Marine Cargo Handling', 4, 1360 FROM dual
UNION ALL SELECT 1361, 'NAICS', 488310, 'Port and Harbor Operations', 5, 1362 FROM dual
UNION ALL SELECT 1363, 'NAICS', 488320, 'Marine Cargo Handling', 5, 1364 FROM dual
UNION ALL SELECT 1365, 'NAICS', 488330, 'Navigational Services to Shipping', 5, 1366 FROM dual
UNION ALL SELECT 1367, 'NAICS', 488390, 'Other Support Activities for Water Transportation', 5, 1368 FROM dual
UNION ALL SELECT 1373, 'NAICS', 48849, 'Other Support Activities for Road Transportation', 4, 1369 FROM dual
UNION ALL SELECT 1371, 'NAICS', 48841, 'Motor Vehicle Towing', 4, 1369 FROM dual
UNION ALL SELECT 1370, 'NAICS', 488410, 'Motor Vehicle Towing', 5, 1371 FROM dual
UNION ALL SELECT 1372, 'NAICS', 488490, 'Other Support Activities for Road Transportation', 5, 1373 FROM dual
UNION ALL SELECT 1376, 'NAICS', 48851, 'Freight Transportation Arrangement', 4, 1374 FROM dual
UNION ALL SELECT 1375, 'NAICS', 488510, 'Freight Transportation Arrangement', 5, 1376 FROM dual
UNION ALL SELECT 1378, 'NAICS', 48899, 'Other Support Activities for Transportation', 4, 1377 FROM dual
UNION ALL SELECT 1379, 'NAICS', 488991, 'Packing and Crating', 5, 1378 FROM dual
UNION ALL SELECT 1380, 'NAICS', 488999, 'All Other Support Activities for Transportation', 5, 1378 FROM dual
UNION ALL SELECT 1382, 'NAICS', 4911, 'Postal Service', 3, 1381 FROM dual
UNION ALL SELECT 1384, 'NAICS', 49111, 'Postal Service', 4, 1382 FROM dual
UNION ALL SELECT 1383, 'NAICS', 491110, 'Postal Service', 5, 1384 FROM dual
UNION ALL SELECT 1386, 'NAICS', 4921, 'Couriers and Express Delivery Services', 3, 1385 FROM dual
UNION ALL SELECT 1389, 'NAICS', 4922, 'Local Messengers and Local Delivery', 3, 1385 FROM dual
UNION ALL SELECT 1388, 'NAICS', 49211, 'Couriers and Express Delivery Services', 4, 1386 FROM dual
UNION ALL SELECT 1387, 'NAICS', 492110, 'Couriers and Express Delivery Services', 5, 1388 FROM dual
UNION ALL SELECT 1391, 'NAICS', 49221, 'Local Messengers and Local Delivery', 4, 1389 FROM dual
UNION ALL SELECT 1390, 'NAICS', 492210, 'Local Messengers and Local Delivery', 5, 1391 FROM dual
UNION ALL SELECT 1393, 'NAICS', 4931, 'Warehousing and Storage', 3, 1392 FROM dual
UNION ALL SELECT 1401, 'NAICS', 49319, 'Other Warehousing and Storage', 4, 1393 FROM dual
UNION ALL SELECT 1395, 'NAICS', 49311, 'General Warehousing and Storage', 4, 1393 FROM dual
UNION ALL SELECT 1397, 'NAICS', 49312, 'Refrigerated Warehousing and Storage', 4, 1393 FROM dual
UNION ALL SELECT 1399, 'NAICS', 49313, 'Farm Product Warehousing and Storage', 4, 1393 FROM dual
UNION ALL SELECT 1394, 'NAICS', 493110, 'General Warehousing and Storage', 5, 1395 FROM dual
UNION ALL SELECT 1396, 'NAICS', 493120, 'Refrigerated Warehousing and Storage', 5, 1397 FROM dual
UNION ALL SELECT 1398, 'NAICS', 493130, 'Farm Product Warehousing and Storage', 5, 1399 FROM dual
UNION ALL SELECT 1400, 'NAICS', 493190, 'Other Warehousing and Storage', 5, 1401 FROM dual
UNION ALL SELECT 1403, 'NAICS', 511, 'Publishing Industries (except Internet)', 2, 1402 FROM dual
UNION ALL SELECT 1442, 'NAICS', 515, 'Broadcasting (except Internet)', 2, 1402 FROM dual
UNION ALL SELECT 1470, 'NAICS', 519, 'Other Information Services', 2, 1402 FROM dual
UNION ALL SELECT 1419, 'NAICS', 512, 'Motion Picture and Sound Recording Industries', 2, 1402 FROM dual
UNION ALL SELECT 1452, 'NAICS', 517, 'Telecommunications', 2, 1402 FROM dual
UNION ALL SELECT 1466, 'NAICS', 518, 'Data Processing, Hosting, and Related Services', 2, 1402 FROM dual
UNION ALL SELECT 1416, 'NAICS', 5112, 'Software Publishers', 3, 1403 FROM dual
UNION ALL SELECT 1404, 'NAICS', 5111, 'Newspaper, Periodical, Book, and Directory Publishers', 3, 1403 FROM dual
UNION ALL SELECT 1412, 'NAICS', 51114, 'Directory and Mailing List Publishers', 4, 1404 FROM dual
UNION ALL SELECT 1413, 'NAICS', 51119, 'Other Publishers', 4, 1404 FROM dual
UNION ALL SELECT 1408, 'NAICS', 51112, 'Periodical Publishers', 4, 1404 FROM dual
UNION ALL SELECT 1410, 'NAICS', 51113, 'Book Publishers', 4, 1404 FROM dual
UNION ALL SELECT 1406, 'NAICS', 51111, 'Newspaper Publishers', 4, 1404 FROM dual
UNION ALL SELECT 1405, 'NAICS', 511110, 'Newspaper Publishers', 5, 1406 FROM dual
UNION ALL SELECT 1407, 'NAICS', 511120, 'Periodical Publishers', 5, 1408 FROM dual
UNION ALL SELECT 1409, 'NAICS', 511130, 'Book Publishers', 5, 1410 FROM dual
UNION ALL SELECT 1411, 'NAICS', 511140, 'Directory and Mailing List Publishers', 5, 1412 FROM dual
UNION ALL SELECT 1414, 'NAICS', 511191, 'Greeting Card Publishers', 5, 1413 FROM dual
UNION ALL SELECT 1415, 'NAICS', 511199, 'All Other Publishers', 5, 1413 FROM dual
UNION ALL SELECT 1418, 'NAICS', 51121, 'Software Publishers', 4, 1416 FROM dual
UNION ALL SELECT 1417, 'NAICS', 511210, 'Software Publishers', 5, 1418 FROM dual
UNION ALL SELECT 1420, 'NAICS', 5121, 'Motion Picture and Video Industries', 3, 1419 FROM dual
UNION ALL SELECT 1431, 'NAICS', 5122, 'Sound Recording Industries', 3, 1419 FROM dual
UNION ALL SELECT 1428, 'NAICS', 51219, 'Postproduction Services and Other Motion Picture and Video Industries', 4, 1420 FROM dual
UNION ALL SELECT 1422, 'NAICS', 51211, 'Motion Picture and Video Production', 4, 1420 FROM dual
UNION ALL SELECT 1424, 'NAICS', 51212, 'Motion Picture and Video Distribution', 4, 1420 FROM dual
UNION ALL SELECT 1425, 'NAICS', 51213, 'Motion Picture and Video Exhibition', 4, 1420 FROM dual
UNION ALL SELECT 1421, 'NAICS', 512110, 'Motion Picture and Video Production', 5, 1422 FROM dual
UNION ALL SELECT 1423, 'NAICS', 512120, 'Motion Picture and Video Distribution', 5, 1424 FROM dual
UNION ALL SELECT 1426, 'NAICS', 512131, 'Motion Picture Theaters (except Drive-Ins)', 5, 1425 FROM dual
UNION ALL SELECT 1427, 'NAICS', 512132, 'Drive-In Motion Picture Theaters', 5, 1425 FROM dual
UNION ALL SELECT 1430, 'NAICS', 512199, 'Other Motion Picture and Video Industries', 5, 1428 FROM dual
UNION ALL SELECT 1429, 'NAICS', 512191, 'Teleproduction and Other Postproduction Services', 5, 1428 FROM dual
UNION ALL SELECT 1441, 'NAICS', 51229, 'Other Sound Recording Industries', 4, 1431 FROM dual
UNION ALL SELECT 1435, 'NAICS', 51222, 'Integrated Record Production/Distribution', 4, 1431 FROM dual
UNION ALL SELECT 1437, 'NAICS', 51223, 'Music Publishers', 4, 1431 FROM dual
UNION ALL SELECT 1433, 'NAICS', 51221, 'Record Production', 4, 1431 FROM dual
UNION ALL SELECT 1439, 'NAICS', 51224, 'Sound Recording Studios', 4, 1431 FROM dual
UNION ALL SELECT 1432, 'NAICS', 512210, 'Record Production', 5, 1433 FROM dual
UNION ALL SELECT 1434, 'NAICS', 512220, 'Integrated Record Production/Distribution', 5, 1435 FROM dual
UNION ALL SELECT 1436, 'NAICS', 512230, 'Music Publishers', 5, 1437 FROM dual
UNION ALL SELECT 1438, 'NAICS', 512240, 'Sound Recording Studios', 5, 1439 FROM dual
UNION ALL SELECT 1440, 'NAICS', 512290, 'Other Sound Recording Industries', 5, 1441 FROM dual
UNION ALL SELECT 1449, 'NAICS', 5152, 'Cable and Other Subscription Programming', 3, 1442 FROM dual
UNION ALL SELECT 1443, 'NAICS', 5151, 'Radio and Television Broadcasting', 3, 1442 FROM dual
UNION ALL SELECT 1444, 'NAICS', 51511, 'Radio Broadcasting', 4, 1443 FROM dual
UNION ALL SELECT 1448, 'NAICS', 51512, 'Television Broadcasting', 4, 1443 FROM dual
UNION ALL SELECT 1445, 'NAICS', 515111, 'Radio Networks', 5, 1444 FROM dual
UNION ALL SELECT 1446, 'NAICS', 515112, 'Radio Stations', 5, 1444 FROM dual
UNION ALL SELECT 1447, 'NAICS', 515120, 'Television Broadcasting', 5, 1448 FROM dual
UNION ALL SELECT 1451, 'NAICS', 51521, 'Cable and Other Subscription Programming', 4, 1449 FROM dual
UNION ALL SELECT 1450, 'NAICS', 515210, 'Cable and Other Subscription Programming', 5, 1451 FROM dual
UNION ALL SELECT 1462, 'NAICS', 5179, 'Other Telecommunications', 3, 1452 FROM dual
UNION ALL SELECT 1456, 'NAICS', 5172, 'Wireless Telecommunications Carriers (except Satellite)', 3, 1452 FROM dual
UNION ALL SELECT 1459, 'NAICS', 5174, 'Satellite Telecommunications', 3, 1452 FROM dual
UNION ALL SELECT 1453, 'NAICS', 5171, 'Wired Telecommunications Carriers', 3, 1452 FROM dual
UNION ALL SELECT 1455, 'NAICS', 51711, 'Wired Telecommunications Carriers', 4, 1453 FROM dual
UNION ALL SELECT 1454, 'NAICS', 517110, 'Wired Telecommunications Carriers', 5, 1455 FROM dual
UNION ALL SELECT 1458, 'NAICS', 51721, 'Wireless Telecommunications Carriers (except Satellite)', 4, 1456 FROM dual
UNION ALL SELECT 1457, 'NAICS', 517210, 'Wireless Telecommunications Carriers (except Satellite)', 5, 1458 FROM dual
UNION ALL SELECT 1461, 'NAICS', 51741, 'Satellite Telecommunications', 4, 1459 FROM dual
UNION ALL SELECT 1460, 'NAICS', 517410, 'Satellite Telecommunications', 5, 1461 FROM dual
UNION ALL SELECT 1463, 'NAICS', 51791, 'Other Telecommunications', 4, 1462 FROM dual
UNION ALL SELECT 1465, 'NAICS', 517919, 'All Other Telecommunications', 5, 1463 FROM dual
UNION ALL SELECT 1464, 'NAICS', 517911, 'Telecommunications Resellers', 5, 1463 FROM dual
UNION ALL SELECT 1467, 'NAICS', 5182, 'Data Processing, Hosting, and Related Services', 3, 1466 FROM dual
UNION ALL SELECT 1469, 'NAICS', 51821, 'Data Processing, Hosting, and Related Services', 4, 1467 FROM dual
UNION ALL SELECT 1468, 'NAICS', 518210, 'Data Processing, Hosting, and Related Services', 5, 1469 FROM dual
UNION ALL SELECT 1471, 'NAICS', 5191, 'Other Information Services', 3, 1470 FROM dual
UNION ALL SELECT 1479, 'NAICS', 51919, 'All Other Information Services', 4, 1471 FROM dual
UNION ALL SELECT 1473, 'NAICS', 51911, 'News Syndicates', 4, 1471 FROM dual
UNION ALL SELECT 1475, 'NAICS', 51912, 'Libraries and Archives', 4, 1471 FROM dual
UNION ALL SELECT 1477, 'NAICS', 51913, 'Internet Publishing and Broadcasting and Web Search Portals', 4, 1471 FROM dual
UNION ALL SELECT 1472, 'NAICS', 519110, 'News Syndicates', 5, 1473 FROM dual
UNION ALL SELECT 1474, 'NAICS', 519120, 'Libraries and Archives', 5, 1475 FROM dual
UNION ALL SELECT 1476, 'NAICS', 519130, 'Internet Publishing and Broadcasting and Web Search Portals', 5, 1477 FROM dual
UNION ALL SELECT 1478, 'NAICS', 519190, 'All Other Information Services', 5, 1479 FROM dual
UNION ALL SELECT 1485, 'NAICS', 522, 'Credit Intermediation and Related Activities', 2, 1480 FROM dual
UNION ALL SELECT 1554, 'NAICS', 525, 'Funds, Trusts, and Other Financial Vehicles', 2, 1480 FROM dual
UNION ALL SELECT 1481, 'NAICS', 521, 'Monetary Authorities-Central Bank', 2, 1480 FROM dual
UNION ALL SELECT 1513, 'NAICS', 523, 'Securities, Commodity Contracts, and Other Financial Investments and Related Activities', 2, 1480 FROM dual
UNION ALL SELECT 1536, 'NAICS', 524, 'Insurance Carriers and Related Activities', 2, 1480 FROM dual
UNION ALL SELECT 1482, 'NAICS', 5211, 'Monetary Authorities-Central Bank', 3, 1481 FROM dual
UNION ALL SELECT 1484, 'NAICS', 52111, 'Monetary Authorities-Central Bank', 4, 1482 FROM dual
UNION ALL SELECT 1483, 'NAICS', 521110, 'Monetary Authorities-Central Bank', 5, 1484 FROM dual
UNION ALL SELECT 1486, 'NAICS', 5221, 'Depository Credit Intermediation', 3, 1485 FROM dual
UNION ALL SELECT 1495, 'NAICS', 5222, 'Nondepository Credit Intermediation', 3, 1485 FROM dual
UNION ALL SELECT 1506, 'NAICS', 5223, 'Activities Related to Credit Intermediation', 3, 1485 FROM dual
UNION ALL SELECT 1488, 'NAICS', 52211, 'Commercial Banking', 4, 1486 FROM dual
UNION ALL SELECT 1492, 'NAICS', 52213, 'Credit Unions', 4, 1486 FROM dual
UNION ALL SELECT 1494, 'NAICS', 52219, 'Other Depository Credit Intermediation', 4, 1486 FROM dual
UNION ALL SELECT 1490, 'NAICS', 52212, 'Savings Institutions', 4, 1486 FROM dual
UNION ALL SELECT 1487, 'NAICS', 522110, 'Commercial Banking', 5, 1488 FROM dual
UNION ALL SELECT 1489, 'NAICS', 522120, 'Savings Institutions', 5, 1490 FROM dual
UNION ALL SELECT 1491, 'NAICS', 522130, 'Credit Unions', 5, 1492 FROM dual
UNION ALL SELECT 1493, 'NAICS', 522190, 'Other Depository Credit Intermediation', 5, 1494 FROM dual
UNION ALL SELECT 1497, 'NAICS', 52221, 'Credit Card Issuing', 4, 1495 FROM dual
UNION ALL SELECT 1499, 'NAICS', 52222, 'Sales Financing', 4, 1495 FROM dual
UNION ALL SELECT 1500, 'NAICS', 52229, 'Other Nondepository Credit Intermediation', 4, 1495 FROM dual
UNION ALL SELECT 1496, 'NAICS', 522210, 'Credit Card Issuing', 5, 1497 FROM dual
UNION ALL SELECT 1498, 'NAICS', 522220, 'Sales Financing', 5, 1499 FROM dual
UNION ALL SELECT 1504, 'NAICS', 522294, 'Secondary Market Financing', 5, 1500 FROM dual
UNION ALL SELECT 1505, 'NAICS', 522298, 'All Other Nondepository Credit Intermediation', 5, 1500 FROM dual
UNION ALL SELECT 1502, 'NAICS', 522292, 'Real Estate Credit', 5, 1500 FROM dual
UNION ALL SELECT 1503, 'NAICS', 522293, 'International Trade Financing', 5, 1500 FROM dual
UNION ALL SELECT 1501, 'NAICS', 522291, 'Consumer Lending', 5, 1500 FROM dual
UNION ALL SELECT 1510, 'NAICS', 52232, 'Financial Transactions Processing, Reserve, and Clearinghouse Activities', 4, 1506 FROM dual
UNION ALL SELECT 1508, 'NAICS', 52231, 'Mortgage and Nonmortgage Loan Brokers', 4, 1506 FROM dual
UNION ALL SELECT 1512, 'NAICS', 52239, 'Other Activities Related to Credit Intermediation', 4, 1506 FROM dual
UNION ALL SELECT 1507, 'NAICS', 522310, 'Mortgage and Nonmortgage Loan Brokers', 5, 1508 FROM dual
UNION ALL SELECT 1509, 'NAICS', 522320, 'Financial Transactions Processing, Reserve, and Clearinghouse Activities', 5, 1510 FROM dual
UNION ALL SELECT 1511, 'NAICS', 522390, 'Other Activities Related to Credit Intermediation', 5, 1512 FROM dual
UNION ALL SELECT 1514, 'NAICS', 5231, 'Securities and Commodity Contracts Intermediation and Brokerage', 3, 1513 FROM dual
UNION ALL SELECT 1526, 'NAICS', 5239, 'Other Financial Investment Activities', 3, 1513 FROM dual
UNION ALL SELECT 1523, 'NAICS', 5232, 'Securities and Commodity Exchanges', 3, 1513 FROM dual
UNION ALL SELECT 1518, 'NAICS', 52312, 'Securities Brokerage', 4, 1514 FROM dual
UNION ALL SELECT 1516, 'NAICS', 52311, 'Investment Banking and Securities Dealing', 4, 1514 FROM dual
UNION ALL SELECT 1520, 'NAICS', 52313, 'Commodity Contracts Dealing', 4, 1514 FROM dual
UNION ALL SELECT 1522, 'NAICS', 52314, 'Commodity Contracts Brokerage', 4, 1514 FROM dual
UNION ALL SELECT 1515, 'NAICS', 523110, 'Investment Banking and Securities Dealing', 5, 1516 FROM dual
UNION ALL SELECT 1519, 'NAICS', 523130, 'Commodity Contracts Dealing', 5, 1520 FROM dual
UNION ALL SELECT 1521, 'NAICS', 523140, 'Commodity Contracts Brokerage', 5, 1522 FROM dual
UNION ALL SELECT 1525, 'NAICS', 52321, 'Securities and Commodity Exchanges', 4, 1523 FROM dual
UNION ALL SELECT 1524, 'NAICS', 523210, 'Securities and Commodity Exchanges', 5, 1525 FROM dual
UNION ALL SELECT 1530, 'NAICS', 52392, 'Portfolio Management', 4, 1526 FROM dual
UNION ALL SELECT 1528, 'NAICS', 52391, 'Miscellaneous Intermediation', 4, 1526 FROM dual
UNION ALL SELECT 1532, 'NAICS', 52393, 'Investment Advice', 4, 1526 FROM dual
UNION ALL SELECT 1533, 'NAICS', 52399, 'All Other Financial Investment Activities', 4, 1526 FROM dual
UNION ALL SELECT 1527, 'NAICS', 523910, 'Miscellaneous Intermediation', 5, 1528 FROM dual
UNION ALL SELECT 1529, 'NAICS', 523920, 'Portfolio Management', 5, 1530 FROM dual
UNION ALL SELECT 1531, 'NAICS', 523930, 'Investment Advice', 5, 1532 FROM dual
UNION ALL SELECT 1535, 'NAICS', 523999, 'Miscellaneous Financial Investment Activities', 5, 1533 FROM dual
UNION ALL SELECT 1534, 'NAICS', 523991, 'Trust, Fiduciary, and Custody Activities', 5, 1533 FROM dual
UNION ALL SELECT 1537, 'NAICS', 5241, 'Insurance Carriers', 3, 1536 FROM dual
UNION ALL SELECT 1547, 'NAICS', 5242, 'Agencies, Brokerages, and Other Insurance Related Activities', 3, 1536 FROM dual
UNION ALL SELECT 1538, 'NAICS', 52411, 'Direct Life, Health, and Medical Insurance Carriers', 4, 1537 FROM dual
UNION ALL SELECT 1546, 'NAICS', 52413, 'Reinsurance Carriers', 4, 1537 FROM dual
UNION ALL SELECT 1541, 'NAICS', 52412, 'Direct Insurance (except Life, Health, and Medical) Carriers', 4, 1537 FROM dual
UNION ALL SELECT 1539, 'NAICS', 524113, 'Direct Life Insurance Carriers', 5, 1538 FROM dual
UNION ALL SELECT 1540, 'NAICS', 524114, 'Direct Health and Medical Insurance Carriers', 5, 1538 FROM dual
UNION ALL SELECT 1920, 'NAICS', 624, 'Social Assistance', 2, 1850 FROM dual
UNION ALL SELECT 1544, 'NAICS', 524128, 'Other Direct Insurance (except Life, Health, and Medical) Carriers', 5, 1541 FROM dual
UNION ALL SELECT 1543, 'NAICS', 524127, 'Direct Title Insurance Carriers', 5, 1541 FROM dual
UNION ALL SELECT 1542, 'NAICS', 524126, 'Direct Property and Casualty Insurance Carriers', 5, 1541 FROM dual
UNION ALL SELECT 1545, 'NAICS', 524130, 'Reinsurance Carriers', 5, 1546 FROM dual
UNION ALL SELECT 1550, 'NAICS', 52429, 'Other Insurance Related Activities', 4, 1547 FROM dual
UNION ALL SELECT 1549, 'NAICS', 52421, 'Insurance Agencies and Brokerages', 4, 1547 FROM dual
UNION ALL SELECT 1548, 'NAICS', 524210, 'Insurance Agencies and Brokerages', 5, 1549 FROM dual
UNION ALL SELECT 1551, 'NAICS', 524291, 'Claims Adjusting', 5, 1550 FROM dual
UNION ALL SELECT 1552, 'NAICS', 524292, 'Third Party Administration of Insurance and Pension Funds', 5, 1550 FROM dual
UNION ALL SELECT 1553, 'NAICS', 524298, 'All Other Insurance Related Activities', 5, 1550 FROM dual
UNION ALL SELECT 1562, 'NAICS', 5259, 'Other Investment Pools and Funds', 3, 1554 FROM dual
UNION ALL SELECT 1555, 'NAICS', 5251, 'Insurance and Employee Benefit Funds', 3, 1554 FROM dual
UNION ALL SELECT 1557, 'NAICS', 52511, 'Pension Funds', 4, 1555 FROM dual
UNION ALL SELECT 1559, 'NAICS', 52512, 'Health and Welfare Funds', 4, 1555 FROM dual
UNION ALL SELECT 1561, 'NAICS', 52519, 'Other Insurance Funds', 4, 1555 FROM dual
UNION ALL SELECT 1556, 'NAICS', 525110, 'Pension Funds', 5, 1557 FROM dual
UNION ALL SELECT 1558, 'NAICS', 525120, 'Health and Welfare Funds', 5, 1559 FROM dual
UNION ALL SELECT 1560, 'NAICS', 525190, 'Other Insurance Funds', 5, 1561 FROM dual
UNION ALL SELECT 1568, 'NAICS', 52599, 'Other Financial Vehicles', 4, 1562 FROM dual
UNION ALL SELECT 1564, 'NAICS', 52591, 'Open-End Investment Funds', 4, 1562 FROM dual
UNION ALL SELECT 1566, 'NAICS', 52592, 'Trusts, Estates, and Agency Accounts', 4, 1562 FROM dual
UNION ALL SELECT 1563, 'NAICS', 525910, 'Open-End Investment Funds', 5, 1564 FROM dual
UNION ALL SELECT 1565, 'NAICS', 525920, 'Trusts, Estates, and Agency Accounts', 5, 1566 FROM dual
UNION ALL SELECT 1567, 'NAICS', 525990, 'Other Financial Vehicles', 5, 1568 FROM dual
UNION ALL SELECT 1620, 'NAICS', 533, 'Lessors of Nonfinancial Intangible Assets (except Copyrighted Works)', 2, 1569 FROM dual
UNION ALL SELECT 1570, 'NAICS', 531, 'Real Estate', 2, 1569 FROM dual
UNION ALL SELECT 1580, 'NAICS', 5312, 'Offices of Real Estate Agents and Brokers', 3, 1570 FROM dual
UNION ALL SELECT 1571, 'NAICS', 5311, 'Lessors of Real Estate', 3, 1570 FROM dual
UNION ALL SELECT 1583, 'NAICS', 5313, 'Activities Related to Real Estate', 3, 1570 FROM dual
UNION ALL SELECT 1575, 'NAICS', 53112, 'Lessors of Nonresidential Buildings (except Miniwarehouses)', 4, 1571 FROM dual
UNION ALL SELECT 1577, 'NAICS', 53113, 'Lessors of Miniwarehouses and Self-Storage Units', 4, 1571 FROM dual
UNION ALL SELECT 1579, 'NAICS', 53119, 'Lessors of Other Real Estate Property', 4, 1571 FROM dual
UNION ALL SELECT 1573, 'NAICS', 53111, 'Lessors of Residential Buildings and Dwellings', 4, 1571 FROM dual
UNION ALL SELECT 1572, 'NAICS', 531110, 'Lessors of Residential Buildings and Dwellings', 5, 1573 FROM dual
UNION ALL SELECT 1574, 'NAICS', 531120, 'Lessors of Nonresidential Buildings (except Miniwarehouses)', 5, 1575 FROM dual
UNION ALL SELECT 1576, 'NAICS', 531130, 'Lessors of Miniwarehouses and Self-Storage Units', 5, 1577 FROM dual
UNION ALL SELECT 1578, 'NAICS', 531190, 'Lessors of Other Real Estate Property', 5, 1579 FROM dual
UNION ALL SELECT 1582, 'NAICS', 53121, 'Offices of Real Estate Agents and Brokers', 4, 1580 FROM dual
UNION ALL SELECT 1581, 'NAICS', 531210, 'Offices of Real Estate Agents and Brokers', 5, 1582 FROM dual
UNION ALL SELECT 1588, 'NAICS', 53132, 'Offices of Real Estate Appraisers', 4, 1583 FROM dual
UNION ALL SELECT 1590, 'NAICS', 53139, 'Other Activities Related to Real Estate', 4, 1583 FROM dual
UNION ALL SELECT 1584, 'NAICS', 53131, 'Real Estate Property Managers', 4, 1583 FROM dual
UNION ALL SELECT 1585, 'NAICS', 531311, 'Residential Property Managers', 5, 1584 FROM dual
UNION ALL SELECT 1586, 'NAICS', 531312, 'Nonresidential Property Managers', 5, 1584 FROM dual
UNION ALL SELECT 1587, 'NAICS', 531320, 'Offices of Real Estate Appraisers', 5, 1588 FROM dual
UNION ALL SELECT 1589, 'NAICS', 531390, 'Other Activities Related to Real Estate', 5, 1590 FROM dual
UNION ALL SELECT 1609, 'NAICS', 5323, 'General Rental Centers', 3, 1591 FROM dual
UNION ALL SELECT 1612, 'NAICS', 5324, 'Commercial and Industrial Machinery and Equipment Rental and Leasing', 3, 1591 FROM dual
UNION ALL SELECT 1592, 'NAICS', 5321, 'Automotive Equipment Rental and Leasing', 3, 1591 FROM dual
UNION ALL SELECT 1598, 'NAICS', 5322, 'Consumer Goods Rental', 3, 1591 FROM dual
UNION ALL SELECT 1597, 'NAICS', 53212, 'Truck, Utility Trailer, and RV (Recreational Vehicle) Rental and Leasing', 4, 1592 FROM dual
UNION ALL SELECT 1593, 'NAICS', 53211, 'Passenger Car Rental and Leasing', 4, 1592 FROM dual
UNION ALL SELECT 1594, 'NAICS', 532111, 'Passenger Car Rental', 5, 1593 FROM dual
UNION ALL SELECT 1595, 'NAICS', 532112, 'Passenger Car Leasing', 5, 1593 FROM dual
UNION ALL SELECT 1596, 'NAICS', 532120, 'Truck, Utility Trailer, and RV (Recreational Vehicle) Rental and Leasing', 5, 1597 FROM dual
UNION ALL SELECT 1604, 'NAICS', 53223, 'Video Tape and Disc Rental', 4, 1598 FROM dual
UNION ALL SELECT 1605, 'NAICS', 53229, 'Other Consumer Goods Rental', 4, 1598 FROM dual
UNION ALL SELECT 1600, 'NAICS', 53221, 'Consumer Electronics and Appliances Rental', 4, 1598 FROM dual
UNION ALL SELECT 1602, 'NAICS', 53222, 'Formal Wear and Costume Rental', 4, 1598 FROM dual
UNION ALL SELECT 1599, 'NAICS', 532210, 'Consumer Electronics and Appliances Rental', 5, 1600 FROM dual
UNION ALL SELECT 1601, 'NAICS', 532220, 'Formal Wear and Costume Rental', 5, 1602 FROM dual
UNION ALL SELECT 1603, 'NAICS', 532230, 'Video Tape and Disc Rental', 5, 1604 FROM dual
UNION ALL SELECT 1608, 'NAICS', 532299, 'All Other Consumer Goods Rental', 5, 1605 FROM dual
UNION ALL SELECT 1607, 'NAICS', 532292, 'Recreational Goods Rental', 5, 1605 FROM dual
UNION ALL SELECT 1606, 'NAICS', 532291, 'Home Health Equipment Rental', 5, 1605 FROM dual
UNION ALL SELECT 1611, 'NAICS', 53231, 'General Rental Centers', 4, 1609 FROM dual
UNION ALL SELECT 1610, 'NAICS', 532310, 'General Rental Centers', 5, 1611 FROM dual
UNION ALL SELECT 1619, 'NAICS', 53249, 'Other Commercial and Industrial Machinery and Equipment Rental and Leasing', 4, 1612 FROM dual
UNION ALL SELECT 1613, 'NAICS', 53241, 'Construction, Transportation, Mining, and Forestry Machinery and Equipment Rental and Leasing', 4, 1612 FROM dual
UNION ALL SELECT 1617, 'NAICS', 53242, 'Office Machinery and Equipment Rental and Leasing', 4, 1612 FROM dual
UNION ALL SELECT 1614, 'NAICS', 532411, 'Commercial Air, Rail, and Water Transportation Equipment Rental and Leasing', 5, 1613 FROM dual
UNION ALL SELECT 1615, 'NAICS', 532412, 'Construction, Mining, and Forestry Machinery and Equipment Rental and Leasing', 5, 1613 FROM dual
UNION ALL SELECT 1616, 'NAICS', 532420, 'Office Machinery and Equipment Rental and Leasing', 5, 1617 FROM dual
UNION ALL SELECT 1618, 'NAICS', 532490, 'Other Commercial and Industrial Machinery and Equipment Rental and Leasing', 5, 1619 FROM dual
UNION ALL SELECT 1621, 'NAICS', 5331, 'Lessors of Nonfinancial Intangible Assets (except Copyrighted Works)', 3, 1620 FROM dual
UNION ALL SELECT 1623, 'NAICS', 53311, 'Lessors of Nonfinancial Intangible Assets (except Copyrighted Works)', 4, 1621 FROM dual
UNION ALL SELECT 1622, 'NAICS', 533110, 'Lessors of Nonfinancial Intangible Assets (except Copyrighted Works)', 5, 1623 FROM dual
UNION ALL SELECT 1625, 'NAICS', 541, 'Professional, Scientific, and Technical Services', 2, 1624 FROM dual
UNION ALL SELECT 1626, 'NAICS', 5411, 'Legal Services', 3, 1625 FROM dual
UNION ALL SELECT 1689, 'NAICS', 5418, 'Advertising, Public Relations, and Related Services', 3, 1625 FROM dual
UNION ALL SELECT 1666, 'NAICS', 5415, 'Computer Systems Design and Related Services', 3, 1625 FROM dual
UNION ALL SELECT 1634, 'NAICS', 5412, 'Accounting, Tax Preparation, Bookkeeping, and Payroll Services', 3, 1625 FROM dual
UNION ALL SELECT 1672, 'NAICS', 5416, 'Management, Scientific, and Technical Consulting Services', 3, 1625 FROM dual
UNION ALL SELECT 1657, 'NAICS', 5414, 'Specialized Design Services', 3, 1625 FROM dual
UNION ALL SELECT 1706, 'NAICS', 5419, 'Other Professional, Scientific, and Technical Services', 3, 1625 FROM dual
UNION ALL SELECT 1683, 'NAICS', 5417, 'Scientific Research and Development Services', 3, 1625 FROM dual
UNION ALL SELECT 1640, 'NAICS', 5413, 'Architectural, Engineering, and Related Services', 3, 1625 FROM dual
UNION ALL SELECT 1631, 'NAICS', 54119, 'Other Legal Services', 4, 1626 FROM dual
UNION ALL SELECT 1628, 'NAICS', 54111, 'Offices of Lawyers', 4, 1626 FROM dual
UNION ALL SELECT 1630, 'NAICS', 54112, 'Offices of Notaries', 4, 1626 FROM dual
UNION ALL SELECT 1627, 'NAICS', 541110, 'Offices of Lawyers', 5, 1628 FROM dual
UNION ALL SELECT 1629, 'NAICS', 541120, 'Offices of Notaries', 5, 1630 FROM dual
UNION ALL SELECT 1632, 'NAICS', 541191, 'Title Abstract and Settlement Offices', 5, 1631 FROM dual
UNION ALL SELECT 1633, 'NAICS', 541199, 'All Other Legal Services', 5, 1631 FROM dual
UNION ALL SELECT 1635, 'NAICS', 54121, 'Accounting, Tax Preparation, Bookkeeping, and Payroll Services', 4, 1634 FROM dual
UNION ALL SELECT 1639, 'NAICS', 541219, 'Other Accounting Services', 5, 1635 FROM dual
UNION ALL SELECT 1638, 'NAICS', 541214, 'Payroll Services', 5, 1635 FROM dual
UNION ALL SELECT 1637, 'NAICS', 541213, 'Tax Preparation Services', 5, 1635 FROM dual;

INSERT INTO "industry" (industry_id, industry_classification, industry_code, industry_description, depth, parent_id) 
          SELECT 1636, 'NAICS', 541211, 'Offices of Certified Public Accountants', 5, 1635 FROM dual
UNION ALL SELECT 1656, 'NAICS', 54138, 'Testing Laboratories', 4, 1640 FROM dual
UNION ALL SELECT 1642, 'NAICS', 54131, 'Architectural Services', 4, 1640 FROM dual
UNION ALL SELECT 1644, 'NAICS', 54132, 'Landscape Architectural Services', 4, 1640 FROM dual
UNION ALL SELECT 1646, 'NAICS', 54133, 'Engineering Services', 4, 1640 FROM dual
UNION ALL SELECT 1648, 'NAICS', 54134, 'Drafting Services', 4, 1640 FROM dual
UNION ALL SELECT 1650, 'NAICS', 54135, 'Building Inspection Services', 4, 1640 FROM dual
UNION ALL SELECT 1652, 'NAICS', 54136, 'Geophysical Surveying and Mapping Services', 4, 1640 FROM dual
UNION ALL SELECT 1654, 'NAICS', 54137, 'Surveying and Mapping (except Geophysical) Services', 4, 1640 FROM dual
UNION ALL SELECT 1641, 'NAICS', 541310, 'Architectural Services', 5, 1642 FROM dual
UNION ALL SELECT 1643, 'NAICS', 541320, 'Landscape Architectural Services', 5, 1644 FROM dual
UNION ALL SELECT 1645, 'NAICS', 541330, 'Engineering Services', 5, 1646 FROM dual
UNION ALL SELECT 1647, 'NAICS', 541340, 'Drafting Services', 5, 1648 FROM dual
UNION ALL SELECT 1649, 'NAICS', 541350, 'Building Inspection Services', 5, 1650 FROM dual
UNION ALL SELECT 1651, 'NAICS', 541360, 'Geophysical Surveying and Mapping Services', 5, 1652 FROM dual
UNION ALL SELECT 1653, 'NAICS', 541370, 'Surveying and Mapping (except Geophysical) Services', 5, 1654 FROM dual
UNION ALL SELECT 1655, 'NAICS', 541380, 'Testing Laboratories', 5, 1656 FROM dual
UNION ALL SELECT 1659, 'NAICS', 54141, 'Interior Design Services', 4, 1657 FROM dual
UNION ALL SELECT 1806, 'NAICS', 56291, 'Remediation Services', 4, 1804 FROM dual
UNION ALL SELECT 1665, 'NAICS', 54149, 'Other Specialized Design Services', 4, 1657 FROM dual
UNION ALL SELECT 1661, 'NAICS', 54142, 'Industrial Design Services', 4, 1657 FROM dual
UNION ALL SELECT 1663, 'NAICS', 54143, 'Graphic Design Services', 4, 1657 FROM dual
UNION ALL SELECT 1658, 'NAICS', 541410, 'Interior Design Services', 5, 1659 FROM dual
UNION ALL SELECT 1660, 'NAICS', 541420, 'Industrial Design Services', 5, 1661 FROM dual
UNION ALL SELECT 1662, 'NAICS', 541430, 'Graphic Design Services', 5, 1663 FROM dual
UNION ALL SELECT 1664, 'NAICS', 541490, 'Other Specialized Design Services', 5, 1665 FROM dual
UNION ALL SELECT 1667, 'NAICS', 54151, 'Computer Systems Design and Related Services', 4, 1666 FROM dual
UNION ALL SELECT 1670, 'NAICS', 541513, 'Computer Facilities Management Services', 5, 1667 FROM dual
UNION ALL SELECT 1668, 'NAICS', 541511, 'Custom Computer Programming Services', 5, 1667 FROM dual
UNION ALL SELECT 1669, 'NAICS', 541512, 'Computer Systems Design Services', 5, 1667 FROM dual
UNION ALL SELECT 1671, 'NAICS', 541519, 'Other Computer Related Services', 5, 1667 FROM dual
UNION ALL SELECT 1682, 'NAICS', 54169, 'Other Scientific and Technical Consulting Services', 4, 1672 FROM dual
UNION ALL SELECT 1673, 'NAICS', 54161, 'Management Consulting Services', 4, 1672 FROM dual
UNION ALL SELECT 1680, 'NAICS', 54162, 'Environmental Consulting Services', 4, 1672 FROM dual
UNION ALL SELECT 1678, 'NAICS', 541618, 'Other Management Consulting Services', 5, 1673 FROM dual
UNION ALL SELECT 1676, 'NAICS', 541613, 'Marketing Consulting Services', 5, 1673 FROM dual
UNION ALL SELECT 1677, 'NAICS', 541614, 'Process, Physical Distribution, and Logistics Consulting Services', 5, 1673 FROM dual
UNION ALL SELECT 1675, 'NAICS', 541612, 'Human Resources Consulting Services', 5, 1673 FROM dual
UNION ALL SELECT 1674, 'NAICS', 541611, 'Administrative Management and General Management Consulting Services', 5, 1673 FROM dual
UNION ALL SELECT 1679, 'NAICS', 541620, 'Environmental Consulting Services', 5, 1680 FROM dual
UNION ALL SELECT 1681, 'NAICS', 541690, 'Other Scientific and Technical Consulting Services', 5, 1682 FROM dual
UNION ALL SELECT 1688, 'NAICS', 54172, 'Research and Development in the Social Sciences and Humanities', 4, 1683 FROM dual
UNION ALL SELECT 1684, 'NAICS', 54171, 'Research and Development in the Physical, Engineering, and Life Sciences', 4, 1683 FROM dual
UNION ALL SELECT 1685, 'NAICS', 541711, 'Research and Development in Biotechnology', 5, 1684 FROM dual
UNION ALL SELECT 1686, 'NAICS', 541712, 'Research and Development in the Physical, Engineering, and Life Sciences (except Biotechnology)', 5, 1684 FROM dual
UNION ALL SELECT 1687, 'NAICS', 541720, 'Research and Development in the Social Sciences and Humanities', 5, 1688 FROM dual
UNION ALL SELECT 1703, 'NAICS', 54187, 'Advertising Material Distribution Services', 4, 1689 FROM dual
UNION ALL SELECT 1705, 'NAICS', 54189, 'Other Services Related to Advertising', 4, 1689 FROM dual
UNION ALL SELECT 1691, 'NAICS', 54181, 'Advertising Agencies', 4, 1689 FROM dual
UNION ALL SELECT 1693, 'NAICS', 54182, 'Public Relations Agencies', 4, 1689 FROM dual
UNION ALL SELECT 1695, 'NAICS', 54183, 'Media Buying Agencies', 4, 1689 FROM dual
UNION ALL SELECT 1697, 'NAICS', 54184, 'Media Representatives', 4, 1689 FROM dual
UNION ALL SELECT 1699, 'NAICS', 54185, 'Outdoor Advertising', 4, 1689 FROM dual
UNION ALL SELECT 1701, 'NAICS', 54186, 'Direct Mail Advertising', 4, 1689 FROM dual
UNION ALL SELECT 1690, 'NAICS', 541810, 'Advertising Agencies', 5, 1691 FROM dual
UNION ALL SELECT 1692, 'NAICS', 541820, 'Public Relations Agencies', 5, 1693 FROM dual
UNION ALL SELECT 1694, 'NAICS', 541830, 'Media Buying Agencies', 5, 1695 FROM dual
UNION ALL SELECT 1696, 'NAICS', 541840, 'Media Representatives', 5, 1697 FROM dual
UNION ALL SELECT 1698, 'NAICS', 541850, 'Outdoor Advertising', 5, 1699 FROM dual
UNION ALL SELECT 1700, 'NAICS', 541860, 'Direct Mail Advertising', 5, 1701 FROM dual
UNION ALL SELECT 1702, 'NAICS', 541870, 'Advertising Material Distribution Services', 5, 1703 FROM dual
UNION ALL SELECT 1704, 'NAICS', 541890, 'Other Services Related to Advertising', 5, 1705 FROM dual
UNION ALL SELECT 1717, 'NAICS', 54199, 'All Other Professional, Scientific, and Technical Services', 4, 1706 FROM dual
UNION ALL SELECT 1708, 'NAICS', 54191, 'Marketing Research and Public Opinion Polling', 4, 1706 FROM dual
UNION ALL SELECT 1709, 'NAICS', 54192, 'Photographic Services', 4, 1706 FROM dual
UNION ALL SELECT 1713, 'NAICS', 54193, 'Translation and Interpretation Services', 4, 1706 FROM dual
UNION ALL SELECT 1715, 'NAICS', 54194, 'Veterinary Services', 4, 1706 FROM dual
UNION ALL SELECT 1707, 'NAICS', 541910, 'Marketing Research and Public Opinion Polling', 5, 1708 FROM dual
UNION ALL SELECT 1710, 'NAICS', 541921, 'Photography Studios, Portrait', 5, 1709 FROM dual
UNION ALL SELECT 1711, 'NAICS', 541922, 'Commercial Photography', 5, 1709 FROM dual
UNION ALL SELECT 1712, 'NAICS', 541930, 'Translation and Interpretation Services', 5, 1713 FROM dual
UNION ALL SELECT 1714, 'NAICS', 541940, 'Veterinary Services', 5, 1715 FROM dual
UNION ALL SELECT 1716, 'NAICS', 541990, 'All Other Professional, Scientific, and Technical Services', 5, 1717 FROM dual
UNION ALL SELECT 1719, 'NAICS', 551, 'Management of Companies and Enterprises', 2, 1718 FROM dual
UNION ALL SELECT 1720, 'NAICS', 5511, 'Management of Companies and Enterprises', 3, 1719 FROM dual
UNION ALL SELECT 1721, 'NAICS', 55111, 'Management of Companies and Enterprises', 4, 1720 FROM dual
UNION ALL SELECT 1722, 'NAICS', 551111, 'Offices of Bank Holding Companies', 5, 1721 FROM dual
UNION ALL SELECT 1723, 'NAICS', 551112, 'Offices of Other Holding Companies', 5, 1721 FROM dual
UNION ALL SELECT 1724, 'NAICS', 551114, 'Corporate, Subsidiary, and Regional Managing Offices', 5, 1721 FROM dual
UNION ALL SELECT 1726, 'NAICS', 561, 'Administrative and Support Services', 2, 1725 FROM dual
UNION ALL SELECT 1792, 'NAICS', 562, 'Waste Management and Remediation Services', 2, 1725 FROM dual
UNION ALL SELECT 1766, 'NAICS', 5616, 'Investigation and Security Services', 3, 1726 FROM dual
UNION ALL SELECT 1785, 'NAICS', 5619, 'Other Support Services', 3, 1726 FROM dual
UNION ALL SELECT 1727, 'NAICS', 5611, 'Office Administrative Services', 3, 1726 FROM dual
UNION ALL SELECT 1758, 'NAICS', 5615, 'Travel Arrangement and Reservation Services', 3, 1726 FROM dual
UNION ALL SELECT 1774, 'NAICS', 5617, 'Services to Buildings and Dwellings', 3, 1726 FROM dual
UNION ALL SELECT 1730, 'NAICS', 5612, 'Facilities Support Services', 3, 1726 FROM dual
UNION ALL SELECT 1733, 'NAICS', 5613, 'Employment Services', 3, 1726 FROM dual
UNION ALL SELECT 1741, 'NAICS', 5614, 'Business Support Services', 3, 1726 FROM dual
UNION ALL SELECT 1729, 'NAICS', 56111, 'Office Administrative Services', 4, 1727 FROM dual
UNION ALL SELECT 1728, 'NAICS', 561110, 'Office Administrative Services', 5, 1729 FROM dual
UNION ALL SELECT 1732, 'NAICS', 56121, 'Facilities Support Services', 4, 1730 FROM dual
UNION ALL SELECT 1731, 'NAICS', 561210, 'Facilities Support Services', 5, 1732 FROM dual
UNION ALL SELECT 1734, 'NAICS', 56131, 'Employment Placement Agencies and Executive Search Services', 4, 1733 FROM dual
UNION ALL SELECT 1740, 'NAICS', 56133, 'Professional Employer Organizations', 4, 1733 FROM dual
UNION ALL SELECT 1738, 'NAICS', 56132, 'Temporary Help Services', 4, 1733 FROM dual
UNION ALL SELECT 1736, 'NAICS', 561312, 'Executive Search Services', 5, 1734 FROM dual
UNION ALL SELECT 1735, 'NAICS', 561311, 'Employment Placement Agencies', 5, 1734 FROM dual
UNION ALL SELECT 1737, 'NAICS', 561320, 'Temporary Help Services', 5, 1738 FROM dual
UNION ALL SELECT 1739, 'NAICS', 561330, 'Professional Employer Organizations', 5, 1740 FROM dual
UNION ALL SELECT 1754, 'NAICS', 56149, 'Other Business Support Services', 4, 1741 FROM dual
UNION ALL SELECT 1743, 'NAICS', 56141, 'Document Preparation Services', 4, 1741 FROM dual
UNION ALL SELECT 1744, 'NAICS', 56142, 'Telephone Call Centers', 4, 1741 FROM dual
UNION ALL SELECT 1747, 'NAICS', 56143, 'Business Service Centers', 4, 1741 FROM dual
UNION ALL SELECT 1751, 'NAICS', 56144, 'Collection Agencies', 4, 1741 FROM dual
UNION ALL SELECT 1753, 'NAICS', 56145, 'Credit Bureaus', 4, 1741 FROM dual
UNION ALL SELECT 1742, 'NAICS', 561410, 'Document Preparation Services', 5, 1743 FROM dual
UNION ALL SELECT 1745, 'NAICS', 561421, 'Telephone Answering Services', 5, 1744 FROM dual
UNION ALL SELECT 1746, 'NAICS', 561422, 'Telemarketing Bureaus and Other Contact Centers', 5, 1744 FROM dual
UNION ALL SELECT 1748, 'NAICS', 561431, 'Private Mail Centers', 5, 1747 FROM dual
UNION ALL SELECT 1749, 'NAICS', 561439, 'Other Business Service Centers (including Copy Shops)', 5, 1747 FROM dual
UNION ALL SELECT 1750, 'NAICS', 561440, 'Collection Agencies', 5, 1751 FROM dual
UNION ALL SELECT 1752, 'NAICS', 561450, 'Credit Bureaus', 5, 1753 FROM dual
UNION ALL SELECT 1757, 'NAICS', 561499, 'All Other Business Support Services', 5, 1754 FROM dual
UNION ALL SELECT 1755, 'NAICS', 561491, 'Repossession Services', 5, 1754 FROM dual
UNION ALL SELECT 1756, 'NAICS', 561492, 'Court Reporting and Stenotype Services', 5, 1754 FROM dual
UNION ALL SELECT 1762, 'NAICS', 56152, 'Tour Operators', 4, 1758 FROM dual
UNION ALL SELECT 1760, 'NAICS', 56151, 'Travel Agencies', 4, 1758 FROM dual
UNION ALL SELECT 1763, 'NAICS', 56159, 'Other Travel Arrangement and Reservation Services', 4, 1758 FROM dual
UNION ALL SELECT 1759, 'NAICS', 561510, 'Travel Agencies', 5, 1760 FROM dual
UNION ALL SELECT 1761, 'NAICS', 561520, 'Tour Operators', 5, 1762 FROM dual
UNION ALL SELECT 1765, 'NAICS', 561599, 'All Other Travel Arrangement and Reservation Services', 5, 1763 FROM dual
UNION ALL SELECT 1764, 'NAICS', 561591, 'Convention and Visitors Bureaus', 5, 1763 FROM dual
UNION ALL SELECT 1767, 'NAICS', 56161, 'Investigation, Guard, and Armored Car Services', 4, 1766 FROM dual
UNION ALL SELECT 1771, 'NAICS', 56162, 'Security Systems Services', 4, 1766 FROM dual
UNION ALL SELECT 1770, 'NAICS', 561613, 'Armored Car Services', 5, 1767 FROM dual
UNION ALL SELECT 1768, 'NAICS', 561611, 'Investigation Services', 5, 1767 FROM dual
UNION ALL SELECT 1769, 'NAICS', 561612, 'Security Guards and Patrol Services', 5, 1767 FROM dual
UNION ALL SELECT 1772, 'NAICS', 561621, 'Security Systems Services (except Locksmiths)', 5, 1771 FROM dual
UNION ALL SELECT 1784, 'NAICS', 56179, 'Other Services to Buildings and Dwellings', 4, 1774 FROM dual
UNION ALL SELECT 1776, 'NAICS', 56171, 'Exterminating and Pest Control Services', 4, 1774 FROM dual
UNION ALL SELECT 1778, 'NAICS', 56172, 'Janitorial Services', 4, 1774 FROM dual
UNION ALL SELECT 1780, 'NAICS', 56173, 'Landscaping Services', 4, 1774 FROM dual
UNION ALL SELECT 1782, 'NAICS', 56174, 'Carpet and Upholstery Cleaning Services', 4, 1774 FROM dual
UNION ALL SELECT 1775, 'NAICS', 561710, 'Exterminating and Pest Control Services', 5, 1776 FROM dual
UNION ALL SELECT 1777, 'NAICS', 561720, 'Janitorial Services', 5, 1778 FROM dual
UNION ALL SELECT 1779, 'NAICS', 561730, 'Landscaping Services', 5, 1780 FROM dual
UNION ALL SELECT 1781, 'NAICS', 561740, 'Carpet and Upholstery Cleaning Services', 5, 1782 FROM dual
UNION ALL SELECT 1783, 'NAICS', 561790, 'Other Services to Buildings and Dwellings', 5, 1784 FROM dual
UNION ALL SELECT 1789, 'NAICS', 56192, 'Convention and Trade Show Organizers', 4, 1785 FROM dual
UNION ALL SELECT 1787, 'NAICS', 56191, 'Packaging and Labeling Services', 4, 1785 FROM dual
UNION ALL SELECT 1791, 'NAICS', 56199, 'All Other Support Services', 4, 1785 FROM dual
UNION ALL SELECT 1786, 'NAICS', 561910, 'Packaging and Labeling Services', 5, 1787 FROM dual
UNION ALL SELECT 1788, 'NAICS', 561920, 'Convention and Trade Show Organizers', 5, 1789 FROM dual
UNION ALL SELECT 1790, 'NAICS', 561990, 'All Other Support Services', 5, 1791 FROM dual
UNION ALL SELECT 1804, 'NAICS', 5629, 'Remediation and Other Waste Management Services', 3, 1792 FROM dual
UNION ALL SELECT 1793, 'NAICS', 5621, 'Waste Collection', 3, 1792 FROM dual
UNION ALL SELECT 1798, 'NAICS', 5622, 'Waste Treatment and Disposal', 3, 1792 FROM dual
UNION ALL SELECT 1794, 'NAICS', 56211, 'Waste Collection', 4, 1793 FROM dual
UNION ALL SELECT 1796, 'NAICS', 562112, 'Hazardous Waste Collection', 5, 1794 FROM dual
UNION ALL SELECT 1797, 'NAICS', 562119, 'Other Waste Collection', 5, 1794 FROM dual
UNION ALL SELECT 1795, 'NAICS', 562111, 'Solid Waste Collection', 5, 1794 FROM dual
UNION ALL SELECT 1799, 'NAICS', 56221, 'Waste Treatment and Disposal', 4, 1798 FROM dual
UNION ALL SELECT 1802, 'NAICS', 562213, 'Solid Waste Combustors and Incinerators', 5, 1799 FROM dual
UNION ALL SELECT 1803, 'NAICS', 562219, 'Other Nonhazardous Waste Treatment and Disposal', 5, 1799 FROM dual
UNION ALL SELECT 1801, 'NAICS', 562212, 'Solid Waste Landfill', 5, 1799 FROM dual
UNION ALL SELECT 1800, 'NAICS', 562211, 'Hazardous Waste Treatment and Disposal', 5, 1799 FROM dual
UNION ALL SELECT 1808, 'NAICS', 56292, 'Materials Recovery Facilities', 4, 1804 FROM dual
UNION ALL SELECT 1809, 'NAICS', 56299, 'All Other Waste Management Services', 4, 1804 FROM dual
UNION ALL SELECT 1805, 'NAICS', 562910, 'Remediation Services', 5, 1806 FROM dual
UNION ALL SELECT 1807, 'NAICS', 562920, 'Materials Recovery Facilities', 5, 1808 FROM dual
UNION ALL SELECT 1811, 'NAICS', 562998, 'All Other Miscellaneous Waste Management Services', 5, 1809 FROM dual
UNION ALL SELECT 1810, 'NAICS', 562991, 'Septic Tank and Related Services', 5, 1809 FROM dual
UNION ALL SELECT 1813, 'NAICS', 611, 'Educational Services', 2, 1812 FROM dual
UNION ALL SELECT 1814, 'NAICS', 6111, 'Elementary and Secondary Schools', 3, 1813 FROM dual
UNION ALL SELECT 1817, 'NAICS', 6112, 'Junior Colleges', 3, 1813 FROM dual
UNION ALL SELECT 1820, 'NAICS', 6113, 'Colleges, Universities, and Professional Schools', 3, 1813 FROM dual
UNION ALL SELECT 1823, 'NAICS', 6114, 'Business Schools and Computer and Management Training', 3, 1813 FROM dual
UNION ALL SELECT 1830, 'NAICS', 6115, 'Technical and Trade Schools', 3, 1813 FROM dual
UNION ALL SELECT 1836, 'NAICS', 6116, 'Other Schools and Instruction', 3, 1813 FROM dual
UNION ALL SELECT 1847, 'NAICS', 6117, 'Educational Support Services', 3, 1813 FROM dual
UNION ALL SELECT 1816, 'NAICS', 61111, 'Elementary and Secondary Schools', 4, 1814 FROM dual
UNION ALL SELECT 1815, 'NAICS', 611110, 'Elementary and Secondary Schools', 5, 1816 FROM dual
UNION ALL SELECT 1819, 'NAICS', 61121, 'Junior Colleges', 4, 1817 FROM dual
UNION ALL SELECT 1818, 'NAICS', 611210, 'Junior Colleges', 5, 1819 FROM dual
UNION ALL SELECT 1822, 'NAICS', 61131, 'Colleges, Universities, and Professional Schools', 4, 1820 FROM dual
UNION ALL SELECT 1821, 'NAICS', 611310, 'Colleges, Universities, and Professional Schools', 5, 1822 FROM dual
UNION ALL SELECT 1827, 'NAICS', 61142, 'Computer Training', 4, 1823 FROM dual
UNION ALL SELECT 1829, 'NAICS', 61143, 'Professional and Management Development Training', 4, 1823 FROM dual
UNION ALL SELECT 1825, 'NAICS', 61141, 'Business and Secretarial Schools', 4, 1823 FROM dual
UNION ALL SELECT 1824, 'NAICS', 611410, 'Business and Secretarial Schools', 5, 1825 FROM dual
UNION ALL SELECT 1826, 'NAICS', 611420, 'Computer Training', 5, 1827 FROM dual
UNION ALL SELECT 1828, 'NAICS', 611430, 'Professional and Management Development Training', 5, 1829 FROM dual
UNION ALL SELECT 1831, 'NAICS', 61151, 'Technical and Trade Schools', 4, 1830 FROM dual
UNION ALL SELECT 1835, 'NAICS', 611519, 'Other Technical and Trade Schools', 5, 1831 FROM dual
UNION ALL SELECT 1834, 'NAICS', 611513, 'Apprenticeship Training', 5, 1831 FROM dual
UNION ALL SELECT 1833, 'NAICS', 611512, 'Flight Training', 5, 1831 FROM dual
UNION ALL SELECT 1832, 'NAICS', 611511, 'Cosmetology and Barber Schools', 5, 1831 FROM dual
UNION ALL SELECT 1838, 'NAICS', 61161, 'Fine Arts Schools', 4, 1836 FROM dual
UNION ALL SELECT 1843, 'NAICS', 61169, 'All Other Schools and Instruction', 4, 1836 FROM dual
UNION ALL SELECT 1842, 'NAICS', 61163, 'Language Schools', 4, 1836 FROM dual
UNION ALL SELECT 1840, 'NAICS', 61162, 'Sports and Recreation Instruction', 4, 1836 FROM dual
UNION ALL SELECT 1837, 'NAICS', 611610, 'Fine Arts Schools', 5, 1838 FROM dual
UNION ALL SELECT 1839, 'NAICS', 611620, 'Sports and Recreation Instruction', 5, 1840 FROM dual
UNION ALL SELECT 1841, 'NAICS', 611630, 'Language Schools', 5, 1842 FROM dual
UNION ALL SELECT 1844, 'NAICS', 611691, 'Exam Preparation and Tutoring', 5, 1843 FROM dual
UNION ALL SELECT 1845, 'NAICS', 611692, 'Automobile Driving Schools', 5, 1843 FROM dual
UNION ALL SELECT 1846, 'NAICS', 611699, 'All Other Miscellaneous Schools and Instruction', 5, 1843 FROM dual
UNION ALL SELECT 1849, 'NAICS', 61171, 'Educational Support Services', 4, 1847 FROM dual
UNION ALL SELECT 1848, 'NAICS', 611710, 'Educational Support Services', 5, 1849 FROM dual
UNION ALL SELECT 1894, 'NAICS', 622, 'Hospitals', 2, 1850 FROM dual
UNION ALL SELECT 1851, 'NAICS', 621, 'Ambulatory Health Care Services', 2, 1850 FROM dual
UNION ALL SELECT 1904, 'NAICS', 623, 'Nursing and Residential Care Facilities', 2, 1850 FROM dual
UNION ALL SELECT 1859, 'NAICS', 6213, 'Offices of Other Health Practitioners', 3, 1851 FROM dual
UNION ALL SELECT 1881, 'NAICS', 6215, 'Medical and Diagnostic Laboratories', 3, 1851 FROM dual
UNION ALL SELECT 1856, 'NAICS', 6212, 'Offices of Dentists', 3, 1851 FROM dual
UNION ALL SELECT 1871, 'NAICS', 6214, 'Outpatient Care Centers', 3, 1851 FROM dual
UNION ALL SELECT 1852, 'NAICS', 6211, 'Offices of Physicians', 3, 1851 FROM dual
UNION ALL SELECT 1885, 'NAICS', 6216, 'Home Health Care Services', 3, 1851 FROM dual
UNION ALL SELECT 1888, 'NAICS', 6219, 'Other Ambulatory Health Care Services', 3, 1851 FROM dual
UNION ALL SELECT 1853, 'NAICS', 62111, 'Offices of Physicians', 4, 1852 FROM dual
UNION ALL SELECT 1855, 'NAICS', 621112, 'Offices of Physicians, Mental Health Specialists', 5, 1853 FROM dual
UNION ALL SELECT 1854, 'NAICS', 621111, 'Offices of Physicians (except Mental Health Specialists)', 5, 1853 FROM dual
UNION ALL SELECT 1858, 'NAICS', 62121, 'Offices of Dentists', 4, 1856 FROM dual
UNION ALL SELECT 1857, 'NAICS', 621210, 'Offices of Dentists', 5, 1858 FROM dual
UNION ALL SELECT 1867, 'NAICS', 62134, 'Offices of Physical, Occupational and Speech Therapists, and Audiologists', 4, 1859 FROM dual
UNION ALL SELECT 1861, 'NAICS', 62131, 'Offices of Chiropractors', 4, 1859 FROM dual
UNION ALL SELECT 1863, 'NAICS', 62132, 'Offices of Optometrists', 4, 1859 FROM dual
UNION ALL SELECT 1865, 'NAICS', 62133, 'Offices of Mental Health Practitioners (except Physicians)', 4, 1859 FROM dual
UNION ALL SELECT 1868, 'NAICS', 62139, 'Offices of All Other Health Practitioners', 4, 1859 FROM dual
UNION ALL SELECT 1860, 'NAICS', 621310, 'Offices of Chiropractors', 5, 1861 FROM dual
UNION ALL SELECT 1862, 'NAICS', 621320, 'Offices of Optometrists', 5, 1863 FROM dual
UNION ALL SELECT 1864, 'NAICS', 621330, 'Offices of Mental Health Practitioners (except Physicians)', 5, 1865 FROM dual
UNION ALL SELECT 1866, 'NAICS', 621340, 'Offices of Physical, Occupational and Speech Therapists, and Audiologists', 5, 1867 FROM dual
UNION ALL SELECT 1870, 'NAICS', 621399, 'Offices of All Other Miscellaneous Health Practitioners', 5, 1868 FROM dual
UNION ALL SELECT 1869, 'NAICS', 621391, 'Offices of Podiatrists', 5, 1868 FROM dual
UNION ALL SELECT 1875, 'NAICS', 62142, 'Outpatient Mental Health and Substance Abuse Centers', 4, 1871 FROM dual
UNION ALL SELECT 1876, 'NAICS', 62149, 'Other Outpatient Care Centers', 4, 1871 FROM dual
UNION ALL SELECT 1873, 'NAICS', 62141, 'Family Planning Centers', 4, 1871 FROM dual
UNION ALL SELECT 1872, 'NAICS', 621410, 'Family Planning Centers', 5, 1873 FROM dual
UNION ALL SELECT 1874, 'NAICS', 621420, 'Outpatient Mental Health and Substance Abuse Centers', 5, 1875 FROM dual
UNION ALL SELECT 1878, 'NAICS', 621492, 'Kidney Dialysis Centers', 5, 1876 FROM dual
UNION ALL SELECT 1880, 'NAICS', 621498, 'All Other Outpatient Care Centers', 5, 1876 FROM dual
UNION ALL SELECT 1877, 'NAICS', 621491, 'HMO Medical Centers', 5, 1876 FROM dual
UNION ALL SELECT 1879, 'NAICS', 621493, 'Freestanding Ambulatory Surgical and Emergency Centers', 5, 1876 FROM dual
UNION ALL SELECT 1882, 'NAICS', 62151, 'Medical and Diagnostic Laboratories', 4, 1881 FROM dual
UNION ALL SELECT 1884, 'NAICS', 621512, 'Diagnostic Imaging Centers', 5, 1882 FROM dual
UNION ALL SELECT 1883, 'NAICS', 621511, 'Medical Laboratories', 5, 1882 FROM dual
UNION ALL SELECT 1887, 'NAICS', 62161, 'Home Health Care Services', 4, 1885 FROM dual
UNION ALL SELECT 1886, 'NAICS', 621610, 'Home Health Care Services', 5, 1887 FROM dual
UNION ALL SELECT 1890, 'NAICS', 62191, 'Ambulance Services', 4, 1888 FROM dual
UNION ALL SELECT 1891, 'NAICS', 62199, 'All Other Ambulatory Health Care Services', 4, 1888 FROM dual
UNION ALL SELECT 1889, 'NAICS', 621910, 'Ambulance Services', 5, 1890 FROM dual
UNION ALL SELECT 1892, 'NAICS', 621991, 'Blood and Organ Banks', 5, 1891 FROM dual
UNION ALL SELECT 1893, 'NAICS', 621999, 'All Other Miscellaneous Ambulatory Health Care Services', 5, 1891 FROM dual
UNION ALL SELECT 1901, 'NAICS', 6223, 'Specialty (except Psychiatric and Substance Abuse) Hospitals', 3, 1894 FROM dual
UNION ALL SELECT 1895, 'NAICS', 6221, 'General Medical and Surgical Hospitals', 3, 1894 FROM dual
UNION ALL SELECT 1898, 'NAICS', 6222, 'Psychiatric and Substance Abuse Hospitals', 3, 1894 FROM dual
UNION ALL SELECT 1897, 'NAICS', 62211, 'General Medical and Surgical Hospitals', 4, 1895 FROM dual
UNION ALL SELECT 1896, 'NAICS', 622110, 'General Medical and Surgical Hospitals', 5, 1897 FROM dual
UNION ALL SELECT 1900, 'NAICS', 62221, 'Psychiatric and Substance Abuse Hospitals', 4, 1898 FROM dual
UNION ALL SELECT 1899, 'NAICS', 622210, 'Psychiatric and Substance Abuse Hospitals', 5, 1900 FROM dual
UNION ALL SELECT 1903, 'NAICS', 62231, 'Specialty (except Psychiatric and Substance Abuse) Hospitals', 4, 1901 FROM dual
UNION ALL SELECT 1902, 'NAICS', 622310, 'Specialty (except Psychiatric and Substance Abuse) Hospitals', 5, 1903 FROM dual
UNION ALL SELECT 1913, 'NAICS', 6233, 'Continuing Care Retirement Communities and Assisted Living Facilities for the Elderly', 3, 1904 FROM dual
UNION ALL SELECT 1908, 'NAICS', 6232, 'Residential Intellectual and Developmental Disability, Mental Health, and Substance Abuse Facilities', 3, 1904 FROM dual
UNION ALL SELECT 1905, 'NAICS', 6231, 'Nursing Care Facilities (Skilled Nursing Facilities)', 3, 1904 FROM dual
UNION ALL SELECT 1917, 'NAICS', 6239, 'Other Residential Care Facilities', 3, 1904 FROM dual
UNION ALL SELECT 1907, 'NAICS', 62311, 'Nursing Care Facilities (Skilled Nursing Facilities)', 4, 1905 FROM dual
UNION ALL SELECT 1906, 'NAICS', 623110, 'Nursing Care Facilities (Skilled Nursing Facilities)', 5, 1907 FROM dual
UNION ALL SELECT 1912, 'NAICS', 62322, 'Residential Mental Health and Substance Abuse Facilities', 4, 1908 FROM dual
UNION ALL SELECT 1910, 'NAICS', 62321, 'Residential Intellectual and Developmental Disability Facilities', 4, 1908 FROM dual
UNION ALL SELECT 1909, 'NAICS', 623210, 'Residential Intellectual and Developmental Disability Facilities', 5, 1910 FROM dual
UNION ALL SELECT 1911, 'NAICS', 623220, 'Residential Mental Health and Substance Abuse Facilities', 5, 1912 FROM dual
UNION ALL SELECT 1914, 'NAICS', 62331, 'Continuing Care Retirement Communities and Assisted Living Facilities for the Elderly', 4, 1913 FROM dual
UNION ALL SELECT 1915, 'NAICS', 623311, 'Continuing Care Retirement Communities', 5, 1914 FROM dual
UNION ALL SELECT 1916, 'NAICS', 623312, 'Assisted Living Facilities for the Elderly', 5, 1914 FROM dual
UNION ALL SELECT 1919, 'NAICS', 62399, 'Other Residential Care Facilities', 4, 1917 FROM dual
UNION ALL SELECT 1918, 'NAICS', 623990, 'Other Residential Care Facilities', 5, 1919 FROM dual
UNION ALL SELECT 1928, 'NAICS', 6242, 'Community Food and Housing, and Emergency and Other Relief Services', 3, 1920 FROM dual
UNION ALL SELECT 1921, 'NAICS', 6241, 'Individual and Family Services', 3, 1920 FROM dual
UNION ALL SELECT 1936, 'NAICS', 6243, 'Vocational Rehabilitation Services', 3, 1920 FROM dual
UNION ALL SELECT 1939, 'NAICS', 6244, 'Child Day Care Services', 3, 1920 FROM dual
UNION ALL SELECT 1927, 'NAICS', 62419, 'Other Individual and Family Services', 4, 1921 FROM dual
UNION ALL SELECT 1923, 'NAICS', 62411, 'Child and Youth Services', 4, 1921 FROM dual
UNION ALL SELECT 2004, 'NAICS', 721, 'Accommodation', 2, 2003 FROM dual
UNION ALL SELECT 1925, 'NAICS', 62412, 'Services for the Elderly and Persons with Disabilities', 4, 1921 FROM dual
UNION ALL SELECT 1922, 'NAICS', 624110, 'Child and Youth Services', 5, 1923 FROM dual
UNION ALL SELECT 1924, 'NAICS', 624120, 'Services for the Elderly and Persons with Disabilities', 5, 1925 FROM dual
UNION ALL SELECT 1926, 'NAICS', 624190, 'Other Individual and Family Services', 5, 1927 FROM dual
UNION ALL SELECT 1931, 'NAICS', 62422, 'Community Housing Services', 4, 1928 FROM dual
UNION ALL SELECT 1935, 'NAICS', 62423, 'Emergency and Other Relief Services', 4, 1928 FROM dual
UNION ALL SELECT 1930, 'NAICS', 62421, 'Community Food Services', 4, 1928 FROM dual
UNION ALL SELECT 1929, 'NAICS', 624210, 'Community Food Services', 5, 1930 FROM dual
UNION ALL SELECT 1932, 'NAICS', 624221, 'Temporary Shelters', 5, 1931 FROM dual
UNION ALL SELECT 1933, 'NAICS', 624229, 'Other Community Housing Services', 5, 1931 FROM dual
UNION ALL SELECT 1934, 'NAICS', 624230, 'Emergency and Other Relief Services', 5, 1935 FROM dual
UNION ALL SELECT 1938, 'NAICS', 62431, 'Vocational Rehabilitation Services', 4, 1936 FROM dual
UNION ALL SELECT 1937, 'NAICS', 624310, 'Vocational Rehabilitation Services', 5, 1938 FROM dual
UNION ALL SELECT 1941, 'NAICS', 62441, 'Child Day Care Services', 4, 1939 FROM dual
UNION ALL SELECT 1940, 'NAICS', 624410, 'Child Day Care Services', 5, 1941 FROM dual
UNION ALL SELECT 1943, 'NAICS', 711, 'Performing Arts, Spectator Sports, and Related Industries', 2, 1942 FROM dual
UNION ALL SELECT 1969, 'NAICS', 712, 'Museums, Historical Sites, and Similar Institutions', 2, 1942 FROM dual
UNION ALL SELECT 1979, 'NAICS', 713, 'Amusement, Gambling, and Recreation Industries', 2, 1942 FROM dual
UNION ALL SELECT 1958, 'NAICS', 7113, 'Promoters of Performing Arts, Sports, and Similar Events', 3, 1943 FROM dual
UNION ALL SELECT 1953, 'NAICS', 7112, 'Spectator Sports', 3, 1943 FROM dual
UNION ALL SELECT 1966, 'NAICS', 7115, 'Independent Artists, Writers, and Performers', 3, 1943 FROM dual
UNION ALL SELECT 1944, 'NAICS', 7111, 'Performing Arts Companies', 3, 1943 FROM dual
UNION ALL SELECT 1963, 'NAICS', 7114, 'Agents and Managers for Artists, Athletes, Entertainers, and Other Public Figures', 3, 1943 FROM dual
UNION ALL SELECT 1948, 'NAICS', 71112, 'Dance Companies', 4, 1944 FROM dual
UNION ALL SELECT 1950, 'NAICS', 71113, 'Musical Groups and Artists', 4, 1944 FROM dual
UNION ALL SELECT 1952, 'NAICS', 71119, 'Other Performing Arts Companies', 4, 1944 FROM dual
UNION ALL SELECT 1946, 'NAICS', 71111, 'Theater Companies and Dinner Theaters', 4, 1944 FROM dual
UNION ALL SELECT 1945, 'NAICS', 711110, 'Theater Companies and Dinner Theaters', 5, 1946 FROM dual
UNION ALL SELECT 1947, 'NAICS', 711120, 'Dance Companies', 5, 1948 FROM dual
UNION ALL SELECT 1949, 'NAICS', 711130, 'Musical Groups and Artists', 5, 1950 FROM dual
UNION ALL SELECT 1951, 'NAICS', 711190, 'Other Performing Arts Companies', 5, 1952 FROM dual
UNION ALL SELECT 1954, 'NAICS', 71121, 'Spectator Sports', 4, 1953 FROM dual
UNION ALL SELECT 1955, 'NAICS', 711211, 'Sports Teams and Clubs', 5, 1954 FROM dual
UNION ALL SELECT 1957, 'NAICS', 711219, 'Other Spectator Sports', 5, 1954 FROM dual
UNION ALL SELECT 1956, 'NAICS', 711212, 'Racetracks', 5, 1954 FROM dual
UNION ALL SELECT 1962, 'NAICS', 71132, 'Promoters of Performing Arts, Sports, and Similar Events without Facilities', 4, 1958 FROM dual
UNION ALL SELECT 1960, 'NAICS', 71131, 'Promoters of Performing Arts, Sports, and Similar Events with Facilities', 4, 1958 FROM dual
UNION ALL SELECT 1959, 'NAICS', 711310, 'Promoters of Performing Arts, Sports, and Similar Events with Facilities', 5, 1960 FROM dual
UNION ALL SELECT 1961, 'NAICS', 711320, 'Promoters of Performing Arts, Sports, and Similar Events without Facilities', 5, 1962 FROM dual
UNION ALL SELECT 1965, 'NAICS', 71141, 'Agents and Managers for Artists, Athletes, Entertainers, and Other Public Figures', 4, 1963 FROM dual
UNION ALL SELECT 1964, 'NAICS', 711410, 'Agents and Managers for Artists, Athletes, Entertainers, and Other Public Figures', 5, 1965 FROM dual
UNION ALL SELECT 1968, 'NAICS', 71151, 'Independent Artists, Writers, and Performers', 4, 1966 FROM dual
UNION ALL SELECT 1967, 'NAICS', 711510, 'Independent Artists, Writers, and Performers', 5, 1968 FROM dual
UNION ALL SELECT 1970, 'NAICS', 7121, 'Museums, Historical Sites, and Similar Institutions', 3, 1969 FROM dual
UNION ALL SELECT 1974, 'NAICS', 71212, 'Historical Sites', 4, 1970 FROM dual
UNION ALL SELECT 1976, 'NAICS', 71213, 'Zoos and Botanical Gardens', 4, 1970 FROM dual
UNION ALL SELECT 1978, 'NAICS', 71219, 'Nature Parks and Other Similar Institutions', 4, 1970 FROM dual
UNION ALL SELECT 1972, 'NAICS', 71211, 'Museums', 4, 1970 FROM dual
UNION ALL SELECT 1971, 'NAICS', 712110, 'Museums', 5, 1972 FROM dual
UNION ALL SELECT 1973, 'NAICS', 712120, 'Historical Sites', 5, 1974 FROM dual
UNION ALL SELECT 1975, 'NAICS', 712130, 'Zoos and Botanical Gardens', 5, 1976 FROM dual
UNION ALL SELECT 1977, 'NAICS', 712190, 'Nature Parks and Other Similar Institutions', 5, 1978 FROM dual
UNION ALL SELECT 1980, 'NAICS', 7131, 'Amusement Parks and Arcades', 3, 1979 FROM dual
UNION ALL SELECT 1985, 'NAICS', 7132, 'Gambling Industries', 3, 1979 FROM dual
UNION ALL SELECT 1990, 'NAICS', 7139, 'Other Amusement and Recreation Industries', 3, 1979 FROM dual
UNION ALL SELECT 1984, 'NAICS', 71312, 'Amusement Arcades', 4, 1980 FROM dual
UNION ALL SELECT 1982, 'NAICS', 71311, 'Amusement and Theme Parks', 4, 1980 FROM dual
UNION ALL SELECT 1981, 'NAICS', 713110, 'Amusement and Theme Parks', 5, 1982 FROM dual
UNION ALL SELECT 1983, 'NAICS', 713120, 'Amusement Arcades', 5, 1984 FROM dual
UNION ALL SELECT 1987, 'NAICS', 71321, 'Casinos (except Casino Hotels)', 4, 1985 FROM dual
UNION ALL SELECT 1989, 'NAICS', 71329, 'Other Gambling Industries', 4, 1985 FROM dual
UNION ALL SELECT 1986, 'NAICS', 713210, 'Casinos (except Casino Hotels)', 5, 1987 FROM dual
UNION ALL SELECT 1988, 'NAICS', 713290, 'Other Gambling Industries', 5, 1989 FROM dual
UNION ALL SELECT 1992, 'NAICS', 71391, 'Golf Courses and Country Clubs', 4, 1990 FROM dual
UNION ALL SELECT 1994, 'NAICS', 71392, 'Skiing Facilities', 4, 1990 FROM dual
UNION ALL SELECT 1996, 'NAICS', 71393, 'Marinas', 4, 1990 FROM dual
UNION ALL SELECT 1998, 'NAICS', 71394, 'Fitness and Recreational Sports Centers', 4, 1990 FROM dual
UNION ALL SELECT 2000, 'NAICS', 71395, 'Bowling Centers', 4, 1990 FROM dual
UNION ALL SELECT 2002, 'NAICS', 71399, 'All Other Amusement and Recreation Industries', 4, 1990 FROM dual
UNION ALL SELECT 1991, 'NAICS', 713910, 'Golf Courses and Country Clubs', 5, 1992 FROM dual
UNION ALL SELECT 1993, 'NAICS', 713920, 'Skiing Facilities', 5, 1994 FROM dual
UNION ALL SELECT 1995, 'NAICS', 713930, 'Marinas', 5, 1996 FROM dual
UNION ALL SELECT 1997, 'NAICS', 713940, 'Fitness and Recreational Sports Centers', 5, 1998 FROM dual
UNION ALL SELECT 1999, 'NAICS', 713950, 'Bowling Centers', 5, 2000 FROM dual
UNION ALL SELECT 2001, 'NAICS', 713990, 'All Other Amusement and Recreation Industries', 5, 2002 FROM dual
UNION ALL SELECT 2020, 'NAICS', 722, 'Food Services and Drinking Places', 2, 2003 FROM dual
UNION ALL SELECT 2005, 'NAICS', 7211, 'Traveler Accommodation', 3, 2004 FROM dual
UNION ALL SELECT 2013, 'NAICS', 7212, 'RV (Recreational Vehicle) Parks and Recreational Camps', 3, 2004 FROM dual
UNION ALL SELECT 2017, 'NAICS', 7213, 'Rooming and Boarding Houses', 3, 2004 FROM dual
UNION ALL SELECT 2010, 'NAICS', 72119, 'Other Traveler Accommodation', 4, 2005 FROM dual
UNION ALL SELECT 2007, 'NAICS', 72111, 'Hotels (except Casino Hotels) and Motels', 4, 2005 FROM dual
UNION ALL SELECT 2009, 'NAICS', 72112, 'Casino Hotels', 4, 2005 FROM dual
UNION ALL SELECT 2006, 'NAICS', 721110, 'Hotels (except Casino Hotels) and Motels', 5, 2007 FROM dual
UNION ALL SELECT 2008, 'NAICS', 721120, 'Casino Hotels', 5, 2009 FROM dual
UNION ALL SELECT 2011, 'NAICS', 721191, 'Bed-and-Breakfast Inns', 5, 2010 FROM dual
UNION ALL SELECT 2012, 'NAICS', 721199, 'All Other Traveler Accommodation', 5, 2010 FROM dual
UNION ALL SELECT 2014, 'NAICS', 72121, 'RV (Recreational Vehicle) Parks and Recreational Camps', 4, 2013 FROM dual
UNION ALL SELECT 2016, 'NAICS', 721214, 'Recreational and Vacation Camps (except Campgrounds)', 5, 2014 FROM dual
UNION ALL SELECT 2015, 'NAICS', 721211, 'RV (Recreational Vehicle) Parks and Campgrounds', 5, 2014 FROM dual
UNION ALL SELECT 2019, 'NAICS', 72131, 'Rooming and Boarding Houses', 4, 2017 FROM dual
UNION ALL SELECT 2018, 'NAICS', 721310, 'Rooming and Boarding Houses', 5, 2019 FROM dual
UNION ALL SELECT 2021, 'NAICS', 7223, 'Special Food Services', 3, 2020 FROM dual
UNION ALL SELECT 2031, 'NAICS', 7225, 'Restaurants and Other Eating Places', 3, 2020 FROM dual
UNION ALL SELECT 2028, 'NAICS', 7224, 'Drinking Places (Alcoholic Beverages)', 3, 2020 FROM dual
UNION ALL SELECT 2023, 'NAICS', 72231, 'Food Service Contractors', 4, 2021 FROM dual
UNION ALL SELECT 2025, 'NAICS', 72232, 'Caterers', 4, 2021 FROM dual
UNION ALL SELECT 2027, 'NAICS', 72233, 'Mobile Food Services', 4, 2021 FROM dual
UNION ALL SELECT 2022, 'NAICS', 722310, 'Food Service Contractors', 5, 2023 FROM dual
UNION ALL SELECT 2024, 'NAICS', 722320, 'Caterers', 5, 2025 FROM dual
UNION ALL SELECT 2026, 'NAICS', 722330, 'Mobile Food Services', 5, 2027 FROM dual
UNION ALL SELECT 2030, 'NAICS', 72241, 'Drinking Places (Alcoholic Beverages)', 4, 2028 FROM dual
UNION ALL SELECT 2029, 'NAICS', 722410, 'Drinking Places (Alcoholic Beverages)', 5, 2030 FROM dual
UNION ALL SELECT 2032, 'NAICS', 72251, 'Restaurants and Other Eating Places', 4, 2031 FROM dual
UNION ALL SELECT 2036, 'NAICS', 722515, 'Snack and Nonalcoholic Beverage Bars', 5, 2032 FROM dual
UNION ALL SELECT 2033, 'NAICS', 722511, 'Full-Service Restaurants', 5, 2032 FROM dual
UNION ALL SELECT 2034, 'NAICS', 722513, 'Limited-Service Restaurants', 5, 2032 FROM dual
UNION ALL SELECT 2035, 'NAICS', 722514, 'Cafeterias, Grill Buffets, and Buffets', 5, 2032 FROM dual
UNION ALL SELECT 2131, 'NAICS', 814, 'Private Households', 2, 2037 FROM dual
UNION ALL SELECT 2103, 'NAICS', 813, 'Religious, Grantmaking, Civic, Professional, and Similar Organizations', 2, 2037 FROM dual
UNION ALL SELECT 2038, 'NAICS', 811, 'Repair and Maintenance', 2, 2037 FROM dual
UNION ALL SELECT 2061, 'NAICS', 8114, 'Personal and Household Goods Repair and Maintenance', 3, 2038 FROM dual
UNION ALL SELECT 2058, 'NAICS', 8113, 'Commercial and Industrial Machinery and Equipment (except Automotive and Electronic) Repair and Maintenance', 3, 2038 FROM dual
UNION ALL SELECT 2052, 'NAICS', 8112, 'Electronic and Precision Equipment Repair and Maintenance', 3, 2038 FROM dual
UNION ALL SELECT 2039, 'NAICS', 8111, 'Automotive Repair and Maintenance', 3, 2038 FROM dual
UNION ALL SELECT 2045, 'NAICS', 81112, 'Automotive Body, Paint, Interior, and Glass Repair', 4, 2039 FROM dual
UNION ALL SELECT 2040, 'NAICS', 81111, 'Automotive Mechanical and Electrical Repair and Maintenance', 4, 2039 FROM dual
UNION ALL SELECT 2048, 'NAICS', 81119, 'Other Automotive Repair and Maintenance', 4, 2039 FROM dual
UNION ALL SELECT 2042, 'NAICS', 811112, 'Automotive Exhaust System Repair', 5, 2040 FROM dual
UNION ALL SELECT 2043, 'NAICS', 811113, 'Automotive Transmission Repair', 5, 2040 FROM dual
UNION ALL SELECT 2041, 'NAICS', 811111, 'General Automotive Repair', 5, 2040 FROM dual
UNION ALL SELECT 2044, 'NAICS', 811118, 'Other Automotive Mechanical and Electrical Repair and Maintenance', 5, 2040 FROM dual
UNION ALL SELECT 2047, 'NAICS', 811122, 'Automotive Glass Replacement Shops', 5, 2045 FROM dual
UNION ALL SELECT 2046, 'NAICS', 811121, 'Automotive Body, Paint, and Interior Repair and Maintenance', 5, 2045 FROM dual
UNION ALL SELECT 2049, 'NAICS', 811191, 'Automotive Oil Change and Lubrication Shops', 5, 2048 FROM dual
UNION ALL SELECT 2050, 'NAICS', 811192, 'Car Washes', 5, 2048 FROM dual
UNION ALL SELECT 2051, 'NAICS', 811198, 'All Other Automotive Repair and Maintenance', 5, 2048 FROM dual
UNION ALL SELECT 2053, 'NAICS', 81121, 'Electronic and Precision Equipment Repair and Maintenance', 4, 2052 FROM dual
UNION ALL SELECT 2057, 'NAICS', 811219, 'Other Electronic and Precision Equipment Repair and Maintenance', 5, 2053 FROM dual
UNION ALL SELECT 2054, 'NAICS', 811211, 'Consumer Electronics Repair and Maintenance', 5, 2053 FROM dual
UNION ALL SELECT 2055, 'NAICS', 811212, 'Computer and Office Machine Repair and Maintenance', 5, 2053 FROM dual
UNION ALL SELECT 2056, 'NAICS', 811213, 'Communication Equipment Repair and Maintenance', 5, 2053 FROM dual
UNION ALL SELECT 2060, 'NAICS', 81131, 'Commercial and Industrial Machinery and Equipment (except Automotive and Electronic) Repair and Maintenance', 4, 2058 FROM dual
UNION ALL SELECT 2059, 'NAICS', 811310, 'Commercial and Industrial Machinery and Equipment (except Automotive and Electronic) Repair and Maintenance', 5, 2060 FROM dual
UNION ALL SELECT 2062, 'NAICS', 81141, 'Home and Garden Equipment and Appliance Repair and Maintenance', 4, 2061 FROM dual
UNION ALL SELECT 2070, 'NAICS', 81149, 'Other Personal and Household Goods Repair and Maintenance', 4, 2061 FROM dual
UNION ALL SELECT 2068, 'NAICS', 81143, 'Footwear and Leather Goods Repair', 4, 2061 FROM dual
UNION ALL SELECT 2066, 'NAICS', 81142, 'Reupholstery and Furniture Repair', 4, 2061 FROM dual
UNION ALL SELECT 2063, 'NAICS', 811411, 'Home and Garden Equipment Repair and Maintenance', 5, 2062 FROM dual
UNION ALL SELECT 2064, 'NAICS', 811412, 'Appliance Repair and Maintenance', 5, 2062 FROM dual
UNION ALL SELECT 2065, 'NAICS', 811420, 'Reupholstery and Furniture Repair', 5, 2066 FROM dual
UNION ALL SELECT 2067, 'NAICS', 811430, 'Footwear and Leather Goods Repair', 5, 2068 FROM dual
UNION ALL SELECT 2069, 'NAICS', 811490, 'Other Personal and Household Goods Repair and Maintenance', 5, 2070 FROM dual
UNION ALL SELECT 2093, 'NAICS', 8129, 'Other Personal Services', 3, 2071 FROM dual
UNION ALL SELECT 2080, 'NAICS', 8122, 'Death Care Services', 3, 2071 FROM dual
UNION ALL SELECT 2072, 'NAICS', 8121, 'Personal Care Services', 3, 2071 FROM dual
UNION ALL SELECT 2085, 'NAICS', 8123, 'Drycleaning and Laundry Services', 3, 2071 FROM dual
UNION ALL SELECT 2077, 'NAICS', 81219, 'Other Personal Care Services', 4, 2072 FROM dual
UNION ALL SELECT 2073, 'NAICS', 81211, 'Hair, Nail, and Skin Care Services', 4, 2072 FROM dual
UNION ALL SELECT 2076, 'NAICS', 812113, 'Nail Salons', 5, 2073 FROM dual
UNION ALL SELECT 2075, 'NAICS', 812112, 'Beauty Salons', 5, 2073 FROM dual
UNION ALL SELECT 2074, 'NAICS', 812111, 'Barber Shops', 5, 2073 FROM dual
UNION ALL SELECT 2078, 'NAICS', 812191, 'Diet and Weight Reducing Centers', 5, 2077 FROM dual
UNION ALL SELECT 2079, 'NAICS', 812199, 'Other Personal Care Services', 5, 2077 FROM dual
UNION ALL SELECT 2082, 'NAICS', 81221, 'Funeral Homes and Funeral Services', 4, 2080 FROM dual
UNION ALL SELECT 2084, 'NAICS', 81222, 'Cemeteries and Crematories', 4, 2080 FROM dual
UNION ALL SELECT 2081, 'NAICS', 812210, 'Funeral Homes and Funeral Services', 5, 2082 FROM dual
UNION ALL SELECT 2083, 'NAICS', 812220, 'Cemeteries and Crematories', 5, 2084 FROM dual
UNION ALL SELECT 2087, 'NAICS', 81231, 'Coin-Operated Laundries and Drycleaners', 4, 2085 FROM dual
UNION ALL SELECT 2090, 'NAICS', 81233, 'Linen and Uniform Supply', 4, 2085 FROM dual
UNION ALL SELECT 2089, 'NAICS', 81232, 'Drycleaning and Laundry Services (except Coin-Operated)', 4, 2085 FROM dual
UNION ALL SELECT 2086, 'NAICS', 812310, 'Coin-Operated Laundries and Drycleaners', 5, 2087 FROM dual
UNION ALL SELECT 2088, 'NAICS', 812320, 'Drycleaning and Laundry Services (except Coin-Operated)', 5, 2089 FROM dual
UNION ALL SELECT 2092, 'NAICS', 812332, 'Industrial Launderers', 5, 2090 FROM dual
UNION ALL SELECT 2091, 'NAICS', 812331, 'Linen Supply', 5, 2090 FROM dual
UNION ALL SELECT 2102, 'NAICS', 81299, 'All Other Personal Services', 4, 2093 FROM dual
UNION ALL SELECT 2100, 'NAICS', 81293, 'Parking Lots and Garages', 4, 2093 FROM dual
UNION ALL SELECT 2096, 'NAICS', 81292, 'Photofinishing', 4, 2093 FROM dual
UNION ALL SELECT 2095, 'NAICS', 81291, 'Pet Care (except Veterinary) Services', 4, 2093 FROM dual
UNION ALL SELECT 2094, 'NAICS', 812910, 'Pet Care (except Veterinary) Services', 5, 2095 FROM dual
UNION ALL SELECT 2098, 'NAICS', 812922, 'One-Hour Photofinishing', 5, 2096 FROM dual
UNION ALL SELECT 2097, 'NAICS', 812921, 'Photofinishing Laboratories (except One-Hour)', 5, 2096 FROM dual
UNION ALL SELECT 2099, 'NAICS', 812930, 'Parking Lots and Garages', 5, 2100 FROM dual
UNION ALL SELECT 2101, 'NAICS', 812990, 'All Other Personal Services', 5, 2102 FROM dual
UNION ALL SELECT 2120, 'NAICS', 8139, 'Business, Professional, Labor, Political, and Similar Organizations', 3, 2103 FROM dual
UNION ALL SELECT 2107, 'NAICS', 8132, 'Grantmaking and Giving Services', 3, 2103 FROM dual
UNION ALL SELECT 2104, 'NAICS', 8131, 'Religious Organizations', 3, 2103 FROM dual
UNION ALL SELECT 2117, 'NAICS', 8134, 'Civic and Social Organizations', 3, 2103 FROM dual
UNION ALL SELECT 2112, 'NAICS', 8133, 'Social Advocacy Organizations', 3, 2103 FROM dual
UNION ALL SELECT 2106, 'NAICS', 81311, 'Religious Organizations', 4, 2104 FROM dual
UNION ALL SELECT 2105, 'NAICS', 813110, 'Religious Organizations', 5, 2106 FROM dual
UNION ALL SELECT 2108, 'NAICS', 81321, 'Grantmaking and Giving Services', 4, 2107 FROM dual
UNION ALL SELECT 2110, 'NAICS', 813212, 'Voluntary Health Organizations', 5, 2108 FROM dual
UNION ALL SELECT 2111, 'NAICS', 813219, 'Other Grantmaking and Giving Services', 5, 2108 FROM dual
UNION ALL SELECT 2109, 'NAICS', 813211, 'Grantmaking Foundations', 5, 2108 FROM dual
UNION ALL SELECT 2113, 'NAICS', 81331, 'Social Advocacy Organizations', 4, 2112 FROM dual
UNION ALL SELECT 2115, 'NAICS', 813312, 'Environment, Conservation and Wildlife Organizations', 5, 2113 FROM dual
UNION ALL SELECT 2114, 'NAICS', 813311, 'Human Rights Organizations', 5, 2113 FROM dual
UNION ALL SELECT 2116, 'NAICS', 813319, 'Other Social Advocacy Organizations', 5, 2113 FROM dual
UNION ALL SELECT 2119, 'NAICS', 81341, 'Civic and Social Organizations', 4, 2117 FROM dual
UNION ALL SELECT 2118, 'NAICS', 813410, 'Civic and Social Organizations', 5, 2119 FROM dual
UNION ALL SELECT 2124, 'NAICS', 81392, 'Professional Organizations', 4, 2120 FROM dual
UNION ALL SELECT 2122, 'NAICS', 81391, 'Business Associations', 4, 2120 FROM dual
UNION ALL SELECT 2130, 'NAICS', 81399, 'Other Similar Organizations (except Business, Professional, Labor, and Political Organizations)', 4, 2120 FROM dual
UNION ALL SELECT 2126, 'NAICS', 81393, 'Labor Unions and Similar Labor Organizations', 4, 2120 FROM dual
UNION ALL SELECT 2128, 'NAICS', 81394, 'Political Organizations', 4, 2120 FROM dual
UNION ALL SELECT 2121, 'NAICS', 813910, 'Business Associations', 5, 2122 FROM dual
UNION ALL SELECT 2123, 'NAICS', 813920, 'Professional Organizations', 5, 2124 FROM dual
UNION ALL SELECT 2125, 'NAICS', 813930, 'Labor Unions and Similar Labor Organizations', 5, 2126 FROM dual
UNION ALL SELECT 2127, 'NAICS', 813940, 'Political Organizations', 5, 2128 FROM dual
UNION ALL SELECT 2129, 'NAICS', 813990, 'Other Similar Organizations (except Business, Professional, Labor, and Political Organizations)', 5, 2130 FROM dual
UNION ALL SELECT 2132, 'NAICS', 8141, 'Private Households', 3, 2131 FROM dual
UNION ALL SELECT 2134, 'NAICS', 81411, 'Private Households', 4, 2132 FROM dual
UNION ALL SELECT 2133, 'NAICS', 814110, 'Private Households', 5, 2134 FROM dual
UNION ALL SELECT 2182, 'NAICS', 925, 'Administration of Housing Programs, Urban Planning, and Community Development', 2, 2135 FROM dual
UNION ALL SELECT 2204, 'NAICS', 928, 'National Security and International Affairs', 2, 2135 FROM dual
UNION ALL SELECT 2200, 'NAICS', 927, 'Space Research and Technology', 2, 2135 FROM dual
UNION ALL SELECT 2188, 'NAICS', 926, 'Administration of Economic Programs', 2, 2135 FROM dual
UNION ALL SELECT 2176, 'NAICS', 924, 'Administration of Environmental Quality Programs', 2, 2135 FROM dual
UNION ALL SELECT 2166, 'NAICS', 923, 'Administration of Human Resource Programs', 2, 2135 FROM dual
UNION ALL SELECT 2136, 'NAICS', 921, 'Executive, Legislative, and Other General Government Support', 2, 2135 FROM dual
UNION ALL SELECT 2150, 'NAICS', 922, 'Justice, Public Order, and Safety Activities', 2, 2135 FROM dual
UNION ALL SELECT 2137, 'NAICS', 9211, 'Executive, Legislative, and Other General Government Support', 3, 2136 FROM dual
UNION ALL SELECT 2149, 'NAICS', 92119, 'Other General Government Support', 4, 2137 FROM dual
UNION ALL SELECT 2143, 'NAICS', 92113, 'Public Finance Activities', 4, 2137 FROM dual
UNION ALL SELECT 2147, 'NAICS', 92115, 'American Indian and Alaska Native Tribal Governments', 4, 2137 FROM dual
UNION ALL SELECT 2141, 'NAICS', 92112, 'Legislative Bodies', 4, 2137 FROM dual
UNION ALL SELECT 2145, 'NAICS', 92114, 'Executive and Legislative Offices, Combined', 4, 2137 FROM dual
UNION ALL SELECT 2139, 'NAICS', 92111, 'Executive Offices', 4, 2137 FROM dual
UNION ALL SELECT 2138, 'NAICS', 921110, 'Executive Offices', 5, 2139 FROM dual
UNION ALL SELECT 2140, 'NAICS', 921120, 'Legislative Bodies', 5, 2141 FROM dual
UNION ALL SELECT 2142, 'NAICS', 921130, 'Public Finance Activities', 5, 2143 FROM dual
UNION ALL SELECT 2144, 'NAICS', 921140, 'Executive and Legislative Offices, Combined', 5, 2145 FROM dual
UNION ALL SELECT 2146, 'NAICS', 921150, 'American Indian and Alaska Native Tribal Governments', 5, 2147 FROM dual
UNION ALL SELECT 2148, 'NAICS', 921190, 'Other General Government Support', 5, 2149 FROM dual
UNION ALL SELECT 2476, 'SEC', 3800, 'INSTRUMENTS & RELATED PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2151, 'NAICS', 9221, 'Justice, Public Order, and Safety Activities', 3, 2150 FROM dual
UNION ALL SELECT 2157, 'NAICS', 92213, 'Legal Counsel and Prosecution', 4, 2151 FROM dual
UNION ALL SELECT 2161, 'NAICS', 92215, 'Parole Offices and Probation Offices', 4, 2151 FROM dual
UNION ALL SELECT 2165, 'NAICS', 92219, 'Other Justice, Public Order, and Safety Activities', 4, 2151 FROM dual
UNION ALL SELECT 2153, 'NAICS', 92211, 'Courts', 4, 2151 FROM dual
UNION ALL SELECT 2155, 'NAICS', 92212, 'Police Protection', 4, 2151 FROM dual
UNION ALL SELECT 2159, 'NAICS', 92214, 'Correctional Institutions', 4, 2151 FROM dual
UNION ALL SELECT 2152, 'NAICS', 922110, 'Courts', 5, 2153 FROM dual
UNION ALL SELECT 2154, 'NAICS', 922120, 'Police Protection', 5, 2155 FROM dual
UNION ALL SELECT 2156, 'NAICS', 922130, 'Legal Counsel and Prosecution', 5, 2157 FROM dual
UNION ALL SELECT 2158, 'NAICS', 922140, 'Correctional Institutions', 5, 2159 FROM dual
UNION ALL SELECT 2160, 'NAICS', 922150, 'Parole Offices and Probation Offices', 5, 2161 FROM dual
UNION ALL SELECT 2162, 'NAICS', 922160, 'Fire Protection', 5, 2163 FROM dual
UNION ALL SELECT 2164, 'NAICS', 922190, 'Other Justice, Public Order, and Safety Activities', 5, 2165 FROM dual
UNION ALL SELECT 2167, 'NAICS', 9231, 'Administration of Human Resource Programs', 3, 2166 FROM dual
UNION ALL SELECT 2173, 'NAICS', 92313, 'Administration of Human Resource Programs (except Education, Public Health, and Veterans'' Affairs Programs)', 4, 2167 FROM dual
UNION ALL SELECT 2169, 'NAICS', 92311, 'Administration of Education Programs', 4, 2167 FROM dual
UNION ALL SELECT 2171, 'NAICS', 92312, 'Administration of Public Health Programs', 4, 2167 FROM dual
UNION ALL SELECT 2175, 'NAICS', 92314, 'Administration of Veterans'' Affairs', 4, 2167 FROM dual
UNION ALL SELECT 2168, 'NAICS', 923110, 'Administration of Education Programs', 5, 2169 FROM dual
UNION ALL SELECT 2170, 'NAICS', 923120, 'Administration of Public Health Programs', 5, 2171 FROM dual
UNION ALL SELECT 2172, 'NAICS', 923130, 'Administration of Human Resource Programs (except Education, Public Health, and Veterans'' Affairs Programs)', 5, 2173 FROM dual
UNION ALL SELECT 2174, 'NAICS', 923140, 'Administration of Veterans'' Affairs', 5, 2175 FROM dual
UNION ALL SELECT 2177, 'NAICS', 9241, 'Administration of Environmental Quality Programs', 3, 2176 FROM dual
UNION ALL SELECT 2179, 'NAICS', 92411, 'Administration of Air and Water Resource and Solid Waste Management Programs', 4, 2177 FROM dual
UNION ALL SELECT 2181, 'NAICS', 92412, 'Administration of Conservation Programs', 4, 2177 FROM dual
UNION ALL SELECT 2178, 'NAICS', 924110, 'Administration of Air and Water Resource and Solid Waste Management Programs', 5, 2179 FROM dual
UNION ALL SELECT 2180, 'NAICS', 924120, 'Administration of Conservation Programs', 5, 2181 FROM dual
UNION ALL SELECT 2183, 'NAICS', 9251, 'Administration of Housing Programs, Urban Planning, and Community Development', 3, 2182 FROM dual
UNION ALL SELECT 2185, 'NAICS', 92511, 'Administration of Housing Programs', 4, 2183 FROM dual
UNION ALL SELECT 2187, 'NAICS', 92512, 'Administration of Urban Planning and Community and Rural Development', 4, 2183 FROM dual
UNION ALL SELECT 2184, 'NAICS', 925110, 'Administration of Housing Programs', 5, 2185 FROM dual
UNION ALL SELECT 2186, 'NAICS', 925120, 'Administration of Urban Planning and Community and Rural Development', 5, 2187 FROM dual
UNION ALL SELECT 2189, 'NAICS', 9261, 'Administration of Economic Program', 3, 2188 FROM dual
UNION ALL SELECT 2193, 'NAICS', 92612, 'Regulation and Administration of Transportation Programs', 4, 2189 FROM dual
UNION ALL SELECT 2199, 'NAICS', 92615, 'Regulation, Licensing, and Inspection of Miscellaneous Commercial Sectors', 4, 2189 FROM dual
UNION ALL SELECT 2197, 'NAICS', 92614, 'Regulation of Agricultural Marketing and Commodities', 4, 2189 FROM dual
UNION ALL SELECT 2195, 'NAICS', 92613, 'Regulation and Administration of Communications, Electric, Gas, and Other Utilities', 4, 2189 FROM dual
UNION ALL SELECT 2191, 'NAICS', 92611, 'Administration of General Economic Programs', 4, 2189 FROM dual
UNION ALL SELECT 2190, 'NAICS', 926110, 'Administration of General Economic Programs', 5, 2191 FROM dual
UNION ALL SELECT 2192, 'NAICS', 926120, 'Regulation and Administration of Transportation Programs', 5, 2193 FROM dual
UNION ALL SELECT 2194, 'NAICS', 926130, 'Regulation and Administration of Communications, Electric, Gas, and Other Utilities', 5, 2195 FROM dual
UNION ALL SELECT 2196, 'NAICS', 926140, 'Regulation of Agricultural Marketing and Commodities', 5, 2197 FROM dual
UNION ALL SELECT 2198, 'NAICS', 926150, 'Regulation, Licensing, and Inspection of Miscellaneous Commercial Sectors', 5, 2199 FROM dual
UNION ALL SELECT 2201, 'NAICS', 9271, 'Space Research and Technology', 3, 2200 FROM dual
UNION ALL SELECT 2203, 'NAICS', 92711, 'Space Research and Technology', 4, 2201 FROM dual
UNION ALL SELECT 2202, 'NAICS', 927110, 'Space Research and Technology', 5, 2203 FROM dual
UNION ALL SELECT 2205, 'NAICS', 9281, 'National Security and International Affairs', 3, 2204 FROM dual
UNION ALL SELECT 2209, 'NAICS', 92812, 'International Affairs', 4, 2205 FROM dual
UNION ALL SELECT 2207, 'NAICS', 92811, 'National Security', 4, 2205 FROM dual
UNION ALL SELECT 2206, 'NAICS', 928110, 'National Security', 5, 2207 FROM dual
UNION ALL SELECT 2208, 'NAICS', 928120, 'International Affairs', 5, 2209 FROM dual
UNION ALL SELECT 2228, 'SEC', 1400, 'NONMETALLIC MINERALS, EXCEPT FUELS', 2, 2789 FROM dual
UNION ALL SELECT 2221, 'SEC', 1300, 'OIL & GAS EXTRACTION', 2, 2789 FROM dual
UNION ALL SELECT 2215, 'SEC', 1000, 'METAL MINING', 2, 2789 FROM dual
UNION ALL SELECT 2918, 'SIC', 1400, 'NONMETALLIC MINERALS, EXCEPT FUELS', 2, 4306 FROM dual
UNION ALL SELECT 2884, 'SIC', 1000, 'METAL MINING', 2, 4306 FROM dual
UNION ALL SELECT 2909, 'SIC', 1300, 'OIL & GAS EXTRACTION', 2, 4306 FROM dual
UNION ALL SELECT 2901, 'SIC', 1200, 'COAL MINING', 2, 4306 FROM dual
UNION ALL SELECT 2234, 'SEC', 1600, 'HEAVY CONSTRUCTION, EXCEPT BUILDING', 2, 2790 FROM dual
UNION ALL SELECT 2229, 'SEC', 1500, 'GENERAL BUILDING CONTRACTORS', 2, 2790 FROM dual
UNION ALL SELECT 2237, 'SEC', 1700, 'SPECIAL TRADE CONTRACTORS', 2, 2790 FROM dual
UNION ALL SELECT 2939, 'SIC', 1500, 'GENERAL BUILDING CONTRACTORS', 2, 4307 FROM dual
UNION ALL SELECT 2955, 'SIC', 1700, 'SPECIAL TRADE CONTRACTORS', 2, 4307 FROM dual
UNION ALL SELECT 2948, 'SIC', 1600, 'HEAVY CONSTRUCTION, EXCEPT BUILDING', 2, 4307 FROM dual
UNION ALL SELECT 2302, 'SEC', 2700, 'PRINTING & PUBLISHING', 2, 2791 FROM dual
UNION ALL SELECT 2292, 'SEC', 2600, 'PAPER & ALLIED PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2283, 'SEC', 2500, 'FURNITURE & FIXTURES', 2, 2791 FROM dual
UNION ALL SELECT 2276, 'SEC', 2400, 'LUMBER & WOOD PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2271, 'SEC', 2300, 'APPAREL & OTHER TEXTILE PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2262, 'SEC', 2200, 'TEXTILE MILL PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2259, 'SEC', 2100, 'TOBACCO PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2240, 'SEC', 2000, 'FOOD & KINDRED PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2500, 'SEC', 3900, 'MISCELLANEOUS MANUFACTURING INDUSTRIES', 2, 2791 FROM dual
UNION ALL SELECT 2337, 'SEC', 2900, 'PETROLEUM & COAL PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2342, 'SEC', 3000, 'RUBBER & MISCELLANEOUS PLASTICS PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2353, 'SEC', 3100, 'LEATHER & ALLIED PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2355, 'SEC', 3200, 'STONE, CLAY & GLASS PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2371, 'SEC', 3300, 'PRIMARY METAL INDUSTRIES', 2, 2791 FROM dual
UNION ALL SELECT 2384, 'SEC', 3400, 'FABRICATED METAL PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 2403, 'SEC', 3500, 'INDUSTRIAL MACHINERY & EQUIPMENT', 2, 2791 FROM dual
UNION ALL SELECT 2434, 'SEC', 3600, 'ELECTRONIC & OTHER ELECTRIC INDUSTRIES', 2, 2791 FROM dual
UNION ALL SELECT 2458, 'SEC', 3700, 'TRANSPORTATION EQUIPMENT', 2, 2791 FROM dual
UNION ALL SELECT 2319, 'SEC', 2800, 'CHEMICALS & ALLIED PRODUCTS', 2, 2791 FROM dual
UNION ALL SELECT 3190, 'SIC', 2700, 'PRINTING & PUBLISHING', 2, 4308 FROM dual
UNION ALL SELECT 3124, 'SIC', 2400, 'LUMBER & WOOD PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3083, 'SIC', 2300, 'APPAREL & OTHER TEXTILE PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3050, 'SIC', 2200, 'TEXTILE MILL PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3041, 'SIC', 2100, 'TOBACCO PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 2982, 'SIC', 2000, 'FOOD & KINDRED PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3526, 'SIC', 3700, 'TRANSPORTATION EQUIPMENT', 2, 4308 FROM dual
UNION ALL SELECT 3282, 'SIC', 3100, 'LEATHER & ALLIED PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3576, 'SIC', 3900, 'MISCELLANEOUS MANUFACTURING INDUSTRIES', 2, 4308 FROM dual
UNION ALL SELECT 3552, 'SIC', 3800, 'INSTRUMENTS & RELATED PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3480, 'SIC', 3600, 'ELECTRONIC & OTHER ELECTRIC INDUSTRIES', 2, 4308 FROM dual
UNION ALL SELECT 3419, 'SIC', 3500, 'INDUSTRIAL MACHINERY & EQUIPMENT', 2, 4308 FROM dual
UNION ALL SELECT 3371, 'SIC', 3400, 'FABRICATED METAL PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3337, 'SIC', 3300, 'PRIMARY METAL INDUSTRIES', 2, 4308 FROM dual
UNION ALL SELECT 3301, 'SIC', 3200, 'STONE, CLAY & GLASS PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3261, 'SIC', 3000, 'RUBBER & MISCELLANEOUS PLASTICS PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3252, 'SIC', 2900, 'PETROLEUM & COAL PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3214, 'SIC', 2800, 'CHEMICALS & ALLIED PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3167, 'SIC', 2600, 'PAPER & ALLIED PRODUCTS', 2, 4308 FROM dual
UNION ALL SELECT 3148, 'SIC', 2500, 'FURNITURE & FIXTURES', 2, 4308 FROM dual
UNION ALL SELECT 2512, 'SEC', 4000, 'RAILROAD TRANSPORTATION', 2, 2792 FROM dual
UNION ALL SELECT 2516, 'SEC', 4100, 'LOCAL & INTERURBAN PASSENGER TRANSIT', 2, 2792 FROM dual
UNION ALL SELECT 2552, 'SEC', 4900, 'ELECTRIC, GAS & SANITARY SERVICES', 2, 2792 FROM dual
UNION ALL SELECT 2539, 'SEC', 4800, 'COMMUNICATIONS', 2, 2792 FROM dual
UNION ALL SELECT 2536, 'SEC', 4700, 'TRANSPORTATION SERVICES', 2, 2792 FROM dual
UNION ALL SELECT 2534, 'SEC', 4600, 'PIPELINES, EXCEPT NATURAL GAS', 2, 2792 FROM dual
UNION ALL SELECT 2526, 'SEC', 4500, 'TRANSPORT BY AIR', 2, 2792 FROM dual
UNION ALL SELECT 2523, 'SEC', 4400, 'WATER TRANSPORTATION', 2, 2792 FROM dual
UNION ALL SELECT 2517, 'SEC', 4200, 'TRUCKING & WAREHOUSING', 2, 2792 FROM dual
UNION ALL SELECT 3633, 'SIC', 4300, 'US POSTAL SERVICE', 2, 4309 FROM dual
UNION ALL SELECT 3620, 'SIC', 4200, 'TRUCKING & WAREHOUSING', 2, 4309 FROM dual
UNION ALL SELECT 3601, 'SIC', 4000, 'RAILROAD TRANSPORTATION', 2, 4309 FROM dual
UNION ALL SELECT 3605, 'SIC', 4100, 'LOCAL & INTERURBAN PASSENGER TRANSIT', 2, 4309 FROM dual
UNION ALL SELECT 3693, 'SIC', 4900, 'ELECTRIC, GAS & SANITARY SERVICES', 2, 4309 FROM dual
UNION ALL SELECT 3680, 'SIC', 4800, 'COMMUNICATIONS', 2, 4309 FROM dual
UNION ALL SELECT 3667, 'SIC', 4700, 'TRANSPORTATION SERVICES', 2, 4309 FROM dual
UNION ALL SELECT 3662, 'SIC', 4600, 'PIPELINES, EXCEPT NATURAL GAS', 2, 4309 FROM dual
UNION ALL SELECT 3654, 'SIC', 4500, 'TRANSPORT BY AIR', 2, 4309 FROM dual
UNION ALL SELECT 3636, 'SIC', 4400, 'WATER TRANSPORTATION', 2, 4309 FROM dual
UNION ALL SELECT 2568, 'SEC', 5000, 'WHOLESALE TRADE - DURABLE GOODS', 2, 2793 FROM dual
UNION ALL SELECT 2591, 'SEC', 5100, 'WHOLESALE TRADE - NONDURABLE GOODS', 2, 2793 FROM dual
UNION ALL SELECT 3762, 'SIC', 5100, 'WHOLESALE TRADE - NONDURABLE GOODS', 2, 4310 FROM dual
UNION ALL SELECT 3715, 'SIC', 5000, 'WHOLESALE TRADE - DURABLE GOODS', 2, 4310 FROM dual
UNION ALL SELECT 2670, 'SEC', 6200, 'SECURITY & COMMODITY BROKERS', 2, 2795 FROM dual
UNION ALL SELECT 2620, 'SEC', 5500, 'AUTOMOTIVE DEALERS & SERVICE STATIONS', 2, 2794 FROM dual
UNION ALL SELECT 2623, 'SEC', 5600, 'APPAREL & ACCESSORY STORES', 2, 2794 FROM dual
UNION ALL SELECT 2617, 'SEC', 5400, 'FOOD STORES', 2, 2794 FROM dual
UNION ALL SELECT 2605, 'SEC', 5200, 'BUILDING MATERIALS & GARDEN SUPPLIES', 2, 2794 FROM dual
UNION ALL SELECT 2610, 'SEC', 5300, 'GENERAL MERCHANDISE STORES', 2, 2794 FROM dual
UNION ALL SELECT 2630, 'SEC', 5700, 'FURNITURE & HOMEFURNISHING STORES', 2, 2794 FROM dual
UNION ALL SELECT 2637, 'SEC', 5800, 'EATING & DRINKING PLACES', 2, 2794 FROM dual
UNION ALL SELECT 2640, 'SEC', 5900, 'MISCELLANEOUS RETAIL', 2, 2794 FROM dual
UNION ALL SELECT 3837, 'SIC', 5500, 'AUTOMOTIVE DEALERS & SERVICE STATIONS', 2, 4311 FROM dual
UNION ALL SELECT 3854, 'SIC', 5600, 'APPAREL & ACCESSORY STORES', 2, 4311 FROM dual
UNION ALL SELECT 3882, 'SIC', 5800, 'EATING & DRINKING PLACES', 2, 4311 FROM dual
UNION ALL SELECT 3869, 'SIC', 5700, 'FURNITURE & HOMEFURNISHING STORES', 2, 4311 FROM dual
UNION ALL SELECT 3886, 'SIC', 5900, 'MISCELLANEOUS RETAIL', 2, 4311 FROM dual
UNION ALL SELECT 3804, 'SIC', 5200, 'BUILDING MATERIALS & GARDEN SUPPLIES', 2, 4311 FROM dual
UNION ALL SELECT 3815, 'SIC', 5300, 'GENERAL MERCHANDISE STORES', 2, 4311 FROM dual
UNION ALL SELECT 3822, 'SIC', 5400, 'FOOD STORES', 2, 4311 FROM dual
UNION ALL SELECT 2649, 'SEC', 6000, 'DEPOSITORY INSTITUTIONS', 2, 2795 FROM dual
UNION ALL SELECT 2677, 'SEC', 6300, 'INSURANCE CARRIERS', 2, 2795 FROM dual
UNION ALL SELECT 2691, 'SEC', 6400, 'INSURANCE AGENTS, BROKERS & SERVICE', 2, 2795 FROM dual
UNION ALL SELECT 2659, 'SEC', 6100, 'NONDEPOSITORY INSTITUTIONS', 2, 2795 FROM dual
UNION ALL SELECT 2703, 'SEC', 6700, 'HOLDING & OTHER INVESTMENT OFFICES', 2, 2795 FROM dual
UNION ALL SELECT 2694, 'SEC', 6500, 'REAL ESTATE', 2, 2795 FROM dual
UNION ALL SELECT 3948, 'SIC', 6200, 'SECURITY & COMMODITY BROKERS', 2, 4312 FROM dual
UNION ALL SELECT 3937, 'SIC', 6100, 'NONDEPOSITORY INSTITUTIONS', 2, 4312 FROM dual
UNION ALL SELECT 3917, 'SIC', 6000, 'DEPOSITORY INSTITUTIONS', 2, 4312 FROM dual
UNION ALL SELECT 3992, 'SIC', 6700, 'HOLDING & OTHER INVESTMENT OFFICES', 2, 4312 FROM dual
UNION ALL SELECT 3977, 'SIC', 6500, 'REAL ESTATE', 2, 4312 FROM dual
UNION ALL SELECT 3974, 'SIC', 6400, 'INSURANCE AGENTS, BROKERS & SERVICE', 2, 4312 FROM dual
UNION ALL SELECT 3958, 'SIC', 6300, 'INSURANCE CARRIERS', 2, 4312 FROM dual
UNION ALL SELECT 2766, 'SEC', 8100, 'LEGAL SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2712, 'SEC', 7200, 'PERSONAL SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2713, 'SEC', 7300, 'BUSINESS SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2735, 'SEC', 7500, 'AUTO REPAIR, SERVICES & PARKING', 2, 2796 FROM dual
UNION ALL SELECT 2737, 'SEC', 7600, 'MISCELLANEOUS REPAIR SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2738, 'SEC', 7800, 'MOTION PICTURES', 2, 2796 FROM dual
UNION ALL SELECT 2769, 'SEC', 8200, 'EDUCATIONAL SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2770, 'SEC', 8300, 'SOCIAL SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2773, 'SEC', 8600, 'MEMBERSHIP ORGANIZATIONS', 2, 2796 FROM dual
UNION ALL SELECT 2774, 'SEC', 8700, 'ENGINEERING & MANAGEMENT SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2784, 'SEC', 8900, 'SERVICES, NEC', 2, 2796 FROM dual
UNION ALL SELECT 2748, 'SEC', 7900, 'AMUSEMENT & RECREATIONAL SERVICES', 2, 2796 FROM dual
UNION ALL SELECT 2753, 'SEC', 8000, 'HEALTH SERVICE', 2, 2796 FROM dual
UNION ALL SELECT 2709, 'SEC', 7000, 'HOTELS & OTHER LODGING PLACES', 2, 2796 FROM dual
UNION ALL SELECT 4203, 'SIC', 8400, 'MUSEUMS, BOTANICAL, ZOOLOGICAL GARDENS', 2, 4313 FROM dual
UNION ALL SELECT 4244, 'SIC', 8900, 'SERVICES, NEC', 2, 4313 FROM dual
UNION ALL SELECT 4241, 'SIC', 8800, 'PRIVATE HOUSEHOLDS', 2, 4313 FROM dual
UNION ALL SELECT 4223, 'SIC', 8700, 'ENGINEERING & MANAGEMENT SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4208, 'SIC', 8600, 'MEMBERSHIP ORGANIZATIONS', 2, 4313 FROM dual
UNION ALL SELECT 4007, 'SIC', 7000, 'HOTELS & OTHER LODGING PLACES', 2, 4313 FROM dual
UNION ALL SELECT 4017, 'SIC', 7200, 'PERSONAL SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4040, 'SIC', 7300, 'BUSINESS SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4081, 'SIC', 7500, 'AUTO REPAIR, SERVICES & PARKING', 2, 4313 FROM dual
UNION ALL SELECT 4100, 'SIC', 7600, 'MISCELLANEOUS REPAIR SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4113, 'SIC', 7800, 'MOTION PICTURES', 2, 4313 FROM dual
UNION ALL SELECT 4125, 'SIC', 7900, 'AMUSEMENT & RECREATIONAL SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4146, 'SIC', 8000, 'HEALTH SERVICE', 2, 4313 FROM dual
UNION ALL SELECT 4175, 'SIC', 8100, 'LEGAL SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4178, 'SIC', 8200, 'EDUCATIONAL SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 4192, 'SIC', 8300, 'SOCIAL SERVICES', 2, 4313 FROM dual
UNION ALL SELECT 2785, 'SEC', 9700, 'NATIONAL SECURITY & INTL. AFFAIRS', 2, 2797 FROM dual
UNION ALL SELECT 4256, 'SIC', 9200, 'JUSTICE, PUBLIC ORDER & SAFETY', 2, 4314 FROM dual
UNION ALL SELECT 4277, 'SIC', 9500, 'ENVIRONMENTAL QUALITY & HOUSING', 2, 4314 FROM dual
UNION ALL SELECT 4247, 'SIC', 9100, 'EXECUTIVE, LEGISLATIVE & GENERAL', 2, 4314 FROM dual
UNION ALL SELECT 4265, 'SIC', 9300, 'FINANCE, TAXATION & MONETARY POLICY', 2, 4314 FROM dual
UNION ALL SELECT 4302, 'SIC', 9900, 'NONCLASSIFIABLE ESTABLISHMENTS', 2, 4314 FROM dual
UNION ALL SELECT 4268, 'SIC', 9400, 'ADMINISTRATION OF HUMAN RESOURCES', 2, 4314 FROM dual
UNION ALL SELECT 4297, 'SIC', 9700, 'NATIONAL SECURITY & INTL. AFFAIRS', 2, 4314 FROM dual
UNION ALL SELECT 4284, 'SIC', 9600, 'ADMINISTRATION OF ECONOMIC PROGRAMS', 2, 4314 FROM dual
UNION ALL SELECT 2845, 'SIC', 290, 'General Farms, Primarily Animal', 3, 2825 FROM dual
UNION ALL SELECT 2826, 'SIC', 210, 'Livestock, Except Dairy & Poultry', 3, 2825 FROM dual
UNION ALL SELECT 2840, 'SIC', 270, 'Animal Specialties', 3, 2825 FROM dual
UNION ALL SELECT 2832, 'SIC', 240, 'Dairy Farms', 3, 2825 FROM dual
UNION ALL SELECT 2834, 'SIC', 250, 'Poultry & Eggs', 3, 2825 FROM dual
UNION ALL SELECT 2855, 'SIC', 740, 'Veterinary Services', 3, 2847 FROM dual
UNION ALL SELECT 2864, 'SIC', 780, 'Landscape & Horticultural Services', 3, 2847 FROM dual
UNION ALL SELECT 2861, 'SIC', 760, 'Farm Labor & Management Services', 3, 2847 FROM dual
UNION ALL SELECT 2850, 'SIC', 720, 'Crop Services', 3, 2847 FROM dual
UNION ALL SELECT 2848, 'SIC', 710, 'Soil Preparation Services', 3, 2847 FROM dual
UNION ALL SELECT 2858, 'SIC', 750, 'Animal Services, Except Veterinary', 3, 2847 FROM dual
UNION ALL SELECT 2871, 'SIC', 830, 'Forest Products', 3, 2868 FROM dual
UNION ALL SELECT 2869, 'SIC', 810, 'Timber Tracts', 3, 2868 FROM dual
UNION ALL SELECT 2873, 'SIC', 850, 'Forestry Services', 3, 2868 FROM dual
UNION ALL SELECT 2882, 'SIC', 970, 'Hunting, Trapping & Game Propagation', 3, 2875 FROM dual
UNION ALL SELECT 2880, 'SIC', 920, 'Fish Hatcheries & Preserves', 3, 2875 FROM dual
UNION ALL SELECT 2876, 'SIC', 910, 'Commercial Fishing', 3, 2875 FROM dual
UNION ALL SELECT 2216, 'SEC', 1040, 'Gold & Silver Ores', 3, 2215 FROM dual
UNION ALL SELECT 2217, 'SEC', 1090, 'Miscellaneous Metal Ores', 3, 2215 FROM dual
UNION ALL SELECT 2891, 'SIC', 1040, 'Gold & Silver Ores', 3, 2884 FROM dual
UNION ALL SELECT 2898, 'SIC', 1090, 'Miscellaneous Metal Ores', 3, 2884 FROM dual
UNION ALL SELECT 2889, 'SIC', 1030, 'Lead & Zinc Ores', 3, 2884 FROM dual
UNION ALL SELECT 2885, 'SIC', 1010, 'Iron Ores', 3, 2884 FROM dual
UNION ALL SELECT 2887, 'SIC', 1020, 'Copper Ores', 3, 2884 FROM dual
UNION ALL SELECT 2894, 'SIC', 1060, 'Ferroalloy Ores, Except Vanadium', 3, 2884 FROM dual
UNION ALL SELECT 2896, 'SIC', 1080, 'Metal Mining Services', 3, 2884 FROM dual
UNION ALL SELECT 2219, 'SEC', 1220, 'Bituminous Coal & Lignite Mining', 3, 2218 FROM dual
UNION ALL SELECT 2907, 'SIC', 1240, 'Coal Mining Services', 3, 2901 FROM dual
UNION ALL SELECT 2905, 'SIC', 1230, 'Anthracite Mining', 3, 2901 FROM dual
UNION ALL SELECT 2902, 'SIC', 1220, 'Bituminous Coal & Lignite Mining', 3, 2901 FROM dual
UNION ALL SELECT 2224, 'SEC', 1380, 'Oil & Gas Field Services', 3, 2221 FROM dual
UNION ALL SELECT 2222, 'SEC', 1310, 'Crude Petroleum & Natural Gas', 3, 2221 FROM dual
UNION ALL SELECT 2910, 'SIC', 1310, 'Crude Petroleum & Natural Gas', 3, 2909 FROM dual
UNION ALL SELECT 2914, 'SIC', 1380, 'Oil & Gas Field Services', 3, 2909 FROM dual
UNION ALL SELECT 2912, 'SIC', 1320, 'Natural Gas Liquids', 3, 2909 FROM dual
UNION ALL SELECT 2931, 'SIC', 1470, 'Chemical & Fertilizer Minerals', 3, 2918 FROM dual
UNION ALL SELECT 2935, 'SIC', 1480, 'Nonmetallic Minerals Services', 3, 2918 FROM dual
UNION ALL SELECT 2937, 'SIC', 1490, 'Miscellaneous Nonmetallic Minerals', 3, 2918 FROM dual
UNION ALL SELECT 2919, 'SIC', 1410, 'Dimension Stone', 3, 2918 FROM dual
UNION ALL SELECT 2921, 'SIC', 1420, 'Crushed & Broken Stone', 3, 2918 FROM dual
UNION ALL SELECT 2925, 'SIC', 1440, 'Sand & Gravel', 3, 2918 FROM dual
UNION ALL SELECT 2928, 'SIC', 1450, 'Clay, Ceramic & Refractory Minerals', 3, 2918 FROM dual
UNION ALL SELECT 2230, 'SEC', 1520, 'Residential Building Construction', 3, 2229 FROM dual
UNION ALL SELECT 2231, 'SEC', 1530, 'Operative Builders', 3, 2229 FROM dual
UNION ALL SELECT 2233, 'SEC', 1540, 'Nonresidential Building Construction', 3, 2229 FROM dual
UNION ALL SELECT 2943, 'SIC', 1530, 'Operative Builders', 3, 2939 FROM dual
UNION ALL SELECT 2940, 'SIC', 1520, 'Residential Building Construction', 3, 2939 FROM dual
UNION ALL SELECT 2945, 'SIC', 1540, 'Nonresidential Building Construction', 3, 2939 FROM dual
UNION ALL SELECT 2235, 'SEC', 1620, 'Heavy Construction, Except Highway', 3, 2234 FROM dual
UNION ALL SELECT 2951, 'SIC', 1620, 'Heavy Construction, Except Highway', 3, 2948 FROM dual
UNION ALL SELECT 2949, 'SIC', 1610, 'Highway & Street Construction', 3, 2948 FROM dual
UNION ALL SELECT 2238, 'SEC', 1730, 'Electrical Work', 3, 2237 FROM dual
UNION ALL SELECT 2973, 'SIC', 1780, 'Water Well Drilling', 3, 2955 FROM dual
UNION ALL SELECT 2960, 'SIC', 1730, 'Electrical Work', 3, 2955 FROM dual
UNION ALL SELECT 2966, 'SIC', 1750, 'Carpentry & Floor Work', 3, 2955 FROM dual
UNION ALL SELECT 2969, 'SIC', 1760, 'Roofing, Siding & Sheet Metal Work', 3, 2955 FROM dual
UNION ALL SELECT 2962, 'SIC', 1740, 'Masonry, Stonework & Plastering', 3, 2955 FROM dual
UNION ALL SELECT 2956, 'SIC', 1710, 'Plumbing, Heating, Air-Conditioning', 3, 2955 FROM dual
UNION ALL SELECT 2971, 'SIC', 1770, 'Concrete Work', 3, 2955 FROM dual
UNION ALL SELECT 2975, 'SIC', 1790, 'Miscellaneous Special Trade Contractors', 3, 2955 FROM dual
UNION ALL SELECT 2958, 'SIC', 1720, 'Painting & Paper Hanging', 3, 2955 FROM dual
UNION ALL SELECT 2257, 'SEC', 2090, 'Miscellaneous Food & Kindred Products', 3, 2240 FROM dual
UNION ALL SELECT 2241, 'SEC', 2010, 'Meat Products', 3, 2240 FROM dual
UNION ALL SELECT 2245, 'SEC', 2020, 'Dairy Products', 3, 2240 FROM dual
UNION ALL SELECT 2247, 'SEC', 2030, 'Preserved Fruits & Vegetables', 3, 2240 FROM dual
UNION ALL SELECT 2249, 'SEC', 2040, 'Grain Mill Products', 3, 2240 FROM dual
UNION ALL SELECT 2250, 'SEC', 2050, 'Bakery Products', 3, 2240 FROM dual
UNION ALL SELECT 2252, 'SEC', 2060, 'Sugar & Confectionery Products', 3, 2240 FROM dual
UNION ALL SELECT 2253, 'SEC', 2070, 'Fats & Oils', 3, 2240 FROM dual
UNION ALL SELECT 2254, 'SEC', 2080, 'Beverages', 3, 2240 FROM dual
UNION ALL SELECT 3026, 'SIC', 2080, 'Beverages', 3, 2982 FROM dual
UNION ALL SELECT 3008, 'SIC', 2050, 'Bakery Products', 3, 2982 FROM dual
UNION ALL SELECT 3020, 'SIC', 2070, 'Fats & Oils', 3, 2982 FROM dual
UNION ALL SELECT 2993, 'SIC', 2030, 'Preserved Fruits & Vegetables', 3, 2982 FROM dual
UNION ALL SELECT 2983, 'SIC', 2010, 'Meat Products', 3, 2982 FROM dual
UNION ALL SELECT 2987, 'SIC', 2020, 'Dairy Products', 3, 2982 FROM dual
UNION ALL SELECT 3012, 'SIC', 2060, 'Sugar & Confectionery Products', 3, 2982 FROM dual
UNION ALL SELECT 3033, 'SIC', 2090, 'Miscellaneous Food & Kindred Products', 3, 2982 FROM dual
UNION ALL SELECT 3000, 'SIC', 2040, 'Grain Mill Products', 3, 2982 FROM dual
UNION ALL SELECT 3048, 'SIC', 2140, 'Tobacco Stemming & Redrying', 3, 3041 FROM dual
UNION ALL SELECT 3046, 'SIC', 2130, 'Chewing & Smoking Tobacco', 3, 3041 FROM dual
UNION ALL SELECT 3044, 'SIC', 2120, 'Cigars', 3, 3041 FROM dual
UNION ALL SELECT 3042, 'SIC', 2110, 'Cigarettes', 3, 3041 FROM dual
UNION ALL SELECT 2265, 'SEC', 2220, 'Broadwoven Fabric Mills, Manmade', 3, 2262 FROM dual
UNION ALL SELECT 2267, 'SEC', 2250, 'Knitting Mills', 3, 2262 FROM dual
UNION ALL SELECT 2269, 'SEC', 2270, 'Carpets & Rugs', 3, 2262 FROM dual
UNION ALL SELECT 2263, 'SEC', 2210, 'Broadwoven Fabric Mills, Cotton', 3, 2262 FROM dual
UNION ALL SELECT 3051, 'SIC', 2210, 'Broadwoven Fabric Mills, Cotton', 3, 3050 FROM dual
UNION ALL SELECT 3059, 'SIC', 2250, 'Knitting Mills', 3, 3050 FROM dual
UNION ALL SELECT 3073, 'SIC', 2280, 'Yarn & Thread Mills', 3, 3050 FROM dual
UNION ALL SELECT 3071, 'SIC', 2270, 'Carpets & Rugs', 3, 3050 FROM dual
UNION ALL SELECT 3053, 'SIC', 2220, 'Broadwoven Fabric Mills, Manmade', 3, 3050 FROM dual
UNION ALL SELECT 3057, 'SIC', 2240, 'Narrow Fabric Mills', 3, 3050 FROM dual
UNION ALL SELECT 3077, 'SIC', 2290, 'Miscellaneous Textile Goods', 3, 3050 FROM dual
UNION ALL SELECT 3067, 'SIC', 2260, 'Textile Finishing, Except Wool', 3, 3050 FROM dual
UNION ALL SELECT 3055, 'SIC', 2230, 'Broadwoven Fabric Mills, Wool', 3, 3050 FROM dual
UNION ALL SELECT 2272, 'SEC', 2320, 'Men''s & Boys'' Furnishings', 3, 2271 FROM dual
UNION ALL SELECT 2275, 'SEC', 2390, 'Miscellaneous Fabricated Textile Products', 3, 2271 FROM dual
UNION ALL SELECT 2274, 'SEC', 2340, 'Women''s & Children''s Undergarments', 3, 2271 FROM dual
UNION ALL SELECT 2273, 'SEC', 2330, 'Women''s & Misses'' Outerwear', 3, 2271 FROM dual
UNION ALL SELECT 3115, 'SIC', 2390, 'Miscellaneous Fabricated Textile Products', 3, 3083 FROM dual
UNION ALL SELECT 3098, 'SIC', 2340, 'Women''s & Children''s Undergarments', 3, 3083 FROM dual
UNION ALL SELECT 3084, 'SIC', 2310, 'Men''s & Boys'' Suits & Coats', 3, 3083 FROM dual
UNION ALL SELECT 3086, 'SIC', 2320, 'Men''s & Boys'' Furnishings', 3, 3083 FROM dual
UNION ALL SELECT 3093, 'SIC', 2330, 'Women''s & Misses'' Outerwear', 3, 3083 FROM dual
UNION ALL SELECT 3101, 'SIC', 2350, 'Hats, Caps & Millinery', 3, 3083 FROM dual
UNION ALL SELECT 3103, 'SIC', 2360, 'Girl''s & Children''s Outerwear', 3, 3083 FROM dual
UNION ALL SELECT 3108, 'SIC', 2380, 'Miscellaneous Apparel & Accessories', 3, 3083 FROM dual
UNION ALL SELECT 2280, 'SEC', 2450, 'Wood Buildings & Mobile Homes', 3, 2276 FROM dual
UNION ALL SELECT 2277, 'SEC', 2420, 'Sawmills & Planing Mills', 3, 2276 FROM dual
UNION ALL SELECT 2279, 'SEC', 2430, 'Millwork, Plywood, & Structural Members', 3, 2276 FROM dual
UNION ALL SELECT 3144, 'SIC', 2490, 'Miscellaneous Wood Products', 3, 3124 FROM dual
UNION ALL SELECT 3141, 'SIC', 2450, 'Wood Buildings & Mobile Homes', 3, 3124 FROM dual
UNION ALL SELECT 3137, 'SIC', 2440, 'Wood Containers', 3, 3124 FROM dual
UNION ALL SELECT 3125, 'SIC', 2410, 'Logging', 3, 3124 FROM dual
UNION ALL SELECT 3131, 'SIC', 2430, 'Millwork, Plywood, & Structural Members', 3, 3124 FROM dual
UNION ALL SELECT 2291, 'SEC', 2590, 'Miscellaneous Furniture & Fixtures', 3, 2283 FROM dual
UNION ALL SELECT 2288, 'SEC', 2530, 'Public Building & Related Furniture', 3, 2283 FROM dual
UNION ALL SELECT 2286, 'SEC', 2520, 'Office Furniture', 3, 2283 FROM dual
UNION ALL SELECT 2284, 'SEC', 2510, 'Household Furniture', 3, 2283 FROM dual
UNION ALL SELECT 2290, 'SEC', 2540, 'Partitions & Fixtures', 3, 2283 FROM dual
UNION ALL SELECT 3164, 'SIC', 2590, 'Miscellaneous Furniture & Fixtures', 3, 3148 FROM dual
UNION ALL SELECT 3156, 'SIC', 2520, 'Office Furniture', 3, 3148 FROM dual
UNION ALL SELECT 3149, 'SIC', 2510, 'Household Furniture', 3, 3148 FROM dual
UNION ALL SELECT 3159, 'SIC', 2530, 'Public Building & Related Furniture', 3, 3148 FROM dual
UNION ALL SELECT 3161, 'SIC', 2540, 'Partitions & Fixtures', 3, 3148 FROM dual
UNION ALL SELECT 2297, 'SEC', 2630, 'Paperboard Mills', 3, 2292 FROM dual
UNION ALL SELECT 2299, 'SEC', 2650, 'Paperboard Containers & Boxes', 3, 2292 FROM dual
UNION ALL SELECT 2300, 'SEC', 2670, 'Miscellaneous Converted Paper Products', 3, 2292 FROM dual
UNION ALL SELECT 2293, 'SEC', 2610, 'Pulp Mills', 3, 2292 FROM dual
UNION ALL SELECT 2295, 'SEC', 2620, 'Paper Mills', 3, 2292 FROM dual
UNION ALL SELECT 3180, 'SIC', 2670, 'Miscellaneous Converted Paper Products', 3, 3167 FROM dual
UNION ALL SELECT 3168, 'SIC', 2610, 'Pulp Mills', 3, 3167 FROM dual
UNION ALL SELECT 3174, 'SIC', 2650, 'Paperboard Containers & Boxes', 3, 3167 FROM dual
UNION ALL SELECT 3172, 'SIC', 2630, 'Paperboard Mills', 3, 3167 FROM dual
UNION ALL SELECT 3170, 'SIC', 2620, 'Paper Mills', 3, 3167 FROM dual
UNION ALL SELECT 2317, 'SEC', 2780, 'Blankbooks & Bookbinding', 3, 2302 FROM dual
UNION ALL SELECT 2303, 'SEC', 2710, 'Newspapers', 3, 2302 FROM dual
UNION ALL SELECT 2305, 'SEC', 2720, 'Periodicals', 3, 2302 FROM dual
UNION ALL SELECT 2310, 'SEC', 2740, 'Miscellaneous Publishing', 3, 2302 FROM dual
UNION ALL SELECT 2312, 'SEC', 2750, 'Commercial Printing', 3, 2302 FROM dual
UNION ALL SELECT 2307, 'SEC', 2730, 'Books', 3, 2302 FROM dual
UNION ALL SELECT 2313, 'SEC', 2760, 'Manifold Business Forms', 3, 2302 FROM dual
UNION ALL SELECT 2318, 'SEC', 2790, 'Printing Trade Services', 3, 2302 FROM dual
UNION ALL SELECT 3206, 'SIC', 2770, 'Greeting Cards', 3, 3190 FROM dual
UNION ALL SELECT 3211, 'SIC', 2790, 'Printing Trade Services', 3, 3190 FROM dual
UNION ALL SELECT 3195, 'SIC', 2730, 'Books', 3, 3190 FROM dual
UNION ALL SELECT 3208, 'SIC', 2780, 'Blankbooks & Bookbinding', 3, 3190 FROM dual
UNION ALL SELECT 3191, 'SIC', 2710, 'Newspapers', 3, 3190 FROM dual
UNION ALL SELECT 3193, 'SIC', 2720, 'Periodicals', 3, 3190 FROM dual
UNION ALL SELECT 3198, 'SIC', 2740, 'Miscellaneous Publishing', 3, 3190 FROM dual
UNION ALL SELECT 3200, 'SIC', 2750, 'Commercial Printing', 3, 3190 FROM dual
UNION ALL SELECT 3204, 'SIC', 2760, 'Manifold Business Forms', 3, 3190 FROM dual
UNION ALL SELECT 2321, 'SEC', 2820, 'Plastics Materials & Synthetics', 3, 2319 FROM dual
UNION ALL SELECT 2320, 'SEC', 2810, 'Industrial Inorganic Chemicals', 3, 2319 FROM dual
UNION ALL SELECT 2328, 'SEC', 2840, 'Soap, Cleaners & Toilet Goods', 3, 2319 FROM dual
UNION ALL SELECT 2333, 'SEC', 2860, 'Industrial Organic Chemicals', 3, 2319 FROM dual
UNION ALL SELECT 2335, 'SEC', 2890, 'Miscellaneous Chemical Products', 3, 2319 FROM dual
UNION ALL SELECT 2331, 'SEC', 2850, 'Paints & Allied Products', 3, 2319 FROM dual
UNION ALL SELECT 2334, 'SEC', 2870, 'Agricultural Chemicals', 3, 2319 FROM dual
UNION ALL SELECT 2323, 'SEC', 2830, 'Drugs', 3, 2319 FROM dual
UNION ALL SELECT 3241, 'SIC', 2870, 'Agricultural Chemicals', 3, 3214 FROM dual
UNION ALL SELECT 3235, 'SIC', 2850, 'Paints & Allied Products', 3, 3214 FROM dual
UNION ALL SELECT 3230, 'SIC', 2840, 'Soap, Cleaners & Toilet Goods', 3, 3214 FROM dual
UNION ALL SELECT 3225, 'SIC', 2830, 'Drugs', 3, 3214 FROM dual
UNION ALL SELECT 3215, 'SIC', 2810, 'Industrial Inorganic Chemicals', 3, 3214 FROM dual
UNION ALL SELECT 3246, 'SIC', 2890, 'Miscellaneous Chemical Products', 3, 3214 FROM dual
UNION ALL SELECT 3220, 'SIC', 2820, 'Plastics Materials & Synthetics', 3, 3214 FROM dual
UNION ALL SELECT 3237, 'SIC', 2860, 'Industrial Organic Chemicals', 3, 3214 FROM dual
UNION ALL SELECT 2338, 'SEC', 2910, 'Petroleum Refining', 3, 2337 FROM dual
UNION ALL SELECT 2341, 'SEC', 2990, 'Miscellaneous Petroleum & Coal Products', 3, 2337 FROM dual
UNION ALL SELECT 2340, 'SEC', 2950, 'Asphalt Paving & Roofing Materials', 3, 2337 FROM dual
UNION ALL SELECT 3255, 'SIC', 2950, 'Asphalt Paving & Roofing Materials', 3, 3252 FROM dual
UNION ALL SELECT 3258, 'SIC', 2990, 'Miscellaneous Petroleum & Coal Products', 3, 3252 FROM dual
UNION ALL SELECT 3253, 'SIC', 2910, 'Petroleum Refining', 3, 3252 FROM dual
UNION ALL SELECT 2343, 'SEC', 3010, 'Tires & Inner Tubes', 3, 2342 FROM dual
UNION ALL SELECT 2349, 'SEC', 3080, 'Miscellaneous Plastics Products, nec', 3, 2342 FROM dual
UNION ALL SELECT 2347, 'SEC', 3050, 'Hose & Belting & Gaskets & Packing', 3, 2342 FROM dual
UNION ALL SELECT 2345, 'SEC', 3020, 'Rubber & Plastics Footwear', 3, 2342 FROM dual
UNION ALL SELECT 2348, 'SEC', 3060, 'Fabricated Rubber Products, nec', 3, 2342 FROM dual
UNION ALL SELECT 3269, 'SIC', 3060, 'Fabricated Rubber Products, nec', 3, 3261 FROM dual
UNION ALL SELECT 3266, 'SIC', 3050, 'Hose & Belting & Gaskets & Packing', 3, 3261 FROM dual
UNION ALL SELECT 3262, 'SIC', 3010, 'Tires & Inner Tubes', 3, 3261 FROM dual
UNION ALL SELECT 3272, 'SIC', 3080, 'Miscellaneous Plastics Products, nec', 3, 3261 FROM dual
UNION ALL SELECT 3264, 'SIC', 3020, 'Rubber & Plastics Footwear', 3, 3261 FROM dual
UNION ALL SELECT 2354, 'SEC', 3140, 'Footwear, Except Rubber', 3, 2353 FROM dual
UNION ALL SELECT 3292, 'SIC', 3150, 'Leather Gloves & Mittens', 3, 3282 FROM dual
UNION ALL SELECT 3299, 'SIC', 3190, 'Leather Goods, nec', 3, 3282 FROM dual
UNION ALL SELECT 3294, 'SIC', 3160, 'Luggage', 3, 3282 FROM dual
UNION ALL SELECT 3287, 'SIC', 3140, 'Footwear, Except Rubber', 3, 3282 FROM dual
UNION ALL SELECT 3285, 'SIC', 3130, 'Footwear Cut Stock', 3, 3282 FROM dual
UNION ALL SELECT 3283, 'SIC', 3110, 'Leather Tanning & Finishing', 3, 3282 FROM dual
UNION ALL SELECT 3296, 'SIC', 3170, 'Handbags & Personal Leather Goods', 3, 3282 FROM dual
UNION ALL SELECT 2370, 'SEC', 3290, 'Miscellaneous Nonmetallic Mineral Products', 3, 2355 FROM dual
UNION ALL SELECT 2364, 'SEC', 3250, 'Structural Clay Products', 3, 2355 FROM dual
UNION ALL SELECT 2356, 'SEC', 3210, 'Flat Glass', 3, 2355 FROM dual
UNION ALL SELECT 2358, 'SEC', 3220, 'Glass & Glassware, Pressed or Blown', 3, 2355 FROM dual
UNION ALL SELECT 2360, 'SEC', 3230, 'Products of Purchased Glass', 3, 2355 FROM dual
UNION ALL SELECT 2362, 'SEC', 3240, 'Cement, Hydraulic', 3, 2355 FROM dual
UNION ALL SELECT 2365, 'SEC', 3260, 'Pottery & Related Products', 3, 2355 FROM dual
UNION ALL SELECT 2366, 'SEC', 3270, 'Concrete, Gypsum & Plaster Products', 3, 2355 FROM dual
UNION ALL SELECT 2368, 'SEC', 3280, 'Cut Stone & Stone Products', 3, 2355 FROM dual
UNION ALL SELECT 3311, 'SIC', 3250, 'Structural Clay Products', 3, 3301 FROM dual
UNION ALL SELECT 3309, 'SIC', 3240, 'Cement, Hydraulic', 3, 3301 FROM dual
UNION ALL SELECT 3307, 'SIC', 3230, 'Products of Purchased Glass', 3, 3301 FROM dual
UNION ALL SELECT 3302, 'SIC', 3210, 'Flat Glass', 3, 3301 FROM dual
UNION ALL SELECT 3304, 'SIC', 3220, 'Glass & Glassware, Pressed or Blown', 3, 3301 FROM dual
UNION ALL SELECT 3322, 'SIC', 3270, 'Concrete, Gypsum & Plaster Products', 3, 3301 FROM dual
UNION ALL SELECT 3328, 'SIC', 3280, 'Cut Stone & Stone Products', 3, 3301 FROM dual
UNION ALL SELECT 3330, 'SIC', 3290, 'Miscellaneous Nonmetallic Mineral Products', 3, 3301 FROM dual
UNION ALL SELECT 3316, 'SIC', 3260, 'Pottery & Related Products', 3, 3301 FROM dual
UNION ALL SELECT 2382, 'SEC', 3360, 'Nonferrous Foundries (Castings)', 3, 2371 FROM dual
UNION ALL SELECT 2376, 'SEC', 3330, 'Primary Nonferrous Metals', 3, 2371 FROM dual
UNION ALL SELECT 2375, 'SEC', 3320, 'Iron & Steel Foundries', 3, 2371 FROM dual
UNION ALL SELECT 2372, 'SEC', 3310, 'Blast Furnace & Basic Steel Products', 3, 2371 FROM dual
UNION ALL SELECT 2380, 'SEC', 3350, 'Nonferrous Rolling & Drawing', 3, 2371 FROM dual
UNION ALL SELECT 2383, 'SEC', 3390, 'Miscellaneous Primary Metal Industries', 3, 2371 FROM dual
UNION ALL SELECT 2378, 'SEC', 3340, 'Secondary Nonferrous Metals', 3, 2371 FROM dual
UNION ALL SELECT 3368, 'SIC', 3390, 'Miscellaneous Primary Metal Industries', 3, 3337 FROM dual
UNION ALL SELECT 3362, 'SIC', 3360, 'Nonferrous Foundries (Castings)', 3, 3337 FROM dual
UNION ALL SELECT 3355, 'SIC', 3350, 'Nonferrous Rolling & Drawing', 3, 3337 FROM dual
UNION ALL SELECT 3353, 'SIC', 3340, 'Secondary Nonferrous Metals', 3, 3337 FROM dual
UNION ALL SELECT 3349, 'SIC', 3330, 'Primary Nonferrous Metals', 3, 3337 FROM dual
UNION ALL SELECT 3344, 'SIC', 3320, 'Iron & Steel Foundries', 3, 3337 FROM dual
UNION ALL SELECT 3338, 'SIC', 3310, 'Blast Furnace & Basic Steel Products', 3, 3337 FROM dual
UNION ALL SELECT 2402, 'SEC', 3490, 'Miscellaneous Fabricated Metal Products', 3, 2384 FROM dual
UNION ALL SELECT 2399, 'SEC', 3460, 'Metal Forgings & Stampings', 3, 2384 FROM dual
UNION ALL SELECT 2396, 'SEC', 3450, 'Screw Machine Products, Bolts, Etc.', 3, 2384 FROM dual
UNION ALL SELECT 2391, 'SEC', 3440, 'Fabricated Structural Metal Products', 3, 2384 FROM dual
UNION ALL SELECT 2389, 'SEC', 3430, 'Plumbing & Heating, Except Electric', 3, 2384 FROM dual
UNION ALL SELECT 2388, 'SEC', 3420, 'Cutlery, Handtools & Hardware', 3, 2384 FROM dual
UNION ALL SELECT 2401, 'SEC', 3480, 'Ordnance & Accessories, nec', 3, 2384 FROM dual
UNION ALL SELECT 2385, 'SEC', 3410, 'Metal Cans & Shipping Containers', 3, 2384 FROM dual
UNION ALL SELECT 3380, 'SIC', 3430, 'Plumbing & Heating, Except Electric', 3, 3371 FROM dual
UNION ALL SELECT 3395, 'SIC', 3460, 'Metal Forgings & Stampings', 3, 3371 FROM dual
UNION ALL SELECT 3392, 'SIC', 3450, 'Screw Machine Products, Bolts, Etc.', 3, 3371 FROM dual
UNION ALL SELECT 3372, 'SIC', 3410, 'Metal Cans & Shipping Containers', 3, 3371 FROM dual
UNION ALL SELECT 3375, 'SIC', 3420, 'Cutlery, Handtools & Hardware', 3, 3371 FROM dual
UNION ALL SELECT 3404, 'SIC', 3480, 'Ordnance & Accessories, nec', 3, 3371 FROM dual
UNION ALL SELECT 3401, 'SIC', 3470, 'Metal Services, nec', 3, 3371 FROM dual
UNION ALL SELECT 3384, 'SIC', 3440, 'Fabricated Structural Metal Products', 3, 3371 FROM dual
UNION ALL SELECT 3409, 'SIC', 3490, 'Miscellaneous Fabricated Metal Products', 3, 3371 FROM dual
UNION ALL SELECT 2408, 'SEC', 3530, 'Construction & Related Machinery', 3, 2403 FROM dual
UNION ALL SELECT 2405, 'SEC', 3520, 'Farm & Garden Machinery', 3, 2403 FROM dual
UNION ALL SELECT 2404, 'SEC', 3510, 'Engines & Turbines', 3, 2403 FROM dual
UNION ALL SELECT 2418, 'SEC', 3560, 'General Industry Machinery', 3, 2403 FROM dual
UNION ALL SELECT 2415, 'SEC', 3550, 'Special Industry Machinery', 3, 2403 FROM dual
UNION ALL SELECT 2413, 'SEC', 3540, 'Metalworking Machinery', 3, 2403 FROM dual
UNION ALL SELECT 2424, 'SEC', 3570, 'Computer & Office Equipment', 3, 2403 FROM dual
UNION ALL SELECT 2431, 'SEC', 3580, 'Refrigeration & Service Industry', 3, 2403 FROM dual
UNION ALL SELECT 2433, 'SEC', 3590, 'Industrial Machinery, nec', 3, 2403 FROM dual
UNION ALL SELECT 3434, 'SIC', 3540, 'Metalworking Machinery', 3, 3419 FROM dual
UNION ALL SELECT 3426, 'SIC', 3530, 'Construction & Related Machinery', 3, 3419 FROM dual
UNION ALL SELECT 3474, 'SIC', 3590, 'Industrial Machinery, nec', 3, 3419 FROM dual
UNION ALL SELECT 3468, 'SIC', 3580, 'Refrigeration & Service Industry', 3, 3419 FROM dual
UNION ALL SELECT 3461, 'SIC', 3570, 'Computer & Office Equipment', 3, 3419 FROM dual
UNION ALL SELECT 3451, 'SIC', 3560, 'General Industry Machinery', 3, 3419 FROM dual
UNION ALL SELECT 3444, 'SIC', 3550, 'Special Industry Machinery', 3, 3419 FROM dual
UNION ALL SELECT 3423, 'SIC', 3520, 'Farm & Garden Machinery', 3, 3419 FROM dual
UNION ALL SELECT 2442, 'SEC', 3640, 'Electric Lighting & Wiring Equipment', 3, 2434 FROM dual
UNION ALL SELECT 2443, 'SEC', 3650, 'Household Audio & Video Equipment', 3, 2434 FROM dual
UNION ALL SELECT 2446, 'SEC', 3660, 'Communications Equipment', 3, 2434 FROM dual
UNION ALL SELECT 2440, 'SEC', 3630, 'Household Appliances', 3, 2434 FROM dual
UNION ALL SELECT 2435, 'SEC', 3610, 'Electric Distribution Equipment', 3, 2434 FROM dual
UNION ALL SELECT 2450, 'SEC', 3670, 'Electronic Components & Accessories', 3, 2434 FROM dual
UNION ALL SELECT 2438, 'SEC', 3620, 'Electrical Industrial Apparatus', 3, 2434 FROM dual
UNION ALL SELECT 2456, 'SEC', 3690, 'Miscellaneous Electrical Equipment & Supplies', 3, 2434 FROM dual
UNION ALL SELECT 3484, 'SIC', 3620, 'Electrical Industrial Apparatus', 3, 3480 FROM dual
UNION ALL SELECT 3507, 'SIC', 3660, 'Communications Equipment', 3, 3480 FROM dual
UNION ALL SELECT 3504, 'SIC', 3650, 'Household Audio & Video Equipment', 3, 3480 FROM dual
UNION ALL SELECT 3520, 'SIC', 3690, 'Miscellaneous Electrical Equipment & Supplies', 3, 3480 FROM dual
UNION ALL SELECT 3496, 'SIC', 3640, 'Electric Lighting & Wiring Equipment', 3, 3480 FROM dual
UNION ALL SELECT 3489, 'SIC', 3630, 'Household Appliances', 3, 3480 FROM dual
UNION ALL SELECT 3511, 'SIC', 3670, 'Electronic Components & Accessories', 3, 3480 FROM dual
UNION ALL SELECT 3481, 'SIC', 3610, 'Electric Distribution Equipment', 3, 3480 FROM dual
UNION ALL SELECT 2474, 'SEC', 3760, 'Guided Missiles, Space Vehicles, Parts', 3, 2458 FROM dual
UNION ALL SELECT 2475, 'SEC', 3790, 'Miscellaneous Transportation Equipment', 3, 2458 FROM dual
UNION ALL SELECT 2459, 'SEC', 3710, 'Motor Vehicles & Equipment', 3, 2458 FROM dual
UNION ALL SELECT 2465, 'SEC', 3720, 'Aircraft & Parts', 3, 2458 FROM dual
UNION ALL SELECT 2469, 'SEC', 3730, 'Ship & Boat Building & Repairing', 3, 2458 FROM dual
UNION ALL SELECT 2470, 'SEC', 3740, 'Railroad Equipment', 3, 2458 FROM dual
UNION ALL SELECT 2472, 'SEC', 3750, 'Motorcycles, Bicycles & Parts', 3, 2458 FROM dual
UNION ALL SELECT 3544, 'SIC', 3760, 'Guided Missiles, Space Vehicles, Parts', 3, 3526 FROM dual
UNION ALL SELECT 3537, 'SIC', 3730, 'Ship & Boat Building & Repairing', 3, 3526 FROM dual
UNION ALL SELECT 3533, 'SIC', 3720, 'Aircraft & Parts', 3, 3526 FROM dual
UNION ALL SELECT 3548, 'SIC', 3790, 'Miscellaneous Transportation Equipment', 3, 3526 FROM dual
UNION ALL SELECT 3527, 'SIC', 3710, 'Motor Vehicles & Equipment', 3, 3526 FROM dual
UNION ALL SELECT 3540, 'SIC', 3740, 'Railroad Equipment', 3, 3526 FROM dual
UNION ALL SELECT 3542, 'SIC', 3750, 'Motorcycles, Bicycles & Parts', 3, 3526 FROM dual
UNION ALL SELECT 2479, 'SEC', 3820, 'Measuring & Controlling Devices', 3, 2476 FROM dual
UNION ALL SELECT 2488, 'SEC', 3840, 'Medical Instruments & Supplies', 3, 2476 FROM dual
UNION ALL SELECT 2477, 'SEC', 3810, 'Search & Navigation Equipment', 3, 2476 FROM dual
UNION ALL SELECT 2498, 'SEC', 3870, 'Watches, Clocks, Watchcases & Parts', 3, 2476 FROM dual
UNION ALL SELECT 2494, 'SEC', 3850, 'Ophthalmic Goods', 3, 2476 FROM dual
UNION ALL SELECT 2496, 'SEC', 3860, 'Photographic Equipment & Supplies', 3, 2476 FROM dual
UNION ALL SELECT 3570, 'SIC', 3850, 'Ophthalmic Goods', 3, 3552 FROM dual
UNION ALL SELECT 3564, 'SIC', 3840, 'Medical Instruments & Supplies', 3, 3552 FROM dual
UNION ALL SELECT 3555, 'SIC', 3820, 'Measuring & Controlling Devices', 3, 3552 FROM dual
UNION ALL SELECT 3553, 'SIC', 3810, 'Search & Navigation Equipment', 3, 3552 FROM dual
UNION ALL SELECT 3572, 'SIC', 3860, 'Photographic Equipment & Supplies', 3, 3552 FROM dual
UNION ALL SELECT 3574, 'SIC', 3870, 'Watches, Clocks, Watchcases & Parts', 3, 3552 FROM dual
UNION ALL SELECT 2505, 'SEC', 3940, 'Toys & Sporting Goods', 3, 2500 FROM dual
UNION ALL SELECT 2503, 'SEC', 3930, 'Musical Instruments', 3, 2500 FROM dual
UNION ALL SELECT 2510, 'SEC', 3960, 'Costume Jewelry & Notions', 3, 2500 FROM dual;

INSERT INTO "industry" (industry_id, industry_classification, industry_code, industry_description, depth, parent_id) 
          SELECT 2511, 'SEC', 3990, 'Miscellaneous Manufacturers', 3, 2500 FROM dual
UNION ALL SELECT 2501, 'SEC', 3910, 'Jewelry, Silverware & Plated Ware', 3, 2500 FROM dual
UNION ALL SELECT 2509, 'SEC', 3950, 'Pens, Pencils, Office & Art Supplies', 3, 2500 FROM dual
UNION ALL SELECT 3595, 'SIC', 3990, 'Miscellaneous Manufacturers', 3, 3576 FROM dual
UNION ALL SELECT 3581, 'SIC', 3930, 'Musical Instruments', 3, 3576 FROM dual
UNION ALL SELECT 3577, 'SIC', 3910, 'Jewelry, Silverware & Plated Ware', 3, 3576 FROM dual
UNION ALL SELECT 3583, 'SIC', 3940, 'Toys & Sporting Goods', 3, 3576 FROM dual
UNION ALL SELECT 3587, 'SIC', 3950, 'Pens, Pencils, Office & Art Supplies', 3, 3576 FROM dual
UNION ALL SELECT 3592, 'SIC', 3960, 'Costume Jewelry & Notions', 3, 3576 FROM dual
UNION ALL SELECT 3602, 'SIC', 4010, 'Railroads', 3, 3601 FROM dual
UNION ALL SELECT 3609, 'SIC', 4120, 'Taxicabs', 3, 3605 FROM dual
UNION ALL SELECT 3606, 'SIC', 4110, 'Local & Suburban Transportation', 3, 3605 FROM dual
UNION ALL SELECT 3613, 'SIC', 4140, 'Bus Charter Service', 3, 3605 FROM dual
UNION ALL SELECT 3618, 'SIC', 4170, 'Bus Terminal & Service Facilities', 3, 3605 FROM dual
UNION ALL SELECT 3616, 'SIC', 4150, 'School Buses', 3, 3605 FROM dual
UNION ALL SELECT 3611, 'SIC', 4130, 'Intercity & Rural Bus Transportation', 3, 3605 FROM dual
UNION ALL SELECT 2521, 'SEC', 4230, 'Trucking Terminal Facilities', 3, 2517 FROM dual
UNION ALL SELECT 2520, 'SEC', 4220, 'Public Warehousing & Storage', 3, 2517 FROM dual
UNION ALL SELECT 2518, 'SEC', 4210, 'Trucking & Courier Services, Except Air', 3, 2517 FROM dual
UNION ALL SELECT 3626, 'SIC', 4220, 'Public Warehousing & Storage', 3, 3620 FROM dual
UNION ALL SELECT 3631, 'SIC', 4230, 'Trucking Terminal Facilities', 3, 3620 FROM dual
UNION ALL SELECT 3621, 'SIC', 4210, 'Trucking & Courier Services, Except Air', 3, 3620 FROM dual
UNION ALL SELECT 3634, 'SIC', 4310, 'US Postal Service', 3, 3633 FROM dual
UNION ALL SELECT 2524, 'SEC', 4410, 'Deep Sea Foreign Transport of Freight', 3, 2523 FROM dual
UNION ALL SELECT 3637, 'SIC', 4410, 'Deep Sea Foreign Transport of Freight', 3, 3636 FROM dual
UNION ALL SELECT 3639, 'SIC', 4420, 'Deep Sea Domestic Transport of Freight', 3, 3636 FROM dual
UNION ALL SELECT 3641, 'SIC', 4430, 'Freight Transport on The Great Lakes', 3, 3636 FROM dual
UNION ALL SELECT 3645, 'SIC', 4480, 'Water Transportation of Passengers', 3, 3636 FROM dual
UNION ALL SELECT 3649, 'SIC', 4490, 'Water Transportation Services', 3, 3636 FROM dual
UNION ALL SELECT 3643, 'SIC', 4440, 'Water Transportation of Freight, nec', 3, 3636 FROM dual
UNION ALL SELECT 2527, 'SEC', 4510, 'Air Transportation, Scheduled', 3, 2526 FROM dual
UNION ALL SELECT 2530, 'SEC', 4520, 'Air Transportation, Nonscheduled', 3, 2526 FROM dual
UNION ALL SELECT 2532, 'SEC', 4580, 'Airports, Flying Fields & Services', 3, 2526 FROM dual
UNION ALL SELECT 3660, 'SIC', 4580, 'Airports, Flying Fields & Services', 3, 3654 FROM dual
UNION ALL SELECT 3655, 'SIC', 4510, 'Air Transportation, Scheduled', 3, 3654 FROM dual
UNION ALL SELECT 3658, 'SIC', 4520, 'Air Transportation, Nonscheduled', 3, 3654 FROM dual
UNION ALL SELECT 2535, 'SEC', 4610, 'Pipelines, Except Natural Gas', 3, 2534 FROM dual
UNION ALL SELECT 3663, 'SIC', 4610, 'Pipelines, Except Natural Gas', 3, 3662 FROM dual
UNION ALL SELECT 2537, 'SEC', 4730, 'Freight Transportation Arrangement', 3, 2536 FROM dual
UNION ALL SELECT 3668, 'SIC', 4720, 'Passenger Transportation Arrangement', 3, 3667 FROM dual
UNION ALL SELECT 3674, 'SIC', 4740, 'Rental of Railroad Cars', 3, 3667 FROM dual
UNION ALL SELECT 3676, 'SIC', 4780, 'Miscellaneous Transportation Services', 3, 3667 FROM dual
UNION ALL SELECT 3672, 'SIC', 4730, 'Freight Transportation Arrangement', 3, 3667 FROM dual
UNION ALL SELECT 2548, 'SEC', 4840, 'Cable & Other Pay TV Services', 3, 2539 FROM dual
UNION ALL SELECT 2540, 'SEC', 4810, 'Telephone Communications', 3, 2539 FROM dual
UNION ALL SELECT 2550, 'SEC', 4890, 'Communications Services, nec', 3, 2539 FROM dual
UNION ALL SELECT 2543, 'SEC', 4820, 'Telegraph & Other Communications', 3, 2539 FROM dual
UNION ALL SELECT 2545, 'SEC', 4830, 'Radio & Television Broadcasting', 3, 2539 FROM dual
UNION ALL SELECT 3686, 'SIC', 4830, 'Radio & Television Broadcasting', 3, 3680 FROM dual
UNION ALL SELECT 3689, 'SIC', 4840, 'Cable & Other Pay TV Services', 3, 3680 FROM dual
UNION ALL SELECT 3681, 'SIC', 4810, 'Telephone Communications', 3, 3680 FROM dual
UNION ALL SELECT 3691, 'SIC', 4890, 'Communications Services, nec', 3, 3680 FROM dual
UNION ALL SELECT 3684, 'SIC', 4820, 'Telegraph & Other Communications', 3, 3680 FROM dual
UNION ALL SELECT 2566, 'SEC', 4960, 'Steam & Air Conditioning Supply', 3, 2552 FROM dual
UNION ALL SELECT 2564, 'SEC', 4950, 'Sanitary Services', 3, 2552 FROM dual
UNION ALL SELECT 2562, 'SEC', 4940, 'Water Supply', 3, 2552 FROM dual
UNION ALL SELECT 2559, 'SEC', 4930, 'Combination Utility Services', 3, 2552 FROM dual
UNION ALL SELECT 2555, 'SEC', 4920, 'Gas Production & Distribution', 3, 2552 FROM dual
UNION ALL SELECT 2553, 'SEC', 4910, 'Electric Services', 3, 2552 FROM dual
UNION ALL SELECT 3701, 'SIC', 4930, 'Combination Utility Services', 3, 3693 FROM dual
UNION ALL SELECT 3707, 'SIC', 4950, 'Sanitary Services', 3, 3693 FROM dual
UNION ALL SELECT 3711, 'SIC', 4960, 'Steam & Air Conditioning Supply', 3, 3693 FROM dual
UNION ALL SELECT 3696, 'SIC', 4920, 'Gas Production & Distribution', 3, 3693 FROM dual
UNION ALL SELECT 3694, 'SIC', 4910, 'Electric Services', 3, 3693 FROM dual
UNION ALL SELECT 3713, 'SIC', 4970, 'Irrigation Systems', 3, 3693 FROM dual
UNION ALL SELECT 3705, 'SIC', 4940, 'Water Supply', 3, 3693 FROM dual
UNION ALL SELECT 2583, 'SEC', 5070, 'Hardware, Plumbing & Heating Equipment', 3, 2568 FROM dual
UNION ALL SELECT 2588, 'SEC', 5090, 'Miscellaneous Durable Goods', 3, 2568 FROM dual
UNION ALL SELECT 2569, 'SEC', 5010, 'Motor Vehicles, Parts & Supplies', 3, 2568 FROM dual
UNION ALL SELECT 2579, 'SEC', 5060, 'Electrical Goods', 3, 2568 FROM dual
UNION ALL SELECT 2572, 'SEC', 5030, 'Lumber & Construction Materials', 3, 2568 FROM dual
UNION ALL SELECT 2571, 'SEC', 5020, 'Furniture & Home Furnishings', 3, 2568 FROM dual
UNION ALL SELECT 2577, 'SEC', 5050, 'Metals & Minerals, Except Petroleum', 3, 2568 FROM dual
UNION ALL SELECT 2574, 'SEC', 5040, 'Professional & Commercial Equipment', 3, 2568 FROM dual
UNION ALL SELECT 2585, 'SEC', 5080, 'Machinery Equipment & Supplies', 3, 2568 FROM dual
UNION ALL SELECT 3737, 'SIC', 5050, 'Metals & Minerals, Except Petroleum', 3, 3715 FROM dual
UNION ALL SELECT 3729, 'SIC', 5040, 'Professional & Commercial Equipment', 3, 3715 FROM dual
UNION ALL SELECT 3749, 'SIC', 5080, 'Machinery Equipment & Supplies', 3, 3715 FROM dual
UNION ALL SELECT 3740, 'SIC', 5060, 'Electrical Goods', 3, 3715 FROM dual
UNION ALL SELECT 3721, 'SIC', 5020, 'Furniture & Home Furnishings', 3, 3715 FROM dual
UNION ALL SELECT 3716, 'SIC', 5010, 'Motor Vehicles, Parts & Supplies', 3, 3715 FROM dual
UNION ALL SELECT 3744, 'SIC', 5070, 'Hardware, Plumbing & Heating Equipment', 3, 3715 FROM dual
UNION ALL SELECT 3724, 'SIC', 5030, 'Lumber & Construction Materials', 3, 3715 FROM dual
UNION ALL SELECT 3756, 'SIC', 5090, 'Miscellaneous Durable Goods', 3, 3715 FROM dual
UNION ALL SELECT 2600, 'SEC', 5170, 'Petroleum & Petroleum Products', 3, 2591 FROM dual
UNION ALL SELECT 2598, 'SEC', 5150, 'Farm-Product Raw Materials', 3, 2591 FROM dual
UNION ALL SELECT 2592, 'SEC', 5110, 'Paper & Paper Products', 3, 2591 FROM dual
UNION ALL SELECT 2593, 'SEC', 5120, 'Drugs, Proprietaries & Sundries', 3, 2591 FROM dual
UNION ALL SELECT 2595, 'SEC', 5130, 'Apparel, Piece Goods & Notations', 3, 2591 FROM dual
UNION ALL SELECT 2596, 'SEC', 5140, 'Groceries & Related Products', 3, 2591 FROM dual
UNION ALL SELECT 2599, 'SEC', 5160, 'Chemicals & Allied Products', 3, 2591 FROM dual
UNION ALL SELECT 2604, 'SEC', 5190, 'Miscellaneous Nondurable Goods', 3, 2591 FROM dual
UNION ALL SELECT 2603, 'SEC', 5180, 'Beer, Wine & Distilled Beverages', 3, 2591 FROM dual
UNION ALL SELECT 3769, 'SIC', 5130, 'Apparel, Piece Goods & Notations', 3, 3762 FROM dual
UNION ALL SELECT 3797, 'SIC', 5190, 'Miscellaneous Nondurable Goods', 3, 3762 FROM dual
UNION ALL SELECT 3763, 'SIC', 5110, 'Paper & Paper Products', 3, 3762 FROM dual
UNION ALL SELECT 3784, 'SIC', 5150, 'Farm-Product Raw Materials', 3, 3762 FROM dual
UNION ALL SELECT 3788, 'SIC', 5160, 'Chemicals & Allied Products', 3, 3762 FROM dual
UNION ALL SELECT 3767, 'SIC', 5120, 'Drugs, Proprietaries & Sundries', 3, 3762 FROM dual
UNION ALL SELECT 3774, 'SIC', 5140, 'Groceries & Related Products', 3, 3762 FROM dual
UNION ALL SELECT 3791, 'SIC', 5170, 'Petroleum & Petroleum Products', 3, 3762 FROM dual
UNION ALL SELECT 3794, 'SIC', 5180, 'Beer, Wine & Distilled Beverages', 3, 3762 FROM dual
UNION ALL SELECT 2606, 'SEC', 5210, 'Lumber & Other Building Materials', 3, 2605 FROM dual
UNION ALL SELECT 2608, 'SEC', 5270, 'Mobile Homes Dealers', 3, 2605 FROM dual
UNION ALL SELECT 3813, 'SIC', 5270, 'Mobile Homes Dealers', 3, 3804 FROM dual
UNION ALL SELECT 3807, 'SIC', 5230, 'Paint, Glass & Wallpaper Stores', 3, 3804 FROM dual
UNION ALL SELECT 3809, 'SIC', 5250, 'Hardware Stores', 3, 3804 FROM dual
UNION ALL SELECT 3811, 'SIC', 5260, 'Retail Nurseries & Garden Stores', 3, 3804 FROM dual
UNION ALL SELECT 3805, 'SIC', 5210, 'Lumber & Other Building Materials', 3, 3804 FROM dual
UNION ALL SELECT 2615, 'SEC', 5390, 'Miscellaneous General Merchandise Stores', 3, 2610 FROM dual
UNION ALL SELECT 2613, 'SEC', 5330, 'Variety Stores', 3, 2610 FROM dual
UNION ALL SELECT 2611, 'SEC', 5310, 'Department Stores', 3, 2610 FROM dual
UNION ALL SELECT 3820, 'SIC', 5390, 'Miscellaneous General Merchandise Stores', 3, 3815 FROM dual
UNION ALL SELECT 3818, 'SIC', 5330, 'Variety Stores', 3, 3815 FROM dual
UNION ALL SELECT 3816, 'SIC', 5310, 'Department Stores', 3, 3815 FROM dual
UNION ALL SELECT 2618, 'SEC', 5410, 'Grocery Stores', 3, 2617 FROM dual
UNION ALL SELECT 3833, 'SIC', 5460, 'Retail Bakeries', 3, 3822 FROM dual
UNION ALL SELECT 3831, 'SIC', 5450, 'Dairy Products Stores', 3, 3822 FROM dual
UNION ALL SELECT 3827, 'SIC', 5430, 'Fruit & Vegetable Markets', 3, 3822 FROM dual
UNION ALL SELECT 3829, 'SIC', 5440, 'Candy, Nut & Confectionery Stores', 3, 3822 FROM dual
UNION ALL SELECT 3835, 'SIC', 5490, 'Miscellaneous Food Stores', 3, 3822 FROM dual
UNION ALL SELECT 3823, 'SIC', 5410, 'Grocery Stores', 3, 3822 FROM dual
UNION ALL SELECT 3825, 'SIC', 5420, 'Meat & Fish Markets', 3, 3822 FROM dual
UNION ALL SELECT 2621, 'SEC', 5530, 'Auto & Home Supply Stores', 3, 2620 FROM dual
UNION ALL SELECT 3850, 'SIC', 5570, 'Motorcycle Dealers', 3, 3837 FROM dual
UNION ALL SELECT 3852, 'SIC', 5590, 'Automotive Dealers, nec', 3, 3837 FROM dual
UNION ALL SELECT 3838, 'SIC', 5510, 'New & Used Car Dealers', 3, 3837 FROM dual
UNION ALL SELECT 3842, 'SIC', 5530, 'Auto & Home Supply Stores', 3, 3837 FROM dual
UNION ALL SELECT 3844, 'SIC', 5540, 'Gasoline Service Stations', 3, 3837 FROM dual
UNION ALL SELECT 3846, 'SIC', 5550, 'Boat Dealers', 3, 3837 FROM dual
UNION ALL SELECT 3848, 'SIC', 5560, 'Recreational Vehicle Dealers', 3, 3837 FROM dual
UNION ALL SELECT 2624, 'SEC', 5620, 'Women''s Clothing Stores', 3, 2623 FROM dual
UNION ALL SELECT 2626, 'SEC', 5650, 'Family Clothing Stores', 3, 2623 FROM dual
UNION ALL SELECT 2628, 'SEC', 5660, 'Shoe Stores', 3, 2623 FROM dual
UNION ALL SELECT 3865, 'SIC', 5660, 'Shoe Stores', 3, 3854 FROM dual
UNION ALL SELECT 3857, 'SIC', 5620, 'Women''s Clothing Stores', 3, 3854 FROM dual
UNION ALL SELECT 4254, 'SIC', 9190, 'General Government, nec', 3, 4247 FROM dual
UNION ALL SELECT 3859, 'SIC', 5630, 'Women''s Accessory & Specialty Stores', 3, 3854 FROM dual
UNION ALL SELECT 3861, 'SIC', 5640, 'Children''s & Infants'' Wear Stores', 3, 3854 FROM dual
UNION ALL SELECT 3855, 'SIC', 5610, 'Men''s & Boys'' Clothing Stores', 3, 3854 FROM dual
UNION ALL SELECT 3863, 'SIC', 5650, 'Family Clothing Stores', 3, 3854 FROM dual
UNION ALL SELECT 3867, 'SIC', 5690, 'Miscellaneous Apparel & Accessory Stores', 3, 3854 FROM dual
UNION ALL SELECT 2633, 'SEC', 5730, 'Radio, Television & Computer Stores', 3, 2630 FROM dual
UNION ALL SELECT 2631, 'SEC', 5710, 'Furniture & Homefurnishing Stores', 3, 2630 FROM dual
UNION ALL SELECT 3870, 'SIC', 5710, 'Furniture & Homefurnishing Stores', 3, 3869 FROM dual
UNION ALL SELECT 3877, 'SIC', 5730, 'Radio, Television & Computer Stores', 3, 3869 FROM dual
UNION ALL SELECT 3875, 'SIC', 5720, 'Household Appliance Stores', 3, 3869 FROM dual
UNION ALL SELECT 2638, 'SEC', 5810, 'Eating & Drinking Places', 3, 2637 FROM dual
UNION ALL SELECT 3883, 'SIC', 5810, 'Eating & Drinking Places', 3, 3882 FROM dual
UNION ALL SELECT 2641, 'SEC', 5910, 'Drug Stores & Proprietary Stores', 3, 2640 FROM dual
UNION ALL SELECT 2648, 'SEC', 5990, 'Retail Stores, nec', 3, 2640 FROM dual
UNION ALL SELECT 2643, 'SEC', 5940, 'Miscellaneous Shopping Goods Stores', 3, 2640 FROM dual
UNION ALL SELECT 2646, 'SEC', 5960, 'Nonstore Retailers', 3, 2640 FROM dual
UNION ALL SELECT 3907, 'SIC', 5980, 'Fuel Dealers', 3, 3886 FROM dual
UNION ALL SELECT 3911, 'SIC', 5990, 'Retail Stores, nec', 3, 3886 FROM dual
UNION ALL SELECT 3891, 'SIC', 5930, 'Used Merchandise Stores', 3, 3886 FROM dual
UNION ALL SELECT 3887, 'SIC', 5910, 'Drug Stores & Proprietary Stores', 3, 3886 FROM dual
UNION ALL SELECT 3889, 'SIC', 5920, 'Liquor Stores', 3, 3886 FROM dual
UNION ALL SELECT 3893, 'SIC', 5940, 'Miscellaneous Shopping Goods Stores', 3, 3886 FROM dual
UNION ALL SELECT 3903, 'SIC', 5960, 'Nonstore Retailers', 3, 3886 FROM dual
UNION ALL SELECT 2654, 'SEC', 6030, 'Savings Institutions', 3, 2649 FROM dual
UNION ALL SELECT 2650, 'SEC', 6020, 'Commercial Banks', 3, 2649 FROM dual
UNION ALL SELECT 2657, 'SEC', 6090, 'Functions Closely Related to Banking', 3, 2649 FROM dual
UNION ALL SELECT 3925, 'SIC', 6030, 'Savings Institutions', 3, 3917 FROM dual
UNION ALL SELECT 3921, 'SIC', 6020, 'Commercial Banks', 3, 3917 FROM dual
UNION ALL SELECT 3928, 'SIC', 6060, 'Credit Unions', 3, 3917 FROM dual
UNION ALL SELECT 3934, 'SIC', 6090, 'Functions Closely Related to Banking', 3, 3917 FROM dual
UNION ALL SELECT 3918, 'SIC', 6010, 'Central Reserve Depositories', 3, 3917 FROM dual
UNION ALL SELECT 3931, 'SIC', 6080, 'Foreign Bank & Branches & Agencies', 3, 3917 FROM dual
UNION ALL SELECT 2667, 'SEC', 6160, 'Mortgage Bankers & Brokers', 3, 2659 FROM dual
UNION ALL SELECT 2664, 'SEC', 6150, 'Business Credit Institutions', 3, 2659 FROM dual
UNION ALL SELECT 2662, 'SEC', 6140, 'Personal Credit Unions', 3, 2659 FROM dual
UNION ALL SELECT 2660, 'SEC', 6110, 'Federal & Federally Sponsored Credit Agencies', 3, 2659 FROM dual
UNION ALL SELECT 3942, 'SIC', 6150, 'Business Credit Institutions', 3, 3937 FROM dual
UNION ALL SELECT 3945, 'SIC', 6160, 'Mortgage Bankers & Brokers', 3, 3937 FROM dual
UNION ALL SELECT 3940, 'SIC', 6140, 'Personal Credit Unions', 3, 3937 FROM dual
UNION ALL SELECT 3938, 'SIC', 6110, 'Federal & Federally Sponsored Credit Agencies', 3, 3937 FROM dual
UNION ALL SELECT 2675, 'SEC', 6280, 'Security & Commodity Services', 3, 2670 FROM dual
UNION ALL SELECT 2673, 'SEC', 6220, 'Commodity Contracts Brokers, Dealers', 3, 2670 FROM dual
UNION ALL SELECT 2671, 'SEC', 6210, 'Security Brokers & Dealers', 3, 2670 FROM dual
UNION ALL SELECT 3955, 'SIC', 6280, 'Security & Commodity Services', 3, 3948 FROM dual
UNION ALL SELECT 3949, 'SIC', 6210, 'Security Brokers & Dealers', 3, 3948 FROM dual
UNION ALL SELECT 3951, 'SIC', 6220, 'Commodity Contracts Brokers, Dealers', 3, 3948 FROM dual
UNION ALL SELECT 3953, 'SIC', 6230, 'Security & Commodity Exchanges', 3, 3948 FROM dual
UNION ALL SELECT 2689, 'SEC', 6390, 'Insurance Carriers, nec', 3, 2677 FROM dual
UNION ALL SELECT 2683, 'SEC', 6330, 'Fire, Marine & Casualty Insurance', 3, 2677 FROM dual
UNION ALL SELECT 2680, 'SEC', 6320, 'Medical Service & Health Insurance', 3, 2677 FROM dual
UNION ALL SELECT 2678, 'SEC', 6310, 'Life Insurance', 3, 2677 FROM dual
UNION ALL SELECT 2687, 'SEC', 6360, 'Title Insurance', 3, 2677 FROM dual
UNION ALL SELECT 2685, 'SEC', 6350, 'Surety Insurance', 3, 2677 FROM dual
UNION ALL SELECT 3970, 'SIC', 6370, 'Pension, Health & Welfare Funds', 3, 3958 FROM dual
UNION ALL SELECT 3959, 'SIC', 6310, 'Life Insurance', 3, 3958 FROM dual
UNION ALL SELECT 3961, 'SIC', 6320, 'Medical Service & Health Insurance', 3, 3958 FROM dual
UNION ALL SELECT 3964, 'SIC', 6330, 'Fire, Marine & Casualty Insurance', 3, 3958 FROM dual
UNION ALL SELECT 3966, 'SIC', 6350, 'Surety Insurance', 3, 3958 FROM dual
UNION ALL SELECT 3968, 'SIC', 6360, 'Title Insurance', 3, 3958 FROM dual
UNION ALL SELECT 3972, 'SIC', 6390, 'Insurance Carriers, nec', 3, 3958 FROM dual
UNION ALL SELECT 2692, 'SEC', 6410, 'Insurance Agents, Brokers & Service', 3, 2691 FROM dual
UNION ALL SELECT 3975, 'SIC', 6410, 'Insurance Agents, Brokers & Service', 3, 3974 FROM dual
UNION ALL SELECT 2695, 'SEC', 6510, 'Real Estate Operators & Lessors', 3, 2694 FROM dual
UNION ALL SELECT 2699, 'SEC', 6530, 'Real Estate Agents & Managers', 3, 2694 FROM dual
UNION ALL SELECT 2701, 'SEC', 6550, 'Subdividers & Developers', 3, 2694 FROM dual
UNION ALL SELECT 3989, 'SIC', 6550, 'Subdividers & Developers', 3, 3977 FROM dual
UNION ALL SELECT 3985, 'SIC', 6530, 'Real Estate Agents & Managers', 3, 3977 FROM dual
UNION ALL SELECT 3987, 'SIC', 6540, 'Title Abstract Offices', 3, 3977 FROM dual
UNION ALL SELECT 3978, 'SIC', 6510, 'Real Estate Operators & Lessors', 3, 3977 FROM dual
UNION ALL SELECT 2704, 'SEC', 6790, 'Miscellaneous Investing', 3, 2703 FROM dual
UNION ALL SELECT 3993, 'SIC', 6710, 'Holding Offices', 3, 3992 FROM dual
UNION ALL SELECT 3996, 'SIC', 6720, 'Investment Offices', 3, 3992 FROM dual
UNION ALL SELECT 4002, 'SIC', 6790, 'Miscellaneous Investing', 3, 3992 FROM dual
UNION ALL SELECT 3999, 'SIC', 6730, 'Trusts', 3, 3992 FROM dual
UNION ALL SELECT 2710, 'SEC', 7010, 'Hotels & Motels', 3, 2709 FROM dual
UNION ALL SELECT 4012, 'SIC', 7030, 'Camps & Recreational Vehicle Parks', 3, 4007 FROM dual
UNION ALL SELECT 4015, 'SIC', 7040, 'Membership Basis Organization Hotels', 3, 4007 FROM dual
UNION ALL SELECT 4010, 'SIC', 7020, 'Rooming & Boarding Houses', 3, 4007 FROM dual
UNION ALL SELECT 4008, 'SIC', 7010, 'Hotels & Motels', 3, 4007 FROM dual
UNION ALL SELECT 4037, 'SIC', 7290, 'Miscellaneous Personal Services', 3, 4017 FROM dual
UNION ALL SELECT 4029, 'SIC', 7230, 'Beauty Shops', 3, 4017 FROM dual
UNION ALL SELECT 4033, 'SIC', 7250, 'Shoe Repair & Shoeshine Parlors', 3, 4017 FROM dual
UNION ALL SELECT 4031, 'SIC', 7240, 'Barber Shops', 3, 4017 FROM dual
UNION ALL SELECT 4018, 'SIC', 7210, 'Laundry, Cleaning & Garment Services', 3, 4017 FROM dual
UNION ALL SELECT 4035, 'SIC', 7260, 'Funeral Service & Crematories', 3, 4017 FROM dual
UNION ALL SELECT 4027, 'SIC', 7220, 'Photographic Studios, Portrait', 3, 4017 FROM dual
UNION ALL SELECT 2725, 'SEC', 7370, 'Computer & Data Processing Services', 3, 2713 FROM dual
UNION ALL SELECT 2720, 'SEC', 7350, 'Misc. Equipment Rental & Leasing', 3, 2713 FROM dual
UNION ALL SELECT 2719, 'SEC', 7340, 'Services to Buildings', 3, 2713 FROM dual
UNION ALL SELECT 2717, 'SEC', 7330, 'Mailing, Reproductive, Stenographic', 3, 2713 FROM dual
UNION ALL SELECT 2716, 'SEC', 7320, 'Credit Reporting & Collection', 3, 2713 FROM dual
UNION ALL SELECT 2714, 'SEC', 7310, 'Advertising', 3, 2713 FROM dual
UNION ALL SELECT 2731, 'SEC', 7380, 'Miscellaneous Business Services', 3, 2713 FROM dual
UNION ALL SELECT 2722, 'SEC', 7360, 'Personnel Supply Services', 3, 2713 FROM dual
UNION ALL SELECT 4055, 'SIC', 7340, 'Services to Buildings', 3, 4040 FROM dual
UNION ALL SELECT 4075, 'SIC', 7380, 'Miscellaneous Business Services', 3, 4040 FROM dual
UNION ALL SELECT 4058, 'SIC', 7350, 'Misc. Equipment Rental & Leasing', 3, 4040 FROM dual
UNION ALL SELECT 4049, 'SIC', 7330, 'Mailing, Reproductive, Stenographic', 3, 4040 FROM dual
UNION ALL SELECT 4046, 'SIC', 7320, 'Credit Reporting & Collection', 3, 4040 FROM dual
UNION ALL SELECT 4065, 'SIC', 7370, 'Computer & Data Processing Services', 3, 4040 FROM dual
UNION ALL SELECT 4062, 'SIC', 7360, 'Personnel Supply Services', 3, 4040 FROM dual
UNION ALL SELECT 4041, 'SIC', 7310, 'Advertising', 3, 4040 FROM dual
UNION ALL SELECT 2736, 'SEC', 7510, 'Automotive Rentals, No Drivers', 3, 2735 FROM dual
UNION ALL SELECT 4097, 'SIC', 7540, 'Automotive Services, Except Repair', 3, 4081 FROM dual
UNION ALL SELECT 4087, 'SIC', 7520, 'Automobile Parking', 3, 4081 FROM dual
UNION ALL SELECT 4089, 'SIC', 7530, 'Automotive Repair Shops', 3, 4081 FROM dual
UNION ALL SELECT 4082, 'SIC', 7510, 'Automotive Rentals, No Drivers', 3, 4081 FROM dual
UNION ALL SELECT 4101, 'SIC', 7620, 'Electrical Repair Shops', 3, 4100 FROM dual
UNION ALL SELECT 4109, 'SIC', 7690, 'Miscellaneous Repair Shops', 3, 4100 FROM dual
UNION ALL SELECT 4105, 'SIC', 7630, 'Watch, Clock & Jewelry Repair', 3, 4100 FROM dual
UNION ALL SELECT 4107, 'SIC', 7640, 'Reupholstery & Furniture Repair', 3, 4100 FROM dual
UNION ALL SELECT 2745, 'SEC', 7830, 'Motion Picture Theaters', 3, 2738 FROM dual
UNION ALL SELECT 2746, 'SEC', 7840, 'Video Tape Rental', 3, 2738 FROM dual
UNION ALL SELECT 2742, 'SEC', 7820, 'Motion Picture Distribution & Services', 3, 2738 FROM dual
UNION ALL SELECT 2739, 'SEC', 7810, 'Motion Picture Production & Services', 3, 2738 FROM dual
UNION ALL SELECT 4120, 'SIC', 7830, 'Motion Picture Theaters', 3, 4113 FROM dual
UNION ALL SELECT 4117, 'SIC', 7820, 'Motion Picture Distribution & Services', 3, 4113 FROM dual
UNION ALL SELECT 4114, 'SIC', 7810, 'Motion Picture Production & Services', 3, 4113 FROM dual
UNION ALL SELECT 4123, 'SIC', 7840, 'Video Tape Rental', 3, 4113 FROM dual
UNION ALL SELECT 2749, 'SEC', 7940, 'Commercial Sports', 3, 2748 FROM dual
UNION ALL SELECT 2751, 'SEC', 7990, 'Misc. Amusement & Recreation Services', 3, 2748 FROM dual
UNION ALL SELECT 4128, 'SIC', 7920, 'Producers, Orchestras, Entertainers', 3, 4125 FROM dual
UNION ALL SELECT 4131, 'SIC', 7930, 'Bowling Centers', 3, 4125 FROM dual
UNION ALL SELECT 4133, 'SIC', 7940, 'Commercial Sports', 3, 4125 FROM dual
UNION ALL SELECT 4136, 'SIC', 7950, 'Ski Facilities', 3, 4125 FROM dual
UNION ALL SELECT 4139, 'SIC', 7990, 'Misc. Amusement & Recreation Services', 3, 4125 FROM dual
UNION ALL SELECT 4126, 'SIC', 7910, 'Dance Studios, Schools & Halls', 3, 4125 FROM dual
UNION ALL SELECT 2764, 'SEC', 8090, 'Health & Allied Services, nec', 3, 2753 FROM dual
UNION ALL SELECT 2762, 'SEC', 8080, 'Home Health Care Services', 3, 2753 FROM dual
UNION ALL SELECT 2754, 'SEC', 8010, 'Offices & Clinics of Medical Doctors', 3, 2753 FROM dual
UNION ALL SELECT 2756, 'SEC', 8050, 'Nursing & Personal Care Facilities', 3, 2753 FROM dual
UNION ALL SELECT 2758, 'SEC', 8060, 'Hospitals', 3, 2753 FROM dual
UNION ALL SELECT 2760, 'SEC', 8070, 'Medical & Dental Laboratories', 3, 2753 FROM dual
UNION ALL SELECT 4171, 'SIC', 8090, 'Health & Allied Services, nec', 3, 4146 FROM dual
UNION ALL SELECT 4158, 'SIC', 8050, 'Nursing & Personal Care Facilities', 3, 4146 FROM dual
UNION ALL SELECT 4153, 'SIC', 8040, 'Offices of Other Health Practitioners', 3, 4146 FROM dual
UNION ALL SELECT 4169, 'SIC', 8080, 'Home Health Care Services', 3, 4146 FROM dual
UNION ALL SELECT 4151, 'SIC', 8030, 'Offices of Osteopathic Physicians', 3, 4146 FROM dual
UNION ALL SELECT 4149, 'SIC', 8020, 'Offices & Clinics of Dentists', 3, 4146 FROM dual
UNION ALL SELECT 4166, 'SIC', 8070, 'Medical & Dental Laboratories', 3, 4146 FROM dual
UNION ALL SELECT 4147, 'SIC', 8010, 'Offices & Clinics of Medical Doctors', 3, 4146 FROM dual
UNION ALL SELECT 4162, 'SIC', 8060, 'Hospitals', 3, 4146 FROM dual
UNION ALL SELECT 2767, 'SEC', 8110, 'Legal Services', 3, 2766 FROM dual
UNION ALL SELECT 4176, 'SIC', 8110, 'Legal Services', 3, 4175 FROM dual
UNION ALL SELECT 4179, 'SIC', 8210, 'Elementary & Secondary Schools', 3, 4178 FROM dual
UNION ALL SELECT 4186, 'SIC', 8240, 'Vocational Schools', 3, 4178 FROM dual
UNION ALL SELECT 4181, 'SIC', 8220, 'Colleges & Universities', 3, 4178 FROM dual
UNION ALL SELECT 4190, 'SIC', 8290, 'Schools & Educational Services, nec', 3, 4178 FROM dual
UNION ALL SELECT 4184, 'SIC', 8230, 'Libraries', 3, 4178 FROM dual
UNION ALL SELECT 2771, 'SEC', 8350, 'Child Day Care Services', 3, 2770 FROM dual
UNION ALL SELECT 4193, 'SIC', 8320, 'Individual & Family Services', 3, 4192 FROM dual
UNION ALL SELECT 4197, 'SIC', 8350, 'Child Day Care Services', 3, 4192 FROM dual
UNION ALL SELECT 4201, 'SIC', 8390, 'Social Services, nec', 3, 4192 FROM dual
UNION ALL SELECT 4195, 'SIC', 8330, 'Job Training & Related Services', 3, 4192 FROM dual
UNION ALL SELECT 4199, 'SIC', 8360, 'Residential Care', 3, 4192 FROM dual
UNION ALL SELECT 4204, 'SIC', 8410, 'Museums & Art Galleries', 3, 4203 FROM dual
UNION ALL SELECT 4206, 'SIC', 8420, 'Botanical & Zoological Gardens', 3, 4203 FROM dual
UNION ALL SELECT 4217, 'SIC', 8650, 'Political Organizations', 3, 4208 FROM dual
UNION ALL SELECT 4221, 'SIC', 8690, 'Membership Organizations, nec', 3, 4208 FROM dual
UNION ALL SELECT 4213, 'SIC', 8630, 'Labor Organizations', 3, 4208 FROM dual
UNION ALL SELECT 4209, 'SIC', 8610, 'Business Associations', 3, 4208 FROM dual
UNION ALL SELECT 4211, 'SIC', 8620, 'Professional Organizations', 3, 4208 FROM dual
UNION ALL SELECT 4215, 'SIC', 8640, 'Civic & Social Organizations', 3, 4208 FROM dual
UNION ALL SELECT 4219, 'SIC', 8660, 'Religious Organizations', 3, 4208 FROM dual
UNION ALL SELECT 2780, 'SEC', 8740, 'Management & Public Relations', 3, 2774 FROM dual
UNION ALL SELECT 2775, 'SEC', 8710, 'Engineering & Architectural Services', 3, 2774 FROM dual
UNION ALL SELECT 2777, 'SEC', 8730, 'Research & Testing Services', 3, 2774 FROM dual
UNION ALL SELECT 4228, 'SIC', 8720, 'Accounting, Auditing & Bookkeeping', 3, 4223 FROM dual
UNION ALL SELECT 4235, 'SIC', 8740, 'Management & Public Relations', 3, 4223 FROM dual
UNION ALL SELECT 4224, 'SIC', 8710, 'Engineering & Architectural Services', 3, 4223 FROM dual
UNION ALL SELECT 4230, 'SIC', 8730, 'Research & Testing Services', 3, 4223 FROM dual
UNION ALL SELECT 4242, 'SIC', 8810, 'Private Households', 3, 4241 FROM dual
UNION ALL SELECT 4245, 'SIC', 8990, 'Services, nec', 3, 4244 FROM dual
UNION ALL SELECT 4252, 'SIC', 9130, 'Executive & Legislative Combined', 3, 4247 FROM dual
UNION ALL SELECT 4250, 'SIC', 9120, 'Legislative Bodies', 3, 4247 FROM dual
UNION ALL SELECT 4248, 'SIC', 9110, 'Executive Offices', 3, 4247 FROM dual
UNION ALL SELECT 4259, 'SIC', 9220, 'Public Order & Safety', 3, 4256 FROM dual
UNION ALL SELECT 4266, 'SIC', 9310, 'Finance, Taxation & Monetary Policy', 3, 4265 FROM dual
UNION ALL SELECT 4275, 'SIC', 9450, 'Administration of Veteran''s Affairs', 3, 4268 FROM dual
UNION ALL SELECT 4271, 'SIC', 9430, 'Admin. of Public Health Programs', 3, 4268 FROM dual
UNION ALL SELECT 4273, 'SIC', 9440, 'Admin. of Social & Manpower Programs', 3, 4268 FROM dual
UNION ALL SELECT 4269, 'SIC', 9410, 'Admin. of Educational Programs', 3, 4268 FROM dual
UNION ALL SELECT 4278, 'SIC', 9510, 'Environmental Quality', 3, 4277 FROM dual
UNION ALL SELECT 4281, 'SIC', 9530, 'Housing & Urban Development', 3, 4277 FROM dual
UNION ALL SELECT 4289, 'SIC', 9630, 'Regulation, Admin. of Utilities', 3, 4284 FROM dual
UNION ALL SELECT 4293, 'SIC', 9650, 'Regulation Misc. Commercial Sectors', 3, 4284 FROM dual
UNION ALL SELECT 4295, 'SIC', 9660, 'Space Research & Technology', 3, 4284 FROM dual
UNION ALL SELECT 4287, 'SIC', 9620, 'Regulation, Admin. of Transportation', 3, 4284 FROM dual
UNION ALL SELECT 4285, 'SIC', 9610, 'Administration of General Economic Programs', 3, 4284 FROM dual
UNION ALL SELECT 4291, 'SIC', 9640, 'Regulation of Agricultural Marketing', 3, 4284 FROM dual
UNION ALL SELECT 2786, 'SEC', 9720, 'International Affairs', 3, 2785 FROM dual
UNION ALL SELECT 4300, 'SIC', 9720, 'International Affairs', 3, 4297 FROM dual
UNION ALL SELECT 4298, 'SIC', 9710, 'National Security', 3, 4297 FROM dual
UNION ALL SELECT 4303, 'SIC', 9990, 'Nonclassifiable Establishments', 3, 4302 FROM dual
;

CREATE TABLE "industry_level" (
    industry_level_id number(19) NOT NULL,
    industry_classification varchar2(16),
    ancestor_id number(19),
    ancestor_code integer,
    ancestor_depth integer,
    descendant_id number(19),
    descendant_code integer,
    descendant_depth integer,
    PRIMARY KEY (industry_level_id)
);

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 1, 'SEC', 2677, 6300, 2, 2689, 6390, 3 FROM dual
UNION ALL SELECT 2, 'NAICS', 1666, 5415, 3, 1671, 541519, 5 FROM dual
UNION ALL SELECT 3, 'NAICS', 931, 423, 2, 999, 42384, 4 FROM dual
UNION ALL SELECT 4, 'SEC', 2791, 20, 1, 2362, 3240, 3 FROM dual
UNION ALL SELECT 5, 'SEC', 2792, 40, 1, 2538, 4731, 4 FROM dual
UNION ALL SELECT 6, 'SIC', 3815, 5300, 2, 3820, 5390, 3 FROM dual
UNION ALL SELECT 7, 'NAICS', 1809, 56299, 4, 1810, 562991, 5 FROM dual
UNION ALL SELECT 8, 'SIC', 4314, 90, 1, 4251, 9121, 4 FROM dual
UNION ALL SELECT 9, 'NAICS', 1612, 5324, 3, 1617, 53242, 4 FROM dual
UNION ALL SELECT 10, 'NAICS', 1868, 62139, 4, 1870, 621399, 5 FROM dual
UNION ALL SELECT 11, 'NAICS', 1026, 4243, 3, 1031, 424330, 5 FROM dual
UNION ALL SELECT 12, 'SIC', 4311, 52, 1, 3890, 5921, 4 FROM dual
UNION ALL SELECT 13, 'NAICS', 1942, 71, 1, 1971, 712110, 5 FROM dual
UNION ALL SELECT 14, 'SIC', 4313, 70, 1, 4067, 7372, 4 FROM dual
UNION ALL SELECT 15, 'SIC', 3474, 3590, 3, 3475, 3592, 4 FROM dual
UNION ALL SELECT 16, 'SIC', 4308, 20, 1, 3275, 3083, 4 FROM dual
UNION ALL SELECT 17, 'NAICS', 2037, 81, 1, 2083, 812220, 5 FROM dual
UNION ALL SELECT 18, 'SIC', 4314, 90, 1, 4254, 9190, 3 FROM dual
UNION ALL SELECT 19, 'SIC', 2834, 250, 3, 2838, 254, 4 FROM dual
UNION ALL SELECT 20, 'NAICS', 56, 112, 2, 93, 11292, 4 FROM dual
UNION ALL SELECT 21, 'NAICS', 1555, 5251, 3, 1561, 52519, 4 FROM dual
UNION ALL SELECT 22, 'SEC', 2384, 3400, 2, 2395, 3448, 4 FROM dual
UNION ALL SELECT 23, 'SIC', 3409, 3490, 3, 3418, 3499, 4 FROM dual
UNION ALL SELECT 24, 'NAICS', 1, 11, 1, 130, 115310, 5 FROM dual
UNION ALL SELECT 25, 'NAICS', 1625, 541, 2, 1629, 541120, 5 FROM dual
UNION ALL SELECT 26, 'SIC', 3050, 2200, 2, 3072, 2273, 4 FROM dual
UNION ALL SELECT 27, 'SIC', 4040, 7300, 2, 4054, 7338, 4 FROM dual
UNION ALL SELECT 28, 'SEC', 2795, 60, 1, 2695, 6510, 3 FROM dual
UNION ALL SELECT 29, 'NAICS', 1431, 5122, 3, 1437, 51223, 4 FROM dual
UNION ALL SELECT 30, 'NAICS', 930, 42, 1, 1088, 42499, 4 FROM dual
UNION ALL SELECT 31, 'NAICS', 1684, 54171, 4, 1686, 541712, 5 FROM dual
UNION ALL SELECT 32, 'NAICS', 1726, 561, 2, 1754, 56149, 4 FROM dual
UNION ALL SELECT 33, 'SIC', 4082, 7510, 3, 4086, 7519, 4 FROM dual
UNION ALL SELECT 34, 'SIC', 3744, 5070, 3, 3745, 5072, 4 FROM dual
UNION ALL SELECT 35, 'SEC', 2795, 60, 1, 2680, 6320, 3 FROM dual
UNION ALL SELECT 36, 'NAICS', 930, 42, 1, 1016, 4241, 3 FROM dual
UNION ALL SELECT 37, 'NAICS', 1792, 562, 2, 1804, 5629, 3 FROM dual
UNION ALL SELECT 38, 'NAICS', 236, 2381, 3, 247, 238160, 5 FROM dual
UNION ALL SELECT 39, 'SEC', 2572, 5030, 3, 2573, 5031, 4 FROM dual
UNION ALL SELECT 40, 'NAICS', 1555, 5251, 3, 1556, 525110, 5 FROM dual
UNION ALL SELECT 41, 'NAICS', 1570, 531, 2, 1574, 531120, 5 FROM dual
UNION ALL SELECT 42, 'SIC', 3246, 2890, 3, 3247, 2891, 4 FROM dual
UNION ALL SELECT 43, 'NAICS', 134, 2111, 3, 137, 211112, 5 FROM dual
UNION ALL SELECT 44, 'NAICS', 1725, 56, 1, 1732, 56121, 4 FROM dual
UNION ALL SELECT 45, 'NAICS', 1972, 71211, 4, 1971, 712110, 5 FROM dual
UNION ALL SELECT 46, 'NAICS', 1624, 54, 1, 1654, 54137, 4 FROM dual
UNION ALL SELECT 47, 'NAICS', 2003, 72, 1, 2023, 72231, 4 FROM dual
UNION ALL SELECT 48, 'SEC', 2791, 20, 1, 2407, 3524, 4 FROM dual
UNION ALL SELECT 49, 'NAICS', 1026, 4243, 3, 1033, 424340, 5 FROM dual
UNION ALL SELECT 50, 'NAICS', 1738, 56132, 4, 1737, 561320, 5 FROM dual
UNION ALL SELECT 51, 'NAICS', 1943, 711, 2, 1961, 711320, 5 FROM dual
UNION ALL SELECT 52, 'SIC', 4308, 20, 1, 3486, 3624, 4 FROM dual
UNION ALL SELECT 53, 'NAICS', 1625, 541, 2, 1633, 541199, 5 FROM dual
UNION ALL SELECT 54, 'NAICS', 2131, 814, 2, 2133, 814110, 5 FROM dual
UNION ALL SELECT 55, 'SIC', 4075, 7380, 3, 4080, 7389, 4 FROM dual
UNION ALL SELECT 56, 'SIC', 2982, 2000, 2, 3036, 2095, 4 FROM dual
UNION ALL SELECT 57, 'SIC', 4311, 52, 1, 3905, 5962, 4 FROM dual
UNION ALL SELECT 58, 'NAICS', 68, 1123, 3, 78, 11239, 4 FROM dual
UNION ALL SELECT 59, 'SEC', 2794, 52, 1, 2620, 5500, 2 FROM dual
UNION ALL SELECT 60, 'NAICS', 1402, 51, 1, 1472, 519110, 5 FROM dual
UNION ALL SELECT 61, 'SIC', 4308, 20, 1, 3055, 2230, 3 FROM dual
UNION ALL SELECT 62, 'NAICS', 2189, 9261, 3, 2192, 926120, 5 FROM dual
UNION ALL SELECT 63, 'SIC', 3587, 3950, 3, 3588, 3951, 4 FROM dual
UNION ALL SELECT 64, 'SIC', 4310, 50, 1, 3752, 5084, 4 FROM dual
UNION ALL SELECT 65, 'SIC', 2798, 100, 2, 2821, 181, 4 FROM dual
UNION ALL SELECT 66, 'SIC', 4311, 52, 1, 3832, 5451, 4 FROM dual
UNION ALL SELECT 67, 'NAICS', 132, 21, 1, 177, 213113, 5 FROM dual
UNION ALL SELECT 68, 'NAICS', 1725, 56, 1, 1772, 561621, 5 FROM dual
UNION ALL SELECT 69, 'SIC', 3371, 3400, 2, 3410, 3491, 4 FROM dual
UNION ALL SELECT 70, 'SIC', 4308, 20, 1, 3280, 3088, 4 FROM dual
UNION ALL SELECT 71, 'SIC', 4308, 20, 1, 3182, 2672, 4 FROM dual
UNION ALL SELECT 72, 'NAICS', 1979, 713, 2, 1999, 713950, 5 FROM dual
UNION ALL SELECT 73, 'NAICS', 1850, 62, 1, 1939, 6244, 3 FROM dual
UNION ALL SELECT 74, 'NAICS', 1733, 5613, 3, 1736, 561312, 5 FROM dual
UNION ALL SELECT 75, 'NAICS', 182, 2211, 3, 188, 221115, 5 FROM dual
UNION ALL SELECT 76, 'NAICS', 2135, 92, 1, 2194, 926130, 5 FROM dual
UNION ALL SELECT 77, 'SIC', 3419, 3500, 2, 3468, 3580, 3 FROM dual
UNION ALL SELECT 78, 'NAICS', 138, 212, 2, 170, 212393, 5 FROM dual
UNION ALL SELECT 79, 'SIC', 2811, 160, 3, 2812, 161, 4 FROM dual
UNION ALL SELECT 80, 'SIC', 4308, 20, 1, 3199, 2741, 4 FROM dual
UNION ALL SELECT 81, 'NAICS', 1010, 42393, 4, 1009, 423930, 5 FROM dual
UNION ALL SELECT 82, 'SEC', 2748, 7900, 2, 2752, 7997, 4 FROM dual
UNION ALL SELECT 83, 'SIC', 4306, 10, 1, 2928, 1450, 3 FROM dual
UNION ALL SELECT 84, 'SIC', 3050, 2200, 2, 3078, 2295, 4 FROM dual
UNION ALL SELECT 85, 'NAICS', 930, 42, 1, 951, 423330, 5 FROM dual
UNION ALL SELECT 86, 'NAICS', 2, 111, 2, 8, 111130, 5 FROM dual
UNION ALL SELECT 87, 'SIC', 4308, 20, 1, 3243, 2874, 4 FROM dual
UNION ALL SELECT 88, 'SIC', 2825, 200, 2, 2844, 279, 4 FROM dual
UNION ALL SELECT 89, 'SEC', 2795, 60, 1, 2694, 6500, 2 FROM dual
UNION ALL SELECT 90, 'SIC', 4305, 1, 1, 2834, 250, 3 FROM dual
UNION ALL SELECT 91, 'SEC', 2791, 20, 1, 2326, 2835, 4 FROM dual
UNION ALL SELECT 92, 'SIC', 4002, 6790, 3, 4005, 6798, 4 FROM dual
UNION ALL SELECT 93, 'SIC', 4314, 90, 1, 4252, 9130, 3 FROM dual
UNION ALL SELECT 94, 'NAICS', 1485, 522, 2, 1502, 522292, 5 FROM dual
UNION ALL SELECT 95, 'NAICS', 138, 212, 2, 167, 21239, 4 FROM dual
UNION ALL SELECT 96, 'SIC', 4312, 60, 1, 3968, 6360, 3 FROM dual
UNION ALL SELECT 97, 'NAICS', 1979, 713, 2, 1996, 71393, 4 FROM dual
UNION ALL SELECT 98, 'SIC', 3337, 3300, 2, 3358, 3354, 4 FROM dual
UNION ALL SELECT 99, 'SIC', 3937, 6100, 2, 3938, 6110, 3 FROM dual
UNION ALL SELECT 100, 'NAICS', 1985, 7132, 3, 1988, 713290, 5 FROM dual
UNION ALL SELECT 101, 'NAICS', 1591, 532, 2, 1597, 53212, 4 FROM dual
UNION ALL SELECT 102, 'NAICS', 56, 112, 2, 73, 112330, 5 FROM dual
UNION ALL SELECT 103, 'SIC', 3083, 2300, 2, 3102, 2353, 4 FROM dual
UNION ALL SELECT 104, 'SIC', 4268, 9400, 2, 4274, 9441, 4 FROM dual
UNION ALL SELECT 105, 'NAICS', 931, 423, 2, 1008, 42392, 4 FROM dual
UNION ALL SELECT 106, 'SIC', 4308, 20, 1, 3025, 2079, 4 FROM dual
UNION ALL SELECT 107, 'SIC', 3570, 3850, 3, 3571, 3851, 4 FROM dual
UNION ALL SELECT 108, 'NAICS', 1813, 611, 2, 1825, 61141, 4 FROM dual
UNION ALL SELECT 109, 'SEC', 2689, 6390, 3, 2690, 6399, 4 FROM dual
UNION ALL SELECT 110, 'NAICS', 1943, 711, 2, 1964, 711410, 5 FROM dual
UNION ALL SELECT 111, 'SIC', 3496, 3640, 3, 3503, 3648, 4 FROM dual
UNION ALL SELECT 112, 'SIC', 4308, 20, 1, 3431, 3535, 4 FROM dual
UNION ALL SELECT 113, 'SIC', 4309, 40, 1, 3705, 4940, 3 FROM dual
UNION ALL SELECT 114, 'SIC', 4308, 20, 1, 3240, 2869, 4 FROM dual
UNION ALL SELECT 115, 'SIC', 3480, 3600, 2, 3514, 3674, 4 FROM dual
UNION ALL SELECT 116, 'SIC', 4101, 7620, 3, 4103, 7623, 4 FROM dual
UNION ALL SELECT 117, 'NAICS', 236, 2381, 3, 237, 238110, 5 FROM dual
UNION ALL SELECT 118, 'NAICS', 930, 42, 1, 1010, 42393, 4 FROM dual
UNION ALL SELECT 119, 'NAICS', 1569, 53, 1, 1622, 533110, 5 FROM dual
UNION ALL SELECT 120, 'SIC', 4055, 7340, 3, 4057, 7349, 4 FROM dual
UNION ALL SELECT 121, 'NAICS', 205, 23, 1, 245, 238150, 5 FROM dual
UNION ALL SELECT 122, 'SEC', 2384, 3400, 2, 2385, 3410, 3 FROM dual
UNION ALL SELECT 123, 'SEC', 2526, 4500, 2, 2527, 4510, 3 FROM dual
UNION ALL SELECT 124, 'NAICS', 2, 111, 2, 30, 111332, 5 FROM dual
UNION ALL SELECT 125, 'SIC', 3527, 3710, 3, 3531, 3715, 4 FROM dual
UNION ALL SELECT 126, 'SIC', 3526, 3700, 2, 3544, 3760, 3 FROM dual
UNION ALL SELECT 127, 'SIC', 4018, 7210, 3, 4019, 7211, 4 FROM dual
UNION ALL SELECT 128, 'SEC', 2791, 20, 1, 2361, 3231, 4 FROM dual
UNION ALL SELECT 129, 'SIC', 4040, 7300, 2, 4080, 7389, 4 FROM dual
UNION ALL SELECT 130, 'NAICS', 931, 423, 2, 961, 42343, 4 FROM dual
UNION ALL SELECT 131, 'SIC', 3769, 5130, 3, 3773, 5139, 4 FROM dual
UNION ALL SELECT 132, 'NAICS', 1060, 42459, 4, 1059, 424590, 5 FROM dual
UNION ALL SELECT 133, 'SIC', 2847, 700, 2, 2851, 721, 4 FROM dual
UNION ALL SELECT 134, 'NAICS', 2071, 812, 2, 2102, 81299, 4 FROM dual
UNION ALL SELECT 135, 'NAICS', 931, 423, 2, 978, 423620, 5 FROM dual
UNION ALL SELECT 136, 'NAICS', 1969, 712, 2, 1976, 71213, 4 FROM dual
UNION ALL SELECT 137, 'NAICS', 2021, 7223, 3, 2027, 72233, 4 FROM dual
UNION ALL SELECT 138, 'NAICS', 2080, 8122, 3, 2084, 81222, 4 FROM dual
UNION ALL SELECT 139, 'SIC', 4208, 8600, 2, 4211, 8620, 3 FROM dual
UNION ALL SELECT 140, 'SEC', 2385, 3410, 3, 2387, 3412, 4 FROM dual
UNION ALL SELECT 141, 'SIC', 4309, 40, 1, 3714, 4971, 4 FROM dual
UNION ALL SELECT 142, 'NAICS', 1, 11, 1, 124, 115115, 5 FROM dual
UNION ALL SELECT 143, 'SEC', 2384, 3400, 2, 2402, 3490, 3 FROM dual
UNION ALL SELECT 144, 'SEC', 2796, 70, 1, 2718, 7331, 4 FROM dual
UNION ALL SELECT 145, 'SIC', 3384, 3440, 3, 3390, 3448, 4 FROM dual
UNION ALL SELECT 146, 'NAICS', 2103, 813, 2, 2110, 813212, 5 FROM dual
UNION ALL SELECT 147, 'NAICS', 1920, 624, 2, 1926, 624190, 5 FROM dual
UNION ALL SELECT 148, 'NAICS', 930, 42, 1, 1035, 4244, 3 FROM dual
UNION ALL SELECT 149, 'SIC', 4307, 15, 1, 2981, 1799, 4 FROM dual
UNION ALL SELECT 150, 'NAICS', 1726, 561, 2, 1742, 561410, 5 FROM dual
UNION ALL SELECT 151, 'SIC', 3261, 3000, 2, 3279, 3087, 4 FROM dual
UNION ALL SELECT 152, 'SEC', 2280, 2450, 3, 2282, 2452, 4 FROM dual
UNION ALL SELECT 153, 'SEC', 2738, 7800, 2, 2741, 7819, 4 FROM dual
UNION ALL SELECT 154, 'SIC', 3667, 4700, 2, 3675, 4741, 4 FROM dual
UNION ALL SELECT 155, 'SIC', 4308, 20, 1, 3165, 2591, 4 FROM dual
UNION ALL SELECT 156, 'NAICS', 2037, 81, 1, 2048, 81119, 4 FROM dual
UNION ALL SELECT 157, 'NAICS', 1725, 56, 1, 1739, 561330, 5 FROM dual
UNION ALL SELECT 158, 'SIC', 3384, 3440, 3, 3386, 3442, 4 FROM dual
UNION ALL SELECT 159, 'SIC', 3992, 6700, 2, 4005, 6798, 4 FROM dual
UNION ALL SELECT 160, 'NAICS', 930, 42, 1, 945, 42322, 4 FROM dual
UNION ALL SELECT 161, 'SIC', 3715, 5000, 2, 3718, 5013, 4 FROM dual
UNION ALL SELECT 162, 'NAICS', 1717, 54199, 4, 1716, 541990, 5 FROM dual
UNION ALL SELECT 163, 'SEC', 2337, 2900, 2, 2341, 2990, 3 FROM dual
UNION ALL SELECT 164, 'SIC', 4306, 10, 1, 2894, 1060, 3 FROM dual
UNION ALL SELECT 165, 'NAICS', 931, 423, 2, 998, 423840, 5 FROM dual
UNION ALL SELECT 166, 'NAICS', 1, 11, 1, 111, 114111, 5 FROM dual
UNION ALL SELECT 167, 'NAICS', 2, 111, 2, 29, 111331, 5 FROM dual
UNION ALL SELECT 168, 'SIC', 3721, 5020, 3, 3722, 5021, 4 FROM dual
UNION ALL SELECT 169, 'NAICS', 1774, 5617, 3, 1780, 56173, 4 FROM dual
UNION ALL SELECT 170, 'SIC', 3059, 2250, 3, 3062, 2253, 4 FROM dual
UNION ALL SELECT 171, 'NAICS', 2150, 922, 2, 2156, 922130, 5 FROM dual
UNION ALL SELECT 172, 'NAICS', 1513, 523, 2, 1527, 523910, 5 FROM dual
UNION ALL SELECT 173, 'SIC', 3837, 5500, 2, 3847, 5551, 4 FROM dual
UNION ALL SELECT 174, 'NAICS', 156, 2123, 3, 167, 21239, 4 FROM dual
UNION ALL SELECT 175, 'SIC', 3854, 5600, 2, 3866, 5661, 4 FROM dual
UNION ALL SELECT 176, 'SIC', 4302, 9900, 2, 4303, 9990, 3 FROM dual
UNION ALL SELECT 177, 'NAICS', 1962, 71132, 4, 1961, 711320, 5 FROM dual
UNION ALL SELECT 178, 'NAICS', 2058, 8113, 3, 2060, 81131, 4 FROM dual
UNION ALL SELECT 179, 'SIC', 4308, 20, 1, 3417, 3498, 4 FROM dual
UNION ALL SELECT 180, 'SIC', 4309, 40, 1, 3688, 4833, 4 FROM dual
UNION ALL SELECT 181, 'SIC', 4307, 15, 1, 2975, 1790, 3 FROM dual
UNION ALL SELECT 182, 'NAICS', 1419, 512, 2, 1428, 51219, 4 FROM dual
UNION ALL SELECT 183, 'SIC', 2971, 1770, 3, 2972, 1771, 4 FROM dual
UNION ALL SELECT 184, 'SIC', 3301, 3200, 2, 3309, 3240, 3 FROM dual
UNION ALL SELECT 185, 'NAICS', 1726, 561, 2, 1768, 561611, 5 FROM dual
UNION ALL SELECT 186, 'SIC', 3877, 5730, 3, 3878, 5731, 4 FROM dual
UNION ALL SELECT 187, 'SEC', 2713, 7300, 2, 2730, 7377, 4 FROM dual
UNION ALL SELECT 188, 'SIC', 3190, 2700, 2, 3213, 2796, 4 FROM dual
UNION ALL SELECT 189, 'SIC', 4126, 7910, 3, 4127, 7911, 4 FROM dual
UNION ALL SELECT 190, 'NAICS', 156, 2123, 3, 170, 212393, 5 FROM dual
UNION ALL SELECT 191, 'NAICS', 930, 42, 1, 1012, 42394, 4 FROM dual
UNION ALL SELECT 192, 'NAICS', 132, 21, 1, 153, 21229, 4 FROM dual
UNION ALL SELECT 193, 'SIC', 3526, 3700, 2, 3531, 3715, 4 FROM dual
UNION ALL SELECT 194, 'SEC', 2791, 20, 1, 2329, 2842, 4 FROM dual
UNION ALL SELECT 195, 'NAICS', 1570, 531, 2, 1572, 531110, 5 FROM dual
UNION ALL SELECT 196, 'SIC', 4313, 70, 1, 4164, 8063, 4 FROM dual
UNION ALL SELECT 197, 'NAICS', 219, 2371, 3, 221, 23711, 4 FROM dual
UNION ALL SELECT 198, 'NAICS', 1547, 5242, 3, 1549, 52421, 4 FROM dual
UNION ALL SELECT 199, 'NAICS', 1, 11, 1, 92, 112920, 5 FROM dual
UNION ALL SELECT 200, 'NAICS', 2071, 812, 2, 2094, 812910, 5 FROM dual
UNION ALL SELECT 201, 'NAICS', 930, 42, 1, 1039, 42442, 4 FROM dual
UNION ALL SELECT 202, 'NAICS', 1624, 54, 1, 1689, 5418, 3 FROM dual
UNION ALL SELECT 203, 'NAICS', 1859, 6213, 3, 1870, 621399, 5 FROM dual
UNION ALL SELECT 204, 'SIC', 2884, 1000, 2, 2892, 1041, 4 FROM dual
UNION ALL SELECT 205, 'SIC', 4308, 20, 1, 3440, 3546, 4 FROM dual
UNION ALL SELECT 206, 'SIC', 2798, 100, 2, 2824, 191, 4 FROM dual
UNION ALL SELECT 207, 'NAICS', 1943, 711, 2, 1958, 7113, 3 FROM dual
UNION ALL SELECT 208, 'SIC', 4146, 8000, 2, 4157, 8049, 4 FROM dual
UNION ALL SELECT 209, 'NAICS', 98, 113, 2, 105, 1133, 3 FROM dual
UNION ALL SELECT 210, 'NAICS', 1, 11, 1, 44, 111910, 5 FROM dual
UNION ALL SELECT 211, 'NAICS', 1859, 6213, 3, 1868, 62139, 4 FROM dual
UNION ALL SELECT 212, 'SEC', 2664, 6150, 3, 2665, 6153, 4 FROM dual
UNION ALL SELECT 213, 'SEC', 2791, 20, 1, 2497, 3861, 4 FROM dual
UNION ALL SELECT 214, 'NAICS', 1526, 5239, 3, 1534, 523991, 5 FROM dual
UNION ALL SELECT 215, 'NAICS', 930, 42, 1, 971, 423510, 5 FROM dual
UNION ALL SELECT 216, 'NAICS', 1624, 54, 1, 1661, 54142, 4 FROM dual
UNION ALL SELECT 217, 'NAICS', 56, 112, 2, 96, 112990, 5 FROM dual
UNION ALL SELECT 218, 'SIC', 3815, 5300, 2, 3819, 5331, 4 FROM dual
UNION ALL SELECT 219, 'SIC', 3294, 3160, 3, 3295, 3161, 4 FROM dual
UNION ALL SELECT 220, 'NAICS', 1980, 7131, 3, 1984, 71312, 4 FROM dual
UNION ALL SELECT 221, 'NAICS', 138, 212, 2, 152, 212234, 5 FROM dual
UNION ALL SELECT 222, 'NAICS', 1016, 4241, 3, 1017, 424110, 5 FROM dual
UNION ALL SELECT 223, 'SIC', 4153, 8040, 3, 4156, 8043, 4 FROM dual
UNION ALL SELECT 224, 'SIC', 4308, 20, 1, 3377, 3423, 4 FROM dual
UNION ALL SELECT 225, 'SIC', 3886, 5900, 2, 3915, 5995, 4 FROM dual
UNION ALL SELECT 226, 'SIC', 4312, 60, 1, 3939, 6111, 4 FROM dual
UNION ALL SELECT 227, 'NAICS', 931, 423, 2, 992, 423810, 5 FROM dual
UNION ALL SELECT 228, 'NAICS', 930, 42, 1, 975, 4236, 3 FROM dual
UNION ALL SELECT 229, 'SIC', 4306, 10, 1, 2886, 1011, 4 FROM dual
UNION ALL SELECT 230, 'NAICS', 946, 4233, 3, 950, 42332, 4 FROM dual
UNION ALL SELECT 231, 'SIC', 3774, 5140, 3, 3780, 5146, 4 FROM dual
UNION ALL SELECT 232, 'SIC', 3167, 2600, 2, 3187, 2677, 4 FROM dual
UNION ALL SELECT 233, 'SIC', 3620, 4200, 2, 3625, 4215, 4 FROM dual
UNION ALL SELECT 234, 'SIC', 2955, 1700, 2, 2976, 1791, 4 FROM dual
UNION ALL SELECT 235, 'SEC', 2796, 70, 1, 2758, 8060, 3 FROM dual
UNION ALL SELECT 236, 'NAICS', 1467, 5182, 3, 1468, 518210, 5 FROM dual
UNION ALL SELECT 237, 'NAICS', 2010, 72119, 4, 2012, 721199, 5 FROM dual
UNION ALL SELECT 238, 'SIC', 3552, 3800, 2, 3561, 3826, 4 FROM dual
UNION ALL SELECT 239, 'SIC', 3636, 4400, 2, 3637, 4410, 3 FROM dual
UNION ALL SELECT 240, 'SIC', 3886, 5900, 2, 3906, 5963, 4 FROM dual
UNION ALL SELECT 241, 'NAICS', 1851, 621, 2, 1884, 621512, 5 FROM dual
UNION ALL SELECT 242, 'NAICS', 1, 11, 1, 115, 114210, 5 FROM dual
UNION ALL SELECT 243, 'NAICS', 2135, 92, 1, 2175, 92314, 4 FROM dual
UNION ALL SELECT 244, 'NAICS', 153, 21229, 4, 155, 212299, 5 FROM dual
UNION ALL SELECT 245, 'SIC', 4313, 70, 1, 4021, 7213, 4 FROM dual
UNION ALL SELECT 246, 'SIC', 3214, 2800, 2, 3226, 2833, 4 FROM dual
UNION ALL SELECT 247, 'NAICS', 2137, 9211, 3, 2148, 921190, 5 FROM dual
UNION ALL SELECT 248, 'SEC', 2794, 52, 1, 2625, 5621, 4 FROM dual
UNION ALL SELECT 249, 'NAICS', 1624, 54, 1, 1664, 541490, 5 FROM dual
UNION ALL SELECT 250, 'NAICS', 132, 21, 1, 137, 211112, 5 FROM dual
UNION ALL SELECT 251, 'SIC', 4308, 20, 1, 3317, 3261, 4 FROM dual
UNION ALL SELECT 252, 'NAICS', 1506, 5223, 3, 1512, 52239, 4 FROM dual
UNION ALL SELECT 253, 'NAICS', 1871, 6214, 3, 1874, 621420, 5 FROM dual
UNION ALL SELECT 254, 'SIC', 3067, 2260, 3, 3069, 2262, 4 FROM dual
UNION ALL SELECT 255, 'SIC', 3917, 6000, 2, 3931, 6080, 3 FROM dual
UNION ALL SELECT 256, 'NAICS', 2182, 925, 2, 2186, 925120, 5 FROM dual
UNION ALL SELECT 257, 'SIC', 3576, 3900, 2, 3593, 3961, 4 FROM dual
UNION ALL SELECT 258, 'SIC', 4308, 20, 1, 3137, 2440, 3 FROM dual
UNION ALL SELECT 259, 'NAICS', 1625, 541, 2, 1680, 54162, 4 FROM dual
UNION ALL SELECT 260, 'SIC', 4309, 40, 1, 3622, 4212, 4 FROM dual
UNION ALL SELECT 261, 'NAICS', 1792, 562, 2, 1798, 5622, 3 FROM dual
UNION ALL SELECT 262, 'NAICS', 236, 2381, 3, 246, 23815, 4 FROM dual
UNION ALL SELECT 263, 'SEC', 2224, 1380, 3, 2225, 1381, 4 FROM dual
UNION ALL SELECT 264, 'NAICS', 1, 11, 1, 114, 1142, 3 FROM dual
UNION ALL SELECT 265, 'SIC', 4308, 20, 1, 3327, 3275, 4 FROM dual
UNION ALL SELECT 266, 'SIC', 3371, 3400, 2, 3404, 3480, 3 FROM dual
UNION ALL SELECT 267, 'SIC', 4311, 52, 1, 3866, 5661, 4 FROM dual
UNION ALL SELECT 268, 'NAICS', 1625, 541, 2, 1659, 54141, 4 FROM dual
UNION ALL SELECT 269, 'SIC', 4208, 8600, 2, 4215, 8640, 3 FROM dual
UNION ALL SELECT 270, 'NAICS', 1569, 53, 1, 1599, 532210, 5 FROM dual
UNION ALL SELECT 271, 'NAICS', 1726, 561, 2, 1736, 561312, 5 FROM dual
UNION ALL SELECT 272, 'SIC', 4268, 9400, 2, 4269, 9410, 3 FROM dual
UNION ALL SELECT 273, 'NAICS', 1071, 4248, 3, 1075, 42482, 4 FROM dual
UNION ALL SELECT 274, 'SIC', 3309, 3240, 3, 3310, 3241, 4 FROM dual
UNION ALL SELECT 275, 'SIC', 4308, 20, 1, 3288, 3142, 4 FROM dual
UNION ALL SELECT 276, 'SIC', 4312, 60, 1, 3918, 6010, 3 FROM dual
UNION ALL SELECT 277, 'NAICS', 1, 11, 1, 9, 11113, 4 FROM dual
UNION ALL SELECT 278, 'NAICS', 1740, 56133, 4, 1739, 561330, 5 FROM dual
UNION ALL SELECT 279, 'SEC', 2795, 60, 1, 2699, 6530, 3 FROM dual
UNION ALL SELECT 280, 'SIC', 3762, 5100, 2, 3788, 5160, 3 FROM dual
UNION ALL SELECT 281, 'NAICS', 930, 42, 1, 989, 423740, 5 FROM dual
UNION ALL SELECT 282, 'NAICS', 1420, 5121, 3, 1430, 512199, 5 FROM dual
UNION ALL SELECT 283, 'SEC', 2229, 1500, 2, 2231, 1530, 3 FROM dual
UNION ALL SELECT 284, 'SIC', 2882, 970, 3, 2883, 971, 4 FROM dual
UNION ALL SELECT 285, 'SIC', 3301, 3200, 2, 3336, 3299, 4 FROM dual
UNION ALL SELECT 286, 'SIC', 3605, 4100, 2, 3606, 4110, 3 FROM dual
UNION ALL SELECT 287, 'SEC', 2588, 5090, 3, 2589, 5094, 4 FROM dual
UNION ALL SELECT 288, 'NAICS', 167, 21239, 4, 168, 212391, 5 FROM dual
UNION ALL SELECT 289, 'NAICS', 930, 42, 1, 1004, 4239, 3 FROM dual
UNION ALL SELECT 290, 'SIC', 4305, 1, 1, 2806, 131, 4 FROM dual
UNION ALL SELECT 291, 'SIC', 2858, 750, 3, 2860, 752, 4 FROM dual
UNION ALL SELECT 292, 'NAICS', 205, 23, 1, 229, 2373, 3 FROM dual
UNION ALL SELECT 293, 'SIC', 3468, 3580, 3, 3472, 3586, 4 FROM dual
UNION ALL SELECT 294, 'NAICS', 931, 423, 2, 1013, 423990, 5 FROM dual
UNION ALL SELECT 295, 'SEC', 2434, 3600, 2, 2444, 3651, 4 FROM dual
UNION ALL SELECT 296, 'SIC', 4308, 20, 1, 3184, 2674, 4 FROM dual
UNION ALL SELECT 297, 'SIC', 3083, 2300, 2, 3105, 2369, 4 FROM dual
UNION ALL SELECT 298, 'SIC', 3925, 6030, 3, 3926, 6035, 4 FROM dual
UNION ALL SELECT 299, 'SIC', 3000, 2040, 3, 3002, 2043, 4 FROM dual
UNION ALL SELECT 300, 'SIC', 3371, 3400, 2, 3379, 3429, 4 FROM dual
UNION ALL SELECT 301, 'SEC', 2791, 20, 1, 2390, 3433, 4 FROM dual
UNION ALL SELECT 302, 'SIC', 3886, 5900, 2, 3911, 5990, 3 FROM dual
UNION ALL SELECT 303, 'SIC', 4313, 70, 1, 4122, 7833, 4 FROM dual
UNION ALL SELECT 304, 'SEC', 2795, 60, 1, 2658, 6099, 4 FROM dual
UNION ALL SELECT 305, 'NAICS', 1850, 62, 1, 1891, 62199, 4 FROM dual
UNION ALL SELECT 306, 'SIC', 4308, 20, 1, 3131, 2430, 3 FROM dual
UNION ALL SELECT 307, 'NAICS', 1859, 6213, 3, 1861, 62131, 4 FROM dual
UNION ALL SELECT 308, 'SIC', 3564, 3840, 3, 3565, 3841, 4 FROM dual
UNION ALL SELECT 309, 'SIC', 4055, 7340, 3, 4056, 7342, 4 FROM dual
UNION ALL SELECT 310, 'NAICS', 205, 23, 1, 235, 238, 2 FROM dual
UNION ALL SELECT 311, 'NAICS', 1640, 5413, 3, 1646, 54133, 4 FROM dual
UNION ALL SELECT 312, 'NAICS', 1554, 525, 2, 1568, 52599, 4 FROM dual
UNION ALL SELECT 313, 'SEC', 2791, 20, 1, 2421, 3564, 4 FROM dual
UNION ALL SELECT 314, 'NAICS', 1402, 51, 1, 1411, 511140, 5 FROM dual
UNION ALL SELECT 315, 'SEC', 2683, 6330, 3, 2684, 6331, 4 FROM dual
UNION ALL SELECT 316, 'SIC', 4305, 1, 1, 2801, 112, 4 FROM dual
UNION ALL SELECT 317, 'NAICS', 1813, 611, 2, 1822, 61131, 4 FROM dual
UNION ALL SELECT 318, 'SEC', 2753, 8000, 2, 2762, 8080, 3 FROM dual
UNION ALL SELECT 319, 'NAICS', 2037, 81, 1, 2086, 812310, 5 FROM dual
UNION ALL SELECT 320, 'NAICS', 2003, 72, 1, 2012, 721199, 5 FROM dual
UNION ALL SELECT 321, 'SIC', 4308, 20, 1, 3248, 2892, 4 FROM dual
UNION ALL SELECT 322, 'SIC', 3419, 3500, 2, 3476, 3593, 4 FROM dual
UNION ALL SELECT 323, 'NAICS', 1725, 56, 1, 1804, 5629, 3 FROM dual
UNION ALL SELECT 324, 'SEC', 2791, 20, 1, 2300, 2670, 3 FROM dual
UNION ALL SELECT 325, 'NAICS', 1500, 52229, 4, 1501, 522291, 5 FROM dual
UNION ALL SELECT 326, 'NAICS', 1850, 62, 1, 1883, 621511, 5 FROM dual
UNION ALL SELECT 327, 'SIC', 4308, 20, 1, 3101, 2350, 3 FROM dual
UNION ALL SELECT 328, 'SIC', 4313, 70, 1, 4243, 8811, 4 FROM dual
UNION ALL SELECT 329, 'SIC', 2939, 1500, 2, 2947, 1542, 4 FROM dual
UNION ALL SELECT 330, 'SIC', 3992, 6700, 2, 3993, 6710, 3 FROM dual
UNION ALL SELECT 331, 'NAICS', 1452, 517, 2, 1459, 5174, 3 FROM dual
UNION ALL SELECT 332, 'NAICS', 1015, 424, 2, 1072, 424810, 5 FROM dual
UNION ALL SELECT 333, 'NAICS', 1549, 52421, 4, 1548, 524210, 5 FROM dual
UNION ALL SELECT 334, 'NAICS', 1624, 54, 1, 1645, 541330, 5 FROM dual
UNION ALL SELECT 335, 'SEC', 2539, 4800, 2, 2541, 4812, 4 FROM dual
UNION ALL SELECT 336, 'NAICS', 2151, 9221, 3, 2163, 92216, 4 FROM dual
UNION ALL SELECT 337, 'SIC', 3215, 2810, 3, 3217, 2813, 4 FROM dual
UNION ALL SELECT 338, 'NAICS', 1657, 5414, 3, 1658, 541410, 5 FROM dual
UNION ALL SELECT 339, 'NAICS', 1726, 561, 2, 1759, 561510, 5 FROM dual
UNION ALL SELECT 340, 'NAICS', 205, 23, 1, 208, 23611, 4 FROM dual
UNION ALL SELECT 341, 'NAICS', 1089, 425, 2, 1094, 42512, 4 FROM dual
UNION ALL SELECT 342, 'NAICS', 1920, 624, 2, 1936, 6243, 3 FROM dual
UNION ALL SELECT 343, 'NAICS', 2103, 813, 2, 2123, 813920, 5 FROM dual
UNION ALL SELECT 344, 'SIC', 2825, 200, 2, 2840, 270, 3 FROM dual
UNION ALL SELECT 345, 'SEC', 2796, 70, 1, 2744, 7829, 4 FROM dual
UNION ALL SELECT 346, 'SIC', 3419, 3500, 2, 3425, 3524, 4 FROM dual
UNION ALL SELECT 347, 'SEC', 2790, 15, 1, 2230, 1520, 3 FROM dual
UNION ALL SELECT 348, 'SIC', 3012, 2060, 3, 3013, 2061, 4 FROM dual
UNION ALL SELECT 349, 'NAICS', 1065, 42469, 4, 1064, 424690, 5 FROM dual
UNION ALL SELECT 350, 'NAICS', 1804, 5629, 3, 1808, 56292, 4 FROM dual
UNION ALL SELECT 351, 'SIC', 4313, 70, 1, 4044, 7313, 4 FROM dual
UNION ALL SELECT 352, 'NAICS', 2020, 722, 2, 2030, 72241, 4 FROM dual
UNION ALL SELECT 353, 'SIC', 4305, 1, 1, 2882, 970, 3 FROM dual
UNION ALL SELECT 354, 'SIC', 3419, 3500, 2, 3449, 3556, 4 FROM dual
UNION ALL SELECT 355, 'NAICS', 3, 1111, 3, 5, 11111, 4 FROM dual
UNION ALL SELECT 356, 'SIC', 3083, 2300, 2, 3113, 2387, 4 FROM dual
UNION ALL SELECT 357, 'SIC', 3667, 4700, 2, 3670, 4725, 4 FROM dual
UNION ALL SELECT 358, 'NAICS', 218, 237, 2, 228, 23721, 4 FROM dual
UNION ALL SELECT 359, 'NAICS', 1470, 519, 2, 1479, 51919, 4 FROM dual
UNION ALL SELECT 360, 'SIC', 4313, 70, 1, 4231, 8731, 4 FROM dual
UNION ALL SELECT 361, 'NAICS', 2113, 81331, 4, 2115, 813312, 5 FROM dual
UNION ALL SELECT 362, 'SIC', 3762, 5100, 2, 3767, 5120, 3 FROM dual
UNION ALL SELECT 363, 'NAICS', 1725, 56, 1, 1747, 56143, 4 FROM dual
UNION ALL SELECT 364, 'NAICS', 930, 42, 1, 932, 4231, 3 FROM dual
UNION ALL SELECT 365, 'SEC', 2371, 3300, 2, 2379, 3341, 4 FROM dual
UNION ALL SELECT 366, 'NAICS', 273, 2389, 3, 274, 238910, 5 FROM dual
UNION ALL SELECT 367, 'NAICS', 1998, 71394, 4, 1997, 713940, 5 FROM dual
UNION ALL SELECT 368, 'NAICS', 1990, 7139, 3, 1993, 713920, 5 FROM dual
UNION ALL SELECT 369, 'SEC', 2677, 6300, 2, 2682, 6324, 4 FROM dual
UNION ALL SELECT 370, 'NAICS', 102, 1132, 3, 104, 11321, 4 FROM dual
UNION ALL SELECT 371, 'NAICS', 931, 423, 2, 964, 423450, 5 FROM dual
UNION ALL SELECT 372, 'NAICS', 1015, 424, 2, 1030, 42432, 4 FROM dual
UNION ALL SELECT 373, 'NAICS', 180, 22, 1, 199, 221310, 5 FROM dual
UNION ALL SELECT 374, 'SIC', 3891, 5930, 3, 3892, 5932, 4 FROM dual
UNION ALL SELECT 375, 'NAICS', 930, 42, 1, 1032, 42433, 4 FROM dual
UNION ALL SELECT 376, 'NAICS', 1471, 5191, 3, 1478, 519190, 5 FROM dual
UNION ALL SELECT 377, 'NAICS', 982, 4237, 3, 984, 42371, 4 FROM dual
UNION ALL SELECT 378, 'SIC', 3131, 2430, 3, 3132, 2431, 4 FROM dual
UNION ALL SELECT 379, 'SIC', 3489, 3630, 3, 3493, 3634, 4 FROM dual
UNION ALL SELECT 380, 'SIC', 4312, 60, 1, 3994, 6712, 4 FROM dual
UNION ALL SELECT 381, 'SIC', 4313, 70, 1, 4025, 7218, 4 FROM dual
UNION ALL SELECT 382, 'SIC', 3020, 2070, 3, 3023, 2076, 4 FROM dual
UNION ALL SELECT 383, 'SIC', 4040, 7300, 2, 4078, 7383, 4 FROM dual
UNION ALL SELECT 384, 'NAICS', 1076, 4249, 3, 1086, 42495, 4 FROM dual
UNION ALL SELECT 385, 'NAICS', 2108, 81321, 4, 2109, 813211, 5 FROM dual
UNION ALL SELECT 386, 'SIC', 4307, 15, 1, 2974, 1781, 4 FROM dual
UNION ALL SELECT 387, 'SIC', 3050, 2200, 2, 3070, 2269, 4 FROM dual
UNION ALL SELECT 388, 'SIC', 4311, 52, 1, 3876, 5722, 4 FROM dual
UNION ALL SELECT 389, 'SIC', 4309, 40, 1, 3614, 4141, 4 FROM dual
UNION ALL SELECT 390, 'SIC', 4308, 20, 1, 3098, 2340, 3 FROM dual
UNION ALL SELECT 391, 'SIC', 3893, 5940, 3, 3897, 5944, 4 FROM dual
UNION ALL SELECT 392, 'SIC', 2909, 1300, 2, 2913, 1321, 4 FROM dual
UNION ALL SELECT 393, 'NAICS', 1611, 53231, 4, 1610, 532310, 5 FROM dual
UNION ALL SELECT 394, 'NAICS', 2, 111, 2, 53, 111991, 5 FROM dual
UNION ALL SELECT 395, 'SIC', 3048, 2140, 3, 3049, 2141, 4 FROM dual
UNION ALL SELECT 396, 'NAICS', 1812, 61, 1, 1840, 61162, 4 FROM dual
UNION ALL SELECT 397, 'SEC', 2789, 10, 1, 2225, 1381, 4 FROM dual
UNION ALL SELECT 398, 'NAICS', 62, 11212, 4, 61, 112120, 5 FROM dual
UNION ALL SELECT 399, 'SEC', 2791, 20, 1, 2444, 3651, 4 FROM dual
UNION ALL SELECT 400, 'SIC', 4308, 20, 1, 3332, 3292, 4 FROM dual
UNION ALL SELECT 401, 'SIC', 3693, 4900, 2, 3708, 4952, 4 FROM dual
UNION ALL SELECT 402, 'SIC', 3886, 5900, 2, 3903, 5960, 3 FROM dual
UNION ALL SELECT 403, 'SEC', 2355, 3200, 2, 2370, 3290, 3 FROM dual
UNION ALL SELECT 404, 'SIC', 4308, 20, 1, 3181, 2671, 4 FROM dual
UNION ALL SELECT 405, 'SIC', 3083, 2300, 2, 3119, 2394, 4 FROM dual
UNION ALL SELECT 406, 'SEC', 2791, 20, 1, 2499, 3873, 4 FROM dual
UNION ALL SELECT 407, 'NAICS', 1402, 51, 1, 1441, 51229, 4 FROM dual
UNION ALL SELECT 408, 'SIC', 4308, 20, 1, 3306, 3229, 4 FROM dual
UNION ALL SELECT 409, 'SEC', 2262, 2200, 2, 2267, 2250, 3 FROM dual
UNION ALL SELECT 410, 'SEC', 2793, 50, 1, 2603, 5180, 3 FROM dual
UNION ALL SELECT 411, 'SIC', 3576, 3900, 2, 3579, 3914, 4 FROM dual
UNION ALL SELECT 412, 'NAICS', 932, 4231, 3, 935, 423120, 5 FROM dual
UNION ALL SELECT 413, 'NAICS', 1725, 56, 1, 1746, 561422, 5 FROM dual
UNION ALL SELECT 414, 'NAICS', 1452, 517, 2, 1461, 51741, 4 FROM dual
UNION ALL SELECT 415, 'SIC', 4308, 20, 1, 2982, 2000, 2 FROM dual
UNION ALL SELECT 416, 'SIC', 3555, 3820, 3, 3563, 3829, 4 FROM dual
UNION ALL SELECT 417, 'SIC', 3461, 3570, 3, 3463, 3572, 4 FROM dual
UNION ALL SELECT 418, 'SEC', 2795, 60, 1, 2702, 6552, 4 FROM dual
UNION ALL SELECT 419, 'NAICS', 2080, 8122, 3, 2082, 81221, 4 FROM dual
UNION ALL SELECT 420, 'SEC', 2794, 52, 1, 2627, 5651, 4 FROM dual
UNION ALL SELECT 421, 'SIC', 4312, 60, 1, 3952, 6221, 4 FROM dual
UNION ALL SELECT 422, 'SIC', 4313, 70, 1, 4229, 8721, 4 FROM dual
UNION ALL SELECT 423, 'SIC', 3180, 2670, 3, 3189, 2679, 4 FROM dual
UNION ALL SELECT 424, 'SIC', 4040, 7300, 2, 4051, 7334, 4 FROM dual
UNION ALL SELECT 425, 'SIC', 3634, 4310, 3, 3635, 4311, 4 FROM dual
UNION ALL SELECT 426, 'SIC', 4308, 20, 1, 3444, 3550, 3 FROM dual
UNION ALL SELECT 427, 'SIC', 4313, 70, 1, 4023, 7216, 4 FROM dual
UNION ALL SELECT 428, 'NAICS', 1, 11, 1, 49, 11193, 4 FROM dual
UNION ALL SELECT 429, 'SEC', 2791, 20, 1, 2261, 2111, 4 FROM dual
UNION ALL SELECT 430, 'NAICS', 1876, 62149, 4, 1879, 621493, 5 FROM dual
UNION ALL SELECT 431, 'SIC', 3854, 5600, 2, 3857, 5620, 3 FROM dual
UNION ALL SELECT 432, 'SEC', 2659, 6100, 2, 4320, 6170, 3 FROM dual
UNION ALL SELECT 433, 'NAICS', 1084, 42494, 4, 1083, 424940, 5 FROM dual
UNION ALL SELECT 434, 'SIC', 3167, 2600, 2, 3175, 2652, 4 FROM dual
UNION ALL SELECT 435, 'NAICS', 205, 23, 1, 269, 238350, 5 FROM dual
UNION ALL SELECT 436, 'SIC', 2948, 1600, 2, 2954, 1629, 4 FROM dual
UNION ALL SELECT 437, 'NAICS', 1471, 5191, 3, 1477, 51913, 4 FROM dual
UNION ALL SELECT 438, 'NAICS', 1871, 6214, 3, 1877, 621491, 5 FROM dual
UNION ALL SELECT 439, 'NAICS', 1836, 6116, 3, 1839, 611620, 5 FROM dual
UNION ALL SELECT 440, 'NAICS', 1851, 621, 2, 1886, 621610, 5 FROM dual
UNION ALL SELECT 441, 'SIC', 4247, 9100, 2, 4249, 9111, 4 FROM dual
UNION ALL SELECT 442, 'SIC', 3511, 3670, 3, 3513, 3672, 4 FROM dual
UNION ALL SELECT 443, 'SIC', 4308, 20, 1, 3488, 3629, 4 FROM dual
UNION ALL SELECT 444, 'NAICS', 1035, 4244, 3, 1053, 42449, 4 FROM dual
UNION ALL SELECT 445, 'NAICS', 1624, 54, 1, 1683, 5417, 3 FROM dual
UNION ALL SELECT 446, 'SIC', 4312, 60, 1, 3978, 6510, 3 FROM dual
UNION ALL SELECT 447, 'SIC', 2955, 1700, 2, 2957, 1711, 4 FROM dual
UNION ALL SELECT 448, 'SEC', 2794, 52, 1, 2646, 5960, 3 FROM dual
UNION ALL SELECT 449, 'SEC', 2791, 20, 1, 2247, 2030, 3 FROM dual
UNION ALL SELECT 450, 'NAICS', 117, 115, 2, 125, 115116, 5 FROM dual
UNION ALL SELECT 451, 'SIC', 3301, 3200, 2, 3302, 3210, 3 FROM dual
UNION ALL SELECT 452, 'SIC', 4308, 20, 1, 3129, 2426, 4 FROM dual
UNION ALL SELECT 453, 'SIC', 3419, 3500, 2, 3462, 3571, 4 FROM dual
UNION ALL SELECT 454, 'SIC', 3804, 5200, 2, 3812, 5261, 4 FROM dual
UNION ALL SELECT 455, 'NAICS', 1624, 54, 1, 1634, 5412, 3 FROM dual
UNION ALL SELECT 456, 'NAICS', 1, 11, 1, 58, 11211, 4 FROM dual
UNION ALL SELECT 457, 'SIC', 4314, 90, 1, 4255, 9191, 4 FROM dual
UNION ALL SELECT 458, 'SIC', 4082, 7510, 3, 4083, 7513, 4 FROM dual
UNION ALL SELECT 459, 'SEC', 2791, 20, 1, 2397, 3451, 4 FROM dual
UNION ALL SELECT 460, 'NAICS', 1741, 5614, 3, 1750, 561440, 5 FROM dual
UNION ALL SELECT 461, 'SEC', 2403, 3500, 2, 2409, 3531, 4 FROM dual
UNION ALL SELECT 462, 'NAICS', 56, 112, 2, 85, 11251, 4 FROM dual
UNION ALL SELECT 463, 'NAICS', 930, 42, 1, 940, 42314, 4 FROM dual
UNION ALL SELECT 464, 'NAICS', 1547, 5242, 3, 1553, 524298, 5 FROM dual
UNION ALL SELECT 465, 'NAICS', 2093, 8129, 3, 2098, 812922, 5 FROM dual
UNION ALL SELECT 466, 'SIC', 2798, 100, 2, 2804, 119, 4 FROM dual
UNION ALL SELECT 467, 'SIC', 2909, 1300, 2, 2911, 1311, 4 FROM dual
UNION ALL SELECT 468, 'NAICS', 2137, 9211, 3, 2147, 92115, 4 FROM dual
UNION ALL SELECT 469, 'SEC', 2657, 6090, 3, 2658, 6099, 4 FROM dual
UNION ALL SELECT 470, 'SIC', 4308, 20, 1, 3313, 3253, 4 FROM dual
UNION ALL SELECT 471, 'SIC', 4310, 50, 1, 3716, 5010, 3 FROM dual
UNION ALL SELECT 472, 'SIC', 3055, 2230, 3, 3056, 2231, 4 FROM dual
UNION ALL SELECT 473, 'SIC', 4309, 40, 1, 3639, 4420, 3 FROM dual
UNION ALL SELECT 474, 'SIC', 3083, 2300, 2, 3114, 2389, 4 FROM dual
UNION ALL SELECT 475, 'NAICS', 138, 212, 2, 140, 21211, 4 FROM dual
UNION ALL SELECT 476, 'NAICS', 1066, 4247, 3, 1069, 424720, 5 FROM dual
UNION ALL SELECT 477, 'SEC', 2791, 20, 1, 2265, 2220, 3 FROM dual
UNION ALL SELECT 478, 'SIC', 3993, 6710, 3, 3995, 6719, 4 FROM dual
UNION ALL SELECT 479, 'NAICS', 1850, 62, 1, 1859, 6213, 3 FROM dual
UNION ALL SELECT 480, 'SIC', 4269, 9410, 3, 4270, 9411, 4 FROM dual
UNION ALL SELECT 481, 'NAICS', 1569, 53, 1, 1614, 532411, 5 FROM dual
UNION ALL SELECT 482, 'NAICS', 1625, 541, 2, 1705, 54189, 4 FROM dual
UNION ALL SELECT 483, 'SIC', 4228, 8720, 3, 4229, 8721, 4 FROM dual
UNION ALL SELECT 484, 'SEC', 2791, 20, 1, 2476, 3800, 2 FROM dual
UNION ALL SELECT 485, 'SEC', 2680, 6320, 3, 2681, 6321, 4 FROM dual
UNION ALL SELECT 486, 'SIC', 2918, 1400, 2, 2920, 1411, 4 FROM dual
UNION ALL SELECT 487, 'SIC', 3696, 4920, 3, 3700, 4925, 4 FROM dual
UNION ALL SELECT 488, 'NAICS', 1871, 6214, 3, 1873, 62141, 4 FROM dual
UNION ALL SELECT 489, 'SIC', 4308, 20, 1, 3268, 3053, 4 FROM dual
UNION ALL SELECT 490, 'SIC', 4308, 20, 1, 3305, 3221, 4 FROM dual
UNION ALL SELECT 491, 'SIC', 3083, 2300, 2, 3090, 2325, 4 FROM dual
UNION ALL SELECT 492, 'SIC', 3167, 2600, 2, 3183, 2673, 4 FROM dual
UNION ALL SELECT 493, 'NAICS', 1591, 532, 2, 1613, 53241, 4 FROM dual
UNION ALL SELECT 494, 'SIC', 3693, 4900, 2, 3698, 4923, 4 FROM dual
UNION ALL SELECT 495, 'NAICS', 1625, 541, 2, 1662, 541430, 5 FROM dual
UNION ALL SELECT 496, 'NAICS', 2167, 9231, 3, 2172, 923130, 5 FROM dual
UNION ALL SELECT 497, 'SIC', 3266, 3050, 3, 3267, 3052, 4 FROM dual
UNION ALL SELECT 498, 'NAICS', 1443, 5151, 3, 1444, 51511, 4 FROM dual
UNION ALL SELECT 499, 'NAICS', 1928, 6242, 3, 1933, 624229, 5 FROM dual
UNION ALL SELECT 500, 'SIC', 3338, 3310, 3, 3343, 3317, 4 FROM dual
UNION ALL SELECT 501, 'SEC', 2792, 40, 1, 2565, 4953, 4 FROM dual
UNION ALL SELECT 502, 'NAICS', 2037, 81, 1, 2134, 81411, 4 FROM dual
UNION ALL SELECT 503, 'NAICS', 1881, 6215, 3, 1883, 621511, 5 FROM dual
UNION ALL SELECT 504, 'NAICS', 1402, 51, 1, 1418, 51121, 4 FROM dual
UNION ALL SELECT 505, 'SEC', 2791, 20, 1, 2285, 2511, 4 FROM dual
UNION ALL SELECT 506, 'SIC', 3384, 3440, 3, 3385, 3441, 4 FROM dual
UNION ALL SELECT 507, 'NAICS', 1598, 5322, 3, 1599, 532210, 5 FROM dual
UNION ALL SELECT 508, 'SIC', 4308, 20, 1, 3163, 2542, 4 FROM dual
UNION ALL SELECT 509, 'SEC', 2649, 6000, 2, 2652, 6022, 4 FROM dual
UNION ALL SELECT 510, 'SIC', 4100, 7600, 2, 4102, 7622, 4 FROM dual
UNION ALL SELECT 511, 'NAICS', 182, 2211, 3, 185, 221112, 5 FROM dual
UNION ALL SELECT 512, 'NAICS', 1591, 532, 2, 1606, 532291, 5 FROM dual
UNION ALL SELECT 513, 'NAICS', 1943, 711, 2, 1967, 711510, 5 FROM dual
UNION ALL SELECT 514, 'NAICS', 1526, 5239, 3, 1535, 523999, 5 FROM dual
UNION ALL SELECT 515, 'SEC', 2568, 5000, 2, 2574, 5040, 3 FROM dual
UNION ALL SELECT 516, 'SIC', 4314, 90, 1, 4287, 9620, 3 FROM dual
UNION ALL SELECT 517, 'SIC', 4308, 20, 1, 3513, 3672, 4 FROM dual
UNION ALL SELECT 518, 'NAICS', 2135, 92, 1, 2178, 924110, 5 FROM dual
UNION ALL SELECT 519, 'SIC', 4311, 52, 1, 3857, 5620, 3 FROM dual
UNION ALL SELECT 520, 'NAICS', 1554, 525, 2, 1558, 525120, 5 FROM dual
UNION ALL SELECT 521, 'NAICS', 1, 11, 1, 94, 112930, 5 FROM dual
UNION ALL SELECT 522, 'NAICS', 98, 113, 2, 103, 113210, 5 FROM dual
UNION ALL SELECT 523, 'NAICS', 931, 423, 2, 985, 423720, 5 FROM dual
UNION ALL SELECT 524, 'NAICS', 1969, 712, 2, 1972, 71211, 4 FROM dual
UNION ALL SELECT 525, 'SIC', 3985, 6530, 3, 3986, 6531, 4 FROM dual
UNION ALL SELECT 526, 'SIC', 4308, 20, 1, 3554, 3812, 4 FROM dual
UNION ALL SELECT 527, 'SIC', 3077, 2290, 3, 3080, 2297, 4 FROM dual
UNION ALL SELECT 528, 'NAICS', 1485, 522, 2, 1488, 52211, 4 FROM dual
UNION ALL SELECT 529, 'NAICS', 1583, 5313, 3, 1590, 53139, 4 FROM dual
UNION ALL SELECT 530, 'SIC', 4075, 7380, 3, 4078, 7383, 4 FROM dual
UNION ALL SELECT 531, 'SIC', 3137, 2440, 3, 3139, 2448, 4 FROM dual
UNION ALL SELECT 532, 'SIC', 4018, 7210, 3, 4020, 7212, 4 FROM dual
UNION ALL SELECT 533, 'SIC', 3050, 2200, 2, 3079, 2296, 4 FROM dual
UNION ALL SELECT 534, 'SIC', 4311, 52, 1, 3883, 5810, 3 FROM dual
UNION ALL SELECT 535, 'SIC', 2982, 2000, 2, 3001, 2041, 4 FROM dual
UNION ALL SELECT 536, 'SIC', 4309, 40, 1, 3689, 4840, 3 FROM dual
UNION ALL SELECT 537, 'NAICS', 1536, 524, 2, 1538, 52411, 4 FROM dual
UNION ALL SELECT 538, 'SIC', 3480, 3600, 2, 3504, 3650, 3 FROM dual
UNION ALL SELECT 539, 'NAICS', 1852, 6211, 3, 1854, 621111, 5 FROM dual
UNION ALL SELECT 540, 'SIC', 4309, 40, 1, 3658, 4520, 3 FROM dual
UNION ALL SELECT 541, 'NAICS', 1979, 713, 2, 1981, 713110, 5 FROM dual
UNION ALL SELECT 542, 'NAICS', 1812, 61, 1, 1817, 6112, 3 FROM dual
UNION ALL SELECT 543, 'SIC', 4306, 10, 1, 2884, 1000, 2 FROM dual
UNION ALL SELECT 544, 'NAICS', 1725, 56, 1, 1726, 561, 2 FROM dual
UNION ALL SELECT 545, 'SIC', 3282, 3100, 2, 3287, 3140, 3 FROM dual
UNION ALL SELECT 546, 'NAICS', 1640, 5413, 3, 1642, 54131, 4 FROM dual
UNION ALL SELECT 547, 'NAICS', 205, 23, 1, 267, 238340, 5 FROM dual
UNION ALL SELECT 548, 'NAICS', 930, 42, 1, 1038, 424420, 5 FROM dual
UNION ALL SELECT 549, 'SIC', 4308, 20, 1, 3343, 3317, 4 FROM dual
UNION ALL SELECT 550, 'NAICS', 2126, 81393, 4, 2125, 813930, 5 FROM dual
UNION ALL SELECT 551, 'NAICS', 2038, 811, 2, 2048, 81119, 4 FROM dual
UNION ALL SELECT 552, 'NAICS', 1741, 5614, 3, 1753, 56145, 4 FROM dual
UNION ALL SELECT 553, 'NAICS', 1035, 4244, 3, 1040, 424430, 5 FROM dual
UNION ALL SELECT 554, 'NAICS', 1809, 56299, 4, 1811, 562998, 5 FROM dual
UNION ALL SELECT 555, 'NAICS', 1725, 56, 1, 1798, 5622, 3 FROM dual
UNION ALL SELECT 556, 'NAICS', 1882, 62151, 4, 1883, 621511, 5 FROM dual
UNION ALL SELECT 557, 'SEC', 2670, 6200, 2, 2676, 6282, 4 FROM dual
UNION ALL SELECT 558, 'SIC', 3419, 3500, 2, 3460, 3569, 4 FROM dual
UNION ALL SELECT 559, 'NAICS', 1453, 5171, 3, 1454, 517110, 5 FROM dual
UNION ALL SELECT 560, 'SIC', 4308, 20, 1, 3263, 3011, 4 FROM dual
UNION ALL SELECT 561, 'SIC', 3252, 2900, 2, 3258, 2990, 3 FROM dual
UNION ALL SELECT 562, 'SIC', 3680, 4800, 2, 3691, 4890, 3 FROM dual
UNION ALL SELECT 563, 'SIC', 4311, 52, 1, 3844, 5540, 3 FROM dual
UNION ALL SELECT 564, 'NAICS', 1480, 52, 1, 1529, 523920, 5 FROM dual
UNION ALL SELECT 565, 'SEC', 2774, 8700, 2, 2781, 8741, 4 FROM dual
UNION ALL SELECT 566, 'SIC', 3059, 2250, 3, 3064, 2257, 4 FROM dual
UNION ALL SELECT 567, 'NAICS', 2072, 8121, 3, 2075, 812112, 5 FROM dual
UNION ALL SELECT 568, 'NAICS', 1699, 54185, 4, 1698, 541850, 5 FROM dual
UNION ALL SELECT 569, 'NAICS', 1634, 5412, 3, 1637, 541213, 5 FROM dual
UNION ALL SELECT 570, 'NAICS', 235, 238, 2, 251, 238190, 5 FROM dual
UNION ALL SELECT 571, 'NAICS', 982, 4237, 3, 986, 42372, 4 FROM dual
UNION ALL SELECT 572, 'NAICS', 1591, 532, 2, 1612, 5324, 3 FROM dual
UNION ALL SELECT 573, 'SIC', 3330, 3290, 3, 3331, 3291, 4 FROM dual
UNION ALL SELECT 574, 'NAICS', 1015, 424, 2, 1065, 42469, 4 FROM dual
UNION ALL SELECT 575, 'NAICS', 2037, 81, 1, 2114, 813311, 5 FROM dual
UNION ALL SELECT 576, 'SIC', 3654, 4500, 2, 3657, 4513, 4 FROM dual
UNION ALL SELECT 577, 'SEC', 2302, 2700, 2, 2318, 2790, 3 FROM dual
UNION ALL SELECT 578, 'NAICS', 1480, 52, 1, 1488, 52211, 4 FROM dual
UNION ALL SELECT 579, 'SIC', 4310, 50, 1, 3725, 5031, 4 FROM dual
UNION ALL SELECT 580, 'SEC', 2384, 3400, 2, 2389, 3430, 3 FROM dual
UNION ALL SELECT 581, 'SEC', 2791, 20, 1, 2352, 3089, 4 FROM dual
UNION ALL SELECT 582, 'SIC', 3869, 5700, 2, 3874, 5719, 4 FROM dual
UNION ALL SELECT 583, 'SEC', 2240, 2000, 2, 2248, 2033, 4 FROM dual
UNION ALL SELECT 584, 'NAICS', 1625, 541, 2, 1671, 541519, 5 FROM dual
UNION ALL SELECT 585, 'SIC', 3886, 5900, 2, 3904, 5961, 4 FROM dual
UNION ALL SELECT 586, 'NAICS', 1725, 56, 1, 1803, 562219, 5 FROM dual
UNION ALL SELECT 587, 'SIC', 2884, 1000, 2, 2894, 1060, 3 FROM dual
UNION ALL SELECT 588, 'SIC', 4313, 70, 1, 4016, 7041, 4 FROM dual
UNION ALL SELECT 589, 'NAICS', 205, 23, 1, 274, 238910, 5 FROM dual
UNION ALL SELECT 590, 'NAICS', 2108, 81321, 4, 2111, 813219, 5 FROM dual
UNION ALL SELECT 591, 'NAICS', 2037, 81, 1, 2079, 812199, 5 FROM dual
UNION ALL SELECT 592, 'NAICS', 28, 11133, 4, 31, 111333, 5 FROM dual
UNION ALL SELECT 593, 'NAICS', 1734, 56131, 4, 1736, 561312, 5 FROM dual
UNION ALL SELECT 594, 'SIC', 3489, 3630, 3, 3495, 3639, 4 FROM dual
UNION ALL SELECT 595, 'NAICS', 1726, 561, 2, 1755, 561491, 5 FROM dual
UNION ALL SELECT 596, 'NAICS', 2058, 8113, 3, 2059, 811310, 5 FROM dual
UNION ALL SELECT 597, 'SIC', 4309, 40, 1, 3702, 4931, 4 FROM dual
UNION ALL SELECT 598, 'SIC', 4081, 7500, 2, 4098, 7542, 4 FROM dual
UNION ALL SELECT 599, 'NAICS', 1513, 523, 2, 1514, 5231, 3 FROM dual
UNION ALL SELECT 600, 'SIC', 4309, 40, 1, 3672, 4730, 3 FROM dual
UNION ALL SELECT 601, 'NAICS', 2038, 811, 2, 2068, 81143, 4 FROM dual
UNION ALL SELECT 602, 'SEC', 2794, 52, 1, 2606, 5210, 3 FROM dual
UNION ALL SELECT 603, 'SIC', 4313, 70, 1, 4242, 8810, 3 FROM dual
UNION ALL SELECT 604, 'SEC', 4324, 6190, 3, 4325, 6199, 4 FROM dual
UNION ALL SELECT 605, 'SIC', 4308, 20, 1, 3017, 2066, 4 FROM dual
UNION ALL SELECT 606, 'SEC', 2793, 50, 1, 2587, 5084, 4 FROM dual
UNION ALL SELECT 607, 'NAICS', 1725, 56, 1, 1750, 561440, 5 FROM dual
UNION ALL SELECT 608, 'NAICS', 2038, 811, 2, 2058, 8113, 3 FROM dual
UNION ALL SELECT 609, 'SIC', 4308, 20, 1, 3582, 3931, 4 FROM dual
UNION ALL SELECT 610, 'SIC', 3355, 3350, 3, 3358, 3354, 4 FROM dual
UNION ALL SELECT 611, 'SIC', 4308, 20, 1, 3418, 3499, 4 FROM dual
UNION ALL SELECT 612, 'NAICS', 1850, 62, 1, 1885, 6216, 3 FROM dual
UNION ALL SELECT 613, 'SIC', 4308, 20, 1, 3477, 3594, 4 FROM dual
UNION ALL SELECT 614, 'NAICS', 975, 4236, 3, 979, 42362, 4 FROM dual
UNION ALL SELECT 615, 'NAICS', 1015, 424, 2, 1026, 4243, 3 FROM dual
UNION ALL SELECT 616, 'NAICS', 1726, 561, 2, 1764, 561591, 5 FROM dual
UNION ALL SELECT 617, 'SIC', 4305, 1, 1, 2832, 240, 3 FROM dual
UNION ALL SELECT 618, 'SIC', 4311, 52, 1, 3870, 5710, 3 FROM dual
UNION ALL SELECT 619, 'NAICS', 132, 21, 1, 144, 2122, 3 FROM dual
UNION ALL SELECT 620, 'NAICS', 1613, 53241, 4, 1615, 532412, 5 FROM dual
UNION ALL SELECT 621, 'NAICS', 930, 42, 1, 1022, 42413, 4 FROM dual
UNION ALL SELECT 622, 'SIC', 3964, 6330, 3, 3965, 6331, 4 FROM dual
UNION ALL SELECT 623, 'SIC', 3715, 5000, 2, 3744, 5070, 3 FROM dual
UNION ALL SELECT 624, 'SEC', 2791, 20, 1, 2269, 2270, 3 FROM dual
UNION ALL SELECT 625, 'SIC', 4311, 52, 1, 3907, 5980, 3 FROM dual
UNION ALL SELECT 626, 'NAICS', 1, 11, 1, 64, 11213, 4 FROM dual
UNION ALL SELECT 627, 'SEC', 2791, 20, 1, 2431, 3580, 3 FROM dual
UNION ALL SELECT 628, 'SIC', 3822, 5400, 2, 3828, 5431, 4 FROM dual
UNION ALL SELECT 629, 'SIC', 3316, 3260, 3, 3318, 3262, 4 FROM dual
UNION ALL SELECT 630, 'NAICS', 132, 21, 1, 147, 21222, 4 FROM dual
UNION ALL SELECT 631, 'NAICS', 1513, 523, 2, 1515, 523110, 5 FROM dual
UNION ALL SELECT 632, 'SIC', 3371, 3400, 2, 3372, 3410, 3 FROM dual
UNION ALL SELECT 633, 'SEC', 2552, 4900, 2, 2559, 4930, 3 FROM dual
UNION ALL SELECT 634, 'SIC', 4308, 20, 1, 3388, 3444, 4 FROM dual
UNION ALL SELECT 635, 'SIC', 4306, 10, 1, 2923, 1423, 4 FROM dual
UNION ALL SELECT 636, 'SIC', 4308, 20, 1, 3260, 2999, 4 FROM dual
UNION ALL SELECT 637, 'NAICS', 1513, 523, 2, 1523, 5232, 3 FROM dual
UNION ALL SELECT 638, 'SIC', 4306, 10, 1, 2892, 1041, 4 FROM dual
UNION ALL SELECT 639, 'SEC', 2791, 20, 1, 2425, 3571, 4 FROM dual
UNION ALL SELECT 640, 'SIC', 4289, 9630, 3, 4290, 9631, 4 FROM dual
UNION ALL SELECT 641, 'NAICS', 198, 2213, 3, 203, 221330, 5 FROM dual
UNION ALL SELECT 642, 'NAICS', 1462, 5179, 3, 1463, 51791, 4 FROM dual
UNION ALL SELECT 643, 'SIC', 4313, 70, 1, 4027, 7220, 3 FROM dual
UNION ALL SELECT 644, 'SIC', 3762, 5100, 2, 3793, 5172, 4 FROM dual
UNION ALL SELECT 645, 'SIC', 3729, 5040, 3, 3735, 5048, 4 FROM dual
UNION ALL SELECT 646, 'NAICS', 1850, 62, 1, 1896, 622110, 5 FROM dual
UNION ALL SELECT 647, 'NAICS', 2013, 7212, 3, 2016, 721214, 5 FROM dual
UNION ALL SELECT 648, 'NAICS', 931, 423, 2, 1003, 42386, 4 FROM dual
UNION ALL SELECT 649, 'NAICS', 930, 42, 1, 1051, 42448, 4 FROM dual
UNION ALL SELECT 650, 'NAICS', 1942, 71, 1, 1958, 7113, 3 FROM dual
UNION ALL SELECT 651, 'NAICS', 2003, 72, 1, 2036, 722515, 5 FROM dual
UNION ALL SELECT 652, 'SEC', 2796, 70, 1, 2775, 8710, 3 FROM dual
UNION ALL SELECT 653, 'NAICS', 98, 113, 2, 107, 11331, 4 FROM dual
UNION ALL SELECT 654, 'NAICS', 1561, 52519, 4, 1560, 525190, 5 FROM dual
UNION ALL SELECT 655, 'NAICS', 1570, 531, 2, 1575, 53112, 4 FROM dual
UNION ALL SELECT 656, 'NAICS', 1725, 56, 1, 1769, 561612, 5 FROM dual
UNION ALL SELECT 657, 'SEC', 2518, 4210, 3, 2519, 4213, 4 FROM dual
UNION ALL SELECT 658, 'SIC', 3301, 3200, 2, 3304, 3220, 3 FROM dual
UNION ALL SELECT 659, 'SIC', 4203, 8400, 2, 4204, 8410, 3 FROM dual
UNION ALL SELECT 660, 'NAICS', 1635, 54121, 4, 1637, 541213, 5 FROM dual
UNION ALL SELECT 661, 'SIC', 4310, 50, 1, 3794, 5180, 3 FROM dual
UNION ALL SELECT 662, 'NAICS', 1725, 56, 1, 1778, 56172, 4 FROM dual
UNION ALL SELECT 663, 'NAICS', 1850, 62, 1, 1920, 624, 2 FROM dual
UNION ALL SELECT 664, 'NAICS', 2038, 811, 2, 2050, 811192, 5 FROM dual
UNION ALL SELECT 665, 'SIC', 4308, 20, 1, 3126, 2411, 4 FROM dual
UNION ALL SELECT 666, 'SIC', 3451, 3560, 3, 3457, 3566, 4 FROM dual
UNION ALL SELECT 667, 'NAICS', 2, 111, 2, 40, 11142, 4 FROM dual
UNION ALL SELECT 668, 'NAICS', 1403, 511, 2, 1410, 51113, 4 FROM dual
UNION ALL SELECT 669, 'SIC', 3241, 2870, 3, 3245, 2879, 4 FROM dual
UNION ALL SELECT 670, 'SIC', 3564, 3840, 3, 3568, 3844, 4 FROM dual
UNION ALL SELECT 671, 'SIC', 3552, 3800, 2, 3554, 3812, 4 FROM dual
UNION ALL SELECT 672, 'NAICS', 1813, 611, 2, 1831, 61151, 4 FROM dual
UNION ALL SELECT 673, 'NAICS', 132, 21, 1, 159, 212312, 5 FROM dual
UNION ALL SELECT 674, 'NAICS', 2135, 92, 1, 2155, 92212, 4 FROM dual
UNION ALL SELECT 675, 'SEC', 2791, 20, 1, 2377, 3334, 4 FROM dual
UNION ALL SELECT 676, 'SIC', 3968, 6360, 3, 3969, 6361, 4 FROM dual
UNION ALL SELECT 677, 'SEC', 2283, 2500, 2, 2285, 2511, 4 FROM dual
UNION ALL SELECT 678, 'SIC', 3419, 3500, 2, 3436, 3542, 4 FROM dual
UNION ALL SELECT 679, 'NAICS', 1, 11, 1, 11, 11114, 4 FROM dual
UNION ALL SELECT 680, 'NAICS', 1510, 52232, 4, 1509, 522320, 5 FROM dual
UNION ALL SELECT 681, 'NAICS', 1402, 51, 1, 1403, 511, 2 FROM dual
UNION ALL SELECT 682, 'SEC', 2234, 1600, 2, 2235, 1620, 3 FROM dual
UNION ALL SELECT 683, 'SIC', 3833, 5460, 3, 3834, 5461, 4 FROM dual
UNION ALL SELECT 684, 'NAICS', 931, 423, 2, 1011, 423940, 5 FROM dual
UNION ALL SELECT 685, 'NAICS', 931, 423, 2, 994, 423820, 5 FROM dual
UNION ALL SELECT 686, 'NAICS', 1431, 5122, 3, 1436, 512230, 5 FROM dual
UNION ALL SELECT 687, 'SIC', 3083, 2300, 2, 3095, 2335, 4 FROM dual
UNION ALL SELECT 688, 'NAICS', 1979, 713, 2, 2002, 71399, 4 FROM dual
UNION ALL SELECT 689, 'SEC', 2479, 3820, 3, 2486, 3827, 4 FROM dual
UNION ALL SELECT 690, 'SIC', 4313, 70, 1, 4208, 8600, 2 FROM dual
UNION ALL SELECT 691, 'SEC', 2792, 40, 1, 2518, 4210, 3 FROM dual
UNION ALL SELECT 692, 'SIC', 4303, 9990, 3, 4304, 9999, 4 FROM dual
UNION ALL SELECT 693, 'NAICS', 253, 2382, 3, 258, 238290, 5 FROM dual
UNION ALL SELECT 694, 'SIC', 3434, 3540, 3, 3442, 3548, 4 FROM dual
UNION ALL SELECT 695, 'NAICS', 173, 2131, 3, 175, 213111, 5 FROM dual
UNION ALL SELECT 696, 'SEC', 2796, 70, 1, 2737, 7600, 2 FROM dual
UNION ALL SELECT 697, 'SIC', 2884, 1000, 2, 2886, 1011, 4 FROM dual
UNION ALL SELECT 698, 'NAICS', 182, 2211, 3, 184, 221111, 5 FROM dual
UNION ALL SELECT 699, 'NAICS', 1831, 61151, 4, 1833, 611512, 5 FROM dual
UNION ALL SELECT 700, 'SEC', 2749, 7940, 3, 2750, 7948, 4 FROM dual
UNION ALL SELECT 701, 'SIC', 3423, 3520, 3, 3425, 3524, 4 FROM dual
UNION ALL SELECT 702, 'SIC', 4305, 1, 1, 2831, 219, 4 FROM dual
UNION ALL SELECT 703, 'SIC', 2918, 1400, 2, 2934, 1479, 4 FROM dual
UNION ALL SELECT 704, 'SIC', 4311, 52, 1, 3909, 5984, 4 FROM dual
UNION ALL SELECT 705, 'SIC', 3124, 2400, 2, 3147, 2499, 4 FROM dual
UNION ALL SELECT 706, 'SIC', 4308, 20, 1, 3429, 3533, 4 FROM dual
UNION ALL SELECT 707, 'SIC', 4312, 60, 1, 3946, 6162, 4 FROM dual
UNION ALL SELECT 708, 'NAICS', 1813, 611, 2, 1838, 61161, 4 FROM dual
UNION ALL SELECT 709, 'SIC', 3353, 3340, 3, 3354, 3341, 4 FROM dual
UNION ALL SELECT 710, 'SIC', 4312, 60, 1, 3919, 6011, 4 FROM dual
UNION ALL SELECT 711, 'SIC', 4313, 70, 1, 4192, 8300, 2 FROM dual
UNION ALL SELECT 712, 'SEC', 2713, 7300, 2, 2720, 7350, 3 FROM dual
UNION ALL SELECT 713, 'NAICS', 931, 423, 2, 1007, 423920, 5 FROM dual
UNION ALL SELECT 714, 'NAICS', 1419, 512, 2, 1431, 5122, 3 FROM dual
UNION ALL SELECT 715, 'NAICS', 156, 2123, 3, 162, 21232, 4 FROM dual
UNION ALL SELECT 716, 'SEC', 2789, 10, 1, 2223, 1311, 4 FROM dual
UNION ALL SELECT 717, 'NAICS', 1078, 42491, 4, 1077, 424910, 5 FROM dual
UNION ALL SELECT 718, 'SIC', 4313, 70, 1, 4176, 8110, 3 FROM dual
UNION ALL SELECT 719, 'SEC', 2791, 20, 1, 2346, 3021, 4 FROM dual
UNION ALL SELECT 720, 'SIC', 2799, 110, 3, 2801, 112, 4 FROM dual
UNION ALL SELECT 721, 'SIC', 3774, 5140, 3, 3779, 5145, 4 FROM dual
UNION ALL SELECT 722, 'NAICS', 930, 42, 1, 941, 4232, 3 FROM dual
UNION ALL SELECT 723, 'NAICS', 2003, 72, 1, 2008, 721120, 5 FROM dual
UNION ALL SELECT 724, 'SIC', 3840, 5520, 3, 3841, 5521, 4 FROM dual
UNION ALL SELECT 725, 'NAICS', 1970, 7121, 3, 1972, 71211, 4 FROM dual
UNION ALL SELECT 726, 'NAICS', 1402, 51, 1, 1475, 51912, 4 FROM dual
UNION ALL SELECT 727, 'NAICS', 1725, 56, 1, 1730, 5612, 3 FROM dual
UNION ALL SELECT 728, 'SIC', 4308, 20, 1, 3093, 2330, 3 FROM dual
UNION ALL SELECT 729, 'NAICS', 931, 423, 2, 966, 423460, 5 FROM dual
UNION ALL SELECT 730, 'NAICS', 1920, 624, 2, 1923, 62411, 4 FROM dual
UNION ALL SELECT 731, 'SIC', 4308, 20, 1, 3376, 3421, 4 FROM dual
UNION ALL SELECT 732, 'SIC', 3999, 6730, 3, 4001, 6733, 4 FROM dual
UNION ALL SELECT 733, 'NAICS', 85, 11251, 4, 86, 112511, 5 FROM dual
UNION ALL SELECT 734, 'SEC', 2405, 3520, 3, 2407, 3524, 4 FROM dual
UNION ALL SELECT 735, 'NAICS', 1942, 71, 1, 1966, 7115, 3 FROM dual
UNION ALL SELECT 736, 'NAICS', 2135, 92, 1, 2162, 922160, 5 FROM dual
UNION ALL SELECT 737, 'SEC', 2791, 20, 1, 2430, 3579, 4 FROM dual
UNION ALL SELECT 738, 'SIC', 3875, 5720, 3, 3876, 5722, 4 FROM dual
UNION ALL SELECT 739, 'NAICS', 1667, 54151, 4, 1671, 541519, 5 FROM dual
UNION ALL SELECT 740, 'SEC', 2789, 10, 1, 2221, 1300, 2 FROM dual
UNION ALL SELECT 741, 'NAICS', 930, 42, 1, 1085, 424950, 5 FROM dual
UNION ALL SELECT 742, 'SIC', 3715, 5000, 2, 3752, 5084, 4 FROM dual
UNION ALL SELECT 743, 'SEC', 2345, 3020, 3, 2346, 3021, 4 FROM dual
UNION ALL SELECT 744, 'NAICS', 2003, 72, 1, 2014, 72121, 4 FROM dual
UNION ALL SELECT 745, 'SEC', 2450, 3670, 3, 2452, 3674, 4 FROM dual
UNION ALL SELECT 746, 'SEC', 2796, 70, 1, 2769, 8200, 2 FROM dual
UNION ALL SELECT 747, 'SEC', 2791, 20, 1, 2494, 3850, 3 FROM dual
UNION ALL SELECT 748, 'SEC', 2792, 40, 1, 2523, 4400, 2 FROM dual
UNION ALL SELECT 749, 'NAICS', 1725, 56, 1, 1738, 56132, 4 FROM dual
UNION ALL SELECT 750, 'SIC', 3371, 3400, 2, 3412, 3493, 4 FROM dual
UNION ALL SELECT 751, 'SIC', 4284, 9600, 2, 4293, 9650, 3 FROM dual
UNION ALL SELECT 752, 'SIC', 4125, 7900, 2, 4129, 7922, 4 FROM dual
UNION ALL SELECT 753, 'SIC', 3451, 3560, 3, 3459, 3568, 4 FROM dual
UNION ALL SELECT 754, 'SIC', 2798, 100, 2, 2811, 160, 3 FROM dual
UNION ALL SELECT 755, 'SEC', 2221, 1300, 2, 2225, 1381, 4 FROM dual
UNION ALL SELECT 756, 'SIC', 4277, 9500, 2, 4280, 9512, 4 FROM dual
UNION ALL SELECT 757, 'SEC', 2739, 7810, 3, 2740, 7812, 4 FROM dual
UNION ALL SELECT 758, 'NAICS', 43, 1119, 3, 44, 111910, 5 FROM dual
UNION ALL SELECT 759, 'NAICS', 1914, 62331, 4, 1916, 623312, 5 FROM dual
UNION ALL SELECT 760, 'SIC', 3480, 3600, 2, 3502, 3647, 4 FROM dual
UNION ALL SELECT 761, 'SIC', 3855, 5610, 3, 3856, 5611, 4 FROM dual
UNION ALL SELECT 762, 'SIC', 4040, 7300, 2, 4053, 7336, 4 FROM dual
UNION ALL SELECT 763, 'SIC', 2847, 700, 2, 2850, 720, 3 FROM dual
UNION ALL SELECT 764, 'SIC', 3000, 2040, 3, 3007, 2048, 4 FROM dual
UNION ALL SELECT 765, 'SIC', 4309, 40, 1, 3657, 4513, 4 FROM dual
UNION ALL SELECT 766, 'NAICS', 1683, 5417, 3, 1686, 541712, 5 FROM dual
UNION ALL SELECT 767, 'SEC', 2458, 3700, 2, 2461, 3713, 4 FROM dual
UNION ALL SELECT 768, 'SIC', 4309, 40, 1, 3607, 4111, 4 FROM dual
UNION ALL SELECT 769, 'NAICS', 1591, 532, 2, 1601, 532220, 5 FROM dual
UNION ALL SELECT 770, 'SIC', 4313, 70, 1, 4086, 7519, 4 FROM dual
UNION ALL SELECT 771, 'NAICS', 2072, 8121, 3, 2074, 812111, 5 FROM dual
UNION ALL SELECT 772, 'SIC', 3161, 2540, 3, 3163, 2542, 4 FROM dual
UNION ALL SELECT 773, 'SIC', 3663, 4610, 3, 3664, 4612, 4 FROM dual
UNION ALL SELECT 774, 'SEC', 2524, 4410, 3, 2525, 4412, 4 FROM dual
UNION ALL SELECT 775, 'SEC', 2794, 52, 1, 2629, 5661, 4 FROM dual
UNION ALL SELECT 776, 'SIC', 3214, 2800, 2, 3246, 2890, 3 FROM dual
UNION ALL SELECT 777, 'SIC', 3837, 5500, 2, 3843, 5531, 4 FROM dual
UNION ALL SELECT 778, 'SIC', 4277, 9500, 2, 4279, 9511, 4 FROM dual
UNION ALL SELECT 779, 'SIC', 4313, 70, 1, 4173, 8093, 4 FROM dual
UNION ALL SELECT 780, 'SIC', 4309, 40, 1, 3610, 4121, 4 FROM dual
UNION ALL SELECT 781, 'NAICS', 1725, 56, 1, 1753, 56145, 4 FROM dual
UNION ALL SELECT 782, 'SEC', 2791, 20, 1, 2309, 2732, 4 FROM dual
UNION ALL SELECT 783, 'SEC', 2403, 3500, 2, 2431, 3580, 3 FROM dual
UNION ALL SELECT 784, 'NAICS', 1942, 71, 1, 1963, 7114, 3 FROM dual
UNION ALL SELECT 785, 'NAICS', 2183, 9251, 3, 2184, 925110, 5 FROM dual
UNION ALL SELECT 786, 'SIC', 3384, 3440, 3, 3391, 3449, 4 FROM dual
UNION ALL SELECT 787, 'NAICS', 139, 2121, 3, 141, 212111, 5 FROM dual
UNION ALL SELECT 788, 'NAICS', 1850, 62, 1, 1854, 621111, 5 FROM dual
UNION ALL SELECT 789, 'SIC', 3371, 3400, 2, 3407, 3484, 4 FROM dual
UNION ALL SELECT 790, 'NAICS', 246, 23815, 4, 245, 238150, 5 FROM dual
UNION ALL SELECT 791, 'NAICS', 119, 11511, 4, 120, 115111, 5 FROM dual
UNION ALL SELECT 792, 'SIC', 4208, 8600, 2, 4220, 8661, 4 FROM dual
UNION ALL SELECT 793, 'SIC', 3098, 2340, 3, 3099, 2341, 4 FROM dual
UNION ALL SELECT 794, 'SIC', 3301, 3200, 2, 3330, 3290, 3 FROM dual
UNION ALL SELECT 795, 'SIC', 3667, 4700, 2, 3668, 4720, 3 FROM dual
UNION ALL SELECT 796, 'NAICS', 1452, 517, 2, 1463, 51791, 4 FROM dual
UNION ALL SELECT 797, 'NAICS', 1, 11, 1, 21, 111211, 5 FROM dual
UNION ALL SELECT 798, 'NAICS', 2037, 81, 1, 2132, 8141, 3 FROM dual
UNION ALL SELECT 799, 'SIC', 3693, 4900, 2, 3706, 4941, 4 FROM dual
UNION ALL SELECT 800, 'SIC', 3409, 3490, 3, 3417, 3498, 4 FROM dual
UNION ALL SELECT 801, 'NAICS', 1850, 62, 1, 1933, 624229, 5 FROM dual
UNION ALL SELECT 802, 'SIC', 4308, 20, 1, 3018, 2067, 4 FROM dual
UNION ALL SELECT 803, 'NAICS', 1850, 62, 1, 1922, 624110, 5 FROM dual
UNION ALL SELECT 804, 'NAICS', 1015, 424, 2, 1070, 42472, 4 FROM dual
UNION ALL SELECT 805, 'NAICS', 1831, 61151, 4, 1835, 611519, 5 FROM dual
UNION ALL SELECT 806, 'SIC', 3520, 3690, 3, 3522, 3692, 4 FROM dual
UNION ALL SELECT 807, 'NAICS', 941, 4232, 3, 943, 42321, 4 FROM dual
UNION ALL SELECT 808, 'SIC', 2798, 100, 2, 2801, 112, 4 FROM dual
UNION ALL SELECT 809, 'NAICS', 1090, 4251, 3, 1091, 425110, 5 FROM dual
UNION ALL SELECT 810, 'SEC', 2384, 3400, 2, 2396, 3450, 3 FROM dual
UNION ALL SELECT 811, 'SIC', 4040, 7300, 2, 4064, 7363, 4 FROM dual
UNION ALL SELECT 812, 'SIC', 4305, 1, 1, 2819, 179, 4 FROM dual
UNION ALL SELECT 813, 'SIC', 3716, 5010, 3, 3717, 5012, 4 FROM dual
UNION ALL SELECT 814, 'SIC', 4305, 1, 1, 2836, 252, 4 FROM dual
UNION ALL SELECT 815, 'NAICS', 2165, 92219, 4, 2164, 922190, 5 FROM dual
UNION ALL SELECT 816, 'SEC', 2791, 20, 1, 2371, 3300, 2 FROM dual
UNION ALL SELECT 817, 'NAICS', 1536, 524, 2, 1552, 524292, 5 FROM dual
UNION ALL SELECT 818, 'SEC', 2796, 70, 1, 2747, 7841, 4 FROM dual
UNION ALL SELECT 819, 'NAICS', 930, 42, 1, 1049, 42447, 4 FROM dual
UNION ALL SELECT 820, 'SIC', 4308, 20, 1, 3251, 2899, 4 FROM dual
UNION ALL SELECT 821, 'SIC', 4313, 70, 1, 4163, 8062, 4 FROM dual
UNION ALL SELECT 822, 'NAICS', 2, 111, 2, 19, 1112, 3 FROM dual
UNION ALL SELECT 823, 'NAICS', 931, 423, 2, 969, 42349, 4 FROM dual
UNION ALL SELECT 824, 'NAICS', 68, 1123, 3, 70, 11231, 4 FROM dual
UNION ALL SELECT 825, 'SIC', 4308, 20, 1, 2999, 2038, 4 FROM dual
UNION ALL SELECT 826, 'NAICS', 205, 23, 1, 254, 238210, 5 FROM dual
UNION ALL SELECT 827, 'SEC', 2795, 60, 1, 2679, 6311, 4 FROM dual
UNION ALL SELECT 828, 'SIC', 4310, 50, 1, 3784, 5150, 3 FROM dual
UNION ALL SELECT 829, 'NAICS', 2191, 92611, 4, 2190, 926110, 5 FROM dual
UNION ALL SELECT 830, 'SIC', 3301, 3200, 2, 3303, 3211, 4 FROM dual
UNION ALL SELECT 831, 'SIC', 3762, 5100, 2, 3796, 5182, 4 FROM dual
UNION ALL SELECT 832, 'SIC', 2847, 700, 2, 2860, 752, 4 FROM dual
UNION ALL SELECT 833, 'SIC', 3167, 2600, 2, 3170, 2620, 3 FROM dual
UNION ALL SELECT 834, 'SEC', 2391, 3440, 3, 2392, 3442, 4 FROM dual
UNION ALL SELECT 835, 'SIC', 4308, 20, 1, 3112, 2386, 4 FROM dual
UNION ALL SELECT 836, 'SIC', 4309, 40, 1, 3641, 4430, 3 FROM dual
UNION ALL SELECT 837, 'SIC', 4308, 20, 1, 3525, 3699, 4 FROM dual
UNION ALL SELECT 838, 'SIC', 3609, 4120, 3, 3610, 4121, 4 FROM dual
UNION ALL SELECT 839, 'NAICS', 2, 111, 2, 10, 111140, 5 FROM dual
UNION ALL SELECT 840, 'NAICS', 1823, 6114, 3, 1827, 61142, 4 FROM dual
UNION ALL SELECT 841, 'NAICS', 1813, 611, 2, 1832, 611511, 5 FROM dual
UNION ALL SELECT 842, 'SIC', 4308, 20, 1, 3110, 2384, 4 FROM dual
UNION ALL SELECT 843, 'SIC', 4311, 52, 1, 3913, 5993, 4 FROM dual
UNION ALL SELECT 844, 'NAICS', 1555, 5251, 3, 1559, 52512, 4 FROM dual
UNION ALL SELECT 845, 'SIC', 3769, 5130, 3, 3771, 5136, 4 FROM dual
UNION ALL SELECT 846, 'SIC', 4308, 20, 1, 3403, 3479, 4 FROM dual
UNION ALL SELECT 847, 'NAICS', 930, 42, 1, 1069, 424720, 5 FROM dual
UNION ALL SELECT 848, 'NAICS', 1792, 562, 2, 1807, 562920, 5 FROM dual
UNION ALL SELECT 849, 'NAICS', 2137, 9211, 3, 2146, 921150, 5 FROM dual
UNION ALL SELECT 850, 'SIC', 4203, 8400, 2, 4205, 8412, 4 FROM dual
UNION ALL SELECT 851, 'NAICS', 1816, 61111, 4, 1815, 611110, 5 FROM dual
UNION ALL SELECT 852, 'NAICS', 2112, 8133, 3, 2113, 81331, 4 FROM dual
UNION ALL SELECT 853, 'SIC', 3012, 2060, 3, 3019, 2068, 4 FROM dual
UNION ALL SELECT 854, 'SEC', 2785, 9700, 2, 2787, 9721, 4 FROM dual
UNION ALL SELECT 855, 'NAICS', 2135, 92, 1, 2158, 922140, 5 FROM dual
UNION ALL SELECT 856, 'SIC', 2982, 2000, 2, 3030, 2085, 4 FROM dual
UNION ALL SELECT 857, 'SIC', 2982, 2000, 2, 3037, 2096, 4 FROM dual
UNION ALL SELECT 858, 'SIC', 3693, 4900, 2, 3696, 4920, 3 FROM dual
UNION ALL SELECT 859, 'NAICS', 1850, 62, 1, 1910, 62321, 4 FROM dual
UNION ALL SELECT 860, 'NAICS', 2112, 8133, 3, 2115, 813312, 5 FROM dual
UNION ALL SELECT 861, 'SEC', 2286, 2520, 3, 2287, 2522, 4 FROM dual
UNION ALL SELECT 862, 'SIC', 4314, 90, 1, 4288, 9621, 4 FROM dual
UNION ALL SELECT 863, 'SEC', 2791, 20, 1, 2320, 2810, 3 FROM dual
UNION ALL SELECT 864, 'SEC', 2611, 5310, 3, 2612, 5311, 4 FROM dual
UNION ALL SELECT 865, 'SIC', 4308, 20, 1, 3023, 2076, 4 FROM dual
UNION ALL SELECT 866, 'NAICS', 205, 23, 1, 252, 23819, 4 FROM dual
UNION ALL SELECT 867, 'NAICS', 109, 1141, 3, 110, 11411, 4 FROM dual
UNION ALL SELECT 868, 'NAICS', 223, 23712, 4, 222, 237120, 5 FROM dual
UNION ALL SELECT 869, 'NAICS', 930, 42, 1, 952, 42333, 4 FROM dual
UNION ALL SELECT 870, 'SIC', 2798, 100, 2, 2806, 131, 4 FROM dual
UNION ALL SELECT 871, 'NAICS', 2039, 8111, 3, 2043, 811113, 5 FROM dual
UNION ALL SELECT 872, 'SIC', 3791, 5170, 3, 3792, 5171, 4 FROM dual
UNION ALL SELECT 873, 'SEC', 2526, 4500, 2, 2533, 4581, 4 FROM dual
UNION ALL SELECT 874, 'SEC', 2753, 8000, 2, 2757, 8051, 4 FROM dual
UNION ALL SELECT 875, 'SEC', 2694, 6500, 2, 2701, 6550, 3 FROM dual
UNION ALL SELECT 876, 'SIC', 3937, 6100, 2, 3939, 6111, 4 FROM dual
UNION ALL SELECT 877, 'NAICS', 235, 238, 2, 241, 238130, 5 FROM dual
UNION ALL SELECT 878, 'NAICS', 930, 42, 1, 977, 42361, 4 FROM dual
UNION ALL SELECT 879, 'SIC', 3815, 5300, 2, 3821, 5399, 4 FROM dual
UNION ALL SELECT 880, 'NAICS', 2103, 813, 2, 2109, 813211, 5 FROM dual
UNION ALL SELECT 881, 'NAICS', 1463, 51791, 4, 1465, 517919, 5 FROM dual
UNION ALL SELECT 882, 'NAICS', 16, 11119, 4, 17, 111191, 5 FROM dual
UNION ALL SELECT 883, 'NAICS', 931, 423, 2, 976, 423610, 5 FROM dual
UNION ALL SELECT 884, 'SIC', 3168, 2610, 3, 3169, 2611, 4 FROM dual
UNION ALL SELECT 885, 'NAICS', 1812, 61, 1, 1814, 6111, 3 FROM dual
UNION ALL SELECT 886, 'NAICS', 1625, 541, 2, 1685, 541711, 5 FROM dual
UNION ALL SELECT 887, 'NAICS', 218, 237, 2, 225, 23713, 4 FROM dual
UNION ALL SELECT 888, 'SIC', 4307, 15, 1, 2946, 1541, 4 FROM dual
UNION ALL SELECT 889, 'SIC', 3426, 3530, 3, 3427, 3531, 4 FROM dual
UNION ALL SELECT 890, 'SIC', 3180, 2670, 3, 3187, 2677, 4 FROM dual
UNION ALL SELECT 891, 'SEC', 2791, 20, 1, 2317, 2780, 3 FROM dual
UNION ALL SELECT 892, 'NAICS', 930, 42, 1, 1093, 425120, 5 FROM dual
UNION ALL SELECT 893, 'NAICS', 2135, 92, 1, 2207, 92811, 4 FROM dual
UNION ALL SELECT 894, 'NAICS', 2204, 928, 2, 2206, 928110, 5 FROM dual
UNION ALL SELECT 895, 'NAICS', 85, 11251, 4, 87, 112512, 5 FROM dual
UNION ALL SELECT 896, 'SIC', 2918, 1400, 2, 2930, 1459, 4 FROM dual
UNION ALL SELECT 897, 'NAICS', 138, 212, 2, 162, 21232, 4 FROM dual
UNION ALL SELECT 898, 'SIC', 3474, 3590, 3, 3476, 3593, 4 FROM dual
UNION ALL SELECT 899, 'SIC', 3073, 2280, 3, 3076, 2284, 4 FROM dual
UNION ALL SELECT 900, 'SIC', 3526, 3700, 2, 3547, 3769, 4 FROM dual
UNION ALL SELECT 901, 'SIC', 4308, 20, 1, 3411, 3492, 4 FROM dual
UNION ALL SELECT 902, 'NAICS', 1420, 5121, 3, 1425, 51213, 4 FROM dual
UNION ALL SELECT 903, 'SIC', 4306, 10, 1, 2904, 1222, 4 FROM dual
UNION ALL SELECT 904, 'SIC', 3611, 4130, 3, 3612, 4131, 4 FROM dual
UNION ALL SELECT 905, 'NAICS', 930, 42, 1, 942, 423210, 5 FROM dual
UNION ALL SELECT 906, 'NAICS', 2136, 921, 2, 2142, 921130, 5 FROM dual
UNION ALL SELECT 907, 'SIC', 3211, 2790, 3, 3212, 2791, 4 FROM dual
UNION ALL SELECT 908, 'NAICS', 2037, 81, 1, 2108, 81321, 4 FROM dual
UNION ALL SELECT 909, 'NAICS', 1402, 51, 1, 1476, 519130, 5 FROM dual
UNION ALL SELECT 910, 'SIC', 3958, 6300, 2, 3962, 6321, 4 FROM dual
UNION ALL SELECT 911, 'NAICS', 242, 23813, 4, 241, 238130, 5 FROM dual
UNION ALL SELECT 912, 'SIC', 3338, 3310, 3, 3341, 3315, 4 FROM dual
UNION ALL SELECT 913, 'SIC', 4310, 50, 1, 3718, 5013, 4 FROM dual
UNION ALL SELECT 914, 'NAICS', 132, 21, 1, 174, 21311, 4 FROM dual
UNION ALL SELECT 915, 'SIC', 3921, 6020, 3, 3922, 6021, 4 FROM dual
UNION ALL SELECT 916, 'NAICS', 89, 1129, 3, 97, 11299, 4 FROM dual
UNION ALL SELECT 917, 'SIC', 3220, 2820, 3, 3224, 2824, 4 FROM dual
UNION ALL SELECT 918, 'SIC', 4313, 70, 1, 4189, 8249, 4 FROM dual
UNION ALL SELECT 919, 'NAICS', 2037, 81, 1, 2074, 812111, 5 FROM dual
UNION ALL SELECT 920, 'SIC', 4308, 20, 1, 3224, 2824, 4 FROM dual
UNION ALL SELECT 921, 'SIC', 4313, 70, 1, 4112, 7699, 4 FROM dual
UNION ALL SELECT 922, 'NAICS', 1, 11, 1, 27, 11132, 4 FROM dual
UNION ALL SELECT 923, 'NAICS', 1942, 71, 1, 1961, 711320, 5 FROM dual
UNION ALL SELECT 924, 'SEC', 2552, 4900, 2, 2562, 4940, 3 FROM dual
UNION ALL SELECT 925, 'NAICS', 2177, 9241, 3, 2179, 92411, 4 FROM dual
UNION ALL SELECT 926, 'SIC', 3124, 2400, 2, 3141, 2450, 3 FROM dual
UNION ALL SELECT 927, 'NAICS', 1726, 561, 2, 1745, 561421, 5 FROM dual
UNION ALL SELECT 928, 'NAICS', 2183, 9251, 3, 2186, 925120, 5 FROM dual
UNION ALL SELECT 929, 'SIC', 3911, 5990, 3, 3915, 5995, 4 FROM dual
UNION ALL SELECT 930, 'SIC', 3101, 2350, 3, 3102, 2353, 4 FROM dual
UNION ALL SELECT 931, 'NAICS', 1741, 5614, 3, 1747, 56143, 4 FROM dual
UNION ALL SELECT 932, 'NAICS', 2037, 81, 1, 2075, 812112, 5 FROM dual
UNION ALL SELECT 933, 'SIC', 4178, 8200, 2, 4181, 8220, 3 FROM dual
UNION ALL SELECT 934, 'NAICS', 1051, 42448, 4, 1050, 424480, 5 FROM dual
UNION ALL SELECT 935, 'NAICS', 2135, 92, 1, 2150, 922, 2 FROM dual
UNION ALL SELECT 936, 'SIC', 3797, 5190, 3, 3802, 5198, 4 FROM dual
UNION ALL SELECT 937, 'SIC', 4311, 52, 1, 3848, 5560, 3 FROM dual
UNION ALL SELECT 938, 'SIC', 2955, 1700, 2, 2967, 1751, 4 FROM dual
UNION ALL SELECT 939, 'SIC', 3480, 3600, 2, 3507, 3660, 3 FROM dual
UNION ALL SELECT 940, 'NAICS', 1015, 424, 2, 1060, 42459, 4 FROM dual
UNION ALL SELECT 941, 'NAICS', 1625, 541, 2, 1675, 541612, 5 FROM dual
UNION ALL SELECT 942, 'SIC', 4305, 1, 1, 2818, 175, 4 FROM dual
UNION ALL SELECT 943, 'SIC', 2813, 170, 3, 2816, 173, 4 FROM dual
UNION ALL SELECT 944, 'NAICS', 1402, 51, 1, 1458, 51721, 4 FROM dual
UNION ALL SELECT 945, 'SIC', 4308, 20, 1, 3231, 2841, 4 FROM dual
UNION ALL SELECT 946, 'SIC', 3258, 2990, 3, 3259, 2992, 4 FROM dual
UNION ALL SELECT 947, 'NAICS', 1480, 52, 1, 1509, 522320, 5 FROM dual
UNION ALL SELECT 948, 'NAICS', 1672, 5416, 3, 1673, 54161, 4 FROM dual
UNION ALL SELECT 949, 'SIC', 4146, 8000, 2, 4150, 8021, 4 FROM dual
UNION ALL SELECT 950, 'SEC', 2713, 7300, 2, 2727, 7372, 4 FROM dual
UNION ALL SELECT 951, 'SEC', 2323, 2830, 3, 2324, 2833, 4 FROM dual
UNION ALL SELECT 952, 'NAICS', 1942, 71, 1, 1978, 71219, 4 FROM dual
UNION ALL SELECT 953, 'SIC', 3680, 4800, 2, 3689, 4840, 3 FROM dual
UNION ALL SELECT 954, 'SIC', 3958, 6300, 2, 3968, 6360, 3 FROM dual
UNION ALL SELECT 955, 'NAICS', 930, 42, 1, 1045, 42445, 4 FROM dual
UNION ALL SELECT 956, 'SIC', 2982, 2000, 2, 3015, 2063, 4 FROM dual
UNION ALL SELECT 957, 'NAICS', 181, 221, 2, 203, 221330, 5 FROM dual
UNION ALL SELECT 958, 'SIC', 4305, 1, 1, 2846, 291, 4 FROM dual
UNION ALL SELECT 959, 'SEC', 2789, 10, 1, 2224, 1380, 3 FROM dual
UNION ALL SELECT 960, 'SIC', 3917, 6000, 2, 3936, 6099, 4 FROM dual
UNION ALL SELECT 961, 'NAICS', 2037, 81, 1, 2098, 812922, 5 FROM dual
UNION ALL SELECT 962, 'SIC', 3127, 2420, 3, 3130, 2429, 4 FROM dual
UNION ALL SELECT 963, 'SEC', 2703, 6700, 2, 4327, 6770, 3 FROM dual
UNION ALL SELECT 964, 'NAICS', 963, 42344, 4, 962, 423440, 5 FROM dual
UNION ALL SELECT 965, 'SIC', 3050, 2200, 2, 3064, 2257, 4 FROM dual
UNION ALL SELECT 966, 'SIC', 4081, 7500, 2, 4087, 7520, 3 FROM dual
UNION ALL SELECT 967, 'SIC', 3419, 3500, 2, 3471, 3585, 4 FROM dual
UNION ALL SELECT 968, 'SIC', 4309, 40, 1, 3710, 4959, 4 FROM dual
UNION ALL SELECT 969, 'NAICS', 1806, 56291, 4, 1805, 562910, 5 FROM dual
UNION ALL SELECT 970, 'SIC', 3911, 5990, 3, 3912, 5992, 4 FROM dual
UNION ALL SELECT 971, 'SIC', 4307, 15, 1, 2971, 1770, 3 FROM dual
UNION ALL SELECT 972, 'NAICS', 2183, 9251, 3, 2185, 92511, 4 FROM dual
UNION ALL SELECT 973, 'SEC', 2552, 4900, 2, 2561, 4932, 4 FROM dual
UNION ALL SELECT 974, 'NAICS', 1763, 56159, 4, 1765, 561599, 5 FROM dual
UNION ALL SELECT 975, 'NAICS', 2167, 9231, 3, 2175, 92314, 4 FROM dual
UNION ALL SELECT 976, 'SIC', 3287, 3140, 3, 3289, 3143, 4 FROM dual
UNION ALL SELECT 977, 'NAICS', 1402, 51, 1, 1453, 5171, 3 FROM dual
UNION ALL SELECT 978, 'NAICS', 1944, 7111, 3, 1945, 711110, 5 FROM dual
UNION ALL SELECT 979, 'SIC', 2993, 2030, 3, 2995, 2033, 4 FROM dual
UNION ALL SELECT 980, 'NAICS', 207, 2361, 3, 208, 23611, 4 FROM dual
UNION ALL SELECT 981, 'SIC', 2973, 1780, 3, 2974, 1781, 4 FROM dual
UNION ALL SELECT 982, 'SIC', 4308, 20, 1, 3091, 2326, 4 FROM dual
UNION ALL SELECT 983, 'SIC', 3917, 6000, 2, 3935, 6091, 4 FROM dual
UNION ALL SELECT 984, 'NAICS', 1605, 53229, 4, 1607, 532292, 5 FROM dual
UNION ALL SELECT 985, 'NAICS', 1484, 52111, 4, 1483, 521110, 5 FROM dual
UNION ALL SELECT 986, 'SEC', 2465, 3720, 3, 2468, 3728, 4 FROM dual
UNION ALL SELECT 987, 'NAICS', 1979, 713, 2, 1995, 713930, 5 FROM dual
UNION ALL SELECT 988, 'NAICS', 2188, 926, 2, 2195, 92613, 4 FROM dual
UNION ALL SELECT 989, 'NAICS', 1943, 711, 2, 1949, 711130, 5 FROM dual
UNION ALL SELECT 990, 'SEC', 2796, 70, 1, 2730, 7377, 4 FROM dual
UNION ALL SELECT 991, 'NAICS', 930, 42, 1, 1046, 424460, 5 FROM dual
UNION ALL SELECT 992, 'NAICS', 1850, 62, 1, 1872, 621410, 5 FROM dual
UNION ALL SELECT 993, 'SIC', 4308, 20, 1, 3503, 3648, 4 FROM dual
UNION ALL SELECT 994, 'SIC', 3237, 2860, 3, 3240, 2869, 4 FROM dual
UNION ALL SELECT 995, 'SEC', 2555, 4920, 3, 2558, 4924, 4 FROM dual
UNION ALL SELECT 996, 'SIC', 2982, 2000, 2, 2985, 2013, 4 FROM dual
UNION ALL SELECT 997, 'NAICS', 1569, 53, 1, 1576, 531130, 5 FROM dual
UNION ALL SELECT 998, 'SEC', 2371, 3300, 2, 2375, 3320, 3 FROM dual
UNION ALL SELECT 999, 'SEC', 2758, 8060, 3, 2759, 8062, 4 FROM dual
UNION ALL SELECT 1000, 'SIC', 4277, 9500, 2, 4281, 9530, 3 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 1001, 'SIC', 3621, 4210, 3, 3622, 4212, 4 FROM dual
UNION ALL SELECT 1002, 'NAICS', 56, 112, 2, 76, 11234, 4 FROM dual
UNION ALL SELECT 1003, 'SIC', 4312, 60, 1, 3984, 6519, 4 FROM dual
UNION ALL SELECT 1004, 'NAICS', 1624, 54, 1, 1704, 541890, 5 FROM dual
UNION ALL SELECT 1005, 'NAICS', 1, 11, 1, 77, 112390, 5 FROM dual
UNION ALL SELECT 1006, 'NAICS', 2072, 8121, 3, 2079, 812199, 5 FROM dual
UNION ALL SELECT 1007, 'SIC', 3444, 3550, 3, 3450, 3559, 4 FROM dual
UNION ALL SELECT 1008, 'SIC', 3174, 2650, 3, 3179, 2657, 4 FROM dual
UNION ALL SELECT 1009, 'NAICS', 1, 11, 1, 125, 115116, 5 FROM dual
UNION ALL SELECT 1010, 'NAICS', 1076, 4249, 3, 1080, 42492, 4 FROM dual
UNION ALL SELECT 1011, 'SIC', 3368, 3390, 3, 3370, 3399, 4 FROM dual
UNION ALL SELECT 1012, 'SEC', 2791, 20, 1, 2387, 3412, 4 FROM dual
UNION ALL SELECT 1013, 'SIC', 4309, 40, 1, 3691, 4890, 3 FROM dual
UNION ALL SELECT 1014, 'SIC', 3942, 6150, 3, 3944, 6159, 4 FROM dual
UNION ALL SELECT 1015, 'NAICS', 1466, 518, 2, 1468, 518210, 5 FROM dual
UNION ALL SELECT 1016, 'NAICS', 1851, 621, 2, 1853, 62111, 4 FROM dual
UNION ALL SELECT 1017, 'SIC', 4310, 50, 1, 3738, 5051, 4 FROM dual
UNION ALL SELECT 1018, 'SIC', 4308, 20, 1, 3109, 2381, 4 FROM dual
UNION ALL SELECT 1019, 'NAICS', 1402, 51, 1, 1468, 518210, 5 FROM dual
UNION ALL SELECT 1020, 'SEC', 2792, 40, 1, 2557, 4923, 4 FROM dual
UNION ALL SELECT 1021, 'SIC', 2909, 1300, 2, 2910, 1310, 3 FROM dual
UNION ALL SELECT 1022, 'SIC', 4171, 8090, 3, 4173, 8093, 4 FROM dual
UNION ALL SELECT 1023, 'NAICS', 83, 11242, 4, 82, 112420, 5 FROM dual
UNION ALL SELECT 1024, 'NAICS', 23, 1113, 3, 27, 11132, 4 FROM dual
UNION ALL SELECT 1025, 'SIC', 3837, 5500, 2, 3845, 5541, 4 FROM dual
UNION ALL SELECT 1026, 'SEC', 2459, 3710, 3, 2460, 3711, 4 FROM dual
UNION ALL SELECT 1027, 'SIC', 4305, 1, 1, 2853, 723, 4 FROM dual
UNION ALL SELECT 1028, 'NAICS', 1, 11, 1, 20, 11121, 4 FROM dual
UNION ALL SELECT 1029, 'NAICS', 1942, 71, 1, 1964, 711410, 5 FROM dual
UNION ALL SELECT 1030, 'SIC', 3829, 5440, 3, 3830, 5441, 4 FROM dual
UNION ALL SELECT 1031, 'NAICS', 180, 22, 1, 202, 22132, 4 FROM dual
UNION ALL SELECT 1032, 'SEC', 2753, 8000, 2, 2764, 8090, 3 FROM dual
UNION ALL SELECT 1033, 'SIC', 4312, 60, 1, 3953, 6230, 3 FROM dual
UNION ALL SELECT 1034, 'SEC', 2271, 2300, 2, 2274, 2340, 3 FROM dual
UNION ALL SELECT 1035, 'SIC', 4284, 9600, 2, 4290, 9631, 4 FROM dual
UNION ALL SELECT 1036, 'NAICS', 2039, 8111, 3, 2041, 811111, 5 FROM dual
UNION ALL SELECT 1037, 'SEC', 2713, 7300, 2, 2718, 7331, 4 FROM dual
UNION ALL SELECT 1038, 'NAICS', 2038, 811, 2, 2059, 811310, 5 FROM dual
UNION ALL SELECT 1039, 'NAICS', 2038, 811, 2, 2049, 811191, 5 FROM dual
UNION ALL SELECT 1040, 'SIC', 4305, 1, 1, 2808, 133, 4 FROM dual
UNION ALL SELECT 1041, 'SIC', 4308, 20, 1, 3089, 2323, 4 FROM dual
UNION ALL SELECT 1042, 'SIC', 4313, 70, 1, 4056, 7342, 4 FROM dual
UNION ALL SELECT 1043, 'NAICS', 2169, 92311, 4, 2168, 923110, 5 FROM dual
UNION ALL SELECT 1044, 'SIC', 4175, 8100, 2, 4177, 8111, 4 FROM dual
UNION ALL SELECT 1045, 'NAICS', 2161, 92215, 4, 2160, 922150, 5 FROM dual
UNION ALL SELECT 1046, 'SEC', 2791, 20, 1, 2292, 2600, 2 FROM dual
UNION ALL SELECT 1047, 'NAICS', 1500, 52229, 4, 1505, 522298, 5 FROM dual
UNION ALL SELECT 1048, 'SIC', 3426, 3530, 3, 3432, 3536, 4 FROM dual
UNION ALL SELECT 1049, 'SIC', 2975, 1790, 3, 2976, 1791, 4 FROM dual
UNION ALL SELECT 1050, 'SIC', 4308, 20, 1, 3511, 3670, 3 FROM dual
UNION ALL SELECT 1051, 'SIC', 3167, 2600, 2, 3171, 2621, 4 FROM dual
UNION ALL SELECT 1052, 'NAICS', 1942, 71, 1, 1952, 71119, 4 FROM dual
UNION ALL SELECT 1053, 'SEC', 2500, 3900, 2, 2509, 3950, 3 FROM dual
UNION ALL SELECT 1054, 'NAICS', 2089, 81232, 4, 2088, 812320, 5 FROM dual
UNION ALL SELECT 1055, 'NAICS', 1600, 53221, 4, 1599, 532210, 5 FROM dual
UNION ALL SELECT 1056, 'SEC', 2791, 20, 1, 2464, 3716, 4 FROM dual
UNION ALL SELECT 1057, 'SIC', 3214, 2800, 2, 3221, 2821, 4 FROM dual
UNION ALL SELECT 1058, 'SIC', 4100, 7600, 2, 4104, 7629, 4 FROM dual
UNION ALL SELECT 1059, 'SIC', 3266, 3050, 3, 3268, 3053, 4 FROM dual
UNION ALL SELECT 1060, 'SIC', 4308, 20, 1, 3267, 3052, 4 FROM dual
UNION ALL SELECT 1061, 'SIC', 4311, 52, 1, 3827, 5430, 3 FROM dual
UNION ALL SELECT 1062, 'SEC', 2796, 70, 1, 2724, 7363, 4 FROM dual
UNION ALL SELECT 1063, 'NAICS', 1979, 713, 2, 1994, 71392, 4 FROM dual
UNION ALL SELECT 1064, 'NAICS', 129, 1153, 3, 131, 11531, 4 FROM dual
UNION ALL SELECT 1065, 'NAICS', 1061, 4246, 3, 1063, 42461, 4 FROM dual
UNION ALL SELECT 1066, 'NAICS', 2037, 81, 1, 2090, 81233, 4 FROM dual
UNION ALL SELECT 1067, 'NAICS', 2071, 812, 2, 2101, 812990, 5 FROM dual
UNION ALL SELECT 1068, 'SIC', 4305, 1, 1, 2805, 130, 3 FROM dual
UNION ALL SELECT 1069, 'NAICS', 206, 236, 2, 208, 23611, 4 FROM dual
UNION ALL SELECT 1070, 'SIC', 3825, 5420, 3, 3826, 5421, 4 FROM dual
UNION ALL SELECT 1071, 'NAICS', 2072, 8121, 3, 2078, 812191, 5 FROM dual
UNION ALL SELECT 1072, 'SIC', 4308, 20, 1, 3490, 3631, 4 FROM dual
UNION ALL SELECT 1073, 'SIC', 4308, 20, 1, 3341, 3315, 4 FROM dual
UNION ALL SELECT 1074, 'SEC', 2791, 20, 1, 2253, 2070, 3 FROM dual
UNION ALL SELECT 1075, 'SEC', 2458, 3700, 2, 2467, 3724, 4 FROM dual
UNION ALL SELECT 1076, 'NAICS', 56, 112, 2, 79, 1124, 3 FROM dual
UNION ALL SELECT 1077, 'NAICS', 181, 221, 2, 194, 221122, 5 FROM dual
UNION ALL SELECT 1078, 'SEC', 2418, 3560, 3, 2423, 3569, 4 FROM dual
UNION ALL SELECT 1079, 'NAICS', 2135, 92, 1, 2196, 926140, 5 FROM dual
UNION ALL SELECT 1080, 'SEC', 2435, 3610, 3, 2437, 3613, 4 FROM dual
UNION ALL SELECT 1081, 'SEC', 2633, 5730, 3, 2634, 5731, 4 FROM dual
UNION ALL SELECT 1082, 'SEC', 2791, 20, 1, 2383, 3390, 3 FROM dual
UNION ALL SELECT 1083, 'NAICS', 2, 111, 2, 33, 111335, 5 FROM dual
UNION ALL SELECT 1084, 'NAICS', 1741, 5614, 3, 1746, 561422, 5 FROM dual
UNION ALL SELECT 1085, 'NAICS', 1625, 541, 2, 1643, 541320, 5 FROM dual
UNION ALL SELECT 1086, 'NAICS', 2038, 811, 2, 2040, 81111, 4 FROM dual
UNION ALL SELECT 1087, 'SIC', 3141, 2450, 3, 3142, 2451, 4 FROM dual
UNION ALL SELECT 1088, 'SIC', 4308, 20, 1, 3031, 2086, 4 FROM dual
UNION ALL SELECT 1089, 'NAICS', 1, 11, 1, 71, 112320, 5 FROM dual
UNION ALL SELECT 1090, 'SEC', 2254, 2080, 3, 2255, 2082, 4 FROM dual
UNION ALL SELECT 1091, 'SIC', 4313, 70, 1, 4055, 7340, 3 FROM dual
UNION ALL SELECT 1092, 'NAICS', 1850, 62, 1, 1940, 624410, 5 FROM dual
UNION ALL SELECT 1093, 'NAICS', 1706, 5419, 3, 1711, 541922, 5 FROM dual
UNION ALL SELECT 1094, 'SIC', 4313, 70, 1, 4035, 7260, 3 FROM dual
UNION ALL SELECT 1095, 'NAICS', 132, 21, 1, 134, 2111, 3 FROM dual
UNION ALL SELECT 1096, 'NAICS', 43, 1119, 3, 45, 11191, 4 FROM dual
UNION ALL SELECT 1097, 'NAICS', 2112, 8133, 3, 2114, 813311, 5 FROM dual
UNION ALL SELECT 1098, 'SIC', 4146, 8000, 2, 4151, 8030, 3 FROM dual
UNION ALL SELECT 1099, 'NAICS', 1536, 524, 2, 1545, 524130, 5 FROM dual
UNION ALL SELECT 1100, 'SIC', 4311, 52, 1, 3833, 5460, 3 FROM dual
UNION ALL SELECT 1101, 'SIC', 3392, 3450, 3, 3393, 3451, 4 FROM dual
UNION ALL SELECT 1102, 'SIC', 3371, 3400, 2, 3415, 3496, 4 FROM dual
UNION ALL SELECT 1103, 'SIC', 4312, 60, 1, 3934, 6090, 3 FROM dual
UNION ALL SELECT 1104, 'SIC', 2955, 1700, 2, 2977, 1793, 4 FROM dual
UNION ALL SELECT 1105, 'SEC', 2649, 6000, 2, 2651, 6021, 4 FROM dual
UNION ALL SELECT 1106, 'SIC', 3167, 2600, 2, 3189, 2679, 4 FROM dual
UNION ALL SELECT 1107, 'SIC', 3822, 5400, 2, 3834, 5461, 4 FROM dual
UNION ALL SELECT 1108, 'SIC', 4017, 7200, 2, 4034, 7251, 4 FROM dual
UNION ALL SELECT 1109, 'NAICS', 1428, 51219, 4, 1429, 512191, 5 FROM dual
UNION ALL SELECT 1110, 'SEC', 2791, 20, 1, 2404, 3510, 3 FROM dual
UNION ALL SELECT 1111, 'SIC', 4314, 90, 1, 4271, 9430, 3 FROM dual
UNION ALL SELECT 1112, 'NAICS', 1480, 52, 1, 1513, 523, 2 FROM dual
UNION ALL SELECT 1113, 'SEC', 2796, 70, 1, 2753, 8000, 2 FROM dual
UNION ALL SELECT 1114, 'SEC', 2796, 70, 1, 2767, 8110, 3 FROM dual
UNION ALL SELECT 1115, 'SIC', 3507, 3660, 3, 3508, 3661, 4 FROM dual
UNION ALL SELECT 1116, 'NAICS', 101, 11311, 4, 100, 113110, 5 FROM dual
UNION ALL SELECT 1117, 'SIC', 4308, 20, 1, 3337, 3300, 2 FROM dual
UNION ALL SELECT 1118, 'SIC', 3255, 2950, 3, 3256, 2951, 4 FROM dual
UNION ALL SELECT 1119, 'SIC', 4305, 1, 1, 2879, 919, 4 FROM dual
UNION ALL SELECT 1120, 'SIC', 4308, 20, 1, 3588, 3951, 4 FROM dual
UNION ALL SELECT 1121, 'SIC', 4158, 8050, 3, 4161, 8059, 4 FROM dual
UNION ALL SELECT 1122, 'NAICS', 2031, 7225, 3, 2033, 722511, 5 FROM dual
UNION ALL SELECT 1123, 'SIC', 4125, 7900, 2, 4128, 7920, 3 FROM dual
UNION ALL SELECT 1124, 'SEC', 2791, 20, 1, 2369, 3281, 4 FROM dual
UNION ALL SELECT 1125, 'SIC', 3526, 3700, 2, 3528, 3711, 4 FROM dual
UNION ALL SELECT 1126, 'NAICS', 1667, 54151, 4, 1669, 541512, 5 FROM dual
UNION ALL SELECT 1127, 'NAICS', 1719, 551, 2, 1721, 55111, 4 FROM dual
UNION ALL SELECT 1128, 'SIC', 3917, 6000, 2, 3933, 6082, 4 FROM dual
UNION ALL SELECT 1129, 'SIC', 4081, 7500, 2, 4095, 7538, 4 FROM dual
UNION ALL SELECT 1130, 'SIC', 4113, 7800, 2, 4115, 7812, 4 FROM dual
UNION ALL SELECT 1131, 'NAICS', 1683, 5417, 3, 1688, 54172, 4 FROM dual
UNION ALL SELECT 1132, 'SIC', 4313, 70, 1, 4145, 7999, 4 FROM dual
UNION ALL SELECT 1133, 'SEC', 2791, 20, 1, 2252, 2060, 3 FROM dual
UNION ALL SELECT 1134, 'SIC', 3654, 4500, 2, 3659, 4522, 4 FROM dual
UNION ALL SELECT 1135, 'SIC', 3660, 4580, 3, 3661, 4581, 4 FROM dual
UNION ALL SELECT 1136, 'NAICS', 1706, 5419, 3, 1716, 541990, 5 FROM dual
UNION ALL SELECT 1137, 'SIC', 4311, 52, 1, 3893, 5940, 3 FROM dual
UNION ALL SELECT 1138, 'SIC', 2847, 700, 2, 2852, 722, 4 FROM dual
UNION ALL SELECT 1139, 'SEC', 2559, 4930, 3, 2561, 4932, 4 FROM dual
UNION ALL SELECT 1140, 'SIC', 4312, 60, 1, 3980, 6513, 4 FROM dual
UNION ALL SELECT 1141, 'SIC', 4309, 40, 1, 3662, 4600, 2 FROM dual
UNION ALL SELECT 1142, 'SEC', 2384, 3400, 2, 2391, 3440, 3 FROM dual
UNION ALL SELECT 1143, 'NAICS', 955, 4234, 3, 963, 42344, 4 FROM dual
UNION ALL SELECT 1144, 'NAICS', 1820, 6113, 3, 1821, 611310, 5 FROM dual
UNION ALL SELECT 1145, 'SIC', 3837, 5500, 2, 3839, 5511, 4 FROM dual
UNION ALL SELECT 1146, 'SIC', 2928, 1450, 3, 2930, 1459, 4 FROM dual
UNION ALL SELECT 1147, 'NAICS', 1015, 424, 2, 1085, 424950, 5 FROM dual
UNION ALL SELECT 1148, 'NAICS', 2136, 921, 2, 2141, 92112, 4 FROM dual
UNION ALL SELECT 1149, 'NAICS', 1706, 5419, 3, 1709, 54192, 4 FROM dual
UNION ALL SELECT 1150, 'NAICS', 1792, 562, 2, 1803, 562219, 5 FROM dual
UNION ALL SELECT 1151, 'SIC', 4313, 70, 1, 4146, 8000, 2 FROM dual
UNION ALL SELECT 1152, 'NAICS', 1954, 71121, 4, 1957, 711219, 5 FROM dual
UNION ALL SELECT 1153, 'SIC', 2847, 700, 2, 2861, 760, 3 FROM dual
UNION ALL SELECT 1154, 'NAICS', 931, 423, 2, 949, 423320, 5 FROM dual
UNION ALL SELECT 1155, 'NAICS', 2073, 81211, 4, 2076, 812113, 5 FROM dual
UNION ALL SELECT 1156, 'NAICS', 1625, 541, 2, 1695, 54183, 4 FROM dual
UNION ALL SELECT 1157, 'SEC', 2789, 10, 1, 2216, 1040, 3 FROM dual
UNION ALL SELECT 1158, 'NAICS', 236, 2381, 3, 251, 238190, 5 FROM dual
UNION ALL SELECT 1159, 'NAICS', 1624, 54, 1, 1690, 541810, 5 FROM dual
UNION ALL SELECT 1160, 'NAICS', 1851, 621, 2, 1887, 62161, 4 FROM dual
UNION ALL SELECT 1161, 'NAICS', 2061, 8114, 3, 2068, 81143, 4 FROM dual
UNION ALL SELECT 1162, 'SIC', 2982, 2000, 2, 2994, 2032, 4 FROM dual
UNION ALL SELECT 1163, 'NAICS', 1480, 52, 1, 1486, 5221, 3 FROM dual
UNION ALL SELECT 1164, 'SEC', 2670, 6200, 2, 2674, 6221, 4 FROM dual
UNION ALL SELECT 1165, 'SIC', 4311, 52, 1, 3908, 5983, 4 FROM dual
UNION ALL SELECT 1166, 'NAICS', 2005, 7211, 3, 2006, 721110, 5 FROM dual
UNION ALL SELECT 1167, 'SEC', 2424, 3570, 3, 2426, 3572, 4 FROM dual
UNION ALL SELECT 1168, 'SEC', 2703, 6700, 2, 2707, 6798, 4 FROM dual
UNION ALL SELECT 1169, 'NAICS', 2037, 81, 1, 2050, 811192, 5 FROM dual
UNION ALL SELECT 1170, 'SEC', 2791, 20, 1, 2278, 2421, 4 FROM dual
UNION ALL SELECT 1171, 'NAICS', 138, 212, 2, 144, 2122, 3 FROM dual
UNION ALL SELECT 1172, 'NAICS', 2136, 921, 2, 2143, 92113, 4 FROM dual
UNION ALL SELECT 1173, 'SIC', 4081, 7500, 2, 4091, 7533, 4 FROM dual
UNION ALL SELECT 1174, 'SEC', 2792, 40, 1, 2546, 4832, 4 FROM dual
UNION ALL SELECT 1175, 'NAICS', 2151, 9221, 3, 2154, 922120, 5 FROM dual
UNION ALL SELECT 1176, 'NAICS', 1014, 42399, 4, 1013, 423990, 5 FROM dual
UNION ALL SELECT 1177, 'NAICS', 930, 42, 1, 995, 42382, 4 FROM dual
UNION ALL SELECT 1178, 'SIC', 3012, 2060, 3, 3018, 2067, 4 FROM dual
UNION ALL SELECT 1179, 'SIC', 2820, 180, 3, 2821, 181, 4 FROM dual
UNION ALL SELECT 1180, 'NAICS', 2103, 813, 2, 2119, 81341, 4 FROM dual
UNION ALL SELECT 1181, 'SIC', 3026, 2080, 3, 3031, 2086, 4 FROM dual
UNION ALL SELECT 1182, 'NAICS', 1043, 42444, 4, 1042, 424440, 5 FROM dual
UNION ALL SELECT 1183, 'SIC', 4306, 10, 1, 2893, 1044, 4 FROM dual
UNION ALL SELECT 1184, 'NAICS', 1953, 7112, 3, 1955, 711211, 5 FROM dual
UNION ALL SELECT 1185, 'SEC', 2791, 20, 1, 2487, 3829, 4 FROM dual
UNION ALL SELECT 1186, 'SIC', 4313, 70, 1, 4137, 7951, 4 FROM dual
UNION ALL SELECT 1187, 'NAICS', 89, 1129, 3, 90, 112910, 5 FROM dual
UNION ALL SELECT 1188, 'NAICS', 138, 212, 2, 147, 21222, 4 FROM dual
UNION ALL SELECT 1189, 'SIC', 4308, 20, 1, 3446, 3553, 4 FROM dual
UNION ALL SELECT 1190, 'NAICS', 1, 11, 1, 128, 11521, 4 FROM dual
UNION ALL SELECT 1191, 'NAICS', 205, 23, 1, 214, 236210, 5 FROM dual
UNION ALL SELECT 1192, 'SIC', 4275, 9450, 3, 4276, 9451, 4 FROM dual
UNION ALL SELECT 1193, 'SEC', 2793, 50, 1, 2572, 5030, 3 FROM dual
UNION ALL SELECT 1194, 'SIC', 2987, 2020, 3, 2992, 2026, 4 FROM dual
UNION ALL SELECT 1195, 'NAICS', 1015, 424, 2, 1051, 42448, 4 FROM dual
UNION ALL SELECT 1196, 'SIC', 4308, 20, 1, 3019, 2068, 4 FROM dual
UNION ALL SELECT 1197, 'SEC', 2649, 6000, 2, 2655, 6035, 4 FROM dual
UNION ALL SELECT 1198, 'SIC', 4309, 40, 1, 3635, 4311, 4 FROM dual
UNION ALL SELECT 1199, 'NAICS', 1785, 5619, 3, 1787, 56191, 4 FROM dual
UNION ALL SELECT 1200, 'SIC', 4310, 50, 1, 3803, 5199, 4 FROM dual
UNION ALL SELECT 1201, 'NAICS', 84, 1125, 3, 88, 112519, 5 FROM dual
UNION ALL SELECT 1202, 'NAICS', 183, 22111, 4, 187, 221114, 5 FROM dual
UNION ALL SELECT 1203, 'NAICS', 2096, 81292, 4, 2098, 812922, 5 FROM dual
UNION ALL SELECT 1204, 'NAICS', 1604, 53223, 4, 1603, 532230, 5 FROM dual
UNION ALL SELECT 1205, 'NAICS', 1706, 5419, 3, 1714, 541940, 5 FROM dual
UNION ALL SELECT 1206, 'NAICS', 43, 1119, 3, 49, 11193, 4 FROM dual
UNION ALL SELECT 1207, 'NAICS', 931, 423, 2, 1006, 42391, 4 FROM dual
UNION ALL SELECT 1208, 'NAICS', 2135, 92, 1, 2138, 921110, 5 FROM dual
UNION ALL SELECT 1209, 'SIC', 4245, 8990, 3, 4246, 8999, 4 FROM dual
UNION ALL SELECT 1210, 'NAICS', 1528, 52391, 4, 1527, 523910, 5 FROM dual
UNION ALL SELECT 1211, 'NAICS', 1970, 7121, 3, 1976, 71213, 4 FROM dual
UNION ALL SELECT 1212, 'SIC', 4311, 52, 1, 3880, 5735, 4 FROM dual
UNION ALL SELECT 1213, 'SIC', 4308, 20, 1, 3259, 2992, 4 FROM dual
UNION ALL SELECT 1214, 'NAICS', 7, 11112, 4, 6, 111120, 5 FROM dual
UNION ALL SELECT 1215, 'NAICS', 1990, 7139, 3, 2001, 713990, 5 FROM dual
UNION ALL SELECT 1216, 'SIC', 4308, 20, 1, 3439, 3545, 4 FROM dual
UNION ALL SELECT 1217, 'NAICS', 1894, 622, 2, 1897, 62211, 4 FROM dual
UNION ALL SELECT 1218, 'SEC', 2791, 20, 1, 2303, 2710, 3 FROM dual
UNION ALL SELECT 1219, 'SIC', 4313, 70, 1, 4010, 7020, 3 FROM dual
UNION ALL SELECT 1220, 'NAICS', 1591, 532, 2, 1595, 532112, 5 FROM dual
UNION ALL SELECT 1221, 'NAICS', 1626, 5411, 3, 1631, 54119, 4 FROM dual
UNION ALL SELECT 1222, 'NAICS', 2039, 8111, 3, 2047, 811122, 5 FROM dual
UNION ALL SELECT 1223, 'NAICS', 1480, 52, 1, 1567, 525990, 5 FROM dual
UNION ALL SELECT 1224, 'NAICS', 2003, 72, 1, 2016, 721214, 5 FROM dual
UNION ALL SELECT 1225, 'SIC', 3893, 5940, 3, 3895, 5942, 4 FROM dual
UNION ALL SELECT 1226, 'NAICS', 1431, 5122, 3, 1440, 512290, 5 FROM dual
UNION ALL SELECT 1227, 'SIC', 3667, 4700, 2, 3669, 4724, 4 FROM dual
UNION ALL SELECT 1228, 'NAICS', 2021, 7223, 3, 2025, 72232, 4 FROM dual
UNION ALL SELECT 1229, 'NAICS', 2037, 81, 1, 2058, 8113, 3 FROM dual
UNION ALL SELECT 1230, 'SEC', 2791, 20, 1, 2492, 3844, 4 FROM dual
UNION ALL SELECT 1231, 'NAICS', 1571, 5311, 3, 1578, 531190, 5 FROM dual
UNION ALL SELECT 1232, 'SIC', 4308, 20, 1, 3147, 2499, 4 FROM dual
UNION ALL SELECT 1233, 'SIC', 3917, 6000, 2, 3920, 6019, 4 FROM dual
UNION ALL SELECT 1234, 'SEC', 2791, 20, 1, 2482, 3823, 4 FROM dual
UNION ALL SELECT 1235, 'SIC', 4308, 20, 1, 3149, 2510, 3 FROM dual
UNION ALL SELECT 1236, 'SEC', 2794, 52, 1, 2621, 5530, 3 FROM dual
UNION ALL SELECT 1237, 'SIC', 3225, 2830, 3, 3226, 2833, 4 FROM dual
UNION ALL SELECT 1238, 'NAICS', 1015, 424, 2, 1022, 42413, 4 FROM dual
UNION ALL SELECT 1239, 'SIC', 4313, 70, 1, 4048, 7323, 4 FROM dual
UNION ALL SELECT 1240, 'SIC', 3613, 4140, 3, 3614, 4141, 4 FROM dual
UNION ALL SELECT 1241, 'SIC', 3948, 6200, 2, 3953, 6230, 3 FROM dual
UNION ALL SELECT 1242, 'NAICS', 2037, 81, 1, 2125, 813930, 5 FROM dual
UNION ALL SELECT 1243, 'SIC', 4136, 7950, 3, 4138, 7952, 4 FROM dual
UNION ALL SELECT 1244, 'NAICS', 36, 1114, 3, 37, 11141, 4 FROM dual
UNION ALL SELECT 1245, 'SIC', 3844, 5540, 3, 3845, 5541, 4 FROM dual
UNION ALL SELECT 1246, 'SIC', 4309, 40, 1, 3648, 4489, 4 FROM dual
UNION ALL SELECT 1247, 'NAICS', 1569, 53, 1, 1609, 5323, 3 FROM dual
UNION ALL SELECT 1248, 'SIC', 4308, 20, 1, 3569, 3845, 4 FROM dual
UNION ALL SELECT 1249, 'SIC', 3434, 3540, 3, 3437, 3543, 4 FROM dual
UNION ALL SELECT 1250, 'SIC', 4312, 60, 1, 3921, 6020, 3 FROM dual
UNION ALL SELECT 1251, 'NAICS', 98, 113, 2, 102, 1132, 3 FROM dual
UNION ALL SELECT 1252, 'NAICS', 1865, 62133, 4, 1864, 621330, 5 FROM dual
UNION ALL SELECT 1253, 'SEC', 2794, 52, 1, 2645, 5945, 4 FROM dual
UNION ALL SELECT 1254, 'NAICS', 2037, 81, 1, 2068, 81143, 4 FROM dual
UNION ALL SELECT 1255, 'NAICS', 1725, 56, 1, 1743, 56141, 4 FROM dual
UNION ALL SELECT 1256, 'NAICS', 2150, 922, 2, 2152, 922110, 5 FROM dual
UNION ALL SELECT 1257, 'SIC', 2891, 1040, 3, 2893, 1044, 4 FROM dual
UNION ALL SELECT 1258, 'NAICS', 1402, 51, 1, 1457, 517210, 5 FROM dual
UNION ALL SELECT 1259, 'NAICS', 1850, 62, 1, 1897, 62211, 4 FROM dual
UNION ALL SELECT 1260, 'SIC', 3083, 2300, 2, 3118, 2393, 4 FROM dual
UNION ALL SELECT 1261, 'SIC', 3253, 2910, 3, 3254, 2911, 4 FROM dual
UNION ALL SELECT 1262, 'SEC', 2403, 3500, 2, 2404, 3510, 3 FROM dual
UNION ALL SELECT 1263, 'SIC', 3371, 3400, 2, 3406, 3483, 4 FROM dual
UNION ALL SELECT 1264, 'NAICS', 1569, 53, 1, 1596, 532120, 5 FROM dual
UNION ALL SELECT 1265, 'NAICS', 1689, 5418, 3, 1695, 54183, 4 FROM dual
UNION ALL SELECT 1266, 'SIC', 4046, 7320, 3, 4048, 7323, 4 FROM dual
UNION ALL SELECT 1267, 'NAICS', 1942, 71, 1, 1967, 711510, 5 FROM dual
UNION ALL SELECT 1268, 'SEC', 2431, 3580, 3, 2432, 3585, 4 FROM dual
UNION ALL SELECT 1269, 'SIC', 3626, 4220, 3, 3630, 4226, 4 FROM dual
UNION ALL SELECT 1270, 'SIC', 4128, 7920, 3, 4130, 7929, 4 FROM dual
UNION ALL SELECT 1271, 'NAICS', 982, 4237, 3, 990, 42374, 4 FROM dual
UNION ALL SELECT 1272, 'SIC', 3124, 2400, 2, 3126, 2411, 4 FROM dual
UNION ALL SELECT 1273, 'SIC', 4313, 70, 1, 4157, 8049, 4 FROM dual
UNION ALL SELECT 1274, 'NAICS', 157, 21231, 4, 161, 212319, 5 FROM dual
UNION ALL SELECT 1275, 'SIC', 4313, 70, 1, 4028, 7221, 4 FROM dual
UNION ALL SELECT 1276, 'SIC', 4313, 70, 1, 4175, 8100, 2 FROM dual
UNION ALL SELECT 1277, 'NAICS', 1485, 522, 2, 1492, 52213, 4 FROM dual
UNION ALL SELECT 1278, 'SIC', 3344, 3320, 3, 3348, 3325, 4 FROM dual
UNION ALL SELECT 1279, 'SIC', 3715, 5000, 2, 3741, 5063, 4 FROM dual
UNION ALL SELECT 1280, 'SEC', 2789, 10, 1, 2220, 1221, 4 FROM dual
UNION ALL SELECT 1281, 'NAICS', 1402, 51, 1, 1408, 51112, 4 FROM dual
UNION ALL SELECT 1282, 'NAICS', 2203, 92711, 4, 2202, 927110, 5 FROM dual
UNION ALL SELECT 1283, 'NAICS', 931, 423, 2, 954, 42339, 4 FROM dual
UNION ALL SELECT 1284, 'SIC', 3041, 2100, 2, 3049, 2141, 4 FROM dual
UNION ALL SELECT 1285, 'SIC', 4311, 52, 1, 3841, 5521, 4 FROM dual
UNION ALL SELECT 1286, 'NAICS', 1, 11, 1, 17, 111191, 5 FROM dual
UNION ALL SELECT 1287, 'NAICS', 117, 115, 2, 130, 115310, 5 FROM dual
UNION ALL SELECT 1288, 'NAICS', 984, 42371, 4, 983, 423710, 5 FROM dual
UNION ALL SELECT 1289, 'SIC', 4308, 20, 1, 3391, 3449, 4 FROM dual
UNION ALL SELECT 1290, 'NAICS', 1442, 515, 2, 1444, 51511, 4 FROM dual
UNION ALL SELECT 1291, 'SEC', 2796, 70, 1, 2778, 8731, 4 FROM dual
UNION ALL SELECT 1292, 'NAICS', 138, 212, 2, 159, 212312, 5 FROM dual
UNION ALL SELECT 1293, 'NAICS', 1620, 533, 2, 1623, 53311, 4 FROM dual
UNION ALL SELECT 1294, 'NAICS', 1402, 51, 1, 1416, 5112, 3 FROM dual
UNION ALL SELECT 1295, 'NAICS', 1624, 54, 1, 1665, 54149, 4 FROM dual
UNION ALL SELECT 1296, 'SEC', 2408, 3530, 3, 2409, 3531, 4 FROM dual
UNION ALL SELECT 1297, 'NAICS', 182, 2211, 3, 193, 221121, 5 FROM dual
UNION ALL SELECT 1298, 'NAICS', 930, 42, 1, 1026, 4243, 3 FROM dual
UNION ALL SELECT 1299, 'SIC', 3380, 3430, 3, 3381, 3431, 4 FROM dual
UNION ALL SELECT 1300, 'NAICS', 1812, 61, 1, 1839, 611620, 5 FROM dual
UNION ALL SELECT 1301, 'SEC', 2795, 60, 1, 2675, 6280, 3 FROM dual
UNION ALL SELECT 1302, 'NAICS', 1812, 61, 1, 1815, 611110, 5 FROM dual
UNION ALL SELECT 1303, 'SIC', 4308, 20, 1, 3057, 2240, 3 FROM dual
UNION ALL SELECT 1304, 'SIC', 3371, 3400, 2, 3389, 3446, 4 FROM dual
UNION ALL SELECT 1305, 'NAICS', 2080, 8122, 3, 2083, 812220, 5 FROM dual
UNION ALL SELECT 1306, 'SEC', 2283, 2500, 2, 2288, 2530, 3 FROM dual
UNION ALL SELECT 1307, 'SIC', 3461, 3570, 3, 3462, 3571, 4 FROM dual
UNION ALL SELECT 1308, 'NAICS', 57, 1121, 3, 59, 112111, 5 FROM dual
UNION ALL SELECT 1309, 'NAICS', 1990, 7139, 3, 1991, 713910, 5 FROM dual
UNION ALL SELECT 1310, 'SIC', 2948, 1600, 2, 2949, 1610, 3 FROM dual
UNION ALL SELECT 1311, 'SIC', 4309, 40, 1, 3694, 4910, 3 FROM dual
UNION ALL SELECT 1312, 'SIC', 2826, 210, 3, 2831, 219, 4 FROM dual
UNION ALL SELECT 1313, 'SEC', 2796, 70, 1, 2771, 8350, 3 FROM dual
UNION ALL SELECT 1314, 'SIC', 4146, 8000, 2, 4164, 8063, 4 FROM dual
UNION ALL SELECT 1315, 'SIC', 4305, 1, 1, 2824, 191, 4 FROM dual
UNION ALL SELECT 1316, 'NAICS', 1570, 531, 2, 1582, 53121, 4 FROM dual
UNION ALL SELECT 1317, 'SIC', 4308, 20, 1, 3069, 2262, 4 FROM dual
UNION ALL SELECT 1318, 'NAICS', 2095, 81291, 4, 2094, 812910, 5 FROM dual
UNION ALL SELECT 1319, 'SIC', 3903, 5960, 3, 3906, 5963, 4 FROM dual
UNION ALL SELECT 1320, 'SIC', 3917, 6000, 2, 3927, 6036, 4 FROM dual
UNION ALL SELECT 1321, 'SIC', 3917, 6000, 2, 3928, 6060, 3 FROM dual
UNION ALL SELECT 1322, 'NAICS', 198, 2213, 3, 201, 221320, 5 FROM dual
UNION ALL SELECT 1323, 'NAICS', 1015, 424, 2, 1018, 42411, 4 FROM dual
UNION ALL SELECT 1324, 'SEC', 2496, 3860, 3, 2497, 3861, 4 FROM dual
UNION ALL SELECT 1325, 'SIC', 3999, 6730, 3, 4000, 6732, 4 FROM dual
UNION ALL SELECT 1326, 'NAICS', 1, 11, 1, 37, 11141, 4 FROM dual
UNION ALL SELECT 1327, 'NAICS', 1625, 541, 2, 1669, 541512, 5 FROM dual
UNION ALL SELECT 1328, 'SIC', 4308, 20, 1, 3330, 3290, 3 FROM dual
UNION ALL SELECT 1329, 'SIC', 3174, 2650, 3, 3176, 2653, 4 FROM dual
UNION ALL SELECT 1330, 'SEC', 2476, 3800, 2, 2479, 3820, 3 FROM dual
UNION ALL SELECT 1331, 'SIC', 4308, 20, 1, 3350, 3331, 4 FROM dual
UNION ALL SELECT 1332, 'NAICS', 1404, 5111, 3, 1405, 511110, 5 FROM dual
UNION ALL SELECT 1333, 'NAICS', 2013, 7212, 3, 2014, 72121, 4 FROM dual
UNION ALL SELECT 1334, 'NAICS', 1004, 4239, 3, 1010, 42393, 4 FROM dual
UNION ALL SELECT 1335, 'SIC', 3715, 5000, 2, 3756, 5090, 3 FROM dual
UNION ALL SELECT 1336, 'NAICS', 2004, 721, 2, 2011, 721191, 5 FROM dual
UNION ALL SELECT 1337, 'NAICS', 1546, 52413, 4, 1545, 524130, 5 FROM dual
UNION ALL SELECT 1338, 'NAICS', 2102, 81299, 4, 2101, 812990, 5 FROM dual
UNION ALL SELECT 1339, 'SIC', 4309, 40, 1, 3659, 4522, 4 FROM dual
UNION ALL SELECT 1340, 'SIC', 4313, 70, 1, 4198, 8351, 4 FROM dual
UNION ALL SELECT 1341, 'NAICS', 1813, 611, 2, 1842, 61163, 4 FROM dual
UNION ALL SELECT 1342, 'SIC', 4308, 20, 1, 3359, 3355, 4 FROM dual
UNION ALL SELECT 1343, 'SIC', 4313, 70, 1, 4089, 7530, 3 FROM dual
UNION ALL SELECT 1344, 'SIC', 3715, 5000, 2, 3746, 5074, 4 FROM dual
UNION ALL SELECT 1345, 'SIC', 3581, 3930, 3, 3582, 3931, 4 FROM dual
UNION ALL SELECT 1346, 'NAICS', 2037, 81, 1, 2055, 811212, 5 FROM dual
UNION ALL SELECT 1347, 'NAICS', 2151, 9221, 3, 2153, 92211, 4 FROM dual
UNION ALL SELECT 1348, 'SIC', 2982, 2000, 2, 3040, 2099, 4 FROM dual
UNION ALL SELECT 1349, 'NAICS', 930, 42, 1, 974, 42352, 4 FROM dual
UNION ALL SELECT 1350, 'NAICS', 1424, 51212, 4, 1423, 512120, 5 FROM dual
UNION ALL SELECT 1351, 'NAICS', 1634, 5412, 3, 1635, 54121, 4 FROM dual
UNION ALL SELECT 1352, 'SIC', 4312, 60, 1, 3944, 6159, 4 FROM dual
UNION ALL SELECT 1353, 'SIC', 4313, 70, 1, 4129, 7922, 4 FROM dual
UNION ALL SELECT 1354, 'SEC', 2792, 40, 1, 2549, 4841, 4 FROM dual
UNION ALL SELECT 1355, 'SIC', 3762, 5100, 2, 3801, 5194, 4 FROM dual
UNION ALL SELECT 1356, 'NAICS', 182, 2211, 3, 186, 221113, 5 FROM dual
UNION ALL SELECT 1357, 'SEC', 2568, 5000, 2, 2577, 5050, 3 FROM dual
UNION ALL SELECT 1358, 'NAICS', 1592, 5321, 3, 1596, 532120, 5 FROM dual
UNION ALL SELECT 1359, 'NAICS', 2108, 81321, 4, 2110, 813212, 5 FROM dual
UNION ALL SELECT 1360, 'SIC', 2868, 800, 2, 2872, 831, 4 FROM dual
UNION ALL SELECT 1361, 'NAICS', 2037, 81, 1, 2091, 812331, 5 FROM dual
UNION ALL SELECT 1362, 'SIC', 4308, 20, 1, 3561, 3826, 4 FROM dual
UNION ALL SELECT 1363, 'SIC', 3724, 5030, 3, 3725, 5031, 4 FROM dual
UNION ALL SELECT 1364, 'NAICS', 2189, 9261, 3, 2191, 92611, 4 FROM dual
UNION ALL SELECT 1365, 'SEC', 2795, 60, 1, 2690, 6399, 4 FROM dual
UNION ALL SELECT 1366, 'NAICS', 1942, 71, 1, 1955, 711211, 5 FROM dual
UNION ALL SELECT 1367, 'SEC', 2791, 20, 1, 2282, 2452, 4 FROM dual
UNION ALL SELECT 1368, 'SIC', 3349, 3330, 3, 3351, 3334, 4 FROM dual
UNION ALL SELECT 1369, 'SIC', 4203, 8400, 2, 4206, 8420, 3 FROM dual
UNION ALL SELECT 1370, 'NAICS', 181, 221, 2, 190, 221117, 5 FROM dual
UNION ALL SELECT 1371, 'NAICS', 1583, 5313, 3, 1584, 53131, 4 FROM dual
UNION ALL SELECT 1372, 'NAICS', 1532, 52393, 4, 1531, 523930, 5 FROM dual
UNION ALL SELECT 1373, 'SIC', 3544, 3760, 3, 3546, 3764, 4 FROM dual
UNION ALL SELECT 1374, 'NAICS', 1402, 51, 1, 1417, 511210, 5 FROM dual
UNION ALL SELECT 1375, 'NAICS', 208, 23611, 4, 211, 236117, 5 FROM dual
UNION ALL SELECT 1376, 'SIC', 4313, 70, 1, 4105, 7630, 3 FROM dual
UNION ALL SELECT 1377, 'NAICS', 930, 42, 1, 935, 423120, 5 FROM dual
UNION ALL SELECT 1378, 'SEC', 2742, 7820, 3, 2743, 7822, 4 FROM dual
UNION ALL SELECT 1379, 'NAICS', 930, 42, 1, 1065, 42469, 4 FROM dual
UNION ALL SELECT 1380, 'NAICS', 930, 42, 1, 937, 423130, 5 FROM dual
UNION ALL SELECT 1381, 'SIC', 2921, 1420, 3, 2924, 1429, 4 FROM dual
UNION ALL SELECT 1382, 'SIC', 4308, 20, 1, 3303, 3211, 4 FROM dual
UNION ALL SELECT 1383, 'SIC', 4308, 20, 1, 3241, 2870, 3 FROM dual
UNION ALL SELECT 1384, 'SIC', 2876, 910, 3, 2877, 912, 4 FROM dual
UNION ALL SELECT 1385, 'NAICS', 2135, 92, 1, 2172, 923130, 5 FROM dual
UNION ALL SELECT 1386, 'SIC', 3511, 3670, 3, 3516, 3676, 4 FROM dual
UNION ALL SELECT 1387, 'SIC', 3937, 6100, 2, 3940, 6140, 3 FROM dual
UNION ALL SELECT 1388, 'NAICS', 236, 2381, 3, 241, 238130, 5 FROM dual
UNION ALL SELECT 1389, 'NAICS', 2037, 81, 1, 2059, 811310, 5 FROM dual
UNION ALL SELECT 1390, 'NAICS', 2037, 81, 1, 2049, 811191, 5 FROM dual
UNION ALL SELECT 1391, 'NAICS', 56, 112, 2, 90, 112910, 5 FROM dual
UNION ALL SELECT 1392, 'SIC', 4313, 70, 1, 4057, 7349, 4 FROM dual
UNION ALL SELECT 1393, 'SEC', 2403, 3500, 2, 2423, 3569, 4 FROM dual
UNION ALL SELECT 1394, 'SIC', 4308, 20, 1, 3148, 2500, 2 FROM dual
UNION ALL SELECT 1395, 'NAICS', 56, 112, 2, 59, 112111, 5 FROM dual
UNION ALL SELECT 1396, 'NAICS', 205, 23, 1, 221, 23711, 4 FROM dual
UNION ALL SELECT 1397, 'NAICS', 2003, 72, 1, 2013, 7212, 3 FROM dual
UNION ALL SELECT 1398, 'SIC', 3715, 5000, 2, 3716, 5010, 3 FROM dual
UNION ALL SELECT 1399, 'SIC', 3701, 4930, 3, 3703, 4932, 4 FROM dual
UNION ALL SELECT 1400, 'SIC', 3940, 6140, 3, 3941, 6141, 4 FROM dual
UNION ALL SELECT 1401, 'SIC', 4309, 40, 1, 3625, 4215, 4 FROM dual
UNION ALL SELECT 1402, 'SIC', 4311, 52, 1, 3855, 5610, 3 FROM dual
UNION ALL SELECT 1403, 'NAICS', 2037, 81, 1, 2126, 81393, 4 FROM dual
UNION ALL SELECT 1404, 'SEC', 2476, 3800, 2, 2486, 3827, 4 FROM dual
UNION ALL SELECT 1405, 'NAICS', 1536, 524, 2, 1537, 5241, 3 FROM dual
UNION ALL SELECT 1406, 'SIC', 3419, 3500, 2, 3475, 3592, 4 FROM dual
UNION ALL SELECT 1407, 'NAICS', 1470, 519, 2, 1474, 519120, 5 FROM dual
UNION ALL SELECT 1408, 'SIC', 3012, 2060, 3, 3017, 2066, 4 FROM dual
UNION ALL SELECT 1409, 'NAICS', 226, 2372, 3, 228, 23721, 4 FROM dual
UNION ALL SELECT 1410, 'SIC', 3050, 2200, 2, 3058, 2241, 4 FROM dual
UNION ALL SELECT 1411, 'NAICS', 1049, 42447, 4, 1048, 424470, 5 FROM dual
UNION ALL SELECT 1412, 'NAICS', 1908, 6232, 3, 1910, 62321, 4 FROM dual
UNION ALL SELECT 1413, 'NAICS', 2, 111, 2, 24, 111310, 5 FROM dual
UNION ALL SELECT 1414, 'SEC', 2791, 20, 1, 2354, 3140, 3 FROM dual
UNION ALL SELECT 1415, 'NAICS', 180, 22, 1, 181, 221, 2 FROM dual
UNION ALL SELECT 1416, 'SIC', 2813, 170, 3, 2814, 171, 4 FROM dual
UNION ALL SELECT 1417, 'NAICS', 991, 4238, 3, 995, 42382, 4 FROM dual
UNION ALL SELECT 1418, 'NAICS', 931, 423, 2, 970, 4235, 3 FROM dual
UNION ALL SELECT 1419, 'NAICS', 1726, 561, 2, 1756, 561492, 5 FROM dual
UNION ALL SELECT 1420, 'SIC', 2901, 1200, 2, 2907, 1240, 3 FROM dual
UNION ALL SELECT 1421, 'NAICS', 2037, 81, 1, 2115, 813312, 5 FROM dual
UNION ALL SELECT 1422, 'SEC', 2791, 20, 1, 2260, 2110, 3 FROM dual
UNION ALL SELECT 1423, 'SIC', 3020, 2070, 3, 3025, 2079, 4 FROM dual
UNION ALL SELECT 1424, 'NAICS', 931, 423, 2, 996, 423830, 5 FROM dual
UNION ALL SELECT 1425, 'NAICS', 2, 111, 2, 32, 111334, 5 FROM dual
UNION ALL SELECT 1426, 'NAICS', 1725, 56, 1, 1791, 56199, 4 FROM dual
UNION ALL SELECT 1427, 'NAICS', 181, 221, 2, 192, 22112, 4 FROM dual
UNION ALL SELECT 1428, 'NAICS', 2037, 81, 1, 2113, 81331, 4 FROM dual
UNION ALL SELECT 1429, 'NAICS', 1725, 56, 1, 1807, 562920, 5 FROM dual
UNION ALL SELECT 1430, 'SIC', 3526, 3700, 2, 3550, 3795, 4 FROM dual
UNION ALL SELECT 1431, 'SEC', 2685, 6350, 3, 2686, 6351, 4 FROM dual
UNION ALL SELECT 1432, 'NAICS', 2037, 81, 1, 2040, 81111, 4 FROM dual
UNION ALL SELECT 1433, 'SIC', 3680, 4800, 2, 3688, 4833, 4 FROM dual
UNION ALL SELECT 1434, 'NAICS', 1825, 61141, 4, 1824, 611410, 5 FROM dual
UNION ALL SELECT 1435, 'SIC', 3882, 5800, 2, 3883, 5810, 3 FROM dual
UNION ALL SELECT 1436, 'SIC', 4312, 60, 1, 3988, 6541, 4 FROM dual
UNION ALL SELECT 1437, 'SIC', 4306, 10, 1, 2922, 1422, 4 FROM dual
UNION ALL SELECT 1438, 'NAICS', 205, 23, 1, 218, 237, 2 FROM dual
UNION ALL SELECT 1439, 'NAICS', 1082, 42493, 4, 1081, 424930, 5 FROM dual
UNION ALL SELECT 1440, 'SIC', 3468, 3580, 3, 3470, 3582, 4 FROM dual
UNION ALL SELECT 1441, 'SIC', 4312, 60, 1, 4006, 6799, 4 FROM dual
UNION ALL SELECT 1442, 'SIC', 4307, 15, 1, 2964, 1742, 4 FROM dual
UNION ALL SELECT 1443, 'SIC', 4308, 20, 1, 3433, 3537, 4 FROM dual
UNION ALL SELECT 1444, 'NAICS', 1431, 5122, 3, 1438, 512240, 5 FROM dual
UNION ALL SELECT 1445, 'NAICS', 1569, 53, 1, 1583, 5313, 3 FROM dual
UNION ALL SELECT 1446, 'NAICS', 2009, 72112, 4, 2008, 721120, 5 FROM dual
UNION ALL SELECT 1447, 'SIC', 4308, 20, 1, 3355, 3350, 3 FROM dual
UNION ALL SELECT 1448, 'NAICS', 1547, 5242, 3, 1550, 52429, 4 FROM dual
UNION ALL SELECT 1449, 'NAICS', 1402, 51, 1, 1450, 515210, 5 FROM dual
UNION ALL SELECT 1450, 'NAICS', 1787, 56191, 4, 1786, 561910, 5 FROM dual
UNION ALL SELECT 1451, 'SEC', 2583, 5070, 3, 2584, 5072, 4 FROM dual
UNION ALL SELECT 1452, 'SIC', 2982, 2000, 2, 2984, 2011, 4 FROM dual
UNION ALL SELECT 1453, 'SIC', 4308, 20, 1, 3524, 3695, 4 FROM dual
UNION ALL SELECT 1454, 'SIC', 3371, 3400, 2, 3378, 3425, 4 FROM dual
UNION ALL SELECT 1455, 'NAICS', 931, 423, 2, 946, 4233, 3 FROM dual
UNION ALL SELECT 1456, 'SEC', 2796, 70, 1, 2779, 8734, 4 FROM dual
UNION ALL SELECT 1457, 'SIC', 4306, 10, 1, 2929, 1455, 4 FROM dual
UNION ALL SELECT 1458, 'SIC', 3555, 3820, 3, 3556, 3821, 4 FROM dual
UNION ALL SELECT 1459, 'NAICS', 2085, 8123, 3, 2088, 812320, 5 FROM dual
UNION ALL SELECT 1460, 'SEC', 2793, 50, 1, 2579, 5060, 3 FROM dual
UNION ALL SELECT 1461, 'SIC', 3296, 3170, 3, 3297, 3171, 4 FROM dual
UNION ALL SELECT 1462, 'NAICS', 1537, 5241, 3, 1540, 524114, 5 FROM dual
UNION ALL SELECT 1463, 'NAICS', 1015, 424, 2, 1046, 424460, 5 FROM dual
UNION ALL SELECT 1464, 'NAICS', 1004, 4239, 3, 1012, 42394, 4 FROM dual
UNION ALL SELECT 1465, 'NAICS', 2135, 92, 1, 2180, 924120, 5 FROM dual
UNION ALL SELECT 1466, 'SIC', 2902, 1220, 3, 2904, 1222, 4 FROM dual
UNION ALL SELECT 1467, 'SEC', 2380, 3350, 3, 2381, 3357, 4 FROM dual
UNION ALL SELECT 1468, 'SEC', 2434, 3600, 2, 2445, 3652, 4 FROM dual
UNION ALL SELECT 1469, 'SEC', 2704, 6790, 3, 2707, 6798, 4 FROM dual
UNION ALL SELECT 1470, 'NAICS', 1, 11, 1, 5, 11111, 4 FROM dual
UNION ALL SELECT 1471, 'SIC', 3419, 3500, 2, 3478, 3596, 4 FROM dual
UNION ALL SELECT 1472, 'SEC', 2260, 2110, 3, 2261, 2111, 4 FROM dual
UNION ALL SELECT 1473, 'SIC', 3715, 5000, 2, 3735, 5048, 4 FROM dual
UNION ALL SELECT 1474, 'NAICS', 2084, 81222, 4, 2083, 812220, 5 FROM dual
UNION ALL SELECT 1475, 'SIC', 3886, 5900, 2, 3891, 5930, 3 FROM dual
UNION ALL SELECT 1476, 'NAICS', 931, 423, 2, 997, 42383, 4 FROM dual
UNION ALL SELECT 1477, 'SEC', 2670, 6200, 2, 2675, 6280, 3 FROM dual
UNION ALL SELECT 1478, 'SIC', 4308, 20, 1, 3141, 2450, 3 FROM dual
UNION ALL SELECT 1479, 'SIC', 3605, 4100, 2, 3619, 4173, 4 FROM dual
UNION ALL SELECT 1480, 'NAICS', 1480, 52, 1, 1496, 522210, 5 FROM dual
UNION ALL SELECT 1481, 'SIC', 3093, 2330, 3, 3097, 2339, 4 FROM dual
UNION ALL SELECT 1482, 'SIC', 2918, 1400, 2, 2921, 1420, 3 FROM dual
UNION ALL SELECT 1483, 'SIC', 3854, 5600, 2, 3863, 5650, 3 FROM dual
UNION ALL SELECT 1484, 'SIC', 4313, 70, 1, 4029, 7230, 3 FROM dual
UNION ALL SELECT 1485, 'SIC', 3676, 4780, 3, 3679, 4789, 4 FROM dual
UNION ALL SELECT 1486, 'NAICS', 205, 23, 1, 232, 2379, 3 FROM dual
UNION ALL SELECT 1487, 'SIC', 4308, 20, 1, 3304, 3220, 3 FROM dual
UNION ALL SELECT 1488, 'NAICS', 156, 2123, 3, 159, 212312, 5 FROM dual
UNION ALL SELECT 1489, 'NAICS', 1920, 624, 2, 1938, 62431, 4 FROM dual
UNION ALL SELECT 1490, 'NAICS', 1836, 6116, 3, 1840, 61162, 4 FROM dual
UNION ALL SELECT 1491, 'SIC', 4311, 52, 1, 3873, 5714, 4 FROM dual
UNION ALL SELECT 1492, 'SIC', 4308, 20, 1, 3244, 2875, 4 FROM dual
UNION ALL SELECT 1493, 'NAICS', 180, 22, 1, 198, 2213, 3 FROM dual
UNION ALL SELECT 1494, 'NAICS', 118, 1151, 3, 124, 115115, 5 FROM dual
UNION ALL SELECT 1495, 'SIC', 3620, 4200, 2, 3622, 4212, 4 FROM dual
UNION ALL SELECT 1496, 'NAICS', 1569, 53, 1, 1579, 53119, 4 FROM dual
UNION ALL SELECT 1497, 'NAICS', 932, 4231, 3, 940, 42314, 4 FROM dual
UNION ALL SELECT 1498, 'SIC', 3419, 3500, 2, 3450, 3559, 4 FROM dual
UNION ALL SELECT 1499, 'NAICS', 1495, 5222, 3, 1499, 52222, 4 FROM dual
UNION ALL SELECT 1500, 'SIC', 3419, 3500, 2, 3463, 3572, 4 FROM dual
UNION ALL SELECT 1501, 'NAICS', 1480, 52, 1, 1502, 522292, 5 FROM dual
UNION ALL SELECT 1502, 'NAICS', 1516, 52311, 4, 1515, 523110, 5 FROM dual
UNION ALL SELECT 1503, 'NAICS', 2003, 72, 1, 2010, 72119, 4 FROM dual
UNION ALL SELECT 1504, 'NAICS', 930, 42, 1, 1060, 42459, 4 FROM dual
UNION ALL SELECT 1505, 'NAICS', 2037, 81, 1, 2092, 812332, 5 FROM dual
UNION ALL SELECT 1506, 'SEC', 2791, 20, 1, 2271, 2300, 2 FROM dual
UNION ALL SELECT 1507, 'SIC', 3961, 6320, 3, 3963, 6324, 4 FROM dual
UNION ALL SELECT 1508, 'NAICS', 1944, 7111, 3, 1946, 71111, 4 FROM dual
UNION ALL SELECT 1509, 'SIC', 4308, 20, 1, 2997, 2035, 4 FROM dual
UNION ALL SELECT 1510, 'SEC', 2791, 20, 1, 2408, 3530, 3 FROM dual
UNION ALL SELECT 1511, 'SIC', 3693, 4900, 2, 3713, 4970, 3 FROM dual
UNION ALL SELECT 1512, 'SEC', 2654, 6030, 3, 2655, 6035, 4 FROM dual
UNION ALL SELECT 1513, 'SIC', 2982, 2000, 2, 2996, 2034, 4 FROM dual
UNION ALL SELECT 1514, 'NAICS', 132, 21, 1, 162, 21232, 4 FROM dual
UNION ALL SELECT 1515, 'SIC', 4192, 8300, 2, 4197, 8350, 3 FROM dual
UNION ALL SELECT 1516, 'NAICS', 1, 11, 1, 26, 111320, 5 FROM dual
UNION ALL SELECT 1517, 'NAICS', 1518, 52312, 4, 1517, 523120, 5 FROM dual
UNION ALL SELECT 1518, 'SIC', 3480, 3600, 2, 3495, 3639, 4 FROM dual
UNION ALL SELECT 1519, 'SIC', 4017, 7200, 2, 4031, 7240, 3 FROM dual
UNION ALL SELECT 1520, 'SIC', 4204, 8410, 3, 4205, 8412, 4 FROM dual
UNION ALL SELECT 1521, 'SIC', 4305, 1, 1, 2798, 100, 2 FROM dual
UNION ALL SELECT 1522, 'NAICS', 2005, 7211, 3, 2011, 721191, 5 FROM dual
UNION ALL SELECT 1523, 'NAICS', 2150, 922, 2, 2151, 9221, 3 FROM dual
UNION ALL SELECT 1524, 'NAICS', 1015, 424, 2, 1045, 42445, 4 FROM dual
UNION ALL SELECT 1525, 'NAICS', 205, 23, 1, 239, 238120, 5 FROM dual
UNION ALL SELECT 1526, 'NAICS', 1, 11, 1, 2, 111, 2 FROM dual
UNION ALL SELECT 1527, 'NAICS', 1978, 71219, 4, 1977, 712190, 5 FROM dual
UNION ALL SELECT 1528, 'SIC', 4308, 20, 1, 3516, 3676, 4 FROM dual
UNION ALL SELECT 1529, 'NAICS', 2087, 81231, 4, 2086, 812310, 5 FROM dual
UNION ALL SELECT 1530, 'SIC', 4309, 40, 1, 3697, 4922, 4 FROM dual
UNION ALL SELECT 1531, 'SIC', 4308, 20, 1, 3383, 3433, 4 FROM dual
UNION ALL SELECT 1532, 'SIC', 3258, 2990, 3, 3260, 2999, 4 FROM dual
UNION ALL SELECT 1533, 'NAICS', 117, 115, 2, 124, 115115, 5 FROM dual
UNION ALL SELECT 1534, 'SEC', 2791, 20, 1, 2423, 3569, 4 FROM dual
UNION ALL SELECT 1535, 'NAICS', 1402, 51, 1, 1407, 511120, 5 FROM dual
UNION ALL SELECT 1536, 'SIC', 4308, 20, 1, 3500, 3645, 4 FROM dual
UNION ALL SELECT 1537, 'NAICS', 1624, 54, 1, 1653, 541370, 5 FROM dual
UNION ALL SELECT 1538, 'NAICS', 1850, 62, 1, 1878, 621492, 5 FROM dual
UNION ALL SELECT 1539, 'SEC', 2302, 2700, 2, 2316, 2771, 4 FROM dual
UNION ALL SELECT 1540, 'NAICS', 941, 4232, 3, 944, 423220, 5 FROM dual
UNION ALL SELECT 1541, 'SEC', 2568, 5000, 2, 2590, 5099, 4 FROM dual
UNION ALL SELECT 1542, 'SIC', 4313, 70, 1, 4083, 7513, 4 FROM dual
UNION ALL SELECT 1543, 'NAICS', 1418, 51121, 4, 1417, 511210, 5 FROM dual
UNION ALL SELECT 1544, 'NAICS', 2037, 81, 1, 2084, 81222, 4 FROM dual
UNION ALL SELECT 1545, 'SEC', 2791, 20, 1, 2291, 2590, 3 FROM dual
UNION ALL SELECT 1546, 'SIC', 4256, 9200, 2, 4258, 9211, 4 FROM dual
UNION ALL SELECT 1547, 'SIC', 4089, 7530, 3, 4090, 7532, 4 FROM dual
UNION ALL SELECT 1548, 'NAICS', 52, 11199, 4, 54, 111992, 5 FROM dual
UNION ALL SELECT 1549, 'NAICS', 117, 115, 2, 129, 1153, 3 FROM dual
UNION ALL SELECT 1550, 'SIC', 2966, 1750, 3, 2967, 1751, 4 FROM dual
UNION ALL SELECT 1551, 'SIC', 4313, 70, 1, 4191, 8299, 4 FROM dual
UNION ALL SELECT 1552, 'SIC', 3375, 3420, 3, 3377, 3423, 4 FROM dual
UNION ALL SELECT 1553, 'SIC', 4309, 40, 1, 3645, 4480, 3 FROM dual
UNION ALL SELECT 1554, 'SEC', 2527, 4510, 3, 2529, 4513, 4 FROM dual
UNION ALL SELECT 1555, 'SIC', 2914, 1380, 3, 2916, 1382, 4 FROM dual
UNION ALL SELECT 1556, 'SIC', 3083, 2300, 2, 3120, 2395, 4 FROM dual
UNION ALL SELECT 1557, 'NAICS', 1904, 623, 2, 1905, 6231, 3 FROM dual
UNION ALL SELECT 1558, 'SIC', 3180, 2670, 3, 3183, 2673, 4 FROM dual
UNION ALL SELECT 1559, 'SIC', 3869, 5700, 2, 3877, 5730, 3 FROM dual
UNION ALL SELECT 1560, 'SIC', 4139, 7990, 3, 4145, 7999, 4 FROM dual
UNION ALL SELECT 1561, 'NAICS', 1513, 523, 2, 1520, 52313, 4 FROM dual
UNION ALL SELECT 1562, 'NAICS', 173, 2131, 3, 176, 213112, 5 FROM dual
UNION ALL SELECT 1563, 'NAICS', 205, 23, 1, 266, 23833, 4 FROM dual
UNION ALL SELECT 1564, 'SIC', 4308, 20, 1, 3154, 2517, 4 FROM dual
UNION ALL SELECT 1565, 'SIC', 3854, 5600, 2, 3855, 5610, 3 FROM dual
UNION ALL SELECT 1566, 'SIC', 4100, 7600, 2, 4107, 7640, 3 FROM dual
UNION ALL SELECT 1567, 'SIC', 3190, 2700, 2, 3207, 2771, 4 FROM dual
UNION ALL SELECT 1568, 'SEC', 2791, 20, 1, 2445, 3652, 4 FROM dual
UNION ALL SELECT 1569, 'SEC', 2791, 20, 1, 2313, 2760, 3 FROM dual
UNION ALL SELECT 1570, 'SIC', 4314, 90, 1, 4273, 9440, 3 FROM dual
UNION ALL SELECT 1571, 'NAICS', 250, 23817, 4, 249, 238170, 5 FROM dual
UNION ALL SELECT 1572, 'NAICS', 1813, 611, 2, 1816, 61111, 4 FROM dual
UNION ALL SELECT 1573, 'SIC', 4314, 90, 1, 4289, 9630, 3 FROM dual
UNION ALL SELECT 1574, 'SIC', 3115, 2390, 3, 3122, 2397, 4 FROM dual
UNION ALL SELECT 1575, 'SIC', 3527, 3710, 3, 3528, 3711, 4 FROM dual
UNION ALL SELECT 1576, 'SIC', 4308, 20, 1, 3159, 2530, 3 FROM dual
UNION ALL SELECT 1577, 'SIC', 4314, 90, 1, 4303, 9990, 3 FROM dual
UNION ALL SELECT 1578, 'SIC', 2982, 2000, 2, 2990, 2023, 4 FROM dual
UNION ALL SELECT 1579, 'SIC', 3749, 5080, 3, 3752, 5084, 4 FROM dual
UNION ALL SELECT 1580, 'NAICS', 1725, 56, 1, 1760, 56151, 4 FROM dual
UNION ALL SELECT 1581, 'SEC', 2568, 5000, 2, 2570, 5013, 4 FROM dual
UNION ALL SELECT 1582, 'SIC', 3083, 2300, 2, 3085, 2311, 4 FROM dual
UNION ALL SELECT 1583, 'SIC', 3000, 2040, 3, 3003, 2044, 4 FROM dual
UNION ALL SELECT 1584, 'SIC', 3797, 5190, 3, 3799, 5192, 4 FROM dual
UNION ALL SELECT 1585, 'SEC', 2500, 3900, 2, 2502, 3911, 4 FROM dual
UNION ALL SELECT 1586, 'NAICS', 1569, 53, 1, 1593, 53211, 4 FROM dual
UNION ALL SELECT 1587, 'SEC', 2795, 60, 1, 2674, 6221, 4 FROM dual
UNION ALL SELECT 1588, 'SIC', 4007, 7000, 2, 4009, 7011, 4 FROM dual
UNION ALL SELECT 1589, 'SIC', 4313, 70, 1, 4059, 7352, 4 FROM dual
UNION ALL SELECT 1590, 'SEC', 2552, 4900, 2, 2563, 4941, 4 FROM dual
UNION ALL SELECT 1591, 'NAICS', 1942, 71, 1, 1986, 713210, 5 FROM dual
UNION ALL SELECT 1592, 'SIC', 4305, 1, 1, 2839, 259, 4 FROM dual
UNION ALL SELECT 1593, 'NAICS', 2153, 92211, 4, 2152, 922110, 5 FROM dual
UNION ALL SELECT 1594, 'NAICS', 3, 1111, 3, 9, 11113, 4 FROM dual
UNION ALL SELECT 1595, 'NAICS', 218, 237, 2, 227, 237210, 5 FROM dual
UNION ALL SELECT 1596, 'NAICS', 23, 1113, 3, 26, 111320, 5 FROM dual
UNION ALL SELECT 1597, 'NAICS', 1592, 5321, 3, 1593, 53211, 4 FROM dual
UNION ALL SELECT 1598, 'NAICS', 2135, 92, 1, 2167, 9231, 3 FROM dual
UNION ALL SELECT 1599, 'SEC', 2458, 3700, 2, 2473, 3751, 4 FROM dual
UNION ALL SELECT 1600, 'SEC', 2793, 50, 1, 2592, 5110, 3 FROM dual
UNION ALL SELECT 1601, 'NAICS', 3, 1111, 3, 6, 111120, 5 FROM dual
UNION ALL SELECT 1602, 'SIC', 4311, 52, 1, 3869, 5700, 2 FROM dual
UNION ALL SELECT 1603, 'SIC', 4308, 20, 1, 3484, 3620, 3 FROM dual
UNION ALL SELECT 1604, 'SIC', 3033, 2090, 3, 3035, 2092, 4 FROM dual
UNION ALL SELECT 1605, 'SIC', 2955, 1700, 2, 2956, 1710, 3 FROM dual
UNION ALL SELECT 1606, 'SIC', 4308, 20, 1, 3178, 2656, 4 FROM dual
UNION ALL SELECT 1607, 'SEC', 2319, 2800, 2, 2330, 2844, 4 FROM dual
UNION ALL SELECT 1608, 'SIC', 3131, 2430, 3, 3134, 2435, 4 FROM dual
UNION ALL SELECT 1609, 'SIC', 3148, 2500, 2, 3161, 2540, 3 FROM dual
UNION ALL SELECT 1610, 'SEC', 2795, 60, 1, 2688, 6361, 4 FROM dual
UNION ALL SELECT 1611, 'SIC', 4307, 15, 1, 2969, 1760, 3 FROM dual
UNION ALL SELECT 1612, 'SIC', 2982, 2000, 2, 3022, 2075, 4 FROM dual
UNION ALL SELECT 1613, 'NAICS', 1419, 512, 2, 1422, 51211, 4 FROM dual
UNION ALL SELECT 1614, 'SEC', 2465, 3720, 3, 2466, 3721, 4 FROM dual
UNION ALL SELECT 1615, 'SEC', 2753, 8000, 2, 2758, 8060, 3 FROM dual
UNION ALL SELECT 1616, 'SIC', 3384, 3440, 3, 3388, 3444, 4 FROM dual
UNION ALL SELECT 1617, 'NAICS', 52, 11199, 4, 53, 111991, 5 FROM dual
UNION ALL SELECT 1618, 'NAICS', 1015, 424, 2, 1069, 424720, 5 FROM dual
UNION ALL SELECT 1619, 'NAICS', 1568, 52599, 4, 1567, 525990, 5 FROM dual
UNION ALL SELECT 1620, 'NAICS', 2151, 9221, 3, 2165, 92219, 4 FROM dual
UNION ALL SELECT 1621, 'SEC', 2240, 2000, 2, 2256, 2086, 4 FROM dual
UNION ALL SELECT 1622, 'SIC', 3419, 3500, 2, 3434, 3540, 3 FROM dual
UNION ALL SELECT 1623, 'SIC', 3322, 3270, 3, 3323, 3271, 4 FROM dual
UNION ALL SELECT 1624, 'SIC', 2940, 1520, 3, 2941, 1521, 4 FROM dual
UNION ALL SELECT 1625, 'NAICS', 1942, 71, 1, 1982, 71311, 4 FROM dual
UNION ALL SELECT 1626, 'SIC', 4305, 1, 1, 2821, 181, 4 FROM dual
UNION ALL SELECT 1627, 'NAICS', 1431, 5122, 3, 1432, 512210, 5 FROM dual
UNION ALL SELECT 1628, 'NAICS', 181, 221, 2, 201, 221320, 5 FROM dual
UNION ALL SELECT 1629, 'NAICS', 1480, 52, 1, 1528, 52391, 4 FROM dual
UNION ALL SELECT 1630, 'SEC', 2637, 5800, 2, 2638, 5810, 3 FROM dual
UNION ALL SELECT 1631, 'SEC', 2403, 3500, 2, 2408, 3530, 3 FROM dual
UNION ALL SELECT 1632, 'SIC', 3526, 3700, 2, 3542, 3750, 3 FROM dual
UNION ALL SELECT 1633, 'SIC', 4158, 8050, 3, 4159, 8051, 4 FROM dual
UNION ALL SELECT 1634, 'SEC', 2791, 20, 1, 2299, 2650, 3 FROM dual
UNION ALL SELECT 1635, 'SIC', 3548, 3790, 3, 3550, 3795, 4 FROM dual
UNION ALL SELECT 1636, 'SIC', 3480, 3600, 2, 3509, 3663, 4 FROM dual
UNION ALL SELECT 1637, 'SEC', 2780, 8740, 3, 2782, 8742, 4 FROM dual
UNION ALL SELECT 1638, 'NAICS', 56, 112, 2, 69, 112310, 5 FROM dual
UNION ALL SELECT 1639, 'NAICS', 2003, 72, 1, 2029, 722410, 5 FROM dual
UNION ALL SELECT 1640, 'SIC', 2875, 900, 2, 2877, 912, 4 FROM dual
UNION ALL SELECT 1641, 'NAICS', 1843, 61169, 4, 1845, 611692, 5 FROM dual
UNION ALL SELECT 1642, 'SIC', 3552, 3800, 2, 3558, 3823, 4 FROM dual
UNION ALL SELECT 1643, 'SIC', 3762, 5100, 2, 3795, 5181, 4 FROM dual
UNION ALL SELECT 1644, 'SIC', 2868, 800, 2, 2871, 830, 3 FROM dual
UNION ALL SELECT 1645, 'NAICS', 1402, 51, 1, 1452, 517, 2 FROM dual
UNION ALL SELECT 1646, 'NAICS', 1480, 52, 1, 1539, 524113, 5 FROM dual
UNION ALL SELECT 1647, 'NAICS', 1741, 5614, 3, 1748, 561431, 5 FROM dual
UNION ALL SELECT 1648, 'SEC', 2605, 5200, 2, 2609, 5271, 4 FROM dual
UNION ALL SELECT 1649, 'NAICS', 2004, 721, 2, 2006, 721110, 5 FROM dual
UNION ALL SELECT 1650, 'SEC', 2795, 60, 1, 2684, 6331, 4 FROM dual
UNION ALL SELECT 1651, 'NAICS', 2135, 92, 1, 2202, 927110, 5 FROM dual
UNION ALL SELECT 1652, 'SEC', 2774, 8700, 2, 2777, 8730, 3 FROM dual
UNION ALL SELECT 1653, 'SIC', 3636, 4400, 2, 3646, 4481, 4 FROM dual
UNION ALL SELECT 1654, 'SIC', 4309, 40, 1, 3676, 4780, 3 FROM dual
UNION ALL SELECT 1655, 'SEC', 2791, 20, 1, 2344, 3011, 4 FROM dual
UNION ALL SELECT 1656, 'SIC', 4311, 52, 1, 3816, 5310, 3 FROM dual
UNION ALL SELECT 1657, 'NAICS', 1495, 5222, 3, 1498, 522220, 5 FROM dual
UNION ALL SELECT 1658, 'SIC', 4312, 60, 1, 3983, 6517, 4 FROM dual
UNION ALL SELECT 1659, 'NAICS', 1015, 424, 2, 1049, 42447, 4 FROM dual
UNION ALL SELECT 1660, 'NAICS', 2028, 7224, 3, 2030, 72241, 4 FROM dual
UNION ALL SELECT 1661, 'SIC', 4308, 20, 1, 3207, 2771, 4 FROM dual
UNION ALL SELECT 1662, 'NAICS', 1, 11, 1, 83, 11242, 4 FROM dual
UNION ALL SELECT 1663, 'SIC', 3419, 3500, 2, 3452, 3561, 4 FROM dual
UNION ALL SELECT 1664, 'SIC', 4311, 52, 1, 3863, 5650, 3 FROM dual
UNION ALL SELECT 1665, 'SEC', 2500, 3900, 2, 2508, 3949, 4 FROM dual
UNION ALL SELECT 1666, 'NAICS', 930, 42, 1, 1070, 42472, 4 FROM dual
UNION ALL SELECT 1667, 'SIC', 3083, 2300, 2, 3103, 2360, 3 FROM dual
UNION ALL SELECT 1668, 'SIC', 4309, 40, 1, 3628, 4222, 4 FROM dual
UNION ALL SELECT 1669, 'SIC', 3083, 2300, 2, 3117, 2392, 4 FROM dual
UNION ALL SELECT 1670, 'SIC', 4230, 8730, 3, 4234, 8734, 4 FROM dual
UNION ALL SELECT 1671, 'NAICS', 1968, 71151, 4, 1967, 711510, 5 FROM dual
UNION ALL SELECT 1672, 'SIC', 4310, 50, 1, 3723, 5023, 4 FROM dual
UNION ALL SELECT 1673, 'NAICS', 1706, 5419, 3, 1708, 54191, 4 FROM dual
UNION ALL SELECT 1674, 'SEC', 2791, 20, 1, 2466, 3721, 4 FROM dual
UNION ALL SELECT 1675, 'SIC', 3511, 3670, 3, 3518, 3678, 4 FROM dual
UNION ALL SELECT 1676, 'SIC', 4314, 90, 1, 4300, 9720, 3 FROM dual
UNION ALL SELECT 1677, 'SIC', 3124, 2400, 2, 3138, 2441, 4 FROM dual
UNION ALL SELECT 1678, 'SIC', 4311, 52, 1, 3882, 5800, 2 FROM dual
UNION ALL SELECT 1679, 'SIC', 4313, 70, 1, 4234, 8734, 4 FROM dual
UNION ALL SELECT 1680, 'NAICS', 2, 111, 2, 36, 1114, 3 FROM dual
UNION ALL SELECT 1681, 'NAICS', 205, 23, 1, 247, 238160, 5 FROM dual
UNION ALL SELECT 1682, 'SEC', 2371, 3300, 2, 2377, 3334, 4 FROM dual
UNION ALL SELECT 1683, 'SIC', 4313, 70, 1, 4162, 8060, 3 FROM dual
UNION ALL SELECT 1684, 'NAICS', 2136, 921, 2, 2138, 921110, 5 FROM dual
UNION ALL SELECT 1685, 'SEC', 2791, 20, 1, 2511, 3990, 3 FROM dual
UNION ALL SELECT 1686, 'SIC', 4268, 9400, 2, 4273, 9440, 3 FROM dual
UNION ALL SELECT 1687, 'SIC', 4313, 70, 1, 4156, 8043, 4 FROM dual
UNION ALL SELECT 1688, 'NAICS', 1726, 561, 2, 1734, 56131, 4 FROM dual
UNION ALL SELECT 1689, 'NAICS', 109, 1141, 3, 112, 114112, 5 FROM dual
UNION ALL SELECT 1690, 'SIC', 3282, 3100, 2, 3291, 3149, 4 FROM dual
UNION ALL SELECT 1691, 'SIC', 3715, 5000, 2, 3733, 5046, 4 FROM dual
UNION ALL SELECT 1692, 'SIC', 4223, 8700, 2, 4225, 8711, 4 FROM dual
UNION ALL SELECT 1693, 'NAICS', 1706, 5419, 3, 1715, 54194, 4 FROM dual
UNION ALL SELECT 1694, 'NAICS', 1942, 71, 1, 1951, 711190, 5 FROM dual
UNION ALL SELECT 1695, 'NAICS', 1626, 5411, 3, 1630, 54112, 4 FROM dual
UNION ALL SELECT 1696, 'NAICS', 1, 11, 1, 126, 1152, 3 FROM dual
UNION ALL SELECT 1697, 'NAICS', 1920, 624, 2, 1929, 624210, 5 FROM dual
UNION ALL SELECT 1698, 'NAICS', 2038, 811, 2, 2057, 811219, 5 FROM dual
UNION ALL SELECT 1699, 'SEC', 2791, 20, 1, 2411, 3533, 4 FROM dual
UNION ALL SELECT 1700, 'SIC', 4308, 20, 1, 3406, 3483, 4 FROM dual
UNION ALL SELECT 1701, 'SEC', 2237, 1700, 2, 2238, 1730, 3 FROM dual
UNION ALL SELECT 1702, 'SEC', 2349, 3080, 3, 2350, 3081, 4 FROM dual
UNION ALL SELECT 1703, 'SIC', 2825, 200, 2, 2836, 252, 4 FROM dual
UNION ALL SELECT 1704, 'SEC', 2791, 20, 1, 2504, 3931, 4 FROM dual
UNION ALL SELECT 1705, 'NAICS', 2189, 9261, 3, 2195, 92613, 4 FROM dual
UNION ALL SELECT 1706, 'SEC', 2794, 52, 1, 2631, 5710, 3 FROM dual
UNION ALL SELECT 1707, 'SIC', 4308, 20, 1, 3571, 3851, 4 FROM dual
UNION ALL SELECT 1708, 'SIC', 4018, 7210, 3, 4022, 7215, 4 FROM dual
UNION ALL SELECT 1709, 'NAICS', 931, 423, 2, 933, 423110, 5 FROM dual
UNION ALL SELECT 1710, 'NAICS', 132, 21, 1, 142, 212112, 5 FROM dual
UNION ALL SELECT 1711, 'NAICS', 1, 11, 1, 88, 112519, 5 FROM dual
UNION ALL SELECT 1712, 'NAICS', 1882, 62151, 4, 1884, 621512, 5 FROM dual
UNION ALL SELECT 1713, 'SIC', 3762, 5100, 2, 3786, 5154, 4 FROM dual
UNION ALL SELECT 1714, 'SIC', 3371, 3400, 2, 3391, 3449, 4 FROM dual
UNION ALL SELECT 1715, 'SIC', 3886, 5900, 2, 3890, 5921, 4 FROM dual
UNION ALL SELECT 1716, 'SEC', 2615, 5390, 3, 2616, 5399, 4 FROM dual
UNION ALL SELECT 1717, 'NAICS', 1942, 71, 1, 1948, 71112, 4 FROM dual
UNION ALL SELECT 1718, 'SIC', 4308, 20, 1, 3389, 3446, 4 FROM dual
UNION ALL SELECT 1719, 'NAICS', 2135, 92, 1, 2143, 92113, 4 FROM dual
UNION ALL SELECT 1720, 'SEC', 2791, 20, 1, 2279, 2430, 3 FROM dual
UNION ALL SELECT 1721, 'NAICS', 1942, 71, 1, 1995, 713930, 5 FROM dual
UNION ALL SELECT 1722, 'NAICS', 2071, 812, 2, 2089, 81232, 4 FROM dual
UNION ALL SELECT 1723, 'NAICS', 2037, 81, 1, 2071, 812, 2 FROM dual
UNION ALL SELECT 1724, 'SIC', 4313, 70, 1, 4245, 8990, 3 FROM dual
UNION ALL SELECT 1725, 'NAICS', 930, 42, 1, 1084, 42494, 4 FROM dual
UNION ALL SELECT 1726, 'NAICS', 2038, 811, 2, 2052, 8112, 3 FROM dual
UNION ALL SELECT 1727, 'NAICS', 150, 21223, 4, 152, 212234, 5 FROM dual
UNION ALL SELECT 1728, 'NAICS', 1726, 561, 2, 1747, 56143, 4 FROM dual
UNION ALL SELECT 1729, 'SEC', 2791, 20, 1, 2336, 2891, 4 FROM dual
UNION ALL SELECT 1730, 'SIC', 3252, 2900, 2, 3254, 2911, 4 FROM dual
UNION ALL SELECT 1731, 'NAICS', 2, 111, 2, 23, 1113, 3 FROM dual
UNION ALL SELECT 1732, 'SIC', 4312, 60, 1, 3993, 6710, 3 FROM dual
UNION ALL SELECT 1733, 'NAICS', 195, 2212, 3, 197, 22121, 4 FROM dual
UNION ALL SELECT 1734, 'SIC', 2875, 900, 2, 2879, 919, 4 FROM dual
UNION ALL SELECT 1735, 'SEC', 2523, 4400, 2, 2524, 4410, 3 FROM dual
UNION ALL SELECT 1736, 'SIC', 4312, 60, 1, 3943, 6153, 4 FROM dual
UNION ALL SELECT 1737, 'SIC', 3012, 2060, 3, 3014, 2062, 4 FROM dual
UNION ALL SELECT 1738, 'NAICS', 2209, 92812, 4, 2208, 928120, 5 FROM dual
UNION ALL SELECT 1739, 'SIC', 4309, 40, 1, 3626, 4220, 3 FROM dual
UNION ALL SELECT 1740, 'SIC', 4313, 70, 1, 4161, 8059, 4 FROM dual
UNION ALL SELECT 1741, 'NAICS', 930, 42, 1, 972, 42351, 4 FROM dual
UNION ALL SELECT 1742, 'SIC', 3174, 2650, 3, 3175, 2652, 4 FROM dual
UNION ALL SELECT 1743, 'SIC', 3886, 5900, 2, 3905, 5962, 4 FROM dual
UNION ALL SELECT 1744, 'SIC', 4313, 70, 1, 4220, 8661, 4 FROM dual
UNION ALL SELECT 1745, 'SIC', 4310, 50, 1, 3731, 5044, 4 FROM dual
UNION ALL SELECT 1746, 'SIC', 4310, 50, 1, 3761, 5099, 4 FROM dual
UNION ALL SELECT 1747, 'SIC', 4308, 20, 1, 3238, 2861, 4 FROM dual
UNION ALL SELECT 1748, 'SEC', 2791, 20, 1, 2459, 3710, 3 FROM dual
UNION ALL SELECT 1749, 'NAICS', 2062, 81141, 4, 2063, 811411, 5 FROM dual
UNION ALL SELECT 1750, 'NAICS', 1726, 561, 2, 1746, 561422, 5 FROM dual
UNION ALL SELECT 1751, 'SIC', 3419, 3500, 2, 3420, 3510, 3 FROM dual
UNION ALL SELECT 1752, 'NAICS', 1912, 62322, 4, 1911, 623220, 5 FROM dual
UNION ALL SELECT 1753, 'NAICS', 2071, 812, 2, 2100, 81293, 4 FROM dual
UNION ALL SELECT 1754, 'SEC', 2771, 8350, 3, 2772, 8351, 4 FROM dual
UNION ALL SELECT 1755, 'SIC', 2868, 800, 2, 2874, 851, 4 FROM dual
UNION ALL SELECT 1756, 'NAICS', 1470, 519, 2, 1475, 51912, 4 FROM dual
UNION ALL SELECT 1757, 'NAICS', 2135, 92, 1, 2141, 92112, 4 FROM dual
UNION ALL SELECT 1758, 'SIC', 3842, 5530, 3, 3843, 5531, 4 FROM dual
UNION ALL SELECT 1759, 'SIC', 4040, 7300, 2, 4074, 7379, 4 FROM dual
UNION ALL SELECT 1760, 'SEC', 2795, 60, 1, 4324, 6190, 3 FROM dual
UNION ALL SELECT 1761, 'SIC', 3033, 2090, 3, 3034, 2091, 4 FROM dual
UNION ALL SELECT 1762, 'SIC', 4312, 60, 1, 4004, 6794, 4 FROM dual
UNION ALL SELECT 1763, 'NAICS', 1456, 5172, 3, 1458, 51721, 4 FROM dual
UNION ALL SELECT 1764, 'SIC', 4308, 20, 1, 3498, 3643, 4 FROM dual
UNION ALL SELECT 1765, 'SEC', 2545, 4830, 3, 2546, 4832, 4 FROM dual
UNION ALL SELECT 1766, 'SIC', 4312, 60, 1, 3965, 6331, 4 FROM dual
UNION ALL SELECT 1767, 'SIC', 2798, 100, 2, 2799, 110, 3 FROM dual
UNION ALL SELECT 1768, 'SIC', 4213, 8630, 3, 4214, 8631, 4 FROM dual
UNION ALL SELECT 1769, 'NAICS', 1881, 6215, 3, 1884, 621512, 5 FROM dual
UNION ALL SELECT 1770, 'NAICS', 156, 2123, 3, 165, 212324, 5 FROM dual
UNION ALL SELECT 1771, 'SIC', 4308, 20, 1, 3000, 2040, 3 FROM dual
UNION ALL SELECT 1772, 'SIC', 4309, 40, 1, 3605, 4100, 2 FROM dual
UNION ALL SELECT 1773, 'SIC', 3715, 5000, 2, 3724, 5030, 3 FROM dual
UNION ALL SELECT 1774, 'NAICS', 1625, 541, 2, 1653, 541370, 5 FROM dual
UNION ALL SELECT 1775, 'NAICS', 2150, 922, 2, 2158, 922140, 5 FROM dual
UNION ALL SELECT 1776, 'NAICS', 2151, 9221, 3, 2157, 92213, 4 FROM dual
UNION ALL SELECT 1777, 'SIC', 2982, 2000, 2, 3020, 2070, 3 FROM dual
UNION ALL SELECT 1778, 'NAICS', 180, 22, 1, 200, 22131, 4 FROM dual
UNION ALL SELECT 1779, 'SEC', 2391, 3440, 3, 2395, 3448, 4 FROM dual
UNION ALL SELECT 1780, 'NAICS', 134, 2111, 3, 136, 211111, 5 FROM dual
UNION ALL SELECT 1781, 'NAICS', 181, 221, 2, 182, 2211, 3 FROM dual
UNION ALL SELECT 1782, 'NAICS', 2038, 811, 2, 2051, 811198, 5 FROM dual
UNION ALL SELECT 1783, 'SIC', 3304, 3220, 3, 3306, 3229, 4 FROM dual
UNION ALL SELECT 1784, 'NAICS', 2090, 81233, 4, 2091, 812331, 5 FROM dual
UNION ALL SELECT 1785, 'SIC', 3992, 6700, 2, 3998, 6726, 4 FROM dual
UNION ALL SELECT 1786, 'NAICS', 205, 23, 1, 222, 237120, 5 FROM dual
UNION ALL SELECT 1787, 'SEC', 2796, 70, 1, 2760, 8070, 3 FROM dual
UNION ALL SELECT 1788, 'NAICS', 2038, 811, 2, 2062, 81141, 4 FROM dual
UNION ALL SELECT 1789, 'NAICS', 1, 11, 1, 74, 11233, 4 FROM dual
UNION ALL SELECT 1790, 'NAICS', 1942, 71, 1, 1969, 712, 2 FROM dual
UNION ALL SELECT 1791, 'NAICS', 2093, 8129, 3, 2096, 81292, 4 FROM dual
UNION ALL SELECT 1792, 'SEC', 2319, 2800, 2, 2326, 2835, 4 FROM dual
UNION ALL SELECT 1793, 'SEC', 2680, 6320, 3, 2682, 6324, 4 FROM dual
UNION ALL SELECT 1794, 'NAICS', 40, 11142, 4, 42, 111422, 5 FROM dual
UNION ALL SELECT 1795, 'NAICS', 1620, 533, 2, 1621, 5331, 3 FROM dual
UNION ALL SELECT 1796, 'SIC', 3148, 2500, 2, 3163, 2542, 4 FROM dual
UNION ALL SELECT 1797, 'SEC', 2738, 7800, 2, 2743, 7822, 4 FROM dual
UNION ALL SELECT 1798, 'SIC', 4314, 90, 1, 4248, 9110, 3 FROM dual
UNION ALL SELECT 1799, 'NAICS', 1942, 71, 1, 1994, 71392, 4 FROM dual
UNION ALL SELECT 1800, 'SIC', 4311, 52, 1, 3825, 5420, 3 FROM dual
UNION ALL SELECT 1801, 'SIC', 4313, 70, 1, 4014, 7033, 4 FROM dual
UNION ALL SELECT 1802, 'NAICS', 1928, 6242, 3, 1934, 624230, 5 FROM dual
UNION ALL SELECT 1803, 'NAICS', 1480, 52, 1, 1534, 523991, 5 FROM dual
UNION ALL SELECT 1804, 'SEC', 2292, 2600, 2, 2299, 2650, 3 FROM dual
UNION ALL SELECT 1805, 'SEC', 2513, 4010, 3, 2514, 4011, 4 FROM dual
UNION ALL SELECT 1806, 'NAICS', 1990, 7139, 3, 1997, 713940, 5 FROM dual
UNION ALL SELECT 1807, 'SIC', 2945, 1540, 3, 2946, 1541, 4 FROM dual
UNION ALL SELECT 1808, 'SIC', 4308, 20, 1, 3279, 3087, 4 FROM dual
UNION ALL SELECT 1809, 'SIC', 4308, 20, 1, 3518, 3678, 4 FROM dual
UNION ALL SELECT 1810, 'SIC', 3616, 4150, 3, 3617, 4151, 4 FROM dual
UNION ALL SELECT 1811, 'SEC', 2269, 2270, 3, 2270, 2273, 4 FROM dual
UNION ALL SELECT 1812, 'SIC', 4308, 20, 1, 3245, 2879, 4 FROM dual
UNION ALL SELECT 1813, 'NAICS', 273, 2389, 3, 276, 238990, 5 FROM dual
UNION ALL SELECT 1814, 'NAICS', 37, 11141, 4, 39, 111419, 5 FROM dual
UNION ALL SELECT 1815, 'NAICS', 1485, 522, 2, 1487, 522110, 5 FROM dual
UNION ALL SELECT 1816, 'NAICS', 1621, 5331, 3, 1622, 533110, 5 FROM dual
UNION ALL SELECT 1817, 'SIC', 3059, 2250, 3, 3060, 2251, 4 FROM dual
UNION ALL SELECT 1818, 'SIC', 4308, 20, 1, 3197, 2732, 4 FROM dual
UNION ALL SELECT 1819, 'SIC', 4113, 7800, 2, 4123, 7840, 3 FROM dual
UNION ALL SELECT 1820, 'SIC', 4308, 20, 1, 3415, 3496, 4 FROM dual
UNION ALL SELECT 1821, 'SIC', 3419, 3500, 2, 3421, 3511, 4 FROM dual
UNION ALL SELECT 1822, 'SIC', 3261, 3000, 2, 3278, 3086, 4 FROM dual
UNION ALL SELECT 1823, 'NAICS', 1725, 56, 1, 1793, 5621, 3 FROM dual
UNION ALL SELECT 1824, 'SEC', 2792, 40, 1, 2559, 4930, 3 FROM dual
UNION ALL SELECT 1825, 'SIC', 4146, 8000, 2, 4171, 8090, 3 FROM dual
UNION ALL SELECT 1826, 'SEC', 2543, 4820, 3, 2544, 4822, 4 FROM dual
UNION ALL SELECT 1827, 'SIC', 4012, 7030, 3, 4013, 7032, 4 FROM dual
UNION ALL SELECT 1828, 'NAICS', 1969, 712, 2, 1973, 712120, 5 FROM dual
UNION ALL SELECT 1829, 'SIC', 3124, 2400, 2, 3136, 2439, 4 FROM dual
UNION ALL SELECT 1830, 'SIC', 4308, 20, 1, 3016, 2064, 4 FROM dual
UNION ALL SELECT 1831, 'SIC', 4308, 20, 1, 3526, 3700, 2 FROM dual
UNION ALL SELECT 1832, 'SIC', 4235, 8740, 3, 4237, 8742, 4 FROM dual
UNION ALL SELECT 1833, 'SIC', 4309, 40, 1, 3703, 4932, 4 FROM dual
UNION ALL SELECT 1834, 'NAICS', 1428, 51219, 4, 1430, 512199, 5 FROM dual
UNION ALL SELECT 1835, 'NAICS', 180, 22, 1, 187, 221114, 5 FROM dual
UNION ALL SELECT 1836, 'NAICS', 930, 42, 1, 968, 423490, 5 FROM dual
UNION ALL SELECT 1837, 'SIC', 4309, 40, 1, 3608, 4119, 4 FROM dual
UNION ALL SELECT 1838, 'NAICS', 58, 11211, 4, 60, 112112, 5 FROM dual
UNION ALL SELECT 1839, 'SEC', 2403, 3500, 2, 2411, 3533, 4 FROM dual
UNION ALL SELECT 1840, 'NAICS', 1554, 525, 2, 1559, 52512, 4 FROM dual
UNION ALL SELECT 1841, 'SEC', 2512, 4000, 2, 2514, 4011, 4 FROM dual
UNION ALL SELECT 1842, 'SIC', 3636, 4400, 2, 3639, 4420, 3 FROM dual
UNION ALL SELECT 1843, 'SIC', 4018, 7210, 3, 4026, 7219, 4 FROM dual
UNION ALL SELECT 1844, 'NAICS', 1626, 5411, 3, 1628, 54111, 4 FROM dual
UNION ALL SELECT 1845, 'NAICS', 2037, 81, 1, 2111, 813219, 5 FROM dual
UNION ALL SELECT 1846, 'NAICS', 172, 213, 2, 173, 2131, 3 FROM dual
UNION ALL SELECT 1847, 'SIC', 2825, 200, 2, 2846, 291, 4 FROM dual
UNION ALL SELECT 1848, 'SIC', 4308, 20, 1, 3497, 3641, 4 FROM dual
UNION ALL SELECT 1849, 'SIC', 3484, 3620, 3, 3486, 3624, 4 FROM dual
UNION ALL SELECT 1850, 'SEC', 2579, 5060, 3, 2580, 5063, 4 FROM dual
UNION ALL SELECT 1851, 'NAICS', 1402, 51, 1, 1440, 512290, 5 FROM dual
UNION ALL SELECT 1852, 'SIC', 4308, 20, 1, 3548, 3790, 3 FROM dual
UNION ALL SELECT 1853, 'NAICS', 205, 23, 1, 237, 238110, 5 FROM dual
UNION ALL SELECT 1854, 'SIC', 3337, 3300, 2, 3355, 3350, 3 FROM dual
UNION ALL SELECT 1855, 'SIC', 3667, 4700, 2, 3672, 4730, 3 FROM dual
UNION ALL SELECT 1856, 'NAICS', 1942, 71, 1, 1976, 71213, 4 FROM dual
UNION ALL SELECT 1857, 'NAICS', 2037, 81, 1, 2053, 81121, 4 FROM dual
UNION ALL SELECT 1858, 'SEC', 2548, 4840, 3, 2549, 4841, 4 FROM dual
UNION ALL SELECT 1859, 'NAICS', 2176, 924, 2, 2179, 92411, 4 FROM dual
UNION ALL SELECT 1860, 'SEC', 2791, 20, 1, 2246, 2024, 4 FROM dual
UNION ALL SELECT 1861, 'SIC', 4308, 20, 1, 3469, 3581, 4 FROM dual
UNION ALL SELECT 1862, 'NAICS', 965, 42345, 4, 964, 423450, 5 FROM dual
UNION ALL SELECT 1863, 'SEC', 2476, 3800, 2, 2484, 3825, 4 FROM dual
UNION ALL SELECT 1864, 'SIC', 3480, 3600, 2, 3494, 3635, 4 FROM dual
UNION ALL SELECT 1865, 'SIC', 3763, 5110, 3, 3766, 5113, 4 FROM dual
UNION ALL SELECT 1866, 'SIC', 4308, 20, 1, 3331, 3291, 4 FROM dual
UNION ALL SELECT 1867, 'NAICS', 1537, 5241, 3, 1545, 524130, 5 FROM dual
UNION ALL SELECT 1868, 'NAICS', 206, 236, 2, 215, 23621, 4 FROM dual
UNION ALL SELECT 1869, 'SIC', 3480, 3600, 2, 3512, 3671, 4 FROM dual
UNION ALL SELECT 1870, 'SEC', 2372, 3310, 3, 2374, 3317, 4 FROM dual
UNION ALL SELECT 1871, 'NAICS', 79, 1124, 3, 83, 11242, 4 FROM dual
UNION ALL SELECT 1872, 'SIC', 4308, 20, 1, 3543, 3751, 4 FROM dual
UNION ALL SELECT 1873, 'SIC', 4018, 7210, 3, 4024, 7217, 4 FROM dual
UNION ALL SELECT 1874, 'NAICS', 1015, 424, 2, 1068, 42471, 4 FROM dual
UNION ALL SELECT 1875, 'NAICS', 236, 2381, 3, 245, 238150, 5 FROM dual
UNION ALL SELECT 1876, 'SEC', 2725, 7370, 3, 2726, 7371, 4 FROM dual
UNION ALL SELECT 1877, 'SIC', 2955, 1700, 2, 2960, 1730, 3 FROM dual
UNION ALL SELECT 1878, 'SIC', 4310, 50, 1, 3770, 5131, 4 FROM dual
UNION ALL SELECT 1879, 'NAICS', 1015, 424, 2, 1027, 424310, 5 FROM dual
UNION ALL SELECT 1880, 'SIC', 2949, 1610, 3, 2950, 1611, 4 FROM dual
UNION ALL SELECT 1881, 'SIC', 4065, 7370, 3, 4072, 7377, 4 FROM dual
UNION ALL SELECT 1882, 'NAICS', 1, 11, 1, 84, 1125, 3 FROM dual
UNION ALL SELECT 1883, 'NAICS', 1554, 525, 2, 1563, 525910, 5 FROM dual
UNION ALL SELECT 1884, 'NAICS', 1942, 71, 1, 1950, 71113, 4 FROM dual
UNION ALL SELECT 1885, 'SEC', 2792, 40, 1, 2548, 4840, 3 FROM dual
UNION ALL SELECT 1886, 'SEC', 2793, 50, 1, 2588, 5090, 3 FROM dual
UNION ALL SELECT 1887, 'SIC', 4306, 10, 1, 2901, 1200, 2 FROM dual
UNION ALL SELECT 1888, 'SIC', 3124, 2400, 2, 3144, 2490, 3 FROM dual
UNION ALL SELECT 1889, 'SIC', 4040, 7300, 2, 4066, 7371, 4 FROM dual
UNION ALL SELECT 1890, 'SEC', 2795, 60, 1, 2668, 6162, 4 FROM dual
UNION ALL SELECT 1891, 'SIC', 3419, 3500, 2, 3466, 3578, 4 FROM dual
UNION ALL SELECT 1892, 'NAICS', 1420, 5121, 3, 1422, 51211, 4 FROM dual
UNION ALL SELECT 1893, 'SIC', 4312, 60, 1, 3923, 6022, 4 FROM dual
UNION ALL SELECT 1894, 'NAICS', 1741, 5614, 3, 1745, 561421, 5 FROM dual
UNION ALL SELECT 1895, 'NAICS', 1904, 623, 2, 1913, 6233, 3 FROM dual
UNION ALL SELECT 1896, 'SIC', 3301, 3200, 2, 3316, 3260, 3 FROM dual
UNION ALL SELECT 1897, 'NAICS', 1851, 621, 2, 1861, 62131, 4 FROM dual
UNION ALL SELECT 1898, 'SIC', 2805, 130, 3, 2808, 133, 4 FROM dual
UNION ALL SELECT 1899, 'NAICS', 1725, 56, 1, 1759, 561510, 5 FROM dual
UNION ALL SELECT 1900, 'SEC', 2288, 2530, 3, 2289, 2531, 4 FROM dual
UNION ALL SELECT 1901, 'NAICS', 930, 42, 1, 973, 423520, 5 FROM dual
UNION ALL SELECT 1902, 'SIC', 2955, 1700, 2, 2965, 1743, 4 FROM dual
UNION ALL SELECT 1903, 'SIC', 4125, 7900, 2, 4144, 7997, 4 FROM dual
UNION ALL SELECT 1904, 'NAICS', 1, 11, 1, 61, 112120, 5 FROM dual
UNION ALL SELECT 1905, 'NAICS', 1402, 51, 1, 1423, 512120, 5 FROM dual
UNION ALL SELECT 1906, 'SEC', 2792, 40, 1, 2545, 4830, 3 FROM dual
UNION ALL SELECT 1907, 'NAICS', 140, 21211, 4, 141, 212111, 5 FROM dual
UNION ALL SELECT 1908, 'SIC', 4306, 10, 1, 2914, 1380, 3 FROM dual
UNION ALL SELECT 1909, 'SIC', 3190, 2700, 2, 3197, 2732, 4 FROM dual
UNION ALL SELECT 1910, 'SIC', 4308, 20, 1, 3442, 3548, 4 FROM dual
UNION ALL SELECT 1911, 'NAICS', 182, 2211, 3, 192, 22112, 4 FROM dual
UNION ALL SELECT 1912, 'NAICS', 1626, 5411, 3, 1627, 541110, 5 FROM dual
UNION ALL SELECT 1913, 'SIC', 4105, 7630, 3, 4106, 7631, 4 FROM dual
UNION ALL SELECT 1914, 'NAICS', 2120, 8139, 3, 2127, 813940, 5 FROM dual
UNION ALL SELECT 1915, 'SEC', 2794, 52, 1, 2643, 5940, 3 FROM dual
UNION ALL SELECT 1916, 'SEC', 2418, 3560, 3, 2420, 3562, 4 FROM dual
UNION ALL SELECT 1917, 'SIC', 4235, 8740, 3, 4239, 8744, 4 FROM dual
UNION ALL SELECT 1918, 'SIC', 2876, 910, 3, 2879, 919, 4 FROM dual
UNION ALL SELECT 1919, 'NAICS', 931, 423, 2, 944, 423220, 5 FROM dual
UNION ALL SELECT 1920, 'SIC', 4308, 20, 1, 3506, 3652, 4 FROM dual
UNION ALL SELECT 1921, 'SIC', 4310, 50, 1, 3774, 5140, 3 FROM dual
UNION ALL SELECT 1922, 'SIC', 3992, 6700, 2, 3999, 6730, 3 FROM dual
UNION ALL SELECT 1923, 'SIC', 3989, 6550, 3, 3991, 6553, 4 FROM dual
UNION ALL SELECT 1924, 'SEC', 2479, 3820, 3, 2483, 3824, 4 FROM dual
UNION ALL SELECT 1925, 'NAICS', 1927, 62419, 4, 1926, 624190, 5 FROM dual
UNION ALL SELECT 1926, 'NAICS', 2071, 812, 2, 2077, 81219, 4 FROM dual
UNION ALL SELECT 1927, 'NAICS', 1480, 52, 1, 1550, 52429, 4 FROM dual
UNION ALL SELECT 1928, 'SEC', 2791, 20, 1, 2507, 3944, 4 FROM dual
UNION ALL SELECT 1929, 'SIC', 3762, 5100, 2, 3766, 5113, 4 FROM dual
UNION ALL SELECT 1930, 'SEC', 2795, 60, 1, 2650, 6020, 3 FROM dual
UNION ALL SELECT 1931, 'NAICS', 1015, 424, 2, 1066, 4247, 3 FROM dual
UNION ALL SELECT 1932, 'SEC', 2796, 70, 1, 2721, 7359, 4 FROM dual
UNION ALL SELECT 1933, 'SEC', 2748, 7900, 2, 2749, 7940, 3 FROM dual
UNION ALL SELECT 1934, 'NAICS', 1571, 5311, 3, 1576, 531130, 5 FROM dual
UNION ALL SELECT 1935, 'SIC', 3992, 6700, 2, 3994, 6712, 4 FROM dual
UNION ALL SELECT 1936, 'SIC', 3762, 5100, 2, 3802, 5198, 4 FROM dual
UNION ALL SELECT 1937, 'SIC', 4308, 20, 1, 3574, 3870, 3 FROM dual
UNION ALL SELECT 1938, 'NAICS', 1591, 532, 2, 1600, 53221, 4 FROM dual
UNION ALL SELECT 1939, 'NAICS', 1766, 5616, 3, 1772, 561621, 5 FROM dual
UNION ALL SELECT 1940, 'NAICS', 1774, 5617, 3, 1776, 56171, 4 FROM dual
UNION ALL SELECT 1941, 'SIC', 4307, 15, 1, 2966, 1750, 3 FROM dual
UNION ALL SELECT 1942, 'SIC', 4308, 20, 1, 3527, 3710, 3 FROM dual
UNION ALL SELECT 1943, 'NAICS', 1023, 4242, 3, 1024, 424210, 5 FROM dual
UNION ALL SELECT 1944, 'SIC', 4306, 10, 1, 2920, 1411, 4 FROM dual
UNION ALL SELECT 1945, 'NAICS', 2003, 72, 1, 2024, 722320, 5 FROM dual
UNION ALL SELECT 1946, 'SEC', 2355, 3200, 2, 2364, 3250, 3 FROM dual
UNION ALL SELECT 1947, 'SIC', 3552, 3800, 2, 3565, 3841, 4 FROM dual
UNION ALL SELECT 1948, 'NAICS', 1774, 5617, 3, 1777, 561720, 5 FROM dual
UNION ALL SELECT 1949, 'SIC', 2847, 700, 2, 2848, 710, 3 FROM dual
UNION ALL SELECT 1950, 'NAICS', 1076, 4249, 3, 1077, 424910, 5 FROM dual
UNION ALL SELECT 1951, 'SIC', 2955, 1700, 2, 2970, 1761, 4 FROM dual
UNION ALL SELECT 1952, 'SIC', 3705, 4940, 3, 3706, 4941, 4 FROM dual
UNION ALL SELECT 1953, 'SIC', 4307, 15, 1, 2959, 1721, 4 FROM dual
UNION ALL SELECT 1954, 'NAICS', 1402, 51, 1, 1470, 519, 2 FROM dual
UNION ALL SELECT 1955, 'NAICS', 1850, 62, 1, 1880, 621498, 5 FROM dual
UNION ALL SELECT 1956, 'SEC', 2796, 70, 1, 2711, 7011, 4 FROM dual
UNION ALL SELECT 1957, 'SIC', 2982, 2000, 2, 3002, 2043, 4 FROM dual
UNION ALL SELECT 1958, 'NAICS', 1931, 62422, 4, 1933, 624229, 5 FROM dual
UNION ALL SELECT 1959, 'NAICS', 1850, 62, 1, 1888, 6219, 3 FROM dual
UNION ALL SELECT 1960, 'NAICS', 2003, 72, 1, 2026, 722330, 5 FROM dual
UNION ALL SELECT 1961, 'SEC', 2476, 3800, 2, 2493, 3845, 4 FROM dual
UNION ALL SELECT 1962, 'SIC', 4309, 40, 1, 3690, 4841, 4 FROM dual
UNION ALL SELECT 1963, 'SIC', 4081, 7500, 2, 4090, 7532, 4 FROM dual
UNION ALL SELECT 1964, 'NAICS', 931, 423, 2, 934, 42311, 4 FROM dual
UNION ALL SELECT 1965, 'SIC', 4311, 52, 1, 3853, 5599, 4 FROM dual
UNION ALL SELECT 1966, 'NAICS', 1672, 5416, 3, 1677, 541614, 5 FROM dual
UNION ALL SELECT 1967, 'SEC', 2568, 5000, 2, 2578, 5051, 4 FROM dual
UNION ALL SELECT 1968, 'SIC', 3371, 3400, 2, 3383, 3433, 4 FROM dual
UNION ALL SELECT 1969, 'SIC', 3645, 4480, 3, 3646, 4481, 4 FROM dual
UNION ALL SELECT 1970, 'SIC', 3419, 3500, 2, 3448, 3555, 4 FROM dual
UNION ALL SELECT 1971, 'NAICS', 205, 23, 1, 220, 237110, 5 FROM dual
UNION ALL SELECT 1972, 'NAICS', 1624, 54, 1, 1669, 541512, 5 FROM dual
UNION ALL SELECT 1973, 'NAICS', 2020, 722, 2, 2021, 7223, 3 FROM dual
UNION ALL SELECT 1974, 'SIC', 4308, 20, 1, 3451, 3560, 3 FROM dual
UNION ALL SELECT 1975, 'SIC', 4125, 7900, 2, 4135, 7948, 4 FROM dual
UNION ALL SELECT 1976, 'SIC', 3724, 5030, 3, 3726, 5032, 4 FROM dual
UNION ALL SELECT 1977, 'NAICS', 2, 111, 2, 7, 11112, 4 FROM dual
UNION ALL SELECT 1978, 'NAICS', 1908, 6232, 3, 1909, 623210, 5 FROM dual
UNION ALL SELECT 1979, 'SIC', 3301, 3200, 2, 3329, 3281, 4 FROM dual
UNION ALL SELECT 1980, 'SIC', 4311, 52, 1, 3862, 5641, 4 FROM dual
UNION ALL SELECT 1981, 'NAICS', 182, 2211, 3, 190, 221117, 5 FROM dual
UNION ALL SELECT 1982, 'SEC', 2796, 70, 1, 2752, 7997, 4 FROM dual
UNION ALL SELECT 1983, 'SEC', 2539, 4800, 2, 2550, 4890, 3 FROM dual
UNION ALL SELECT 1984, 'SIC', 4244, 8900, 2, 4245, 8990, 3 FROM dual
UNION ALL SELECT 1985, 'NAICS', 1613, 53241, 4, 1614, 532411, 5 FROM dual
UNION ALL SELECT 1986, 'SIC', 4100, 7600, 2, 4112, 7699, 4 FROM dual
UNION ALL SELECT 1987, 'NAICS', 930, 42, 1, 1057, 424520, 5 FROM dual
UNION ALL SELECT 1988, 'SIC', 4040, 7300, 2, 4047, 7322, 4 FROM dual
UNION ALL SELECT 1989, 'NAICS', 930, 42, 1, 1053, 42449, 4 FROM dual
UNION ALL SELECT 1990, 'NAICS', 1402, 51, 1, 1438, 512240, 5 FROM dual
UNION ALL SELECT 1991, 'NAICS', 1706, 5419, 3, 1712, 541930, 5 FROM dual
UNION ALL SELECT 1992, 'SEC', 2791, 20, 1, 2401, 3480, 3 FROM dual
UNION ALL SELECT 1993, 'SIC', 4311, 52, 1, 3847, 5551, 4 FROM dual
UNION ALL SELECT 1994, 'NAICS', 2135, 92, 1, 2209, 92812, 4 FROM dual
UNION ALL SELECT 1995, 'SIC', 3000, 2040, 3, 3001, 2041, 4 FROM dual
UNION ALL SELECT 1996, 'SIC', 3848, 5560, 3, 3849, 5561, 4 FROM dual
UNION ALL SELECT 1997, 'SIC', 3576, 3900, 2, 3581, 3930, 3 FROM dual
UNION ALL SELECT 1998, 'NAICS', 181, 221, 2, 186, 221113, 5 FROM dual
UNION ALL SELECT 1999, 'SIC', 3762, 5100, 2, 3764, 5111, 4 FROM dual
UNION ALL SELECT 2000, 'SIC', 3989, 6550, 3, 3990, 6552, 4 FROM dual
UNION ALL SELECT 2001, 'NAICS', 235, 238, 2, 269, 238350, 5 FROM dual
UNION ALL SELECT 2002, 'NAICS', 1930, 62421, 4, 1929, 624210, 5 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 2003, 'SIC', 3050, 2200, 2, 3060, 2251, 4 FROM dual
UNION ALL SELECT 2004, 'SIC', 3167, 2600, 2, 3177, 2655, 4 FROM dual
UNION ALL SELECT 2005, 'SEC', 2792, 40, 1, 4318, 4991, 3 FROM dual
UNION ALL SELECT 2006, 'SEC', 2780, 8740, 3, 2781, 8741, 4 FROM dual
UNION ALL SELECT 2007, 'SEC', 2568, 5000, 2, 2569, 5010, 3 FROM dual
UNION ALL SELECT 2008, 'NAICS', 930, 42, 1, 1058, 42452, 4 FROM dual
UNION ALL SELECT 2009, 'SEC', 2793, 50, 1, 2594, 5122, 4 FROM dual
UNION ALL SELECT 2010, 'SIC', 4089, 7530, 3, 4091, 7533, 4 FROM dual
UNION ALL SELECT 2011, 'SEC', 2605, 5200, 2, 2607, 5211, 4 FROM dual
UNION ALL SELECT 2012, 'NAICS', 56, 112, 2, 72, 11232, 4 FROM dual
UNION ALL SELECT 2013, 'NAICS', 2037, 81, 1, 2097, 812921, 5 FROM dual
UNION ALL SELECT 2014, 'SIC', 4311, 52, 1, 3812, 5261, 4 FROM dual
UNION ALL SELECT 2015, 'SEC', 2791, 20, 1, 2375, 3320, 3 FROM dual
UNION ALL SELECT 2016, 'NAICS', 1725, 56, 1, 1784, 56179, 4 FROM dual
UNION ALL SELECT 2017, 'SIC', 3362, 3360, 3, 3367, 3369, 4 FROM dual
UNION ALL SELECT 2018, 'SIC', 3167, 2600, 2, 3169, 2611, 4 FROM dual
UNION ALL SELECT 2019, 'NAICS', 56, 112, 2, 89, 1129, 3 FROM dual
UNION ALL SELECT 2020, 'NAICS', 1640, 5413, 3, 1641, 541310, 5 FROM dual
UNION ALL SELECT 2021, 'SEC', 2476, 3800, 2, 2488, 3840, 3 FROM dual
UNION ALL SELECT 2022, 'SIC', 3261, 3000, 2, 3280, 3088, 4 FROM dual
UNION ALL SELECT 2023, 'NAICS', 1480, 52, 1, 1557, 52511, 4 FROM dual
UNION ALL SELECT 2024, 'SIC', 2850, 720, 3, 2853, 723, 4 FROM dual
UNION ALL SELECT 2025, 'SIC', 4308, 20, 1, 3584, 3942, 4 FROM dual
UNION ALL SELECT 2026, 'SIC', 4310, 50, 1, 3751, 5083, 4 FROM dual
UNION ALL SELECT 2027, 'NAICS', 931, 423, 2, 950, 42332, 4 FROM dual
UNION ALL SELECT 2028, 'NAICS', 1499, 52222, 4, 1498, 522220, 5 FROM dual
UNION ALL SELECT 2029, 'NAICS', 2, 111, 2, 42, 111422, 5 FROM dual
UNION ALL SELECT 2030, 'NAICS', 1513, 523, 2, 1532, 52393, 4 FROM dual
UNION ALL SELECT 2031, 'NAICS', 1843, 61169, 4, 1846, 611699, 5 FROM dual
UNION ALL SELECT 2032, 'SEC', 2552, 4900, 2, 2557, 4923, 4 FROM dual
UNION ALL SELECT 2033, 'SIC', 3451, 3560, 3, 3460, 3569, 4 FROM dual
UNION ALL SELECT 2034, 'SIC', 4313, 70, 1, 4130, 7929, 4 FROM dual
UNION ALL SELECT 2035, 'NAICS', 1689, 5418, 3, 1690, 541810, 5 FROM dual
UNION ALL SELECT 2036, 'SEC', 2791, 20, 1, 2302, 2700, 2 FROM dual
UNION ALL SELECT 2037, 'SIC', 3816, 5310, 3, 3817, 5311, 4 FROM dual
UNION ALL SELECT 2038, 'NAICS', 1942, 71, 1, 2002, 71399, 4 FROM dual
UNION ALL SELECT 2039, 'NAICS', 1, 11, 1, 60, 112112, 5 FROM dual
UNION ALL SELECT 2040, 'NAICS', 2037, 81, 1, 2123, 813920, 5 FROM dual
UNION ALL SELECT 2041, 'SIC', 4017, 7200, 2, 4022, 7215, 4 FROM dual
UNION ALL SELECT 2042, 'NAICS', 931, 423, 2, 960, 423430, 5 FROM dual
UNION ALL SELECT 2043, 'SIC', 4312, 60, 1, 3929, 6061, 4 FROM dual
UNION ALL SELECT 2044, 'NAICS', 129, 1153, 3, 130, 115310, 5 FROM dual
UNION ALL SELECT 2045, 'SIC', 3707, 4950, 3, 3709, 4953, 4 FROM dual
UNION ALL SELECT 2046, 'NAICS', 1920, 624, 2, 1930, 62421, 4 FROM dual
UNION ALL SELECT 2047, 'SIC', 3337, 3300, 2, 3359, 3355, 4 FROM dual
UNION ALL SELECT 2048, 'NAICS', 2167, 9231, 3, 2168, 923110, 5 FROM dual
UNION ALL SELECT 2049, 'NAICS', 1, 11, 1, 99, 1131, 3 FROM dual
UNION ALL SELECT 2050, 'SEC', 2302, 2700, 2, 2307, 2730, 3 FROM dual
UNION ALL SELECT 2051, 'SIC', 3921, 6020, 3, 3924, 6029, 4 FROM dual
UNION ALL SELECT 2052, 'SEC', 2791, 20, 1, 2481, 3822, 4 FROM dual
UNION ALL SELECT 2053, 'SIC', 4089, 7530, 3, 4095, 7538, 4 FROM dual
UNION ALL SELECT 2054, 'NAICS', 2, 111, 2, 43, 1119, 3 FROM dual
UNION ALL SELECT 2055, 'NAICS', 181, 221, 2, 193, 221121, 5 FROM dual
UNION ALL SELECT 2056, 'SIC', 3948, 6200, 2, 3951, 6220, 3 FROM dual
UNION ALL SELECT 2057, 'SIC', 4128, 7920, 3, 4129, 7922, 4 FROM dual
UNION ALL SELECT 2058, 'SIC', 4308, 20, 1, 3298, 3172, 4 FROM dual
UNION ALL SELECT 2059, 'SIC', 3337, 3300, 2, 3350, 3331, 4 FROM dual
UNION ALL SELECT 2060, 'SEC', 2760, 8070, 3, 2761, 8071, 4 FROM dual
UNION ALL SELECT 2061, 'SIC', 3214, 2800, 2, 3220, 2820, 3 FROM dual
UNION ALL SELECT 2062, 'SIC', 3763, 5110, 3, 3764, 5111, 4 FROM dual
UNION ALL SELECT 2063, 'SIC', 4310, 50, 1, 3747, 5075, 4 FROM dual
UNION ALL SELECT 2064, 'NAICS', 1850, 62, 1, 1937, 624310, 5 FROM dual
UNION ALL SELECT 2065, 'SIC', 2825, 200, 2, 2832, 240, 3 FROM dual
UNION ALL SELECT 2066, 'NAICS', 1403, 511, 2, 1412, 51114, 4 FROM dual
UNION ALL SELECT 2067, 'SEC', 2791, 20, 1, 2398, 3452, 4 FROM dual
UNION ALL SELECT 2068, 'SEC', 2792, 40, 1, 2561, 4932, 4 FROM dual
UNION ALL SELECT 2069, 'NAICS', 95, 11293, 4, 94, 112930, 5 FROM dual
UNION ALL SELECT 2070, 'SEC', 2319, 2800, 2, 2329, 2842, 4 FROM dual
UNION ALL SELECT 2071, 'NAICS', 43, 1119, 3, 55, 111998, 5 FROM dual
UNION ALL SELECT 2072, 'NAICS', 1943, 711, 2, 1944, 7111, 3 FROM dual
UNION ALL SELECT 2073, 'SIC', 2956, 1710, 3, 2957, 1711, 4 FROM dual
UNION ALL SELECT 2074, 'NAICS', 213, 2362, 3, 217, 23622, 4 FROM dual
UNION ALL SELECT 2075, 'SIC', 3576, 3900, 2, 3594, 3965, 4 FROM dual
UNION ALL SELECT 2076, 'NAICS', 1, 11, 1, 22, 111219, 5 FROM dual
UNION ALL SELECT 2077, 'NAICS', 1086, 42495, 4, 1085, 424950, 5 FROM dual
UNION ALL SELECT 2078, 'SIC', 4305, 1, 1, 2870, 811, 4 FROM dual
UNION ALL SELECT 2079, 'SIC', 4308, 20, 1, 3473, 3589, 4 FROM dual
UNION ALL SELECT 2080, 'NAICS', 173, 2131, 3, 177, 213113, 5 FROM dual
UNION ALL SELECT 2081, 'SEC', 2371, 3300, 2, 2383, 3390, 3 FROM dual
UNION ALL SELECT 2082, 'SIC', 4311, 52, 1, 3912, 5992, 4 FROM dual
UNION ALL SELECT 2083, 'NAICS', 119, 11511, 4, 122, 115113, 5 FROM dual
UNION ALL SELECT 2084, 'SEC', 2283, 2500, 2, 2284, 2510, 3 FROM dual
UNION ALL SELECT 2085, 'SIC', 3958, 6300, 2, 3966, 6350, 3 FROM dual
UNION ALL SELECT 2086, 'SIC', 2861, 760, 3, 2862, 761, 4 FROM dual
UNION ALL SELECT 2087, 'NAICS', 1733, 5613, 3, 1734, 56131, 4 FROM dual
UNION ALL SELECT 2088, 'NAICS', 2062, 81141, 4, 2064, 811412, 5 FROM dual
UNION ALL SELECT 2089, 'NAICS', 1514, 5231, 3, 1518, 52312, 4 FROM dual
UNION ALL SELECT 2090, 'NAICS', 1625, 541, 2, 1665, 54149, 4 FROM dual
UNION ALL SELECT 2091, 'NAICS', 1944, 7111, 3, 1947, 711120, 5 FROM dual
UNION ALL SELECT 2092, 'SEC', 2791, 20, 1, 2286, 2520, 3 FROM dual
UNION ALL SELECT 2093, 'SIC', 3282, 3100, 2, 3293, 3151, 4 FROM dual
UNION ALL SELECT 2094, 'SIC', 3827, 5430, 3, 3828, 5431, 4 FROM dual
UNION ALL SELECT 2095, 'NAICS', 144, 2122, 3, 153, 21229, 4 FROM dual
UNION ALL SELECT 2096, 'NAICS', 1513, 523, 2, 1531, 523930, 5 FROM dual
UNION ALL SELECT 2097, 'SEC', 2319, 2800, 2, 2323, 2830, 3 FROM dual
UNION ALL SELECT 2098, 'SIC', 4313, 70, 1, 4148, 8011, 4 FROM dual
UNION ALL SELECT 2099, 'SIC', 4113, 7800, 2, 4114, 7810, 3 FROM dual
UNION ALL SELECT 2100, 'SEC', 2313, 2760, 3, 2314, 2761, 4 FROM dual
UNION ALL SELECT 2101, 'SIC', 4314, 90, 1, 4297, 9700, 2 FROM dual
UNION ALL SELECT 2102, 'SIC', 3214, 2800, 2, 3236, 2851, 4 FROM dual
UNION ALL SELECT 2103, 'NAICS', 2166, 923, 2, 2174, 923140, 5 FROM dual
UNION ALL SELECT 2104, 'SEC', 2791, 20, 1, 2416, 3555, 4 FROM dual
UNION ALL SELECT 2105, 'SIC', 4308, 20, 1, 3568, 3844, 4 FROM dual
UNION ALL SELECT 2106, 'SIC', 4313, 70, 1, 4049, 7330, 3 FROM dual
UNION ALL SELECT 2107, 'SIC', 3715, 5000, 2, 3759, 5093, 4 FROM dual
UNION ALL SELECT 2108, 'SIC', 3886, 5900, 2, 3901, 5948, 4 FROM dual
UNION ALL SELECT 2109, 'SEC', 2262, 2200, 2, 2270, 2273, 4 FROM dual
UNION ALL SELECT 2110, 'NAICS', 1920, 624, 2, 1941, 62441, 4 FROM dual
UNION ALL SELECT 2111, 'NAICS', 2128, 81394, 4, 2127, 813940, 5 FROM dual
UNION ALL SELECT 2112, 'SEC', 2307, 2730, 3, 2309, 2732, 4 FROM dual
UNION ALL SELECT 2113, 'SEC', 2792, 40, 1, 2528, 4512, 4 FROM dual
UNION ALL SELECT 2114, 'NAICS', 1480, 52, 1, 1521, 523140, 5 FROM dual
UNION ALL SELECT 2115, 'SEC', 2355, 3200, 2, 2358, 3220, 3 FROM dual
UNION ALL SELECT 2116, 'SIC', 3893, 5940, 3, 3896, 5943, 4 FROM dual
UNION ALL SELECT 2117, 'SIC', 4313, 70, 1, 4217, 8650, 3 FROM dual
UNION ALL SELECT 2118, 'NAICS', 1850, 62, 1, 1884, 621512, 5 FROM dual
UNION ALL SELECT 2119, 'SIC', 3676, 4780, 3, 3678, 4785, 4 FROM dual
UNION ALL SELECT 2120, 'SEC', 2792, 40, 1, 2562, 4940, 3 FROM dual
UNION ALL SELECT 2121, 'SEC', 2677, 6300, 2, 2681, 6321, 4 FROM dual
UNION ALL SELECT 2122, 'SIC', 4311, 52, 1, 3915, 5995, 4 FROM dual
UNION ALL SELECT 2123, 'SIC', 4313, 70, 1, 4104, 7629, 4 FROM dual
UNION ALL SELECT 2124, 'NAICS', 1495, 5222, 3, 1503, 522293, 5 FROM dual
UNION ALL SELECT 2125, 'NAICS', 1836, 6116, 3, 1843, 61169, 4 FROM dual
UNION ALL SELECT 2126, 'SIC', 4017, 7200, 2, 4026, 7219, 4 FROM dual
UNION ALL SELECT 2127, 'SIC', 4313, 70, 1, 4147, 8010, 3 FROM dual
UNION ALL SELECT 2128, 'SEC', 2795, 60, 1, 2686, 6351, 4 FROM dual
UNION ALL SELECT 2129, 'SEC', 2795, 60, 1, 2692, 6410, 3 FROM dual
UNION ALL SELECT 2130, 'NAICS', 930, 42, 1, 948, 42331, 4 FROM dual
UNION ALL SELECT 2131, 'SIC', 3301, 3200, 2, 3324, 3272, 4 FROM dual
UNION ALL SELECT 2132, 'SIC', 4310, 50, 1, 3765, 5112, 4 FROM dual
UNION ALL SELECT 2133, 'NAICS', 1741, 5614, 3, 1755, 561491, 5 FROM dual
UNION ALL SELECT 2134, 'SEC', 2791, 20, 1, 2388, 3420, 3 FROM dual
UNION ALL SELECT 2135, 'NAICS', 1762, 56152, 4, 1761, 561520, 5 FROM dual
UNION ALL SELECT 2136, 'SIC', 3601, 4000, 2, 3603, 4011, 4 FROM dual
UNION ALL SELECT 2137, 'SIC', 4310, 50, 1, 3783, 5149, 4 FROM dual
UNION ALL SELECT 2138, 'SIC', 3426, 3530, 3, 3431, 3535, 4 FROM dual
UNION ALL SELECT 2139, 'NAICS', 2037, 81, 1, 2131, 814, 2 FROM dual
UNION ALL SELECT 2140, 'NAICS', 975, 4236, 3, 981, 42369, 4 FROM dual
UNION ALL SELECT 2141, 'NAICS', 1625, 541, 2, 1690, 541810, 5 FROM dual
UNION ALL SELECT 2142, 'NAICS', 1402, 51, 1, 1426, 512131, 5 FROM dual
UNION ALL SELECT 2143, 'SIC', 3552, 3800, 2, 3560, 3825, 4 FROM dual
UNION ALL SELECT 2144, 'SIC', 4040, 7300, 2, 4060, 7353, 4 FROM dual
UNION ALL SELECT 2145, 'NAICS', 132, 21, 1, 149, 212222, 5 FROM dual
UNION ALL SELECT 2146, 'NAICS', 930, 42, 1, 990, 42374, 4 FROM dual
UNION ALL SELECT 2147, 'SIC', 3767, 5120, 3, 3768, 5122, 4 FROM dual
UNION ALL SELECT 2148, 'NAICS', 1015, 424, 2, 1044, 424450, 5 FROM dual
UNION ALL SELECT 2149, 'SEC', 2789, 10, 1, 2222, 1310, 3 FROM dual
UNION ALL SELECT 2150, 'NAICS', 11, 11114, 4, 10, 111140, 5 FROM dual
UNION ALL SELECT 2151, 'SIC', 3537, 3730, 3, 3539, 3732, 4 FROM dual
UNION ALL SELECT 2152, 'SIC', 4311, 52, 1, 3840, 5520, 3 FROM dual
UNION ALL SELECT 2153, 'NAICS', 931, 423, 2, 967, 42346, 4 FROM dual
UNION ALL SELECT 2154, 'SIC', 3854, 5600, 2, 3862, 5641, 4 FROM dual
UNION ALL SELECT 2155, 'SIC', 2825, 200, 2, 2831, 219, 4 FROM dual
UNION ALL SELECT 2156, 'SEC', 2791, 20, 1, 2325, 2834, 4 FROM dual
UNION ALL SELECT 2157, 'SIC', 4184, 8230, 3, 4185, 8231, 4 FROM dual
UNION ALL SELECT 2158, 'NAICS', 1448, 51512, 4, 1447, 515120, 5 FROM dual
UNION ALL SELECT 2159, 'SEC', 2713, 7300, 2, 2723, 7361, 4 FROM dual
UNION ALL SELECT 2160, 'SIC', 3214, 2800, 2, 3235, 2850, 3 FROM dual
UNION ALL SELECT 2161, 'SEC', 2797, 90, 1, 2786, 9720, 3 FROM dual
UNION ALL SELECT 2162, 'SEC', 2618, 5410, 3, 2619, 5411, 4 FROM dual
UNION ALL SELECT 2163, 'SIC', 3214, 2800, 2, 3234, 2844, 4 FROM dual
UNION ALL SELECT 2164, 'SIC', 3214, 2800, 2, 3227, 2834, 4 FROM dual
UNION ALL SELECT 2165, 'NAICS', 1513, 523, 2, 1524, 523210, 5 FROM dual
UNION ALL SELECT 2166, 'SIC', 3349, 3330, 3, 3352, 3339, 4 FROM dual
UNION ALL SELECT 2167, 'SIC', 4017, 7200, 2, 4024, 7217, 4 FROM dual
UNION ALL SELECT 2168, 'SEC', 2476, 3800, 2, 2478, 3812, 4 FROM dual
UNION ALL SELECT 2169, 'NAICS', 2038, 811, 2, 2039, 8111, 3 FROM dual
UNION ALL SELECT 2170, 'SIC', 3552, 3800, 2, 3564, 3840, 3 FROM dual
UNION ALL SELECT 2171, 'SIC', 4309, 40, 1, 3633, 4300, 2 FROM dual
UNION ALL SELECT 2172, 'NAICS', 2135, 92, 1, 2170, 923120, 5 FROM dual
UNION ALL SELECT 2173, 'SIC', 3311, 3250, 3, 3313, 3253, 4 FROM dual
UNION ALL SELECT 2174, 'SIC', 3444, 3550, 3, 3447, 3554, 4 FROM dual
UNION ALL SELECT 2175, 'SIC', 4310, 50, 1, 3800, 5193, 4 FROM dual
UNION ALL SELECT 2176, 'NAICS', 2150, 922, 2, 2162, 922160, 5 FROM dual
UNION ALL SELECT 2177, 'NAICS', 1958, 7113, 3, 1960, 71131, 4 FROM dual
UNION ALL SELECT 2178, 'SIC', 4268, 9400, 2, 4271, 9430, 3 FROM dual
UNION ALL SELECT 2179, 'SIC', 3762, 5100, 2, 3780, 5146, 4 FROM dual
UNION ALL SELECT 2180, 'SIC', 4308, 20, 1, 3378, 3425, 4 FROM dual
UNION ALL SELECT 2181, 'NAICS', 138, 212, 2, 165, 212324, 5 FROM dual
UNION ALL SELECT 2182, 'SIC', 4049, 7330, 3, 4054, 7338, 4 FROM dual
UNION ALL SELECT 2183, 'NAICS', 1624, 54, 1, 1625, 541, 2 FROM dual
UNION ALL SELECT 2184, 'SIC', 2982, 2000, 2, 3005, 2046, 4 FROM dual
UNION ALL SELECT 2185, 'NAICS', 1969, 712, 2, 1971, 712110, 5 FROM dual
UNION ALL SELECT 2186, 'SIC', 3261, 3000, 2, 3275, 3083, 4 FROM dual
UNION ALL SELECT 2187, 'NAICS', 205, 23, 1, 246, 23815, 4 FROM dual
UNION ALL SELECT 2188, 'SIC', 4308, 20, 1, 3315, 3259, 4 FROM dual
UNION ALL SELECT 2189, 'SIC', 4313, 70, 1, 4159, 8051, 4 FROM dual
UNION ALL SELECT 2190, 'NAICS', 260, 2383, 3, 266, 23833, 4 FROM dual
UNION ALL SELECT 2191, 'NAICS', 1812, 61, 1, 1847, 6117, 3 FROM dual
UNION ALL SELECT 2192, 'SIC', 3322, 3270, 3, 3327, 3275, 4 FROM dual
UNION ALL SELECT 2193, 'SIC', 4305, 1, 1, 2877, 912, 4 FROM dual
UNION ALL SELECT 2194, 'SIC', 4309, 40, 1, 3611, 4130, 3 FROM dual
UNION ALL SELECT 2195, 'SIC', 3740, 5060, 3, 3743, 5065, 4 FROM dual
UNION ALL SELECT 2196, 'SIC', 4311, 52, 1, 3900, 5947, 4 FROM dual
UNION ALL SELECT 2197, 'NAICS', 931, 423, 2, 959, 42342, 4 FROM dual
UNION ALL SELECT 2198, 'NAICS', 1480, 52, 1, 1493, 522190, 5 FROM dual
UNION ALL SELECT 2199, 'NAICS', 1904, 623, 2, 1915, 623311, 5 FROM dual
UNION ALL SELECT 2200, 'SIC', 3050, 2200, 2, 3056, 2231, 4 FROM dual
UNION ALL SELECT 2201, 'NAICS', 1453, 5171, 3, 1455, 51711, 4 FROM dual
UNION ALL SELECT 2202, 'SIC', 3928, 6060, 3, 3930, 6062, 4 FROM dual
UNION ALL SELECT 2203, 'SIC', 4181, 8220, 3, 4183, 8222, 4 FROM dual
UNION ALL SELECT 2204, 'NAICS', 930, 42, 1, 1028, 42431, 4 FROM dual
UNION ALL SELECT 2205, 'SIC', 3304, 3220, 3, 3305, 3221, 4 FROM dual
UNION ALL SELECT 2206, 'NAICS', 1570, 531, 2, 1580, 5312, 3 FROM dual
UNION ALL SELECT 2207, 'NAICS', 1725, 56, 1, 1790, 561990, 5 FROM dual
UNION ALL SELECT 2208, 'NAICS', 1812, 61, 1, 1829, 61143, 4 FROM dual
UNION ALL SELECT 2209, 'NAICS', 931, 423, 2, 991, 4238, 3 FROM dual
UNION ALL SELECT 2210, 'NAICS', 1422, 51211, 4, 1421, 512110, 5 FROM dual
UNION ALL SELECT 2211, 'NAICS', 2179, 92411, 4, 2178, 924110, 5 FROM dual
UNION ALL SELECT 2212, 'SIC', 4309, 40, 1, 3682, 4812, 4 FROM dual
UNION ALL SELECT 2213, 'SIC', 3911, 5990, 3, 3913, 5993, 4 FROM dual
UNION ALL SELECT 2214, 'SIC', 4310, 50, 1, 3778, 5144, 4 FROM dual
UNION ALL SELECT 2215, 'NAICS', 1813, 611, 2, 1823, 6114, 3 FROM dual
UNION ALL SELECT 2216, 'NAICS', 1, 11, 1, 62, 11212, 4 FROM dual
UNION ALL SELECT 2217, 'NAICS', 5, 11111, 4, 4, 111110, 5 FROM dual
UNION ALL SELECT 2218, 'SIC', 4313, 70, 1, 4126, 7910, 3 FROM dual
UNION ALL SELECT 2219, 'NAICS', 1035, 4244, 3, 1038, 424420, 5 FROM dual
UNION ALL SELECT 2220, 'SEC', 2795, 60, 1, 2696, 6512, 4 FROM dual
UNION ALL SELECT 2221, 'NAICS', 132, 21, 1, 136, 211111, 5 FROM dual
UNION ALL SELECT 2222, 'SIC', 4313, 70, 1, 4236, 8741, 4 FROM dual
UNION ALL SELECT 2223, 'NAICS', 1813, 611, 2, 1837, 611610, 5 FROM dual
UNION ALL SELECT 2224, 'NAICS', 2004, 721, 2, 2005, 7211, 3 FROM dual
UNION ALL SELECT 2225, 'NAICS', 1402, 51, 1, 1415, 511199, 5 FROM dual
UNION ALL SELECT 2226, 'SIC', 4308, 20, 1, 3127, 2420, 3 FROM dual
UNION ALL SELECT 2227, 'SIC', 4311, 52, 1, 3910, 5989, 4 FROM dual
UNION ALL SELECT 2228, 'SIC', 2951, 1620, 3, 2954, 1629, 4 FROM dual
UNION ALL SELECT 2229, 'SIC', 3083, 2300, 2, 3092, 2329, 4 FROM dual
UNION ALL SELECT 2230, 'NAICS', 930, 42, 1, 1040, 424430, 5 FROM dual
UNION ALL SELECT 2231, 'NAICS', 2103, 813, 2, 2120, 8139, 3 FROM dual
UNION ALL SELECT 2232, 'SEC', 2342, 3000, 2, 2343, 3010, 3 FROM dual
UNION ALL SELECT 2233, 'SIC', 3083, 2300, 2, 3121, 2396, 4 FROM dual
UNION ALL SELECT 2234, 'SIC', 4309, 40, 1, 3668, 4720, 3 FROM dual
UNION ALL SELECT 2235, 'SEC', 2713, 7300, 2, 2732, 7381, 4 FROM dual
UNION ALL SELECT 2236, 'SIC', 3124, 2400, 2, 3135, 2436, 4 FROM dual
UNION ALL SELECT 2237, 'NAICS', 235, 238, 2, 267, 238340, 5 FROM dual
UNION ALL SELECT 2238, 'NAICS', 2037, 81, 1, 2095, 81291, 4 FROM dual
UNION ALL SELECT 2239, 'SIC', 4311, 52, 1, 3906, 5963, 4 FROM dual
UNION ALL SELECT 2240, 'SEC', 2512, 4000, 2, 2513, 4010, 3 FROM dual
UNION ALL SELECT 2241, 'SEC', 2694, 6500, 2, 2697, 6513, 4 FROM dual
UNION ALL SELECT 2242, 'NAICS', 1480, 52, 1, 1544, 524128, 5 FROM dual
UNION ALL SELECT 2243, 'NAICS', 2090, 81233, 4, 2092, 812332, 5 FROM dual
UNION ALL SELECT 2244, 'SIC', 4265, 9300, 2, 4267, 9311, 4 FROM dual
UNION ALL SELECT 2245, 'NAICS', 205, 23, 1, 212, 236118, 5 FROM dual
UNION ALL SELECT 2246, 'SIC', 3282, 3100, 2, 3292, 3150, 3 FROM dual
UNION ALL SELECT 2247, 'NAICS', 932, 4231, 3, 938, 42313, 4 FROM dual
UNION ALL SELECT 2248, 'NAICS', 1470, 519, 2, 1476, 519130, 5 FROM dual
UNION ALL SELECT 2249, 'NAICS', 1598, 5322, 3, 1605, 53229, 4 FROM dual
UNION ALL SELECT 2250, 'NAICS', 1624, 54, 1, 1695, 54183, 4 FROM dual
UNION ALL SELECT 2251, 'SEC', 2753, 8000, 2, 2765, 8093, 4 FROM dual
UNION ALL SELECT 2252, 'SEC', 2476, 3800, 2, 2477, 3810, 3 FROM dual
UNION ALL SELECT 2253, 'SIC', 4314, 90, 1, 4291, 9640, 3 FROM dual
UNION ALL SELECT 2254, 'SIC', 2975, 1790, 3, 2978, 1794, 4 FROM dual
UNION ALL SELECT 2255, 'SIC', 2975, 1790, 3, 2979, 1795, 4 FROM dual
UNION ALL SELECT 2256, 'SEC', 2342, 3000, 2, 2349, 3080, 3 FROM dual
UNION ALL SELECT 2257, 'SEC', 2795, 60, 1, 2687, 6360, 3 FROM dual
UNION ALL SELECT 2258, 'NAICS', 930, 42, 1, 1005, 423910, 5 FROM dual
UNION ALL SELECT 2259, 'SIC', 3041, 2100, 2, 3045, 2121, 4 FROM dual
UNION ALL SELECT 2260, 'SEC', 2791, 20, 1, 2245, 2020, 3 FROM dual
UNION ALL SELECT 2261, 'NAICS', 1536, 524, 2, 1540, 524114, 5 FROM dual
UNION ALL SELECT 2262, 'NAICS', 1402, 51, 1, 1432, 512210, 5 FROM dual
UNION ALL SELECT 2263, 'NAICS', 1871, 6214, 3, 1875, 62142, 4 FROM dual
UNION ALL SELECT 2264, 'NAICS', 1569, 53, 1, 1607, 532292, 5 FROM dual
UNION ALL SELECT 2265, 'NAICS', 2150, 922, 2, 2155, 92212, 4 FROM dual
UNION ALL SELECT 2266, 'SEC', 2403, 3500, 2, 2416, 3555, 4 FROM dual
UNION ALL SELECT 2267, 'SEC', 2568, 5000, 2, 2575, 5045, 4 FROM dual
UNION ALL SELECT 2268, 'SEC', 2796, 70, 1, 2725, 7370, 3 FROM dual
UNION ALL SELECT 2269, 'SIC', 3886, 5900, 2, 3914, 5994, 4 FROM dual
UNION ALL SELECT 2270, 'NAICS', 2, 111, 2, 50, 111940, 5 FROM dual
UNION ALL SELECT 2271, 'NAICS', 1920, 624, 2, 1924, 624120, 5 FROM dual
UNION ALL SELECT 2272, 'SIC', 4308, 20, 1, 3423, 3520, 3 FROM dual
UNION ALL SELECT 2273, 'SIC', 3461, 3570, 3, 3465, 3577, 4 FROM dual
UNION ALL SELECT 2274, 'SIC', 3636, 4400, 2, 3640, 4424, 4 FROM dual
UNION ALL SELECT 2275, 'NAICS', 930, 42, 1, 955, 4234, 3 FROM dual
UNION ALL SELECT 2276, 'NAICS', 1571, 5311, 3, 1579, 53119, 4 FROM dual
UNION ALL SELECT 2277, 'SIC', 3434, 3540, 3, 3439, 3545, 4 FROM dual
UNION ALL SELECT 2278, 'SIC', 4311, 52, 1, 3911, 5990, 3 FROM dual
UNION ALL SELECT 2279, 'SIC', 4058, 7350, 3, 4059, 7352, 4 FROM dual
UNION ALL SELECT 2280, 'NAICS', 117, 115, 2, 131, 11531, 4 FROM dual
UNION ALL SELECT 2281, 'NAICS', 1480, 52, 1, 1497, 52221, 4 FROM dual
UNION ALL SELECT 2282, 'SIC', 4309, 40, 1, 3637, 4410, 3 FROM dual
UNION ALL SELECT 2283, 'NAICS', 182, 2211, 3, 194, 221122, 5 FROM dual
UNION ALL SELECT 2284, 'NAICS', 930, 42, 1, 1066, 4247, 3 FROM dual
UNION ALL SELECT 2285, 'NAICS', 65, 1122, 3, 66, 112210, 5 FROM dual
UNION ALL SELECT 2286, 'NAICS', 1774, 5617, 3, 1781, 561740, 5 FROM dual
UNION ALL SELECT 2287, 'SEC', 2796, 70, 1, 2740, 7812, 4 FROM dual
UNION ALL SELECT 2288, 'SIC', 4306, 10, 1, 2917, 1389, 4 FROM dual
UNION ALL SELECT 2289, 'SIC', 4312, 60, 1, 3951, 6220, 3 FROM dual
UNION ALL SELECT 2290, 'NAICS', 1851, 621, 2, 1883, 621511, 5 FROM dual
UNION ALL SELECT 2291, 'NAICS', 138, 212, 2, 142, 212112, 5 FROM dual
UNION ALL SELECT 2292, 'SIC', 4308, 20, 1, 3047, 2131, 4 FROM dual
UNION ALL SELECT 2293, 'NAICS', 1495, 5222, 3, 1501, 522291, 5 FROM dual
UNION ALL SELECT 2294, 'SIC', 4308, 20, 1, 3291, 3149, 4 FROM dual
UNION ALL SELECT 2295, 'NAICS', 15, 11116, 4, 14, 111160, 5 FROM dual
UNION ALL SELECT 2296, 'NAICS', 1513, 523, 2, 1516, 52311, 4 FROM dual
UNION ALL SELECT 2297, 'NAICS', 56, 112, 2, 82, 112420, 5 FROM dual
UNION ALL SELECT 2298, 'NAICS', 1942, 71, 1, 1972, 71211, 4 FROM dual
UNION ALL SELECT 2299, 'SIC', 3576, 3900, 2, 3589, 3952, 4 FROM dual
UNION ALL SELECT 2300, 'NAICS', 943, 42321, 4, 942, 423210, 5 FROM dual
UNION ALL SELECT 2301, 'SIC', 4178, 8200, 2, 4184, 8230, 3 FROM dual
UNION ALL SELECT 2302, 'SEC', 2796, 70, 1, 2784, 8900, 2 FROM dual
UNION ALL SELECT 2303, 'SIC', 3190, 2700, 2, 3210, 2789, 4 FROM dual
UNION ALL SELECT 2304, 'NAICS', 2135, 92, 1, 2151, 9221, 3 FROM dual
UNION ALL SELECT 2305, 'NAICS', 1593, 53211, 4, 1594, 532111, 5 FROM dual
UNION ALL SELECT 2306, 'NAICS', 1402, 51, 1, 1434, 512220, 5 FROM dual
UNION ALL SELECT 2307, 'NAICS', 110, 11411, 4, 111, 114111, 5 FROM dual
UNION ALL SELECT 2308, 'SIC', 4308, 20, 1, 3599, 3996, 4 FROM dual
UNION ALL SELECT 2309, 'SIC', 3480, 3600, 2, 3487, 3625, 4 FROM dual
UNION ALL SELECT 2310, 'SIC', 4308, 20, 1, 3250, 2895, 4 FROM dual
UNION ALL SELECT 2311, 'SIC', 4313, 70, 1, 4238, 8743, 4 FROM dual
UNION ALL SELECT 2312, 'NAICS', 1562, 5259, 3, 1567, 525990, 5 FROM dual
UNION ALL SELECT 2313, 'NAICS', 1404, 5111, 3, 1411, 511140, 5 FROM dual
UNION ALL SELECT 2314, 'NAICS', 1733, 5613, 3, 1735, 561311, 5 FROM dual
UNION ALL SELECT 2315, 'NAICS', 1850, 62, 1, 1918, 623990, 5 FROM dual
UNION ALL SELECT 2316, 'SIC', 4313, 70, 1, 4088, 7521, 4 FROM dual
UNION ALL SELECT 2317, 'NAICS', 162, 21232, 4, 166, 212325, 5 FROM dual
UNION ALL SELECT 2318, 'SIC', 3200, 2750, 3, 3203, 2759, 4 FROM dual
UNION ALL SELECT 2319, 'NAICS', 1526, 5239, 3, 1529, 523920, 5 FROM dual
UNION ALL SELECT 2320, 'NAICS', 990, 42374, 4, 989, 423740, 5 FROM dual
UNION ALL SELECT 2321, 'SIC', 4308, 20, 1, 3549, 3792, 4 FROM dual
UNION ALL SELECT 2322, 'SIC', 3711, 4960, 3, 3712, 4961, 4 FROM dual
UNION ALL SELECT 2323, 'SIC', 4308, 20, 1, 3464, 3575, 4 FROM dual
UNION ALL SELECT 2324, 'SIC', 3762, 5100, 2, 3785, 5153, 4 FROM dual
UNION ALL SELECT 2325, 'SIC', 4311, 52, 1, 3823, 5410, 3 FROM dual
UNION ALL SELECT 2326, 'NAICS', 1624, 54, 1, 1643, 541320, 5 FROM dual
UNION ALL SELECT 2327, 'SEC', 2796, 70, 1, 2736, 7510, 3 FROM dual
UNION ALL SELECT 2328, 'SEC', 2403, 3500, 2, 2420, 3562, 4 FROM dual
UNION ALL SELECT 2329, 'NAICS', 930, 42, 1, 1027, 424310, 5 FROM dual
UNION ALL SELECT 2330, 'NAICS', 1726, 561, 2, 1772, 561621, 5 FROM dual
UNION ALL SELECT 2331, 'NAICS', 205, 23, 1, 275, 23891, 4 FROM dual
UNION ALL SELECT 2332, 'NAICS', 3, 1111, 3, 15, 11116, 4 FROM dual
UNION ALL SELECT 2333, 'SIC', 4017, 7200, 2, 4033, 7250, 3 FROM dual
UNION ALL SELECT 2334, 'NAICS', 1406, 51111, 4, 1405, 511110, 5 FROM dual
UNION ALL SELECT 2335, 'NAICS', 1089, 425, 2, 1090, 4251, 3 FROM dual
UNION ALL SELECT 2336, 'SIC', 4308, 20, 1, 3138, 2441, 4 FROM dual
UNION ALL SELECT 2337, 'SIC', 3214, 2800, 2, 3247, 2891, 4 FROM dual
UNION ALL SELECT 2338, 'NAICS', 260, 2383, 3, 264, 23832, 4 FROM dual
UNION ALL SELECT 2339, 'SIC', 3067, 2260, 3, 3068, 2261, 4 FROM dual
UNION ALL SELECT 2340, 'SIC', 4186, 8240, 3, 4188, 8244, 4 FROM dual
UNION ALL SELECT 2341, 'NAICS', 2135, 92, 1, 2189, 9261, 3 FROM dual
UNION ALL SELECT 2342, 'NAICS', 2004, 721, 2, 2017, 7213, 3 FROM dual
UNION ALL SELECT 2343, 'NAICS', 1812, 61, 1, 1836, 6116, 3 FROM dual
UNION ALL SELECT 2344, 'NAICS', 1480, 52, 1, 1562, 5259, 3 FROM dual
UNION ALL SELECT 2345, 'SIC', 2912, 1320, 3, 2913, 1321, 4 FROM dual
UNION ALL SELECT 2346, 'NAICS', 1550, 52429, 4, 1551, 524291, 5 FROM dual
UNION ALL SELECT 2347, 'SIC', 3419, 3500, 2, 3472, 3586, 4 FROM dual
UNION ALL SELECT 2348, 'SIC', 3680, 4800, 2, 3684, 4820, 3 FROM dual
UNION ALL SELECT 2349, 'NAICS', 930, 42, 1, 962, 423440, 5 FROM dual
UNION ALL SELECT 2350, 'NAICS', 931, 423, 2, 943, 42321, 4 FROM dual
UNION ALL SELECT 2351, 'NAICS', 1402, 51, 1, 1421, 512110, 5 FROM dual
UNION ALL SELECT 2352, 'NAICS', 1812, 61, 1, 1835, 611519, 5 FROM dual
UNION ALL SELECT 2353, 'SIC', 4313, 70, 1, 4197, 8350, 3 FROM dual
UNION ALL SELECT 2354, 'NAICS', 930, 42, 1, 953, 423390, 5 FROM dual
UNION ALL SELECT 2355, 'SIC', 4308, 20, 1, 3210, 2789, 4 FROM dual
UNION ALL SELECT 2356, 'SIC', 4308, 20, 1, 3520, 3690, 3 FROM dual
UNION ALL SELECT 2357, 'NAICS', 1979, 713, 2, 1982, 71311, 4 FROM dual
UNION ALL SELECT 2358, 'SEC', 2579, 5060, 3, 2581, 5064, 4 FROM dual
UNION ALL SELECT 2359, 'SIC', 4314, 90, 1, 4258, 9211, 4 FROM dual
UNION ALL SELECT 2360, 'NAICS', 1480, 52, 1, 1535, 523999, 5 FROM dual
UNION ALL SELECT 2361, 'NAICS', 1753, 56145, 4, 1752, 561450, 5 FROM dual
UNION ALL SELECT 2362, 'NAICS', 1725, 56, 1, 1758, 5615, 3 FROM dual
UNION ALL SELECT 2363, 'NAICS', 2003, 72, 1, 2032, 72251, 4 FROM dual
UNION ALL SELECT 2364, 'NAICS', 1402, 51, 1, 1442, 515, 2 FROM dual
UNION ALL SELECT 2365, 'NAICS', 2201, 9271, 3, 2203, 92711, 4 FROM dual
UNION ALL SELECT 2366, 'NAICS', 931, 423, 2, 982, 4237, 3 FROM dual
UNION ALL SELECT 2367, 'NAICS', 1624, 54, 1, 1675, 541612, 5 FROM dual
UNION ALL SELECT 2368, 'SEC', 2703, 6700, 2, 2704, 6790, 3 FROM dual
UNION ALL SELECT 2369, 'SIC', 4308, 20, 1, 3324, 3272, 4 FROM dual
UNION ALL SELECT 2370, 'SIC', 4313, 70, 1, 4223, 8700, 2 FROM dual
UNION ALL SELECT 2371, 'NAICS', 2037, 81, 1, 2110, 813212, 5 FROM dual
UNION ALL SELECT 2372, 'SIC', 3371, 3400, 2, 3376, 3421, 4 FROM dual
UNION ALL SELECT 2373, 'SIC', 4308, 20, 1, 3164, 2590, 3 FROM dual
UNION ALL SELECT 2374, 'NAICS', 930, 42, 1, 1068, 42471, 4 FROM dual
UNION ALL SELECT 2375, 'SIC', 3693, 4900, 2, 3695, 4911, 4 FROM dual
UNION ALL SELECT 2376, 'SIC', 3261, 3000, 2, 3263, 3011, 4 FROM dual
UNION ALL SELECT 2377, 'NAICS', 2135, 92, 1, 2198, 926150, 5 FROM dual
UNION ALL SELECT 2378, 'NAICS', 119, 11511, 4, 121, 115112, 5 FROM dual
UNION ALL SELECT 2379, 'NAICS', 1813, 611, 2, 1844, 611691, 5 FROM dual
UNION ALL SELECT 2380, 'SEC', 2391, 3440, 3, 2393, 3443, 4 FROM dual
UNION ALL SELECT 2381, 'NAICS', 1726, 561, 2, 1748, 561431, 5 FROM dual
UNION ALL SELECT 2382, 'NAICS', 240, 23812, 4, 239, 238120, 5 FROM dual
UNION ALL SELECT 2383, 'NAICS', 1979, 713, 2, 1986, 713210, 5 FROM dual
UNION ALL SELECT 2384, 'NAICS', 1625, 541, 2, 1704, 541890, 5 FROM dual
UNION ALL SELECT 2385, 'SEC', 2240, 2000, 2, 2249, 2040, 3 FROM dual
UNION ALL SELECT 2386, 'NAICS', 1871, 6214, 3, 1879, 621493, 5 FROM dual
UNION ALL SELECT 2387, 'SEC', 2795, 60, 1, 4323, 6189, 4 FROM dual
UNION ALL SELECT 2388, 'NAICS', 1851, 621, 2, 1891, 62199, 4 FROM dual
UNION ALL SELECT 2389, 'SIC', 2962, 1740, 3, 2965, 1743, 4 FROM dual
UNION ALL SELECT 2390, 'SIC', 3301, 3200, 2, 3315, 3259, 4 FROM dual
UNION ALL SELECT 2391, 'SIC', 4308, 20, 1, 3412, 3493, 4 FROM dual
UNION ALL SELECT 2392, 'SIC', 4313, 70, 1, 4203, 8400, 2 FROM dual
UNION ALL SELECT 2393, 'SEC', 2434, 3600, 2, 2455, 3679, 4 FROM dual
UNION ALL SELECT 2394, 'SIC', 4309, 40, 1, 3669, 4724, 4 FROM dual
UNION ALL SELECT 2395, 'SIC', 3149, 2510, 3, 3154, 2517, 4 FROM dual
UNION ALL SELECT 2396, 'NAICS', 1850, 62, 1, 1928, 6242, 3 FROM dual
UNION ALL SELECT 2397, 'SIC', 4313, 70, 1, 4052, 7335, 4 FROM dual
UNION ALL SELECT 2398, 'SIC', 3337, 3300, 2, 3341, 3315, 4 FROM dual
UNION ALL SELECT 2399, 'SIC', 4309, 40, 1, 3686, 4830, 3 FROM dual
UNION ALL SELECT 2400, 'NAICS', 2061, 8114, 3, 2062, 81141, 4 FROM dual
UNION ALL SELECT 2401, 'SIC', 3409, 3490, 3, 3410, 3491, 4 FROM dual
UNION ALL SELECT 2402, 'NAICS', 180, 22, 1, 204, 22133, 4 FROM dual
UNION ALL SELECT 2403, 'SEC', 2305, 2720, 3, 2306, 2721, 4 FROM dual
UNION ALL SELECT 2404, 'SEC', 2791, 20, 1, 2267, 2250, 3 FROM dual
UNION ALL SELECT 2405, 'SEC', 2791, 20, 1, 2468, 3728, 4 FROM dual
UNION ALL SELECT 2406, 'SIC', 3769, 5130, 3, 3772, 5137, 4 FROM dual
UNION ALL SELECT 2407, 'SEC', 2703, 6700, 2, 2705, 6792, 4 FROM dual
UNION ALL SELECT 2408, 'SIC', 4308, 20, 1, 3222, 2822, 4 FROM dual
UNION ALL SELECT 2409, 'NAICS', 1876, 62149, 4, 1877, 621491, 5 FROM dual
UNION ALL SELECT 2410, 'SIC', 3220, 2820, 3, 3222, 2822, 4 FROM dual
UNION ALL SELECT 2411, 'SIC', 3272, 3080, 3, 3275, 3083, 4 FROM dual
UNION ALL SELECT 2412, 'NAICS', 56, 112, 2, 91, 11291, 4 FROM dual
UNION ALL SELECT 2413, 'SIC', 3870, 5710, 3, 3874, 5719, 4 FROM dual
UNION ALL SELECT 2414, 'NAICS', 946, 4233, 3, 947, 423310, 5 FROM dual
UNION ALL SELECT 2415, 'SIC', 3822, 5400, 2, 3836, 5499, 4 FROM dual
UNION ALL SELECT 2416, 'NAICS', 1419, 512, 2, 1425, 51213, 4 FROM dual
UNION ALL SELECT 2417, 'NAICS', 1830, 6115, 3, 1835, 611519, 5 FROM dual
UNION ALL SELECT 2418, 'NAICS', 1794, 56211, 4, 1795, 562111, 5 FROM dual
UNION ALL SELECT 2419, 'SIC', 3740, 5060, 3, 3742, 5064, 4 FROM dual
UNION ALL SELECT 2420, 'NAICS', 2038, 811, 2, 2053, 81121, 4 FROM dual
UNION ALL SELECT 2421, 'SEC', 2650, 6020, 3, 2651, 6021, 4 FROM dual
UNION ALL SELECT 2422, 'SIC', 4308, 20, 1, 3424, 3523, 4 FROM dual
UNION ALL SELECT 2423, 'NAICS', 180, 22, 1, 195, 2212, 3 FROM dual
UNION ALL SELECT 2424, 'NAICS', 1624, 54, 1, 1685, 541711, 5 FROM dual
UNION ALL SELECT 2425, 'NAICS', 114, 1142, 3, 116, 11421, 4 FROM dual
UNION ALL SELECT 2426, 'NAICS', 1936, 6243, 3, 1938, 62431, 4 FROM dual
UNION ALL SELECT 2427, 'SIC', 3148, 2500, 2, 3150, 2511, 4 FROM dual
UNION ALL SELECT 2428, 'SIC', 4049, 7330, 3, 4051, 7334, 4 FROM dual
UNION ALL SELECT 2429, 'SIC', 4313, 70, 1, 4121, 7832, 4 FROM dual
UNION ALL SELECT 2430, 'NAICS', 1402, 51, 1, 1436, 512230, 5 FROM dual
UNION ALL SELECT 2431, 'SIC', 4308, 20, 1, 3559, 3824, 4 FROM dual
UNION ALL SELECT 2432, 'SEC', 2791, 20, 1, 2455, 3679, 4 FROM dual
UNION ALL SELECT 2433, 'NAICS', 1480, 52, 1, 1504, 522294, 5 FROM dual
UNION ALL SELECT 2434, 'SIC', 3041, 2100, 2, 3048, 2140, 3 FROM dual
UNION ALL SELECT 2435, 'NAICS', 1812, 61, 1, 1848, 611710, 5 FROM dual
UNION ALL SELECT 2436, 'SEC', 2505, 3940, 3, 2508, 3949, 4 FROM dual
UNION ALL SELECT 2437, 'SEC', 2385, 3410, 3, 2386, 3411, 4 FROM dual
UNION ALL SELECT 2438, 'SIC', 4313, 70, 1, 4045, 7319, 4 FROM dual
UNION ALL SELECT 2439, 'SEC', 4336, 99, 1, 4337, 8880, 2 FROM dual
UNION ALL SELECT 2440, 'SIC', 3375, 3420, 3, 3379, 3429, 4 FROM dual
UNION ALL SELECT 2441, 'SIC', 4314, 90, 1, 4292, 9641, 4 FROM dual
UNION ALL SELECT 2442, 'NAICS', 1774, 5617, 3, 1779, 561730, 5 FROM dual
UNION ALL SELECT 2443, 'SEC', 2591, 5100, 2, 2599, 5160, 3 FROM dual
UNION ALL SELECT 2444, 'SIC', 4284, 9600, 2, 4294, 9651, 4 FROM dual
UNION ALL SELECT 2445, 'NAICS', 1792, 562, 2, 1793, 5621, 3 FROM dual
UNION ALL SELECT 2446, 'SEC', 2792, 40, 1, 2514, 4011, 4 FROM dual
UNION ALL SELECT 2447, 'SIC', 4100, 7600, 2, 4105, 7630, 3 FROM dual
UNION ALL SELECT 2448, 'SEC', 2271, 2300, 2, 2275, 2390, 3 FROM dual
UNION ALL SELECT 2449, 'SIC', 2918, 1400, 2, 2925, 1440, 3 FROM dual
UNION ALL SELECT 2450, 'SIC', 4308, 20, 1, 3038, 2097, 4 FROM dual
UNION ALL SELECT 2451, 'SIC', 4223, 8700, 2, 4230, 8730, 3 FROM dual
UNION ALL SELECT 2452, 'NAICS', 955, 4234, 3, 958, 423420, 5 FROM dual
UNION ALL SELECT 2453, 'SEC', 2794, 52, 1, 2636, 5735, 4 FROM dual
UNION ALL SELECT 2454, 'SEC', 2791, 20, 1, 2420, 3562, 4 FROM dual
UNION ALL SELECT 2455, 'SIC', 4308, 20, 1, 3136, 2439, 4 FROM dual
UNION ALL SELECT 2456, 'SIC', 2918, 1400, 2, 2936, 1481, 4 FROM dual
UNION ALL SELECT 2457, 'NAICS', 1741, 5614, 3, 1756, 561492, 5 FROM dual
UNION ALL SELECT 2458, 'SIC', 4146, 8000, 2, 4153, 8040, 3 FROM dual
UNION ALL SELECT 2459, 'NAICS', 1913, 6233, 3, 1916, 623312, 5 FROM dual
UNION ALL SELECT 2460, 'SIC', 2982, 2000, 2, 3010, 2052, 4 FROM dual
UNION ALL SELECT 2461, 'NAICS', 1853, 62111, 4, 1855, 621112, 5 FROM dual
UNION ALL SELECT 2462, 'NAICS', 1730, 5612, 3, 1731, 561210, 5 FROM dual
UNION ALL SELECT 2463, 'SIC', 4308, 20, 1, 3009, 2051, 4 FROM dual
UNION ALL SELECT 2464, 'SIC', 3887, 5910, 3, 3888, 5912, 4 FROM dual
UNION ALL SELECT 2465, 'NAICS', 1850, 62, 1, 1921, 6241, 3 FROM dual
UNION ALL SELECT 2466, 'NAICS', 2037, 81, 1, 2051, 811198, 5 FROM dual
UNION ALL SELECT 2467, 'SEC', 2257, 2090, 3, 2258, 2092, 4 FROM dual
UNION ALL SELECT 2468, 'NAICS', 1471, 5191, 3, 1476, 519130, 5 FROM dual
UNION ALL SELECT 2469, 'SEC', 2796, 70, 1, 2728, 7373, 4 FROM dual
UNION ALL SELECT 2470, 'SIC', 2871, 830, 3, 2872, 831, 4 FROM dual
UNION ALL SELECT 2471, 'SEC', 2450, 3670, 3, 2454, 3678, 4 FROM dual
UNION ALL SELECT 2472, 'SIC', 3977, 6500, 2, 3986, 6531, 4 FROM dual
UNION ALL SELECT 2473, 'NAICS', 2037, 81, 1, 2062, 81141, 4 FROM dual
UNION ALL SELECT 2474, 'SIC', 4010, 7020, 3, 4011, 7021, 4 FROM dual
UNION ALL SELECT 2475, 'NAICS', 2005, 7211, 3, 2007, 72111, 4 FROM dual
UNION ALL SELECT 2476, 'NAICS', 2, 111, 2, 28, 11133, 4 FROM dual
UNION ALL SELECT 2477, 'NAICS', 1433, 51221, 4, 1432, 512210, 5 FROM dual
UNION ALL SELECT 2478, 'NAICS', 1612, 5324, 3, 1618, 532490, 5 FROM dual
UNION ALL SELECT 2479, 'SIC', 4178, 8200, 2, 4180, 8211, 4 FROM dual
UNION ALL SELECT 2480, 'NAICS', 49, 11193, 4, 48, 111930, 5 FROM dual
UNION ALL SELECT 2481, 'SIC', 3419, 3500, 2, 3422, 3519, 4 FROM dual
UNION ALL SELECT 2482, 'NAICS', 2037, 81, 1, 2057, 811219, 5 FROM dual
UNION ALL SELECT 2483, 'NAICS', 1689, 5418, 3, 1704, 541890, 5 FROM dual
UNION ALL SELECT 2484, 'NAICS', 180, 22, 1, 197, 22121, 4 FROM dual
UNION ALL SELECT 2485, 'SEC', 2774, 8700, 2, 2782, 8742, 4 FROM dual
UNION ALL SELECT 2486, 'SIC', 2873, 850, 3, 2874, 851, 4 FROM dual
UNION ALL SELECT 2487, 'SIC', 2884, 1000, 2, 2885, 1010, 3 FROM dual
UNION ALL SELECT 2488, 'SEC', 2241, 2010, 3, 2244, 2015, 4 FROM dual
UNION ALL SELECT 2489, 'SEC', 2667, 6160, 3, 2669, 6163, 4 FROM dual
UNION ALL SELECT 2490, 'SIC', 2825, 200, 2, 2839, 259, 4 FROM dual
UNION ALL SELECT 2491, 'SIC', 4308, 20, 1, 3557, 3822, 4 FROM dual
UNION ALL SELECT 2492, 'NAICS', 1725, 56, 1, 1782, 56174, 4 FROM dual
UNION ALL SELECT 2493, 'SIC', 4308, 20, 1, 3329, 3281, 4 FROM dual
UNION ALL SELECT 2494, 'SIC', 3371, 3400, 2, 3418, 3499, 4 FROM dual
UNION ALL SELECT 2495, 'SIC', 3261, 3000, 2, 3268, 3053, 4 FROM dual
UNION ALL SELECT 2496, 'SIC', 4199, 8360, 3, 4200, 8361, 4 FROM dual
UNION ALL SELECT 2497, 'NAICS', 1480, 52, 1, 1547, 5242, 3 FROM dual
UNION ALL SELECT 2498, 'NAICS', 2096, 81292, 4, 2097, 812921, 5 FROM dual
UNION ALL SELECT 2499, 'SIC', 3762, 5100, 2, 3799, 5192, 4 FROM dual
UNION ALL SELECT 2500, 'NAICS', 1452, 517, 2, 1454, 517110, 5 FROM dual
UNION ALL SELECT 2501, 'SEC', 2659, 6100, 2, 4340, 6172, 4 FROM dual
UNION ALL SELECT 2502, 'SIC', 2939, 1500, 2, 2940, 1520, 3 FROM dual
UNION ALL SELECT 2503, 'SEC', 2443, 3650, 3, 2445, 3652, 4 FROM dual
UNION ALL SELECT 2504, 'SIC', 3636, 4400, 2, 3638, 4412, 4 FROM dual
UNION ALL SELECT 2505, 'SIC', 4308, 20, 1, 3050, 2200, 2 FROM dual
UNION ALL SELECT 2506, 'SIC', 3148, 2500, 2, 3165, 2591, 4 FROM dual
UNION ALL SELECT 2507, 'SIC', 3237, 2860, 3, 3239, 2865, 4 FROM dual
UNION ALL SELECT 2508, 'NAICS', 1015, 424, 2, 1084, 42494, 4 FROM dual
UNION ALL SELECT 2509, 'NAICS', 2205, 9281, 3, 2206, 928110, 5 FROM dual
UNION ALL SELECT 2510, 'SIC', 2909, 1300, 2, 2915, 1381, 4 FROM dual
UNION ALL SELECT 2511, 'NAICS', 1402, 51, 1, 1464, 517911, 5 FROM dual
UNION ALL SELECT 2512, 'NAICS', 1725, 56, 1, 1754, 56149, 4 FROM dual
UNION ALL SELECT 2513, 'NAICS', 2037, 81, 1, 2052, 8112, 3 FROM dual
UNION ALL SELECT 2514, 'SIC', 3520, 3690, 3, 3523, 3694, 4 FROM dual
UNION ALL SELECT 2515, 'NAICS', 43, 1119, 3, 47, 11192, 4 FROM dual
UNION ALL SELECT 2516, 'NAICS', 1726, 561, 2, 1727, 5611, 3 FROM dual
UNION ALL SELECT 2517, 'SIC', 4278, 9510, 3, 4280, 9512, 4 FROM dual
UNION ALL SELECT 2518, 'SIC', 4311, 52, 1, 3808, 5231, 4 FROM dual
UNION ALL SELECT 2519, 'NAICS', 1850, 62, 1, 1857, 621210, 5 FROM dual
UNION ALL SELECT 2520, 'SEC', 2262, 2200, 2, 2266, 2221, 4 FROM dual
UNION ALL SELECT 2521, 'SIC', 2840, 270, 3, 2844, 279, 4 FROM dual
UNION ALL SELECT 2522, 'SIC', 3272, 3080, 3, 3280, 3088, 4 FROM dual
UNION ALL SELECT 2523, 'NAICS', 1591, 532, 2, 1611, 53231, 4 FROM dual
UNION ALL SELECT 2524, 'SIC', 4313, 70, 1, 4144, 7997, 4 FROM dual
UNION ALL SELECT 2525, 'SEC', 2300, 2670, 3, 2301, 2673, 4 FROM dual
UNION ALL SELECT 2526, 'SIC', 4308, 20, 1, 3437, 3543, 4 FROM dual
UNION ALL SELECT 2527, 'SEC', 2335, 2890, 3, 2336, 2891, 4 FROM dual
UNION ALL SELECT 2528, 'SIC', 4311, 52, 1, 3903, 5960, 3 FROM dual
UNION ALL SELECT 2529, 'SIC', 4310, 50, 1, 3754, 5087, 4 FROM dual
UNION ALL SELECT 2530, 'SIC', 4305, 1, 1, 2864, 780, 3 FROM dual
UNION ALL SELECT 2531, 'SIC', 4305, 1, 1, 2847, 700, 2 FROM dual
UNION ALL SELECT 2532, 'SIC', 4308, 20, 1, 3144, 2490, 3 FROM dual
UNION ALL SELECT 2533, 'NAICS', 235, 238, 2, 245, 238150, 5 FROM dual
UNION ALL SELECT 2534, 'NAICS', 2, 111, 2, 4, 111110, 5 FROM dual
UNION ALL SELECT 2535, 'NAICS', 232, 2379, 3, 233, 237990, 5 FROM dual
UNION ALL SELECT 2536, 'SIC', 4265, 9300, 2, 4266, 9310, 3 FROM dual
UNION ALL SELECT 2537, 'NAICS', 1733, 5613, 3, 1739, 561330, 5 FROM dual
UNION ALL SELECT 2538, 'SIC', 4278, 9510, 3, 4279, 9511, 4 FROM dual
UNION ALL SELECT 2539, 'NAICS', 1726, 561, 2, 1732, 56121, 4 FROM dual
UNION ALL SELECT 2540, 'NAICS', 2037, 81, 1, 2128, 81394, 4 FROM dual
UNION ALL SELECT 2541, 'SIC', 4308, 20, 1, 3372, 3410, 3 FROM dual
UNION ALL SELECT 2542, 'SIC', 3371, 3400, 2, 3388, 3444, 4 FROM dual
UNION ALL SELECT 2543, 'SEC', 2630, 5700, 2, 2635, 5734, 4 FROM dual
UNION ALL SELECT 2544, 'NAICS', 1718, 55, 1, 1723, 551112, 5 FROM dual
UNION ALL SELECT 2545, 'SIC', 3576, 3900, 2, 3586, 3949, 4 FROM dual
UNION ALL SELECT 2546, 'SIC', 4308, 20, 1, 3225, 2830, 3 FROM dual
UNION ALL SELECT 2547, 'SEC', 2593, 5120, 3, 2594, 5122, 4 FROM dual
UNION ALL SELECT 2548, 'SIC', 4308, 20, 1, 3456, 3565, 4 FROM dual
UNION ALL SELECT 2549, 'SIC', 3893, 5940, 3, 3902, 5949, 4 FROM dual
UNION ALL SELECT 2550, 'NAICS', 1016, 4241, 3, 1022, 42413, 4 FROM dual
UNION ALL SELECT 2551, 'SEC', 2479, 3820, 3, 2484, 3825, 4 FROM dual
UNION ALL SELECT 2552, 'SEC', 2794, 52, 1, 2637, 5800, 2 FROM dual
UNION ALL SELECT 2553, 'SIC', 4125, 7900, 2, 4130, 7929, 4 FROM dual
UNION ALL SELECT 2554, 'NAICS', 1480, 52, 1, 1511, 522390, 5 FROM dual
UNION ALL SELECT 2555, 'NAICS', 1851, 621, 2, 1859, 6213, 3 FROM dual
UNION ALL SELECT 2556, 'SIC', 2898, 1090, 3, 2899, 1094, 4 FROM dual
UNION ALL SELECT 2557, 'SIC', 3526, 3700, 2, 3533, 3720, 3 FROM dual
UNION ALL SELECT 2558, 'NAICS', 91, 11291, 4, 90, 112910, 5 FROM dual
UNION ALL SELECT 2559, 'NAICS', 1471, 5191, 3, 1475, 51912, 4 FROM dual
UNION ALL SELECT 2560, 'NAICS', 2004, 721, 2, 2007, 72111, 4 FROM dual
UNION ALL SELECT 2561, 'SEC', 2569, 5010, 3, 2570, 5013, 4 FROM dual
UNION ALL SELECT 2562, 'NAICS', 266, 23833, 4, 265, 238330, 5 FROM dual
UNION ALL SELECT 2563, 'NAICS', 1904, 623, 2, 1912, 62322, 4 FROM dual
UNION ALL SELECT 2564, 'SIC', 4308, 20, 1, 3316, 3260, 3 FROM dual
UNION ALL SELECT 2565, 'NAICS', 2137, 9211, 3, 2140, 921120, 5 FROM dual
UNION ALL SELECT 2566, 'SEC', 2476, 3800, 2, 2489, 3841, 4 FROM dual
UNION ALL SELECT 2567, 'SIC', 3583, 3940, 3, 3585, 3944, 4 FROM dual
UNION ALL SELECT 2568, 'SIC', 3667, 4700, 2, 3676, 4780, 3 FROM dual
UNION ALL SELECT 2569, 'SIC', 3086, 2320, 3, 3091, 2326, 4 FROM dual
UNION ALL SELECT 2570, 'SIC', 4271, 9430, 3, 4272, 9431, 4 FROM dual
UNION ALL SELECT 2571, 'SIC', 2982, 2000, 2, 3021, 2074, 4 FROM dual
UNION ALL SELECT 2572, 'NAICS', 2135, 92, 1, 2142, 921130, 5 FROM dual
UNION ALL SELECT 2573, 'SIC', 4308, 20, 1, 3008, 2050, 3 FROM dual
UNION ALL SELECT 2574, 'SIC', 3255, 2950, 3, 3257, 2952, 4 FROM dual
UNION ALL SELECT 2575, 'SIC', 3886, 5900, 2, 3899, 5946, 4 FROM dual
UNION ALL SELECT 2576, 'NAICS', 89, 1129, 3, 91, 11291, 4 FROM dual
UNION ALL SELECT 2577, 'SEC', 2791, 20, 1, 2427, 3575, 4 FROM dual
UNION ALL SELECT 2578, 'NAICS', 1015, 424, 2, 1028, 42431, 4 FROM dual
UNION ALL SELECT 2579, 'SIC', 3854, 5600, 2, 3859, 5630, 3 FROM dual
UNION ALL SELECT 2580, 'NAICS', 1706, 5419, 3, 1707, 541910, 5 FROM dual
UNION ALL SELECT 2581, 'SIC', 4284, 9600, 2, 4285, 9610, 3 FROM dual
UNION ALL SELECT 2582, 'SIC', 4308, 20, 1, 3185, 2675, 4 FROM dual
UNION ALL SELECT 2583, 'NAICS', 1850, 62, 1, 1909, 623210, 5 FROM dual
UNION ALL SELECT 2584, 'SIC', 4308, 20, 1, 3228, 2835, 4 FROM dual
UNION ALL SELECT 2585, 'NAICS', 930, 42, 1, 986, 42372, 4 FROM dual
UNION ALL SELECT 2586, 'NAICS', 1015, 424, 2, 1040, 424430, 5 FROM dual
UNION ALL SELECT 2587, 'SIC', 4308, 20, 1, 3219, 2819, 4 FROM dual
UNION ALL SELECT 2588, 'NAICS', 1513, 523, 2, 1519, 523130, 5 FROM dual
UNION ALL SELECT 2589, 'SIC', 2931, 1470, 3, 2934, 1479, 4 FROM dual
UNION ALL SELECT 2590, 'SEC', 2568, 5000, 2, 2586, 5082, 4 FROM dual
UNION ALL SELECT 2591, 'NAICS', 1852, 6211, 3, 1853, 62111, 4 FROM dual
UNION ALL SELECT 2592, 'SIC', 2918, 1400, 2, 2928, 1450, 3 FROM dual
UNION ALL SELECT 2593, 'SEC', 2319, 2800, 2, 2328, 2840, 3 FROM dual
UNION ALL SELECT 2594, 'SIC', 4313, 70, 1, 4135, 7948, 4 FROM dual
UNION ALL SELECT 2595, 'SIC', 4305, 1, 1, 2858, 750, 3 FROM dual
UNION ALL SELECT 2596, 'SIC', 4310, 50, 1, 3789, 5162, 4 FROM dual
UNION ALL SELECT 2597, 'SIC', 4308, 20, 1, 3505, 3651, 4 FROM dual
UNION ALL SELECT 2598, 'NAICS', 1016, 4241, 3, 1018, 42411, 4 FROM dual
UNION ALL SELECT 2599, 'NAICS', 1871, 6214, 3, 1876, 62149, 4 FROM dual
UNION ALL SELECT 2600, 'SIC', 2982, 2000, 2, 3032, 2087, 4 FROM dual
UNION ALL SELECT 2601, 'SIC', 3838, 5510, 3, 3839, 5511, 4 FROM dual
UNION ALL SELECT 2602, 'NAICS', 1813, 611, 2, 1827, 61142, 4 FROM dual
UNION ALL SELECT 2603, 'SIC', 3167, 2600, 2, 3186, 2676, 4 FROM dual
UNION ALL SELECT 2604, 'SEC', 2791, 20, 1, 2437, 3613, 4 FROM dual
UNION ALL SELECT 2605, 'NAICS', 205, 23, 1, 259, 23829, 4 FROM dual
UNION ALL SELECT 2606, 'SIC', 4306, 10, 1, 2890, 1031, 4 FROM dual
UNION ALL SELECT 2607, 'SIC', 4040, 7300, 2, 4072, 7377, 4 FROM dual
UNION ALL SELECT 2608, 'NAICS', 1758, 5615, 3, 1760, 56151, 4 FROM dual
UNION ALL SELECT 2609, 'SEC', 2792, 40, 1, 2563, 4941, 4 FROM dual
UNION ALL SELECT 2610, 'SEC', 2643, 5940, 3, 2644, 5944, 4 FROM dual
UNION ALL SELECT 2611, 'NAICS', 1726, 561, 2, 1735, 561311, 5 FROM dual
UNION ALL SELECT 2612, 'NAICS', 930, 42, 1, 1044, 424450, 5 FROM dual
UNION ALL SELECT 2613, 'NAICS', 1456, 5172, 3, 1457, 517210, 5 FROM dual
UNION ALL SELECT 2614, 'SIC', 3282, 3100, 2, 3298, 3172, 4 FROM dual
UNION ALL SELECT 2615, 'SIC', 3272, 3080, 3, 3278, 3086, 4 FROM dual
UNION ALL SELECT 2616, 'SIC', 3241, 2870, 3, 3244, 2875, 4 FROM dual
UNION ALL SELECT 2617, 'SIC', 4065, 7370, 3, 4066, 7371, 4 FROM dual
UNION ALL SELECT 2618, 'SEC', 2793, 50, 1, 2576, 5047, 4 FROM dual
UNION ALL SELECT 2619, 'SIC', 4314, 90, 1, 4256, 9200, 2 FROM dual
UNION ALL SELECT 2620, 'SIC', 3419, 3500, 2, 3428, 3532, 4 FROM dual
UNION ALL SELECT 2621, 'NAICS', 1970, 7121, 3, 1978, 71219, 4 FROM dual
UNION ALL SELECT 2622, 'NAICS', 181, 221, 2, 184, 221111, 5 FROM dual
UNION ALL SELECT 2623, 'NAICS', 1480, 52, 1, 1526, 5239, 3 FROM dual
UNION ALL SELECT 2624, 'SIC', 4027, 7220, 3, 4028, 7221, 4 FROM dual
UNION ALL SELECT 2625, 'SEC', 2476, 3800, 2, 2483, 3824, 4 FROM dual
UNION ALL SELECT 2626, 'SIC', 4312, 60, 1, 3972, 6390, 3 FROM dual
UNION ALL SELECT 2627, 'SIC', 3663, 4610, 3, 3666, 4619, 4 FROM dual
UNION ALL SELECT 2628, 'SIC', 4139, 7990, 3, 4144, 7997, 4 FROM dual
UNION ALL SELECT 2629, 'SIC', 4146, 8000, 2, 4158, 8050, 3 FROM dual
UNION ALL SELECT 2630, 'NAICS', 1812, 61, 1, 1830, 6115, 3 FROM dual
UNION ALL SELECT 2631, 'NAICS', 2135, 92, 1, 2166, 923, 2 FROM dual
UNION ALL SELECT 2632, 'SEC', 2791, 20, 1, 2337, 2900, 2 FROM dual
UNION ALL SELECT 2633, 'SIC', 4277, 9500, 2, 4282, 9531, 4 FROM dual
UNION ALL SELECT 2634, 'SIC', 4146, 8000, 2, 4160, 8052, 4 FROM dual
UNION ALL SELECT 2635, 'SEC', 2283, 2500, 2, 2290, 2540, 3 FROM dual
UNION ALL SELECT 2636, 'NAICS', 56, 112, 2, 78, 11239, 4 FROM dual
UNION ALL SELECT 2637, 'NAICS', 114, 1142, 3, 115, 114210, 5 FROM dual
UNION ALL SELECT 2638, 'NAICS', 1725, 56, 1, 1736, 561312, 5 FROM dual
UNION ALL SELECT 2639, 'SIC', 3942, 6150, 3, 3943, 6153, 4 FROM dual
UNION ALL SELECT 2640, 'NAICS', 2188, 926, 2, 2191, 92611, 4 FROM dual
UNION ALL SELECT 2641, 'NAICS', 1851, 621, 2, 1870, 621399, 5 FROM dual
UNION ALL SELECT 2642, 'SIC', 4307, 15, 1, 2972, 1771, 4 FROM dual
UNION ALL SELECT 2643, 'SIC', 3693, 4900, 2, 3709, 4953, 4 FROM dual
UNION ALL SELECT 2644, 'SEC', 2229, 1500, 2, 2232, 1531, 4 FROM dual
UNION ALL SELECT 2645, 'SIC', 4308, 20, 1, 3004, 2045, 4 FROM dual
UNION ALL SELECT 2646, 'SIC', 4308, 20, 1, 3014, 2062, 4 FROM dual
UNION ALL SELECT 2647, 'NAICS', 218, 237, 2, 233, 237990, 5 FROM dual
UNION ALL SELECT 2648, 'SEC', 2796, 70, 1, 2735, 7500, 2 FROM dual
UNION ALL SELECT 2649, 'NAICS', 1812, 61, 1, 1826, 611420, 5 FROM dual
UNION ALL SELECT 2650, 'NAICS', 2037, 81, 1, 2099, 812930, 5 FROM dual
UNION ALL SELECT 2651, 'SIC', 4308, 20, 1, 3024, 2077, 4 FROM dual
UNION ALL SELECT 2652, 'SEC', 2263, 2210, 3, 2264, 2211, 4 FROM dual
UNION ALL SELECT 2653, 'SEC', 2434, 3600, 2, 2447, 3661, 4 FROM dual
UNION ALL SELECT 2654, 'SIC', 4308, 20, 1, 3293, 3151, 4 FROM dual
UNION ALL SELECT 2655, 'SIC', 3419, 3500, 2, 3447, 3554, 4 FROM dual
UNION ALL SELECT 2656, 'SIC', 3301, 3200, 2, 3331, 3291, 4 FROM dual
UNION ALL SELECT 2657, 'SIC', 4312, 60, 1, 4005, 6798, 4 FROM dual
UNION ALL SELECT 2658, 'NAICS', 1851, 621, 2, 1868, 62139, 4 FROM dual
UNION ALL SELECT 2659, 'NAICS', 1850, 62, 1, 1907, 62311, 4 FROM dual
UNION ALL SELECT 2660, 'SIC', 3086, 2320, 3, 3089, 2323, 4 FROM dual
UNION ALL SELECT 2661, 'NAICS', 2020, 722, 2, 2027, 72233, 4 FROM dual
UNION ALL SELECT 2662, 'NAICS', 1004, 4239, 3, 1009, 423930, 5 FROM dual
UNION ALL SELECT 2663, 'NAICS', 126, 1152, 3, 128, 11521, 4 FROM dual
UNION ALL SELECT 2664, 'SIC', 4308, 20, 1, 3075, 2282, 4 FROM dual
UNION ALL SELECT 2665, 'SIC', 3564, 3840, 3, 3569, 3845, 4 FROM dual
UNION ALL SELECT 2666, 'SEC', 2791, 20, 1, 2414, 3541, 4 FROM dual
UNION ALL SELECT 2667, 'NAICS', 1850, 62, 1, 1886, 621610, 5 FROM dual
UNION ALL SELECT 2668, 'NAICS', 2135, 92, 1, 2205, 9281, 3 FROM dual
UNION ALL SELECT 2669, 'SEC', 2791, 20, 1, 2334, 2870, 3 FROM dual
UNION ALL SELECT 2670, 'NAICS', 2037, 81, 1, 2039, 8111, 3 FROM dual
UNION ALL SELECT 2671, 'NAICS', 1920, 624, 2, 1927, 62419, 4 FROM dual
UNION ALL SELECT 2672, 'SEC', 2704, 6790, 3, 2705, 6792, 4 FROM dual
UNION ALL SELECT 2673, 'SEC', 2633, 5730, 3, 2636, 5735, 4 FROM dual
UNION ALL SELECT 2674, 'NAICS', 1850, 62, 1, 1908, 6232, 3 FROM dual
UNION ALL SELECT 2675, 'NAICS', 1598, 5322, 3, 1607, 532292, 5 FROM dual
UNION ALL SELECT 2676, 'SIC', 4311, 52, 1, 3859, 5630, 3 FROM dual
UNION ALL SELECT 2677, 'SEC', 2795, 60, 1, 2703, 6700, 2 FROM dual
UNION ALL SELECT 2678, 'SIC', 2847, 700, 2, 2856, 741, 4 FROM dual
UNION ALL SELECT 2679, 'NAICS', 1725, 56, 1, 1742, 561410, 5 FROM dual
UNION ALL SELECT 2680, 'SEC', 2621, 5530, 3, 2622, 5531, 4 FROM dual
UNION ALL SELECT 2681, 'NAICS', 1570, 531, 2, 1590, 53139, 4 FROM dual
UNION ALL SELECT 2682, 'SIC', 4310, 50, 1, 3798, 5191, 4 FROM dual
UNION ALL SELECT 2683, 'SIC', 4310, 50, 1, 3787, 5159, 4 FROM dual
UNION ALL SELECT 2684, 'SIC', 3762, 5100, 2, 3763, 5110, 3 FROM dual
UNION ALL SELECT 2685, 'NAICS', 207, 2361, 3, 211, 236117, 5 FROM dual
UNION ALL SELECT 2686, 'SEC', 2791, 20, 1, 2447, 3661, 4 FROM dual
UNION ALL SELECT 2687, 'SIC', 4308, 20, 1, 3407, 3484, 4 FROM dual
UNION ALL SELECT 2688, 'SIC', 4314, 90, 1, 4295, 9660, 3 FROM dual
UNION ALL SELECT 2689, 'NAICS', 1939, 6244, 3, 1940, 624410, 5 FROM dual
UNION ALL SELECT 2690, 'SIC', 3797, 5190, 3, 3801, 5194, 4 FROM dual
UNION ALL SELECT 2691, 'SEC', 2795, 60, 1, 2693, 6411, 4 FROM dual
UNION ALL SELECT 2692, 'NAICS', 1550, 52429, 4, 1553, 524298, 5 FROM dual
UNION ALL SELECT 2693, 'SIC', 4308, 20, 1, 3172, 2630, 3 FROM dual
UNION ALL SELECT 2694, 'NAICS', 1814, 6111, 3, 1816, 61111, 4 FROM dual
UNION ALL SELECT 2695, 'NAICS', 1726, 561, 2, 1739, 561330, 5 FROM dual
UNION ALL SELECT 2696, 'SIC', 3937, 6100, 2, 3942, 6150, 3 FROM dual
UNION ALL SELECT 2697, 'NAICS', 1625, 541, 2, 1676, 541613, 5 FROM dual
UNION ALL SELECT 2698, 'SIC', 4305, 1, 1, 2863, 762, 4 FROM dual
UNION ALL SELECT 2699, 'NAICS', 931, 423, 2, 936, 42312, 4 FROM dual
UNION ALL SELECT 2700, 'NAICS', 1402, 51, 1, 1474, 519120, 5 FROM dual
UNION ALL SELECT 2701, 'NAICS', 1569, 53, 1, 1605, 53229, 4 FROM dual
UNION ALL SELECT 2702, 'SIC', 4224, 8710, 3, 4226, 8712, 4 FROM dual
UNION ALL SELECT 2703, 'NAICS', 132, 21, 1, 165, 212324, 5 FROM dual
UNION ALL SELECT 2704, 'NAICS', 144, 2122, 3, 152, 212234, 5 FROM dual
UNION ALL SELECT 2705, 'NAICS', 1513, 523, 2, 1530, 52392, 4 FROM dual
UNION ALL SELECT 2706, 'SEC', 2517, 4200, 2, 2519, 4213, 4 FROM dual
UNION ALL SELECT 2707, 'SIC', 4313, 70, 1, 4099, 7549, 4 FROM dual
UNION ALL SELECT 2708, 'SIC', 4230, 8730, 3, 4233, 8733, 4 FROM dual
UNION ALL SELECT 2709, 'NAICS', 1684, 54171, 4, 1685, 541711, 5 FROM dual
UNION ALL SELECT 2710, 'SIC', 3384, 3440, 3, 3389, 3446, 4 FROM dual
UNION ALL SELECT 2711, 'NAICS', 1812, 61, 1, 1821, 611310, 5 FROM dual
UNION ALL SELECT 2712, 'NAICS', 1442, 515, 2, 1451, 51521, 4 FROM dual
UNION ALL SELECT 2713, 'SIC', 4151, 8030, 3, 4152, 8031, 4 FROM dual
UNION ALL SELECT 2714, 'NAICS', 1, 11, 1, 48, 111930, 5 FROM dual
UNION ALL SELECT 2715, 'SEC', 2790, 15, 1, 2238, 1730, 3 FROM dual
UNION ALL SELECT 2716, 'SEC', 2695, 6510, 3, 2697, 6513, 4 FROM dual
UNION ALL SELECT 2717, 'SIC', 3744, 5070, 3, 3748, 5078, 4 FROM dual
UNION ALL SELECT 2718, 'NAICS', 3, 1111, 3, 12, 111150, 5 FROM dual
UNION ALL SELECT 2719, 'NAICS', 1035, 4244, 3, 1039, 42442, 4 FROM dual
UNION ALL SELECT 2720, 'NAICS', 2085, 8123, 3, 2087, 81231, 4 FROM dual
UNION ALL SELECT 2721, 'SEC', 2568, 5000, 2, 2582, 5065, 4 FROM dual
UNION ALL SELECT 2722, 'SIC', 3480, 3600, 2, 3501, 3646, 4 FROM dual
UNION ALL SELECT 2723, 'NAICS', 1680, 54162, 4, 1679, 541620, 5 FROM dual
UNION ALL SELECT 2724, 'SEC', 2770, 8300, 2, 2772, 8351, 4 FROM dual
UNION ALL SELECT 2725, 'SIC', 3526, 3700, 2, 3541, 3743, 4 FROM dual
UNION ALL SELECT 2726, 'SIC', 4313, 70, 1, 4068, 7373, 4 FROM dual
UNION ALL SELECT 2727, 'NAICS', 1950, 71113, 4, 1949, 711130, 5 FROM dual
UNION ALL SELECT 2728, 'SIC', 4307, 15, 1, 2949, 1610, 3 FROM dual
UNION ALL SELECT 2729, 'NAICS', 264, 23832, 4, 263, 238320, 5 FROM dual
UNION ALL SELECT 2730, 'NAICS', 1804, 5629, 3, 1807, 562920, 5 FROM dual
UNION ALL SELECT 2731, 'NAICS', 1813, 611, 2, 1849, 61171, 4 FROM dual
UNION ALL SELECT 2732, 'SIC', 3701, 4930, 3, 3702, 4931, 4 FROM dual
UNION ALL SELECT 2733, 'NAICS', 1785, 5619, 3, 1789, 56192, 4 FROM dual
UNION ALL SELECT 2734, 'SIC', 3592, 3960, 3, 3594, 3965, 4 FROM dual
UNION ALL SELECT 2735, 'SEC', 2434, 3600, 2, 2437, 3613, 4 FROM dual
UNION ALL SELECT 2736, 'SIC', 3996, 6720, 3, 3998, 6726, 4 FROM dual
UNION ALL SELECT 2737, 'NAICS', 1673, 54161, 4, 1677, 541614, 5 FROM dual
UNION ALL SELECT 2738, 'SEC', 2403, 3500, 2, 2427, 3575, 4 FROM dual
UNION ALL SELECT 2739, 'SIC', 4305, 1, 1, 2859, 751, 4 FROM dual
UNION ALL SELECT 2740, 'SIC', 4313, 70, 1, 4167, 8071, 4 FROM dual
UNION ALL SELECT 2741, 'NAICS', 1850, 62, 1, 1925, 62412, 4 FROM dual
UNION ALL SELECT 2742, 'SIC', 4312, 60, 1, 3982, 6515, 4 FROM dual
UNION ALL SELECT 2743, 'SIC', 3371, 3400, 2, 3403, 3479, 4 FROM dual
UNION ALL SELECT 2744, 'SIC', 4287, 9620, 3, 4288, 9621, 4 FROM dual
UNION ALL SELECT 2745, 'SIC', 4307, 15, 1, 2980, 1796, 4 FROM dual
UNION ALL SELECT 2746, 'SIC', 3837, 5500, 2, 3844, 5540, 3 FROM dual
UNION ALL SELECT 2747, 'NAICS', 138, 212, 2, 149, 212222, 5 FROM dual
UNION ALL SELECT 2748, 'SIC', 4312, 60, 1, 3932, 6081, 4 FROM dual
UNION ALL SELECT 2749, 'SEC', 2795, 60, 1, 2654, 6030, 3 FROM dual
UNION ALL SELECT 2750, 'NAICS', 1625, 541, 2, 1717, 54199, 4 FROM dual
UNION ALL SELECT 2751, 'NAICS', 1538, 52411, 4, 1539, 524113, 5 FROM dual
UNION ALL SELECT 2752, 'SIC', 4310, 50, 1, 3790, 5169, 4 FROM dual
UNION ALL SELECT 2753, 'NAICS', 1497, 52221, 4, 1496, 522210, 5 FROM dual
UNION ALL SELECT 2754, 'SIC', 4308, 20, 1, 3480, 3600, 2 FROM dual
UNION ALL SELECT 2755, 'NAICS', 1569, 53, 1, 1587, 531320, 5 FROM dual
UNION ALL SELECT 2756, 'NAICS', 1720, 5511, 3, 1721, 55111, 4 FROM dual
UNION ALL SELECT 2757, 'NAICS', 1830, 6115, 3, 1833, 611512, 5 FROM dual
UNION ALL SELECT 2758, 'SIC', 3729, 5040, 3, 3733, 5046, 4 FROM dual
UNION ALL SELECT 2759, 'NAICS', 1725, 56, 1, 1749, 561439, 5 FROM dual
UNION ALL SELECT 2760, 'NAICS', 146, 21221, 4, 145, 212210, 5 FROM dual
UNION ALL SELECT 2761, 'NAICS', 253, 2382, 3, 257, 23822, 4 FROM dual
UNION ALL SELECT 2762, 'NAICS', 2120, 8139, 3, 2129, 813990, 5 FROM dual
UNION ALL SELECT 2763, 'SIC', 4065, 7370, 3, 4074, 7379, 4 FROM dual
UNION ALL SELECT 2764, 'SIC', 4313, 70, 1, 4107, 7640, 3 FROM dual
UNION ALL SELECT 2765, 'NAICS', 206, 236, 2, 211, 236117, 5 FROM dual
UNION ALL SELECT 2766, 'NAICS', 1015, 424, 2, 1058, 42452, 4 FROM dual
UNION ALL SELECT 2767, 'NAICS', 1609, 5323, 3, 1611, 53231, 4 FROM dual
UNION ALL SELECT 2768, 'NAICS', 1904, 623, 2, 1919, 62399, 4 FROM dual
UNION ALL SELECT 2769, 'SIC', 4308, 20, 1, 3122, 2397, 4 FROM dual
UNION ALL SELECT 2770, 'NAICS', 1419, 512, 2, 1420, 5121, 3 FROM dual
UNION ALL SELECT 2771, 'SEC', 2262, 2200, 2, 2265, 2220, 3 FROM dual
UNION ALL SELECT 2772, 'NAICS', 1480, 52, 1, 1481, 521, 2 FROM dual
UNION ALL SELECT 2773, 'NAICS', 2134, 81411, 4, 2133, 814110, 5 FROM dual
UNION ALL SELECT 2774, 'SIC', 3012, 2060, 3, 3016, 2064, 4 FROM dual
UNION ALL SELECT 2775, 'NAICS', 2014, 72121, 4, 2016, 721214, 5 FROM dual
UNION ALL SELECT 2776, 'SIC', 4311, 52, 1, 3835, 5490, 3 FROM dual
UNION ALL SELECT 2777, 'NAICS', 1725, 56, 1, 1768, 561611, 5 FROM dual
UNION ALL SELECT 2778, 'SIC', 3552, 3800, 2, 3572, 3860, 3 FROM dual
UNION ALL SELECT 2779, 'SIC', 4314, 90, 1, 4268, 9400, 2 FROM dual
UNION ALL SELECT 2780, 'SIC', 4308, 20, 1, 3135, 2436, 4 FROM dual
UNION ALL SELECT 2781, 'SIC', 4305, 1, 1, 2800, 111, 4 FROM dual
UNION ALL SELECT 2782, 'SEC', 2753, 8000, 2, 2754, 8010, 3 FROM dual
UNION ALL SELECT 2783, 'SIC', 2884, 1000, 2, 2895, 1061, 4 FROM dual
UNION ALL SELECT 2784, 'SEC', 2458, 3700, 2, 2474, 3760, 3 FROM dual
UNION ALL SELECT 2785, 'SIC', 3444, 3550, 3, 3448, 3555, 4 FROM dual
UNION ALL SELECT 2786, 'NAICS', 1583, 5313, 3, 1588, 53132, 4 FROM dual
UNION ALL SELECT 2787, 'SEC', 2623, 5600, 2, 2629, 5661, 4 FROM dual
UNION ALL SELECT 2788, 'NAICS', 1569, 53, 1, 1571, 5311, 3 FROM dual
UNION ALL SELECT 2789, 'SIC', 3822, 5400, 2, 3831, 5450, 3 FROM dual
UNION ALL SELECT 2790, 'NAICS', 1767, 56161, 4, 1770, 561613, 5 FROM dual
UNION ALL SELECT 2791, 'NAICS', 200, 22131, 4, 199, 221310, 5 FROM dual
UNION ALL SELECT 2792, 'NAICS', 1569, 53, 1, 1578, 531190, 5 FROM dual
UNION ALL SELECT 2793, 'SIC', 3131, 2430, 3, 3133, 2434, 4 FROM dual
UNION ALL SELECT 2794, 'SIC', 4305, 1, 1, 2873, 850, 3 FROM dual
UNION ALL SELECT 2795, 'NAICS', 2, 111, 2, 14, 111160, 5 FROM dual
UNION ALL SELECT 2796, 'SEC', 2459, 3710, 3, 2462, 3714, 4 FROM dual
UNION ALL SELECT 2797, 'SIC', 4313, 70, 1, 4120, 7830, 3 FROM dual
UNION ALL SELECT 2798, 'SIC', 3484, 3620, 3, 3488, 3629, 4 FROM dual
UNION ALL SELECT 2799, 'SIC', 4312, 60, 1, 3976, 6411, 4 FROM dual
UNION ALL SELECT 2800, 'SIC', 3371, 3400, 2, 3411, 3492, 4 FROM dual
UNION ALL SELECT 2801, 'SEC', 2792, 40, 1, 2513, 4010, 3 FROM dual
UNION ALL SELECT 2802, 'SIC', 4125, 7900, 2, 4126, 7910, 3 FROM dual
UNION ALL SELECT 2803, 'SIC', 4192, 8300, 2, 4198, 8351, 4 FROM dual
UNION ALL SELECT 2804, 'SIC', 3124, 2400, 2, 3127, 2420, 3 FROM dual
UNION ALL SELECT 2805, 'NAICS', 68, 1123, 3, 73, 112330, 5 FROM dual
UNION ALL SELECT 2806, 'SIC', 3537, 3730, 3, 3538, 3731, 4 FROM dual
UNION ALL SELECT 2807, 'SIC', 4308, 20, 1, 3292, 3150, 3 FROM dual
UNION ALL SELECT 2808, 'NAICS', 2003, 72, 1, 2031, 7225, 3 FROM dual
UNION ALL SELECT 2809, 'SIC', 4308, 20, 1, 3253, 2910, 3 FROM dual
UNION ALL SELECT 2810, 'NAICS', 1039, 42442, 4, 1038, 424420, 5 FROM dual
UNION ALL SELECT 2811, 'NAICS', 1624, 54, 1, 1671, 541519, 5 FROM dual
UNION ALL SELECT 2812, 'NAICS', 2135, 92, 1, 2190, 926110, 5 FROM dual
UNION ALL SELECT 2813, 'NAICS', 931, 423, 2, 1002, 423860, 5 FROM dual
UNION ALL SELECT 2814, 'NAICS', 2135, 92, 1, 2152, 922110, 5 FROM dual
UNION ALL SELECT 2815, 'SIC', 4308, 20, 1, 3173, 2631, 4 FROM dual
UNION ALL SELECT 2816, 'SIC', 4224, 8710, 3, 4227, 8713, 4 FROM dual
UNION ALL SELECT 2817, 'SIC', 3489, 3630, 3, 3494, 3635, 4 FROM dual
UNION ALL SELECT 2818, 'SEC', 2241, 2010, 3, 2243, 2013, 4 FROM dual
UNION ALL SELECT 2819, 'SIC', 3762, 5100, 2, 3773, 5139, 4 FROM dual
UNION ALL SELECT 2820, 'SIC', 4310, 50, 1, 3726, 5032, 4 FROM dual
UNION ALL SELECT 2821, 'SIC', 4313, 70, 1, 4065, 7370, 3 FROM dual
UNION ALL SELECT 2822, 'SEC', 2722, 7360, 3, 2723, 7361, 4 FROM dual
UNION ALL SELECT 2823, 'NAICS', 1015, 424, 2, 1053, 42449, 4 FROM dual
UNION ALL SELECT 2824, 'NAICS', 1812, 61, 1, 1833, 611512, 5 FROM dual
UNION ALL SELECT 2825, 'NAICS', 1015, 424, 2, 1057, 424520, 5 FROM dual
UNION ALL SELECT 2826, 'SEC', 2391, 3440, 3, 2394, 3444, 4 FROM dual
UNION ALL SELECT 2827, 'SEC', 2403, 3500, 2, 2414, 3541, 4 FROM dual
UNION ALL SELECT 2828, 'SIC', 3480, 3600, 2, 3489, 3630, 3 FROM dual
UNION ALL SELECT 2829, 'NAICS', 1442, 515, 2, 1449, 5152, 3 FROM dual
UNION ALL SELECT 2830, 'SEC', 2795, 60, 1, 2649, 6000, 2 FROM dual
UNION ALL SELECT 2831, 'SEC', 2717, 7330, 3, 2718, 7331, 4 FROM dual
UNION ALL SELECT 2832, 'SIC', 4310, 50, 1, 3724, 5030, 3 FROM dual
UNION ALL SELECT 2833, 'NAICS', 931, 423, 2, 1014, 42399, 4 FROM dual
UNION ALL SELECT 2834, 'SIC', 4309, 40, 1, 3684, 4820, 3 FROM dual
UNION ALL SELECT 2835, 'NAICS', 1766, 5616, 3, 1767, 56161, 4 FROM dual
UNION ALL SELECT 2836, 'SEC', 2791, 20, 1, 2319, 2800, 2 FROM dual
UNION ALL SELECT 2837, 'NAICS', 1015, 424, 2, 1073, 42481, 4 FROM dual
UNION ALL SELECT 2838, 'SEC', 2219, 1220, 3, 2220, 1221, 4 FROM dual
UNION ALL SELECT 2839, 'SIC', 4087, 7520, 3, 4088, 7521, 4 FROM dual
UNION ALL SELECT 2840, 'SIC', 3507, 3660, 3, 3510, 3669, 4 FROM dual
UNION ALL SELECT 2841, 'SIC', 3774, 5140, 3, 3782, 5148, 4 FROM dual
UNION ALL SELECT 2842, 'SIC', 3822, 5400, 2, 3830, 5441, 4 FROM dual
UNION ALL SELECT 2843, 'NAICS', 1851, 621, 2, 1885, 6216, 3 FROM dual
UNION ALL SELECT 2844, 'SIC', 2901, 1200, 2, 2904, 1222, 4 FROM dual
UNION ALL SELECT 2845, 'NAICS', 1812, 61, 1, 1843, 61169, 4 FROM dual
UNION ALL SELECT 2846, 'SIC', 4311, 52, 1, 3904, 5961, 4 FROM dual
UNION ALL SELECT 2847, 'SEC', 2788, 1, 1, 2214, 900, 2 FROM dual
UNION ALL SELECT 2848, 'SEC', 2526, 4500, 2, 2530, 4520, 3 FROM dual
UNION ALL SELECT 2849, 'SIC', 3208, 2780, 3, 3209, 2782, 4 FROM dual
UNION ALL SELECT 2850, 'NAICS', 955, 4234, 3, 962, 423440, 5 FROM dual
UNION ALL SELECT 2851, 'SIC', 4208, 8600, 2, 4216, 8641, 4 FROM dual
UNION ALL SELECT 2852, 'SIC', 3818, 5330, 3, 3819, 5331, 4 FROM dual
UNION ALL SELECT 2853, 'NAICS', 1015, 424, 2, 1029, 424320, 5 FROM dual
UNION ALL SELECT 2854, 'NAICS', 1582, 53121, 4, 1581, 531210, 5 FROM dual
UNION ALL SELECT 2855, 'SEC', 4320, 6170, 3, 4340, 6172, 4 FROM dual
UNION ALL SELECT 2856, 'NAICS', 2003, 72, 1, 2035, 722514, 5 FROM dual
UNION ALL SELECT 2857, 'SIC', 4308, 20, 1, 3041, 2100, 2 FROM dual
UNION ALL SELECT 2858, 'SIC', 3371, 3400, 2, 3386, 3442, 4 FROM dual
UNION ALL SELECT 2859, 'SIC', 3715, 5000, 2, 3761, 5099, 4 FROM dual
UNION ALL SELECT 2860, 'SEC', 2704, 6790, 3, 4328, 6795, 4 FROM dual
UNION ALL SELECT 2861, 'SIC', 3715, 5000, 2, 3731, 5044, 4 FROM dual
UNION ALL SELECT 2862, 'NAICS', 1, 11, 1, 55, 111998, 5 FROM dual
UNION ALL SELECT 2863, 'NAICS', 253, 2382, 3, 256, 238220, 5 FROM dual
UNION ALL SELECT 2864, 'NAICS', 1403, 511, 2, 1404, 5111, 3 FROM dual
UNION ALL SELECT 2865, 'SIC', 4307, 15, 1, 2963, 1741, 4 FROM dual
UNION ALL SELECT 2866, 'SIC', 4309, 40, 1, 3646, 4481, 4 FROM dual
UNION ALL SELECT 2867, 'SEC', 2794, 52, 1, 2623, 5600, 2 FROM dual
UNION ALL SELECT 2868, 'NAICS', 56, 112, 2, 86, 112511, 5 FROM dual
UNION ALL SELECT 2869, 'NAICS', 1015, 424, 2, 1017, 424110, 5 FROM dual
UNION ALL SELECT 2870, 'NAICS', 1569, 53, 1, 1585, 531311, 5 FROM dual
UNION ALL SELECT 2871, 'SEC', 2738, 7800, 2, 2745, 7830, 3 FROM dual
UNION ALL SELECT 2872, 'NAICS', 1625, 541, 2, 1648, 54134, 4 FROM dual
UNION ALL SELECT 2873, 'SEC', 2725, 7370, 3, 2729, 7374, 4 FROM dual
UNION ALL SELECT 2874, 'NAICS', 1813, 611, 2, 1820, 6113, 3 FROM dual
UNION ALL SELECT 2875, 'SIC', 2993, 2030, 3, 2994, 2032, 4 FROM dual
UNION ALL SELECT 2876, 'SIC', 4308, 20, 1, 3161, 2540, 3 FROM dual
UNION ALL SELECT 2877, 'SEC', 2796, 70, 1, 2739, 7810, 3 FROM dual
UNION ALL SELECT 2878, 'SIC', 3419, 3500, 2, 3441, 3547, 4 FROM dual
UNION ALL SELECT 2879, 'SIC', 4308, 20, 1, 3061, 2252, 4 FROM dual
UNION ALL SELECT 2880, 'SIC', 3108, 2380, 3, 3113, 2387, 4 FROM dual
UNION ALL SELECT 2881, 'NAICS', 1850, 62, 1, 1915, 623311, 5 FROM dual
UNION ALL SELECT 2882, 'SIC', 3540, 3740, 3, 3541, 3743, 4 FROM dual
UNION ALL SELECT 2883, 'NAICS', 977, 42361, 4, 976, 423610, 5 FROM dual
UNION ALL SELECT 2884, 'NAICS', 1969, 712, 2, 1978, 71219, 4 FROM dual
UNION ALL SELECT 2885, 'SIC', 3301, 3200, 2, 3322, 3270, 3 FROM dual
UNION ALL SELECT 2886, 'SIC', 3680, 4800, 2, 3686, 4830, 3 FROM dual
UNION ALL SELECT 2887, 'SIC', 4309, 40, 1, 3636, 4400, 2 FROM dual
UNION ALL SELECT 2888, 'SIC', 2982, 2000, 2, 2993, 2030, 3 FROM dual
UNION ALL SELECT 2889, 'SIC', 3762, 5100, 2, 3792, 5171, 4 FROM dual
UNION ALL SELECT 2890, 'NAICS', 157, 21231, 4, 159, 212312, 5 FROM dual
UNION ALL SELECT 2891, 'NAICS', 2038, 811, 2, 2056, 811213, 5 FROM dual
UNION ALL SELECT 2892, 'NAICS', 1066, 4247, 3, 1068, 42471, 4 FROM dual
UNION ALL SELECT 2893, 'NAICS', 1726, 561, 2, 1743, 56141, 4 FROM dual
UNION ALL SELECT 2894, 'NAICS', 1, 11, 1, 38, 111411, 5 FROM dual
UNION ALL SELECT 2895, 'SEC', 2240, 2000, 2, 2255, 2082, 4 FROM dual
UNION ALL SELECT 2896, 'SEC', 2384, 3400, 2, 2399, 3460, 3 FROM dual
UNION ALL SELECT 2897, 'SIC', 3059, 2250, 3, 3063, 2254, 4 FROM dual
UNION ALL SELECT 2898, 'NAICS', 2135, 92, 1, 2156, 922130, 5 FROM dual
UNION ALL SELECT 2899, 'SEC', 2795, 60, 1, 4326, 6532, 3 FROM dual
UNION ALL SELECT 2900, 'NAICS', 1984, 71312, 4, 1983, 713120, 5 FROM dual
UNION ALL SELECT 2901, 'NAICS', 1462, 5179, 3, 1465, 517919, 5 FROM dual
UNION ALL SELECT 2902, 'SIC', 3480, 3600, 2, 3499, 3644, 4 FROM dual
UNION ALL SELECT 2903, 'SIC', 4308, 20, 1, 3427, 3531, 4 FROM dual
UNION ALL SELECT 2904, 'SIC', 2861, 760, 3, 2863, 762, 4 FROM dual
UNION ALL SELECT 2905, 'NAICS', 138, 212, 2, 161, 212319, 5 FROM dual
UNION ALL SELECT 2906, 'SEC', 2713, 7300, 2, 2721, 7359, 4 FROM dual
UNION ALL SELECT 2907, 'SIC', 2825, 200, 2, 2837, 253, 4 FROM dual
UNION ALL SELECT 2908, 'SIC', 4314, 90, 1, 4278, 9510, 3 FROM dual
UNION ALL SELECT 2909, 'SIC', 3371, 3400, 2, 3390, 3448, 4 FROM dual
UNION ALL SELECT 2910, 'SIC', 3854, 5600, 2, 3864, 5651, 4 FROM dual
UNION ALL SELECT 2911, 'SIC', 4190, 8290, 3, 4191, 8299, 4 FROM dual
UNION ALL SELECT 2912, 'SIC', 3301, 3200, 2, 3323, 3271, 4 FROM dual
UNION ALL SELECT 2913, 'SIC', 4065, 7370, 3, 4070, 7375, 4 FROM dual
UNION ALL SELECT 2914, 'SIC', 4308, 20, 1, 3443, 3549, 4 FROM dual
UNION ALL SELECT 2915, 'NAICS', 218, 237, 2, 223, 23712, 4 FROM dual
UNION ALL SELECT 2916, 'SEC', 2536, 4700, 2, 2537, 4730, 3 FROM dual
UNION ALL SELECT 2917, 'SIC', 4215, 8640, 3, 4216, 8641, 4 FROM dual
UNION ALL SELECT 2918, 'SIC', 4308, 20, 1, 3530, 3714, 4 FROM dual
UNION ALL SELECT 2919, 'SIC', 3148, 2500, 2, 3159, 2530, 3 FROM dual
UNION ALL SELECT 2920, 'SEC', 2791, 20, 1, 2290, 2540, 3 FROM dual
UNION ALL SELECT 2921, 'SIC', 3261, 3000, 2, 3267, 3052, 4 FROM dual
UNION ALL SELECT 2922, 'SEC', 2791, 20, 1, 2258, 2092, 4 FROM dual
UNION ALL SELECT 2923, 'SIC', 3774, 5140, 3, 3776, 5142, 4 FROM dual
UNION ALL SELECT 2924, 'NAICS', 1625, 541, 2, 1688, 54172, 4 FROM dual
UNION ALL SELECT 2925, 'SIC', 4313, 70, 1, 4211, 8620, 3 FROM dual
UNION ALL SELECT 2926, 'NAICS', 132, 21, 1, 157, 21231, 4 FROM dual
UNION ALL SELECT 2927, 'NAICS', 138, 212, 2, 166, 212325, 5 FROM dual
UNION ALL SELECT 2928, 'NAICS', 1942, 71, 1, 1944, 7111, 3 FROM dual
UNION ALL SELECT 2929, 'SIC', 4040, 7300, 2, 4046, 7320, 3 FROM dual
UNION ALL SELECT 2930, 'NAICS', 205, 23, 1, 211, 236117, 5 FROM dual
UNION ALL SELECT 2931, 'NAICS', 218, 237, 2, 231, 23731, 4 FROM dual
UNION ALL SELECT 2932, 'NAICS', 2013, 7212, 3, 2015, 721211, 5 FROM dual
UNION ALL SELECT 2933, 'SEC', 2434, 3600, 2, 2454, 3678, 4 FROM dual
UNION ALL SELECT 2934, 'NAICS', 982, 4237, 3, 988, 42373, 4 FROM dual
UNION ALL SELECT 2935, 'SEC', 2302, 2700, 2, 2305, 2720, 3 FROM dual
UNION ALL SELECT 2936, 'NAICS', 2141, 92112, 4, 2140, 921120, 5 FROM dual
UNION ALL SELECT 2937, 'NAICS', 1785, 5619, 3, 1788, 561920, 5 FROM dual
UNION ALL SELECT 2938, 'SIC', 3148, 2500, 2, 3154, 2517, 4 FROM dual
UNION ALL SELECT 2939, 'SIC', 4312, 60, 1, 3966, 6350, 3 FROM dual
UNION ALL SELECT 2940, 'NAICS', 1985, 7132, 3, 1987, 71321, 4 FROM dual
UNION ALL SELECT 2941, 'SIC', 3050, 2200, 2, 3066, 2259, 4 FROM dual
UNION ALL SELECT 2942, 'SIC', 4310, 50, 1, 3733, 5046, 4 FROM dual
UNION ALL SELECT 2943, 'SIC', 3636, 4400, 2, 3645, 4480, 3 FROM dual
UNION ALL SELECT 2944, 'SIC', 4040, 7300, 2, 4073, 7378, 4 FROM dual
UNION ALL SELECT 2945, 'SEC', 2302, 2700, 2, 2312, 2750, 3 FROM dual
UNION ALL SELECT 2946, 'SIC', 3886, 5900, 2, 3907, 5980, 3 FROM dual
UNION ALL SELECT 2947, 'SIC', 3921, 6020, 3, 3923, 6022, 4 FROM dual
UNION ALL SELECT 2948, 'NAICS', 2103, 813, 2, 2126, 81393, 4 FROM dual
UNION ALL SELECT 2949, 'SEC', 2792, 40, 1, 2556, 4922, 4 FROM dual
UNION ALL SELECT 2950, 'SIC', 4308, 20, 1, 3146, 2493, 4 FROM dual
UNION ALL SELECT 2951, 'NAICS', 1569, 53, 1, 1616, 532420, 5 FROM dual
UNION ALL SELECT 2952, 'NAICS', 1, 11, 1, 109, 1141, 3 FROM dual
UNION ALL SELECT 2953, 'NAICS', 1794, 56211, 4, 1796, 562112, 5 FROM dual
UNION ALL SELECT 2954, 'NAICS', 2187, 92512, 4, 2186, 925120, 5 FROM dual
UNION ALL SELECT 2955, 'SIC', 3662, 4600, 2, 3665, 4613, 4 FROM dual
UNION ALL SELECT 2956, 'SIC', 4313, 70, 1, 4160, 8052, 4 FROM dual
UNION ALL SELECT 2957, 'SIC', 4307, 15, 1, 2948, 1600, 2 FROM dual
UNION ALL SELECT 2958, 'SIC', 4313, 70, 1, 4042, 7311, 4 FROM dual
UNION ALL SELECT 2959, 'NAICS', 144, 2122, 3, 147, 21222, 4 FROM dual
UNION ALL SELECT 2960, 'SEC', 2667, 6160, 3, 2668, 6162, 4 FROM dual
UNION ALL SELECT 2961, 'SIC', 4081, 7500, 2, 4083, 7513, 4 FROM dual
UNION ALL SELECT 2962, 'NAICS', 36, 1114, 3, 38, 111411, 5 FROM dual
UNION ALL SELECT 2963, 'NAICS', 2103, 813, 2, 2113, 81331, 4 FROM dual
UNION ALL SELECT 2964, 'SIC', 4314, 90, 1, 4276, 9451, 4 FROM dual
UNION ALL SELECT 2965, 'SIC', 4308, 20, 1, 3545, 3761, 4 FROM dual
UNION ALL SELECT 2966, 'NAICS', 2103, 813, 2, 2115, 813312, 5 FROM dual
UNION ALL SELECT 2967, 'SEC', 2791, 20, 1, 2433, 3590, 3 FROM dual
UNION ALL SELECT 2968, 'SIC', 2907, 1240, 3, 2908, 1241, 4 FROM dual
UNION ALL SELECT 2969, 'NAICS', 2181, 92412, 4, 2180, 924120, 5 FROM dual
UNION ALL SELECT 2970, 'SIC', 4313, 70, 1, 4158, 8050, 3 FROM dual
UNION ALL SELECT 2971, 'NAICS', 181, 221, 2, 185, 221112, 5 FROM dual
UNION ALL SELECT 2972, 'NAICS', 2120, 8139, 3, 2122, 81391, 4 FROM dual
UNION ALL SELECT 2973, 'SEC', 2526, 4500, 2, 2529, 4513, 4 FROM dual
UNION ALL SELECT 2974, 'SEC', 2458, 3700, 2, 2471, 3743, 4 FROM dual
UNION ALL SELECT 2975, 'SIC', 3316, 3260, 3, 3320, 3264, 4 FROM dual
UNION ALL SELECT 2976, 'SIC', 3886, 5900, 2, 3887, 5910, 3 FROM dual
UNION ALL SELECT 2977, 'SEC', 2664, 6150, 3, 2666, 6159, 4 FROM dual
UNION ALL SELECT 2978, 'NAICS', 2038, 811, 2, 2054, 811211, 5 FROM dual
UNION ALL SELECT 2979, 'NAICS', 2166, 923, 2, 2173, 92313, 4 FROM dual
UNION ALL SELECT 2980, 'SIC', 3371, 3400, 2, 3396, 3462, 4 FROM dual
UNION ALL SELECT 2981, 'SIC', 3337, 3300, 2, 3343, 3317, 4 FROM dual
UNION ALL SELECT 2982, 'NAICS', 1813, 611, 2, 1828, 611430, 5 FROM dual
UNION ALL SELECT 2983, 'SEC', 2434, 3600, 2, 2446, 3660, 3 FROM dual
UNION ALL SELECT 2984, 'SIC', 4309, 40, 1, 3655, 4510, 3 FROM dual
UNION ALL SELECT 2985, 'SIC', 3654, 4500, 2, 3655, 4510, 3 FROM dual
UNION ALL SELECT 2986, 'NAICS', 2039, 8111, 3, 2046, 811121, 5 FROM dual
UNION ALL SELECT 2987, 'SEC', 2659, 6100, 2, 4322, 6180, 3 FROM dual
UNION ALL SELECT 2988, 'SIC', 4308, 20, 1, 3458, 3567, 4 FROM dual
UNION ALL SELECT 2989, 'SIC', 3886, 5900, 2, 3909, 5984, 4 FROM dual
UNION ALL SELECT 2990, 'NAICS', 202, 22132, 4, 201, 221320, 5 FROM dual
UNION ALL SELECT 2991, 'SIC', 2798, 100, 2, 2800, 111, 4 FROM dual
UNION ALL SELECT 2992, 'NAICS', 273, 2389, 3, 275, 23891, 4 FROM dual
UNION ALL SELECT 2993, 'SIC', 4313, 70, 1, 4061, 7359, 4 FROM dual
UNION ALL SELECT 2994, 'SIC', 3252, 2900, 2, 3256, 2951, 4 FROM dual
UNION ALL SELECT 2995, 'SIC', 3576, 3900, 2, 3595, 3990, 3 FROM dual
UNION ALL SELECT 2996, 'NAICS', 173, 2131, 3, 174, 21311, 4 FROM dual
UNION ALL SELECT 2997, 'SEC', 2476, 3800, 2, 2498, 3870, 3 FROM dual
UNION ALL SELECT 2998, 'SIC', 3282, 3100, 2, 3300, 3199, 4 FROM dual
UNION ALL SELECT 2999, 'SIC', 3301, 3200, 2, 3319, 3263, 4 FROM dual
UNION ALL SELECT 3000, 'NAICS', 1569, 53, 1, 1608, 532299, 5 FROM dual
UNION ALL SELECT 3001, 'NAICS', 1441, 51229, 4, 1440, 512290, 5 FROM dual
UNION ALL SELECT 3002, 'NAICS', 1480, 52, 1, 1553, 524298, 5 FROM dual
UNION ALL SELECT 3003, 'SIC', 4146, 8000, 2, 4167, 8071, 4 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 3004, 'SIC', 4311, 52, 1, 3864, 5651, 4 FROM dual
UNION ALL SELECT 3005, 'NAICS', 218, 237, 2, 224, 237130, 5 FROM dual
UNION ALL SELECT 3006, 'NAICS', 1420, 5121, 3, 1429, 512191, 5 FROM dual
UNION ALL SELECT 3007, 'SIC', 3020, 2070, 3, 3024, 2077, 4 FROM dual
UNION ALL SELECT 3008, 'NAICS', 1741, 5614, 3, 1742, 561410, 5 FROM dual
UNION ALL SELECT 3009, 'NAICS', 2037, 81, 1, 2103, 813, 2 FROM dual
UNION ALL SELECT 3010, 'SEC', 2355, 3200, 2, 2366, 3270, 3 FROM dual
UNION ALL SELECT 3011, 'SEC', 2450, 3670, 3, 2455, 3679, 4 FROM dual
UNION ALL SELECT 3012, 'NAICS', 57, 1121, 3, 63, 112130, 5 FROM dual
UNION ALL SELECT 3013, 'SIC', 4247, 9100, 2, 4250, 9120, 3 FROM dual
UNION ALL SELECT 3014, 'NAICS', 1, 11, 1, 131, 11531, 4 FROM dual
UNION ALL SELECT 3015, 'SEC', 2791, 20, 1, 2446, 3660, 3 FROM dual
UNION ALL SELECT 3016, 'SEC', 2523, 4400, 2, 2525, 4412, 4 FROM dual
UNION ALL SELECT 3017, 'SEC', 2791, 20, 1, 2242, 2011, 4 FROM dual
UNION ALL SELECT 3018, 'SIC', 4308, 20, 1, 3479, 3599, 4 FROM dual
UNION ALL SELECT 3019, 'NAICS', 1804, 5629, 3, 1810, 562991, 5 FROM dual
UNION ALL SELECT 3020, 'SIC', 4308, 20, 1, 3034, 2091, 4 FROM dual
UNION ALL SELECT 3021, 'NAICS', 1624, 54, 1, 1662, 541430, 5 FROM dual
UNION ALL SELECT 3022, 'SIC', 3975, 6410, 3, 3976, 6411, 4 FROM dual
UNION ALL SELECT 3023, 'NAICS', 1569, 53, 1, 1619, 53249, 4 FROM dual
UNION ALL SELECT 3024, 'SIC', 3555, 3820, 3, 3562, 3827, 4 FROM dual
UNION ALL SELECT 3025, 'SIC', 4308, 20, 1, 3249, 2893, 4 FROM dual
UNION ALL SELECT 3026, 'SIC', 4040, 7300, 2, 4076, 7381, 4 FROM dual
UNION ALL SELECT 3027, 'SIC', 2875, 900, 2, 2882, 970, 3 FROM dual
UNION ALL SELECT 3028, 'NAICS', 174, 21311, 4, 177, 213113, 5 FROM dual
UNION ALL SELECT 3029, 'SEC', 2378, 3340, 3, 2379, 3341, 4 FROM dual
UNION ALL SELECT 3030, 'SIC', 4308, 20, 1, 3496, 3640, 3 FROM dual
UNION ALL SELECT 3031, 'NAICS', 931, 423, 2, 983, 423710, 5 FROM dual
UNION ALL SELECT 3032, 'SIC', 4308, 20, 1, 3521, 3691, 4 FROM dual
UNION ALL SELECT 3033, 'SIC', 4308, 20, 1, 3237, 2860, 3 FROM dual
UNION ALL SELECT 3034, 'SIC', 4308, 20, 1, 3379, 3429, 4 FROM dual
UNION ALL SELECT 3035, 'SIC', 4313, 70, 1, 4011, 7021, 4 FROM dual
UNION ALL SELECT 3036, 'SIC', 3026, 2080, 3, 3029, 2084, 4 FROM dual
UNION ALL SELECT 3037, 'SEC', 2791, 20, 1, 2454, 3678, 4 FROM dual
UNION ALL SELECT 3038, 'SEC', 2568, 5000, 2, 2581, 5064, 4 FROM dual
UNION ALL SELECT 3039, 'SIC', 4308, 20, 1, 3065, 2258, 4 FROM dual
UNION ALL SELECT 3040, 'SIC', 4242, 8810, 3, 4243, 8811, 4 FROM dual
UNION ALL SELECT 3041, 'NAICS', 140, 21211, 4, 143, 212113, 5 FROM dual
UNION ALL SELECT 3042, 'NAICS', 167, 21239, 4, 171, 212399, 5 FROM dual
UNION ALL SELECT 3043, 'NAICS', 1413, 51119, 4, 1415, 511199, 5 FROM dual
UNION ALL SELECT 3044, 'NAICS', 1419, 512, 2, 1424, 51212, 4 FROM dual
UNION ALL SELECT 3045, 'SIC', 2864, 780, 3, 2867, 783, 4 FROM dual
UNION ALL SELECT 3046, 'NAICS', 116, 11421, 4, 115, 114210, 5 FROM dual
UNION ALL SELECT 3047, 'SEC', 2791, 20, 1, 2463, 3715, 4 FROM dual
UNION ALL SELECT 3048, 'SIC', 2918, 1400, 2, 2922, 1422, 4 FROM dual
UNION ALL SELECT 3049, 'SIC', 4308, 20, 1, 3432, 3536, 4 FROM dual
UNION ALL SELECT 3050, 'NAICS', 1709, 54192, 4, 1710, 541921, 5 FROM dual
UNION ALL SELECT 3051, 'SIC', 4311, 52, 1, 3843, 5531, 4 FROM dual
UNION ALL SELECT 3052, 'NAICS', 1624, 54, 1, 1705, 54189, 4 FROM dual
UNION ALL SELECT 3053, 'SEC', 2797, 90, 1, 2785, 9700, 2 FROM dual
UNION ALL SELECT 3054, 'SIC', 3241, 2870, 3, 3243, 2874, 4 FROM dual
UNION ALL SELECT 3055, 'SIC', 4313, 70, 1, 4215, 8640, 3 FROM dual
UNION ALL SELECT 3056, 'SIC', 2847, 700, 2, 2867, 783, 4 FROM dual
UNION ALL SELECT 3057, 'NAICS', 79, 1124, 3, 80, 112410, 5 FROM dual
UNION ALL SELECT 3058, 'SIC', 3149, 2510, 3, 3150, 2511, 4 FROM dual
UNION ALL SELECT 3059, 'SEC', 2591, 5100, 2, 2602, 5172, 4 FROM dual
UNION ALL SELECT 3060, 'NAICS', 1569, 53, 1, 1604, 53223, 4 FROM dual
UNION ALL SELECT 3061, 'SEC', 2796, 70, 1, 2772, 8351, 4 FROM dual
UNION ALL SELECT 3062, 'NAICS', 68, 1123, 3, 69, 112310, 5 FROM dual
UNION ALL SELECT 3063, 'NAICS', 1402, 51, 1, 1477, 51913, 4 FROM dual
UNION ALL SELECT 3064, 'SIC', 4305, 1, 1, 2827, 211, 4 FROM dual
UNION ALL SELECT 3065, 'NAICS', 1744, 56142, 4, 1745, 561421, 5 FROM dual
UNION ALL SELECT 3066, 'SIC', 4308, 20, 1, 3033, 2090, 3 FROM dual
UNION ALL SELECT 3067, 'SIC', 3937, 6100, 2, 3943, 6153, 4 FROM dual
UNION ALL SELECT 3068, 'NAICS', 1569, 53, 1, 1589, 531390, 5 FROM dual
UNION ALL SELECT 3069, 'SIC', 4230, 8730, 3, 4232, 8732, 4 FROM dual
UNION ALL SELECT 3070, 'SIC', 4162, 8060, 3, 4165, 8069, 4 FROM dual
UNION ALL SELECT 3071, 'NAICS', 1490, 52212, 4, 1489, 522120, 5 FROM dual
UNION ALL SELECT 3072, 'NAICS', 172, 213, 2, 178, 213114, 5 FROM dual
UNION ALL SELECT 3073, 'NAICS', 1625, 541, 2, 1635, 54121, 4 FROM dual
UNION ALL SELECT 3074, 'NAICS', 1625, 541, 2, 1710, 541921, 5 FROM dual
UNION ALL SELECT 3075, 'SIC', 4306, 10, 1, 2921, 1420, 3 FROM dual
UNION ALL SELECT 3076, 'SIC', 3762, 5100, 2, 3779, 5145, 4 FROM dual
UNION ALL SELECT 3077, 'SIC', 4313, 70, 1, 4062, 7360, 3 FROM dual
UNION ALL SELECT 3078, 'SEC', 2795, 60, 1, 2667, 6160, 3 FROM dual
UNION ALL SELECT 3079, 'NAICS', 1485, 522, 2, 1511, 522390, 5 FROM dual
UNION ALL SELECT 3080, 'SIC', 4311, 52, 1, 3829, 5440, 3 FROM dual
UNION ALL SELECT 3081, 'SIC', 3693, 4900, 2, 3704, 4939, 4 FROM dual
UNION ALL SELECT 3082, 'NAICS', 1741, 5614, 3, 1749, 561439, 5 FROM dual
UNION ALL SELECT 3083, 'SIC', 4309, 40, 1, 3617, 4151, 4 FROM dual
UNION ALL SELECT 3084, 'NAICS', 64, 11213, 4, 63, 112130, 5 FROM dual
UNION ALL SELECT 3085, 'NAICS', 253, 2382, 3, 255, 23821, 4 FROM dual
UNION ALL SELECT 3086, 'NAICS', 930, 42, 1, 1079, 424920, 5 FROM dual
UNION ALL SELECT 3087, 'SIC', 2918, 1400, 2, 2929, 1455, 4 FROM dual
UNION ALL SELECT 3088, 'NAICS', 1908, 6232, 3, 1912, 62322, 4 FROM dual
UNION ALL SELECT 3089, 'SIC', 4313, 70, 1, 4106, 7631, 4 FROM dual
UNION ALL SELECT 3090, 'NAICS', 180, 22, 1, 196, 221210, 5 FROM dual
UNION ALL SELECT 3091, 'NAICS', 2003, 72, 1, 2034, 722513, 5 FROM dual
UNION ALL SELECT 3092, 'SIC', 4310, 50, 1, 3775, 5141, 4 FROM dual
UNION ALL SELECT 3093, 'NAICS', 1402, 51, 1, 1419, 512, 2 FROM dual
UNION ALL SELECT 3094, 'SEC', 2459, 3710, 3, 2461, 3713, 4 FROM dual
UNION ALL SELECT 3095, 'SIC', 4002, 6790, 3, 4006, 6799, 4 FROM dual
UNION ALL SELECT 3096, 'SEC', 2403, 3500, 2, 2433, 3590, 3 FROM dual
UNION ALL SELECT 3097, 'SEC', 2780, 8740, 3, 2783, 8744, 4 FROM dual
UNION ALL SELECT 3098, 'NAICS', 1792, 562, 2, 1808, 56292, 4 FROM dual
UNION ALL SELECT 3099, 'SIC', 3344, 3320, 3, 3347, 3324, 4 FROM dual
UNION ALL SELECT 3100, 'SEC', 2552, 4900, 2, 2566, 4960, 3 FROM dual
UNION ALL SELECT 3101, 'SIC', 3124, 2400, 2, 3128, 2421, 4 FROM dual
UNION ALL SELECT 3102, 'NAICS', 2020, 722, 2, 2025, 72232, 4 FROM dual
UNION ALL SELECT 3103, 'NAICS', 1856, 6212, 3, 1857, 621210, 5 FROM dual
UNION ALL SELECT 3104, 'SEC', 2792, 40, 1, 2526, 4500, 2 FROM dual
UNION ALL SELECT 3105, 'NAICS', 1895, 6221, 3, 1896, 622110, 5 FROM dual
UNION ALL SELECT 3106, 'NAICS', 1089, 425, 2, 1092, 42511, 4 FROM dual
UNION ALL SELECT 3107, 'NAICS', 1766, 5616, 3, 1769, 561612, 5 FROM dual
UNION ALL SELECT 3108, 'SIC', 3409, 3490, 3, 3415, 3496, 4 FROM dual
UNION ALL SELECT 3109, 'NAICS', 1026, 4243, 3, 1034, 42434, 4 FROM dual
UNION ALL SELECT 3110, 'NAICS', 1625, 541, 2, 1634, 5412, 3 FROM dual
UNION ALL SELECT 3111, 'NAICS', 1901, 6223, 3, 1903, 62231, 4 FROM dual
UNION ALL SELECT 3112, 'SIC', 4305, 1, 1, 2855, 740, 3 FROM dual
UNION ALL SELECT 3113, 'NAICS', 1625, 541, 2, 1683, 5417, 3 FROM dual
UNION ALL SELECT 3114, 'SEC', 2234, 1600, 2, 2236, 1623, 4 FROM dual
UNION ALL SELECT 3115, 'NAICS', 930, 42, 1, 958, 423420, 5 FROM dual
UNION ALL SELECT 3116, 'NAICS', 1785, 5619, 3, 1786, 561910, 5 FROM dual
UNION ALL SELECT 3117, 'SIC', 4308, 20, 1, 3029, 2084, 4 FROM dual
UNION ALL SELECT 3118, 'SIC', 2847, 700, 2, 2865, 781, 4 FROM dual
UNION ALL SELECT 3119, 'NAICS', 1555, 5251, 3, 1560, 525190, 5 FROM dual
UNION ALL SELECT 3120, 'NAICS', 1431, 5122, 3, 1441, 51229, 4 FROM dual
UNION ALL SELECT 3121, 'NAICS', 2120, 8139, 3, 2124, 81392, 4 FROM dual
UNION ALL SELECT 3122, 'SEC', 2262, 2200, 2, 2269, 2270, 3 FROM dual
UNION ALL SELECT 3123, 'SIC', 2864, 780, 3, 2865, 781, 4 FROM dual
UNION ALL SELECT 3124, 'SIC', 3480, 3600, 2, 3523, 3694, 4 FROM dual
UNION ALL SELECT 3125, 'NAICS', 930, 42, 1, 979, 42362, 4 FROM dual
UNION ALL SELECT 3126, 'NAICS', 930, 42, 1, 1021, 424130, 5 FROM dual
UNION ALL SELECT 3127, 'NAICS', 1470, 519, 2, 1472, 519110, 5 FROM dual
UNION ALL SELECT 3128, 'SIC', 2982, 2000, 2, 3039, 2098, 4 FROM dual
UNION ALL SELECT 3129, 'SIC', 4308, 20, 1, 3572, 3860, 3 FROM dual
UNION ALL SELECT 3130, 'SIC', 2847, 700, 2, 2857, 742, 4 FROM dual
UNION ALL SELECT 3131, 'NAICS', 1943, 711, 2, 1951, 711190, 5 FROM dual
UNION ALL SELECT 3132, 'SIC', 4308, 20, 1, 3048, 2140, 3 FROM dual
UNION ALL SELECT 3133, 'SIC', 3337, 3300, 2, 3344, 3320, 3 FROM dual
UNION ALL SELECT 3134, 'NAICS', 1942, 71, 1, 2000, 71395, 4 FROM dual
UNION ALL SELECT 3135, 'NAICS', 1640, 5413, 3, 1644, 54132, 4 FROM dual
UNION ALL SELECT 3136, 'SEC', 2424, 3570, 3, 2428, 3577, 4 FROM dual
UNION ALL SELECT 3137, 'SIC', 3237, 2860, 3, 3238, 2861, 4 FROM dual
UNION ALL SELECT 3138, 'NAICS', 1419, 512, 2, 1430, 512199, 5 FROM dual
UNION ALL SELECT 3139, 'SIC', 3822, 5400, 2, 3824, 5411, 4 FROM dual
UNION ALL SELECT 3140, 'NAICS', 2135, 92, 1, 2164, 922190, 5 FROM dual
UNION ALL SELECT 3141, 'NAICS', 236, 2381, 3, 252, 23819, 4 FROM dual
UNION ALL SELECT 3142, 'NAICS', 1071, 4248, 3, 1074, 424820, 5 FROM dual
UNION ALL SELECT 3143, 'SEC', 2791, 20, 1, 2475, 3790, 3 FROM dual
UNION ALL SELECT 3144, 'NAICS', 1942, 71, 1, 1999, 713950, 5 FROM dual
UNION ALL SELECT 3145, 'SIC', 4312, 60, 1, 3989, 6550, 3 FROM dual
UNION ALL SELECT 3146, 'SEC', 2703, 6700, 2, 4328, 6795, 4 FROM dual
UNION ALL SELECT 3147, 'SIC', 4308, 20, 1, 3384, 3440, 3 FROM dual
UNION ALL SELECT 3148, 'SIC', 4312, 60, 1, 3949, 6210, 3 FROM dual
UNION ALL SELECT 3149, 'SEC', 2794, 52, 1, 2613, 5330, 3 FROM dual
UNION ALL SELECT 3150, 'SIC', 2993, 2030, 3, 2996, 2034, 4 FROM dual
UNION ALL SELECT 3151, 'SIC', 3715, 5000, 2, 3747, 5075, 4 FROM dual
UNION ALL SELECT 3152, 'SIC', 3124, 2400, 2, 3142, 2451, 4 FROM dual
UNION ALL SELECT 3153, 'SIC', 3496, 3640, 3, 3498, 3643, 4 FROM dual
UNION ALL SELECT 3154, 'SIC', 3886, 5900, 2, 3913, 5993, 4 FROM dual
UNION ALL SELECT 3155, 'NAICS', 2037, 81, 1, 2119, 81341, 4 FROM dual
UNION ALL SELECT 3156, 'SIC', 2887, 1020, 3, 2888, 1021, 4 FROM dual
UNION ALL SELECT 3157, 'NAICS', 1718, 55, 1, 1720, 5511, 3 FROM dual
UNION ALL SELECT 3158, 'NAICS', 1726, 561, 2, 1760, 56151, 4 FROM dual
UNION ALL SELECT 3159, 'SEC', 2610, 5300, 2, 2611, 5310, 3 FROM dual
UNION ALL SELECT 3160, 'SIC', 4308, 20, 1, 3068, 2261, 4 FROM dual
UNION ALL SELECT 3161, 'NAICS', 1920, 624, 2, 1931, 62422, 4 FROM dual
UNION ALL SELECT 3162, 'SIC', 4268, 9400, 2, 4275, 9450, 3 FROM dual
UNION ALL SELECT 3163, 'NAICS', 2, 111, 2, 34, 111336, 5 FROM dual
UNION ALL SELECT 3164, 'SEC', 2789, 10, 1, 2217, 1090, 3 FROM dual
UNION ALL SELECT 3165, 'NAICS', 1850, 62, 1, 1913, 6233, 3 FROM dual
UNION ALL SELECT 3166, 'SEC', 2713, 7300, 2, 2725, 7370, 3 FROM dual
UNION ALL SELECT 3167, 'SIC', 2955, 1700, 2, 2958, 1720, 3 FROM dual
UNION ALL SELECT 3168, 'SIC', 3214, 2800, 2, 3215, 2810, 3 FROM dual
UNION ALL SELECT 3169, 'SEC', 2794, 52, 1, 2632, 5712, 4 FROM dual
UNION ALL SELECT 3170, 'SIC', 3050, 2200, 2, 3080, 2297, 4 FROM dual
UNION ALL SELECT 3171, 'SIC', 4166, 8070, 3, 4167, 8071, 4 FROM dual
UNION ALL SELECT 3172, 'NAICS', 56, 112, 2, 63, 112130, 5 FROM dual
UNION ALL SELECT 3173, 'NAICS', 1431, 5122, 3, 1433, 51221, 4 FROM dual
UNION ALL SELECT 3174, 'NAICS', 982, 4237, 3, 989, 423740, 5 FROM dual
UNION ALL SELECT 3175, 'NAICS', 2020, 722, 2, 2033, 722511, 5 FROM dual
UNION ALL SELECT 3176, 'SIC', 4309, 40, 1, 3661, 4581, 4 FROM dual
UNION ALL SELECT 3177, 'SEC', 2276, 2400, 2, 2277, 2420, 3 FROM dual
UNION ALL SELECT 3178, 'SIC', 4310, 50, 1, 3759, 5093, 4 FROM dual
UNION ALL SELECT 3179, 'SIC', 4308, 20, 1, 3012, 2060, 3 FROM dual
UNION ALL SELECT 3180, 'SIC', 4178, 8200, 2, 4183, 8222, 4 FROM dual
UNION ALL SELECT 3181, 'SIC', 4081, 7500, 2, 4089, 7530, 3 FROM dual
UNION ALL SELECT 3182, 'NAICS', 1954, 71121, 4, 1956, 711212, 5 FROM dual
UNION ALL SELECT 3183, 'NAICS', 2135, 92, 1, 2168, 923110, 5 FROM dual
UNION ALL SELECT 3184, 'SIC', 4313, 70, 1, 4182, 8221, 4 FROM dual
UNION ALL SELECT 3185, 'NAICS', 2, 111, 2, 18, 111199, 5 FROM dual
UNION ALL SELECT 3186, 'NAICS', 1513, 523, 2, 1525, 52321, 4 FROM dual
UNION ALL SELECT 3187, 'SEC', 2371, 3300, 2, 2373, 3312, 4 FROM dual
UNION ALL SELECT 3188, 'SIC', 4309, 40, 1, 3678, 4785, 4 FROM dual
UNION ALL SELECT 3189, 'NAICS', 1, 11, 1, 25, 11131, 4 FROM dual
UNION ALL SELECT 3190, 'SIC', 4308, 20, 1, 3203, 2759, 4 FROM dual
UNION ALL SELECT 3191, 'SIC', 3911, 5990, 3, 3914, 5994, 4 FROM dual
UNION ALL SELECT 3192, 'NAICS', 105, 1133, 3, 106, 113310, 5 FROM dual
UNION ALL SELECT 3193, 'SEC', 2796, 70, 1, 2732, 7381, 4 FROM dual
UNION ALL SELECT 3194, 'SIC', 4309, 40, 1, 3701, 4930, 3 FROM dual
UNION ALL SELECT 3195, 'SIC', 3451, 3560, 3, 3452, 3561, 4 FROM dual
UNION ALL SELECT 3196, 'NAICS', 104, 11321, 4, 103, 113210, 5 FROM dual
UNION ALL SELECT 3197, 'SIC', 4313, 70, 1, 4119, 7829, 4 FROM dual
UNION ALL SELECT 3198, 'NAICS', 1506, 5223, 3, 1510, 52232, 4 FROM dual
UNION ALL SELECT 3199, 'NAICS', 1943, 711, 2, 1948, 71112, 4 FROM dual
UNION ALL SELECT 3200, 'SIC', 3230, 2840, 3, 3231, 2841, 4 FROM dual
UNION ALL SELECT 3201, 'NAICS', 1480, 52, 1, 1542, 524126, 5 FROM dual
UNION ALL SELECT 3202, 'NAICS', 248, 23816, 4, 247, 238160, 5 FROM dual
UNION ALL SELECT 3203, 'SEC', 2649, 6000, 2, 2653, 6029, 4 FROM dual
UNION ALL SELECT 3204, 'NAICS', 930, 42, 1, 1082, 42493, 4 FROM dual
UNION ALL SELECT 3205, 'NAICS', 2135, 92, 1, 2136, 921, 2 FROM dual
UNION ALL SELECT 3206, 'SEC', 2579, 5060, 3, 2582, 5065, 4 FROM dual
UNION ALL SELECT 3207, 'SEC', 2550, 4890, 3, 2551, 4899, 4 FROM dual
UNION ALL SELECT 3208, 'SIC', 3461, 3570, 3, 3466, 3578, 4 FROM dual
UNION ALL SELECT 3209, 'SEC', 2415, 3550, 3, 2417, 3559, 4 FROM dual
UNION ALL SELECT 3210, 'NAICS', 132, 21, 1, 175, 213111, 5 FROM dual
UNION ALL SELECT 3211, 'NAICS', 930, 42, 1, 1009, 423930, 5 FROM dual
UNION ALL SELECT 3212, 'SEC', 2791, 20, 1, 2365, 3260, 3 FROM dual
UNION ALL SELECT 3213, 'SIC', 3480, 3600, 2, 3481, 3610, 3 FROM dual
UNION ALL SELECT 3214, 'NAICS', 1942, 71, 1, 1990, 7139, 3 FROM dual
UNION ALL SELECT 3215, 'SIC', 4297, 9700, 2, 4300, 9720, 3 FROM dual
UNION ALL SELECT 3216, 'NAICS', 2085, 8123, 3, 2089, 81232, 4 FROM dual
UNION ALL SELECT 3217, 'SIC', 3077, 2290, 3, 3079, 2296, 4 FROM dual
UNION ALL SELECT 3218, 'NAICS', 2038, 811, 2, 2045, 81112, 4 FROM dual
UNION ALL SELECT 3219, 'NAICS', 1485, 522, 2, 1504, 522294, 5 FROM dual
UNION ALL SELECT 3220, 'NAICS', 1851, 621, 2, 1854, 621111, 5 FROM dual
UNION ALL SELECT 3221, 'SEC', 2267, 2250, 3, 2268, 2253, 4 FROM dual
UNION ALL SELECT 3222, 'NAICS', 1480, 52, 1, 1551, 524291, 5 FROM dual
UNION ALL SELECT 3223, 'NAICS', 1672, 5416, 3, 1674, 541611, 5 FROM dual
UNION ALL SELECT 3224, 'NAICS', 1813, 611, 2, 1824, 611410, 5 FROM dual
UNION ALL SELECT 3225, 'SIC', 4017, 7200, 2, 4018, 7210, 3 FROM dual
UNION ALL SELECT 3226, 'SIC', 2799, 110, 3, 2800, 111, 4 FROM dual
UNION ALL SELECT 3227, 'NAICS', 1480, 52, 1, 1536, 524, 2 FROM dual
UNION ALL SELECT 3228, 'NAICS', 1942, 71, 1, 1996, 71393, 4 FROM dual
UNION ALL SELECT 3229, 'SIC', 3636, 4400, 2, 3648, 4489, 4 FROM dual
UNION ALL SELECT 3230, 'SIC', 4310, 50, 1, 3777, 5143, 4 FROM dual
UNION ALL SELECT 3231, 'SEC', 2408, 3530, 3, 2411, 3533, 4 FROM dual
UNION ALL SELECT 3232, 'SEC', 2792, 40, 1, 2540, 4810, 3 FROM dual
UNION ALL SELECT 3233, 'SEC', 2677, 6300, 2, 2678, 6310, 3 FROM dual
UNION ALL SELECT 3234, 'SIC', 3958, 6300, 2, 3965, 6331, 4 FROM dual
UNION ALL SELECT 3235, 'SEC', 2791, 20, 1, 2340, 2950, 3 FROM dual
UNION ALL SELECT 3236, 'SIC', 4308, 20, 1, 2992, 2026, 4 FROM dual
UNION ALL SELECT 3237, 'NAICS', 1480, 52, 1, 1546, 52413, 4 FROM dual
UNION ALL SELECT 3238, 'NAICS', 1905, 6231, 3, 1907, 62311, 4 FROM dual
UNION ALL SELECT 3239, 'SIC', 3190, 2700, 2, 3203, 2759, 4 FROM dual
UNION ALL SELECT 3240, 'SIC', 3854, 5600, 2, 3865, 5660, 3 FROM dual
UNION ALL SELECT 3241, 'SIC', 3992, 6700, 2, 4006, 6799, 4 FROM dual
UNION ALL SELECT 3242, 'SIC', 3715, 5000, 2, 3751, 5083, 4 FROM dual
UNION ALL SELECT 3243, 'SIC', 2982, 2000, 2, 2991, 2024, 4 FROM dual
UNION ALL SELECT 3244, 'NAICS', 1673, 54161, 4, 1678, 541618, 5 FROM dual
UNION ALL SELECT 3245, 'NAICS', 2007, 72111, 4, 2006, 721110, 5 FROM dual
UNION ALL SELECT 3246, 'SIC', 4049, 7330, 3, 4053, 7336, 4 FROM dual
UNION ALL SELECT 3247, 'SEC', 2794, 52, 1, 2639, 5812, 4 FROM dual
UNION ALL SELECT 3248, 'SIC', 2982, 2000, 2, 3003, 2044, 4 FROM dual
UNION ALL SELECT 3249, 'SIC', 3008, 2050, 3, 3010, 2052, 4 FROM dual
UNION ALL SELECT 3250, 'SIC', 3330, 3290, 3, 3336, 3299, 4 FROM dual
UNION ALL SELECT 3251, 'SIC', 4017, 7200, 2, 4038, 7291, 4 FROM dual
UNION ALL SELECT 3252, 'NAICS', 1537, 5241, 3, 1538, 52411, 4 FROM dual
UNION ALL SELECT 3253, 'SIC', 4230, 8730, 3, 4231, 8731, 4 FROM dual
UNION ALL SELECT 3254, 'SIC', 3553, 3810, 3, 3554, 3812, 4 FROM dual
UNION ALL SELECT 3255, 'SIC', 4040, 7300, 2, 4069, 7374, 4 FROM dual
UNION ALL SELECT 3256, 'SEC', 2791, 20, 1, 2315, 2770, 3 FROM dual
UNION ALL SELECT 3257, 'NAICS', 931, 423, 2, 957, 42341, 4 FROM dual
UNION ALL SELECT 3258, 'SEC', 2384, 3400, 2, 2392, 3442, 4 FROM dual
UNION ALL SELECT 3259, 'NAICS', 235, 238, 2, 239, 238120, 5 FROM dual
UNION ALL SELECT 3260, 'NAICS', 260, 2383, 3, 269, 238350, 5 FROM dual
UNION ALL SELECT 3261, 'NAICS', 2003, 72, 1, 2018, 721310, 5 FROM dual
UNION ALL SELECT 3262, 'SIC', 4308, 20, 1, 3082, 2299, 4 FROM dual
UNION ALL SELECT 3263, 'NAICS', 1402, 51, 1, 1478, 519190, 5 FROM dual
UNION ALL SELECT 3264, 'SEC', 2796, 70, 1, 2723, 7361, 4 FROM dual
UNION ALL SELECT 3265, 'SEC', 2434, 3600, 2, 2435, 3610, 3 FROM dual
UNION ALL SELECT 3266, 'SIC', 2825, 200, 2, 2835, 251, 4 FROM dual
UNION ALL SELECT 3267, 'NAICS', 2185, 92511, 4, 2184, 925110, 5 FROM dual
UNION ALL SELECT 3268, 'SIC', 3148, 2500, 2, 3149, 2510, 3 FROM dual
UNION ALL SELECT 3269, 'NAICS', 2, 111, 2, 52, 11199, 4 FROM dual
UNION ALL SELECT 3270, 'NAICS', 1938, 62431, 4, 1937, 624310, 5 FROM dual
UNION ALL SELECT 3271, 'SIC', 3846, 5550, 3, 3847, 5551, 4 FROM dual
UNION ALL SELECT 3272, 'SIC', 3041, 2100, 2, 3047, 2131, 4 FROM dual
UNION ALL SELECT 3273, 'NAICS', 260, 2383, 3, 267, 238340, 5 FROM dual
UNION ALL SELECT 3274, 'NAICS', 1569, 53, 1, 1586, 531312, 5 FROM dual
UNION ALL SELECT 3275, 'SIC', 3167, 2600, 2, 3176, 2653, 4 FROM dual
UNION ALL SELECT 3276, 'NAICS', 23, 1113, 3, 25, 11131, 4 FROM dual
UNION ALL SELECT 3277, 'NAICS', 959, 42342, 4, 958, 423420, 5 FROM dual
UNION ALL SELECT 3278, 'SIC', 4109, 7690, 3, 4111, 7694, 4 FROM dual
UNION ALL SELECT 3279, 'NAICS', 1726, 561, 2, 1791, 56199, 4 FROM dual
UNION ALL SELECT 3280, 'NAICS', 167, 21239, 4, 169, 212392, 5 FROM dual
UNION ALL SELECT 3281, 'NAICS', 1628, 54111, 4, 1627, 541110, 5 FROM dual
UNION ALL SELECT 3282, 'SEC', 2568, 5000, 2, 2583, 5070, 3 FROM dual
UNION ALL SELECT 3283, 'NAICS', 1813, 611, 2, 1834, 611513, 5 FROM dual
UNION ALL SELECT 3284, 'SEC', 2774, 8700, 2, 2780, 8740, 3 FROM dual
UNION ALL SELECT 3285, 'SIC', 4311, 52, 1, 3845, 5541, 4 FROM dual
UNION ALL SELECT 3286, 'SIC', 3301, 3200, 2, 3335, 3297, 4 FROM dual
UNION ALL SELECT 3287, 'NAICS', 1495, 5222, 3, 1505, 522298, 5 FROM dual
UNION ALL SELECT 3288, 'SIC', 3124, 2400, 2, 3140, 2449, 4 FROM dual
UNION ALL SELECT 3289, 'NAICS', 930, 42, 1, 1083, 424940, 5 FROM dual
UNION ALL SELECT 3290, 'SIC', 4308, 20, 1, 3027, 2082, 4 FROM dual
UNION ALL SELECT 3291, 'NAICS', 1562, 5259, 3, 1564, 52591, 4 FROM dual
UNION ALL SELECT 3292, 'NAICS', 2137, 9211, 3, 2144, 921140, 5 FROM dual
UNION ALL SELECT 3293, 'SIC', 4313, 70, 1, 4153, 8040, 3 FROM dual
UNION ALL SELECT 3294, 'SIC', 3344, 3320, 3, 3345, 3321, 4 FROM dual
UNION ALL SELECT 3295, 'NAICS', 930, 42, 1, 984, 42371, 4 FROM dual
UNION ALL SELECT 3296, 'SIC', 4308, 20, 1, 3598, 3995, 4 FROM dual
UNION ALL SELECT 3297, 'NAICS', 1612, 5324, 3, 1613, 53241, 4 FROM dual
UNION ALL SELECT 3298, 'SIC', 3526, 3700, 2, 3538, 3731, 4 FROM dual
UNION ALL SELECT 3299, 'SIC', 3372, 3410, 3, 3374, 3412, 4 FROM dual
UNION ALL SELECT 3300, 'SEC', 2319, 2800, 2, 2320, 2810, 3 FROM dual
UNION ALL SELECT 3301, 'NAICS', 1554, 525, 2, 1561, 52519, 4 FROM dual
UNION ALL SELECT 3302, 'NAICS', 1792, 562, 2, 1805, 562910, 5 FROM dual
UNION ALL SELECT 3303, 'SEC', 2628, 5660, 3, 2629, 5661, 4 FROM dual
UNION ALL SELECT 3304, 'SIC', 3419, 3500, 2, 3430, 3534, 4 FROM dual
UNION ALL SELECT 3305, 'SIC', 2869, 810, 3, 2870, 811, 4 FROM dual
UNION ALL SELECT 3306, 'SIC', 2845, 290, 3, 2846, 291, 4 FROM dual
UNION ALL SELECT 3307, 'SEC', 2791, 20, 1, 2435, 3610, 3 FROM dual
UNION ALL SELECT 3308, 'NAICS', 108, 114, 2, 112, 114112, 5 FROM dual
UNION ALL SELECT 3309, 'NAICS', 1979, 713, 2, 1992, 71391, 4 FROM dual
UNION ALL SELECT 3310, 'SEC', 2240, 2000, 2, 2251, 2052, 4 FROM dual
UNION ALL SELECT 3311, 'SIC', 4305, 1, 1, 2844, 279, 4 FROM dual
UNION ALL SELECT 3312, 'NAICS', 2120, 8139, 3, 2121, 813910, 5 FROM dual
UNION ALL SELECT 3313, 'SIC', 2825, 200, 2, 2834, 250, 3 FROM dual
UNION ALL SELECT 3314, 'SIC', 4313, 70, 1, 4091, 7533, 4 FROM dual
UNION ALL SELECT 3315, 'NAICS', 235, 238, 2, 266, 23833, 4 FROM dual
UNION ALL SELECT 3316, 'SIC', 3807, 5230, 3, 3808, 5231, 4 FROM dual
UNION ALL SELECT 3317, 'NAICS', 2150, 922, 2, 2159, 92214, 4 FROM dual
UNION ALL SELECT 3318, 'SIC', 3552, 3800, 2, 3559, 3824, 4 FROM dual
UNION ALL SELECT 3319, 'SIC', 3083, 2300, 2, 3116, 2391, 4 FROM dual
UNION ALL SELECT 3320, 'SIC', 3434, 3540, 3, 3438, 3544, 4 FROM dual
UNION ALL SELECT 3321, 'NAICS', 1003, 42386, 4, 1002, 423860, 5 FROM dual
UNION ALL SELECT 3322, 'NAICS', 205, 23, 1, 234, 23799, 4 FROM dual
UNION ALL SELECT 3323, 'SIC', 2955, 1700, 2, 2968, 1752, 4 FROM dual
UNION ALL SELECT 3324, 'SIC', 3083, 2300, 2, 3088, 2322, 4 FROM dual
UNION ALL SELECT 3325, 'SIC', 4125, 7900, 2, 4139, 7990, 3 FROM dual
UNION ALL SELECT 3326, 'NAICS', 56, 112, 2, 87, 112512, 5 FROM dual
UNION ALL SELECT 3327, 'NAICS', 156, 2123, 3, 166, 212325, 5 FROM dual
UNION ALL SELECT 3328, 'SIC', 3050, 2200, 2, 3063, 2254, 4 FROM dual
UNION ALL SELECT 3329, 'SIC', 3762, 5100, 2, 3768, 5122, 4 FROM dual
UNION ALL SELECT 3330, 'NAICS', 1480, 52, 1, 1541, 52412, 4 FROM dual
UNION ALL SELECT 3331, 'NAICS', 1402, 51, 1, 1451, 51521, 4 FROM dual
UNION ALL SELECT 3332, 'NAICS', 1625, 541, 2, 1645, 541330, 5 FROM dual
UNION ALL SELECT 3333, 'SIC', 3576, 3900, 2, 3580, 3915, 4 FROM dual
UNION ALL SELECT 3334, 'NAICS', 1452, 517, 2, 1465, 517919, 5 FROM dual
UNION ALL SELECT 3335, 'SIC', 4308, 20, 1, 3491, 3632, 4 FROM dual
UNION ALL SELECT 3336, 'NAICS', 1619, 53249, 4, 1618, 532490, 5 FROM dual
UNION ALL SELECT 3337, 'SEC', 2795, 60, 1, 2663, 6141, 4 FROM dual
UNION ALL SELECT 3338, 'SIC', 4176, 8110, 3, 4177, 8111, 4 FROM dual
UNION ALL SELECT 3339, 'NAICS', 1624, 54, 1, 1663, 54143, 4 FROM dual
UNION ALL SELECT 3340, 'SIC', 3050, 2200, 2, 3051, 2210, 3 FROM dual
UNION ALL SELECT 3341, 'SEC', 2794, 52, 1, 2647, 5961, 4 FROM dual
UNION ALL SELECT 3342, 'SIC', 4017, 7200, 2, 4036, 7261, 4 FROM dual
UNION ALL SELECT 3343, 'NAICS', 2, 111, 2, 39, 111419, 5 FROM dual
UNION ALL SELECT 3344, 'NAICS', 1725, 56, 1, 1756, 561492, 5 FROM dual
UNION ALL SELECT 3345, 'NAICS', 2093, 8129, 3, 2099, 812930, 5 FROM dual
UNION ALL SELECT 3346, 'SEC', 2640, 5900, 2, 2644, 5944, 4 FROM dual
UNION ALL SELECT 3347, 'SEC', 2795, 60, 1, 2657, 6090, 3 FROM dual
UNION ALL SELECT 3348, 'SEC', 2796, 70, 1, 2754, 8010, 3 FROM dual
UNION ALL SELECT 3349, 'SIC', 4311, 52, 1, 3865, 5660, 3 FROM dual
UNION ALL SELECT 3350, 'SIC', 4291, 9640, 3, 4292, 9641, 4 FROM dual
UNION ALL SELECT 3351, 'NAICS', 2103, 813, 2, 2125, 813930, 5 FROM dual
UNION ALL SELECT 3352, 'NAICS', 2039, 8111, 3, 2042, 811112, 5 FROM dual
UNION ALL SELECT 3353, 'NAICS', 1958, 7113, 3, 1959, 711310, 5 FROM dual
UNION ALL SELECT 3354, 'NAICS', 25, 11131, 4, 24, 111310, 5 FROM dual
UNION ALL SELECT 3355, 'SEC', 2292, 2600, 2, 2300, 2670, 3 FROM dual
UNION ALL SELECT 3356, 'SEC', 2791, 20, 1, 2465, 3720, 3 FROM dual
UNION ALL SELECT 3357, 'SIC', 4308, 20, 1, 3492, 3633, 4 FROM dual
UNION ALL SELECT 3358, 'SIC', 4309, 40, 1, 3675, 4741, 4 FROM dual
UNION ALL SELECT 3359, 'SIC', 3749, 5080, 3, 3754, 5087, 4 FROM dual
UNION ALL SELECT 3360, 'SEC', 2458, 3700, 2, 2460, 3711, 4 FROM dual
UNION ALL SELECT 3361, 'SIC', 3337, 3300, 2, 3346, 3322, 4 FROM dual
UNION ALL SELECT 3362, 'SEC', 2793, 50, 1, 2584, 5072, 4 FROM dual
UNION ALL SELECT 3363, 'SIC', 4313, 70, 1, 4095, 7538, 4 FROM dual
UNION ALL SELECT 3364, 'SIC', 3059, 2250, 3, 3066, 2259, 4 FROM dual
UNION ALL SELECT 3365, 'SIC', 3626, 4220, 3, 3629, 4225, 4 FROM dual
UNION ALL SELECT 3366, 'SIC', 3837, 5500, 2, 3848, 5560, 3 FROM dual
UNION ALL SELECT 3367, 'NAICS', 1402, 51, 1, 1449, 5152, 3 FROM dual
UNION ALL SELECT 3368, 'NAICS', 1943, 711, 2, 1950, 71113, 4 FROM dual
UNION ALL SELECT 3369, 'SIC', 3355, 3350, 3, 3359, 3355, 4 FROM dual
UNION ALL SELECT 3370, 'SIC', 3496, 3640, 3, 3497, 3641, 4 FROM dual
UNION ALL SELECT 3371, 'SIC', 3654, 4500, 2, 3661, 4581, 4 FROM dual
UNION ALL SELECT 3372, 'SIC', 4314, 90, 1, 4267, 9311, 4 FROM dual
UNION ALL SELECT 3373, 'SIC', 3605, 4100, 2, 3616, 4150, 3 FROM dual
UNION ALL SELECT 3374, 'SIC', 3026, 2080, 3, 3027, 2082, 4 FROM dual
UNION ALL SELECT 3375, 'SIC', 3576, 3900, 2, 3583, 3940, 3 FROM dual
UNION ALL SELECT 3376, 'SIC', 4075, 7380, 3, 4076, 7381, 4 FROM dual
UNION ALL SELECT 3377, 'SEC', 2623, 5600, 2, 2627, 5651, 4 FROM dual
UNION ALL SELECT 3378, 'NAICS', 1631, 54119, 4, 1633, 541199, 5 FROM dual
UNION ALL SELECT 3379, 'SIC', 4312, 60, 1, 3924, 6029, 4 FROM dual
UNION ALL SELECT 3380, 'NAICS', 988, 42373, 4, 987, 423730, 5 FROM dual
UNION ALL SELECT 3381, 'NAICS', 2167, 9231, 3, 2170, 923120, 5 FROM dual
UNION ALL SELECT 3382, 'SEC', 2292, 2600, 2, 2293, 2610, 3 FROM dual
UNION ALL SELECT 3383, 'SEC', 2240, 2000, 2, 2241, 2010, 3 FROM dual
UNION ALL SELECT 3384, 'SIC', 3426, 3530, 3, 3429, 3533, 4 FROM dual
UNION ALL SELECT 3385, 'SEC', 2458, 3700, 2, 2472, 3750, 3 FROM dual
UNION ALL SELECT 3386, 'SEC', 2591, 5100, 2, 2601, 5171, 4 FROM dual
UNION ALL SELECT 3387, 'NAICS', 156, 2123, 3, 161, 212319, 5 FROM dual
UNION ALL SELECT 3388, 'NAICS', 1485, 522, 2, 1497, 52221, 4 FROM dual
UNION ALL SELECT 3389, 'SEC', 2796, 70, 1, 2766, 8100, 2 FROM dual
UNION ALL SELECT 3390, 'SIC', 3552, 3800, 2, 3557, 3822, 4 FROM dual
UNION ALL SELECT 3391, 'SIC', 3083, 2300, 2, 3104, 2361, 4 FROM dual
UNION ALL SELECT 3392, 'SEC', 2254, 2080, 3, 2256, 2086, 4 FROM dual
UNION ALL SELECT 3393, 'NAICS', 1554, 525, 2, 1556, 525110, 5 FROM dual
UNION ALL SELECT 3394, 'SEC', 2795, 60, 1, 2708, 6799, 4 FROM dual
UNION ALL SELECT 3395, 'SIC', 3693, 4900, 2, 3700, 4925, 4 FROM dual
UNION ALL SELECT 3396, 'SIC', 4309, 40, 1, 3667, 4700, 2 FROM dual
UNION ALL SELECT 3397, 'NAICS', 1913, 6233, 3, 1914, 62331, 4 FROM dual
UNION ALL SELECT 3398, 'NAICS', 1850, 62, 1, 1853, 62111, 4 FROM dual
UNION ALL SELECT 3399, 'SEC', 2659, 6100, 2, 2664, 6150, 3 FROM dual
UNION ALL SELECT 3400, 'NAICS', 975, 4236, 3, 977, 42361, 4 FROM dual
UNION ALL SELECT 3401, 'SIC', 2955, 1700, 2, 2979, 1795, 4 FROM dual
UNION ALL SELECT 3402, 'SIC', 2955, 1700, 2, 2978, 1794, 4 FROM dual
UNION ALL SELECT 3403, 'SIC', 3434, 3540, 3, 3440, 3546, 4 FROM dual
UNION ALL SELECT 3404, 'NAICS', 1015, 424, 2, 1076, 4249, 3 FROM dual
UNION ALL SELECT 3405, 'SIC', 4308, 20, 1, 3026, 2080, 3 FROM dual
UNION ALL SELECT 3406, 'SIC', 4308, 20, 1, 3190, 2700, 2 FROM dual
UNION ALL SELECT 3407, 'SIC', 3215, 2810, 3, 3218, 2816, 4 FROM dual
UNION ALL SELECT 3408, 'SEC', 2792, 40, 1, 2536, 4700, 2 FROM dual
UNION ALL SELECT 3409, 'SIC', 3791, 5170, 3, 3793, 5172, 4 FROM dual
UNION ALL SELECT 3410, 'SIC', 3050, 2200, 2, 3077, 2290, 3 FROM dual
UNION ALL SELECT 3411, 'NAICS', 1946, 71111, 4, 1945, 711110, 5 FROM dual
UNION ALL SELECT 3412, 'SEC', 2231, 1530, 3, 2232, 1531, 4 FROM dual
UNION ALL SELECT 3413, 'SIC', 3419, 3500, 2, 3435, 3541, 4 FROM dual
UNION ALL SELECT 3414, 'SEC', 2476, 3800, 2, 2495, 3851, 4 FROM dual
UNION ALL SELECT 3415, 'SIC', 3108, 2380, 3, 3114, 2389, 4 FROM dual
UNION ALL SELECT 3416, 'SIC', 3784, 5150, 3, 3787, 5159, 4 FROM dual
UNION ALL SELECT 3417, 'NAICS', 1741, 5614, 3, 1754, 56149, 4 FROM dual
UNION ALL SELECT 3418, 'NAICS', 2188, 926, 2, 2192, 926120, 5 FROM dual
UNION ALL SELECT 3419, 'NAICS', 930, 42, 1, 938, 42313, 4 FROM dual
UNION ALL SELECT 3420, 'NAICS', 930, 42, 1, 1063, 42461, 4 FROM dual
UNION ALL SELECT 3421, 'SIC', 3468, 3580, 3, 3471, 3585, 4 FROM dual
UNION ALL SELECT 3422, 'SIC', 3696, 4920, 3, 3698, 4923, 4 FROM dual
UNION ALL SELECT 3423, 'NAICS', 1851, 621, 2, 1872, 621410, 5 FROM dual
UNION ALL SELECT 3424, 'SEC', 2534, 4600, 2, 2535, 4610, 3 FROM dual
UNION ALL SELECT 3425, 'SIC', 4307, 15, 1, 2954, 1629, 4 FROM dual
UNION ALL SELECT 3426, 'NAICS', 2037, 81, 1, 2109, 813211, 5 FROM dual
UNION ALL SELECT 3427, 'NAICS', 2071, 812, 2, 2093, 8129, 3 FROM dual
UNION ALL SELECT 3428, 'SIC', 3409, 3490, 3, 3412, 3493, 4 FROM dual
UNION ALL SELECT 3429, 'NAICS', 1591, 532, 2, 1603, 532230, 5 FROM dual
UNION ALL SELECT 3430, 'NAICS', 56, 112, 2, 70, 11231, 4 FROM dual
UNION ALL SELECT 3431, 'NAICS', 228, 23721, 4, 227, 237210, 5 FROM dual
UNION ALL SELECT 3432, 'SIC', 3749, 5080, 3, 3751, 5083, 4 FROM dual
UNION ALL SELECT 3433, 'NAICS', 1526, 5239, 3, 1528, 52391, 4 FROM dual
UNION ALL SELECT 3434, 'SEC', 2355, 3200, 2, 2359, 3221, 4 FROM dual
UNION ALL SELECT 3435, 'SIC', 3481, 3610, 3, 3483, 3613, 4 FROM dual
UNION ALL SELECT 3436, 'SIC', 4308, 20, 1, 3410, 3491, 4 FROM dual
UNION ALL SELECT 3437, 'NAICS', 1452, 517, 2, 1460, 517410, 5 FROM dual
UNION ALL SELECT 3438, 'NAICS', 1625, 541, 2, 1664, 541490, 5 FROM dual
UNION ALL SELECT 3439, 'SIC', 4310, 50, 1, 3781, 5147, 4 FROM dual
UNION ALL SELECT 3440, 'SEC', 2791, 20, 1, 2379, 3341, 4 FROM dual
UNION ALL SELECT 3441, 'NAICS', 1850, 62, 1, 1934, 624230, 5 FROM dual
UNION ALL SELECT 3442, 'SEC', 2794, 52, 1, 2622, 5531, 4 FROM dual
UNION ALL SELECT 3443, 'NAICS', 930, 42, 1, 1091, 425110, 5 FROM dual
UNION ALL SELECT 3444, 'NAICS', 1402, 51, 1, 1479, 51919, 4 FROM dual
UNION ALL SELECT 3445, 'NAICS', 1555, 5251, 3, 1558, 525120, 5 FROM dual
UNION ALL SELECT 3446, 'NAICS', 1673, 54161, 4, 1674, 541611, 5 FROM dual
UNION ALL SELECT 3447, 'NAICS', 205, 23, 1, 251, 238190, 5 FROM dual
UNION ALL SELECT 3448, 'NAICS', 1980, 7131, 3, 1983, 713120, 5 FROM dual
UNION ALL SELECT 3449, 'NAICS', 2137, 9211, 3, 2139, 92111, 4 FROM dual
UNION ALL SELECT 3450, 'NAICS', 132, 21, 1, 161, 212319, 5 FROM dual
UNION ALL SELECT 3451, 'NAICS', 1569, 53, 1, 1615, 532412, 5 FROM dual
UNION ALL SELECT 3452, 'SEC', 2476, 3800, 2, 2491, 3843, 4 FROM dual
UNION ALL SELECT 3453, 'NAICS', 1015, 424, 2, 1021, 424130, 5 FROM dual
UNION ALL SELECT 3454, 'NAICS', 2071, 812, 2, 2080, 8122, 3 FROM dual
UNION ALL SELECT 3455, 'SEC', 2791, 20, 1, 2403, 3500, 2 FROM dual
UNION ALL SELECT 3456, 'NAICS', 98, 113, 2, 104, 11321, 4 FROM dual
UNION ALL SELECT 3457, 'SEC', 2659, 6100, 2, 2660, 6110, 3 FROM dual
UNION ALL SELECT 3458, 'SIC', 4313, 70, 1, 4071, 7376, 4 FROM dual
UNION ALL SELECT 3459, 'NAICS', 1850, 62, 1, 1887, 62161, 4 FROM dual
UNION ALL SELECT 3460, 'SIC', 3693, 4900, 2, 3707, 4950, 3 FROM dual
UNION ALL SELECT 3461, 'NAICS', 1506, 5223, 3, 1507, 522310, 5 FROM dual
UNION ALL SELECT 3462, 'SIC', 3164, 2590, 3, 3166, 2599, 4 FROM dual
UNION ALL SELECT 3463, 'SIC', 3696, 4920, 3, 3699, 4924, 4 FROM dual
UNION ALL SELECT 3464, 'NAICS', 182, 2211, 3, 191, 221118, 5 FROM dual
UNION ALL SELECT 3465, 'NAICS', 1485, 522, 2, 1493, 522190, 5 FROM dual
UNION ALL SELECT 3466, 'SIC', 4314, 90, 1, 4260, 9221, 4 FROM dual
UNION ALL SELECT 3467, 'SIC', 3893, 5940, 3, 3894, 5941, 4 FROM dual
UNION ALL SELECT 3468, 'SIC', 3886, 5900, 2, 3893, 5940, 3 FROM dual
UNION ALL SELECT 3469, 'NAICS', 1625, 541, 2, 1637, 541213, 5 FROM dual
UNION ALL SELECT 3470, 'NAICS', 1888, 6219, 3, 1891, 62199, 4 FROM dual
UNION ALL SELECT 3471, 'NAICS', 1659, 54141, 4, 1658, 541410, 5 FROM dual
UNION ALL SELECT 3472, 'SEC', 2238, 1730, 3, 2239, 1731, 4 FROM dual
UNION ALL SELECT 3473, 'SEC', 2791, 20, 1, 2250, 2050, 3 FROM dual
UNION ALL SELECT 3474, 'SIC', 3945, 6160, 3, 3947, 6163, 4 FROM dual
UNION ALL SELECT 3475, 'NAICS', 1402, 51, 1, 1437, 51223, 4 FROM dual
UNION ALL SELECT 3476, 'SEC', 2793, 50, 1, 2589, 5094, 4 FROM dual
UNION ALL SELECT 3477, 'NAICS', 195, 2212, 3, 196, 221210, 5 FROM dual
UNION ALL SELECT 3478, 'NAICS', 207, 2361, 3, 212, 236118, 5 FROM dual
UNION ALL SELECT 3479, 'SIC', 3419, 3500, 2, 3459, 3568, 4 FROM dual
UNION ALL SELECT 3480, 'SIC', 4309, 40, 1, 3671, 4729, 4 FROM dual
UNION ALL SELECT 3481, 'NAICS', 1624, 54, 1, 1659, 54141, 4 FROM dual
UNION ALL SELECT 3482, 'SIC', 4308, 20, 1, 3242, 2873, 4 FROM dual
UNION ALL SELECT 3483, 'SIC', 4311, 52, 1, 3818, 5330, 3 FROM dual
UNION ALL SELECT 3484, 'SIC', 4313, 70, 1, 4226, 8712, 4 FROM dual
UNION ALL SELECT 3485, 'NAICS', 1850, 62, 1, 1919, 62399, 4 FROM dual
UNION ALL SELECT 3486, 'NAICS', 23, 1113, 3, 31, 111333, 5 FROM dual
UNION ALL SELECT 3487, 'NAICS', 946, 4233, 3, 954, 42339, 4 FROM dual
UNION ALL SELECT 3488, 'NAICS', 2135, 92, 1, 2185, 92511, 4 FROM dual
UNION ALL SELECT 3489, 'SIC', 3576, 3900, 2, 3591, 3955, 4 FROM dual
UNION ALL SELECT 3490, 'SEC', 2618, 5410, 3, 4319, 5412, 4 FROM dual
UNION ALL SELECT 3491, 'SIC', 4308, 20, 1, 3408, 3489, 4 FROM dual
UNION ALL SELECT 3492, 'SIC', 4305, 1, 1, 2826, 210, 3 FROM dual
UNION ALL SELECT 3493, 'NAICS', 235, 238, 2, 274, 238910, 5 FROM dual
UNION ALL SELECT 3494, 'NAICS', 2103, 813, 2, 2108, 81321, 4 FROM dual
UNION ALL SELECT 3495, 'NAICS', 1965, 71141, 4, 1964, 711410, 5 FROM dual
UNION ALL SELECT 3496, 'NAICS', 2037, 81, 1, 2072, 8121, 3 FROM dual
UNION ALL SELECT 3497, 'NAICS', 1726, 561, 2, 1769, 561612, 5 FROM dual
UNION ALL SELECT 3498, 'SEC', 2605, 5200, 2, 2608, 5270, 3 FROM dual
UNION ALL SELECT 3499, 'NAICS', 1076, 4249, 3, 1081, 424930, 5 FROM dual
UNION ALL SELECT 3500, 'NAICS', 1798, 5622, 3, 1803, 562219, 5 FROM dual
UNION ALL SELECT 3501, 'SIC', 3077, 2290, 3, 3078, 2295, 4 FROM dual
UNION ALL SELECT 3502, 'NAICS', 1726, 561, 2, 1778, 56172, 4 FROM dual
UNION ALL SELECT 3503, 'SIC', 3050, 2200, 2, 3059, 2250, 3 FROM dual
UNION ALL SELECT 3504, 'NAICS', 98, 113, 2, 100, 113110, 5 FROM dual
UNION ALL SELECT 3505, 'SEC', 2458, 3700, 2, 2470, 3740, 3 FROM dual
UNION ALL SELECT 3506, 'SIC', 3595, 3990, 3, 3597, 3993, 4 FROM dual
UNION ALL SELECT 3507, 'NAICS', 1015, 424, 2, 1079, 424920, 5 FROM dual
UNION ALL SELECT 3508, 'SIC', 3380, 3430, 3, 3383, 3433, 4 FROM dual
UNION ALL SELECT 3509, 'NAICS', 1672, 5416, 3, 1678, 541618, 5 FROM dual
UNION ALL SELECT 3510, 'NAICS', 1813, 611, 2, 1818, 611210, 5 FROM dual
UNION ALL SELECT 3511, 'SEC', 2791, 20, 1, 2386, 3411, 4 FROM dual
UNION ALL SELECT 3512, 'SEC', 2591, 5100, 2, 2596, 5140, 3 FROM dual
UNION ALL SELECT 3513, 'NAICS', 1624, 54, 1, 1680, 54162, 4 FROM dual
UNION ALL SELECT 3514, 'SEC', 2791, 20, 1, 2469, 3730, 3 FROM dual
UNION ALL SELECT 3515, 'NAICS', 205, 23, 1, 276, 238990, 5 FROM dual
UNION ALL SELECT 3516, 'SEC', 2774, 8700, 2, 2776, 8711, 4 FROM dual
UNION ALL SELECT 3517, 'SIC', 2982, 2000, 2, 3007, 2048, 4 FROM dual
UNION ALL SELECT 3518, 'NAICS', 1625, 541, 2, 1689, 5418, 3 FROM dual
UNION ALL SELECT 3519, 'SEC', 2342, 3000, 2, 2345, 3020, 3 FROM dual
UNION ALL SELECT 3520, 'NAICS', 1718, 55, 1, 1722, 551111, 5 FROM dual
UNION ALL SELECT 3521, 'SIC', 2825, 200, 2, 2842, 272, 4 FROM dual
UNION ALL SELECT 3522, 'NAICS', 132, 21, 1, 176, 213112, 5 FROM dual
UNION ALL SELECT 3523, 'NAICS', 1726, 561, 2, 1738, 56132, 4 FROM dual
UNION ALL SELECT 3524, 'SEC', 2791, 20, 1, 2347, 3050, 3 FROM dual
UNION ALL SELECT 3525, 'SIC', 4313, 70, 1, 4227, 8713, 4 FROM dual
UNION ALL SELECT 3526, 'NAICS', 1090, 4251, 3, 1093, 425120, 5 FROM dual
UNION ALL SELECT 3527, 'NAICS', 2003, 72, 1, 2015, 721211, 5 FROM dual
UNION ALL SELECT 3528, 'SIC', 2825, 200, 2, 2828, 212, 4 FROM dual
UNION ALL SELECT 3529, 'SEC', 2791, 20, 1, 2429, 3578, 4 FROM dual
UNION ALL SELECT 3530, 'SEC', 2568, 5000, 2, 2580, 5063, 4 FROM dual
UNION ALL SELECT 3531, 'NAICS', 930, 42, 1, 987, 423730, 5 FROM dual
UNION ALL SELECT 3532, 'NAICS', 1942, 71, 1, 1981, 713110, 5 FROM dual
UNION ALL SELECT 3533, 'SEC', 2795, 60, 1, 2685, 6350, 3 FROM dual
UNION ALL SELECT 3534, 'SIC', 3083, 2300, 2, 3096, 2337, 4 FROM dual
UNION ALL SELECT 3535, 'SIC', 3419, 3500, 2, 3457, 3566, 4 FROM dual
UNION ALL SELECT 3536, 'SIC', 3948, 6200, 2, 3949, 6210, 3 FROM dual
UNION ALL SELECT 3537, 'SEC', 2751, 7990, 3, 2752, 7997, 4 FROM dual
UNION ALL SELECT 3538, 'SIC', 4313, 70, 1, 4032, 7241, 4 FROM dual
UNION ALL SELECT 3539, 'SIC', 4146, 8000, 2, 4148, 8011, 4 FROM dual
UNION ALL SELECT 3540, 'NAICS', 235, 238, 2, 264, 23832, 4 FROM dual
UNION ALL SELECT 3541, 'NAICS', 1076, 4249, 3, 1078, 42491, 4 FROM dual
UNION ALL SELECT 3542, 'SIC', 4308, 20, 1, 3086, 2320, 3 FROM dual
UNION ALL SELECT 3543, 'SIC', 4308, 20, 1, 3335, 3297, 4 FROM dual
UNION ALL SELECT 3544, 'SIC', 4314, 90, 1, 4269, 9410, 3 FROM dual
UNION ALL SELECT 3545, 'SIC', 4308, 20, 1, 3510, 3669, 4 FROM dual
UNION ALL SELECT 3546, 'SIC', 3426, 3530, 3, 3433, 3537, 4 FROM dual
UNION ALL SELECT 3547, 'SIC', 3083, 2300, 2, 3107, 2371, 4 FROM dual
UNION ALL SELECT 3548, 'SIC', 3552, 3800, 2, 3568, 3844, 4 FROM dual
UNION ALL SELECT 3549, 'SIC', 4308, 20, 1, 3371, 3400, 2 FROM dual
UNION ALL SELECT 3550, 'NAICS', 1812, 61, 1, 1813, 611, 2 FROM dual
UNION ALL SELECT 3551, 'NAICS', 1726, 561, 2, 1753, 56145, 4 FROM dual
UNION ALL SELECT 3552, 'SIC', 2884, 1000, 2, 2890, 1031, 4 FROM dual
UNION ALL SELECT 3553, 'NAICS', 1480, 52, 1, 1549, 52421, 4 FROM dual
UNION ALL SELECT 3554, 'SIC', 3526, 3700, 2, 3532, 3716, 4 FROM dual
UNION ALL SELECT 3555, 'NAICS', 81, 11241, 4, 80, 112410, 5 FROM dual
UNION ALL SELECT 3556, 'NAICS', 132, 21, 1, 166, 212325, 5 FROM dual
UNION ALL SELECT 3557, 'NAICS', 138, 212, 2, 157, 21231, 4 FROM dual
UNION ALL SELECT 3558, 'SIC', 3214, 2800, 2, 3233, 2843, 4 FROM dual
UNION ALL SELECT 3559, 'SIC', 4309, 40, 1, 3670, 4725, 4 FROM dual
UNION ALL SELECT 3560, 'NAICS', 997, 42383, 4, 996, 423830, 5 FROM dual
UNION ALL SELECT 3561, 'SIC', 3958, 6300, 2, 3972, 6390, 3 FROM dual
UNION ALL SELECT 3562, 'SIC', 3917, 6000, 2, 3930, 6062, 4 FROM dual
UNION ALL SELECT 3563, 'NAICS', 1403, 511, 2, 1409, 511130, 5 FROM dual
UNION ALL SELECT 3564, 'SIC', 3330, 3290, 3, 3332, 3292, 4 FROM dual
UNION ALL SELECT 3565, 'NAICS', 1536, 524, 2, 1548, 524210, 5 FROM dual
UNION ALL SELECT 3566, 'SEC', 2795, 60, 1, 2669, 6163, 4 FROM dual
UNION ALL SELECT 3567, 'SIC', 3124, 2400, 2, 3146, 2493, 4 FROM dual
UNION ALL SELECT 3568, 'SEC', 2713, 7300, 2, 2728, 7373, 4 FROM dual
UNION ALL SELECT 3569, 'SIC', 3083, 2300, 2, 3087, 2321, 4 FROM dual
UNION ALL SELECT 3570, 'SIC', 3033, 2090, 3, 3038, 2097, 4 FROM dual
UNION ALL SELECT 3571, 'NAICS', 1726, 561, 2, 1730, 5612, 3 FROM dual
UNION ALL SELECT 3572, 'SIC', 3548, 3790, 3, 3551, 3799, 4 FROM dual
UNION ALL SELECT 3573, 'SIC', 4209, 8610, 3, 4210, 8611, 4 FROM dual
UNION ALL SELECT 3574, 'NAICS', 206, 236, 2, 212, 236118, 5 FROM dual
UNION ALL SELECT 3575, 'NAICS', 1, 11, 1, 12, 111150, 5 FROM dual
UNION ALL SELECT 3576, 'SIC', 3655, 4510, 3, 3657, 4513, 4 FROM dual
UNION ALL SELECT 3577, 'NAICS', 1625, 541, 2, 1700, 541860, 5 FROM dual
UNION ALL SELECT 3578, 'SIC', 4313, 70, 1, 4087, 7520, 3 FROM dual
UNION ALL SELECT 3579, 'NAICS', 2204, 928, 2, 2208, 928120, 5 FROM dual
UNION ALL SELECT 3580, 'SEC', 2753, 8000, 2, 2760, 8070, 3 FROM dual
UNION ALL SELECT 3581, 'NAICS', 1420, 5121, 3, 1428, 51219, 4 FROM dual
UNION ALL SELECT 3582, 'SEC', 2791, 20, 1, 2367, 3272, 4 FROM dual
UNION ALL SELECT 3583, 'SIC', 4306, 10, 1, 2907, 1240, 3 FROM dual
UNION ALL SELECT 3584, 'SIC', 3886, 5900, 2, 3908, 5983, 4 FROM dual
UNION ALL SELECT 3585, 'NAICS', 931, 423, 2, 956, 423410, 5 FROM dual
UNION ALL SELECT 3586, 'NAICS', 1625, 541, 2, 1661, 54142, 4 FROM dual
UNION ALL SELECT 3587, 'SIC', 3481, 3610, 3, 3482, 3612, 4 FROM dual
UNION ALL SELECT 3588, 'SIC', 4313, 70, 1, 4110, 7692, 4 FROM dual
UNION ALL SELECT 3589, 'SIC', 3392, 3450, 3, 3394, 3452, 4 FROM dual
UNION ALL SELECT 3590, 'SIC', 4308, 20, 1, 3045, 2121, 4 FROM dual
UNION ALL SELECT 3591, 'NAICS', 2132, 8141, 3, 2134, 81411, 4 FROM dual
UNION ALL SELECT 3592, 'SEC', 2283, 2500, 2, 2286, 2520, 3 FROM dual
UNION ALL SELECT 3593, 'SIC', 4178, 8200, 2, 4186, 8240, 3 FROM dual
UNION ALL SELECT 3594, 'SEC', 2791, 20, 1, 2284, 2510, 3 FROM dual
UNION ALL SELECT 3595, 'NAICS', 1, 11, 1, 65, 1122, 3 FROM dual
UNION ALL SELECT 3596, 'NAICS', 2124, 81392, 4, 2123, 813920, 5 FROM dual
UNION ALL SELECT 3597, 'NAICS', 2176, 924, 2, 2177, 9241, 3 FROM dual
UNION ALL SELECT 3598, 'SEC', 2526, 4500, 2, 2532, 4580, 3 FROM dual
UNION ALL SELECT 3599, 'SIC', 4311, 52, 1, 3839, 5511, 4 FROM dual
UNION ALL SELECT 3600, 'SIC', 4308, 20, 1, 3564, 3840, 3 FROM dual
UNION ALL SELECT 3601, 'SIC', 4223, 8700, 2, 4237, 8742, 4 FROM dual
UNION ALL SELECT 3602, 'SIC', 4017, 7200, 2, 4030, 7231, 4 FROM dual
UNION ALL SELECT 3603, 'NAICS', 1904, 623, 2, 1907, 62311, 4 FROM dual
UNION ALL SELECT 3604, 'NAICS', 2135, 92, 1, 2184, 925110, 5 FROM dual
UNION ALL SELECT 3605, 'SIC', 4146, 8000, 2, 4159, 8051, 4 FROM dual
UNION ALL SELECT 3606, 'SIC', 4312, 60, 1, 3937, 6100, 2 FROM dual
UNION ALL SELECT 3607, 'NAICS', 2135, 92, 1, 2203, 92711, 4 FROM dual
UNION ALL SELECT 3608, 'NAICS', 931, 423, 2, 1001, 42385, 4 FROM dual
UNION ALL SELECT 3609, 'NAICS', 117, 115, 2, 126, 1152, 3 FROM dual
UNION ALL SELECT 3610, 'NAICS', 1625, 541, 2, 1686, 541712, 5 FROM dual
UNION ALL SELECT 3611, 'NAICS', 1734, 56131, 4, 1735, 561311, 5 FROM dual
UNION ALL SELECT 3612, 'SIC', 3083, 2300, 2, 3123, 2399, 4 FROM dual
UNION ALL SELECT 3613, 'SEC', 2302, 2700, 2, 2304, 2711, 4 FROM dual
UNION ALL SELECT 3614, 'NAICS', 1904, 623, 2, 1908, 6232, 3 FROM dual
UNION ALL SELECT 3615, 'SIC', 4308, 20, 1, 3300, 3199, 4 FROM dual
UNION ALL SELECT 3616, 'SIC', 3419, 3500, 2, 3426, 3530, 3 FROM dual
UNION ALL SELECT 3617, 'SIC', 3264, 3020, 3, 3265, 3021, 4 FROM dual
UNION ALL SELECT 3618, 'NAICS', 1, 11, 1, 31, 111333, 5 FROM dual
UNION ALL SELECT 3619, 'SIC', 4308, 20, 1, 3229, 2836, 4 FROM dual
UNION ALL SELECT 3620, 'SIC', 4313, 70, 1, 4224, 8710, 3 FROM dual
UNION ALL SELECT 3621, 'NAICS', 68, 1123, 3, 76, 11234, 4 FROM dual
UNION ALL SELECT 3622, 'NAICS', 47, 11192, 4, 46, 111920, 5 FROM dual
UNION ALL SELECT 3623, 'NAICS', 1640, 5413, 3, 1647, 541340, 5 FROM dual
UNION ALL SELECT 3624, 'NAICS', 2166, 923, 2, 2171, 92312, 4 FROM dual
UNION ALL SELECT 3625, 'NAICS', 1584, 53131, 4, 1585, 531311, 5 FROM dual
UNION ALL SELECT 3626, 'NAICS', 1725, 56, 1, 1755, 561491, 5 FROM dual
UNION ALL SELECT 3627, 'NAICS', 1920, 624, 2, 1932, 624221, 5 FROM dual
UNION ALL SELECT 3628, 'SEC', 2293, 2610, 3, 2294, 2611, 4 FROM dual
UNION ALL SELECT 3629, 'SEC', 2500, 3900, 2, 2501, 3910, 3 FROM dual
UNION ALL SELECT 3630, 'NAICS', 1625, 541, 2, 1650, 54135, 4 FROM dual
UNION ALL SELECT 3631, 'NAICS', 1625, 541, 2, 1655, 541380, 5 FROM dual
UNION ALL SELECT 3632, 'SEC', 2247, 2030, 3, 2248, 2033, 4 FROM dual
UNION ALL SELECT 3633, 'SIC', 3715, 5000, 2, 3754, 5087, 4 FROM dual
UNION ALL SELECT 3634, 'NAICS', 218, 237, 2, 219, 2371, 3 FROM dual
UNION ALL SELECT 3635, 'SEC', 2796, 70, 1, 2713, 7300, 2 FROM dual
UNION ALL SELECT 3636, 'SIC', 4309, 40, 1, 3609, 4120, 3 FROM dual
UNION ALL SELECT 3637, 'SIC', 3526, 3700, 2, 3551, 3799, 4 FROM dual
UNION ALL SELECT 3638, 'SIC', 3371, 3400, 2, 3385, 3441, 4 FROM dual
UNION ALL SELECT 3639, 'NAICS', 2037, 81, 1, 2054, 811211, 5 FROM dual
UNION ALL SELECT 3640, 'SIC', 3716, 5010, 3, 3719, 5014, 4 FROM dual
UNION ALL SELECT 3641, 'SIC', 3576, 3900, 2, 3597, 3993, 4 FROM dual
UNION ALL SELECT 3642, 'NAICS', 1402, 51, 1, 1405, 511110, 5 FROM dual
UNION ALL SELECT 3643, 'SIC', 4306, 10, 1, 2916, 1382, 4 FROM dual
UNION ALL SELECT 3644, 'SIC', 4308, 20, 1, 3560, 3825, 4 FROM dual
UNION ALL SELECT 3645, 'NAICS', 979, 42362, 4, 978, 423620, 5 FROM dual
UNION ALL SELECT 3646, 'NAICS', 1921, 6241, 3, 1925, 62412, 4 FROM dual
UNION ALL SELECT 3647, 'NAICS', 2037, 81, 1, 2096, 81292, 4 FROM dual
UNION ALL SELECT 3648, 'NAICS', 1570, 531, 2, 1584, 53131, 4 FROM dual
UNION ALL SELECT 3649, 'NAICS', 946, 4233, 3, 949, 423320, 5 FROM dual
UNION ALL SELECT 3650, 'SEC', 2791, 20, 1, 2298, 2631, 4 FROM dual
UNION ALL SELECT 3651, 'SIC', 3262, 3010, 3, 3263, 3011, 4 FROM dual
UNION ALL SELECT 3652, 'SEC', 2342, 3000, 2, 2350, 3081, 4 FROM dual
UNION ALL SELECT 3653, 'NAICS', 1598, 5322, 3, 1604, 53223, 4 FROM dual
UNION ALL SELECT 3654, 'SIC', 3680, 4800, 2, 3690, 4841, 4 FROM dual
UNION ALL SELECT 3655, 'NAICS', 1035, 4244, 3, 1051, 42448, 4 FROM dual
UNION ALL SELECT 3656, 'NAICS', 1823, 6114, 3, 1825, 61141, 4 FROM dual
UNION ALL SELECT 3657, 'SIC', 4314, 90, 1, 4286, 9611, 4 FROM dual
UNION ALL SELECT 3658, 'SIC', 4308, 20, 1, 3191, 2710, 3 FROM dual
UNION ALL SELECT 3659, 'SIC', 4146, 8000, 2, 4147, 8010, 3 FROM dual
UNION ALL SELECT 3660, 'NAICS', 955, 4234, 3, 968, 423490, 5 FROM dual
UNION ALL SELECT 3661, 'NAICS', 1904, 623, 2, 1909, 623210, 5 FROM dual
UNION ALL SELECT 3662, 'NAICS', 1970, 7121, 3, 1971, 712110, 5 FROM dual
UNION ALL SELECT 3663, 'NAICS', 1, 11, 1, 68, 1123, 3 FROM dual
UNION ALL SELECT 3664, 'NAICS', 1726, 561, 2, 1767, 56161, 4 FROM dual
UNION ALL SELECT 3665, 'NAICS', 1495, 5222, 3, 1500, 52229, 4 FROM dual
UNION ALL SELECT 3666, 'NAICS', 2071, 812, 2, 2087, 81231, 4 FROM dual
UNION ALL SELECT 3667, 'SIC', 4306, 10, 1, 2930, 1459, 4 FROM dual
UNION ALL SELECT 3668, 'SIC', 4306, 10, 1, 2895, 1061, 4 FROM dual
UNION ALL SELECT 3669, 'NAICS', 1715, 54194, 4, 1714, 541940, 5 FROM dual
UNION ALL SELECT 3670, 'SEC', 2403, 3500, 2, 2429, 3578, 4 FROM dual
UNION ALL SELECT 3671, 'SIC', 3544, 3760, 3, 3545, 3761, 4 FROM dual
UNION ALL SELECT 3672, 'SIC', 4305, 1, 1, 2862, 761, 4 FROM dual
UNION ALL SELECT 3673, 'NAICS', 133, 211, 2, 135, 21111, 4 FROM dual
UNION ALL SELECT 3674, 'SIC', 4314, 90, 1, 4266, 9310, 3 FROM dual
UNION ALL SELECT 3675, 'SEC', 2796, 70, 1, 2712, 7200, 2 FROM dual
UNION ALL SELECT 3676, 'SIC', 4223, 8700, 2, 4239, 8744, 4 FROM dual
UNION ALL SELECT 3677, 'NAICS', 1404, 5111, 3, 1408, 51112, 4 FROM dual
UNION ALL SELECT 3678, 'NAICS', 930, 42, 1, 1017, 424110, 5 FROM dual
UNION ALL SELECT 3679, 'SEC', 2704, 6790, 3, 2706, 6794, 4 FROM dual
UNION ALL SELECT 3680, 'SIC', 4308, 20, 1, 3054, 2221, 4 FROM dual
UNION ALL SELECT 3681, 'SIC', 3395, 3460, 3, 3396, 3462, 4 FROM dual
UNION ALL SELECT 3682, 'NAICS', 1689, 5418, 3, 1700, 541860, 5 FROM dual
UNION ALL SELECT 3683, 'SIC', 4308, 20, 1, 3128, 2421, 4 FROM dual
UNION ALL SELECT 3684, 'SIC', 4307, 15, 1, 2951, 1620, 3 FROM dual
UNION ALL SELECT 3685, 'SIC', 2826, 210, 3, 2827, 211, 4 FROM dual
UNION ALL SELECT 3686, 'NAICS', 991, 4238, 3, 1000, 423850, 5 FROM dual
UNION ALL SELECT 3687, 'NAICS', 43, 1119, 3, 48, 111930, 5 FROM dual
UNION ALL SELECT 3688, 'SIC', 2847, 700, 2, 2849, 711, 4 FROM dual
UNION ALL SELECT 3689, 'SIC', 3576, 3900, 2, 3578, 3911, 4 FROM dual
UNION ALL SELECT 3690, 'SIC', 3451, 3560, 3, 3454, 3563, 4 FROM dual
UNION ALL SELECT 3691, 'NAICS', 1569, 53, 1, 1610, 532310, 5 FROM dual
UNION ALL SELECT 3692, 'NAICS', 1813, 611, 2, 1845, 611692, 5 FROM dual
UNION ALL SELECT 3693, 'NAICS', 936, 42312, 4, 935, 423120, 5 FROM dual
UNION ALL SELECT 3694, 'NAICS', 1570, 531, 2, 1573, 53111, 4 FROM dual
UNION ALL SELECT 3695, 'NAICS', 1706, 5419, 3, 1713, 54193, 4 FROM dual
UNION ALL SELECT 3696, 'SIC', 3190, 2700, 2, 3191, 2710, 3 FROM dual
UNION ALL SELECT 3697, 'SIC', 3583, 3940, 3, 3586, 3949, 4 FROM dual
UNION ALL SELECT 3698, 'SIC', 3167, 2600, 2, 3179, 2657, 4 FROM dual
UNION ALL SELECT 3699, 'NAICS', 972, 42351, 4, 971, 423510, 5 FROM dual
UNION ALL SELECT 3700, 'SIC', 4308, 20, 1, 3115, 2390, 3 FROM dual
UNION ALL SELECT 3701, 'SIC', 3837, 5500, 2, 3841, 5521, 4 FROM dual
UNION ALL SELECT 3702, 'NAICS', 2037, 81, 1, 2056, 811213, 5 FROM dual
UNION ALL SELECT 3703, 'NAICS', 930, 42, 1, 1073, 42481, 4 FROM dual
UNION ALL SELECT 3704, 'SIC', 3636, 4400, 2, 3641, 4430, 3 FROM dual
UNION ALL SELECT 3705, 'SIC', 2813, 170, 3, 2815, 172, 4 FROM dual
UNION ALL SELECT 3706, 'NAICS', 2, 111, 2, 46, 111920, 5 FROM dual
UNION ALL SELECT 3707, 'NAICS', 219, 2371, 3, 222, 237120, 5 FROM dual
UNION ALL SELECT 3708, 'NAICS', 1061, 4246, 3, 1065, 42469, 4 FROM dual
UNION ALL SELECT 3709, 'NAICS', 1598, 5322, 3, 1608, 532299, 5 FROM dual
UNION ALL SELECT 3710, 'NAICS', 931, 423, 2, 980, 423690, 5 FROM dual
UNION ALL SELECT 3711, 'SEC', 2623, 5600, 2, 2625, 5621, 4 FROM dual
UNION ALL SELECT 3712, 'SIC', 4308, 20, 1, 3445, 3552, 4 FROM dual
UNION ALL SELECT 3713, 'NAICS', 1726, 561, 2, 1750, 561440, 5 FROM dual
UNION ALL SELECT 3714, 'SIC', 3620, 4200, 2, 3626, 4220, 3 FROM dual
UNION ALL SELECT 3715, 'SIC', 3861, 5640, 3, 3862, 5641, 4 FROM dual
UNION ALL SELECT 3716, 'NAICS', 1591, 532, 2, 1617, 53242, 4 FROM dual
UNION ALL SELECT 3717, 'SEC', 4317, 4990, 2, 4318, 4991, 3 FROM dual
UNION ALL SELECT 3718, 'SEC', 2540, 4810, 3, 2541, 4812, 4 FROM dual
UNION ALL SELECT 3719, 'NAICS', 1895, 6221, 3, 1897, 62211, 4 FROM dual
UNION ALL SELECT 3720, 'NAICS', 180, 22, 1, 189, 221116, 5 FROM dual
UNION ALL SELECT 3721, 'NAICS', 1850, 62, 1, 1912, 62322, 4 FROM dual
UNION ALL SELECT 3722, 'SIC', 3576, 3900, 2, 3585, 3944, 4 FROM dual
UNION ALL SELECT 3723, 'SIC', 4065, 7370, 3, 4069, 7374, 4 FROM dual
UNION ALL SELECT 3724, 'NAICS', 930, 42, 1, 1029, 424320, 5 FROM dual
UNION ALL SELECT 3725, 'NAICS', 1725, 56, 1, 1808, 56292, 4 FROM dual
UNION ALL SELECT 3726, 'SIC', 2982, 2000, 2, 2986, 2015, 4 FROM dual
UNION ALL SELECT 3727, 'NAICS', 1725, 56, 1, 1764, 561591, 5 FROM dual
UNION ALL SELECT 3728, 'NAICS', 2037, 81, 1, 2038, 811, 2 FROM dual
UNION ALL SELECT 3729, 'NAICS', 2135, 92, 1, 2186, 925120, 5 FROM dual
UNION ALL SELECT 3730, 'SIC', 3225, 2830, 3, 3227, 2834, 4 FROM dual
UNION ALL SELECT 3731, 'NAICS', 1718, 55, 1, 1724, 551114, 5 FROM dual
UNION ALL SELECT 3732, 'NAICS', 110, 11411, 4, 113, 114119, 5 FROM dual
UNION ALL SELECT 3733, 'NAICS', 930, 42, 1, 1076, 4249, 3 FROM dual
UNION ALL SELECT 3734, 'SIC', 4153, 8040, 3, 4157, 8049, 4 FROM dual
UNION ALL SELECT 3735, 'SIC', 2805, 130, 3, 2806, 131, 4 FROM dual
UNION ALL SELECT 3736, 'SEC', 2791, 20, 1, 2330, 2844, 4 FROM dual
UNION ALL SELECT 3737, 'NAICS', 1640, 5413, 3, 1656, 54138, 4 FROM dual
UNION ALL SELECT 3738, 'NAICS', 1894, 622, 2, 1902, 622310, 5 FROM dual
UNION ALL SELECT 3739, 'SIC', 4308, 20, 1, 3381, 3431, 4 FROM dual
UNION ALL SELECT 3740, 'NAICS', 1672, 5416, 3, 1681, 541690, 5 FROM dual
UNION ALL SELECT 3741, 'SIC', 3083, 2300, 2, 3106, 2370, 3 FROM dual
UNION ALL SELECT 3742, 'SIC', 3552, 3800, 2, 3571, 3851, 4 FROM dual
UNION ALL SELECT 3743, 'SIC', 3667, 4700, 2, 3679, 4789, 4 FROM dual
UNION ALL SELECT 3744, 'SIC', 3869, 5700, 2, 3878, 5731, 4 FROM dual
UNION ALL SELECT 3745, 'SIC', 4305, 1, 1, 2799, 110, 3 FROM dual
UNION ALL SELECT 3746, 'NAICS', 930, 42, 1, 931, 423, 2 FROM dual
UNION ALL SELECT 3747, 'NAICS', 1015, 424, 2, 1063, 42461, 4 FROM dual
UNION ALL SELECT 3748, 'SIC', 4308, 20, 1, 3482, 3612, 4 FROM dual
UNION ALL SELECT 3749, 'SEC', 2438, 3620, 3, 2439, 3621, 4 FROM dual
UNION ALL SELECT 3750, 'NAICS', 1403, 511, 2, 1406, 51111, 4 FROM dual
UNION ALL SELECT 3751, 'NAICS', 1580, 5312, 3, 1582, 53121, 4 FROM dual
UNION ALL SELECT 3752, 'SIC', 3511, 3670, 3, 3519, 3679, 4 FROM dual
UNION ALL SELECT 3753, 'SIC', 4081, 7500, 2, 4086, 7519, 4 FROM dual
UNION ALL SELECT 3754, 'NAICS', 2189, 9261, 3, 2199, 92615, 4 FROM dual
UNION ALL SELECT 3755, 'SEC', 2791, 20, 1, 2314, 2761, 4 FROM dual
UNION ALL SELECT 3756, 'NAICS', 1480, 52, 1, 1543, 524127, 5 FROM dual
UNION ALL SELECT 3757, 'SIC', 3083, 2300, 2, 3094, 2331, 4 FROM dual
UNION ALL SELECT 3758, 'SIC', 4308, 20, 1, 3142, 2451, 4 FROM dual
UNION ALL SELECT 3759, 'SIC', 4040, 7300, 2, 4058, 7350, 3 FROM dual
UNION ALL SELECT 3760, 'NAICS', 213, 2362, 3, 216, 236220, 5 FROM dual
UNION ALL SELECT 3761, 'SIC', 4308, 20, 1, 3071, 2270, 3 FROM dual
UNION ALL SELECT 3762, 'NAICS', 235, 238, 2, 254, 238210, 5 FROM dual
UNION ALL SELECT 3763, 'SEC', 2218, 1200, 2, 2219, 1220, 3 FROM dual
UNION ALL SELECT 3764, 'SEC', 2403, 3500, 2, 2432, 3585, 4 FROM dual
UNION ALL SELECT 3765, 'SIC', 4306, 10, 1, 2934, 1479, 4 FROM dual
UNION ALL SELECT 3766, 'NAICS', 1733, 5613, 3, 1738, 56132, 4 FROM dual
UNION ALL SELECT 3767, 'NAICS', 1035, 4244, 3, 1049, 42447, 4 FROM dual
UNION ALL SELECT 3768, 'SEC', 2792, 40, 1, 2522, 4231, 4 FROM dual
UNION ALL SELECT 3769, 'SIC', 3050, 2200, 2, 3076, 2284, 4 FROM dual
UNION ALL SELECT 3770, 'NAICS', 1657, 5414, 3, 1660, 541420, 5 FROM dual
UNION ALL SELECT 3771, 'NAICS', 236, 2381, 3, 239, 238120, 5 FROM dual
UNION ALL SELECT 3772, 'SIC', 4313, 70, 1, 4118, 7822, 4 FROM dual
UNION ALL SELECT 3773, 'SIC', 3574, 3870, 3, 3575, 3873, 4 FROM dual
UNION ALL SELECT 3774, 'SIC', 4195, 8330, 3, 4196, 8331, 4 FROM dual
UNION ALL SELECT 3775, 'SIC', 3083, 2300, 2, 3084, 2310, 3 FROM dual
UNION ALL SELECT 3776, 'NAICS', 197, 22121, 4, 196, 221210, 5 FROM dual
UNION ALL SELECT 3777, 'SIC', 3918, 6010, 3, 3920, 6019, 4 FROM dual
UNION ALL SELECT 3778, 'SIC', 4146, 8000, 2, 4156, 8043, 4 FROM dual
UNION ALL SELECT 3779, 'NAICS', 1683, 5417, 3, 1685, 541711, 5 FROM dual
UNION ALL SELECT 3780, 'NAICS', 205, 23, 1, 215, 23621, 4 FROM dual
UNION ALL SELECT 3781, 'SIC', 4308, 20, 1, 3204, 2760, 3 FROM dual
UNION ALL SELECT 3782, 'SIC', 3371, 3400, 2, 3377, 3423, 4 FROM dual
UNION ALL SELECT 3783, 'SIC', 3480, 3600, 2, 3485, 3621, 4 FROM dual
UNION ALL SELECT 3784, 'NAICS', 102, 1132, 3, 103, 113210, 5 FROM dual
UNION ALL SELECT 3785, 'SIC', 4307, 15, 1, 2943, 1530, 3 FROM dual
UNION ALL SELECT 3786, 'SIC', 4309, 40, 1, 3618, 4170, 3 FROM dual
UNION ALL SELECT 3787, 'SIC', 4309, 40, 1, 3712, 4961, 4 FROM dual
UNION ALL SELECT 3788, 'NAICS', 1584, 53131, 4, 1586, 531312, 5 FROM dual
UNION ALL SELECT 3789, 'NAICS', 2093, 8129, 3, 2095, 81291, 4 FROM dual
UNION ALL SELECT 3790, 'SIC', 4146, 8000, 2, 4162, 8060, 3 FROM dual
UNION ALL SELECT 3791, 'SIC', 4313, 70, 1, 4185, 8231, 4 FROM dual
UNION ALL SELECT 3792, 'NAICS', 235, 238, 2, 252, 23819, 4 FROM dual
UNION ALL SELECT 3793, 'SEC', 2675, 6280, 3, 2676, 6282, 4 FROM dual
UNION ALL SELECT 3794, 'NAICS', 76, 11234, 4, 75, 112340, 5 FROM dual
UNION ALL SELECT 3795, 'SIC', 3595, 3990, 3, 3596, 3991, 4 FROM dual
UNION ALL SELECT 3796, 'NAICS', 205, 23, 1, 241, 238130, 5 FROM dual
UNION ALL SELECT 3797, 'SIC', 4308, 20, 1, 3319, 3263, 4 FROM dual
UNION ALL SELECT 3798, 'SIC', 2955, 1700, 2, 2961, 1731, 4 FROM dual
UNION ALL SELECT 3799, 'SIC', 4308, 20, 1, 3367, 3369, 4 FROM dual
UNION ALL SELECT 3800, 'SIC', 2939, 1500, 2, 2942, 1522, 4 FROM dual
UNION ALL SELECT 3801, 'SIC', 4309, 40, 1, 3680, 4800, 2 FROM dual
UNION ALL SELECT 3802, 'NAICS', 1813, 611, 2, 1819, 61121, 4 FROM dual
UNION ALL SELECT 3803, 'SEC', 2302, 2700, 2, 2308, 2731, 4 FROM dual
UNION ALL SELECT 3804, 'NAICS', 1094, 42512, 4, 1093, 425120, 5 FROM dual
UNION ALL SELECT 3805, 'SEC', 2792, 40, 1, 2521, 4230, 3 FROM dual
UNION ALL SELECT 3806, 'SIC', 3419, 3500, 2, 3470, 3582, 4 FROM dual
UNION ALL SELECT 3807, 'NAICS', 1402, 51, 1, 1427, 512132, 5 FROM dual
UNION ALL SELECT 3808, 'NAICS', 2103, 813, 2, 2130, 81399, 4 FROM dual
UNION ALL SELECT 3809, 'SEC', 2792, 40, 1, 2542, 4813, 4 FROM dual
UNION ALL SELECT 3810, 'SIC', 3903, 5960, 3, 3905, 5962, 4 FROM dual
UNION ALL SELECT 3811, 'NAICS', 1804, 5629, 3, 1811, 562998, 5 FROM dual
UNION ALL SELECT 3812, 'NAICS', 2103, 813, 2, 2114, 813311, 5 FROM dual
UNION ALL SELECT 3813, 'NAICS', 1813, 611, 2, 1841, 611630, 5 FROM dual
UNION ALL SELECT 3814, 'NAICS', 128, 11521, 4, 127, 115210, 5 FROM dual
UNION ALL SELECT 3815, 'NAICS', 1480, 52, 1, 1487, 522110, 5 FROM dual
UNION ALL SELECT 3816, 'SIC', 4308, 20, 1, 3358, 3354, 4 FROM dual
UNION ALL SELECT 3817, 'NAICS', 1015, 424, 2, 1083, 424940, 5 FROM dual
UNION ALL SELECT 3818, 'NAICS', 1403, 511, 2, 1413, 51119, 4 FROM dual
UNION ALL SELECT 3819, 'NAICS', 181, 221, 2, 188, 221115, 5 FROM dual
UNION ALL SELECT 3820, 'SIC', 3756, 5090, 3, 3758, 5092, 4 FROM dual
UNION ALL SELECT 3821, 'SIC', 4312, 60, 1, 3948, 6200, 2 FROM dual
UNION ALL SELECT 3822, 'SIC', 3190, 2700, 2, 3204, 2760, 3 FROM dual
UNION ALL SELECT 3823, 'SIC', 3577, 3910, 3, 3579, 3914, 4 FROM dual
UNION ALL SELECT 3824, 'SIC', 4308, 20, 1, 3483, 3613, 4 FROM dual
UNION ALL SELECT 3825, 'NAICS', 119, 11511, 4, 123, 115114, 5 FROM dual
UNION ALL SELECT 3826, 'SIC', 4007, 7000, 2, 4013, 7032, 4 FROM dual
UNION ALL SELECT 3827, 'SEC', 2794, 52, 1, 2648, 5990, 3 FROM dual
UNION ALL SELECT 3828, 'SIC', 4312, 60, 1, 3942, 6150, 3 FROM dual
UNION ALL SELECT 3829, 'SEC', 2791, 20, 1, 2356, 3210, 3 FROM dual
UNION ALL SELECT 3830, 'SIC', 4312, 60, 1, 3926, 6035, 4 FROM dual
UNION ALL SELECT 3831, 'SEC', 2795, 60, 1, 2665, 6153, 4 FROM dual
UNION ALL SELECT 3832, 'SIC', 4308, 20, 1, 3404, 3480, 3 FROM dual
UNION ALL SELECT 3833, 'NAICS', 1747, 56143, 4, 1748, 561431, 5 FROM dual
UNION ALL SELECT 3834, 'NAICS', 1985, 7132, 3, 1989, 71329, 4 FROM dual
UNION ALL SELECT 3835, 'NAICS', 2003, 72, 1, 2009, 72112, 4 FROM dual
UNION ALL SELECT 3836, 'SEC', 2424, 3570, 3, 4315, 3576, 4 FROM dual
UNION ALL SELECT 3837, 'SIC', 4311, 52, 1, 3881, 5736, 4 FROM dual
UNION ALL SELECT 3838, 'SIC', 4313, 70, 1, 4079, 7384, 4 FROM dual
UNION ALL SELECT 3839, 'SIC', 3287, 3140, 3, 3290, 3144, 4 FROM dual
UNION ALL SELECT 3840, 'SIC', 4314, 90, 1, 4274, 9441, 4 FROM dual
UNION ALL SELECT 3841, 'SIC', 4313, 70, 1, 4125, 7900, 2 FROM dual
UNION ALL SELECT 3842, 'NAICS', 1799, 56221, 4, 1803, 562219, 5 FROM dual
UNION ALL SELECT 3843, 'NAICS', 79, 1124, 3, 81, 11241, 4 FROM dual
UNION ALL SELECT 3844, 'SIC', 4146, 8000, 2, 4161, 8059, 4 FROM dual
UNION ALL SELECT 3845, 'SIC', 4311, 52, 1, 3891, 5930, 3 FROM dual
UNION ALL SELECT 3846, 'SIC', 3214, 2800, 2, 3217, 2813, 4 FROM dual
UNION ALL SELECT 3847, 'SIC', 3972, 6390, 3, 3973, 6399, 4 FROM dual
UNION ALL SELECT 3848, 'SIC', 4247, 9100, 2, 4253, 9131, 4 FROM dual
UNION ALL SELECT 3849, 'NAICS', 1569, 53, 1, 1581, 531210, 5 FROM dual
UNION ALL SELECT 3850, 'SIC', 3401, 3470, 3, 3403, 3479, 4 FROM dual
UNION ALL SELECT 3851, 'SIC', 4305, 1, 1, 2840, 270, 3 FROM dual
UNION ALL SELECT 3852, 'SIC', 4309, 40, 1, 3634, 4310, 3 FROM dual
UNION ALL SELECT 3853, 'NAICS', 1554, 525, 2, 1555, 5251, 3 FROM dual
UNION ALL SELECT 3854, 'NAICS', 2037, 81, 1, 2045, 81112, 4 FROM dual
UNION ALL SELECT 3855, 'SIC', 4062, 7360, 3, 4063, 7361, 4 FROM dual
UNION ALL SELECT 3856, 'NAICS', 156, 2123, 3, 157, 21231, 4 FROM dual
UNION ALL SELECT 3857, 'NAICS', 1725, 56, 1, 1805, 562910, 5 FROM dual
UNION ALL SELECT 3858, 'SEC', 2221, 1300, 2, 2222, 1310, 3 FROM dual
UNION ALL SELECT 3859, 'SIC', 3026, 2080, 3, 3028, 2083, 4 FROM dual
UNION ALL SELECT 3860, 'SEC', 2574, 5040, 3, 2575, 5045, 4 FROM dual
UNION ALL SELECT 3861, 'NAICS', 1625, 541, 2, 1654, 54137, 4 FROM dual
UNION ALL SELECT 3862, 'SEC', 2552, 4900, 2, 2565, 4953, 4 FROM dual
UNION ALL SELECT 3863, 'SIC', 3869, 5700, 2, 3879, 5734, 4 FROM dual
UNION ALL SELECT 3864, 'NAICS', 1794, 56211, 4, 1797, 562119, 5 FROM dual
UNION ALL SELECT 3865, 'SIC', 3419, 3500, 2, 3465, 3577, 4 FROM dual
UNION ALL SELECT 3866, 'NAICS', 219, 2371, 3, 220, 237110, 5 FROM dual
UNION ALL SELECT 3867, 'NAICS', 1402, 51, 1, 1462, 5179, 3 FROM dual
UNION ALL SELECT 3868, 'SIC', 3680, 4800, 2, 3682, 4812, 4 FROM dual
UNION ALL SELECT 3869, 'SEC', 2488, 3840, 3, 2492, 3844, 4 FROM dual
UNION ALL SELECT 3870, 'NAICS', 1063, 42461, 4, 1062, 424610, 5 FROM dual
UNION ALL SELECT 3871, 'SIC', 4308, 20, 1, 3239, 2865, 4 FROM dual
UNION ALL SELECT 3872, 'SIC', 4313, 70, 1, 4139, 7990, 3 FROM dual
UNION ALL SELECT 3873, 'NAICS', 931, 423, 2, 965, 42345, 4 FROM dual
UNION ALL SELECT 3874, 'SIC', 3938, 6110, 3, 3939, 6111, 4 FROM dual
UNION ALL SELECT 3875, 'NAICS', 930, 42, 1, 1000, 423850, 5 FROM dual
UNION ALL SELECT 3876, 'NAICS', 1569, 53, 1, 1577, 53113, 4 FROM dual
UNION ALL SELECT 3877, 'SIC', 4309, 40, 1, 3711, 4960, 3 FROM dual
UNION ALL SELECT 3878, 'SIC', 4308, 20, 1, 3519, 3679, 4 FROM dual
UNION ALL SELECT 3879, 'NAICS', 1591, 532, 2, 1594, 532111, 5 FROM dual
UNION ALL SELECT 3880, 'SIC', 3375, 3420, 3, 3378, 3425, 4 FROM dual
UNION ALL SELECT 3881, 'SIC', 4305, 1, 1, 2869, 810, 3 FROM dual
UNION ALL SELECT 3882, 'SIC', 3756, 5090, 3, 3760, 5094, 4 FROM dual
UNION ALL SELECT 3883, 'NAICS', 1, 11, 1, 47, 11192, 4 FROM dual
UNION ALL SELECT 3884, 'SEC', 2794, 52, 1, 2633, 5730, 3 FROM dual
UNION ALL SELECT 3885, 'NAICS', 1035, 4244, 3, 1046, 424460, 5 FROM dual
UNION ALL SELECT 3886, 'SIC', 4313, 70, 1, 4228, 8720, 3 FROM dual
UNION ALL SELECT 3887, 'SIC', 3125, 2410, 3, 3126, 2411, 4 FROM dual
UNION ALL SELECT 3888, 'NAICS', 1471, 5191, 3, 1472, 519110, 5 FROM dual
UNION ALL SELECT 3889, 'NAICS', 1035, 4244, 3, 1045, 42445, 4 FROM dual
UNION ALL SELECT 3890, 'SEC', 2791, 20, 1, 2432, 3585, 4 FROM dual
UNION ALL SELECT 3891, 'SIC', 4309, 40, 1, 3665, 4613, 4 FROM dual
UNION ALL SELECT 3892, 'NAICS', 2025, 72232, 4, 2024, 722320, 5 FROM dual
UNION ALL SELECT 3893, 'SEC', 2539, 4800, 2, 2547, 4833, 4 FROM dual
UNION ALL SELECT 3894, 'SEC', 2355, 3200, 2, 2368, 3280, 3 FROM dual
UNION ALL SELECT 3895, 'SIC', 4312, 60, 1, 3979, 6512, 4 FROM dual
UNION ALL SELECT 3896, 'SIC', 4313, 70, 1, 4098, 7542, 4 FROM dual
UNION ALL SELECT 3897, 'NAICS', 1015, 424, 2, 1082, 42493, 4 FROM dual
UNION ALL SELECT 3898, 'SIC', 3715, 5000, 2, 3726, 5032, 4 FROM dual
UNION ALL SELECT 3899, 'SIC', 3180, 2670, 3, 3186, 2676, 4 FROM dual
UNION ALL SELECT 3900, 'SIC', 4313, 70, 1, 4244, 8900, 2 FROM dual
UNION ALL SELECT 3901, 'SIC', 4313, 70, 1, 4037, 7290, 3 FROM dual
UNION ALL SELECT 3902, 'SIC', 2928, 1450, 3, 2929, 1455, 4 FROM dual
UNION ALL SELECT 3903, 'SIC', 4314, 90, 1, 4296, 9661, 4 FROM dual
UNION ALL SELECT 3904, 'SIC', 4313, 70, 1, 4171, 8090, 3 FROM dual
UNION ALL SELECT 3905, 'SIC', 4040, 7300, 2, 4070, 7375, 4 FROM dual
UNION ALL SELECT 3906, 'SEC', 2443, 3650, 3, 2444, 3651, 4 FROM dual
UNION ALL SELECT 3907, 'NAICS', 1624, 54, 1, 1633, 541199, 5 FROM dual
UNION ALL SELECT 3908, 'SIC', 4308, 20, 1, 3140, 2449, 4 FROM dual
UNION ALL SELECT 3909, 'NAICS', 98, 113, 2, 101, 11311, 4 FROM dual
UNION ALL SELECT 3910, 'SIC', 2966, 1750, 3, 2968, 1752, 4 FROM dual
UNION ALL SELECT 3911, 'SIC', 3605, 4100, 2, 3615, 4142, 4 FROM dual
UNION ALL SELECT 3912, 'NAICS', 1774, 5617, 3, 1775, 561710, 5 FROM dual
UNION ALL SELECT 3913, 'SEC', 2709, 7000, 2, 2711, 7011, 4 FROM dual
UNION ALL SELECT 3914, 'SIC', 3576, 3900, 2, 3596, 3991, 4 FROM dual
UNION ALL SELECT 3915, 'SIC', 3762, 5100, 2, 3771, 5136, 4 FROM dual
UNION ALL SELECT 3916, 'SEC', 2703, 6700, 2, 2706, 6794, 4 FROM dual
UNION ALL SELECT 3917, 'SIC', 3552, 3800, 2, 3574, 3870, 3 FROM dual
UNION ALL SELECT 3918, 'SIC', 3626, 4220, 3, 3627, 4221, 4 FROM dual
UNION ALL SELECT 3919, 'SIC', 2982, 2000, 2, 2995, 2033, 4 FROM dual
UNION ALL SELECT 3920, 'SIC', 2918, 1400, 2, 2933, 1475, 4 FROM dual
UNION ALL SELECT 3921, 'SIC', 4308, 20, 1, 3081, 2298, 4 FROM dual
UNION ALL SELECT 3922, 'SEC', 2568, 5000, 2, 2571, 5020, 3 FROM dual
UNION ALL SELECT 3923, 'SIC', 3480, 3600, 2, 3515, 3675, 4 FROM dual
UNION ALL SELECT 3924, 'NAICS', 1923, 62411, 4, 1922, 624110, 5 FROM dual
UNION ALL SELECT 3925, 'NAICS', 1004, 4239, 3, 1005, 423910, 5 FROM dual
UNION ALL SELECT 3926, 'SIC', 3576, 3900, 2, 3577, 3910, 3 FROM dual
UNION ALL SELECT 3927, 'SIC', 4308, 20, 1, 3370, 3399, 4 FROM dual
UNION ALL SELECT 3928, 'NAICS', 1904, 623, 2, 1918, 623990, 5 FROM dual
UNION ALL SELECT 3929, 'NAICS', 1, 11, 1, 98, 113, 2 FROM dual
UNION ALL SELECT 3930, 'NAICS', 162, 21232, 4, 165, 212324, 5 FROM dual
UNION ALL SELECT 3931, 'SIC', 4065, 7370, 3, 4073, 7378, 4 FROM dual
UNION ALL SELECT 3932, 'SIC', 3729, 5040, 3, 3731, 5044, 4 FROM dual
UNION ALL SELECT 3933, 'SIC', 4309, 40, 1, 3613, 4140, 3 FROM dual
UNION ALL SELECT 3934, 'SEC', 2355, 3200, 2, 2357, 3211, 4 FROM dual
UNION ALL SELECT 3935, 'SIC', 3970, 6370, 3, 3971, 6371, 4 FROM dual
UNION ALL SELECT 3936, 'SIC', 4313, 70, 1, 4200, 8361, 4 FROM dual
UNION ALL SELECT 3937, 'SIC', 3931, 6080, 3, 3932, 6081, 4 FROM dual
UNION ALL SELECT 3938, 'NAICS', 1725, 56, 1, 1745, 561421, 5 FROM dual
UNION ALL SELECT 3939, 'NAICS', 1054, 4245, 3, 1060, 42459, 4 FROM dual
UNION ALL SELECT 3940, 'NAICS', 1076, 4249, 3, 1087, 424990, 5 FROM dual
UNION ALL SELECT 3941, 'SIC', 4308, 20, 1, 3565, 3841, 4 FROM dual
UNION ALL SELECT 3942, 'NAICS', 1851, 621, 2, 1878, 621492, 5 FROM dual
UNION ALL SELECT 3943, 'NAICS', 1654, 54137, 4, 1653, 541370, 5 FROM dual
UNION ALL SELECT 3944, 'SIC', 4306, 10, 1, 2885, 1010, 3 FROM dual
UNION ALL SELECT 3945, 'SIC', 3526, 3700, 2, 3539, 3732, 4 FROM dual
UNION ALL SELECT 3946, 'NAICS', 957, 42341, 4, 956, 423410, 5 FROM dual
UNION ALL SELECT 3947, 'SIC', 4308, 20, 1, 3028, 2083, 4 FROM dual
UNION ALL SELECT 3948, 'SIC', 2918, 1400, 2, 2923, 1423, 4 FROM dual
UNION ALL SELECT 3949, 'NAICS', 1562, 5259, 3, 1566, 52592, 4 FROM dual
UNION ALL SELECT 3950, 'SEC', 2650, 6020, 3, 2652, 6022, 4 FROM dual
UNION ALL SELECT 3951, 'NAICS', 1, 11, 1, 106, 113310, 5 FROM dual
UNION ALL SELECT 3952, 'SIC', 4308, 20, 1, 3323, 3271, 4 FROM dual
UNION ALL SELECT 3953, 'NAICS', 1041, 42443, 4, 1040, 424430, 5 FROM dual
UNION ALL SELECT 3954, 'NAICS', 1480, 52, 1, 1554, 525, 2 FROM dual
UNION ALL SELECT 3955, 'NAICS', 1850, 62, 1, 1902, 622310, 5 FROM dual
UNION ALL SELECT 3956, 'SIC', 4308, 20, 1, 3535, 3724, 4 FROM dual
UNION ALL SELECT 3957, 'SEC', 2791, 20, 1, 2348, 3060, 3 FROM dual
UNION ALL SELECT 3958, 'SEC', 2791, 20, 1, 2496, 3860, 3 FROM dual
UNION ALL SELECT 3959, 'SEC', 2766, 8100, 2, 2767, 8110, 3 FROM dual
UNION ALL SELECT 3960, 'NAICS', 1970, 7121, 3, 1973, 712120, 5 FROM dual
UNION ALL SELECT 3961, 'SIC', 4313, 70, 1, 4102, 7622, 4 FROM dual
UNION ALL SELECT 3962, 'NAICS', 1404, 5111, 3, 1407, 511120, 5 FROM dual
UNION ALL SELECT 3963, 'NAICS', 1624, 54, 1, 1629, 541120, 5 FROM dual
UNION ALL SELECT 3964, 'NAICS', 275, 23891, 4, 274, 238910, 5 FROM dual
UNION ALL SELECT 3965, 'NAICS', 2135, 92, 1, 2201, 9271, 3 FROM dual
UNION ALL SELECT 3966, 'SIC', 3527, 3710, 3, 3532, 3716, 4 FROM dual
UNION ALL SELECT 3967, 'SIC', 3371, 3400, 2, 3417, 3498, 4 FROM dual
UNION ALL SELECT 3968, 'NAICS', 1514, 5231, 3, 1521, 523140, 5 FROM dual
UNION ALL SELECT 3969, 'NAICS', 2093, 8129, 3, 2097, 812921, 5 FROM dual
UNION ALL SELECT 3970, 'SEC', 2302, 2700, 2, 2306, 2721, 4 FROM dual
UNION ALL SELECT 3971, 'NAICS', 1480, 52, 1, 1518, 52312, 4 FROM dual
UNION ALL SELECT 3972, 'SIC', 4308, 20, 1, 3322, 3270, 3 FROM dual
UNION ALL SELECT 3973, 'SEC', 2791, 20, 1, 2283, 2500, 2 FROM dual
UNION ALL SELECT 3974, 'SEC', 2793, 50, 1, 2597, 5141, 4 FROM dual
UNION ALL SELECT 3975, 'SIC', 3870, 5710, 3, 3871, 5712, 4 FROM dual
UNION ALL SELECT 3976, 'NAICS', 2135, 92, 1, 2193, 92612, 4 FROM dual
UNION ALL SELECT 3977, 'NAICS', 1, 11, 1, 15, 11116, 4 FROM dual
UNION ALL SELECT 3978, 'SEC', 2791, 20, 1, 2275, 2390, 3 FROM dual
UNION ALL SELECT 3979, 'SEC', 2796, 70, 1, 2765, 8093, 4 FROM dual
UNION ALL SELECT 3980, 'SIC', 4007, 7000, 2, 4008, 7010, 3 FROM dual
UNION ALL SELECT 3981, 'SIC', 3371, 3400, 2, 3402, 3471, 4 FROM dual
UNION ALL SELECT 3982, 'NAICS', 235, 238, 2, 257, 23822, 4 FROM dual
UNION ALL SELECT 3983, 'SIC', 4307, 15, 1, 2939, 1500, 2 FROM dual
UNION ALL SELECT 3984, 'SIC', 4313, 70, 1, 4094, 7537, 4 FROM dual
UNION ALL SELECT 3985, 'NAICS', 1018, 42411, 4, 1017, 424110, 5 FROM dual
UNION ALL SELECT 3986, 'SIC', 4308, 20, 1, 3264, 3020, 3 FROM dual
UNION ALL SELECT 3987, 'NAICS', 270, 23835, 4, 269, 238350, 5 FROM dual
UNION ALL SELECT 3988, 'NAICS', 1910, 62321, 4, 1909, 623210, 5 FROM dual
UNION ALL SELECT 3989, 'SIC', 4309, 40, 1, 3706, 4941, 4 FROM dual
UNION ALL SELECT 3990, 'SIC', 4305, 1, 1, 2866, 782, 4 FROM dual
UNION ALL SELECT 3991, 'SIC', 4310, 50, 1, 3722, 5021, 4 FROM dual
UNION ALL SELECT 3992, 'NAICS', 1990, 7139, 3, 1996, 71393, 4 FROM dual
UNION ALL SELECT 3993, 'NAICS', 1015, 424, 2, 1059, 424590, 5 FROM dual
UNION ALL SELECT 3994, 'SEC', 2553, 4910, 3, 2554, 4911, 4 FROM dual
UNION ALL SELECT 3995, 'SIC', 4308, 20, 1, 3345, 3321, 4 FROM dual
UNION ALL SELECT 3996, 'NAICS', 1969, 712, 2, 1977, 712190, 5 FROM dual
UNION ALL SELECT 3997, 'SIC', 3480, 3600, 2, 3506, 3652, 4 FROM dual
UNION ALL SELECT 3998, 'SIC', 3552, 3800, 2, 3562, 3827, 4 FROM dual
UNION ALL SELECT 3999, 'NAICS', 931, 423, 2, 1004, 4239, 3 FROM dual
UNION ALL SELECT 4000, 'NAICS', 40, 11142, 4, 41, 111421, 5 FROM dual
UNION ALL SELECT 4001, 'NAICS', 205, 23, 1, 262, 23831, 4 FROM dual
UNION ALL SELECT 4002, 'SIC', 3214, 2800, 2, 3249, 2893, 4 FROM dual
UNION ALL SELECT 4003, 'SIC', 3576, 3900, 2, 3582, 3931, 4 FROM dual
UNION ALL SELECT 4004, 'NAICS', 1625, 541, 2, 1639, 541219, 5 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 4005, 'NAICS', 2071, 812, 2, 2095, 81291, 4 FROM dual
UNION ALL SELECT 4006, 'NAICS', 931, 423, 2, 989, 423740, 5 FROM dual
UNION ALL SELECT 4007, 'SIC', 3214, 2800, 2, 3237, 2860, 3 FROM dual
UNION ALL SELECT 4008, 'SIC', 4307, 15, 1, 2958, 1720, 3 FROM dual
UNION ALL SELECT 4009, 'NAICS', 132, 21, 1, 160, 212313, 5 FROM dual
UNION ALL SELECT 4010, 'SEC', 2271, 2300, 2, 2272, 2320, 3 FROM dual
UNION ALL SELECT 4011, 'SIC', 4065, 7370, 3, 4067, 7372, 4 FROM dual
UNION ALL SELECT 4012, 'SIC', 3050, 2200, 2, 3065, 2258, 4 FROM dual
UNION ALL SELECT 4013, 'SEC', 2791, 20, 1, 2436, 3612, 4 FROM dual
UNION ALL SELECT 4014, 'NAICS', 23, 1113, 3, 35, 111339, 5 FROM dual
UNION ALL SELECT 4015, 'SEC', 2789, 10, 1, 2227, 1389, 4 FROM dual
UNION ALL SELECT 4016, 'SIC', 2918, 1400, 2, 2932, 1474, 4 FROM dual
UNION ALL SELECT 4017, 'SIC', 3504, 3650, 3, 3505, 3651, 4 FROM dual
UNION ALL SELECT 4018, 'NAICS', 1617, 53242, 4, 1616, 532420, 5 FROM dual
UNION ALL SELECT 4019, 'NAICS', 2204, 928, 2, 2209, 92812, 4 FROM dual
UNION ALL SELECT 4020, 'NAICS', 2189, 9261, 3, 2197, 92614, 4 FROM dual
UNION ALL SELECT 4021, 'SIC', 4153, 8040, 3, 4154, 8041, 4 FROM dual
UNION ALL SELECT 4022, 'NAICS', 930, 42, 1, 1013, 423990, 5 FROM dual
UNION ALL SELECT 4023, 'SIC', 3148, 2500, 2, 3162, 2541, 4 FROM dual
UNION ALL SELECT 4024, 'NAICS', 2197, 92614, 4, 2196, 926140, 5 FROM dual
UNION ALL SELECT 4025, 'NAICS', 204, 22133, 4, 203, 221330, 5 FROM dual
UNION ALL SELECT 4026, 'SEC', 2794, 52, 1, 2624, 5620, 3 FROM dual
UNION ALL SELECT 4027, 'NAICS', 1402, 51, 1, 1471, 5191, 3 FROM dual
UNION ALL SELECT 4028, 'NAICS', 268, 23834, 4, 267, 238340, 5 FROM dual
UNION ALL SELECT 4029, 'NAICS', 1625, 541, 2, 1656, 54138, 4 FROM dual
UNION ALL SELECT 4030, 'SIC', 4197, 8350, 3, 4198, 8351, 4 FROM dual
UNION ALL SELECT 4031, 'SIC', 4308, 20, 1, 3448, 3555, 4 FROM dual
UNION ALL SELECT 4032, 'SIC', 4313, 70, 1, 4085, 7515, 4 FROM dual
UNION ALL SELECT 4033, 'NAICS', 1071, 4248, 3, 1072, 424810, 5 FROM dual
UNION ALL SELECT 4034, 'NAICS', 2136, 921, 2, 2147, 92115, 4 FROM dual
UNION ALL SELECT 4035, 'SIC', 3419, 3500, 2, 3451, 3560, 3 FROM dual
UNION ALL SELECT 4036, 'NAICS', 2, 111, 2, 45, 11191, 4 FROM dual
UNION ALL SELECT 4037, 'SIC', 3917, 6000, 2, 3926, 6035, 4 FROM dual
UNION ALL SELECT 4038, 'NAICS', 1402, 51, 1, 1455, 51711, 4 FROM dual
UNION ALL SELECT 4039, 'NAICS', 1793, 5621, 3, 1796, 562112, 5 FROM dual
UNION ALL SELECT 4040, 'SIC', 3837, 5500, 2, 3838, 5510, 3 FROM dual
UNION ALL SELECT 4041, 'SIC', 2925, 1440, 3, 2926, 1442, 4 FROM dual
UNION ALL SELECT 4042, 'SIC', 4309, 40, 1, 3696, 4920, 3 FROM dual
UNION ALL SELECT 4043, 'SIC', 4309, 40, 1, 3650, 4491, 4 FROM dual
UNION ALL SELECT 4044, 'NAICS', 1580, 5312, 3, 1581, 531210, 5 FROM dual
UNION ALL SELECT 4045, 'NAICS', 945, 42322, 4, 944, 423220, 5 FROM dual
UNION ALL SELECT 4046, 'NAICS', 1054, 4245, 3, 1055, 424510, 5 FROM dual
UNION ALL SELECT 4047, 'NAICS', 1537, 5241, 3, 1543, 524127, 5 FROM dual
UNION ALL SELECT 4048, 'NAICS', 2171, 92312, 4, 2170, 923120, 5 FROM dual
UNION ALL SELECT 4049, 'NAICS', 1514, 5231, 3, 1516, 52311, 4 FROM dual
UNION ALL SELECT 4050, 'NAICS', 1673, 54161, 4, 1675, 541612, 5 FROM dual
UNION ALL SELECT 4051, 'SEC', 2791, 20, 1, 2312, 2750, 3 FROM dual
UNION ALL SELECT 4052, 'NAICS', 1792, 562, 2, 1802, 562213, 5 FROM dual
UNION ALL SELECT 4053, 'SEC', 2552, 4900, 2, 2554, 4911, 4 FROM dual
UNION ALL SELECT 4054, 'SEC', 2791, 20, 1, 4315, 3576, 4 FROM dual
UNION ALL SELECT 4055, 'SIC', 4308, 20, 1, 3352, 3339, 4 FROM dual
UNION ALL SELECT 4056, 'SIC', 4308, 20, 1, 3177, 2655, 4 FROM dual
UNION ALL SELECT 4057, 'SEC', 2501, 3910, 3, 2502, 3911, 4 FROM dual
UNION ALL SELECT 4058, 'SIC', 3837, 5500, 2, 3842, 5530, 3 FROM dual
UNION ALL SELECT 4059, 'SIC', 4313, 70, 1, 4113, 7800, 2 FROM dual
UNION ALL SELECT 4060, 'NAICS', 235, 238, 2, 242, 23813, 4 FROM dual
UNION ALL SELECT 4061, 'NAICS', 1442, 515, 2, 1447, 515120, 5 FROM dual
UNION ALL SELECT 4062, 'NAICS', 2, 111, 2, 49, 11193, 4 FROM dual
UNION ALL SELECT 4063, 'NAICS', 1566, 52592, 4, 1565, 525920, 5 FROM dual
UNION ALL SELECT 4064, 'SEC', 2796, 70, 1, 2781, 8741, 4 FROM dual
UNION ALL SELECT 4065, 'SIC', 4314, 90, 1, 4250, 9120, 3 FROM dual
UNION ALL SELECT 4066, 'SIC', 3837, 5500, 2, 3846, 5550, 3 FROM dual
UNION ALL SELECT 4067, 'NAICS', 1547, 5242, 3, 1548, 524210, 5 FROM dual
UNION ALL SELECT 4068, 'SIC', 3419, 3500, 2, 3473, 3589, 4 FROM dual
UNION ALL SELECT 4069, 'NAICS', 68, 1123, 3, 71, 112320, 5 FROM dual
UNION ALL SELECT 4070, 'SIC', 4313, 70, 1, 4241, 8800, 2 FROM dual
UNION ALL SELECT 4071, 'NAICS', 1990, 7139, 3, 2000, 71395, 4 FROM dual
UNION ALL SELECT 4072, 'NAICS', 1015, 424, 2, 1075, 42482, 4 FROM dual
UNION ALL SELECT 4073, 'SEC', 2791, 20, 1, 2305, 2720, 3 FROM dual
UNION ALL SELECT 4074, 'SIC', 3355, 3350, 3, 3357, 3353, 4 FROM dual
UNION ALL SELECT 4075, 'SIC', 4313, 70, 1, 4031, 7240, 3 FROM dual
UNION ALL SELECT 4076, 'SIC', 4305, 1, 1, 2871, 830, 3 FROM dual
UNION ALL SELECT 4077, 'NAICS', 1942, 71, 1, 1947, 711120, 5 FROM dual
UNION ALL SELECT 4078, 'NAICS', 1990, 7139, 3, 1999, 713950, 5 FROM dual
UNION ALL SELECT 4079, 'SIC', 3337, 3300, 2, 3356, 3351, 4 FROM dual
UNION ALL SELECT 4080, 'NAICS', 1032, 42433, 4, 1031, 424330, 5 FROM dual
UNION ALL SELECT 4081, 'NAICS', 1402, 51, 1, 1424, 51212, 4 FROM dual
UNION ALL SELECT 4082, 'NAICS', 2135, 92, 1, 2160, 922150, 5 FROM dual
UNION ALL SELECT 4083, 'SEC', 2620, 5500, 2, 2622, 5531, 4 FROM dual
UNION ALL SELECT 4084, 'NAICS', 1569, 53, 1, 1582, 53121, 4 FROM dual
UNION ALL SELECT 4085, 'NAICS', 198, 2213, 3, 202, 22132, 4 FROM dual
UNION ALL SELECT 4086, 'NAICS', 1850, 62, 1, 1900, 62221, 4 FROM dual
UNION ALL SELECT 4087, 'NAICS', 78, 11239, 4, 77, 112390, 5 FROM dual
UNION ALL SELECT 4088, 'NAICS', 1, 11, 1, 63, 112130, 5 FROM dual
UNION ALL SELECT 4089, 'SEC', 2791, 20, 1, 2406, 3523, 4 FROM dual
UNION ALL SELECT 4090, 'SIC', 3108, 2380, 3, 3109, 2381, 4 FROM dual
UNION ALL SELECT 4091, 'SIC', 4308, 20, 1, 3494, 3635, 4 FROM dual
UNION ALL SELECT 4092, 'SIC', 4308, 20, 1, 3512, 3671, 4 FROM dual
UNION ALL SELECT 4093, 'SIC', 4049, 7330, 3, 4052, 7335, 4 FROM dual
UNION ALL SELECT 4094, 'NAICS', 930, 42, 1, 1037, 42441, 4 FROM dual
UNION ALL SELECT 4095, 'NAICS', 1625, 541, 2, 1658, 541410, 5 FROM dual
UNION ALL SELECT 4096, 'SIC', 4308, 20, 1, 3405, 3482, 4 FROM dual
UNION ALL SELECT 4097, 'SEC', 2479, 3820, 3, 2482, 3823, 4 FROM dual
UNION ALL SELECT 4098, 'SIC', 3654, 4500, 2, 3656, 4512, 4 FROM dual
UNION ALL SELECT 4099, 'SIC', 4312, 60, 1, 3961, 6320, 3 FROM dual
UNION ALL SELECT 4100, 'SIC', 4312, 60, 1, 3917, 6000, 2 FROM dual
UNION ALL SELECT 4101, 'NAICS', 1657, 5414, 3, 1663, 54143, 4 FROM dual
UNION ALL SELECT 4102, 'NAICS', 2038, 811, 2, 2047, 811122, 5 FROM dual
UNION ALL SELECT 4103, 'SEC', 2795, 60, 1, 2701, 6550, 3 FROM dual
UNION ALL SELECT 4104, 'SIC', 4313, 70, 1, 4101, 7620, 3 FROM dual
UNION ALL SELECT 4105, 'SIC', 3480, 3600, 2, 3518, 3678, 4 FROM dual
UNION ALL SELECT 4106, 'SIC', 4040, 7300, 2, 4041, 7310, 3 FROM dual
UNION ALL SELECT 4107, 'SEC', 2792, 40, 1, 2552, 4900, 2 FROM dual
UNION ALL SELECT 4108, 'SIC', 4310, 50, 1, 3764, 5111, 4 FROM dual
UNION ALL SELECT 4109, 'SEC', 2396, 3450, 3, 2397, 3451, 4 FROM dual
UNION ALL SELECT 4110, 'SEC', 2792, 40, 1, 2544, 4822, 4 FROM dual
UNION ALL SELECT 4111, 'SEC', 2791, 20, 1, 2457, 3695, 4 FROM dual
UNION ALL SELECT 4112, 'SIC', 3480, 3600, 2, 3497, 3641, 4 FROM dual
UNION ALL SELECT 4113, 'SIC', 4308, 20, 1, 2993, 2030, 3 FROM dual
UNION ALL SELECT 4114, 'NAICS', 84, 1125, 3, 85, 11251, 4 FROM dual
UNION ALL SELECT 4115, 'NAICS', 2038, 811, 2, 2069, 811490, 5 FROM dual
UNION ALL SELECT 4116, 'SIC', 4311, 52, 1, 3819, 5331, 4 FROM dual
UNION ALL SELECT 4117, 'SIC', 4305, 1, 1, 2851, 721, 4 FROM dual
UNION ALL SELECT 4118, 'SIC', 3144, 2490, 3, 3146, 2493, 4 FROM dual
UNION ALL SELECT 4119, 'SIC', 4308, 20, 1, 3169, 2611, 4 FROM dual
UNION ALL SELECT 4120, 'SIC', 4235, 8740, 3, 4240, 8748, 4 FROM dual
UNION ALL SELECT 4121, 'NAICS', 1856, 6212, 3, 1858, 62121, 4 FROM dual
UNION ALL SELECT 4122, 'SIC', 4017, 7200, 2, 4029, 7230, 3 FROM dual
UNION ALL SELECT 4123, 'SEC', 2792, 40, 1, 2553, 4910, 3 FROM dual
UNION ALL SELECT 4124, 'SIC', 4313, 70, 1, 4168, 8072, 4 FROM dual
UNION ALL SELECT 4125, 'NAICS', 1725, 56, 1, 1789, 56192, 4 FROM dual
UNION ALL SELECT 4126, 'SEC', 2793, 50, 1, 2595, 5130, 3 FROM dual
UNION ALL SELECT 4127, 'SEC', 2649, 6000, 2, 2650, 6020, 3 FROM dual
UNION ALL SELECT 4128, 'SIC', 3215, 2810, 3, 3216, 2812, 4 FROM dual
UNION ALL SELECT 4129, 'NAICS', 1726, 561, 2, 1780, 56173, 4 FROM dual
UNION ALL SELECT 4130, 'SEC', 2591, 5100, 2, 2593, 5120, 3 FROM dual
UNION ALL SELECT 4131, 'NAICS', 1791, 56199, 4, 1790, 561990, 5 FROM dual
UNION ALL SELECT 4132, 'NAICS', 955, 4234, 3, 966, 423460, 5 FROM dual
UNION ALL SELECT 4133, 'SIC', 4313, 70, 1, 4206, 8420, 3 FROM dual
UNION ALL SELECT 4134, 'NAICS', 1004, 4239, 3, 1007, 423920, 5 FROM dual
UNION ALL SELECT 4135, 'SIC', 3252, 2900, 2, 3259, 2992, 4 FROM dual
UNION ALL SELECT 4136, 'SIC', 4310, 50, 1, 3740, 5060, 3 FROM dual
UNION ALL SELECT 4137, 'NAICS', 1850, 62, 1, 1931, 62422, 4 FROM dual
UNION ALL SELECT 4138, 'SIC', 3756, 5090, 3, 3761, 5099, 4 FROM dual
UNION ALL SELECT 4139, 'NAICS', 991, 4238, 3, 993, 42381, 4 FROM dual
UNION ALL SELECT 4140, 'NAICS', 1812, 61, 1, 1831, 61151, 4 FROM dual
UNION ALL SELECT 4141, 'NAICS', 2135, 92, 1, 2144, 921140, 5 FROM dual
UNION ALL SELECT 4142, 'NAICS', 135, 21111, 4, 137, 211112, 5 FROM dual
UNION ALL SELECT 4143, 'NAICS', 138, 212, 2, 146, 21221, 4 FROM dual
UNION ALL SELECT 4144, 'SEC', 2434, 3600, 2, 2457, 3695, 4 FROM dual
UNION ALL SELECT 4145, 'SIC', 4307, 15, 1, 2968, 1752, 4 FROM dual
UNION ALL SELECT 4146, 'SIC', 4147, 8010, 3, 4148, 8011, 4 FROM dual
UNION ALL SELECT 4147, 'SIC', 3804, 5200, 2, 3805, 5210, 3 FROM dual
UNION ALL SELECT 4148, 'NAICS', 930, 42, 1, 964, 423450, 5 FROM dual
UNION ALL SELECT 4149, 'SIC', 3337, 3300, 2, 3351, 3334, 4 FROM dual
UNION ALL SELECT 4150, 'SIC', 4308, 20, 1, 3328, 3280, 3 FROM dual
UNION ALL SELECT 4151, 'SIC', 4192, 8300, 2, 4202, 8399, 4 FROM dual
UNION ALL SELECT 4152, 'NAICS', 1, 11, 1, 35, 111339, 5 FROM dual
UNION ALL SELECT 4153, 'NAICS', 1591, 532, 2, 1607, 532292, 5 FROM dual
UNION ALL SELECT 4154, 'NAICS', 2167, 9231, 3, 2169, 92311, 4 FROM dual
UNION ALL SELECT 4155, 'NAICS', 1004, 4239, 3, 1011, 423940, 5 FROM dual
UNION ALL SELECT 4156, 'SIC', 3451, 3560, 3, 3453, 3562, 4 FROM dual
UNION ALL SELECT 4157, 'NAICS', 2072, 8121, 3, 2077, 81219, 4 FROM dual
UNION ALL SELECT 4158, 'SIC', 4309, 40, 1, 3656, 4512, 4 FROM dual
UNION ALL SELECT 4159, 'NAICS', 931, 423, 2, 932, 4231, 3 FROM dual
UNION ALL SELECT 4160, 'SIC', 2820, 180, 3, 2822, 182, 4 FROM dual
UNION ALL SELECT 4161, 'SEC', 2792, 40, 1, 2532, 4580, 3 FROM dual
UNION ALL SELECT 4162, 'NAICS', 1564, 52591, 4, 1563, 525910, 5 FROM dual
UNION ALL SELECT 4163, 'SIC', 2982, 2000, 2, 3033, 2090, 3 FROM dual
UNION ALL SELECT 4164, 'SIC', 3301, 3200, 2, 3333, 3295, 4 FROM dual
UNION ALL SELECT 4165, 'SIC', 3693, 4900, 2, 3710, 4959, 4 FROM dual
UNION ALL SELECT 4166, 'SIC', 4311, 52, 1, 3885, 5813, 4 FROM dual
UNION ALL SELECT 4167, 'SIC', 3762, 5100, 2, 3778, 5144, 4 FROM dual
UNION ALL SELECT 4168, 'SIC', 3115, 2390, 3, 3117, 2392, 4 FROM dual
UNION ALL SELECT 4169, 'SIC', 4308, 20, 1, 3039, 2098, 4 FROM dual
UNION ALL SELECT 4170, 'SIC', 4125, 7900, 2, 4136, 7950, 3 FROM dual
UNION ALL SELECT 4171, 'SIC', 2805, 130, 3, 2810, 139, 4 FROM dual
UNION ALL SELECT 4172, 'NAICS', 1812, 61, 1, 1838, 61161, 4 FROM dual
UNION ALL SELECT 4173, 'NAICS', 2037, 81, 1, 2107, 8132, 3 FROM dual
UNION ALL SELECT 4174, 'SEC', 2713, 7300, 2, 2719, 7340, 3 FROM dual
UNION ALL SELECT 4175, 'SIC', 3050, 2200, 2, 3061, 2252, 4 FROM dual
UNION ALL SELECT 4176, 'NAICS', 144, 2122, 3, 145, 212210, 5 FROM dual
UNION ALL SELECT 4177, 'SIC', 4308, 20, 1, 3166, 2599, 4 FROM dual
UNION ALL SELECT 4178, 'NAICS', 1871, 6214, 3, 1878, 621492, 5 FROM dual
UNION ALL SELECT 4179, 'SIC', 3763, 5110, 3, 3765, 5112, 4 FROM dual
UNION ALL SELECT 4180, 'SIC', 4313, 70, 1, 4152, 8031, 4 FROM dual
UNION ALL SELECT 4181, 'NAICS', 1640, 5413, 3, 1654, 54137, 4 FROM dual
UNION ALL SELECT 4182, 'SIC', 4284, 9600, 2, 4296, 9661, 4 FROM dual
UNION ALL SELECT 4183, 'NAICS', 1402, 51, 1, 1456, 5172, 3 FROM dual
UNION ALL SELECT 4184, 'NAICS', 1885, 6216, 3, 1887, 62161, 4 FROM dual
UNION ALL SELECT 4185, 'NAICS', 1789, 56192, 4, 1788, 561920, 5 FROM dual
UNION ALL SELECT 4186, 'SIC', 2982, 2000, 2, 3029, 2084, 4 FROM dual
UNION ALL SELECT 4187, 'NAICS', 2143, 92113, 4, 2142, 921130, 5 FROM dual
UNION ALL SELECT 4188, 'SEC', 2753, 8000, 2, 2761, 8071, 4 FROM dual
UNION ALL SELECT 4189, 'SIC', 2823, 190, 3, 2824, 191, 4 FROM dual
UNION ALL SELECT 4190, 'SIC', 3108, 2380, 3, 3110, 2384, 4 FROM dual
UNION ALL SELECT 4191, 'NAICS', 1725, 56, 1, 1741, 5614, 3 FROM dual
UNION ALL SELECT 4192, 'NAICS', 1917, 6239, 3, 1918, 623990, 5 FROM dual
UNION ALL SELECT 4193, 'NAICS', 1830, 6115, 3, 1831, 61151, 4 FROM dual
UNION ALL SELECT 4194, 'SEC', 2785, 9700, 2, 2786, 9720, 3 FROM dual
UNION ALL SELECT 4195, 'SIC', 3020, 2070, 3, 3021, 2074, 4 FROM dual
UNION ALL SELECT 4196, 'SEC', 2479, 3820, 3, 2487, 3829, 4 FROM dual
UNION ALL SELECT 4197, 'SIC', 4313, 70, 1, 4051, 7334, 4 FROM dual
UNION ALL SELECT 4198, 'SIC', 3108, 2380, 3, 3112, 2386, 4 FROM dual
UNION ALL SELECT 4199, 'SEC', 2791, 20, 1, 2322, 2821, 4 FROM dual
UNION ALL SELECT 4200, 'SIC', 3744, 5070, 3, 3746, 5074, 4 FROM dual
UNION ALL SELECT 4201, 'NAICS', 2021, 7223, 3, 2023, 72231, 4 FROM dual
UNION ALL SELECT 4202, 'NAICS', 1894, 622, 2, 1900, 62221, 4 FROM dual
UNION ALL SELECT 4203, 'SEC', 2795, 60, 1, 4320, 6170, 3 FROM dual
UNION ALL SELECT 4204, 'NAICS', 1, 11, 1, 121, 115112, 5 FROM dual
UNION ALL SELECT 4205, 'NAICS', 1624, 54, 1, 1651, 541360, 5 FROM dual
UNION ALL SELECT 4206, 'NAICS', 1904, 623, 2, 1917, 6239, 3 FROM dual
UNION ALL SELECT 4207, 'SEC', 2568, 5000, 2, 2587, 5084, 4 FROM dual
UNION ALL SELECT 4208, 'SIC', 3480, 3600, 2, 3498, 3643, 4 FROM dual
UNION ALL SELECT 4209, 'SIC', 3762, 5100, 2, 3800, 5193, 4 FROM dual
UNION ALL SELECT 4210, 'NAICS', 946, 4233, 3, 953, 423390, 5 FROM dual
UNION ALL SELECT 4211, 'SEC', 2774, 8700, 2, 2775, 8710, 3 FROM dual
UNION ALL SELECT 4212, 'SEC', 2434, 3600, 2, 2436, 3612, 4 FROM dual
UNION ALL SELECT 4213, 'SIC', 4308, 20, 1, 3547, 3769, 4 FROM dual
UNION ALL SELECT 4214, 'SIC', 3261, 3000, 2, 3265, 3021, 4 FROM dual
UNION ALL SELECT 4215, 'SIC', 4310, 50, 1, 3780, 5146, 4 FROM dual
UNION ALL SELECT 4216, 'NAICS', 1767, 56161, 4, 1769, 561612, 5 FROM dual
UNION ALL SELECT 4217, 'NAICS', 2004, 721, 2, 2012, 721199, 5 FROM dual
UNION ALL SELECT 4218, 'SEC', 2403, 3500, 2, 4315, 3576, 4 FROM dual
UNION ALL SELECT 4219, 'SIC', 4313, 70, 1, 4108, 7641, 4 FROM dual
UNION ALL SELECT 4220, 'SIC', 4309, 40, 1, 3632, 4231, 4 FROM dual
UNION ALL SELECT 4221, 'NAICS', 2021, 7223, 3, 2022, 722310, 5 FROM dual
UNION ALL SELECT 4222, 'SIC', 3115, 2390, 3, 3120, 2395, 4 FROM dual
UNION ALL SELECT 4223, 'NAICS', 138, 212, 2, 164, 212322, 5 FROM dual
UNION ALL SELECT 4224, 'SIC', 4307, 15, 1, 2978, 1794, 4 FROM dual
UNION ALL SELECT 4225, 'SIC', 4307, 15, 1, 2979, 1795, 4 FROM dual
UNION ALL SELECT 4226, 'SIC', 3225, 2830, 3, 3228, 2835, 4 FROM dual
UNION ALL SELECT 4227, 'SEC', 2403, 3500, 2, 2406, 3523, 4 FROM dual
UNION ALL SELECT 4228, 'SEC', 2371, 3300, 2, 2381, 3357, 4 FROM dual
UNION ALL SELECT 4229, 'NAICS', 2071, 812, 2, 2097, 812921, 5 FROM dual
UNION ALL SELECT 4230, 'NAICS', 235, 238, 2, 236, 2381, 3 FROM dual
UNION ALL SELECT 4231, 'SEC', 2292, 2600, 2, 2294, 2611, 4 FROM dual
UNION ALL SELECT 4232, 'NAICS', 1, 11, 1, 87, 112512, 5 FROM dual
UNION ALL SELECT 4233, 'SIC', 4308, 20, 1, 3066, 2259, 4 FROM dual
UNION ALL SELECT 4234, 'SIC', 4308, 20, 1, 3573, 3861, 4 FROM dual
UNION ALL SELECT 4235, 'SEC', 2413, 3540, 3, 2414, 3541, 4 FROM dual
UNION ALL SELECT 4236, 'NAICS', 1, 11, 1, 51, 11194, 4 FROM dual
UNION ALL SELECT 4237, 'SIC', 3511, 3670, 3, 3512, 3671, 4 FROM dual
UNION ALL SELECT 4238, 'NAICS', 1035, 4244, 3, 1036, 424410, 5 FROM dual
UNION ALL SELECT 4239, 'SIC', 3672, 4730, 3, 3673, 4731, 4 FROM dual
UNION ALL SELECT 4240, 'SIC', 4040, 7300, 2, 4063, 7361, 4 FROM dual
UNION ALL SELECT 4241, 'SEC', 2408, 3530, 3, 2410, 3532, 4 FROM dual
UNION ALL SELECT 4242, 'NAICS', 132, 21, 1, 143, 212113, 5 FROM dual
UNION ALL SELECT 4243, 'SEC', 2767, 8110, 3, 2768, 8111, 4 FROM dual
UNION ALL SELECT 4244, 'SIC', 3552, 3800, 2, 3570, 3850, 3 FROM dual
UNION ALL SELECT 4245, 'SIC', 4311, 52, 1, 3807, 5230, 3 FROM dual
UNION ALL SELECT 4246, 'SIC', 4312, 60, 1, 4001, 6733, 4 FROM dual
UNION ALL SELECT 4247, 'NAICS', 1976, 71213, 4, 1975, 712130, 5 FROM dual
UNION ALL SELECT 4248, 'SEC', 2240, 2000, 2, 2250, 2050, 3 FROM dual
UNION ALL SELECT 4249, 'SIC', 4031, 7240, 3, 4032, 7241, 4 FROM dual
UNION ALL SELECT 4250, 'NAICS', 138, 212, 2, 158, 212311, 5 FROM dual
UNION ALL SELECT 4251, 'NAICS', 217, 23622, 4, 216, 236220, 5 FROM dual
UNION ALL SELECT 4252, 'SIC', 2987, 2020, 3, 2990, 2023, 4 FROM dual
UNION ALL SELECT 4253, 'SIC', 4313, 70, 1, 4178, 8200, 2 FROM dual
UNION ALL SELECT 4254, 'SIC', 3850, 5570, 3, 3851, 5571, 4 FROM dual
UNION ALL SELECT 4255, 'SIC', 4310, 50, 1, 3721, 5020, 3 FROM dual
UNION ALL SELECT 4256, 'SIC', 4311, 52, 1, 3813, 5270, 3 FROM dual
UNION ALL SELECT 4257, 'NAICS', 930, 42, 1, 1086, 42495, 4 FROM dual
UNION ALL SELECT 4258, 'SEC', 2617, 5400, 2, 4319, 5412, 4 FROM dual
UNION ALL SELECT 4259, 'NAICS', 1015, 424, 2, 1041, 42443, 4 FROM dual
UNION ALL SELECT 4260, 'SIC', 3083, 2300, 2, 3122, 2397, 4 FROM dual
UNION ALL SELECT 4261, 'NAICS', 2020, 722, 2, 2032, 72251, 4 FROM dual
UNION ALL SELECT 4262, 'NAICS', 2103, 813, 2, 2127, 813940, 5 FROM dual
UNION ALL SELECT 4263, 'SIC', 3093, 2330, 3, 3094, 2331, 4 FROM dual
UNION ALL SELECT 4264, 'NAICS', 1402, 51, 1, 1446, 515112, 5 FROM dual
UNION ALL SELECT 4265, 'SEC', 2271, 2300, 2, 2273, 2330, 3 FROM dual
UNION ALL SELECT 4266, 'SIC', 4002, 6790, 3, 4003, 6792, 4 FROM dual
UNION ALL SELECT 4267, 'SIC', 3762, 5100, 2, 3765, 5112, 4 FROM dual
UNION ALL SELECT 4268, 'SIC', 2982, 2000, 2, 3034, 2091, 4 FROM dual
UNION ALL SELECT 4269, 'NAICS', 1047, 42446, 4, 1046, 424460, 5 FROM dual
UNION ALL SELECT 4270, 'SIC', 3362, 3360, 3, 3363, 3363, 4 FROM dual
UNION ALL SELECT 4271, 'NAICS', 930, 42, 1, 947, 423310, 5 FROM dual
UNION ALL SELECT 4272, 'NAICS', 139, 2121, 3, 140, 21211, 4 FROM dual
UNION ALL SELECT 4273, 'SIC', 3395, 3460, 3, 3400, 3469, 4 FROM dual
UNION ALL SELECT 4274, 'SIC', 3762, 5100, 2, 3783, 5149, 4 FROM dual
UNION ALL SELECT 4275, 'SIC', 4040, 7300, 2, 4075, 7380, 3 FROM dual
UNION ALL SELECT 4276, 'NAICS', 1481, 521, 2, 1483, 521110, 5 FROM dual
UNION ALL SELECT 4277, 'NAICS', 1726, 561, 2, 1757, 561499, 5 FROM dual
UNION ALL SELECT 4278, 'NAICS', 2037, 81, 1, 2041, 811111, 5 FROM dual
UNION ALL SELECT 4279, 'SIC', 3621, 4210, 3, 3623, 4213, 4 FROM dual
UNION ALL SELECT 4280, 'SIC', 4310, 50, 1, 3719, 5014, 4 FROM dual
UNION ALL SELECT 4281, 'NAICS', 1830, 6115, 3, 1832, 611511, 5 FROM dual
UNION ALL SELECT 4282, 'SIC', 4313, 70, 1, 4078, 7383, 4 FROM dual
UNION ALL SELECT 4283, 'NAICS', 1766, 5616, 3, 1771, 56162, 4 FROM dual
UNION ALL SELECT 4284, 'SIC', 3620, 4200, 2, 3629, 4225, 4 FROM dual
UNION ALL SELECT 4285, 'NAICS', 1624, 54, 1, 1670, 541513, 5 FROM dual
UNION ALL SELECT 4286, 'SEC', 2795, 60, 1, 2659, 6100, 2 FROM dual
UNION ALL SELECT 4287, 'NAICS', 1894, 622, 2, 1895, 6221, 3 FROM dual
UNION ALL SELECT 4288, 'SIC', 3555, 3820, 3, 3559, 3824, 4 FROM dual
UNION ALL SELECT 4289, 'NAICS', 156, 2123, 3, 158, 212311, 5 FROM dual
UNION ALL SELECT 4290, 'SEC', 2731, 7380, 3, 2733, 7384, 4 FROM dual
UNION ALL SELECT 4291, 'SIC', 4308, 20, 1, 3286, 3131, 4 FROM dual
UNION ALL SELECT 4292, 'SEC', 2259, 2100, 2, 2261, 2111, 4 FROM dual
UNION ALL SELECT 4293, 'SEC', 2795, 60, 1, 2678, 6310, 3 FROM dual
UNION ALL SELECT 4294, 'SIC', 4308, 20, 1, 3308, 3231, 4 FROM dual
UNION ALL SELECT 4295, 'NAICS', 19, 1112, 3, 20, 11121, 4 FROM dual
UNION ALL SELECT 4296, 'NAICS', 1404, 5111, 3, 1410, 51113, 4 FROM dual
UNION ALL SELECT 4297, 'SIC', 3371, 3400, 2, 3380, 3430, 3 FROM dual
UNION ALL SELECT 4298, 'SIC', 4311, 52, 1, 3854, 5600, 2 FROM dual
UNION ALL SELECT 4299, 'SIC', 4305, 1, 1, 2822, 182, 4 FROM dual
UNION ALL SELECT 4300, 'NAICS', 930, 42, 1, 1094, 42512, 4 FROM dual
UNION ALL SELECT 4301, 'SIC', 3409, 3490, 3, 3416, 3497, 4 FROM dual
UNION ALL SELECT 4302, 'NAICS', 180, 22, 1, 203, 221330, 5 FROM dual
UNION ALL SELECT 4303, 'NAICS', 2000, 71395, 4, 1999, 713950, 5 FROM dual
UNION ALL SELECT 4304, 'NAICS', 1823, 6114, 3, 1826, 611420, 5 FROM dual
UNION ALL SELECT 4305, 'SIC', 2955, 1700, 2, 2963, 1741, 4 FROM dual
UNION ALL SELECT 4306, 'NAICS', 1625, 541, 2, 1698, 541850, 5 FROM dual
UNION ALL SELECT 4307, 'SIC', 4308, 20, 1, 3044, 2120, 3 FROM dual
UNION ALL SELECT 4308, 'NAICS', 2037, 81, 1, 2077, 81219, 4 FROM dual
UNION ALL SELECT 4309, 'NAICS', 2082, 81221, 4, 2081, 812210, 5 FROM dual
UNION ALL SELECT 4310, 'SEC', 2756, 8050, 3, 2757, 8051, 4 FROM dual
UNION ALL SELECT 4311, 'NAICS', 1944, 7111, 3, 1948, 71112, 4 FROM dual
UNION ALL SELECT 4312, 'SEC', 2762, 8080, 3, 2763, 8082, 4 FROM dual
UNION ALL SELECT 4313, 'SIC', 4308, 20, 1, 3063, 2254, 4 FROM dual
UNION ALL SELECT 4314, 'NAICS', 1944, 7111, 3, 1951, 711190, 5 FROM dual
UNION ALL SELECT 4315, 'SIC', 3911, 5990, 3, 3916, 5999, 4 FROM dual
UNION ALL SELECT 4316, 'SIC', 3945, 6160, 3, 3946, 6162, 4 FROM dual
UNION ALL SELECT 4317, 'SIC', 4312, 60, 1, 3930, 6062, 4 FROM dual
UNION ALL SELECT 4318, 'SEC', 2659, 6100, 2, 2666, 6159, 4 FROM dual
UNION ALL SELECT 4319, 'NAICS', 205, 23, 1, 206, 236, 2 FROM dual
UNION ALL SELECT 4320, 'NAICS', 2139, 92111, 4, 2138, 921110, 5 FROM dual
UNION ALL SELECT 4321, 'SIC', 4017, 7200, 2, 4035, 7260, 3 FROM dual
UNION ALL SELECT 4322, 'NAICS', 156, 2123, 3, 164, 212322, 5 FROM dual
UNION ALL SELECT 4323, 'NAICS', 1485, 522, 2, 1499, 52222, 4 FROM dual
UNION ALL SELECT 4324, 'SEC', 2353, 3100, 2, 2354, 3140, 3 FROM dual
UNION ALL SELECT 4325, 'SIC', 3272, 3080, 3, 3276, 3084, 4 FROM dual
UNION ALL SELECT 4326, 'SEC', 2591, 5100, 2, 2603, 5180, 3 FROM dual
UNION ALL SELECT 4327, 'SIC', 3715, 5000, 2, 3749, 5080, 3 FROM dual
UNION ALL SELECT 4328, 'SIC', 4097, 7540, 3, 4099, 7549, 4 FROM dual
UNION ALL SELECT 4329, 'SIC', 4273, 9440, 3, 4274, 9441, 4 FROM dual
UNION ALL SELECT 4330, 'SIC', 4125, 7900, 2, 4138, 7952, 4 FROM dual
UNION ALL SELECT 4331, 'NAICS', 183, 22111, 4, 184, 221111, 5 FROM dual
UNION ALL SELECT 4332, 'NAICS', 1522, 52314, 4, 1521, 523140, 5 FROM dual
UNION ALL SELECT 4333, 'SIC', 4308, 20, 1, 2991, 2024, 4 FROM dual
UNION ALL SELECT 4334, 'SEC', 2796, 70, 1, 2726, 7371, 4 FROM dual
UNION ALL SELECT 4335, 'SIC', 4308, 20, 1, 3051, 2210, 3 FROM dual
UNION ALL SELECT 4336, 'NAICS', 1897, 62211, 4, 1896, 622110, 5 FROM dual
UNION ALL SELECT 4337, 'NAICS', 144, 2122, 3, 151, 212231, 5 FROM dual
UNION ALL SELECT 4338, 'NAICS', 1554, 525, 2, 1562, 5259, 3 FROM dual
UNION ALL SELECT 4339, 'SIC', 4308, 20, 1, 3003, 2044, 4 FROM dual
UNION ALL SELECT 4340, 'SIC', 3008, 2050, 3, 3009, 2051, 4 FROM dual
UNION ALL SELECT 4341, 'SIC', 4307, 15, 1, 2952, 1622, 4 FROM dual
UNION ALL SELECT 4342, 'SIC', 4308, 20, 1, 3353, 3340, 3 FROM dual
UNION ALL SELECT 4343, 'SIC', 3857, 5620, 3, 3858, 5621, 4 FROM dual
UNION ALL SELECT 4344, 'NAICS', 1812, 61, 1, 1832, 611511, 5 FROM dual
UNION ALL SELECT 4345, 'SIC', 4309, 40, 1, 3644, 4449, 4 FROM dual
UNION ALL SELECT 4346, 'NAICS', 2166, 923, 2, 2172, 923130, 5 FROM dual
UNION ALL SELECT 4347, 'SIC', 2982, 2000, 2, 2992, 2026, 4 FROM dual
UNION ALL SELECT 4348, 'SIC', 3693, 4900, 2, 3702, 4931, 4 FROM dual
UNION ALL SELECT 4349, 'SIC', 3059, 2250, 3, 3061, 2252, 4 FROM dual
UNION ALL SELECT 4350, 'SIC', 4313, 70, 1, 4034, 7251, 4 FROM dual
UNION ALL SELECT 4351, 'NAICS', 1850, 62, 1, 1906, 623110, 5 FROM dual
UNION ALL SELECT 4352, 'NAICS', 2, 111, 2, 6, 111120, 5 FROM dual
UNION ALL SELECT 4353, 'NAICS', 931, 423, 2, 940, 42314, 4 FROM dual
UNION ALL SELECT 4354, 'NAICS', 2053, 81121, 4, 2057, 811219, 5 FROM dual
UNION ALL SELECT 4355, 'NAICS', 2136, 921, 2, 2145, 92114, 4 FROM dual
UNION ALL SELECT 4356, 'SEC', 2355, 3200, 2, 2369, 3281, 4 FROM dual
UNION ALL SELECT 4357, 'SEC', 2791, 20, 1, 2333, 2860, 3 FROM dual
UNION ALL SELECT 4358, 'NAICS', 1813, 611, 2, 1814, 6111, 3 FROM dual
UNION ALL SELECT 4359, 'NAICS', 2, 111, 2, 9, 11113, 4 FROM dual
UNION ALL SELECT 4360, 'NAICS', 930, 42, 1, 1067, 424710, 5 FROM dual
UNION ALL SELECT 4361, 'SEC', 2786, 9720, 3, 2787, 9721, 4 FROM dual
UNION ALL SELECT 4362, 'SIC', 3555, 3820, 3, 3557, 3822, 4 FROM dual
UNION ALL SELECT 4363, 'SIC', 4305, 1, 1, 2881, 921, 4 FROM dual
UNION ALL SELECT 4364, 'SIC', 4310, 50, 1, 3750, 5082, 4 FROM dual
UNION ALL SELECT 4365, 'NAICS', 74, 11233, 4, 73, 112330, 5 FROM dual
UNION ALL SELECT 4366, 'NAICS', 181, 221, 2, 202, 22132, 4 FROM dual
UNION ALL SELECT 4367, 'NAICS', 2020, 722, 2, 2031, 7225, 3 FROM dual
UNION ALL SELECT 4368, 'SIC', 2993, 2030, 3, 2997, 2035, 4 FROM dual
UNION ALL SELECT 4369, 'NAICS', 2, 111, 2, 41, 111421, 5 FROM dual
UNION ALL SELECT 4370, 'NAICS', 1726, 561, 2, 1783, 561790, 5 FROM dual
UNION ALL SELECT 4371, 'NAICS', 1, 11, 1, 86, 112511, 5 FROM dual
UNION ALL SELECT 4372, 'SIC', 3636, 4400, 2, 3653, 4499, 4 FROM dual
UNION ALL SELECT 4373, 'SIC', 4305, 1, 1, 2872, 831, 4 FROM dual
UNION ALL SELECT 4374, 'SIC', 3674, 4740, 3, 3675, 4741, 4 FROM dual
UNION ALL SELECT 4375, 'SIC', 4310, 50, 1, 3786, 5154, 4 FROM dual
UNION ALL SELECT 4376, 'NAICS', 1980, 7131, 3, 1982, 71311, 4 FROM dual
UNION ALL SELECT 4377, 'NAICS', 205, 23, 1, 219, 2371, 3 FROM dual
UNION ALL SELECT 4378, 'NAICS', 1514, 5231, 3, 1519, 523130, 5 FROM dual
UNION ALL SELECT 4379, 'SIC', 4146, 8000, 2, 4166, 8070, 3 FROM dual
UNION ALL SELECT 4380, 'SEC', 2794, 52, 1, 2640, 5900, 2 FROM dual
UNION ALL SELECT 4381, 'SIC', 4308, 20, 1, 3413, 3494, 4 FROM dual
UNION ALL SELECT 4382, 'NAICS', 1942, 71, 1, 1979, 713, 2 FROM dual
UNION ALL SELECT 4383, 'NAICS', 2005, 7211, 3, 2012, 721199, 5 FROM dual
UNION ALL SELECT 4384, 'NAICS', 2135, 92, 1, 2163, 92216, 4 FROM dual
UNION ALL SELECT 4385, 'SEC', 2408, 3530, 3, 2412, 3537, 4 FROM dual
UNION ALL SELECT 4386, 'NAICS', 1725, 56, 1, 1781, 561740, 5 FROM dual
UNION ALL SELECT 4387, 'SIC', 4178, 8200, 2, 4182, 8221, 4 FROM dual
UNION ALL SELECT 4388, 'SIC', 4040, 7300, 2, 4043, 7312, 4 FROM dual
UNION ALL SELECT 4389, 'NAICS', 2182, 925, 2, 2187, 92512, 4 FROM dual
UNION ALL SELECT 4390, 'SIC', 3149, 2510, 3, 3155, 2519, 4 FROM dual
UNION ALL SELECT 4391, 'NAICS', 205, 23, 1, 273, 2389, 3 FROM dual
UNION ALL SELECT 4392, 'SEC', 2302, 2700, 2, 2315, 2770, 3 FROM dual
UNION ALL SELECT 4393, 'NAICS', 1758, 5615, 3, 1763, 56159, 4 FROM dual
UNION ALL SELECT 4394, 'SIC', 3729, 5040, 3, 3732, 5045, 4 FROM dual
UNION ALL SELECT 4395, 'NAICS', 205, 23, 1, 210, 236116, 5 FROM dual
UNION ALL SELECT 4396, 'NAICS', 1480, 52, 1, 1510, 52232, 4 FROM dual
UNION ALL SELECT 4397, 'NAICS', 1979, 713, 2, 1985, 7132, 3 FROM dual
UNION ALL SELECT 4398, 'SEC', 2713, 7300, 2, 2729, 7374, 4 FROM dual
UNION ALL SELECT 4399, 'SIC', 4259, 9220, 3, 4262, 9223, 4 FROM dual
UNION ALL SELECT 4400, 'SEC', 2424, 3570, 3, 2429, 3578, 4 FROM dual
UNION ALL SELECT 4401, 'NAICS', 180, 22, 1, 194, 221122, 5 FROM dual
UNION ALL SELECT 4402, 'SIC', 3804, 5200, 2, 3810, 5251, 4 FROM dual
UNION ALL SELECT 4403, 'NAICS', 1624, 54, 1, 1713, 54193, 4 FROM dual
UNION ALL SELECT 4404, 'NAICS', 167, 21239, 4, 170, 212393, 5 FROM dual
UNION ALL SELECT 4405, 'SIC', 2982, 2000, 2, 3012, 2060, 3 FROM dual
UNION ALL SELECT 4406, 'NAICS', 955, 4234, 3, 969, 42349, 4 FROM dual
UNION ALL SELECT 4407, 'SIC', 3191, 2710, 3, 3192, 2711, 4 FROM dual
UNION ALL SELECT 4408, 'SIC', 4308, 20, 1, 3420, 3510, 3 FROM dual
UNION ALL SELECT 4409, 'SIC', 4308, 20, 1, 3347, 3324, 4 FROM dual
UNION ALL SELECT 4410, 'SIC', 3301, 3200, 2, 3314, 3255, 4 FROM dual
UNION ALL SELECT 4411, 'SIC', 4313, 70, 1, 4183, 8222, 4 FROM dual
UNION ALL SELECT 4412, 'NAICS', 931, 423, 2, 988, 42373, 4 FROM dual
UNION ALL SELECT 4413, 'NAICS', 1657, 5414, 3, 1662, 541430, 5 FROM dual
UNION ALL SELECT 4414, 'SIC', 4308, 20, 1, 3077, 2290, 3 FROM dual
UNION ALL SELECT 4415, 'NAICS', 1020, 42412, 4, 1019, 424120, 5 FROM dual
UNION ALL SELECT 4416, 'SIC', 3992, 6700, 2, 4003, 6792, 4 FROM dual
UNION ALL SELECT 4417, 'SIC', 3854, 5600, 2, 3867, 5690, 3 FROM dual
UNION ALL SELECT 4418, 'NAICS', 1640, 5413, 3, 1650, 54135, 4 FROM dual
UNION ALL SELECT 4419, 'NAICS', 2004, 721, 2, 2019, 72131, 4 FROM dual
UNION ALL SELECT 4420, 'SIC', 4314, 90, 1, 4302, 9900, 2 FROM dual
UNION ALL SELECT 4421, 'NAICS', 1640, 5413, 3, 1655, 541380, 5 FROM dual
UNION ALL SELECT 4422, 'NAICS', 1, 11, 1, 53, 111991, 5 FROM dual
UNION ALL SELECT 4423, 'SEC', 2791, 20, 1, 2349, 3080, 3 FROM dual
UNION ALL SELECT 4424, 'NAICS', 1569, 53, 1, 1600, 53221, 4 FROM dual
UNION ALL SELECT 4425, 'NAICS', 1836, 6116, 3, 1846, 611699, 5 FROM dual
UNION ALL SELECT 4426, 'SEC', 2476, 3800, 2, 2494, 3850, 3 FROM dual
UNION ALL SELECT 4427, 'NAICS', 2037, 81, 1, 2043, 811113, 5 FROM dual
UNION ALL SELECT 4428, 'NAICS', 1625, 541, 2, 1632, 541191, 5 FROM dual
UNION ALL SELECT 4429, 'SIC', 4308, 20, 1, 3374, 3412, 4 FROM dual
UNION ALL SELECT 4430, 'SIC', 4017, 7200, 2, 4028, 7221, 4 FROM dual
UNION ALL SELECT 4431, 'SIC', 3958, 6300, 2, 3964, 6330, 3 FROM dual
UNION ALL SELECT 4432, 'NAICS', 235, 238, 2, 240, 23812, 4 FROM dual
UNION ALL SELECT 4433, 'NAICS', 2027, 72233, 4, 2026, 722330, 5 FROM dual
UNION ALL SELECT 4434, 'SEC', 2792, 40, 1, 2520, 4220, 3 FROM dual
UNION ALL SELECT 4435, 'SIC', 4310, 50, 1, 3766, 5113, 4 FROM dual
UNION ALL SELECT 4436, 'SEC', 2283, 2500, 2, 2289, 2531, 4 FROM dual
UNION ALL SELECT 4437, 'SIC', 3050, 2200, 2, 3068, 2261, 4 FROM dual
UNION ALL SELECT 4438, 'SEC', 2794, 52, 1, 2626, 5650, 3 FROM dual
UNION ALL SELECT 4439, 'SIC', 4308, 20, 1, 3215, 2810, 3 FROM dual
UNION ALL SELECT 4440, 'SIC', 3762, 5100, 2, 3774, 5140, 3 FROM dual
UNION ALL SELECT 4441, 'SIC', 3613, 4140, 3, 3615, 4142, 4 FROM dual
UNION ALL SELECT 4442, 'NAICS', 1851, 621, 2, 1869, 621391, 5 FROM dual
UNION ALL SELECT 4443, 'SEC', 2791, 20, 1, 2343, 3010, 3 FROM dual
UNION ALL SELECT 4444, 'NAICS', 2070, 81149, 4, 2069, 811490, 5 FROM dual
UNION ALL SELECT 4445, 'SIC', 4310, 50, 1, 3802, 5198, 4 FROM dual
UNION ALL SELECT 4446, 'NAICS', 982, 4237, 3, 983, 423710, 5 FROM dual
UNION ALL SELECT 4447, 'SEC', 2659, 6100, 2, 2662, 6140, 3 FROM dual
UNION ALL SELECT 4448, 'SIC', 3419, 3500, 2, 3469, 3581, 4 FROM dual
UNION ALL SELECT 4449, 'SIC', 4109, 7690, 3, 4110, 7692, 4 FROM dual
UNION ALL SELECT 4450, 'NAICS', 2135, 92, 1, 2137, 9211, 3 FROM dual
UNION ALL SELECT 4451, 'SIC', 3496, 3640, 3, 3499, 3644, 4 FROM dual
UNION ALL SELECT 4452, 'SEC', 2796, 70, 1, 4329, 7385, 4 FROM dual
UNION ALL SELECT 4453, 'SIC', 4007, 7000, 2, 4014, 7033, 4 FROM dual
UNION ALL SELECT 4454, 'SEC', 2788, 1, 1, 2213, 800, 2 FROM dual
UNION ALL SELECT 4455, 'SIC', 2825, 200, 2, 2845, 290, 3 FROM dual
UNION ALL SELECT 4456, 'SEC', 2791, 20, 1, 2338, 2910, 3 FROM dual
UNION ALL SELECT 4457, 'SIC', 4308, 20, 1, 3080, 2297, 4 FROM dual
UNION ALL SELECT 4458, 'SIC', 4284, 9600, 2, 4286, 9611, 4 FROM dual
UNION ALL SELECT 4459, 'SIC', 3928, 6060, 3, 3929, 6061, 4 FROM dual
UNION ALL SELECT 4460, 'SIC', 3093, 2330, 3, 3096, 2337, 4 FROM dual
UNION ALL SELECT 4461, 'NAICS', 173, 2131, 3, 178, 213114, 5 FROM dual
UNION ALL SELECT 4462, 'NAICS', 1741, 5614, 3, 1751, 56144, 4 FROM dual
UNION ALL SELECT 4463, 'NAICS', 1859, 6213, 3, 1867, 62134, 4 FROM dual
UNION ALL SELECT 4464, 'SEC', 2791, 20, 1, 2372, 3310, 3 FROM dual
UNION ALL SELECT 4465, 'SEC', 2753, 8000, 2, 2755, 8011, 4 FROM dual
UNION ALL SELECT 4466, 'NAICS', 1015, 424, 2, 1024, 424210, 5 FROM dual
UNION ALL SELECT 4467, 'SIC', 3337, 3300, 2, 3342, 3316, 4 FROM dual
UNION ALL SELECT 4468, 'SIC', 4101, 7620, 3, 4102, 7622, 4 FROM dual
UNION ALL SELECT 4469, 'NAICS', 235, 238, 2, 271, 238390, 5 FROM dual
UNION ALL SELECT 4470, 'NAICS', 1570, 531, 2, 1583, 5313, 3 FROM dual
UNION ALL SELECT 4471, 'SIC', 3715, 5000, 2, 3730, 5043, 4 FROM dual
UNION ALL SELECT 4472, 'SIC', 4252, 9130, 3, 4253, 9131, 4 FROM dual
UNION ALL SELECT 4473, 'NAICS', 1942, 71, 1, 1970, 7121, 3 FROM dual
UNION ALL SELECT 4474, 'NAICS', 2037, 81, 1, 2100, 81293, 4 FROM dual
UNION ALL SELECT 4475, 'SIC', 3124, 2400, 2, 3139, 2448, 4 FROM dual
UNION ALL SELECT 4476, 'SIC', 2982, 2000, 2, 3026, 2080, 3 FROM dual
UNION ALL SELECT 4477, 'NAICS', 1624, 54, 1, 1677, 541614, 5 FROM dual
UNION ALL SELECT 4478, 'NAICS', 1725, 56, 1, 1740, 56133, 4 FROM dual
UNION ALL SELECT 4479, 'SIC', 4308, 20, 1, 3421, 3511, 4 FROM dual
UNION ALL SELECT 4480, 'NAICS', 1798, 5622, 3, 1799, 56221, 4 FROM dual
UNION ALL SELECT 4481, 'NAICS', 930, 42, 1, 985, 423720, 5 FROM dual
UNION ALL SELECT 4482, 'SIC', 3059, 2250, 3, 3065, 2258, 4 FROM dual
UNION ALL SELECT 4483, 'NAICS', 1485, 522, 2, 1498, 522220, 5 FROM dual
UNION ALL SELECT 4484, 'NAICS', 1666, 5415, 3, 1668, 541511, 5 FROM dual
UNION ALL SELECT 4485, 'NAICS', 1774, 5617, 3, 1782, 56174, 4 FROM dual
UNION ALL SELECT 4486, 'NAICS', 2038, 811, 2, 2065, 811420, 5 FROM dual
UNION ALL SELECT 4487, 'SIC', 4312, 60, 1, 3977, 6500, 2 FROM dual
UNION ALL SELECT 4488, 'NAICS', 1689, 5418, 3, 1698, 541850, 5 FROM dual
UNION ALL SELECT 4489, 'NAICS', 1625, 541, 2, 1647, 541340, 5 FROM dual
UNION ALL SELECT 4490, 'NAICS', 1851, 621, 2, 1862, 621320, 5 FROM dual
UNION ALL SELECT 4491, 'SIC', 3115, 2390, 3, 3118, 2393, 4 FROM dual
UNION ALL SELECT 4492, 'NAICS', 1402, 51, 1, 1404, 5111, 3 FROM dual
UNION ALL SELECT 4493, 'NAICS', 930, 42, 1, 993, 42381, 4 FROM dual
UNION ALL SELECT 4494, 'NAICS', 1960, 71131, 4, 1959, 711310, 5 FROM dual
UNION ALL SELECT 4495, 'SEC', 2446, 3660, 3, 2448, 3663, 4 FROM dual
UNION ALL SELECT 4496, 'NAICS', 1, 11, 1, 54, 111992, 5 FROM dual
UNION ALL SELECT 4497, 'NAICS', 1570, 531, 2, 1579, 53119, 4 FROM dual
UNION ALL SELECT 4498, 'SIC', 4256, 9200, 2, 4262, 9223, 4 FROM dual
UNION ALL SELECT 4499, 'NAICS', 67, 11221, 4, 66, 112210, 5 FROM dual
UNION ALL SELECT 4500, 'NAICS', 930, 42, 1, 939, 423140, 5 FROM dual
UNION ALL SELECT 4501, 'NAICS', 1, 11, 1, 103, 113210, 5 FROM dual
UNION ALL SELECT 4502, 'SEC', 2789, 10, 1, 2226, 1382, 4 FROM dual
UNION ALL SELECT 4503, 'SIC', 4040, 7300, 2, 4044, 7313, 4 FROM dual
UNION ALL SELECT 4504, 'SIC', 2825, 200, 2, 2841, 271, 4 FROM dual
UNION ALL SELECT 4505, 'NAICS', 2037, 81, 1, 2089, 81232, 4 FROM dual
UNION ALL SELECT 4506, 'NAICS', 2017, 7213, 3, 2018, 721310, 5 FROM dual
UNION ALL SELECT 4507, 'NAICS', 1726, 561, 2, 1766, 5616, 3 FROM dual
UNION ALL SELECT 4508, 'NAICS', 2037, 81, 1, 2117, 8134, 3 FROM dual
UNION ALL SELECT 4509, 'SIC', 2805, 130, 3, 2809, 134, 4 FROM dual
UNION ALL SELECT 4510, 'NAICS', 1725, 56, 1, 1795, 562111, 5 FROM dual
UNION ALL SELECT 4511, 'SIC', 4308, 20, 1, 3295, 3161, 4 FROM dual
UNION ALL SELECT 4512, 'SIC', 3762, 5100, 2, 3770, 5131, 4 FROM dual
UNION ALL SELECT 4513, 'SIC', 4311, 52, 1, 3820, 5390, 3 FROM dual
UNION ALL SELECT 4514, 'SIC', 3587, 3950, 3, 3590, 3953, 4 FROM dual
UNION ALL SELECT 4515, 'NAICS', 1850, 62, 1, 1890, 62191, 4 FROM dual
UNION ALL SELECT 4516, 'NAICS', 2003, 72, 1, 2027, 72233, 4 FROM dual
UNION ALL SELECT 4517, 'SIC', 2982, 2000, 2, 3027, 2082, 4 FROM dual
UNION ALL SELECT 4518, 'SIC', 3206, 2770, 3, 3207, 2771, 4 FROM dual
UNION ALL SELECT 4519, 'SIC', 4312, 60, 1, 3967, 6351, 4 FROM dual
UNION ALL SELECT 4520, 'SIC', 4311, 52, 1, 3867, 5690, 3 FROM dual
UNION ALL SELECT 4521, 'NAICS', 1431, 5122, 3, 1439, 51224, 4 FROM dual
UNION ALL SELECT 4522, 'SEC', 2793, 50, 1, 2600, 5170, 3 FROM dual
UNION ALL SELECT 4523, 'NAICS', 132, 21, 1, 168, 212391, 5 FROM dual
UNION ALL SELECT 4524, 'NAICS', 236, 2381, 3, 248, 23816, 4 FROM dual
UNION ALL SELECT 4525, 'NAICS', 19, 1112, 3, 21, 111211, 5 FROM dual
UNION ALL SELECT 4526, 'NAICS', 1851, 621, 2, 1860, 621310, 5 FROM dual
UNION ALL SELECT 4527, 'SIC', 3870, 5710, 3, 3872, 5713, 4 FROM dual
UNION ALL SELECT 4528, 'SEC', 2791, 20, 1, 2391, 3440, 3 FROM dual
UNION ALL SELECT 4529, 'SIC', 3124, 2400, 2, 3143, 2452, 4 FROM dual
UNION ALL SELECT 4530, 'NAICS', 2107, 8132, 3, 2110, 813212, 5 FROM dual
UNION ALL SELECT 4531, 'NAICS', 1402, 51, 1, 1430, 512199, 5 FROM dual
UNION ALL SELECT 4532, 'NAICS', 2040, 81111, 4, 2044, 811118, 5 FROM dual
UNION ALL SELECT 4533, 'NAICS', 2135, 92, 1, 2176, 924, 2 FROM dual
UNION ALL SELECT 4534, 'SIC', 4308, 20, 1, 3294, 3160, 3 FROM dual
UNION ALL SELECT 4535, 'SIC', 3419, 3500, 2, 3442, 3548, 4 FROM dual
UNION ALL SELECT 4536, 'NAICS', 172, 213, 2, 174, 21311, 4 FROM dual
UNION ALL SELECT 4537, 'SIC', 3371, 3400, 2, 3401, 3470, 3 FROM dual
UNION ALL SELECT 4538, 'NAICS', 1850, 62, 1, 1895, 6221, 3 FROM dual
UNION ALL SELECT 4539, 'SIC', 3907, 5980, 3, 3909, 5984, 4 FROM dual
UNION ALL SELECT 4540, 'NAICS', 1792, 562, 2, 1806, 56291, 4 FROM dual
UNION ALL SELECT 4541, 'SIC', 3434, 3540, 3, 3436, 3542, 4 FROM dual
UNION ALL SELECT 4542, 'NAICS', 1725, 56, 1, 1792, 562, 2 FROM dual
UNION ALL SELECT 4543, 'NAICS', 2188, 926, 2, 2193, 92612, 4 FROM dual
UNION ALL SELECT 4544, 'NAICS', 1944, 7111, 3, 1950, 71113, 4 FROM dual
UNION ALL SELECT 4545, 'SIC', 3261, 3000, 2, 3281, 3089, 4 FROM dual
UNION ALL SELECT 4546, 'NAICS', 2130, 81399, 4, 2129, 813990, 5 FROM dual
UNION ALL SELECT 4547, 'NAICS', 1480, 52, 1, 1525, 52321, 4 FROM dual
UNION ALL SELECT 4548, 'SIC', 3526, 3700, 2, 3546, 3764, 4 FROM dual
UNION ALL SELECT 4549, 'SIC', 3050, 2200, 2, 3082, 2299, 4 FROM dual
UNION ALL SELECT 4550, 'SIC', 4308, 20, 1, 3466, 3578, 4 FROM dual
UNION ALL SELECT 4551, 'NAICS', 1987, 71321, 4, 1986, 713210, 5 FROM dual
UNION ALL SELECT 4552, 'SIC', 4309, 40, 1, 3674, 4740, 3 FROM dual
UNION ALL SELECT 4553, 'NAICS', 2166, 923, 2, 2167, 9231, 3 FROM dual
UNION ALL SELECT 4554, 'SEC', 2713, 7300, 2, 2717, 7330, 3 FROM dual
UNION ALL SELECT 4555, 'SIC', 4146, 8000, 2, 4170, 8082, 4 FROM dual
UNION ALL SELECT 4556, 'NAICS', 1850, 62, 1, 1916, 623312, 5 FROM dual
UNION ALL SELECT 4557, 'SIC', 4310, 50, 1, 3728, 5039, 4 FROM dual
UNION ALL SELECT 4558, 'SIC', 4113, 7800, 2, 4118, 7822, 4 FROM dual
UNION ALL SELECT 4559, 'NAICS', 1725, 56, 1, 1779, 561730, 5 FROM dual
UNION ALL SELECT 4560, 'NAICS', 1620, 533, 2, 1622, 533110, 5 FROM dual
UNION ALL SELECT 4561, 'SEC', 2791, 20, 1, 2462, 3714, 4 FROM dual
UNION ALL SELECT 4562, 'SIC', 4311, 52, 1, 3817, 5311, 4 FROM dual
UNION ALL SELECT 4563, 'SEC', 2714, 7310, 3, 2715, 7311, 4 FROM dual
UNION ALL SELECT 4564, 'NAICS', 236, 2381, 3, 242, 23813, 4 FROM dual
UNION ALL SELECT 4565, 'SIC', 4146, 8000, 2, 4174, 8099, 4 FROM dual
UNION ALL SELECT 4566, 'NAICS', 138, 212, 2, 160, 212313, 5 FROM dual
UNION ALL SELECT 4567, 'NAICS', 1, 11, 1, 29, 111331, 5 FROM dual
UNION ALL SELECT 4568, 'NAICS', 1624, 54, 1, 1684, 54171, 4 FROM dual
UNION ALL SELECT 4569, 'SEC', 2631, 5710, 3, 2632, 5712, 4 FROM dual
UNION ALL SELECT 4570, 'SIC', 4308, 20, 1, 3186, 2676, 4 FROM dual
UNION ALL SELECT 4571, 'SIC', 4306, 10, 1, 2927, 1446, 4 FROM dual
UNION ALL SELECT 4572, 'SEC', 2539, 4800, 2, 2546, 4832, 4 FROM dual
UNION ALL SELECT 4573, 'SEC', 2793, 50, 1, 2577, 5050, 3 FROM dual
UNION ALL SELECT 4574, 'NAICS', 1583, 5313, 3, 1589, 531390, 5 FROM dual
UNION ALL SELECT 4575, 'SEC', 2738, 7800, 2, 2747, 7841, 4 FROM dual
UNION ALL SELECT 4576, 'SIC', 4298, 9710, 3, 4299, 9711, 4 FROM dual
UNION ALL SELECT 4577, 'NAICS', 156, 2123, 3, 168, 212391, 5 FROM dual
UNION ALL SELECT 4578, 'NAICS', 1569, 53, 1, 1575, 53112, 4 FROM dual
UNION ALL SELECT 4579, 'SEC', 2276, 2400, 2, 2282, 2452, 4 FROM dual
UNION ALL SELECT 4580, 'SIC', 4306, 10, 1, 2902, 1220, 3 FROM dual
UNION ALL SELECT 4581, 'SIC', 3148, 2500, 2, 3151, 2512, 4 FROM dual
UNION ALL SELECT 4582, 'SIC', 3865, 5660, 3, 3866, 5661, 4 FROM dual
UNION ALL SELECT 4583, 'NAICS', 1015, 424, 2, 1086, 42495, 4 FROM dual
UNION ALL SELECT 4584, 'NAICS', 930, 42, 1, 999, 42384, 4 FROM dual
UNION ALL SELECT 4585, 'NAICS', 1537, 5241, 3, 1542, 524126, 5 FROM dual
UNION ALL SELECT 4586, 'NAICS', 2132, 8141, 3, 2133, 814110, 5 FROM dual
UNION ALL SELECT 4587, 'SIC', 4089, 7530, 3, 4096, 7539, 4 FROM dual
UNION ALL SELECT 4588, 'NAICS', 930, 42, 1, 1041, 42443, 4 FROM dual
UNION ALL SELECT 4589, 'NAICS', 1625, 541, 2, 1644, 54132, 4 FROM dual
UNION ALL SELECT 4590, 'SEC', 2677, 6300, 2, 2685, 6350, 3 FROM dual
UNION ALL SELECT 4591, 'SIC', 4186, 8240, 3, 4189, 8249, 4 FROM dual
UNION ALL SELECT 4592, 'NAICS', 1, 11, 1, 3, 1111, 3 FROM dual
UNION ALL SELECT 4593, 'NAICS', 1851, 621, 2, 1856, 6212, 3 FROM dual
UNION ALL SELECT 4594, 'SIC', 3214, 2800, 2, 3229, 2836, 4 FROM dual
UNION ALL SELECT 4595, 'NAICS', 1554, 525, 2, 1557, 52511, 4 FROM dual
UNION ALL SELECT 4596, 'NAICS', 1829, 61143, 4, 1828, 611430, 5 FROM dual
UNION ALL SELECT 4597, 'SIC', 4277, 9500, 2, 4278, 9510, 3 FROM dual
UNION ALL SELECT 4598, 'SIC', 3172, 2630, 3, 3173, 2631, 4 FROM dual
UNION ALL SELECT 4599, 'SIC', 3740, 5060, 3, 3741, 5063, 4 FROM dual
UNION ALL SELECT 4600, 'SEC', 2591, 5100, 2, 2604, 5190, 3 FROM dual
UNION ALL SELECT 4601, 'SIC', 3949, 6210, 3, 3950, 6211, 4 FROM dual
UNION ALL SELECT 4602, 'SIC', 4313, 70, 1, 4186, 8240, 3 FROM dual
UNION ALL SELECT 4603, 'NAICS', 2037, 81, 1, 2106, 81311, 4 FROM dual
UNION ALL SELECT 4604, 'NAICS', 1838, 61161, 4, 1837, 611610, 5 FROM dual
UNION ALL SELECT 4605, 'NAICS', 1480, 52, 1, 1538, 52411, 4 FROM dual
UNION ALL SELECT 4606, 'SIC', 2918, 1400, 2, 2937, 1490, 3 FROM dual
UNION ALL SELECT 4607, 'NAICS', 1402, 51, 1, 1461, 51741, 4 FROM dual
UNION ALL SELECT 4608, 'NAICS', 1665, 54149, 4, 1664, 541490, 5 FROM dual
UNION ALL SELECT 4609, 'SIC', 3272, 3080, 3, 3281, 3089, 4 FROM dual
UNION ALL SELECT 4610, 'NAICS', 1537, 5241, 3, 1546, 52413, 4 FROM dual
UNION ALL SELECT 4611, 'SIC', 3419, 3500, 2, 3456, 3565, 4 FROM dual
UNION ALL SELECT 4612, 'NAICS', 1979, 713, 2, 1980, 7131, 3 FROM dual
UNION ALL SELECT 4613, 'NAICS', 1725, 56, 1, 1729, 56111, 4 FROM dual
UNION ALL SELECT 4614, 'SEC', 2791, 20, 1, 2428, 3577, 4 FROM dual
UNION ALL SELECT 4615, 'SIC', 3103, 2360, 3, 3104, 2361, 4 FROM dual
UNION ALL SELECT 4616, 'SIC', 3269, 3060, 3, 3271, 3069, 4 FROM dual
UNION ALL SELECT 4617, 'NAICS', 1888, 6219, 3, 1893, 621999, 5 FROM dual
UNION ALL SELECT 4618, 'NAICS', 2020, 722, 2, 2026, 722330, 5 FROM dual
UNION ALL SELECT 4619, 'NAICS', 1004, 4239, 3, 1006, 42391, 4 FROM dual
UNION ALL SELECT 4620, 'NAICS', 970, 4235, 3, 972, 42351, 4 FROM dual
UNION ALL SELECT 4621, 'NAICS', 1850, 62, 1, 1852, 6211, 3 FROM dual
UNION ALL SELECT 4622, 'SIC', 3050, 2200, 2, 3054, 2221, 4 FROM dual
UNION ALL SELECT 4623, 'NAICS', 2204, 928, 2, 2205, 9281, 3 FROM dual
UNION ALL SELECT 4624, 'SIC', 4311, 52, 1, 3894, 5941, 4 FROM dual
UNION ALL SELECT 4625, 'NAICS', 255, 23821, 4, 254, 238210, 5 FROM dual
UNION ALL SELECT 4626, 'NAICS', 1774, 5617, 3, 1784, 56179, 4 FROM dual
UNION ALL SELECT 4627, 'SIC', 4308, 20, 1, 3542, 3750, 3 FROM dual
UNION ALL SELECT 4628, 'SIC', 3918, 6010, 3, 3919, 6011, 4 FROM dual
UNION ALL SELECT 4629, 'NAICS', 1480, 52, 1, 1507, 522310, 5 FROM dual
UNION ALL SELECT 4630, 'SEC', 2791, 20, 1, 2486, 3827, 4 FROM dual
UNION ALL SELECT 4631, 'SEC', 2791, 20, 1, 2458, 3700, 2 FROM dual
UNION ALL SELECT 4632, 'SIC', 3106, 2370, 3, 3107, 2371, 4 FROM dual
UNION ALL SELECT 4633, 'SIC', 3633, 4300, 2, 3634, 4310, 3 FROM dual
UNION ALL SELECT 4634, 'SIC', 3371, 3400, 2, 3400, 3469, 4 FROM dual
UNION ALL SELECT 4635, 'SIC', 3762, 5100, 2, 3789, 5162, 4 FROM dual
UNION ALL SELECT 4636, 'NAICS', 1408, 51112, 4, 1407, 511120, 5 FROM dual
UNION ALL SELECT 4637, 'NAICS', 2020, 722, 2, 2024, 722320, 5 FROM dual
UNION ALL SELECT 4638, 'SIC', 3156, 2520, 3, 3158, 2522, 4 FROM dual
UNION ALL SELECT 4639, 'NAICS', 1979, 713, 2, 1993, 713920, 5 FROM dual
UNION ALL SELECT 4640, 'SIC', 4308, 20, 1, 3271, 3069, 4 FROM dual
UNION ALL SELECT 4641, 'NAICS', 260, 2383, 3, 268, 23834, 4 FROM dual
UNION ALL SELECT 4642, 'SEC', 2384, 3400, 2, 2387, 3412, 4 FROM dual
UNION ALL SELECT 4643, 'SIC', 4308, 20, 1, 3046, 2130, 3 FROM dual
UNION ALL SELECT 4644, 'SIC', 4308, 20, 1, 3447, 3554, 4 FROM dual
UNION ALL SELECT 4645, 'SIC', 3809, 5250, 3, 3810, 5251, 4 FROM dual
UNION ALL SELECT 4646, 'SEC', 2659, 6100, 2, 2661, 6111, 4 FROM dual
UNION ALL SELECT 4647, 'SIC', 3576, 3900, 2, 3588, 3951, 4 FROM dual
UNION ALL SELECT 4648, 'SIC', 3520, 3690, 3, 3521, 3691, 4 FROM dual
UNION ALL SELECT 4649, 'NAICS', 1725, 56, 1, 1802, 562213, 5 FROM dual
UNION ALL SELECT 4650, 'SIC', 4313, 70, 1, 4219, 8660, 3 FROM dual
UNION ALL SELECT 4651, 'SIC', 4041, 7310, 3, 4043, 7312, 4 FROM dual
UNION ALL SELECT 4652, 'SEC', 2488, 3840, 3, 2491, 3843, 4 FROM dual
UNION ALL SELECT 4653, 'SEC', 2792, 40, 1, 2525, 4412, 4 FROM dual
UNION ALL SELECT 4654, 'SEC', 2796, 70, 1, 2738, 7800, 2 FROM dual
UNION ALL SELECT 4655, 'SIC', 3837, 5500, 2, 3851, 5571, 4 FROM dual
UNION ALL SELECT 4656, 'SIC', 4120, 7830, 3, 4122, 7833, 4 FROM dual
UNION ALL SELECT 4657, 'NAICS', 218, 237, 2, 234, 23799, 4 FROM dual
UNION ALL SELECT 4658, 'SIC', 4311, 52, 1, 3830, 5441, 4 FROM dual
UNION ALL SELECT 4659, 'NAICS', 2119, 81341, 4, 2118, 813410, 5 FROM dual
UNION ALL SELECT 4660, 'SIC', 3544, 3760, 3, 3547, 3769, 4 FROM dual
UNION ALL SELECT 4661, 'SIC', 3729, 5040, 3, 3730, 5043, 4 FROM dual
UNION ALL SELECT 4662, 'NAICS', 2207, 92811, 4, 2206, 928110, 5 FROM dual
UNION ALL SELECT 4663, 'SIC', 4308, 20, 1, 3007, 2048, 4 FROM dual
UNION ALL SELECT 4664, 'SEC', 2796, 70, 1, 2734, 7389, 4 FROM dual
UNION ALL SELECT 4665, 'SIC', 2825, 200, 2, 2829, 213, 4 FROM dual
UNION ALL SELECT 4666, 'NAICS', 931, 423, 2, 963, 42344, 4 FROM dual
UNION ALL SELECT 4667, 'SEC', 2240, 2000, 2, 2242, 2011, 4 FROM dual
UNION ALL SELECT 4668, 'SIC', 4313, 70, 1, 4111, 7694, 4 FROM dual
UNION ALL SELECT 4669, 'SIC', 4309, 40, 1, 3713, 4970, 3 FROM dual
UNION ALL SELECT 4670, 'SIC', 4192, 8300, 2, 4193, 8320, 3 FROM dual
UNION ALL SELECT 4671, 'SIC', 3797, 5190, 3, 3803, 5199, 4 FROM dual
UNION ALL SELECT 4672, 'NAICS', 1979, 713, 2, 1998, 71394, 4 FROM dual
UNION ALL SELECT 4673, 'NAICS', 1559, 52512, 4, 1558, 525120, 5 FROM dual
UNION ALL SELECT 4674, 'NAICS', 2061, 8114, 3, 2069, 811490, 5 FROM dual
UNION ALL SELECT 4675, 'SIC', 3637, 4410, 3, 3638, 4412, 4 FROM dual
UNION ALL SELECT 4676, 'NAICS', 180, 22, 1, 190, 221117, 5 FROM dual
UNION ALL SELECT 4677, 'SIC', 4307, 15, 1, 2961, 1731, 4 FROM dual
UNION ALL SELECT 4678, 'SIC', 3200, 2750, 3, 3202, 2754, 4 FROM dual
UNION ALL SELECT 4679, 'NAICS', 1076, 4249, 3, 1088, 42499, 4 FROM dual
UNION ALL SELECT 4680, 'SIC', 3167, 2600, 2, 3185, 2675, 4 FROM dual
UNION ALL SELECT 4681, 'SIC', 4308, 20, 1, 3428, 3532, 4 FROM dual
UNION ALL SELECT 4682, 'NAICS', 991, 4238, 3, 998, 423840, 5 FROM dual
UNION ALL SELECT 4683, 'SIC', 4308, 20, 1, 3360, 3356, 4 FROM dual
UNION ALL SELECT 4684, 'SIC', 4310, 50, 1, 3745, 5072, 4 FROM dual
UNION ALL SELECT 4685, 'NAICS', 1469, 51821, 4, 1468, 518210, 5 FROM dual
UNION ALL SELECT 4686, 'NAICS', 1666, 5415, 3, 1667, 54151, 4 FROM dual
UNION ALL SELECT 4687, 'NAICS', 1402, 51, 1, 1413, 51119, 4 FROM dual
UNION ALL SELECT 4688, 'SIC', 4308, 20, 1, 3550, 3795, 4 FROM dual
UNION ALL SELECT 4689, 'SIC', 4040, 7300, 2, 4050, 7331, 4 FROM dual
UNION ALL SELECT 4690, 'SEC', 2662, 6140, 3, 2663, 6141, 4 FROM dual
UNION ALL SELECT 4691, 'SIC', 4015, 7040, 3, 4016, 7041, 4 FROM dual
UNION ALL SELECT 4692, 'SIC', 4313, 70, 1, 4195, 8330, 3 FROM dual
UNION ALL SELECT 4693, 'NAICS', 1431, 5122, 3, 1435, 51222, 4 FROM dual
UNION ALL SELECT 4694, 'SIC', 4310, 50, 1, 3773, 5139, 4 FROM dual
UNION ALL SELECT 4695, 'SIC', 3252, 2900, 2, 3260, 2999, 4 FROM dual
UNION ALL SELECT 4696, 'SIC', 3693, 4900, 2, 3697, 4922, 4 FROM dual
UNION ALL SELECT 4697, 'SIC', 3605, 4100, 2, 3608, 4119, 4 FROM dual
UNION ALL SELECT 4698, 'NAICS', 2039, 8111, 3, 2050, 811192, 5 FROM dual
UNION ALL SELECT 4699, 'NAICS', 1921, 6241, 3, 1924, 624120, 5 FROM dual
UNION ALL SELECT 4700, 'NAICS', 1015, 424, 2, 1037, 42441, 4 FROM dual
UNION ALL SELECT 4701, 'NAICS', 1871, 6214, 3, 1872, 621410, 5 FROM dual
UNION ALL SELECT 4702, 'SEC', 2796, 70, 1, 2774, 8700, 2 FROM dual
UNION ALL SELECT 4703, 'SIC', 3287, 3140, 3, 3288, 3142, 4 FROM dual
UNION ALL SELECT 4704, 'SIC', 2975, 1790, 3, 2980, 1796, 4 FROM dual
UNION ALL SELECT 4705, 'SEC', 2568, 5000, 2, 2572, 5030, 3 FROM dual
UNION ALL SELECT 4706, 'NAICS', 2200, 927, 2, 2201, 9271, 3 FROM dual
UNION ALL SELECT 4707, 'SIC', 3214, 2800, 2, 3242, 2873, 4 FROM dual
UNION ALL SELECT 4708, 'SIC', 3668, 4720, 3, 3670, 4725, 4 FROM dual
UNION ALL SELECT 4709, 'NAICS', 150, 21223, 4, 151, 212231, 5 FROM dual
UNION ALL SELECT 4710, 'SEC', 2791, 20, 1, 2304, 2711, 4 FROM dual
UNION ALL SELECT 4711, 'SEC', 2792, 40, 1, 2529, 4513, 4 FROM dual
UNION ALL SELECT 4712, 'NAICS', 1640, 5413, 3, 1645, 541330, 5 FROM dual
UNION ALL SELECT 4713, 'NAICS', 2135, 92, 1, 2177, 9241, 3 FROM dual
UNION ALL SELECT 4714, 'SIC', 4313, 70, 1, 4136, 7950, 3 FROM dual
UNION ALL SELECT 4715, 'NAICS', 2071, 812, 2, 2099, 812930, 5 FROM dual
UNION ALL SELECT 4716, 'SIC', 2875, 900, 2, 2880, 920, 3 FROM dual
UNION ALL SELECT 4717, 'NAICS', 2177, 9241, 3, 2178, 924110, 5 FROM dual
UNION ALL SELECT 4718, 'SIC', 3917, 6000, 2, 3924, 6029, 4 FROM dual
UNION ALL SELECT 4719, 'SIC', 4312, 60, 1, 3986, 6531, 4 FROM dual
UNION ALL SELECT 4720, 'SIC', 3474, 3590, 3, 3479, 3599, 4 FROM dual
UNION ALL SELECT 4721, 'NAICS', 1813, 611, 2, 1839, 611620, 5 FROM dual
UNION ALL SELECT 4722, 'SIC', 4308, 20, 1, 3314, 3255, 4 FROM dual
UNION ALL SELECT 4723, 'SIC', 3715, 5000, 2, 3732, 5045, 4 FROM dual
UNION ALL SELECT 4724, 'NAICS', 1813, 611, 2, 1815, 611110, 5 FROM dual
UNION ALL SELECT 4725, 'NAICS', 2037, 81, 1, 2069, 811490, 5 FROM dual
UNION ALL SELECT 4726, 'NAICS', 1928, 6242, 3, 1930, 62421, 4 FROM dual
UNION ALL SELECT 4727, 'SEC', 2792, 40, 1, 4317, 4990, 2 FROM dual
UNION ALL SELECT 4728, 'SIC', 4308, 20, 1, 2986, 2015, 4 FROM dual
UNION ALL SELECT 4729, 'SIC', 3167, 2600, 2, 3173, 2631, 4 FROM dual
UNION ALL SELECT 4730, 'SEC', 2630, 5700, 2, 2634, 5731, 4 FROM dual
UNION ALL SELECT 4731, 'SIC', 3355, 3350, 3, 3361, 3357, 4 FROM dual
UNION ALL SELECT 4732, 'NAICS', 931, 423, 2, 951, 423330, 5 FROM dual
UNION ALL SELECT 4733, 'SIC', 3636, 4400, 2, 3647, 4482, 4 FROM dual
UNION ALL SELECT 4734, 'SIC', 4305, 1, 1, 2883, 971, 4 FROM dual
UNION ALL SELECT 4735, 'SIC', 3480, 3600, 2, 3520, 3690, 3 FROM dual
UNION ALL SELECT 4736, 'SEC', 2789, 10, 1, 2215, 1000, 2 FROM dual
UNION ALL SELECT 4737, 'NAICS', 940, 42314, 4, 939, 423140, 5 FROM dual
UNION ALL SELECT 4738, 'NAICS', 2037, 81, 1, 2047, 811122, 5 FROM dual
UNION ALL SELECT 4739, 'SEC', 2240, 2000, 2, 2258, 2092, 4 FROM dual
UNION ALL SELECT 4740, 'SEC', 2403, 3500, 2, 2428, 3577, 4 FROM dual
UNION ALL SELECT 4741, 'NAICS', 1624, 54, 1, 1646, 54133, 4 FROM dual
UNION ALL SELECT 4742, 'NAICS', 1625, 541, 2, 1626, 5411, 3 FROM dual
UNION ALL SELECT 4743, 'SIC', 4313, 70, 1, 4080, 7389, 4 FROM dual
UNION ALL SELECT 4744, 'SIC', 3302, 3210, 3, 3303, 3211, 4 FROM dual
UNION ALL SELECT 4745, 'NAICS', 2136, 921, 2, 2148, 921190, 5 FROM dual
UNION ALL SELECT 4746, 'NAICS', 1402, 51, 1, 1459, 5174, 3 FROM dual
UNION ALL SELECT 4747, 'NAICS', 132, 21, 1, 146, 21221, 4 FROM dual
UNION ALL SELECT 4748, 'NAICS', 930, 42, 1, 1075, 42482, 4 FROM dual
UNION ALL SELECT 4749, 'NAICS', 1851, 621, 2, 1866, 621340, 5 FROM dual
UNION ALL SELECT 4750, 'SIC', 3707, 4950, 3, 3710, 4959, 4 FROM dual
UNION ALL SELECT 4751, 'SIC', 4081, 7500, 2, 4097, 7540, 3 FROM dual
UNION ALL SELECT 4752, 'SIC', 3601, 4000, 2, 3604, 4013, 4 FROM dual
UNION ALL SELECT 4753, 'SIC', 3822, 5400, 2, 3829, 5440, 3 FROM dual
UNION ALL SELECT 4754, 'SEC', 2276, 2400, 2, 2278, 2421, 4 FROM dual
UNION ALL SELECT 4755, 'SEC', 2539, 4800, 2, 2549, 4841, 4 FROM dual
UNION ALL SELECT 4756, 'NAICS', 23, 1113, 3, 29, 111331, 5 FROM dual
UNION ALL SELECT 4757, 'SIC', 4308, 20, 1, 3059, 2250, 3 FROM dual
UNION ALL SELECT 4758, 'SIC', 3831, 5450, 3, 3832, 5451, 4 FROM dual
UNION ALL SELECT 4759, 'NAICS', 132, 21, 1, 158, 212311, 5 FROM dual
UNION ALL SELECT 4760, 'NAICS', 1758, 5615, 3, 1761, 561520, 5 FROM dual
UNION ALL SELECT 4761, 'SIC', 3042, 2110, 3, 3043, 2111, 4 FROM dual
UNION ALL SELECT 4762, 'SIC', 4223, 8700, 2, 4233, 8733, 4 FROM dual
UNION ALL SELECT 4763, 'SEC', 2791, 20, 1, 2255, 2082, 4 FROM dual
UNION ALL SELECT 4764, 'NAICS', 138, 212, 2, 143, 212113, 5 FROM dual
UNION ALL SELECT 4765, 'NAICS', 1486, 5221, 3, 1487, 522110, 5 FROM dual
UNION ALL SELECT 4766, 'NAICS', 3, 1111, 3, 13, 11115, 4 FROM dual
UNION ALL SELECT 4767, 'SIC', 4100, 7600, 2, 4103, 7623, 4 FROM dual
UNION ALL SELECT 4768, 'SIC', 3870, 5710, 3, 3873, 5714, 4 FROM dual
UNION ALL SELECT 4769, 'NAICS', 1812, 61, 1, 1842, 61163, 4 FROM dual
UNION ALL SELECT 4770, 'SIC', 3762, 5100, 2, 3790, 5169, 4 FROM dual
UNION ALL SELECT 4771, 'NAICS', 1480, 52, 1, 1482, 5211, 3 FROM dual
UNION ALL SELECT 4772, 'NAICS', 1850, 62, 1, 1855, 621112, 5 FROM dual
UNION ALL SELECT 4773, 'SIC', 4306, 10, 1, 2896, 1080, 3 FROM dual
UNION ALL SELECT 4774, 'NAICS', 970, 4235, 3, 973, 423520, 5 FROM dual
UNION ALL SELECT 4775, 'SIC', 4312, 60, 1, 3945, 6160, 3 FROM dual
UNION ALL SELECT 4776, 'SIC', 3301, 3200, 2, 3308, 3231, 4 FROM dual
UNION ALL SELECT 4777, 'SIC', 4308, 20, 1, 3233, 2843, 4 FROM dual
UNION ALL SELECT 4778, 'SIC', 3937, 6100, 2, 3941, 6141, 4 FROM dual
UNION ALL SELECT 4779, 'SIC', 3668, 4720, 3, 3671, 4729, 4 FROM dual
UNION ALL SELECT 4780, 'NAICS', 1798, 5622, 3, 1800, 562211, 5 FROM dual
UNION ALL SELECT 4781, 'NAICS', 1463, 51791, 4, 1464, 517911, 5 FROM dual
UNION ALL SELECT 4782, 'SIC', 3167, 2600, 2, 3172, 2630, 3 FROM dual
UNION ALL SELECT 4783, 'NAICS', 2, 111, 2, 44, 111910, 5 FROM dual
UNION ALL SELECT 4784, 'NAICS', 930, 42, 1, 1008, 42392, 4 FROM dual
UNION ALL SELECT 4785, 'SIC', 4308, 20, 1, 3121, 2396, 4 FROM dual
UNION ALL SELECT 4786, 'SIC', 3684, 4820, 3, 3685, 4822, 4 FROM dual
UNION ALL SELECT 4787, 'NAICS', 1725, 56, 1, 1728, 561110, 5 FROM dual
UNION ALL SELECT 4788, 'NAICS', 2037, 81, 1, 2105, 813110, 5 FROM dual
UNION ALL SELECT 4789, 'NAICS', 930, 42, 1, 1059, 424590, 5 FROM dual
UNION ALL SELECT 4790, 'SIC', 4308, 20, 1, 3092, 2329, 4 FROM dual
UNION ALL SELECT 4791, 'SIC', 4308, 20, 1, 3255, 2950, 3 FROM dual
UNION ALL SELECT 4792, 'SEC', 2215, 1000, 2, 2217, 1090, 3 FROM dual
UNION ALL SELECT 4793, 'SEC', 2795, 60, 1, 4325, 6199, 4 FROM dual
UNION ALL SELECT 4794, 'SEC', 2774, 8700, 2, 2778, 8731, 4 FROM dual
UNION ALL SELECT 4795, 'NAICS', 931, 423, 2, 1010, 42393, 4 FROM dual
UNION ALL SELECT 4796, 'NAICS', 132, 21, 1, 164, 212322, 5 FROM dual
UNION ALL SELECT 4797, 'NAICS', 1624, 54, 1, 1638, 541214, 5 FROM dual
UNION ALL SELECT 4798, 'SIC', 4040, 7300, 2, 4077, 7382, 4 FROM dual
UNION ALL SELECT 4799, 'SIC', 3283, 3110, 3, 3284, 3111, 4 FROM dual
UNION ALL SELECT 4800, 'SIC', 3636, 4400, 2, 3652, 4493, 4 FROM dual
UNION ALL SELECT 4801, 'SIC', 3886, 5900, 2, 3895, 5942, 4 FROM dual
UNION ALL SELECT 4802, 'NAICS', 1625, 541, 2, 1657, 5414, 3 FROM dual
UNION ALL SELECT 4803, 'SIC', 3371, 3400, 2, 3373, 3411, 4 FROM dual
UNION ALL SELECT 4804, 'NAICS', 1785, 5619, 3, 1790, 561990, 5 FROM dual
UNION ALL SELECT 4805, 'NAICS', 1943, 711, 2, 1959, 711310, 5 FROM dual
UNION ALL SELECT 4806, 'SIC', 3261, 3000, 2, 3276, 3084, 4 FROM dual
UNION ALL SELECT 4807, 'SIC', 4133, 7940, 3, 4135, 7948, 4 FROM dual
UNION ALL SELECT 4808, 'NAICS', 1537, 5241, 3, 1541, 52412, 4 FROM dual
UNION ALL SELECT 4809, 'SIC', 4306, 10, 1, 2910, 1310, 3 FROM dual
UNION ALL SELECT 4810, 'SIC', 4146, 8000, 2, 4155, 8042, 4 FROM dual
UNION ALL SELECT 4811, 'SEC', 2792, 40, 1, 2530, 4520, 3 FROM dual
UNION ALL SELECT 4812, 'NAICS', 1624, 54, 1, 1660, 541420, 5 FROM dual
UNION ALL SELECT 4813, 'SEC', 2791, 20, 1, 2376, 3330, 3 FROM dual
UNION ALL SELECT 4814, 'SIC', 4308, 20, 1, 3487, 3625, 4 FROM dual
UNION ALL SELECT 4815, 'NAICS', 1419, 512, 2, 1437, 51223, 4 FROM dual
UNION ALL SELECT 4816, 'SIC', 3658, 4520, 3, 3659, 4522, 4 FROM dual
UNION ALL SELECT 4817, 'SIC', 4308, 20, 1, 3152, 2514, 4 FROM dual
UNION ALL SELECT 4818, 'SIC', 4041, 7310, 3, 4044, 7313, 4 FROM dual
UNION ALL SELECT 4819, 'NAICS', 1, 11, 1, 117, 115, 2 FROM dual
UNION ALL SELECT 4820, 'NAICS', 991, 4238, 3, 992, 423810, 5 FROM dual
UNION ALL SELECT 4821, 'NAICS', 1720, 5511, 3, 1722, 551111, 5 FROM dual
UNION ALL SELECT 4822, 'SIC', 4313, 70, 1, 4204, 8410, 3 FROM dual
UNION ALL SELECT 4823, 'SIC', 4310, 50, 1, 3763, 5110, 3 FROM dual
UNION ALL SELECT 4824, 'SIC', 3762, 5100, 2, 3798, 5191, 4 FROM dual
UNION ALL SELECT 4825, 'SIC', 3762, 5100, 2, 3787, 5159, 4 FROM dual
UNION ALL SELECT 4826, 'SIC', 4312, 60, 1, 4000, 6732, 4 FROM dual
UNION ALL SELECT 4827, 'NAICS', 1402, 51, 1, 1406, 51111, 4 FROM dual
UNION ALL SELECT 4828, 'SEC', 2224, 1380, 3, 2226, 1382, 4 FROM dual
UNION ALL SELECT 4829, 'SEC', 2796, 70, 1, 2716, 7320, 3 FROM dual
UNION ALL SELECT 4830, 'NAICS', 133, 211, 2, 134, 2111, 3 FROM dual
UNION ALL SELECT 4831, 'SEC', 2791, 20, 1, 2396, 3450, 3 FROM dual
UNION ALL SELECT 4832, 'SEC', 2640, 5900, 2, 2643, 5940, 3 FROM dual
UNION ALL SELECT 4833, 'NAICS', 1591, 532, 2, 1605, 53229, 4 FROM dual
UNION ALL SELECT 4834, 'NAICS', 1583, 5313, 3, 1585, 531311, 5 FROM dual
UNION ALL SELECT 4835, 'SIC', 2798, 100, 2, 2803, 116, 4 FROM dual
UNION ALL SELECT 4836, 'NAICS', 20, 11121, 4, 22, 111219, 5 FROM dual
UNION ALL SELECT 4837, 'SIC', 3724, 5030, 3, 3728, 5039, 4 FROM dual
UNION ALL SELECT 4838, 'NAICS', 1640, 5413, 3, 1648, 54134, 4 FROM dual
UNION ALL SELECT 4839, 'NAICS', 23, 1113, 3, 30, 111332, 5 FROM dual
UNION ALL SELECT 4840, 'SEC', 2476, 3800, 2, 2487, 3829, 4 FROM dual
UNION ALL SELECT 4841, 'NAICS', 2117, 8134, 3, 2119, 81341, 4 FROM dual
UNION ALL SELECT 4842, 'SEC', 2302, 2700, 2, 2314, 2761, 4 FROM dual
UNION ALL SELECT 4843, 'SEC', 2794, 52, 1, 2605, 5200, 2 FROM dual
UNION ALL SELECT 4844, 'SIC', 3978, 6510, 3, 3981, 6514, 4 FROM dual
UNION ALL SELECT 4845, 'NAICS', 1523, 5232, 3, 1524, 523210, 5 FROM dual
UNION ALL SELECT 4846, 'NAICS', 1850, 62, 1, 1932, 624221, 5 FROM dual
UNION ALL SELECT 4847, 'NAICS', 1850, 62, 1, 1867, 62134, 4 FROM dual
UNION ALL SELECT 4848, 'SIC', 3620, 4200, 2, 3627, 4221, 4 FROM dual
UNION ALL SELECT 4849, 'NAICS', 56, 112, 2, 65, 1122, 3 FROM dual
UNION ALL SELECT 4850, 'NAICS', 930, 42, 1, 978, 423620, 5 FROM dual
UNION ALL SELECT 4851, 'NAICS', 1992, 71391, 4, 1991, 713910, 5 FROM dual
UNION ALL SELECT 4852, 'SIC', 4192, 8300, 2, 4194, 8322, 4 FROM dual
UNION ALL SELECT 4853, 'SIC', 3715, 5000, 2, 3760, 5094, 4 FROM dual
UNION ALL SELECT 4854, 'NAICS', 2135, 92, 1, 2139, 92111, 4 FROM dual
UNION ALL SELECT 4855, 'NAICS', 946, 4233, 3, 948, 42331, 4 FROM dual
UNION ALL SELECT 4856, 'NAICS', 1443, 5151, 3, 1446, 515112, 5 FROM dual
UNION ALL SELECT 4857, 'SEC', 2796, 70, 1, 2756, 8050, 3 FROM dual
UNION ALL SELECT 4858, 'NAICS', 930, 42, 1, 961, 42343, 4 FROM dual
UNION ALL SELECT 4859, 'SIC', 3282, 3100, 2, 3286, 3131, 4 FROM dual
UNION ALL SELECT 4860, 'NAICS', 1624, 54, 1, 1642, 54131, 4 FROM dual
UNION ALL SELECT 4861, 'SIC', 4308, 20, 1, 3489, 3630, 3 FROM dual
UNION ALL SELECT 4862, 'SIC', 3691, 4890, 3, 3692, 4899, 4 FROM dual
UNION ALL SELECT 4863, 'SIC', 4313, 70, 1, 4138, 7952, 4 FROM dual
UNION ALL SELECT 4864, 'NAICS', 1726, 561, 2, 1744, 56142, 4 FROM dual
UNION ALL SELECT 4865, 'SEC', 2494, 3850, 3, 2495, 3851, 4 FROM dual
UNION ALL SELECT 4866, 'NAICS', 1624, 54, 1, 1652, 54136, 4 FROM dual
UNION ALL SELECT 4867, 'NAICS', 117, 115, 2, 123, 115114, 5 FROM dual
UNION ALL SELECT 4868, 'NAICS', 1626, 5411, 3, 1633, 541199, 5 FROM dual
UNION ALL SELECT 4869, 'SEC', 2224, 1380, 3, 2227, 1389, 4 FROM dual
UNION ALL SELECT 4870, 'SIC', 4178, 8200, 2, 4185, 8231, 4 FROM dual
UNION ALL SELECT 4871, 'SIC', 4308, 20, 1, 3392, 3450, 3 FROM dual
UNION ALL SELECT 4872, 'SIC', 4313, 70, 1, 4040, 7300, 2 FROM dual
UNION ALL SELECT 4873, 'SIC', 3193, 2720, 3, 3194, 2721, 4 FROM dual
UNION ALL SELECT 4874, 'NAICS', 1672, 5416, 3, 1675, 541612, 5 FROM dual
UNION ALL SELECT 4875, 'NAICS', 205, 23, 1, 249, 238170, 5 FROM dual
UNION ALL SELECT 4876, 'NAICS', 1625, 541, 2, 1673, 54161, 4 FROM dual
UNION ALL SELECT 4877, 'SIC', 3322, 3270, 3, 3325, 3273, 4 FROM dual
UNION ALL SELECT 4878, 'SIC', 4308, 20, 1, 3555, 3820, 3 FROM dual
UNION ALL SELECT 4879, 'SIC', 3419, 3500, 2, 3464, 3575, 4 FROM dual
UNION ALL SELECT 4880, 'NAICS', 2189, 9261, 3, 2194, 926130, 5 FROM dual
UNION ALL SELECT 4881, 'SIC', 3907, 5980, 3, 3908, 5983, 4 FROM dual
UNION ALL SELECT 4882, 'SIC', 2993, 2030, 3, 2999, 2038, 4 FROM dual
UNION ALL SELECT 4883, 'NAICS', 1088, 42499, 4, 1087, 424990, 5 FROM dual
UNION ALL SELECT 4884, 'SIC', 3292, 3150, 3, 3293, 3151, 4 FROM dual
UNION ALL SELECT 4885, 'SIC', 3214, 2800, 2, 3239, 2865, 4 FROM dual
UNION ALL SELECT 4886, 'NAICS', 1022, 42413, 4, 1021, 424130, 5 FROM dual
UNION ALL SELECT 4887, 'SIC', 4308, 20, 1, 3567, 3843, 4 FROM dual
UNION ALL SELECT 4888, 'NAICS', 2003, 72, 1, 2004, 721, 2 FROM dual
UNION ALL SELECT 4889, 'SIC', 4311, 52, 1, 3837, 5500, 2 FROM dual
UNION ALL SELECT 4890, 'NAICS', 1570, 531, 2, 1576, 531130, 5 FROM dual
UNION ALL SELECT 4891, 'SEC', 2791, 20, 1, 2241, 2010, 3 FROM dual
UNION ALL SELECT 4892, 'SIC', 2799, 110, 3, 2803, 116, 4 FROM dual
UNION ALL SELECT 4893, 'SEC', 2791, 20, 1, 2442, 3640, 3 FROM dual
UNION ALL SELECT 4894, 'NAICS', 2107, 8132, 3, 2111, 813219, 5 FROM dual
UNION ALL SELECT 4895, 'SIC', 2876, 910, 3, 2878, 913, 4 FROM dual
UNION ALL SELECT 4896, 'NAICS', 930, 42, 1, 998, 423840, 5 FROM dual
UNION ALL SELECT 4897, 'SEC', 2792, 40, 1, 2564, 4950, 3 FROM dual
UNION ALL SELECT 4898, 'SIC', 3337, 3300, 2, 3340, 3313, 4 FROM dual
UNION ALL SELECT 4899, 'SIC', 3419, 3500, 2, 3423, 3520, 3 FROM dual
UNION ALL SELECT 4900, 'SEC', 2552, 4900, 2, 4316, 4955, 3 FROM dual
UNION ALL SELECT 4901, 'SIC', 4313, 70, 1, 4165, 8069, 4 FROM dual
UNION ALL SELECT 4902, 'NAICS', 1513, 523, 2, 1518, 52312, 4 FROM dual
UNION ALL SELECT 4903, 'SIC', 4308, 20, 1, 3194, 2721, 4 FROM dual
UNION ALL SELECT 4904, 'NAICS', 986, 42372, 4, 985, 423720, 5 FROM dual
UNION ALL SELECT 4905, 'NAICS', 1907, 62311, 4, 1906, 623110, 5 FROM dual
UNION ALL SELECT 4906, 'SIC', 3371, 3400, 2, 3393, 3451, 4 FROM dual
UNION ALL SELECT 4907, 'SIC', 2875, 900, 2, 2876, 910, 3 FROM dual
UNION ALL SELECT 4908, 'NAICS', 931, 423, 2, 945, 42322, 4 FROM dual
UNION ALL SELECT 4909, 'SEC', 2555, 4920, 3, 2556, 4922, 4 FROM dual
UNION ALL SELECT 4910, 'NAICS', 931, 423, 2, 981, 42369, 4 FROM dual
UNION ALL SELECT 4911, 'SIC', 2983, 2010, 3, 2986, 2015, 4 FROM dual
UNION ALL SELECT 4912, 'SEC', 2725, 7370, 3, 2728, 7373, 4 FROM dual
UNION ALL SELECT 4913, 'NAICS', 1626, 5411, 3, 1629, 541120, 5 FROM dual
UNION ALL SELECT 4914, 'SIC', 4313, 70, 1, 4205, 8412, 4 FROM dual
UNION ALL SELECT 4915, 'SEC', 2794, 52, 1, 4319, 5412, 4 FROM dual
UNION ALL SELECT 4916, 'SIC', 4308, 20, 1, 3333, 3295, 4 FROM dual
UNION ALL SELECT 4917, 'NAICS', 930, 42, 1, 1024, 424210, 5 FROM dual
UNION ALL SELECT 4918, 'NAICS', 1500, 52229, 4, 1504, 522294, 5 FROM dual
UNION ALL SELECT 4919, 'NAICS', 1888, 6219, 3, 1892, 621991, 5 FROM dual
UNION ALL SELECT 4920, 'NAICS', 1045, 42445, 4, 1044, 424450, 5 FROM dual
UNION ALL SELECT 4921, 'SEC', 2791, 20, 1, 2308, 2731, 4 FROM dual
UNION ALL SELECT 4922, 'SEC', 2793, 50, 1, 2570, 5013, 4 FROM dual
UNION ALL SELECT 4923, 'SIC', 3050, 2200, 2, 3081, 2298, 4 FROM dual
UNION ALL SELECT 4924, 'SIC', 3301, 3200, 2, 3328, 3280, 3 FROM dual
UNION ALL SELECT 4925, 'NAICS', 1969, 712, 2, 1974, 71212, 4 FROM dual
UNION ALL SELECT 4926, 'SIC', 3051, 2210, 3, 3052, 2211, 4 FROM dual
UNION ALL SELECT 4927, 'SIC', 4308, 20, 1, 3501, 3646, 4 FROM dual
UNION ALL SELECT 4928, 'SIC', 3419, 3500, 2, 3424, 3523, 4 FROM dual
UNION ALL SELECT 4929, 'NAICS', 1624, 54, 1, 1649, 541350, 5 FROM dual
UNION ALL SELECT 4930, 'SEC', 2434, 3600, 2, 2439, 3621, 4 FROM dual
UNION ALL SELECT 4931, 'NAICS', 991, 4238, 3, 999, 42384, 4 FROM dual
UNION ALL SELECT 4932, 'NAICS', 1403, 511, 2, 1405, 511110, 5 FROM dual
UNION ALL SELECT 4933, 'NAICS', 1416, 5112, 3, 1417, 511210, 5 FROM dual
UNION ALL SELECT 4934, 'SEC', 2791, 20, 1, 2251, 2052, 4 FROM dual
UNION ALL SELECT 4935, 'NAICS', 180, 22, 1, 201, 221320, 5 FROM dual
UNION ALL SELECT 4936, 'NAICS', 205, 23, 1, 260, 2383, 3 FROM dual
UNION ALL SELECT 4937, 'SIC', 4310, 50, 1, 3785, 5153, 4 FROM dual
UNION ALL SELECT 4938, 'NAICS', 205, 23, 1, 224, 237130, 5 FROM dual
UNION ALL SELECT 4939, 'NAICS', 2038, 811, 2, 2043, 811113, 5 FROM dual
UNION ALL SELECT 4940, 'SEC', 2791, 20, 1, 2263, 2210, 3 FROM dual
UNION ALL SELECT 4941, 'SIC', 3337, 3300, 2, 3364, 3364, 4 FROM dual
UNION ALL SELECT 4942, 'SIC', 3953, 6230, 3, 3954, 6231, 4 FROM dual
UNION ALL SELECT 4943, 'NAICS', 1583, 5313, 3, 1586, 531312, 5 FROM dual
UNION ALL SELECT 4944, 'NAICS', 1904, 623, 2, 1914, 62331, 4 FROM dual
UNION ALL SELECT 4945, 'NAICS', 183, 22111, 4, 186, 221113, 5 FROM dual
UNION ALL SELECT 4946, 'SEC', 2791, 20, 1, 2415, 3550, 3 FROM dual
UNION ALL SELECT 4947, 'NAICS', 1663, 54143, 4, 1662, 541430, 5 FROM dual
UNION ALL SELECT 4948, 'SIC', 4310, 50, 1, 3737, 5050, 3 FROM dual
UNION ALL SELECT 4949, 'SIC', 4311, 52, 1, 3824, 5411, 4 FROM dual
UNION ALL SELECT 4950, 'NAICS', 1625, 541, 2, 1691, 54181, 4 FROM dual
UNION ALL SELECT 4951, 'NAICS', 1624, 54, 1, 1694, 541830, 5 FROM dual
UNION ALL SELECT 4952, 'SIC', 3190, 2700, 2, 3194, 2721, 4 FROM dual
UNION ALL SELECT 4953, 'NAICS', 2037, 81, 1, 2065, 811420, 5 FROM dual
UNION ALL SELECT 4954, 'NAICS', 2136, 921, 2, 2149, 92119, 4 FROM dual
UNION ALL SELECT 4955, 'SIC', 4308, 20, 1, 3160, 2531, 4 FROM dual
UNION ALL SELECT 4956, 'SEC', 2319, 2800, 2, 2324, 2833, 4 FROM dual
UNION ALL SELECT 4957, 'SIC', 3987, 6540, 3, 3988, 6541, 4 FROM dual
UNION ALL SELECT 4958, 'SEC', 2500, 3900, 2, 2507, 3944, 4 FROM dual
UNION ALL SELECT 4959, 'SEC', 2610, 5300, 2, 2612, 5311, 4 FROM dual
UNION ALL SELECT 4960, 'NAICS', 1, 11, 1, 70, 11231, 4 FROM dual
UNION ALL SELECT 4961, 'SEC', 2774, 8700, 2, 2779, 8734, 4 FROM dual
UNION ALL SELECT 4962, 'SIC', 4308, 20, 1, 3472, 3586, 4 FROM dual
UNION ALL SELECT 4963, 'SIC', 4311, 52, 1, 3809, 5250, 3 FROM dual
UNION ALL SELECT 4964, 'NAICS', 931, 423, 2, 1012, 42394, 4 FROM dual
UNION ALL SELECT 4965, 'NAICS', 1942, 71, 1, 1957, 711219, 5 FROM dual
UNION ALL SELECT 4966, 'SEC', 2790, 15, 1, 2232, 1531, 4 FROM dual
UNION ALL SELECT 4967, 'SIC', 2884, 1000, 2, 2888, 1021, 4 FROM dual
UNION ALL SELECT 4968, 'SIC', 4311, 52, 1, 3852, 5590, 3 FROM dual
UNION ALL SELECT 4969, 'SIC', 4306, 10, 1, 2900, 1099, 4 FROM dual
UNION ALL SELECT 4970, 'SEC', 2793, 50, 1, 2590, 5099, 4 FROM dual
UNION ALL SELECT 4971, 'NAICS', 1725, 56, 1, 1776, 56171, 4 FROM dual
UNION ALL SELECT 4972, 'SIC', 4308, 20, 1, 3139, 2448, 4 FROM dual
UNION ALL SELECT 4973, 'NAICS', 1008, 42392, 4, 1007, 423920, 5 FROM dual
UNION ALL SELECT 4974, 'NAICS', 1725, 56, 1, 1733, 5613, 3 FROM dual
UNION ALL SELECT 4975, 'SIC', 2982, 2000, 2, 3028, 2083, 4 FROM dual
UNION ALL SELECT 4976, 'SIC', 4311, 52, 1, 3850, 5570, 3 FROM dual
UNION ALL SELECT 4977, 'NAICS', 1402, 51, 1, 1409, 511130, 5 FROM dual
UNION ALL SELECT 4978, 'SEC', 2500, 3900, 2, 2511, 3990, 3 FROM dual
UNION ALL SELECT 4979, 'NAICS', 1623, 53311, 4, 1622, 533110, 5 FROM dual
UNION ALL SELECT 4980, 'SIC', 4308, 20, 1, 3528, 3711, 4 FROM dual
UNION ALL SELECT 4981, 'NAICS', 1, 11, 1, 122, 115113, 5 FROM dual
UNION ALL SELECT 4982, 'NAICS', 205, 23, 1, 231, 23731, 4 FROM dual
UNION ALL SELECT 4983, 'NAICS', 1569, 53, 1, 1611, 53231, 4 FROM dual
UNION ALL SELECT 4984, 'NAICS', 205, 23, 1, 272, 23839, 4 FROM dual
UNION ALL SELECT 4985, 'SIC', 4192, 8300, 2, 4199, 8360, 3 FROM dual
UNION ALL SELECT 4986, 'NAICS', 1942, 71, 1, 1960, 71131, 4 FROM dual
UNION ALL SELECT 4987, 'NAICS', 205, 23, 1, 223, 23712, 4 FROM dual
UNION ALL SELECT 4988, 'SEC', 2791, 20, 1, 2479, 3820, 3 FROM dual
UNION ALL SELECT 4989, 'SIC', 4313, 70, 1, 4054, 7338, 4 FROM dual
UNION ALL SELECT 4990, 'SIC', 2884, 1000, 2, 2898, 1090, 3 FROM dual
UNION ALL SELECT 4991, 'NAICS', 1480, 52, 1, 1559, 52512, 4 FROM dual
UNION ALL SELECT 4992, 'SEC', 2500, 3900, 2, 2504, 3931, 4 FROM dual
UNION ALL SELECT 4993, 'SIC', 3886, 5900, 2, 3892, 5932, 4 FROM dual
UNION ALL SELECT 4994, 'NAICS', 235, 238, 2, 248, 23816, 4 FROM dual
UNION ALL SELECT 4995, 'SIC', 4075, 7380, 3, 4077, 7382, 4 FROM dual
UNION ALL SELECT 4996, 'SIC', 3977, 6500, 2, 3982, 6515, 4 FROM dual
UNION ALL SELECT 4997, 'SIC', 4308, 20, 1, 2995, 2033, 4 FROM dual
UNION ALL SELECT 4998, 'SIC', 4306, 10, 1, 2931, 1470, 3 FROM dual
UNION ALL SELECT 4999, 'NAICS', 1920, 624, 2, 1934, 624230, 5 FROM dual
UNION ALL SELECT 5000, 'NAICS', 1725, 56, 1, 1777, 561720, 5 FROM dual
UNION ALL SELECT 5001, 'NAICS', 1799, 56221, 4, 1800, 562211, 5 FROM dual
UNION ALL SELECT 5002, 'NAICS', 2032, 72251, 4, 2036, 722515, 5 FROM dual
UNION ALL SELECT 5003, 'SIC', 4310, 50, 1, 3799, 5192, 4 FROM dual
UNION ALL SELECT 5004, 'NAICS', 1954, 71121, 4, 1955, 711211, 5 FROM dual
UNION ALL SELECT 5005, 'NAICS', 180, 22, 1, 192, 22112, 4 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 5006, 'SEC', 2796, 70, 1, 2777, 8730, 3 FROM dual
UNION ALL SELECT 5007, 'SIC', 2939, 1500, 2, 2943, 1530, 3 FROM dual
UNION ALL SELECT 5008, 'SIC', 4308, 20, 1, 3076, 2284, 4 FROM dual
UNION ALL SELECT 5009, 'SIC', 3676, 4780, 3, 3677, 4783, 4 FROM dual
UNION ALL SELECT 5010, 'NAICS', 3, 1111, 3, 10, 111140, 5 FROM dual
UNION ALL SELECT 5011, 'SEC', 2303, 2710, 3, 2304, 2711, 4 FROM dual
UNION ALL SELECT 5012, 'NAICS', 1015, 424, 2, 1067, 424710, 5 FROM dual
UNION ALL SELECT 5013, 'SIC', 3686, 4830, 3, 3688, 4833, 4 FROM dual
UNION ALL SELECT 5014, 'SIC', 3050, 2200, 2, 3071, 2270, 3 FROM dual
UNION ALL SELECT 5015, 'SIC', 3127, 2420, 3, 3129, 2426, 4 FROM dual
UNION ALL SELECT 5016, 'SIC', 4017, 7200, 2, 4027, 7220, 3 FROM dual
UNION ALL SELECT 5017, 'SIC', 3041, 2100, 2, 3043, 2111, 4 FROM dual
UNION ALL SELECT 5018, 'SIC', 4040, 7300, 2, 4067, 7372, 4 FROM dual
UNION ALL SELECT 5019, 'NAICS', 931, 423, 2, 971, 423510, 5 FROM dual
UNION ALL SELECT 5020, 'SIC', 2962, 1740, 3, 2963, 1741, 4 FROM dual
UNION ALL SELECT 5021, 'NAICS', 1, 11, 1, 16, 11119, 4 FROM dual
UNION ALL SELECT 5022, 'SEC', 2791, 20, 1, 2439, 3621, 4 FROM dual
UNION ALL SELECT 5023, 'SIC', 4314, 90, 1, 4253, 9131, 4 FROM dual
UNION ALL SELECT 5024, 'NAICS', 1, 11, 1, 30, 111332, 5 FROM dual
UNION ALL SELECT 5025, 'NAICS', 56, 112, 2, 68, 1123, 3 FROM dual
UNION ALL SELECT 5026, 'SEC', 2568, 5000, 2, 2579, 5060, 3 FROM dual
UNION ALL SELECT 5027, 'SIC', 4313, 70, 1, 4084, 7514, 4 FROM dual
UNION ALL SELECT 5028, 'SIC', 2798, 100, 2, 2812, 161, 4 FROM dual
UNION ALL SELECT 5029, 'NAICS', 1823, 6114, 3, 1829, 61143, 4 FROM dual
UNION ALL SELECT 5030, 'NAICS', 931, 423, 2, 975, 4236, 3 FROM dual
UNION ALL SELECT 5031, 'NAICS', 1720, 5511, 3, 1724, 551114, 5 FROM dual
UNION ALL SELECT 5032, 'SIC', 4308, 20, 1, 3398, 3465, 4 FROM dual
UNION ALL SELECT 5033, 'NAICS', 1419, 512, 2, 1427, 512132, 5 FROM dual
UNION ALL SELECT 5034, 'NAICS', 2003, 72, 1, 2021, 7223, 3 FROM dual
UNION ALL SELECT 5035, 'NAICS', 930, 42, 1, 992, 423810, 5 FROM dual
UNION ALL SELECT 5036, 'NAICS', 1812, 61, 1, 1816, 61111, 4 FROM dual
UNION ALL SELECT 5037, 'NAICS', 2135, 92, 1, 2161, 92215, 4 FROM dual
UNION ALL SELECT 5038, 'SIC', 4314, 90, 1, 4272, 9431, 4 FROM dual
UNION ALL SELECT 5039, 'NAICS', 2061, 8114, 3, 2065, 811420, 5 FROM dual
UNION ALL SELECT 5040, 'SEC', 2791, 20, 1, 2353, 3100, 2 FROM dual
UNION ALL SELECT 5041, 'SIC', 3837, 5500, 2, 3849, 5561, 4 FROM dual
UNION ALL SELECT 5042, 'SIC', 3693, 4900, 2, 3694, 4910, 3 FROM dual
UNION ALL SELECT 5043, 'SIC', 3337, 3300, 2, 3339, 3312, 4 FROM dual
UNION ALL SELECT 5044, 'NAICS', 118, 1151, 3, 123, 115114, 5 FROM dual
UNION ALL SELECT 5045, 'SEC', 2795, 60, 1, 2704, 6790, 3 FROM dual
UNION ALL SELECT 5046, 'SIC', 3552, 3800, 2, 3566, 3842, 4 FROM dual
UNION ALL SELECT 5047, 'NAICS', 1953, 7112, 3, 1957, 711219, 5 FROM dual
UNION ALL SELECT 5048, 'NAICS', 236, 2381, 3, 240, 23812, 4 FROM dual
UNION ALL SELECT 5049, 'NAICS', 1689, 5418, 3, 1691, 54181, 4 FROM dual
UNION ALL SELECT 5050, 'NAICS', 1726, 561, 2, 1771, 56162, 4 FROM dual
UNION ALL SELECT 5051, 'SIC', 3877, 5730, 3, 3880, 5735, 4 FROM dual
UNION ALL SELECT 5052, 'SIC', 3606, 4110, 3, 3608, 4119, 4 FROM dual
UNION ALL SELECT 5053, 'NAICS', 2151, 9221, 3, 2159, 92214, 4 FROM dual
UNION ALL SELECT 5054, 'SEC', 2446, 3660, 3, 2449, 3669, 4 FROM dual
UNION ALL SELECT 5055, 'NAICS', 1480, 52, 1, 1563, 525910, 5 FROM dual
UNION ALL SELECT 5056, 'NAICS', 1851, 621, 2, 1882, 62151, 4 FROM dual
UNION ALL SELECT 5057, 'NAICS', 2039, 8111, 3, 2049, 811191, 5 FROM dual
UNION ALL SELECT 5058, 'SEC', 2791, 20, 1, 2384, 3400, 2 FROM dual
UNION ALL SELECT 5059, 'SIC', 4308, 20, 1, 3217, 2813, 4 FROM dual
UNION ALL SELECT 5060, 'SIC', 3246, 2890, 3, 3251, 2899, 4 FROM dual
UNION ALL SELECT 5061, 'NAICS', 1001, 42385, 4, 1000, 423850, 5 FROM dual
UNION ALL SELECT 5062, 'SEC', 2434, 3600, 2, 2442, 3640, 3 FROM dual
UNION ALL SELECT 5063, 'SIC', 3282, 3100, 2, 3295, 3161, 4 FROM dual
UNION ALL SELECT 5064, 'SEC', 2476, 3800, 2, 2482, 3823, 4 FROM dual
UNION ALL SELECT 5065, 'SEC', 2792, 40, 1, 2512, 4000, 2 FROM dual
UNION ALL SELECT 5066, 'NAICS', 1792, 562, 2, 1795, 562111, 5 FROM dual
UNION ALL SELECT 5067, 'SIC', 3480, 3600, 2, 3505, 3651, 4 FROM dual
UNION ALL SELECT 5068, 'SEC', 2792, 40, 1, 2560, 4931, 4 FROM dual
UNION ALL SELECT 5069, 'NAICS', 231, 23731, 4, 230, 237310, 5 FROM dual
UNION ALL SELECT 5070, 'NAICS', 1851, 621, 2, 1881, 6215, 3 FROM dual
UNION ALL SELECT 5071, 'NAICS', 2038, 811, 2, 2041, 811111, 5 FROM dual
UNION ALL SELECT 5072, 'NAICS', 1793, 5621, 3, 1797, 562119, 5 FROM dual
UNION ALL SELECT 5073, 'SIC', 4307, 15, 1, 2942, 1522, 4 FROM dual
UNION ALL SELECT 5074, 'SIC', 2875, 900, 2, 2878, 913, 4 FROM dual
UNION ALL SELECT 5075, 'NAICS', 1035, 4244, 3, 1050, 424480, 5 FROM dual
UNION ALL SELECT 5076, 'SEC', 2476, 3800, 2, 2492, 3844, 4 FROM dual
UNION ALL SELECT 5077, 'SEC', 2649, 6000, 2, 2654, 6030, 3 FROM dual
UNION ALL SELECT 5078, 'SIC', 3419, 3500, 2, 3437, 3543, 4 FROM dual
UNION ALL SELECT 5079, 'SIC', 4136, 7950, 3, 4137, 7951, 4 FROM dual
UNION ALL SELECT 5080, 'SIC', 3713, 4970, 3, 3714, 4971, 4 FROM dual
UNION ALL SELECT 5081, 'NAICS', 181, 221, 2, 198, 2213, 3 FROM dual
UNION ALL SELECT 5082, 'SIC', 3605, 4100, 2, 3611, 4130, 3 FROM dual
UNION ALL SELECT 5083, 'SIC', 3282, 3100, 2, 3294, 3160, 3 FROM dual
UNION ALL SELECT 5084, 'NAICS', 68, 1123, 3, 77, 112390, 5 FROM dual
UNION ALL SELECT 5085, 'NAICS', 2039, 8111, 3, 2040, 81111, 4 FROM dual
UNION ALL SELECT 5086, 'SIC', 2798, 100, 2, 2807, 132, 4 FROM dual
UNION ALL SELECT 5087, 'SIC', 2918, 1400, 2, 2938, 1499, 4 FROM dual
UNION ALL SELECT 5088, 'SEC', 2795, 60, 1, 2705, 6792, 4 FROM dual
UNION ALL SELECT 5089, 'SIC', 3451, 3560, 3, 3455, 3564, 4 FROM dual
UNION ALL SELECT 5090, 'NAICS', 1, 11, 1, 105, 1133, 3 FROM dual
UNION ALL SELECT 5091, 'SEC', 2280, 2450, 3, 2281, 2451, 4 FROM dual
UNION ALL SELECT 5092, 'SIC', 4308, 20, 1, 3143, 2452, 4 FROM dual
UNION ALL SELECT 5093, 'SEC', 2791, 20, 1, 2355, 3200, 2 FROM dual
UNION ALL SELECT 5094, 'SIC', 4310, 50, 1, 3755, 5088, 4 FROM dual
UNION ALL SELECT 5095, 'NAICS', 2103, 813, 2, 2129, 813990, 5 FROM dual
UNION ALL SELECT 5096, 'SEC', 2791, 20, 1, 2306, 2721, 4 FROM dual
UNION ALL SELECT 5097, 'NAICS', 1550, 52429, 4, 1552, 524292, 5 FROM dual
UNION ALL SELECT 5098, 'NAICS', 2205, 9281, 3, 2207, 92811, 4 FROM dual
UNION ALL SELECT 5099, 'SEC', 2250, 2050, 3, 2251, 2052, 4 FROM dual
UNION ALL SELECT 5100, 'SIC', 2945, 1540, 3, 2947, 1542, 4 FROM dual
UNION ALL SELECT 5101, 'NAICS', 182, 2211, 3, 187, 221114, 5 FROM dual
UNION ALL SELECT 5102, 'SIC', 3620, 4200, 2, 3623, 4213, 4 FROM dual
UNION ALL SELECT 5103, 'NAICS', 1, 11, 1, 8, 111130, 5 FROM dual
UNION ALL SELECT 5104, 'SEC', 2403, 3500, 2, 2415, 3550, 3 FROM dual
UNION ALL SELECT 5105, 'NAICS', 156, 2123, 3, 160, 212313, 5 FROM dual
UNION ALL SELECT 5106, 'NAICS', 138, 212, 2, 168, 212391, 5 FROM dual
UNION ALL SELECT 5107, 'NAICS', 1725, 56, 1, 1806, 56291, 4 FROM dual
UNION ALL SELECT 5108, 'SIC', 4113, 7800, 2, 4119, 7829, 4 FROM dual
UNION ALL SELECT 5109, 'NAICS', 206, 236, 2, 213, 2362, 3 FROM dual
UNION ALL SELECT 5110, 'SIC', 3715, 5000, 2, 3758, 5092, 4 FROM dual
UNION ALL SELECT 5111, 'SIC', 2855, 740, 3, 2856, 741, 4 FROM dual
UNION ALL SELECT 5112, 'NAICS', 232, 2379, 3, 234, 23799, 4 FROM dual
UNION ALL SELECT 5113, 'NAICS', 2200, 927, 2, 2203, 92711, 4 FROM dual
UNION ALL SELECT 5114, 'SIC', 4308, 20, 1, 3422, 3519, 4 FROM dual
UNION ALL SELECT 5115, 'NAICS', 1969, 712, 2, 1975, 712130, 5 FROM dual
UNION ALL SELECT 5116, 'NAICS', 2149, 92119, 4, 2148, 921190, 5 FROM dual
UNION ALL SELECT 5117, 'SEC', 2791, 20, 1, 2389, 3430, 3 FROM dual
UNION ALL SELECT 5118, 'NAICS', 1928, 6242, 3, 1929, 624210, 5 FROM dual
UNION ALL SELECT 5119, 'NAICS', 1921, 6241, 3, 1927, 62419, 4 FROM dual
UNION ALL SELECT 5120, 'NAICS', 1760, 56151, 4, 1759, 561510, 5 FROM dual
UNION ALL SELECT 5121, 'SIC', 4305, 1, 1, 2815, 172, 4 FROM dual
UNION ALL SELECT 5122, 'SIC', 3504, 3650, 3, 3506, 3652, 4 FROM dual
UNION ALL SELECT 5123, 'NAICS', 1598, 5322, 3, 1600, 53221, 4 FROM dual
UNION ALL SELECT 5124, 'NAICS', 2135, 92, 1, 2192, 926120, 5 FROM dual
UNION ALL SELECT 5125, 'NAICS', 1657, 5414, 3, 1659, 54141, 4 FROM dual
UNION ALL SELECT 5126, 'SIC', 4309, 40, 1, 3687, 4832, 4 FROM dual
UNION ALL SELECT 5127, 'SIC', 4312, 60, 1, 3971, 6371, 4 FROM dual
UNION ALL SELECT 5128, 'NAICS', 235, 238, 2, 263, 238320, 5 FROM dual
UNION ALL SELECT 5129, 'SIC', 3951, 6220, 3, 3952, 6221, 4 FROM dual
UNION ALL SELECT 5130, 'NAICS', 1852, 6211, 3, 1855, 621112, 5 FROM dual
UNION ALL SELECT 5131, 'NAICS', 172, 213, 2, 177, 213113, 5 FROM dual
UNION ALL SELECT 5132, 'SIC', 2982, 2000, 2, 3000, 2040, 3 FROM dual
UNION ALL SELECT 5133, 'NAICS', 1743, 56141, 4, 1742, 561410, 5 FROM dual
UNION ALL SELECT 5134, 'NAICS', 205, 23, 1, 270, 23835, 4 FROM dual
UNION ALL SELECT 5135, 'SIC', 4308, 20, 1, 3020, 2070, 3 FROM dual
UNION ALL SELECT 5136, 'NAICS', 1573, 53111, 4, 1572, 531110, 5 FROM dual
UNION ALL SELECT 5137, 'SIC', 4208, 8600, 2, 4221, 8690, 3 FROM dual
UNION ALL SELECT 5138, 'SEC', 2591, 5100, 2, 2592, 5110, 3 FROM dual
UNION ALL SELECT 5139, 'SIC', 3552, 3800, 2, 3567, 3843, 4 FROM dual
UNION ALL SELECT 5140, 'SIC', 4171, 8090, 3, 4174, 8099, 4 FROM dual
UNION ALL SELECT 5141, 'SIC', 4040, 7300, 2, 4059, 7352, 4 FROM dual
UNION ALL SELECT 5142, 'NAICS', 2, 111, 2, 5, 11111, 4 FROM dual
UNION ALL SELECT 5143, 'NAICS', 991, 4238, 3, 997, 42383, 4 FROM dual
UNION ALL SELECT 5144, 'SEC', 2458, 3700, 2, 2464, 3716, 4 FROM dual
UNION ALL SELECT 5145, 'SIC', 4308, 20, 1, 3084, 2310, 3 FROM dual
UNION ALL SELECT 5146, 'SIC', 4146, 8000, 2, 4165, 8069, 4 FROM dual
UNION ALL SELECT 5147, 'NAICS', 2040, 81111, 4, 2043, 811113, 5 FROM dual
UNION ALL SELECT 5148, 'SIC', 3601, 4000, 2, 3602, 4010, 3 FROM dual
UNION ALL SELECT 5149, 'SIC', 3576, 3900, 2, 3592, 3960, 3 FROM dual
UNION ALL SELECT 5150, 'SIC', 4308, 20, 1, 3523, 3694, 4 FROM dual
UNION ALL SELECT 5151, 'SIC', 3693, 4900, 2, 3714, 4971, 4 FROM dual
UNION ALL SELECT 5152, 'NAICS', 1812, 61, 1, 1846, 611699, 5 FROM dual
UNION ALL SELECT 5153, 'NAICS', 1904, 623, 2, 1916, 623312, 5 FROM dual
UNION ALL SELECT 5154, 'SIC', 3762, 5100, 2, 3777, 5143, 4 FROM dual
UNION ALL SELECT 5155, 'SEC', 2791, 20, 1, 2262, 2200, 2 FROM dual
UNION ALL SELECT 5156, 'NAICS', 57, 1121, 3, 61, 112120, 5 FROM dual
UNION ALL SELECT 5157, 'SIC', 2958, 1720, 3, 2959, 1721, 4 FROM dual
UNION ALL SELECT 5158, 'SIC', 4308, 20, 1, 3156, 2520, 3 FROM dual
UNION ALL SELECT 5159, 'SEC', 2424, 3570, 3, 2427, 3575, 4 FROM dual
UNION ALL SELECT 5160, 'SIC', 4311, 52, 1, 3836, 5499, 4 FROM dual
UNION ALL SELECT 5161, 'NAICS', 2004, 721, 2, 2016, 721214, 5 FROM dual
UNION ALL SELECT 5162, 'NAICS', 1624, 54, 1, 1672, 5416, 3 FROM dual
UNION ALL SELECT 5163, 'SEC', 2791, 20, 1, 2412, 3537, 4 FROM dual
UNION ALL SELECT 5164, 'SIC', 3301, 3200, 2, 3320, 3264, 4 FROM dual
UNION ALL SELECT 5165, 'NAICS', 1624, 54, 1, 1674, 541611, 5 FROM dual
UNION ALL SELECT 5166, 'SIC', 3337, 3300, 2, 3349, 3330, 3 FROM dual
UNION ALL SELECT 5167, 'SIC', 4309, 40, 1, 3660, 4580, 3 FROM dual
UNION ALL SELECT 5168, 'SEC', 2791, 20, 1, 2467, 3724, 4 FROM dual
UNION ALL SELECT 5169, 'SIC', 4308, 20, 1, 3094, 2331, 4 FROM dual
UNION ALL SELECT 5170, 'SIC', 4308, 20, 1, 3153, 2515, 4 FROM dual
UNION ALL SELECT 5171, 'SIC', 3149, 2510, 3, 3151, 2512, 4 FROM dual
UNION ALL SELECT 5172, 'SIC', 4308, 20, 1, 3043, 2111, 4 FROM dual
UNION ALL SELECT 5173, 'SIC', 4259, 9220, 3, 4264, 9229, 4 FROM dual
UNION ALL SELECT 5174, 'SIC', 4223, 8700, 2, 4231, 8731, 4 FROM dual
UNION ALL SELECT 5175, 'SEC', 2791, 20, 1, 2281, 2451, 4 FROM dual
UNION ALL SELECT 5176, 'NAICS', 105, 1133, 3, 107, 11331, 4 FROM dual
UNION ALL SELECT 5177, 'NAICS', 99, 1131, 3, 100, 113110, 5 FROM dual
UNION ALL SELECT 5178, 'SIC', 4017, 7200, 2, 4023, 7216, 4 FROM dual
UNION ALL SELECT 5179, 'SIC', 4308, 20, 1, 3369, 3398, 4 FROM dual
UNION ALL SELECT 5180, 'NAICS', 1851, 621, 2, 1876, 62149, 4 FROM dual
UNION ALL SELECT 5181, 'SIC', 4314, 90, 1, 4285, 9610, 3 FROM dual
UNION ALL SELECT 5182, 'NAICS', 1970, 7121, 3, 1974, 71212, 4 FROM dual
UNION ALL SELECT 5183, 'SIC', 4308, 20, 1, 3106, 2370, 3 FROM dual
UNION ALL SELECT 5184, 'SIC', 4308, 20, 1, 3552, 3800, 2 FROM dual
UNION ALL SELECT 5185, 'NAICS', 65, 1122, 3, 67, 11221, 4 FROM dual
UNION ALL SELECT 5186, 'NAICS', 930, 42, 1, 949, 423320, 5 FROM dual
UNION ALL SELECT 5187, 'NAICS', 1630, 54112, 4, 1629, 541120, 5 FROM dual
UNION ALL SELECT 5188, 'NAICS', 1861, 62131, 4, 1860, 621310, 5 FROM dual
UNION ALL SELECT 5189, 'SIC', 2918, 1400, 2, 2924, 1429, 4 FROM dual
UNION ALL SELECT 5190, 'NAICS', 2, 111, 2, 26, 111320, 5 FROM dual
UNION ALL SELECT 5191, 'NAICS', 2188, 926, 2, 2189, 9261, 3 FROM dual
UNION ALL SELECT 5192, 'SIC', 3552, 3800, 2, 3555, 3820, 3 FROM dual
UNION ALL SELECT 5193, 'SIC', 3214, 2800, 2, 3245, 2879, 4 FROM dual
UNION ALL SELECT 5194, 'SIC', 3788, 5160, 3, 3789, 5162, 4 FROM dual
UNION ALL SELECT 5195, 'NAICS', 235, 238, 2, 277, 23899, 4 FROM dual
UNION ALL SELECT 5196, 'NAICS', 1624, 54, 1, 1628, 54111, 4 FROM dual
UNION ALL SELECT 5197, 'SIC', 4208, 8600, 2, 4218, 8651, 4 FROM dual
UNION ALL SELECT 5198, 'NAICS', 1771, 56162, 4, 1773, 561622, 5 FROM dual
UNION ALL SELECT 5199, 'NAICS', 1850, 62, 1, 1877, 621491, 5 FROM dual
UNION ALL SELECT 5200, 'NAICS', 2135, 92, 1, 2183, 9251, 3 FROM dual
UNION ALL SELECT 5201, 'NAICS', 1725, 56, 1, 1797, 562119, 5 FROM dual
UNION ALL SELECT 5202, 'SEC', 2606, 5210, 3, 2607, 5211, 4 FROM dual
UNION ALL SELECT 5203, 'NAICS', 1403, 511, 2, 1415, 511199, 5 FROM dual
UNION ALL SELECT 5204, 'NAICS', 1419, 512, 2, 1423, 512120, 5 FROM dual
UNION ALL SELECT 5205, 'SIC', 3804, 5200, 2, 3814, 5271, 4 FROM dual
UNION ALL SELECT 5206, 'SIC', 4081, 7500, 2, 4093, 7536, 4 FROM dual
UNION ALL SELECT 5207, 'NAICS', 1526, 5239, 3, 1527, 523910, 5 FROM dual
UNION ALL SELECT 5208, 'NAICS', 1935, 62423, 4, 1934, 624230, 5 FROM dual
UNION ALL SELECT 5209, 'NAICS', 2135, 92, 1, 2179, 92411, 4 FROM dual
UNION ALL SELECT 5210, 'SIC', 3480, 3600, 2, 3521, 3691, 4 FROM dual
UNION ALL SELECT 5211, 'SIC', 3480, 3600, 2, 3496, 3640, 3 FROM dual
UNION ALL SELECT 5212, 'SIC', 4311, 52, 1, 3879, 5734, 4 FROM dual
UNION ALL SELECT 5213, 'SIC', 3822, 5400, 2, 3823, 5410, 3 FROM dual
UNION ALL SELECT 5214, 'NAICS', 975, 4236, 3, 980, 423690, 5 FROM dual
UNION ALL SELECT 5215, 'SEC', 2791, 20, 1, 2394, 3444, 4 FROM dual
UNION ALL SELECT 5216, 'NAICS', 1419, 512, 2, 1440, 512290, 5 FROM dual
UNION ALL SELECT 5217, 'NAICS', 1538, 52411, 4, 1540, 524114, 5 FROM dual
UNION ALL SELECT 5218, 'NAICS', 56, 112, 2, 60, 112112, 5 FROM dual
UNION ALL SELECT 5219, 'NAICS', 1894, 622, 2, 1899, 622210, 5 FROM dual
UNION ALL SELECT 5220, 'SIC', 4113, 7800, 2, 4121, 7832, 4 FROM dual
UNION ALL SELECT 5221, 'NAICS', 1850, 62, 1, 1873, 62141, 4 FROM dual
UNION ALL SELECT 5222, 'SIC', 3882, 5800, 2, 3885, 5813, 4 FROM dual
UNION ALL SELECT 5223, 'SEC', 2791, 20, 1, 2382, 3360, 3 FROM dual
UNION ALL SELECT 5224, 'SIC', 3636, 4400, 2, 3649, 4490, 3 FROM dual
UNION ALL SELECT 5225, 'NAICS', 1053, 42449, 4, 1052, 424490, 5 FROM dual
UNION ALL SELECT 5226, 'NAICS', 2003, 72, 1, 2011, 721191, 5 FROM dual
UNION ALL SELECT 5227, 'NAICS', 2038, 811, 2, 2070, 81149, 4 FROM dual
UNION ALL SELECT 5228, 'NAICS', 1624, 54, 1, 1692, 541820, 5 FROM dual
UNION ALL SELECT 5229, 'SIC', 3715, 5000, 2, 3722, 5021, 4 FROM dual
UNION ALL SELECT 5230, 'SIC', 3033, 2090, 3, 3039, 2098, 4 FROM dual
UNION ALL SELECT 5231, 'NAICS', 2188, 926, 2, 2198, 926150, 5 FROM dual
UNION ALL SELECT 5232, 'NAICS', 1625, 541, 2, 1687, 541720, 5 FROM dual
UNION ALL SELECT 5233, 'NAICS', 1, 11, 1, 24, 111310, 5 FROM dual
UNION ALL SELECT 5234, 'SIC', 4305, 1, 1, 2860, 752, 4 FROM dual
UNION ALL SELECT 5235, 'NAICS', 2005, 7211, 3, 2010, 72119, 4 FROM dual
UNION ALL SELECT 5236, 'SEC', 2517, 4200, 2, 2518, 4210, 3 FROM dual
UNION ALL SELECT 5237, 'NAICS', 1, 11, 1, 89, 1129, 3 FROM dual
UNION ALL SELECT 5238, 'NAICS', 1569, 53, 1, 1623, 53311, 4 FROM dual
UNION ALL SELECT 5239, 'NAICS', 2103, 813, 2, 2122, 81391, 4 FROM dual
UNION ALL SELECT 5240, 'SIC', 4308, 20, 1, 3587, 3950, 3 FROM dual
UNION ALL SELECT 5241, 'NAICS', 1, 11, 1, 72, 11232, 4 FROM dual
UNION ALL SELECT 5242, 'NAICS', 931, 423, 2, 995, 42382, 4 FROM dual
UNION ALL SELECT 5243, 'NAICS', 1486, 5221, 3, 1493, 522190, 5 FROM dual
UNION ALL SELECT 5244, 'SEC', 2794, 52, 1, 2609, 5271, 4 FROM dual
UNION ALL SELECT 5245, 'SIC', 4308, 20, 1, 3266, 3050, 3 FROM dual
UNION ALL SELECT 5246, 'SIC', 4223, 8700, 2, 4229, 8721, 4 FROM dual
UNION ALL SELECT 5247, 'SEC', 2794, 52, 1, 2638, 5810, 3 FROM dual
UNION ALL SELECT 5248, 'NAICS', 1, 11, 1, 32, 111334, 5 FROM dual
UNION ALL SELECT 5249, 'NAICS', 1035, 4244, 3, 1052, 424490, 5 FROM dual
UNION ALL SELECT 5250, 'SIC', 4306, 10, 1, 2911, 1311, 4 FROM dual
UNION ALL SELECT 5251, 'SIC', 4100, 7600, 2, 4109, 7690, 3 FROM dual
UNION ALL SELECT 5252, 'NAICS', 991, 4238, 3, 996, 423830, 5 FROM dual
UNION ALL SELECT 5253, 'SIC', 4308, 20, 1, 3362, 3360, 3 FROM dual
UNION ALL SELECT 5254, 'NAICS', 205, 23, 1, 261, 238310, 5 FROM dual
UNION ALL SELECT 5255, 'NAICS', 2037, 81, 1, 2044, 811118, 5 FROM dual
UNION ALL SELECT 5256, 'SIC', 3636, 4400, 2, 3651, 4492, 4 FROM dual
UNION ALL SELECT 5257, 'SIC', 3978, 6510, 3, 3980, 6513, 4 FROM dual
UNION ALL SELECT 5258, 'SEC', 2695, 6510, 3, 2698, 6519, 4 FROM dual
UNION ALL SELECT 5259, 'SEC', 2371, 3300, 2, 2380, 3350, 3 FROM dual
UNION ALL SELECT 5260, 'NAICS', 174, 21311, 4, 178, 213114, 5 FROM dual
UNION ALL SELECT 5261, 'NAICS', 1990, 7139, 3, 2002, 71399, 4 FROM dual
UNION ALL SELECT 5262, 'NAICS', 1602, 53222, 4, 1601, 532220, 5 FROM dual
UNION ALL SELECT 5263, 'NAICS', 2103, 813, 2, 2118, 813410, 5 FROM dual
UNION ALL SELECT 5264, 'SIC', 4281, 9530, 3, 4283, 9532, 4 FROM dual
UNION ALL SELECT 5265, 'SIC', 2935, 1480, 3, 2936, 1481, 4 FROM dual
UNION ALL SELECT 5266, 'NAICS', 2120, 8139, 3, 2126, 81393, 4 FROM dual
UNION ALL SELECT 5267, 'SIC', 3108, 2380, 3, 3111, 2385, 4 FROM dual
UNION ALL SELECT 5268, 'SIC', 4081, 7500, 2, 4092, 7534, 4 FROM dual
UNION ALL SELECT 5269, 'SIC', 4308, 20, 1, 3375, 3420, 3 FROM dual
UNION ALL SELECT 5270, 'NAICS', 1624, 54, 1, 1627, 541110, 5 FROM dual
UNION ALL SELECT 5271, 'SEC', 2703, 6700, 2, 2708, 6799, 4 FROM dual
UNION ALL SELECT 5272, 'NAICS', 205, 23, 1, 213, 2362, 3 FROM dual
UNION ALL SELECT 5273, 'NAICS', 1725, 56, 1, 1773, 561622, 5 FROM dual
UNION ALL SELECT 5274, 'NAICS', 2052, 8112, 3, 2055, 811212, 5 FROM dual
UNION ALL SELECT 5275, 'SEC', 2796, 70, 1, 2719, 7340, 3 FROM dual
UNION ALL SELECT 5276, 'NAICS', 930, 42, 1, 1006, 42391, 4 FROM dual
UNION ALL SELECT 5277, 'SEC', 2434, 3600, 2, 2441, 3634, 4 FROM dual
UNION ALL SELECT 5278, 'SIC', 3301, 3200, 2, 3307, 3230, 3 FROM dual
UNION ALL SELECT 5279, 'NAICS', 140, 21211, 4, 142, 212112, 5 FROM dual
UNION ALL SELECT 5280, 'NAICS', 2100, 81293, 4, 2099, 812930, 5 FROM dual
UNION ALL SELECT 5281, 'SIC', 4082, 7510, 3, 4085, 7515, 4 FROM dual
UNION ALL SELECT 5282, 'SEC', 2795, 60, 1, 4328, 6795, 4 FROM dual
UNION ALL SELECT 5283, 'SIC', 4308, 20, 1, 3382, 3432, 4 FROM dual
UNION ALL SELECT 5284, 'SIC', 4018, 7210, 3, 4025, 7218, 4 FROM dual
UNION ALL SELECT 5285, 'SIC', 4089, 7530, 3, 4094, 7537, 4 FROM dual
UNION ALL SELECT 5286, 'NAICS', 51, 11194, 4, 50, 111940, 5 FROM dual
UNION ALL SELECT 5287, 'NAICS', 2040, 81111, 4, 2041, 811111, 5 FROM dual
UNION ALL SELECT 5288, 'SIC', 3246, 2890, 3, 3248, 2892, 4 FROM dual
UNION ALL SELECT 5289, 'SEC', 2792, 40, 1, 2555, 4920, 3 FROM dual
UNION ALL SELECT 5290, 'NAICS', 1970, 7121, 3, 1975, 712130, 5 FROM dual
UNION ALL SELECT 5291, 'NAICS', 1070, 42472, 4, 1069, 424720, 5 FROM dual
UNION ALL SELECT 5292, 'NAICS', 930, 42, 1, 1031, 424330, 5 FROM dual
UNION ALL SELECT 5293, 'SIC', 2855, 740, 3, 2857, 742, 4 FROM dual
UNION ALL SELECT 5294, 'NAICS', 1480, 52, 1, 1555, 5251, 3 FROM dual
UNION ALL SELECT 5295, 'NAICS', 132, 21, 1, 141, 212111, 5 FROM dual
UNION ALL SELECT 5296, 'NAICS', 1726, 561, 2, 1761, 561520, 5 FROM dual
UNION ALL SELECT 5297, 'SIC', 4311, 52, 1, 3821, 5399, 4 FROM dual
UNION ALL SELECT 5298, 'NAICS', 1090, 4251, 3, 1092, 42511, 4 FROM dual
UNION ALL SELECT 5299, 'SIC', 4256, 9200, 2, 4264, 9229, 4 FROM dual
UNION ALL SELECT 5300, 'SIC', 4284, 9600, 2, 4295, 9660, 3 FROM dual
UNION ALL SELECT 5301, 'NAICS', 208, 23611, 4, 209, 236115, 5 FROM dual
UNION ALL SELECT 5302, 'SEC', 2791, 20, 1, 2249, 2040, 3 FROM dual
UNION ALL SELECT 5303, 'SEC', 2403, 3500, 2, 2412, 3537, 4 FROM dual
UNION ALL SELECT 5304, 'NAICS', 144, 2122, 3, 150, 21223, 4 FROM dual
UNION ALL SELECT 5305, 'SEC', 2532, 4580, 3, 2533, 4581, 4 FROM dual
UNION ALL SELECT 5306, 'NAICS', 1725, 56, 1, 1762, 56152, 4 FROM dual
UNION ALL SELECT 5307, 'SEC', 2791, 20, 1, 2341, 2990, 3 FROM dual
UNION ALL SELECT 5308, 'SIC', 4314, 90, 1, 4263, 9224, 4 FROM dual
UNION ALL SELECT 5309, 'SEC', 2791, 20, 1, 2441, 3634, 4 FROM dual
UNION ALL SELECT 5310, 'SIC', 4247, 9100, 2, 4248, 9110, 3 FROM dual
UNION ALL SELECT 5311, 'SIC', 4308, 20, 1, 3002, 2043, 4 FROM dual
UNION ALL SELECT 5312, 'NAICS', 56, 112, 2, 62, 11212, 4 FROM dual
UNION ALL SELECT 5313, 'SIC', 3204, 2760, 3, 3205, 2761, 4 FROM dual
UNION ALL SELECT 5314, 'NAICS', 930, 42, 1, 954, 42339, 4 FROM dual
UNION ALL SELECT 5315, 'SIC', 3384, 3440, 3, 3387, 3443, 4 FROM dual
UNION ALL SELECT 5316, 'SIC', 3337, 3300, 2, 3348, 3325, 4 FROM dual
UNION ALL SELECT 5317, 'NAICS', 1579, 53119, 4, 1578, 531190, 5 FROM dual
UNION ALL SELECT 5318, 'NAICS', 1850, 62, 1, 1899, 622210, 5 FROM dual
UNION ALL SELECT 5319, 'SEC', 2792, 40, 1, 2567, 4961, 4 FROM dual
UNION ALL SELECT 5320, 'SIC', 2937, 1490, 3, 2938, 1499, 4 FROM dual
UNION ALL SELECT 5321, 'SIC', 4308, 20, 1, 3430, 3534, 4 FROM dual
UNION ALL SELECT 5322, 'SIC', 3526, 3700, 2, 3537, 3730, 3 FROM dual
UNION ALL SELECT 5323, 'NAICS', 1942, 71, 1, 1989, 71329, 4 FROM dual
UNION ALL SELECT 5324, 'NAICS', 1725, 56, 1, 1775, 561710, 5 FROM dual
UNION ALL SELECT 5325, 'SIC', 2798, 100, 2, 2815, 172, 4 FROM dual
UNION ALL SELECT 5326, 'NAICS', 1904, 623, 2, 1906, 623110, 5 FROM dual
UNION ALL SELECT 5327, 'SEC', 2384, 3400, 2, 2397, 3451, 4 FROM dual
UNION ALL SELECT 5328, 'NAICS', 205, 23, 1, 233, 237990, 5 FROM dual
UNION ALL SELECT 5329, 'NAICS', 1402, 51, 1, 1412, 51114, 4 FROM dual
UNION ALL SELECT 5330, 'SEC', 2488, 3840, 3, 2493, 3845, 4 FROM dual
UNION ALL SELECT 5331, 'SIC', 2885, 1010, 3, 2886, 1011, 4 FROM dual
UNION ALL SELECT 5332, 'NAICS', 205, 23, 1, 253, 2382, 3 FROM dual
UNION ALL SELECT 5333, 'SIC', 3046, 2130, 3, 3047, 2131, 4 FROM dual
UNION ALL SELECT 5334, 'SIC', 4040, 7300, 2, 4057, 7349, 4 FROM dual
UNION ALL SELECT 5335, 'NAICS', 1917, 6239, 3, 1919, 62399, 4 FROM dual
UNION ALL SELECT 5336, 'SIC', 3788, 5160, 3, 3790, 5169, 4 FROM dual
UNION ALL SELECT 5337, 'SIC', 3605, 4100, 2, 3609, 4120, 3 FROM dual
UNION ALL SELECT 5338, 'SIC', 3715, 5000, 2, 3721, 5020, 3 FROM dual
UNION ALL SELECT 5339, 'NAICS', 1513, 523, 2, 1534, 523991, 5 FROM dual
UNION ALL SELECT 5340, 'SIC', 4312, 60, 1, 3964, 6330, 3 FROM dual
UNION ALL SELECT 5341, 'SIC', 2847, 700, 2, 2853, 723, 4 FROM dual
UNION ALL SELECT 5342, 'NAICS', 43, 1119, 3, 54, 111992, 5 FROM dual
UNION ALL SELECT 5343, 'NAICS', 213, 2362, 3, 214, 236210, 5 FROM dual
UNION ALL SELECT 5344, 'NAICS', 1624, 54, 1, 1707, 541910, 5 FROM dual
UNION ALL SELECT 5345, 'SIC', 3869, 5700, 2, 3881, 5736, 4 FROM dual
UNION ALL SELECT 5346, 'SIC', 4305, 1, 1, 2807, 132, 4 FROM dual
UNION ALL SELECT 5347, 'NAICS', 1624, 54, 1, 1630, 54112, 4 FROM dual
UNION ALL SELECT 5348, 'SIC', 3337, 3300, 2, 3365, 3365, 4 FROM dual
UNION ALL SELECT 5349, 'NAICS', 1812, 61, 1, 1825, 61141, 4 FROM dual
UNION ALL SELECT 5350, 'SEC', 2794, 52, 1, 2619, 5411, 4 FROM dual
UNION ALL SELECT 5351, 'SIC', 3337, 3300, 2, 3357, 3353, 4 FROM dual
UNION ALL SELECT 5352, 'NAICS', 28, 11133, 4, 34, 111336, 5 FROM dual
UNION ALL SELECT 5353, 'SIC', 4308, 20, 1, 3435, 3541, 4 FROM dual
UNION ALL SELECT 5354, 'SIC', 4311, 52, 1, 3878, 5731, 4 FROM dual
UNION ALL SELECT 5355, 'NAICS', 1920, 624, 2, 1937, 624310, 5 FROM dual
UNION ALL SELECT 5356, 'SEC', 2262, 2200, 2, 2268, 2253, 4 FROM dual
UNION ALL SELECT 5357, 'SIC', 2982, 2000, 2, 3016, 2064, 4 FROM dual
UNION ALL SELECT 5358, 'NAICS', 1, 11, 1, 112, 114112, 5 FROM dual
UNION ALL SELECT 5359, 'SIC', 3572, 3860, 3, 3573, 3861, 4 FROM dual
UNION ALL SELECT 5360, 'SIC', 3886, 5900, 2, 3889, 5920, 3 FROM dual
UNION ALL SELECT 5361, 'SEC', 2793, 50, 1, 2585, 5080, 3 FROM dual
UNION ALL SELECT 5362, 'SIC', 3401, 3470, 3, 3402, 3471, 4 FROM dual
UNION ALL SELECT 5363, 'SIC', 3355, 3350, 3, 3356, 3351, 4 FROM dual
UNION ALL SELECT 5364, 'NAICS', 931, 423, 2, 974, 42352, 4 FROM dual
UNION ALL SELECT 5365, 'SIC', 3715, 5000, 2, 3740, 5060, 3 FROM dual
UNION ALL SELECT 5366, 'NAICS', 1850, 62, 1, 1930, 62421, 4 FROM dual
UNION ALL SELECT 5367, 'SEC', 2292, 2600, 2, 2301, 2673, 4 FROM dual
UNION ALL SELECT 5368, 'SIC', 3925, 6030, 3, 3927, 6036, 4 FROM dual
UNION ALL SELECT 5369, 'NAICS', 23, 1113, 3, 32, 111334, 5 FROM dual
UNION ALL SELECT 5370, 'SIC', 4308, 20, 1, 3176, 2653, 4 FROM dual
UNION ALL SELECT 5371, 'NAICS', 1015, 424, 2, 1062, 424610, 5 FROM dual
UNION ALL SELECT 5372, 'SEC', 2701, 6550, 3, 2702, 6552, 4 FROM dual
UNION ALL SELECT 5373, 'SIC', 2914, 1380, 3, 2915, 1381, 4 FROM dual
UNION ALL SELECT 5374, 'SIC', 4308, 20, 1, 3499, 3644, 4 FROM dual
UNION ALL SELECT 5375, 'NAICS', 236, 2381, 3, 238, 23811, 4 FROM dual
UNION ALL SELECT 5376, 'SIC', 2901, 1200, 2, 2905, 1230, 3 FROM dual
UNION ALL SELECT 5377, 'NAICS', 2103, 813, 2, 2124, 81392, 4 FROM dual
UNION ALL SELECT 5378, 'SIC', 3526, 3700, 2, 3540, 3740, 3 FROM dual
UNION ALL SELECT 5379, 'SIC', 3131, 2430, 3, 3135, 2436, 4 FROM dual
UNION ALL SELECT 5380, 'NAICS', 1026, 4243, 3, 1030, 42432, 4 FROM dual
UNION ALL SELECT 5381, 'SIC', 4308, 20, 1, 3531, 3715, 4 FROM dual
UNION ALL SELECT 5382, 'NAICS', 2135, 92, 1, 2165, 92219, 4 FROM dual
UNION ALL SELECT 5383, 'SIC', 4310, 50, 1, 3768, 5122, 4 FROM dual
UNION ALL SELECT 5384, 'NAICS', 1644, 54132, 4, 1643, 541320, 5 FROM dual
UNION ALL SELECT 5385, 'SIC', 3749, 5080, 3, 3755, 5088, 4 FROM dual
UNION ALL SELECT 5386, 'NAICS', 2135, 92, 1, 2169, 92311, 4 FROM dual
UNION ALL SELECT 5387, 'SIC', 3654, 4500, 2, 3660, 4580, 3 FROM dual
UNION ALL SELECT 5388, 'NAICS', 253, 2382, 3, 254, 238210, 5 FROM dual
UNION ALL SELECT 5389, 'NAICS', 1004, 4239, 3, 1008, 42392, 4 FROM dual
UNION ALL SELECT 5390, 'SIC', 4308, 20, 1, 3409, 3490, 3 FROM dual
UNION ALL SELECT 5391, 'SIC', 3958, 6300, 2, 3967, 6351, 4 FROM dual
UNION ALL SELECT 5392, 'NAICS', 97, 11299, 4, 96, 112990, 5 FROM dual
UNION ALL SELECT 5393, 'NAICS', 23, 1113, 3, 24, 111310, 5 FROM dual
UNION ALL SELECT 5394, 'SEC', 2405, 3520, 3, 2406, 3523, 4 FROM dual
UNION ALL SELECT 5395, 'NAICS', 1612, 5324, 3, 1614, 532411, 5 FROM dual
UNION ALL SELECT 5396, 'SIC', 4308, 20, 1, 3133, 2434, 4 FROM dual
UNION ALL SELECT 5397, 'NAICS', 1, 11, 1, 102, 1132, 3 FROM dual
UNION ALL SELECT 5398, 'NAICS', 1850, 62, 1, 1941, 62441, 4 FROM dual
UNION ALL SELECT 5399, 'SIC', 4308, 20, 1, 3566, 3842, 4 FROM dual
UNION ALL SELECT 5400, 'SIC', 4305, 1, 1, 2812, 161, 4 FROM dual
UNION ALL SELECT 5401, 'NAICS', 930, 42, 1, 1033, 424340, 5 FROM dual
UNION ALL SELECT 5402, 'NAICS', 1035, 4244, 3, 1047, 42446, 4 FROM dual
UNION ALL SELECT 5403, 'SEC', 2337, 2900, 2, 2339, 2911, 4 FROM dual
UNION ALL SELECT 5404, 'SIC', 3214, 2800, 2, 3238, 2861, 4 FROM dual
UNION ALL SELECT 5405, 'SIC', 3917, 6000, 2, 3932, 6081, 4 FROM dual
UNION ALL SELECT 5406, 'NAICS', 931, 423, 2, 937, 423130, 5 FROM dual
UNION ALL SELECT 5407, 'NAICS', 931, 423, 2, 935, 423120, 5 FROM dual
UNION ALL SELECT 5408, 'SIC', 4208, 8600, 2, 4222, 8699, 4 FROM dual
UNION ALL SELECT 5409, 'NAICS', 1480, 52, 1, 1552, 524292, 5 FROM dual
UNION ALL SELECT 5410, 'NAICS', 930, 42, 1, 1019, 424120, 5 FROM dual
UNION ALL SELECT 5411, 'SEC', 2476, 3800, 2, 2497, 3861, 4 FROM dual
UNION ALL SELECT 5412, 'NAICS', 1624, 54, 1, 1699, 54185, 4 FROM dual
UNION ALL SELECT 5413, 'NAICS', 43, 1119, 3, 53, 111991, 5 FROM dual
UNION ALL SELECT 5414, 'SIC', 3316, 3260, 3, 3319, 3263, 4 FROM dual
UNION ALL SELECT 5415, 'NAICS', 219, 2371, 3, 223, 23712, 4 FROM dual
UNION ALL SELECT 5416, 'SIC', 3124, 2400, 2, 3125, 2410, 3 FROM dual
UNION ALL SELECT 5417, 'SIC', 4125, 7900, 2, 4131, 7930, 3 FROM dual
UNION ALL SELECT 5418, 'SIC', 4313, 70, 1, 4081, 7500, 2 FROM dual
UNION ALL SELECT 5419, 'SEC', 2791, 20, 1, 2461, 3713, 4 FROM dual
UNION ALL SELECT 5420, 'NAICS', 181, 221, 2, 183, 22111, 4 FROM dual
UNION ALL SELECT 5421, 'NAICS', 1625, 541, 2, 1631, 54119, 4 FROM dual
UNION ALL SELECT 5422, 'SIC', 4308, 20, 1, 3258, 2990, 3 FROM dual
UNION ALL SELECT 5423, 'NAICS', 2031, 7225, 3, 2036, 722515, 5 FROM dual
UNION ALL SELECT 5424, 'SIC', 3409, 3490, 3, 3413, 3494, 4 FROM dual
UNION ALL SELECT 5425, 'SEC', 2319, 2800, 2, 2331, 2850, 3 FROM dual
UNION ALL SELECT 5426, 'NAICS', 1569, 53, 1, 1588, 53132, 4 FROM dual
UNION ALL SELECT 5427, 'NAICS', 2103, 813, 2, 2116, 813319, 5 FROM dual
UNION ALL SELECT 5428, 'SIC', 4308, 20, 1, 3130, 2429, 4 FROM dual
UNION ALL SELECT 5429, 'NAICS', 1979, 713, 2, 1983, 713120, 5 FROM dual
UNION ALL SELECT 5430, 'SIC', 3419, 3500, 2, 3427, 3531, 4 FROM dual
UNION ALL SELECT 5431, 'SIC', 4311, 52, 1, 3811, 5260, 3 FROM dual
UNION ALL SELECT 5432, 'SIC', 3419, 3500, 2, 3443, 3549, 4 FROM dual
UNION ALL SELECT 5433, 'NAICS', 1726, 561, 2, 1774, 5617, 3 FROM dual
UNION ALL SELECT 5434, 'NAICS', 999, 42384, 4, 998, 423840, 5 FROM dual
UNION ALL SELECT 5435, 'NAICS', 1850, 62, 1, 1865, 62133, 4 FROM dual
UNION ALL SELECT 5436, 'NAICS', 1851, 621, 2, 1879, 621493, 5 FROM dual
UNION ALL SELECT 5437, 'SIC', 4308, 20, 1, 3234, 2844, 4 FROM dual
UNION ALL SELECT 5438, 'NAICS', 930, 42, 1, 970, 4235, 3 FROM dual
UNION ALL SELECT 5439, 'NAICS', 1037, 42441, 4, 1036, 424410, 5 FROM dual
UNION ALL SELECT 5440, 'NAICS', 1598, 5322, 3, 1603, 532230, 5 FROM dual
UNION ALL SELECT 5441, 'SIC', 4308, 20, 1, 3235, 2850, 3 FROM dual
UNION ALL SELECT 5442, 'NAICS', 133, 211, 2, 137, 211112, 5 FROM dual
UNION ALL SELECT 5443, 'SEC', 2403, 3500, 2, 2410, 3532, 4 FROM dual
UNION ALL SELECT 5444, 'NAICS', 930, 42, 1, 1056, 42451, 4 FROM dual
UNION ALL SELECT 5445, 'NAICS', 57, 1121, 3, 62, 11212, 4 FROM dual
UNION ALL SELECT 5446, 'SIC', 3444, 3550, 3, 3445, 3552, 4 FROM dual
UNION ALL SELECT 5447, 'NAICS', 1015, 424, 2, 1080, 42492, 4 FROM dual
UNION ALL SELECT 5448, 'NAICS', 1419, 512, 2, 1432, 512210, 5 FROM dual
UNION ALL SELECT 5449, 'SIC', 3715, 5000, 2, 3750, 5082, 4 FROM dual
UNION ALL SELECT 5450, 'NAICS', 183, 22111, 4, 188, 221115, 5 FROM dual
UNION ALL SELECT 5451, 'SIC', 4308, 20, 1, 3441, 3547, 4 FROM dual
UNION ALL SELECT 5452, 'NAICS', 56, 112, 2, 74, 11233, 4 FROM dual
UNION ALL SELECT 5453, 'SIC', 4097, 7540, 3, 4098, 7542, 4 FROM dual
UNION ALL SELECT 5454, 'NAICS', 2135, 92, 1, 2182, 925, 2 FROM dual
UNION ALL SELECT 5455, 'SEC', 2694, 6500, 2, 2695, 6510, 3 FROM dual
UNION ALL SELECT 5456, 'NAICS', 1571, 5311, 3, 1573, 53111, 4 FROM dual
UNION ALL SELECT 5457, 'NAICS', 1583, 5313, 3, 1587, 531320, 5 FROM dual
UNION ALL SELECT 5458, 'SEC', 2753, 8000, 2, 2756, 8050, 3 FROM dual
UNION ALL SELECT 5459, 'SIC', 4308, 20, 1, 3227, 2834, 4 FROM dual
UNION ALL SELECT 5460, 'NAICS', 117, 115, 2, 122, 115113, 5 FROM dual
UNION ALL SELECT 5461, 'SEC', 2389, 3430, 3, 2390, 3433, 4 FROM dual
UNION ALL SELECT 5462, 'SEC', 2795, 60, 1, 2653, 6029, 4 FROM dual
UNION ALL SELECT 5463, 'SIC', 4311, 52, 1, 3896, 5943, 4 FROM dual
UNION ALL SELECT 5464, 'SIC', 4312, 60, 1, 3992, 6700, 2 FROM dual
UNION ALL SELECT 5465, 'NAICS', 1990, 7139, 3, 1994, 71392, 4 FROM dual
UNION ALL SELECT 5466, 'NAICS', 1015, 424, 2, 1064, 424690, 5 FROM dual
UNION ALL SELECT 5467, 'NAICS', 1569, 53, 1, 1570, 531, 2 FROM dual
UNION ALL SELECT 5468, 'NAICS', 1894, 622, 2, 1903, 62231, 4 FROM dual
UNION ALL SELECT 5469, 'SEC', 2259, 2100, 2, 2260, 2110, 3 FROM dual
UNION ALL SELECT 5470, 'SIC', 2825, 200, 2, 2838, 254, 4 FROM dual
UNION ALL SELECT 5471, 'SEC', 2796, 70, 1, 2717, 7330, 3 FROM dual
UNION ALL SELECT 5472, 'SIC', 3605, 4100, 2, 3618, 4170, 3 FROM dual
UNION ALL SELECT 5473, 'NAICS', 1942, 71, 1, 1997, 713940, 5 FROM dual
UNION ALL SELECT 5474, 'SIC', 4308, 20, 1, 3297, 3171, 4 FROM dual
UNION ALL SELECT 5475, 'SIC', 3480, 3600, 2, 3492, 3633, 4 FROM dual
UNION ALL SELECT 5476, 'SIC', 3977, 6500, 2, 3989, 6550, 3 FROM dual
UNION ALL SELECT 5477, 'SIC', 3784, 5150, 3, 3785, 5153, 4 FROM dual
UNION ALL SELECT 5478, 'NAICS', 930, 42, 1, 996, 423830, 5 FROM dual
UNION ALL SELECT 5479, 'NAICS', 2053, 81121, 4, 2054, 811211, 5 FROM dual
UNION ALL SELECT 5480, 'NAICS', 1562, 5259, 3, 1568, 52599, 4 FROM dual
UNION ALL SELECT 5481, 'NAICS', 2177, 9241, 3, 2181, 92412, 4 FROM dual
UNION ALL SELECT 5482, 'NAICS', 2137, 9211, 3, 2143, 92113, 4 FROM dual
UNION ALL SELECT 5483, 'SEC', 2791, 20, 1, 2501, 3910, 3 FROM dual
UNION ALL SELECT 5484, 'SIC', 4113, 7800, 2, 4120, 7830, 3 FROM dual
UNION ALL SELECT 5485, 'NAICS', 930, 42, 1, 946, 4233, 3 FROM dual
UNION ALL SELECT 5486, 'SIC', 4313, 70, 1, 4174, 8099, 4 FROM dual
UNION ALL SELECT 5487, 'SIC', 4018, 7210, 3, 4023, 7216, 4 FROM dual
UNION ALL SELECT 5488, 'NAICS', 1836, 6116, 3, 1838, 61161, 4 FROM dual
UNION ALL SELECT 5489, 'SIC', 4308, 20, 1, 3289, 3143, 4 FROM dual
UNION ALL SELECT 5490, 'SIC', 3190, 2700, 2, 3202, 2754, 4 FROM dual
UNION ALL SELECT 5491, 'NAICS', 108, 114, 2, 109, 1141, 3 FROM dual
UNION ALL SELECT 5492, 'NAICS', 1473, 51911, 4, 1472, 519110, 5 FROM dual
UNION ALL SELECT 5493, 'SIC', 3555, 3820, 3, 3560, 3825, 4 FROM dual
UNION ALL SELECT 5494, 'NAICS', 1591, 532, 2, 1616, 532420, 5 FROM dual
UNION ALL SELECT 5495, 'NAICS', 2038, 811, 2, 2066, 81142, 4 FROM dual
UNION ALL SELECT 5496, 'NAICS', 1726, 561, 2, 1770, 561613, 5 FROM dual
UNION ALL SELECT 5497, 'SIC', 4040, 7300, 2, 4048, 7323, 4 FROM dual
UNION ALL SELECT 5498, 'SIC', 4314, 90, 1, 4282, 9531, 4 FROM dual
UNION ALL SELECT 5499, 'NAICS', 930, 42, 1, 997, 42383, 4 FROM dual
UNION ALL SELECT 5500, 'SIC', 4312, 60, 1, 4002, 6790, 3 FROM dual
UNION ALL SELECT 5501, 'NAICS', 1943, 711, 2, 1947, 711120, 5 FROM dual
UNION ALL SELECT 5502, 'NAICS', 2135, 92, 1, 2154, 922120, 5 FROM dual
UNION ALL SELECT 5503, 'SEC', 2791, 20, 1, 2272, 2320, 3 FROM dual
UNION ALL SELECT 5504, 'SIC', 4268, 9400, 2, 4270, 9411, 4 FROM dual
UNION ALL SELECT 5505, 'SIC', 4208, 8600, 2, 4210, 8611, 4 FROM dual
UNION ALL SELECT 5506, 'SIC', 4308, 20, 1, 3056, 2231, 4 FROM dual
UNION ALL SELECT 5507, 'SIC', 3311, 3250, 3, 3312, 3251, 4 FROM dual
UNION ALL SELECT 5508, 'SEC', 2792, 40, 1, 2551, 4899, 4 FROM dual
UNION ALL SELECT 5509, 'SEC', 2640, 5900, 2, 2648, 5990, 3 FROM dual
UNION ALL SELECT 5510, 'SIC', 3649, 4490, 3, 3650, 4491, 4 FROM dual
UNION ALL SELECT 5511, 'SIC', 4311, 52, 1, 3815, 5300, 2 FROM dual
UNION ALL SELECT 5512, 'NAICS', 2137, 9211, 3, 2141, 92112, 4 FROM dual
UNION ALL SELECT 5513, 'SEC', 2791, 20, 1, 2307, 2730, 3 FROM dual
UNION ALL SELECT 5514, 'NAICS', 1850, 62, 1, 1917, 6239, 3 FROM dual
UNION ALL SELECT 5515, 'NAICS', 1015, 424, 2, 1071, 4248, 3 FROM dual
UNION ALL SELECT 5516, 'SIC', 4308, 20, 1, 3087, 2321, 4 FROM dual
UNION ALL SELECT 5517, 'SIC', 4308, 20, 1, 3544, 3760, 3 FROM dual
UNION ALL SELECT 5518, 'NAICS', 1996, 71393, 4, 1995, 713930, 5 FROM dual
UNION ALL SELECT 5519, 'SEC', 2791, 20, 1, 2332, 2851, 4 FROM dual
UNION ALL SELECT 5520, 'SIC', 3958, 6300, 2, 3961, 6320, 3 FROM dual
UNION ALL SELECT 5521, 'SEC', 2459, 3710, 3, 2463, 3715, 4 FROM dual
UNION ALL SELECT 5522, 'NAICS', 1073, 42481, 4, 1072, 424810, 5 FROM dual
UNION ALL SELECT 5523, 'NAICS', 955, 4234, 3, 961, 42343, 4 FROM dual
UNION ALL SELECT 5524, 'NAICS', 1452, 517, 2, 1458, 51721, 4 FROM dual
UNION ALL SELECT 5525, 'NAICS', 1851, 621, 2, 1864, 621330, 5 FROM dual
UNION ALL SELECT 5526, 'SIC', 3715, 5000, 2, 3719, 5014, 4 FROM dual
UNION ALL SELECT 5527, 'SIC', 3480, 3600, 2, 3491, 3632, 4 FROM dual
UNION ALL SELECT 5528, 'NAICS', 1624, 54, 1, 1701, 54186, 4 FROM dual
UNION ALL SELECT 5529, 'SIC', 4310, 50, 1, 3792, 5171, 4 FROM dual
UNION ALL SELECT 5530, 'NAICS', 56, 112, 2, 88, 112519, 5 FROM dual
UNION ALL SELECT 5531, 'SIC', 4309, 40, 1, 3623, 4213, 4 FROM dual
UNION ALL SELECT 5532, 'NAICS', 1851, 621, 2, 1863, 62132, 4 FROM dual
UNION ALL SELECT 5533, 'NAICS', 180, 22, 1, 191, 221118, 5 FROM dual
UNION ALL SELECT 5534, 'NAICS', 1990, 7139, 3, 1995, 713930, 5 FROM dual
UNION ALL SELECT 5535, 'NAICS', 1667, 54151, 4, 1668, 541511, 5 FROM dual
UNION ALL SELECT 5536, 'NAICS', 2053, 81121, 4, 2056, 811213, 5 FROM dual
UNION ALL SELECT 5537, 'SIC', 4308, 20, 1, 3107, 2371, 4 FROM dual
UNION ALL SELECT 5538, 'SEC', 2297, 2630, 3, 2298, 2631, 4 FROM dual
UNION ALL SELECT 5539, 'NAICS', 1625, 541, 2, 1697, 54184, 4 FROM dual
UNION ALL SELECT 5540, 'NAICS', 2195, 92613, 4, 2194, 926130, 5 FROM dual
UNION ALL SELECT 5541, 'SEC', 2792, 40, 1, 2543, 4820, 3 FROM dual
UNION ALL SELECT 5542, 'NAICS', 132, 21, 1, 179, 213115, 5 FROM dual
UNION ALL SELECT 5543, 'NAICS', 138, 212, 2, 148, 212221, 5 FROM dual
UNION ALL SELECT 5544, 'NAICS', 1725, 56, 1, 1801, 562212, 5 FROM dual
UNION ALL SELECT 5545, 'NAICS', 205, 23, 1, 258, 238290, 5 FROM dual
UNION ALL SELECT 5546, 'NAICS', 1480, 52, 1, 1524, 523210, 5 FROM dual
UNION ALL SELECT 5547, 'SIC', 4308, 20, 1, 3202, 2754, 4 FROM dual
UNION ALL SELECT 5548, 'SIC', 3083, 2300, 2, 3086, 2320, 3 FROM dual
UNION ALL SELECT 5549, 'NAICS', 2135, 92, 1, 2191, 92611, 4 FROM dual
UNION ALL SELECT 5550, 'NAICS', 2173, 92313, 4, 2172, 923130, 5 FROM dual
UNION ALL SELECT 5551, 'SIC', 4305, 1, 1, 2854, 724, 4 FROM dual
UNION ALL SELECT 5552, 'SIC', 4308, 20, 1, 3096, 2337, 4 FROM dual
UNION ALL SELECT 5553, 'SIC', 3886, 5900, 2, 3898, 5945, 4 FROM dual
UNION ALL SELECT 5554, 'SIC', 4310, 50, 1, 3749, 5080, 3 FROM dual
UNION ALL SELECT 5555, 'SIC', 4223, 8700, 2, 4235, 8740, 3 FROM dual
UNION ALL SELECT 5556, 'SIC', 2901, 1200, 2, 2906, 1231, 4 FROM dual
UNION ALL SELECT 5557, 'SIC', 4306, 10, 1, 2912, 1320, 3 FROM dual
UNION ALL SELECT 5558, 'SIC', 4308, 20, 1, 3416, 3497, 4 FROM dual
UNION ALL SELECT 5559, 'SIC', 3225, 2830, 3, 3229, 2836, 4 FROM dual
UNION ALL SELECT 5560, 'NAICS', 2003, 72, 1, 2006, 721110, 5 FROM dual
UNION ALL SELECT 5561, 'NAICS', 1482, 5211, 3, 1483, 521110, 5 FROM dual
UNION ALL SELECT 5562, 'SEC', 2738, 7800, 2, 2744, 7829, 4 FROM dual
UNION ALL SELECT 5563, 'SIC', 4203, 8400, 2, 4207, 8422, 4 FROM dual
UNION ALL SELECT 5564, 'SEC', 2704, 6790, 3, 2708, 6799, 4 FROM dual
UNION ALL SELECT 5565, 'SIC', 2993, 2030, 3, 2998, 2037, 4 FROM dual
UNION ALL SELECT 5566, 'NAICS', 1452, 517, 2, 1453, 5171, 3 FROM dual
UNION ALL SELECT 5567, 'SIC', 3261, 3000, 2, 3273, 3081, 4 FROM dual
UNION ALL SELECT 5568, 'NAICS', 1419, 512, 2, 1426, 512131, 5 FROM dual
UNION ALL SELECT 5569, 'NAICS', 1928, 6242, 3, 1932, 624221, 5 FROM dual
UNION ALL SELECT 5570, 'SEC', 2713, 7300, 2, 4329, 7385, 4 FROM dual
UNION ALL SELECT 5571, 'SIC', 4007, 7000, 2, 4011, 7021, 4 FROM dual
UNION ALL SELECT 5572, 'NAICS', 2103, 813, 2, 2112, 8133, 3 FROM dual
UNION ALL SELECT 5573, 'SEC', 2479, 3820, 3, 2485, 3826, 4 FROM dual
UNION ALL SELECT 5574, 'NAICS', 182, 2211, 3, 189, 221116, 5 FROM dual
UNION ALL SELECT 5575, 'NAICS', 1850, 62, 1, 1929, 624210, 5 FROM dual
UNION ALL SELECT 5576, 'NAICS', 2045, 81112, 4, 2047, 811122, 5 FROM dual
UNION ALL SELECT 5577, 'NAICS', 1455, 51711, 4, 1454, 517110, 5 FROM dual
UNION ALL SELECT 5578, 'SIC', 3282, 3100, 2, 3283, 3110, 3 FROM dual
UNION ALL SELECT 5579, 'SIC', 3230, 2840, 3, 3232, 2842, 4 FROM dual
UNION ALL SELECT 5580, 'SIC', 4307, 15, 1, 2960, 1730, 3 FROM dual
UNION ALL SELECT 5581, 'SIC', 3419, 3500, 2, 3432, 3536, 4 FROM dual
UNION ALL SELECT 5582, 'SEC', 2237, 1700, 2, 2239, 1731, 4 FROM dual
UNION ALL SELECT 5583, 'SIC', 3395, 3460, 3, 3397, 3463, 4 FROM dual
UNION ALL SELECT 5584, 'SIC', 4314, 90, 1, 4294, 9651, 4 FROM dual
UNION ALL SELECT 5585, 'NAICS', 2, 111, 2, 37, 11141, 4 FROM dual
UNION ALL SELECT 5586, 'SIC', 4284, 9600, 2, 4292, 9641, 4 FROM dual
UNION ALL SELECT 5587, 'NAICS', 1, 11, 1, 123, 115114, 5 FROM dual
UNION ALL SELECT 5588, 'SEC', 2796, 70, 1, 2783, 8744, 4 FROM dual
UNION ALL SELECT 5589, 'SIC', 4181, 8220, 3, 4182, 8221, 4 FROM dual
UNION ALL SELECT 5590, 'NAICS', 1015, 424, 2, 1043, 42444, 4 FROM dual
UNION ALL SELECT 5591, 'NAICS', 2103, 813, 2, 2121, 813910, 5 FROM dual
UNION ALL SELECT 5592, 'SIC', 4308, 20, 1, 3529, 3713, 4 FROM dual
UNION ALL SELECT 5593, 'NAICS', 1016, 4241, 3, 1020, 42412, 4 FROM dual
UNION ALL SELECT 5594, 'NAICS', 235, 238, 2, 244, 23814, 4 FROM dual
UNION ALL SELECT 5595, 'NAICS', 930, 42, 1, 1042, 424440, 5 FROM dual
UNION ALL SELECT 5596, 'SIC', 4223, 8700, 2, 4232, 8732, 4 FROM dual
UNION ALL SELECT 5597, 'SIC', 4308, 20, 1, 3060, 2251, 4 FROM dual
UNION ALL SELECT 5598, 'SEC', 2791, 20, 1, 2273, 2330, 3 FROM dual
UNION ALL SELECT 5599, 'SIC', 4306, 10, 1, 2913, 1321, 4 FROM dual
UNION ALL SELECT 5600, 'SIC', 3762, 5100, 2, 3775, 5141, 4 FROM dual
UNION ALL SELECT 5601, 'NAICS', 2004, 721, 2, 2013, 7212, 3 FROM dual
UNION ALL SELECT 5602, 'SIC', 3992, 6700, 2, 3997, 6722, 4 FROM dual
UNION ALL SELECT 5603, 'NAICS', 1625, 541, 2, 1668, 541511, 5 FROM dual
UNION ALL SELECT 5604, 'SIC', 3115, 2390, 3, 3119, 2394, 4 FROM dual
UNION ALL SELECT 5605, 'SIC', 3423, 3520, 3, 3424, 3523, 4 FROM dual
UNION ALL SELECT 5606, 'SIC', 4305, 1, 1, 2850, 720, 3 FROM dual
UNION ALL SELECT 5607, 'SEC', 2795, 60, 1, 2670, 6200, 2 FROM dual
UNION ALL SELECT 5608, 'NAICS', 1443, 5151, 3, 1447, 515120, 5 FROM dual
UNION ALL SELECT 5609, 'NAICS', 1533, 52399, 4, 1534, 523991, 5 FROM dual
UNION ALL SELECT 5610, 'SEC', 2791, 20, 1, 2410, 3532, 4 FROM dual
UNION ALL SELECT 5611, 'SIC', 4309, 40, 1, 3699, 4924, 4 FROM dual
UNION ALL SELECT 5612, 'NAICS', 1485, 522, 2, 1507, 522310, 5 FROM dual
UNION ALL SELECT 5613, 'SEC', 2677, 6300, 2, 2687, 6360, 3 FROM dual
UNION ALL SELECT 5614, 'SIC', 3893, 5940, 3, 3900, 5947, 4 FROM dual
UNION ALL SELECT 5615, 'NAICS', 1449, 5152, 3, 1451, 51521, 4 FROM dual
UNION ALL SELECT 5616, 'SIC', 4017, 7200, 2, 4025, 7218, 4 FROM dual
UNION ALL SELECT 5617, 'NAICS', 236, 2381, 3, 250, 23817, 4 FROM dual
UNION ALL SELECT 5618, 'SIC', 4309, 40, 1, 3615, 4142, 4 FROM dual
UNION ALL SELECT 5619, 'NAICS', 56, 112, 2, 61, 112120, 5 FROM dual
UNION ALL SELECT 5620, 'SEC', 2791, 20, 1, 2370, 3290, 3 FROM dual
UNION ALL SELECT 5621, 'SIC', 2910, 1310, 3, 2911, 1311, 4 FROM dual
UNION ALL SELECT 5622, 'SIC', 4308, 20, 1, 3481, 3610, 3 FROM dual
UNION ALL SELECT 5623, 'NAICS', 99, 1131, 3, 101, 11311, 4 FROM dual
UNION ALL SELECT 5624, 'SIC', 3715, 5000, 2, 3728, 5039, 4 FROM dual
UNION ALL SELECT 5625, 'SIC', 4305, 1, 1, 2803, 116, 4 FROM dual
UNION ALL SELECT 5626, 'NAICS', 43, 1119, 3, 51, 11194, 4 FROM dual
UNION ALL SELECT 5627, 'NAICS', 235, 238, 2, 243, 238140, 5 FROM dual
UNION ALL SELECT 5628, 'SIC', 4305, 1, 1, 2833, 241, 4 FROM dual
UNION ALL SELECT 5629, 'NAICS', 1591, 532, 2, 1604, 53223, 4 FROM dual
UNION ALL SELECT 5630, 'NAICS', 1480, 52, 1, 1531, 523930, 5 FROM dual
UNION ALL SELECT 5631, 'SEC', 2476, 3800, 2, 2480, 3821, 4 FROM dual
UNION ALL SELECT 5632, 'SEC', 2791, 20, 1, 2321, 2820, 3 FROM dual
UNION ALL SELECT 5633, 'NAICS', 1624, 54, 1, 1706, 5419, 3 FROM dual
UNION ALL SELECT 5634, 'SEC', 2796, 70, 1, 2729, 7374, 4 FROM dual
UNION ALL SELECT 5635, 'SIC', 3131, 2430, 3, 3136, 2439, 4 FROM dual
UNION ALL SELECT 5636, 'SIC', 4308, 20, 1, 3593, 3961, 4 FROM dual
UNION ALL SELECT 5637, 'SIC', 3419, 3500, 2, 3479, 3599, 4 FROM dual
UNION ALL SELECT 5638, 'NAICS', 144, 2122, 3, 155, 212299, 5 FROM dual
UNION ALL SELECT 5639, 'SEC', 2794, 52, 1, 2628, 5660, 3 FROM dual
UNION ALL SELECT 5640, 'SIC', 4308, 20, 1, 3123, 2399, 4 FROM dual
UNION ALL SELECT 5641, 'SEC', 2790, 15, 1, 2235, 1620, 3 FROM dual
UNION ALL SELECT 5642, 'SIC', 4311, 52, 1, 3831, 5450, 3 FROM dual
UNION ALL SELECT 5643, 'SIC', 4040, 7300, 2, 4055, 7340, 3 FROM dual
UNION ALL SELECT 5644, 'NAICS', 2107, 8132, 3, 2109, 813211, 5 FROM dual
UNION ALL SELECT 5645, 'NAICS', 56, 112, 2, 84, 1125, 3 FROM dual
UNION ALL SELECT 5646, 'NAICS', 229, 2373, 3, 230, 237310, 5 FROM dual
UNION ALL SELECT 5647, 'SIC', 3419, 3500, 2, 3458, 3567, 4 FROM dual
UNION ALL SELECT 5648, 'NAICS', 1025, 42421, 4, 1024, 424210, 5 FROM dual
UNION ALL SELECT 5649, 'NAICS', 1625, 541, 2, 1679, 541620, 5 FROM dual
UNION ALL SELECT 5650, 'SIC', 4309, 40, 1, 3627, 4221, 4 FROM dual
UNION ALL SELECT 5651, 'NAICS', 1513, 523, 2, 1521, 523140, 5 FROM dual
UNION ALL SELECT 5652, 'SIC', 3680, 4800, 2, 3687, 4832, 4 FROM dual
UNION ALL SELECT 5653, 'SIC', 3822, 5400, 2, 3835, 5490, 3 FROM dual
UNION ALL SELECT 5654, 'SIC', 4310, 50, 1, 3779, 5145, 4 FROM dual
UNION ALL SELECT 5655, 'SIC', 3978, 6510, 3, 3983, 6517, 4 FROM dual
UNION ALL SELECT 5656, 'SIC', 2955, 1700, 2, 2959, 1721, 4 FROM dual
UNION ALL SELECT 5657, 'SIC', 4307, 15, 1, 2970, 1761, 4 FROM dual
UNION ALL SELECT 5658, 'NAICS', 1689, 5418, 3, 1697, 54184, 4 FROM dual
UNION ALL SELECT 5659, 'SIC', 4308, 20, 1, 3394, 3452, 4 FROM dual
UNION ALL SELECT 5660, 'NAICS', 1850, 62, 1, 1903, 62231, 4 FROM dual
UNION ALL SELECT 5661, 'SIC', 4308, 20, 1, 3236, 2851, 4 FROM dual
UNION ALL SELECT 5662, 'NAICS', 118, 1151, 3, 122, 115113, 5 FROM dual
UNION ALL SELECT 5663, 'NAICS', 260, 2383, 3, 271, 238390, 5 FROM dual
UNION ALL SELECT 5664, 'NAICS', 1591, 532, 2, 1619, 53249, 4 FROM dual
UNION ALL SELECT 5665, 'NAICS', 1792, 562, 2, 1796, 562112, 5 FROM dual
UNION ALL SELECT 5666, 'SIC', 3605, 4100, 2, 3613, 4140, 3 FROM dual
UNION ALL SELECT 5667, 'NAICS', 1885, 6216, 3, 1886, 621610, 5 FROM dual
UNION ALL SELECT 5668, 'SIC', 3174, 2650, 3, 3178, 2656, 4 FROM dual
UNION ALL SELECT 5669, 'NAICS', 1652, 54136, 4, 1651, 541360, 5 FROM dual
UNION ALL SELECT 5670, 'SIC', 4312, 60, 1, 3996, 6720, 3 FROM dual
UNION ALL SELECT 5671, 'NAICS', 1847, 6117, 3, 1849, 61171, 4 FROM dual
UNION ALL SELECT 5672, 'NAICS', 2135, 92, 1, 2153, 92211, 4 FROM dual
UNION ALL SELECT 5673, 'SIC', 2955, 1700, 2, 2966, 1750, 3 FROM dual
UNION ALL SELECT 5674, 'NAICS', 932, 4231, 3, 939, 423140, 5 FROM dual
UNION ALL SELECT 5675, 'NAICS', 1402, 51, 1, 1463, 51791, 4 FROM dual
UNION ALL SELECT 5676, 'NAICS', 219, 2371, 3, 224, 237130, 5 FROM dual
UNION ALL SELECT 5677, 'SIC', 3371, 3400, 2, 3399, 3466, 4 FROM dual
UNION ALL SELECT 5678, 'SIC', 4313, 70, 1, 4155, 8042, 4 FROM dual
UNION ALL SELECT 5679, 'SIC', 2918, 1400, 2, 2926, 1442, 4 FROM dual
UNION ALL SELECT 5680, 'SIC', 3041, 2100, 2, 3046, 2130, 3 FROM dual
UNION ALL SELECT 5681, 'SEC', 2649, 6000, 2, 2657, 6090, 3 FROM dual
UNION ALL SELECT 5682, 'SIC', 3083, 2300, 2, 3115, 2390, 3 FROM dual
UNION ALL SELECT 5683, 'SEC', 2384, 3400, 2, 2390, 3433, 4 FROM dual
UNION ALL SELECT 5684, 'SIC', 4308, 20, 1, 3338, 3310, 3 FROM dual
UNION ALL SELECT 5685, 'SIC', 3693, 4900, 2, 3705, 4940, 3 FROM dual
UNION ALL SELECT 5686, 'SEC', 2545, 4830, 3, 2547, 4833, 4 FROM dual
UNION ALL SELECT 5687, 'SIC', 4308, 20, 1, 3269, 3060, 3 FROM dual
UNION ALL SELECT 5688, 'SIC', 3756, 5090, 3, 3759, 5093, 4 FROM dual
UNION ALL SELECT 5689, 'SIC', 4308, 20, 1, 3005, 2046, 4 FROM dual
UNION ALL SELECT 5690, 'NAICS', 1625, 541, 2, 1636, 541211, 5 FROM dual
UNION ALL SELECT 5691, 'SEC', 2791, 20, 1, 2393, 3443, 4 FROM dual
UNION ALL SELECT 5692, 'SIC', 4307, 15, 1, 2965, 1743, 4 FROM dual
UNION ALL SELECT 5693, 'SIC', 2948, 1600, 2, 2953, 1623, 4 FROM dual
UNION ALL SELECT 5694, 'SIC', 3886, 5900, 2, 3916, 5999, 4 FROM dual
UNION ALL SELECT 5695, 'SEC', 2713, 7300, 2, 2726, 7371, 4 FROM dual
UNION ALL SELECT 5696, 'NAICS', 2, 111, 2, 17, 111191, 5 FROM dual
UNION ALL SELECT 5697, 'SEC', 2677, 6300, 2, 2686, 6351, 4 FROM dual
UNION ALL SELECT 5698, 'SEC', 2795, 60, 1, 2681, 6321, 4 FROM dual
UNION ALL SELECT 5699, 'SIC', 4305, 1, 1, 2825, 200, 2 FROM dual
UNION ALL SELECT 5700, 'NAICS', 2120, 8139, 3, 2125, 813930, 5 FROM dual
UNION ALL SELECT 5701, 'NAICS', 1754, 56149, 4, 1757, 561499, 5 FROM dual
UNION ALL SELECT 5702, 'NAICS', 2004, 721, 2, 2010, 72119, 4 FROM dual
UNION ALL SELECT 5703, 'SIC', 4308, 20, 1, 3158, 2522, 4 FROM dual
UNION ALL SELECT 5704, 'SIC', 4313, 70, 1, 4020, 7212, 4 FROM dual
UNION ALL SELECT 5705, 'NAICS', 1480, 52, 1, 1532, 52393, 4 FROM dual
UNION ALL SELECT 5706, 'SEC', 2292, 2600, 2, 2296, 2621, 4 FROM dual
UNION ALL SELECT 5707, 'NAICS', 277, 23899, 4, 276, 238990, 5 FROM dual
UNION ALL SELECT 5708, 'SIC', 3148, 2500, 2, 3157, 2521, 4 FROM dual
UNION ALL SELECT 5709, 'SIC', 4310, 50, 1, 3730, 5043, 4 FROM dual
UNION ALL SELECT 5710, 'NAICS', 1804, 5629, 3, 1809, 56299, 4 FROM dual
UNION ALL SELECT 5711, 'SIC', 4314, 90, 1, 4261, 9222, 4 FROM dual
UNION ALL SELECT 5712, 'SIC', 3261, 3000, 2, 3277, 3085, 4 FROM dual
UNION ALL SELECT 5713, 'NAICS', 1591, 532, 2, 1608, 532299, 5 FROM dual
UNION ALL SELECT 5714, 'SIC', 4308, 20, 1, 3220, 2820, 3 FROM dual
UNION ALL SELECT 5715, 'NAICS', 2183, 9251, 3, 2187, 92512, 4 FROM dual
UNION ALL SELECT 5716, 'NAICS', 1419, 512, 2, 1438, 512240, 5 FROM dual
UNION ALL SELECT 5717, 'NAICS', 57, 1121, 3, 60, 112112, 5 FROM dual
UNION ALL SELECT 5718, 'NAICS', 2066, 81142, 4, 2065, 811420, 5 FROM dual
UNION ALL SELECT 5719, 'NAICS', 2188, 926, 2, 2190, 926110, 5 FROM dual
UNION ALL SELECT 5720, 'NAICS', 2039, 8111, 3, 2048, 81119, 4 FROM dual
UNION ALL SELECT 5721, 'SIC', 3662, 4600, 2, 3664, 4612, 4 FROM dual
UNION ALL SELECT 5722, 'SEC', 2221, 1300, 2, 2227, 1389, 4 FROM dual
UNION ALL SELECT 5723, 'NAICS', 2137, 9211, 3, 2142, 921130, 5 FROM dual
UNION ALL SELECT 5724, 'NAICS', 2135, 92, 1, 2140, 921120, 5 FROM dual
UNION ALL SELECT 5725, 'NAICS', 1725, 56, 1, 1765, 561599, 5 FROM dual
UNION ALL SELECT 5726, 'SIC', 3917, 6000, 2, 3929, 6061, 4 FROM dual
UNION ALL SELECT 5727, 'SIC', 3496, 3640, 3, 3501, 3646, 4 FROM dual
UNION ALL SELECT 5728, 'NAICS', 1625, 541, 2, 1641, 541310, 5 FROM dual
UNION ALL SELECT 5729, 'SEC', 2791, 20, 1, 2434, 3600, 2 FROM dual
UNION ALL SELECT 5730, 'NAICS', 23, 1113, 3, 33, 111335, 5 FROM dual
UNION ALL SELECT 5731, 'NAICS', 1994, 71392, 4, 1993, 713920, 5 FROM dual
UNION ALL SELECT 5732, 'NAICS', 144, 2122, 3, 154, 212291, 5 FROM dual
UNION ALL SELECT 5733, 'NAICS', 183, 22111, 4, 185, 221112, 5 FROM dual
UNION ALL SELECT 5734, 'NAICS', 1536, 524, 2, 1539, 524113, 5 FROM dual
UNION ALL SELECT 5735, 'NAICS', 1625, 541, 2, 1682, 54169, 4 FROM dual
UNION ALL SELECT 5736, 'SIC', 4266, 9310, 3, 4267, 9311, 4 FROM dual
UNION ALL SELECT 5737, 'SIC', 2969, 1760, 3, 2970, 1761, 4 FROM dual
UNION ALL SELECT 5738, 'NAICS', 1004, 4239, 3, 1013, 423990, 5 FROM dual
UNION ALL SELECT 5739, 'NAICS', 1727, 5611, 3, 1728, 561110, 5 FROM dual
UNION ALL SELECT 5740, 'SIC', 3489, 3630, 3, 3491, 3632, 4 FROM dual
UNION ALL SELECT 5741, 'SIC', 3822, 5400, 2, 3825, 5420, 3 FROM dual
UNION ALL SELECT 5742, 'SIC', 4017, 7200, 2, 4039, 7299, 4 FROM dual
UNION ALL SELECT 5743, 'NAICS', 1015, 424, 2, 1019, 424120, 5 FROM dual
UNION ALL SELECT 5744, 'NAICS', 1015, 424, 2, 1033, 424340, 5 FROM dual
UNION ALL SELECT 5745, 'SEC', 2479, 3820, 3, 2480, 3821, 4 FROM dual
UNION ALL SELECT 5746, 'NAICS', 3, 1111, 3, 8, 111130, 5 FROM dual
UNION ALL SELECT 5747, 'SIC', 3195, 2730, 3, 3196, 2731, 4 FROM dual
UNION ALL SELECT 5748, 'NAICS', 1792, 562, 2, 1797, 562119, 5 FROM dual
UNION ALL SELECT 5749, 'SEC', 2694, 6500, 2, 2698, 6519, 4 FROM dual
UNION ALL SELECT 5750, 'NAICS', 2104, 8131, 3, 2105, 813110, 5 FROM dual
UNION ALL SELECT 5751, 'NAICS', 117, 115, 2, 121, 115112, 5 FROM dual
UNION ALL SELECT 5752, 'NAICS', 1793, 5621, 3, 1795, 562111, 5 FROM dual
UNION ALL SELECT 5753, 'NAICS', 1624, 54, 1, 1702, 541870, 5 FROM dual
UNION ALL SELECT 5754, 'NAICS', 181, 221, 2, 199, 221310, 5 FROM dual
UNION ALL SELECT 5755, 'NAICS', 1416, 5112, 3, 1418, 51121, 4 FROM dual
UNION ALL SELECT 5756, 'NAICS', 1525, 52321, 4, 1524, 523210, 5 FROM dual
UNION ALL SELECT 5757, 'SEC', 2713, 7300, 2, 2716, 7320, 3 FROM dual
UNION ALL SELECT 5758, 'SEC', 2796, 70, 1, 2741, 7819, 4 FROM dual
UNION ALL SELECT 5759, 'SIC', 3715, 5000, 2, 3745, 5072, 4 FROM dual
UNION ALL SELECT 5760, 'SIC', 4312, 60, 1, 3941, 6141, 4 FROM dual
UNION ALL SELECT 5761, 'SIC', 4313, 70, 1, 4008, 7010, 3 FROM dual
UNION ALL SELECT 5762, 'NAICS', 1, 11, 1, 78, 11239, 4 FROM dual
UNION ALL SELECT 5763, 'SEC', 2791, 20, 1, 2473, 3751, 4 FROM dual
UNION ALL SELECT 5764, 'SIC', 3815, 5300, 2, 3816, 5310, 3 FROM dual
UNION ALL SELECT 5765, 'SIC', 4313, 70, 1, 4212, 8621, 4 FROM dual
UNION ALL SELECT 5766, 'NAICS', 967, 42346, 4, 966, 423460, 5 FROM dual
UNION ALL SELECT 5767, 'SIC', 4308, 20, 1, 3536, 3728, 4 FROM dual
UNION ALL SELECT 5768, 'NAICS', 1625, 541, 2, 1714, 541940, 5 FROM dual
UNION ALL SELECT 5769, 'SIC', 3937, 6100, 2, 3945, 6160, 3 FROM dual
UNION ALL SELECT 5770, 'SIC', 4208, 8600, 2, 4213, 8630, 3 FROM dual
UNION ALL SELECT 5771, 'NAICS', 1026, 4243, 3, 1032, 42433, 4 FROM dual
UNION ALL SELECT 5772, 'SIC', 3337, 3300, 2, 3361, 3357, 4 FROM dual
UNION ALL SELECT 5773, 'NAICS', 2, 111, 2, 27, 11132, 4 FROM dual
UNION ALL SELECT 5774, 'NAICS', 2150, 922, 2, 2157, 92213, 4 FROM dual
UNION ALL SELECT 5775, 'SEC', 2476, 3800, 2, 2485, 3826, 4 FROM dual
UNION ALL SELECT 5776, 'NAICS', 2151, 9221, 3, 2158, 922140, 5 FROM dual
UNION ALL SELECT 5777, 'SEC', 2342, 3000, 2, 2347, 3050, 3 FROM dual
UNION ALL SELECT 5778, 'SEC', 2791, 20, 1, 2345, 3020, 3 FROM dual
UNION ALL SELECT 5779, 'NAICS', 235, 238, 2, 268, 23834, 4 FROM dual
UNION ALL SELECT 5780, 'SIC', 3330, 3290, 3, 3334, 3296, 4 FROM dual
UNION ALL SELECT 5781, 'NAICS', 1725, 56, 1, 1737, 561320, 5 FROM dual
UNION ALL SELECT 5782, 'NAICS', 1920, 624, 2, 1925, 62412, 4 FROM dual
UNION ALL SELECT 5783, 'SEC', 2791, 20, 1, 2400, 3470, 3 FROM dual
UNION ALL SELECT 5784, 'SEC', 2552, 4900, 2, 2558, 4924, 4 FROM dual
UNION ALL SELECT 5785, 'NAICS', 930, 42, 1, 1062, 424610, 5 FROM dual
UNION ALL SELECT 5786, 'NAICS', 1709, 54192, 4, 1711, 541922, 5 FROM dual
UNION ALL SELECT 5787, 'SEC', 2792, 40, 1, 2541, 4812, 4 FROM dual
UNION ALL SELECT 5788, 'SEC', 2775, 8710, 3, 2776, 8711, 4 FROM dual
UNION ALL SELECT 5789, 'NAICS', 1089, 425, 2, 1093, 425120, 5 FROM dual
UNION ALL SELECT 5790, 'SIC', 4309, 40, 1, 3664, 4612, 4 FROM dual
UNION ALL SELECT 5791, 'NAICS', 221, 23711, 4, 220, 237110, 5 FROM dual
UNION ALL SELECT 5792, 'SIC', 3681, 4810, 3, 3683, 4813, 4 FROM dual
UNION ALL SELECT 5793, 'SEC', 2790, 15, 1, 2239, 1731, 4 FROM dual
UNION ALL SELECT 5794, 'SIC', 3272, 3080, 3, 3277, 3085, 4 FROM dual
UNION ALL SELECT 5795, 'NAICS', 1850, 62, 1, 1871, 6214, 3 FROM dual
UNION ALL SELECT 5796, 'NAICS', 1851, 621, 2, 1893, 621999, 5 FROM dual
UNION ALL SELECT 5797, 'SIC', 4313, 70, 1, 4184, 8230, 3 FROM dual
UNION ALL SELECT 5798, 'SEC', 2262, 2200, 2, 2264, 2211, 4 FROM dual
UNION ALL SELECT 5799, 'SIC', 4313, 70, 1, 4123, 7840, 3 FROM dual
UNION ALL SELECT 5800, 'NAICS', 2004, 721, 2, 2014, 72121, 4 FROM dual
UNION ALL SELECT 5801, 'SIC', 3542, 3750, 3, 3543, 3751, 4 FROM dual
UNION ALL SELECT 5802, 'NAICS', 1452, 517, 2, 1457, 517210, 5 FROM dual
UNION ALL SELECT 5803, 'NAICS', 1850, 62, 1, 1889, 621910, 5 FROM dual
UNION ALL SELECT 5804, 'NAICS', 1625, 541, 2, 1711, 541922, 5 FROM dual
UNION ALL SELECT 5805, 'SEC', 2792, 40, 1, 2539, 4800, 2 FROM dual
UNION ALL SELECT 5806, 'NAICS', 1006, 42391, 4, 1005, 423910, 5 FROM dual
UNION ALL SELECT 5807, 'NAICS', 1624, 54, 1, 1681, 541690, 5 FROM dual
UNION ALL SELECT 5808, 'SIC', 2798, 100, 2, 2822, 182, 4 FROM dual
UNION ALL SELECT 5809, 'SIC', 4307, 15, 1, 2940, 1520, 3 FROM dual
UNION ALL SELECT 5810, 'NAICS', 2, 111, 2, 21, 111211, 5 FROM dual
UNION ALL SELECT 5811, 'SIC', 2987, 2020, 3, 2988, 2021, 4 FROM dual
UNION ALL SELECT 5812, 'SIC', 3701, 4930, 3, 3704, 4939, 4 FROM dual
UNION ALL SELECT 5813, 'SEC', 2795, 60, 1, 2706, 6794, 4 FROM dual
UNION ALL SELECT 5814, 'SEC', 2796, 70, 1, 2709, 7000, 2 FROM dual
UNION ALL SELECT 5815, 'NAICS', 2061, 8114, 3, 2070, 81149, 4 FROM dual
UNION ALL SELECT 5816, 'NAICS', 1068, 42471, 4, 1067, 424710, 5 FROM dual
UNION ALL SELECT 5817, 'NAICS', 1625, 541, 2, 1667, 54151, 4 FROM dual
UNION ALL SELECT 5818, 'SEC', 2777, 8730, 3, 2778, 8731, 4 FROM dual
UNION ALL SELECT 5819, 'SEC', 2505, 3940, 3, 2506, 3942, 4 FROM dual
UNION ALL SELECT 5820, 'SEC', 2240, 2000, 2, 2246, 2024, 4 FROM dual
UNION ALL SELECT 5821, 'SIC', 4313, 70, 1, 4166, 8070, 3 FROM dual
UNION ALL SELECT 5822, 'SIC', 2955, 1700, 2, 2972, 1771, 4 FROM dual
UNION ALL SELECT 5823, 'NAICS', 2003, 72, 1, 2025, 72232, 4 FROM dual
UNION ALL SELECT 5824, 'SIC', 3996, 6720, 3, 3997, 6722, 4 FROM dual
UNION ALL SELECT 5825, 'NAICS', 2136, 921, 2, 2146, 921150, 5 FROM dual
UNION ALL SELECT 5826, 'SIC', 4012, 7030, 3, 4014, 7033, 4 FROM dual
UNION ALL SELECT 5827, 'NAICS', 1812, 61, 1, 1822, 61131, 4 FROM dual
UNION ALL SELECT 5828, 'NAICS', 2004, 721, 2, 2008, 721120, 5 FROM dual
UNION ALL SELECT 5829, 'SEC', 2791, 20, 1, 2490, 3842, 4 FROM dual
UNION ALL SELECT 5830, 'SEC', 2600, 5170, 3, 2601, 5171, 4 FROM dual
UNION ALL SELECT 5831, 'NAICS', 3, 1111, 3, 16, 11119, 4 FROM dual
UNION ALL SELECT 5832, 'NAICS', 1015, 424, 2, 1031, 424330, 5 FROM dual
UNION ALL SELECT 5833, 'SIC', 4040, 7300, 2, 4056, 7342, 4 FROM dual
UNION ALL SELECT 5834, 'NAICS', 1402, 51, 1, 1447, 515120, 5 FROM dual
UNION ALL SELECT 5835, 'NAICS', 1718, 55, 1, 1719, 551, 2 FROM dual
UNION ALL SELECT 5836, 'NAICS', 2020, 722, 2, 2034, 722513, 5 FROM dual
UNION ALL SELECT 5837, 'SIC', 4308, 20, 1, 3470, 3582, 4 FROM dual
UNION ALL SELECT 5838, 'SIC', 4186, 8240, 3, 4187, 8243, 4 FROM dual
UNION ALL SELECT 5839, 'NAICS', 1480, 52, 1, 1498, 522220, 5 FROM dual
UNION ALL SELECT 5840, 'NAICS', 1, 11, 1, 19, 1112, 3 FROM dual
UNION ALL SELECT 5841, 'SIC', 4312, 60, 1, 3975, 6410, 3 FROM dual
UNION ALL SELECT 5842, 'NAICS', 1943, 711, 2, 1957, 711219, 5 FROM dual
UNION ALL SELECT 5843, 'NAICS', 1, 11, 1, 10, 111140, 5 FROM dual
UNION ALL SELECT 5844, 'NAICS', 981, 42369, 4, 980, 423690, 5 FROM dual
UNION ALL SELECT 5845, 'SIC', 3235, 2850, 3, 3236, 2851, 4 FROM dual
UNION ALL SELECT 5846, 'NAICS', 1689, 5418, 3, 1693, 54182, 4 FROM dual
UNION ALL SELECT 5847, 'NAICS', 1625, 541, 2, 1716, 541990, 5 FROM dual
UNION ALL SELECT 5848, 'SIC', 3489, 3630, 3, 3492, 3633, 4 FROM dual
UNION ALL SELECT 5849, 'SIC', 4223, 8700, 2, 4240, 8748, 4 FROM dual
UNION ALL SELECT 5850, 'NAICS', 1419, 512, 2, 1436, 512230, 5 FROM dual
UNION ALL SELECT 5851, 'SIC', 4306, 10, 1, 2888, 1021, 4 FROM dual
UNION ALL SELECT 5852, 'SEC', 2748, 7900, 2, 2750, 7948, 4 FROM dual
UNION ALL SELECT 5853, 'SIC', 2884, 1000, 2, 2900, 1099, 4 FROM dual
UNION ALL SELECT 5854, 'NAICS', 1624, 54, 1, 1640, 5413, 3 FROM dual
UNION ALL SELECT 5855, 'SEC', 2218, 1200, 2, 2220, 1221, 4 FROM dual
UNION ALL SELECT 5856, 'SIC', 2940, 1520, 3, 2942, 1522, 4 FROM dual
UNION ALL SELECT 5857, 'NAICS', 1625, 541, 2, 1709, 54192, 4 FROM dual
UNION ALL SELECT 5858, 'NAICS', 1979, 713, 2, 1991, 713910, 5 FROM dual
UNION ALL SELECT 5859, 'SIC', 4309, 40, 1, 3698, 4923, 4 FROM dual
UNION ALL SELECT 5860, 'SIC', 3749, 5080, 3, 3750, 5082, 4 FROM dual
UNION ALL SELECT 5861, 'SIC', 3620, 4200, 2, 3632, 4231, 4 FROM dual
UNION ALL SELECT 5862, 'NAICS', 2177, 9241, 3, 2180, 924120, 5 FROM dual
UNION ALL SELECT 5863, 'NAICS', 1850, 62, 1, 1858, 62121, 4 FROM dual
UNION ALL SELECT 5864, 'NAICS', 2189, 9261, 3, 2196, 926140, 5 FROM dual
UNION ALL SELECT 5865, 'SIC', 3667, 4700, 2, 3673, 4731, 4 FROM dual
UNION ALL SELECT 5866, 'SEC', 2791, 20, 1, 2335, 2890, 3 FROM dual
UNION ALL SELECT 5867, 'SIC', 4308, 20, 1, 3247, 2891, 4 FROM dual
UNION ALL SELECT 5868, 'NAICS', 2, 111, 2, 20, 11121, 4 FROM dual
UNION ALL SELECT 5869, 'NAICS', 1624, 54, 1, 1712, 541930, 5 FROM dual
UNION ALL SELECT 5870, 'SEC', 2315, 2770, 3, 2316, 2771, 4 FROM dual
UNION ALL SELECT 5871, 'NAICS', 126, 1152, 3, 127, 115210, 5 FROM dual
UNION ALL SELECT 5872, 'NAICS', 930, 42, 1, 1003, 42386, 4 FROM dual
UNION ALL SELECT 5873, 'NAICS', 2037, 81, 1, 2070, 81149, 4 FROM dual
UNION ALL SELECT 5874, 'SEC', 2355, 3200, 2, 2363, 3241, 4 FROM dual
UNION ALL SELECT 5875, 'SEC', 2678, 6310, 3, 2679, 6311, 4 FROM dual
UNION ALL SELECT 5876, 'SIC', 4306, 10, 1, 2898, 1090, 3 FROM dual
UNION ALL SELECT 5877, 'NAICS', 272, 23839, 4, 271, 238390, 5 FROM dual
UNION ALL SELECT 5878, 'NAICS', 1979, 713, 2, 1988, 713290, 5 FROM dual
UNION ALL SELECT 5879, 'NAICS', 1513, 523, 2, 1535, 523999, 5 FROM dual
UNION ALL SELECT 5880, 'SIC', 2955, 1700, 2, 2980, 1796, 4 FROM dual
UNION ALL SELECT 5881, 'NAICS', 1402, 51, 1, 1473, 51911, 4 FROM dual
UNION ALL SELECT 5882, 'SIC', 2825, 200, 2, 2830, 214, 4 FROM dual
UNION ALL SELECT 5883, 'NAICS', 950, 42332, 4, 949, 423320, 5 FROM dual
UNION ALL SELECT 5884, 'NAICS', 1592, 5321, 3, 1594, 532111, 5 FROM dual
UNION ALL SELECT 5885, 'NAICS', 1851, 621, 2, 1875, 62142, 4 FROM dual
UNION ALL SELECT 5886, 'SEC', 2791, 20, 1, 2289, 2531, 4 FROM dual
UNION ALL SELECT 5887, 'NAICS', 954, 42339, 4, 953, 423390, 5 FROM dual
UNION ALL SELECT 5888, 'NAICS', 1609, 5323, 3, 1610, 532310, 5 FROM dual
UNION ALL SELECT 5889, 'SEC', 2694, 6500, 2, 2700, 6531, 4 FROM dual
UNION ALL SELECT 5890, 'NAICS', 205, 23, 1, 265, 238330, 5 FROM dual
UNION ALL SELECT 5891, 'SEC', 2796, 70, 1, 2746, 7840, 3 FROM dual
UNION ALL SELECT 5892, 'NAICS', 1480, 52, 1, 1561, 52519, 4 FROM dual
UNION ALL SELECT 5893, 'SIC', 3041, 2100, 2, 3044, 2120, 3 FROM dual
UNION ALL SELECT 5894, 'SIC', 2901, 1200, 2, 2903, 1221, 4 FROM dual
UNION ALL SELECT 5895, 'SIC', 4308, 20, 1, 3088, 2322, 4 FROM dual
UNION ALL SELECT 5896, 'SIC', 3461, 3570, 3, 3464, 3575, 4 FROM dual
UNION ALL SELECT 5897, 'SIC', 4308, 20, 1, 3116, 2391, 4 FROM dual
UNION ALL SELECT 5898, 'SIC', 3180, 2670, 3, 3185, 2675, 4 FROM dual
UNION ALL SELECT 5899, 'NAICS', 1402, 51, 1, 1425, 51213, 4 FROM dual
UNION ALL SELECT 5900, 'SIC', 4313, 70, 1, 4096, 7539, 4 FROM dual
UNION ALL SELECT 5901, 'SEC', 2791, 20, 1, 2378, 3340, 3 FROM dual
UNION ALL SELECT 5902, 'NAICS', 1076, 4249, 3, 1085, 424950, 5 FROM dual
UNION ALL SELECT 5903, 'NAICS', 1943, 711, 2, 1960, 71131, 4 FROM dual
UNION ALL SELECT 5904, 'SIC', 4308, 20, 1, 3363, 3363, 4 FROM dual
UNION ALL SELECT 5905, 'NAICS', 9, 11113, 4, 8, 111130, 5 FROM dual
UNION ALL SELECT 5906, 'NAICS', 235, 238, 2, 238, 23811, 4 FROM dual
UNION ALL SELECT 5907, 'SIC', 3668, 4720, 3, 3669, 4724, 4 FROM dual
UNION ALL SELECT 5908, 'NAICS', 162, 21232, 4, 164, 212322, 5 FROM dual
UNION ALL SELECT 5909, 'SEC', 2610, 5300, 2, 2616, 5399, 4 FROM dual
UNION ALL SELECT 5910, 'SEC', 2796, 70, 1, 2761, 8071, 4 FROM dual
UNION ALL SELECT 5911, 'SIC', 4308, 20, 1, 3579, 3914, 4 FROM dual
UNION ALL SELECT 5912, 'NAICS', 207, 2361, 3, 210, 236116, 5 FROM dual
UNION ALL SELECT 5913, 'NAICS', 2020, 722, 2, 2035, 722514, 5 FROM dual
UNION ALL SELECT 5914, 'NAICS', 138, 212, 2, 141, 212111, 5 FROM dual
UNION ALL SELECT 5915, 'SIC', 4313, 70, 1, 4064, 7363, 4 FROM dual
UNION ALL SELECT 5916, 'NAICS', 1444, 51511, 4, 1445, 515111, 5 FROM dual
UNION ALL SELECT 5917, 'SIC', 3057, 2240, 3, 3058, 2241, 4 FROM dual
UNION ALL SELECT 5918, 'SIC', 3322, 3270, 3, 3326, 3274, 4 FROM dual
UNION ALL SELECT 5919, 'NAICS', 930, 42, 1, 1011, 423940, 5 FROM dual
UNION ALL SELECT 5920, 'NAICS', 930, 42, 1, 994, 423820, 5 FROM dual
UNION ALL SELECT 5921, 'SIC', 2901, 1200, 2, 2908, 1241, 4 FROM dual
UNION ALL SELECT 5922, 'SIC', 3214, 2800, 2, 3250, 2895, 4 FROM dual
UNION ALL SELECT 5923, 'SIC', 4306, 10, 1, 2919, 1410, 3 FROM dual
UNION ALL SELECT 5924, 'SIC', 4309, 40, 1, 3642, 4432, 4 FROM dual
UNION ALL SELECT 5925, 'NAICS', 1419, 512, 2, 1421, 512110, 5 FROM dual
UNION ALL SELECT 5926, 'NAICS', 1850, 62, 1, 1914, 62331, 4 FROM dual
UNION ALL SELECT 5927, 'NAICS', 2038, 811, 2, 2044, 811118, 5 FROM dual
UNION ALL SELECT 5928, 'SIC', 4310, 50, 1, 3771, 5136, 4 FROM dual
UNION ALL SELECT 5929, 'NAICS', 1979, 713, 2, 2001, 713990, 5 FROM dual
UNION ALL SELECT 5930, 'NAICS', 1876, 62149, 4, 1880, 621498, 5 FROM dual
UNION ALL SELECT 5931, 'SEC', 2713, 7300, 2, 2734, 7389, 4 FROM dual
UNION ALL SELECT 5932, 'SIC', 3480, 3600, 2, 3510, 3669, 4 FROM dual
UNION ALL SELECT 5933, 'NAICS', 930, 42, 1, 1007, 423920, 5 FROM dual
UNION ALL SELECT 5934, 'NAICS', 1485, 522, 2, 1510, 52232, 4 FROM dual
UNION ALL SELECT 5935, 'NAICS', 1562, 5259, 3, 1565, 525920, 5 FROM dual
UNION ALL SELECT 5936, 'SIC', 4256, 9200, 2, 4259, 9220, 3 FROM dual
UNION ALL SELECT 5937, 'NAICS', 1537, 5241, 3, 1544, 524128, 5 FROM dual
UNION ALL SELECT 5938, 'NAICS', 1, 11, 1, 33, 111335, 5 FROM dual
UNION ALL SELECT 5939, 'SEC', 2355, 3200, 2, 2360, 3230, 3 FROM dual
UNION ALL SELECT 5940, 'SEC', 2456, 3690, 3, 2457, 3695, 4 FROM dual
UNION ALL SELECT 5941, 'SIC', 4313, 70, 1, 4013, 7032, 4 FROM dual
UNION ALL SELECT 5942, 'NAICS', 1730, 5612, 3, 1732, 56121, 4 FROM dual
UNION ALL SELECT 5943, 'NAICS', 1605, 53229, 4, 1606, 532291, 5 FROM dual
UNION ALL SELECT 5944, 'SIC', 4308, 20, 1, 3465, 3577, 4 FROM dual
UNION ALL SELECT 5945, 'NAICS', 1725, 56, 1, 1751, 56144, 4 FROM dual
UNION ALL SELECT 5946, 'SIC', 3272, 3080, 3, 3273, 3081, 4 FROM dual
UNION ALL SELECT 5947, 'SIC', 3721, 5020, 3, 3723, 5023, 4 FROM dual
UNION ALL SELECT 5948, 'SEC', 2792, 40, 1, 2537, 4730, 3 FROM dual
UNION ALL SELECT 5949, 'NAICS', 1901, 6223, 3, 1902, 622310, 5 FROM dual
UNION ALL SELECT 5950, 'SIC', 3214, 2800, 2, 3222, 2822, 4 FROM dual
UNION ALL SELECT 5951, 'SEC', 2319, 2800, 2, 2327, 2836, 4 FROM dual
UNION ALL SELECT 5952, 'SIC', 4313, 70, 1, 4170, 8082, 4 FROM dual
UNION ALL SELECT 5953, 'SIC', 3261, 3000, 2, 3272, 3080, 3 FROM dual
UNION ALL SELECT 5954, 'SIC', 3859, 5630, 3, 3860, 5632, 4 FROM dual
UNION ALL SELECT 5955, 'NAICS', 955, 4234, 3, 964, 423450, 5 FROM dual
UNION ALL SELECT 5956, 'SIC', 4313, 70, 1, 4019, 7211, 4 FROM dual
UNION ALL SELECT 5957, 'SIC', 2982, 2000, 2, 3038, 2097, 4 FROM dual
UNION ALL SELECT 5958, 'NAICS', 1557, 52511, 4, 1556, 525110, 5 FROM dual
UNION ALL SELECT 5959, 'NAICS', 931, 423, 2, 941, 4232, 3 FROM dual
UNION ALL SELECT 5960, 'NAICS', 1726, 561, 2, 1752, 561450, 5 FROM dual
UNION ALL SELECT 5961, 'SEC', 2793, 50, 1, 2573, 5031, 4 FROM dual
UNION ALL SELECT 5962, 'SIC', 3605, 4100, 2, 3617, 4151, 4 FROM dual
UNION ALL SELECT 5963, 'NAICS', 1625, 541, 2, 1693, 54182, 4 FROM dual
UNION ALL SELECT 5964, 'NAICS', 1831, 61151, 4, 1834, 611513, 5 FROM dual
UNION ALL SELECT 5965, 'SEC', 2791, 20, 1, 2350, 3081, 4 FROM dual
UNION ALL SELECT 5966, 'SIC', 3285, 3130, 3, 3286, 3131, 4 FROM dual
UNION ALL SELECT 5967, 'NAICS', 1850, 62, 1, 1874, 621420, 5 FROM dual
UNION ALL SELECT 5968, 'SIC', 3958, 6300, 2, 3971, 6371, 4 FROM dual
UNION ALL SELECT 5969, 'NAICS', 1569, 53, 1, 1574, 531120, 5 FROM dual
UNION ALL SELECT 5970, 'NAICS', 218, 237, 2, 220, 237110, 5 FROM dual
UNION ALL SELECT 5971, 'NAICS', 1480, 52, 1, 1516, 52311, 4 FROM dual
UNION ALL SELECT 5972, 'SEC', 2791, 20, 1, 2374, 3317, 4 FROM dual
UNION ALL SELECT 5973, 'SIC', 3886, 5900, 2, 3897, 5944, 4 FROM dual
UNION ALL SELECT 5974, 'SIC', 4313, 70, 1, 4082, 7510, 3 FROM dual
UNION ALL SELECT 5975, 'NAICS', 1813, 611, 2, 1840, 61162, 4 FROM dual
UNION ALL SELECT 5976, 'NAICS', 1442, 515, 2, 1446, 515112, 5 FROM dual
UNION ALL SELECT 5977, 'NAICS', 1569, 53, 1, 1591, 532, 2 FROM dual
UNION ALL SELECT 5978, 'NAICS', 1495, 5222, 3, 1496, 522210, 5 FROM dual
UNION ALL SELECT 5979, 'SIC', 3124, 2400, 2, 3133, 2434, 4 FROM dual
UNION ALL SELECT 5980, 'NAICS', 930, 42, 1, 966, 423460, 5 FROM dual
UNION ALL SELECT 5981, 'SIC', 4308, 20, 1, 3104, 2361, 4 FROM dual
UNION ALL SELECT 5982, 'SIC', 4309, 40, 1, 3654, 4500, 2 FROM dual
UNION ALL SELECT 5983, 'NAICS', 1480, 52, 1, 1484, 52111, 4 FROM dual
UNION ALL SELECT 5984, 'SIC', 2951, 1620, 3, 2952, 1622, 4 FROM dual
UNION ALL SELECT 5985, 'SIC', 4311, 52, 1, 3806, 5211, 4 FROM dual
UNION ALL SELECT 5986, 'SIC', 4313, 70, 1, 4180, 8211, 4 FROM dual
UNION ALL SELECT 5987, 'NAICS', 206, 236, 2, 210, 236116, 5 FROM dual
UNION ALL SELECT 5988, 'NAICS', 1726, 561, 2, 1731, 561210, 5 FROM dual
UNION ALL SELECT 5989, 'SIC', 4257, 9210, 3, 4258, 9211, 4 FROM dual
UNION ALL SELECT 5990, 'NAICS', 1850, 62, 1, 1927, 62419, 4 FROM dual
UNION ALL SELECT 5991, 'NAICS', 2104, 8131, 3, 2106, 81311, 4 FROM dual
UNION ALL SELECT 5992, 'NAICS', 1480, 52, 1, 1556, 525110, 5 FROM dual
UNION ALL SELECT 5993, 'SEC', 2477, 3810, 3, 2478, 3812, 4 FROM dual
UNION ALL SELECT 5994, 'NAICS', 1480, 52, 1, 1499, 52222, 4 FROM dual
UNION ALL SELECT 5995, 'SIC', 2982, 2000, 2, 3009, 2051, 4 FROM dual
UNION ALL SELECT 5996, 'NAICS', 118, 1151, 3, 121, 115112, 5 FROM dual
UNION ALL SELECT 5997, 'SIC', 3261, 3000, 2, 3270, 3061, 4 FROM dual
UNION ALL SELECT 5998, 'NAICS', 1480, 52, 1, 1537, 5241, 3 FROM dual
UNION ALL SELECT 5999, 'NAICS', 1925, 62412, 4, 1924, 624120, 5 FROM dual
UNION ALL SELECT 6000, 'SIC', 4308, 20, 1, 3010, 2052, 4 FROM dual
UNION ALL SELECT 6001, 'SEC', 2328, 2840, 3, 2329, 2842, 4 FROM dual
UNION ALL SELECT 6002, 'SIC', 3086, 2320, 3, 3090, 2325, 4 FROM dual
UNION ALL SELECT 6003, 'NAICS', 1569, 53, 1, 1594, 532111, 5 FROM dual
UNION ALL SELECT 6004, 'NAICS', 1727, 5611, 3, 1729, 56111, 4 FROM dual
UNION ALL SELECT 6005, 'SIC', 4284, 9600, 2, 4291, 9640, 3 FROM dual
UNION ALL SELECT 6006, 'NAICS', 1, 11, 1, 107, 11331, 4 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 6007, 'NAICS', 1419, 512, 2, 1434, 512220, 5 FROM dual
UNION ALL SELECT 6008, 'NAICS', 1495, 5222, 3, 1502, 522292, 5 FROM dual
UNION ALL SELECT 6009, 'SIC', 2825, 200, 2, 2843, 273, 4 FROM dual
UNION ALL SELECT 6010, 'SIC', 4308, 20, 1, 3395, 3460, 3 FROM dual
UNION ALL SELECT 6011, 'SIC', 4017, 7200, 2, 4021, 7213, 4 FROM dual
UNION ALL SELECT 6012, 'SIC', 4310, 50, 1, 3732, 5045, 4 FROM dual
UNION ALL SELECT 6013, 'SEC', 2792, 40, 1, 2524, 4410, 3 FROM dual
UNION ALL SELECT 6014, 'SIC', 3643, 4440, 3, 3644, 4449, 4 FROM dual
UNION ALL SELECT 6015, 'NAICS', 257, 23822, 4, 256, 238220, 5 FROM dual
UNION ALL SELECT 6016, 'SIC', 4312, 60, 1, 3958, 6300, 2 FROM dual
UNION ALL SELECT 6017, 'SIC', 4166, 8070, 3, 4168, 8072, 4 FROM dual
UNION ALL SELECT 6018, 'SEC', 2796, 70, 1, 2768, 8111, 4 FROM dual
UNION ALL SELECT 6019, 'SEC', 2738, 7800, 2, 2742, 7820, 3 FROM dual
UNION ALL SELECT 6020, 'SEC', 2791, 20, 1, 2402, 3490, 3 FROM dual
UNION ALL SELECT 6021, 'NAICS', 1928, 6242, 3, 1931, 62422, 4 FROM dual
UNION ALL SELECT 6022, 'NAICS', 1035, 4244, 3, 1048, 424470, 5 FROM dual
UNION ALL SELECT 6023, 'SEC', 2600, 5170, 3, 2602, 5172, 4 FROM dual
UNION ALL SELECT 6024, 'SEC', 2240, 2000, 2, 2245, 2020, 3 FROM dual
UNION ALL SELECT 6025, 'SIC', 4082, 7510, 3, 4084, 7514, 4 FROM dual
UNION ALL SELECT 6026, 'SEC', 2794, 52, 1, 2617, 5400, 2 FROM dual
UNION ALL SELECT 6027, 'SIC', 3889, 5920, 3, 3890, 5921, 4 FROM dual
UNION ALL SELECT 6028, 'NAICS', 192, 22112, 4, 194, 221122, 5 FROM dual
UNION ALL SELECT 6029, 'NAICS', 1, 11, 1, 13, 11115, 4 FROM dual
UNION ALL SELECT 6030, 'SIC', 4308, 20, 1, 3387, 3443, 4 FROM dual
UNION ALL SELECT 6031, 'SIC', 3261, 3000, 2, 3262, 3010, 3 FROM dual
UNION ALL SELECT 6032, 'NAICS', 1, 11, 1, 91, 11291, 4 FROM dual
UNION ALL SELECT 6033, 'SIC', 4314, 90, 1, 4257, 9210, 3 FROM dual
UNION ALL SELECT 6034, 'SIC', 4308, 20, 1, 3459, 3568, 4 FROM dual
UNION ALL SELECT 6035, 'SIC', 4309, 40, 1, 3677, 4783, 4 FROM dual
UNION ALL SELECT 6036, 'NAICS', 1569, 53, 1, 1572, 531110, 5 FROM dual
UNION ALL SELECT 6037, 'SIC', 4308, 20, 1, 3032, 2087, 4 FROM dual
UNION ALL SELECT 6038, 'NAICS', 2145, 92114, 4, 2144, 921140, 5 FROM dual
UNION ALL SELECT 6039, 'NAICS', 235, 238, 2, 255, 23821, 4 FROM dual
UNION ALL SELECT 6040, 'SEC', 2791, 20, 1, 2385, 3410, 3 FROM dual
UNION ALL SELECT 6041, 'SIC', 4208, 8600, 2, 4214, 8631, 4 FROM dual
UNION ALL SELECT 6042, 'SIC', 4308, 20, 1, 3562, 3827, 4 FROM dual
UNION ALL SELECT 6043, 'NAICS', 262, 23831, 4, 261, 238310, 5 FROM dual
UNION ALL SELECT 6044, 'SIC', 4208, 8600, 2, 4209, 8610, 3 FROM dual
UNION ALL SELECT 6045, 'SEC', 2792, 40, 1, 2516, 4100, 2 FROM dual
UNION ALL SELECT 6046, 'SIC', 4308, 20, 1, 3515, 3675, 4 FROM dual
UNION ALL SELECT 6047, 'NAICS', 1402, 51, 1, 1420, 5121, 3 FROM dual
UNION ALL SELECT 6048, 'NAICS', 1634, 5412, 3, 1638, 541214, 5 FROM dual
UNION ALL SELECT 6049, 'NAICS', 1920, 624, 2, 1921, 6241, 3 FROM dual
UNION ALL SELECT 6050, 'SIC', 3282, 3100, 2, 3297, 3171, 4 FROM dual
UNION ALL SELECT 6051, 'SIC', 3784, 5150, 3, 3786, 5154, 4 FROM dual
UNION ALL SELECT 6052, 'NAICS', 1903, 62231, 4, 1902, 622310, 5 FROM dual
UNION ALL SELECT 6053, 'NAICS', 1942, 71, 1, 1984, 71312, 4 FROM dual
UNION ALL SELECT 6054, 'SEC', 2694, 6500, 2, 2699, 6530, 3 FROM dual
UNION ALL SELECT 6055, 'SIC', 3053, 2220, 3, 3054, 2221, 4 FROM dual
UNION ALL SELECT 6056, 'NAICS', 930, 42, 1, 969, 42349, 4 FROM dual
UNION ALL SELECT 6057, 'NAICS', 1066, 4247, 3, 1067, 424710, 5 FROM dual
UNION ALL SELECT 6058, 'SIC', 4309, 40, 1, 3643, 4440, 3 FROM dual
UNION ALL SELECT 6059, 'SEC', 2310, 2740, 3, 2311, 2741, 4 FROM dual
UNION ALL SELECT 6060, 'SEC', 2596, 5140, 3, 2597, 5141, 4 FROM dual
UNION ALL SELECT 6061, 'NAICS', 198, 2213, 3, 199, 221310, 5 FROM dual
UNION ALL SELECT 6062, 'SEC', 2610, 5300, 2, 2615, 5390, 3 FROM dual
UNION ALL SELECT 6063, 'SIC', 4308, 20, 1, 3274, 3082, 4 FROM dual
UNION ALL SELECT 6064, 'SIC', 3282, 3100, 2, 3289, 3143, 4 FROM dual
UNION ALL SELECT 6065, 'SIC', 3371, 3400, 2, 3397, 3463, 4 FROM dual
UNION ALL SELECT 6066, 'NAICS', 1851, 621, 2, 1892, 621991, 5 FROM dual
UNION ALL SELECT 6067, 'SIC', 4308, 20, 1, 3125, 2410, 3 FROM dual
UNION ALL SELECT 6068, 'NAICS', 2037, 81, 1, 2087, 81231, 4 FROM dual
UNION ALL SELECT 6069, 'NAICS', 2061, 8114, 3, 2066, 81142, 4 FROM dual
UNION ALL SELECT 6070, 'SIC', 2982, 2000, 2, 3014, 2062, 4 FROM dual
UNION ALL SELECT 6071, 'SIC', 2982, 2000, 2, 3004, 2045, 4 FROM dual
UNION ALL SELECT 6072, 'SEC', 2720, 7350, 3, 2721, 7359, 4 FROM dual
UNION ALL SELECT 6073, 'NAICS', 1726, 561, 2, 1763, 56159, 4 FROM dual
UNION ALL SELECT 6074, 'SIC', 3715, 5000, 2, 3737, 5050, 3 FROM dual
UNION ALL SELECT 6075, 'SIC', 3992, 6700, 2, 3995, 6719, 4 FROM dual
UNION ALL SELECT 6076, 'SIC', 4313, 70, 1, 4131, 7930, 3 FROM dual
UNION ALL SELECT 6077, 'SIC', 3124, 2400, 2, 3130, 2429, 4 FROM dual
UNION ALL SELECT 6078, 'SIC', 2982, 2000, 2, 3024, 2077, 4 FROM dual
UNION ALL SELECT 6079, 'SIC', 4308, 20, 1, 3534, 3721, 4 FROM dual
UNION ALL SELECT 6080, 'SIC', 4308, 20, 1, 3600, 3999, 4 FROM dual
UNION ALL SELECT 6081, 'NAICS', 132, 21, 1, 148, 212221, 5 FROM dual
UNION ALL SELECT 6082, 'NAICS', 1905, 6231, 3, 1906, 623110, 5 FROM dual
UNION ALL SELECT 6083, 'SEC', 2222, 1310, 3, 2223, 1311, 4 FROM dual
UNION ALL SELECT 6084, 'SIC', 3026, 2080, 3, 3032, 2087, 4 FROM dual
UNION ALL SELECT 6085, 'NAICS', 931, 423, 2, 977, 42361, 4 FROM dual
UNION ALL SELECT 6086, 'NAICS', 2135, 92, 1, 2208, 928120, 5 FROM dual
UNION ALL SELECT 6087, 'SIC', 3820, 5390, 3, 3821, 5399, 4 FROM dual
UNION ALL SELECT 6088, 'SIC', 2982, 2000, 2, 3008, 2050, 3 FROM dual
UNION ALL SELECT 6089, 'SIC', 4146, 8000, 2, 4168, 8072, 4 FROM dual
UNION ALL SELECT 6090, 'SIC', 4305, 1, 1, 2861, 760, 3 FROM dual
UNION ALL SELECT 6091, 'NAICS', 68, 1123, 3, 75, 112340, 5 FROM dual
UNION ALL SELECT 6092, 'SIC', 4308, 20, 1, 3021, 2074, 4 FROM dual
UNION ALL SELECT 6093, 'NAICS', 930, 42, 1, 1089, 425, 2 FROM dual
UNION ALL SELECT 6094, 'NAICS', 2071, 812, 2, 2096, 81292, 4 FROM dual
UNION ALL SELECT 6095, 'NAICS', 1689, 5418, 3, 1703, 54187, 4 FROM dual
UNION ALL SELECT 6096, 'NAICS', 931, 423, 2, 952, 42333, 4 FROM dual
UNION ALL SELECT 6097, 'SIC', 4308, 20, 1, 3590, 3953, 4 FROM dual
UNION ALL SELECT 6098, 'SIC', 4305, 1, 1, 2852, 722, 4 FROM dual
UNION ALL SELECT 6099, 'SIC', 3148, 2500, 2, 3155, 2519, 4 FROM dual
UNION ALL SELECT 6100, 'SIC', 4310, 50, 1, 3760, 5094, 4 FROM dual
UNION ALL SELECT 6101, 'SEC', 2488, 3840, 3, 2489, 3841, 4 FROM dual
UNION ALL SELECT 6102, 'SEC', 2434, 3600, 2, 2450, 3670, 3 FROM dual
UNION ALL SELECT 6103, 'SIC', 4306, 10, 1, 2909, 1300, 2 FROM dual
UNION ALL SELECT 6104, 'NAICS', 1480, 52, 1, 1530, 52392, 4 FROM dual
UNION ALL SELECT 6105, 'NAICS', 1506, 5223, 3, 1511, 522390, 5 FROM dual
UNION ALL SELECT 6106, 'NAICS', 1836, 6116, 3, 1842, 61163, 4 FROM dual
UNION ALL SELECT 6107, 'NAICS', 2048, 81119, 4, 2051, 811198, 5 FROM dual
UNION ALL SELECT 6108, 'SEC', 2292, 2600, 2, 2297, 2630, 3 FROM dual
UNION ALL SELECT 6109, 'SIC', 4311, 52, 1, 3902, 5949, 4 FROM dual
UNION ALL SELECT 6110, 'NAICS', 930, 42, 1, 976, 423610, 5 FROM dual
UNION ALL SELECT 6111, 'SIC', 4313, 70, 1, 4053, 7336, 4 FROM dual
UNION ALL SELECT 6112, 'NAICS', 2060, 81131, 4, 2059, 811310, 5 FROM dual
UNION ALL SELECT 6113, 'NAICS', 1625, 541, 2, 1666, 5415, 3 FROM dual
UNION ALL SELECT 6114, 'SIC', 4308, 20, 1, 3457, 3566, 4 FROM dual
UNION ALL SELECT 6115, 'NAICS', 930, 42, 1, 1043, 42444, 4 FROM dual
UNION ALL SELECT 6116, 'NAICS', 1, 11, 1, 82, 112420, 5 FROM dual
UNION ALL SELECT 6117, 'SEC', 2793, 50, 1, 2574, 5040, 3 FROM dual
UNION ALL SELECT 6118, 'SIC', 3395, 3460, 3, 3399, 3466, 4 FROM dual
UNION ALL SELECT 6119, 'SIC', 4308, 20, 1, 3368, 3390, 3 FROM dual
UNION ALL SELECT 6120, 'NAICS', 2093, 8129, 3, 2100, 81293, 4 FROM dual
UNION ALL SELECT 6121, 'NAICS', 2193, 92612, 4, 2192, 926120, 5 FROM dual
UNION ALL SELECT 6122, 'NAICS', 931, 423, 2, 942, 423210, 5 FROM dual
UNION ALL SELECT 6123, 'NAICS', 1569, 53, 1, 1603, 532230, 5 FROM dual
UNION ALL SELECT 6124, 'SEC', 2792, 40, 1, 2517, 4200, 2 FROM dual
UNION ALL SELECT 6125, 'NAICS', 1625, 541, 2, 1696, 541840, 5 FROM dual
UNION ALL SELECT 6126, 'SIC', 3883, 5810, 3, 3884, 5812, 4 FROM dual
UNION ALL SELECT 6127, 'NAICS', 1970, 7121, 3, 1977, 712190, 5 FROM dual
UNION ALL SELECT 6128, 'SEC', 2617, 5400, 2, 2619, 5411, 4 FROM dual
UNION ALL SELECT 6129, 'SIC', 4309, 40, 1, 3620, 4200, 2 FROM dual
UNION ALL SELECT 6130, 'SIC', 4308, 20, 1, 3257, 2952, 4 FROM dual
UNION ALL SELECT 6131, 'SEC', 2355, 3200, 2, 2362, 3240, 3 FROM dual
UNION ALL SELECT 6132, 'NAICS', 1718, 55, 1, 1721, 55111, 4 FROM dual
UNION ALL SELECT 6133, 'NAICS', 1792, 562, 2, 1801, 562212, 5 FROM dual
UNION ALL SELECT 6134, 'SIC', 3480, 3600, 2, 3519, 3679, 4 FROM dual
UNION ALL SELECT 6135, 'SIC', 3886, 5900, 2, 3888, 5912, 4 FROM dual
UNION ALL SELECT 6136, 'NAICS', 1725, 56, 1, 1788, 561920, 5 FROM dual
UNION ALL SELECT 6137, 'NAICS', 1591, 532, 2, 1615, 532412, 5 FROM dual
UNION ALL SELECT 6138, 'SIC', 3762, 5100, 2, 3781, 5147, 4 FROM dual
UNION ALL SELECT 6139, 'NAICS', 1913, 6233, 3, 1915, 623311, 5 FROM dual
UNION ALL SELECT 6140, 'NAICS', 2037, 81, 1, 2104, 8131, 3 FROM dual
UNION ALL SELECT 6141, 'NAICS', 1015, 424, 2, 1042, 424440, 5 FROM dual
UNION ALL SELECT 6142, 'NAICS', 1952, 71119, 4, 1951, 711190, 5 FROM dual
UNION ALL SELECT 6143, 'NAICS', 1672, 5416, 3, 1680, 54162, 4 FROM dual
UNION ALL SELECT 6144, 'NAICS', 1640, 5413, 3, 1653, 541370, 5 FROM dual
UNION ALL SELECT 6145, 'NAICS', 2120, 8139, 3, 2130, 81399, 4 FROM dual
UNION ALL SELECT 6146, 'NAICS', 1624, 54, 1, 1678, 541618, 5 FROM dual
UNION ALL SELECT 6147, 'SEC', 2687, 6360, 3, 2688, 6361, 4 FROM dual
UNION ALL SELECT 6148, 'SIC', 3977, 6500, 2, 3979, 6512, 4 FROM dual
UNION ALL SELECT 6149, 'SIC', 3371, 3400, 2, 3414, 3495, 4 FROM dual
UNION ALL SELECT 6150, 'NAICS', 1624, 54, 1, 1715, 54194, 4 FROM dual
UNION ALL SELECT 6151, 'NAICS', 1526, 5239, 3, 1533, 52399, 4 FROM dual
UNION ALL SELECT 6152, 'SIC', 4308, 20, 1, 3283, 3110, 3 FROM dual
UNION ALL SELECT 6153, 'SIC', 3480, 3600, 2, 3483, 3613, 4 FROM dual
UNION ALL SELECT 6154, 'SIC', 3214, 2800, 2, 3219, 2819, 4 FROM dual
UNION ALL SELECT 6155, 'SEC', 2792, 40, 1, 2534, 4600, 2 FROM dual
UNION ALL SELECT 6156, 'SIC', 3214, 2800, 2, 3228, 2835, 4 FROM dual
UNION ALL SELECT 6157, 'SEC', 2794, 52, 1, 2614, 5331, 4 FROM dual
UNION ALL SELECT 6158, 'NAICS', 1942, 71, 1, 1987, 71321, 4 FROM dual
UNION ALL SELECT 6159, 'SEC', 2476, 3800, 2, 2499, 3873, 4 FROM dual
UNION ALL SELECT 6160, 'SIC', 3917, 6000, 2, 3923, 6022, 4 FROM dual
UNION ALL SELECT 6161, 'NAICS', 36, 1114, 3, 40, 11142, 4 FROM dual
UNION ALL SELECT 6162, 'SEC', 2323, 2830, 3, 2325, 2834, 4 FROM dual
UNION ALL SELECT 6163, 'SIC', 4308, 20, 1, 3261, 3000, 2 FROM dual
UNION ALL SELECT 6164, 'SIC', 4058, 7350, 3, 4060, 7353, 4 FROM dual
UNION ALL SELECT 6165, 'NAICS', 1513, 523, 2, 1526, 5239, 3 FROM dual
UNION ALL SELECT 6166, 'NAICS', 2151, 9221, 3, 2155, 92212, 4 FROM dual
UNION ALL SELECT 6167, 'SIC', 4146, 8000, 2, 4152, 8031, 4 FROM dual
UNION ALL SELECT 6168, 'NAICS', 1523, 5232, 3, 1525, 52321, 4 FROM dual
UNION ALL SELECT 6169, 'NAICS', 1624, 54, 1, 1708, 54191, 4 FROM dual
UNION ALL SELECT 6170, 'NAICS', 1850, 62, 1, 1924, 624120, 5 FROM dual
UNION ALL SELECT 6171, 'NAICS', 2003, 72, 1, 2033, 722511, 5 FROM dual
UNION ALL SELECT 6172, 'SEC', 2355, 3200, 2, 2361, 3231, 4 FROM dual
UNION ALL SELECT 6173, 'SEC', 2526, 4500, 2, 2528, 4512, 4 FROM dual
UNION ALL SELECT 6174, 'NAICS', 260, 2383, 3, 263, 238320, 5 FROM dual
UNION ALL SELECT 6175, 'SEC', 2376, 3330, 3, 2377, 3334, 4 FROM dual
UNION ALL SELECT 6176, 'SEC', 2640, 5900, 2, 2647, 5961, 4 FROM dual
UNION ALL SELECT 6177, 'SEC', 2540, 4810, 3, 2542, 4813, 4 FROM dual
UNION ALL SELECT 6178, 'SIC', 4179, 8210, 3, 4180, 8211, 4 FROM dual
UNION ALL SELECT 6179, 'SEC', 2342, 3000, 2, 2348, 3060, 3 FROM dual
UNION ALL SELECT 6180, 'NAICS', 1569, 53, 1, 1617, 53242, 4 FROM dual
UNION ALL SELECT 6181, 'SIC', 3863, 5650, 3, 3864, 5651, 4 FROM dual
UNION ALL SELECT 6182, 'SEC', 2794, 52, 1, 2644, 5944, 4 FROM dual
UNION ALL SELECT 6183, 'SIC', 3073, 2280, 3, 3074, 2281, 4 FROM dual
UNION ALL SELECT 6184, 'SIC', 4308, 20, 1, 3426, 3530, 3 FROM dual
UNION ALL SELECT 6185, 'SIC', 3000, 2040, 3, 3006, 2047, 4 FROM dual
UNION ALL SELECT 6186, 'SIC', 4313, 70, 1, 4114, 7810, 3 FROM dual
UNION ALL SELECT 6187, 'SIC', 3511, 3670, 3, 3515, 3675, 4 FROM dual
UNION ALL SELECT 6188, 'SIC', 3337, 3300, 2, 3366, 3366, 4 FROM dual
UNION ALL SELECT 6189, 'SIC', 4308, 20, 1, 3307, 3230, 3 FROM dual
UNION ALL SELECT 6190, 'SIC', 4310, 50, 1, 3758, 5092, 4 FROM dual
UNION ALL SELECT 6191, 'NAICS', 1533, 52399, 4, 1535, 523999, 5 FROM dual
UNION ALL SELECT 6192, 'SIC', 2987, 2020, 3, 2989, 2022, 4 FROM dual
UNION ALL SELECT 6193, 'SEC', 2537, 4730, 3, 2538, 4731, 4 FROM dual
UNION ALL SELECT 6194, 'NAICS', 218, 237, 2, 222, 237120, 5 FROM dual
UNION ALL SELECT 6195, 'SEC', 2793, 50, 1, 2598, 5150, 3 FROM dual
UNION ALL SELECT 6196, 'SIC', 3715, 5000, 2, 3755, 5088, 4 FROM dual
UNION ALL SELECT 6197, 'NAICS', 1656, 54138, 4, 1655, 541380, 5 FROM dual
UNION ALL SELECT 6198, 'NAICS', 1859, 6213, 3, 1865, 62133, 4 FROM dual
UNION ALL SELECT 6199, 'NAICS', 1459, 5174, 3, 1461, 51741, 4 FROM dual
UNION ALL SELECT 6200, 'NAICS', 1591, 532, 2, 1610, 532310, 5 FROM dual
UNION ALL SELECT 6201, 'SEC', 2245, 2020, 3, 2246, 2024, 4 FROM dual
UNION ALL SELECT 6202, 'SEC', 2791, 20, 1, 2450, 3670, 3 FROM dual
UNION ALL SELECT 6203, 'SIC', 3214, 2800, 2, 3225, 2830, 3 FROM dual
UNION ALL SELECT 6204, 'NAICS', 1720, 5511, 3, 1723, 551112, 5 FROM dual
UNION ALL SELECT 6205, 'SIC', 4308, 20, 1, 3179, 2657, 4 FROM dual
UNION ALL SELECT 6206, 'SIC', 4309, 40, 1, 3629, 4225, 4 FROM dual
UNION ALL SELECT 6207, 'NAICS', 1683, 5417, 3, 1684, 54171, 4 FROM dual
UNION ALL SELECT 6208, 'SEC', 2796, 70, 1, 2755, 8011, 4 FROM dual
UNION ALL SELECT 6209, 'SIC', 4308, 20, 1, 3354, 3341, 4 FROM dual
UNION ALL SELECT 6210, 'NAICS', 930, 42, 1, 1071, 4248, 3 FROM dual
UNION ALL SELECT 6211, 'NAICS', 1635, 54121, 4, 1638, 541214, 5 FROM dual
UNION ALL SELECT 6212, 'NAICS', 1725, 56, 1, 1785, 5619, 3 FROM dual
UNION ALL SELECT 6213, 'SIC', 3419, 3500, 2, 3445, 3552, 4 FROM dual
UNION ALL SELECT 6214, 'SIC', 4308, 20, 1, 3320, 3264, 4 FROM dual
UNION ALL SELECT 6215, 'NAICS', 1475, 51912, 4, 1474, 519120, 5 FROM dual
UNION ALL SELECT 6216, 'SEC', 2791, 20, 1, 2280, 2450, 3 FROM dual
UNION ALL SELECT 6217, 'SIC', 4101, 7620, 3, 4104, 7629, 4 FROM dual
UNION ALL SELECT 6218, 'SEC', 2777, 8730, 3, 2779, 8734, 4 FROM dual
UNION ALL SELECT 6219, 'SIC', 4309, 40, 1, 3708, 4952, 4 FROM dual
UNION ALL SELECT 6220, 'SIC', 2884, 1000, 2, 2896, 1080, 3 FROM dual
UNION ALL SELECT 6221, 'SIC', 4308, 20, 1, 3570, 3850, 3 FROM dual
UNION ALL SELECT 6222, 'SIC', 4306, 10, 1, 2935, 1480, 3 FROM dual
UNION ALL SELECT 6223, 'NAICS', 991, 4238, 3, 1003, 42386, 4 FROM dual
UNION ALL SELECT 6224, 'NAICS', 930, 42, 1, 1064, 424690, 5 FROM dual
UNION ALL SELECT 6225, 'SIC', 4309, 40, 1, 3616, 4150, 3 FROM dual
UNION ALL SELECT 6226, 'NAICS', 1689, 5418, 3, 1696, 541840, 5 FROM dual
UNION ALL SELECT 6227, 'SEC', 2265, 2220, 3, 2266, 2221, 4 FROM dual
UNION ALL SELECT 6228, 'SEC', 2694, 6500, 2, 2702, 6552, 4 FROM dual
UNION ALL SELECT 6229, 'SIC', 3071, 2270, 3, 3072, 2273, 4 FROM dual
UNION ALL SELECT 6230, 'NAICS', 1813, 611, 2, 1817, 6112, 3 FROM dual
UNION ALL SELECT 6231, 'NAICS', 1, 11, 1, 40, 11142, 4 FROM dual
UNION ALL SELECT 6232, 'NAICS', 1725, 56, 1, 1786, 561910, 5 FROM dual
UNION ALL SELECT 6233, 'NAICS', 1725, 56, 1, 1796, 562112, 5 FROM dual
UNION ALL SELECT 6234, 'NAICS', 2002, 71399, 4, 2001, 713990, 5 FROM dual
UNION ALL SELECT 6235, 'SIC', 4313, 70, 1, 4201, 8390, 3 FROM dual
UNION ALL SELECT 6236, 'NAICS', 2071, 812, 2, 2072, 8121, 3 FROM dual
UNION ALL SELECT 6237, 'SIC', 4308, 20, 1, 3485, 3621, 4 FROM dual
UNION ALL SELECT 6238, 'NAICS', 2010, 72119, 4, 2011, 721191, 5 FROM dual
UNION ALL SELECT 6239, 'NAICS', 236, 2381, 3, 244, 23814, 4 FROM dual
UNION ALL SELECT 6240, 'SIC', 2948, 1600, 2, 2950, 1611, 4 FROM dual
UNION ALL SELECT 6241, 'SEC', 2292, 2600, 2, 2295, 2620, 3 FROM dual
UNION ALL SELECT 6242, 'SEC', 2791, 20, 1, 2395, 3448, 4 FROM dual
UNION ALL SELECT 6243, 'NAICS', 2151, 9221, 3, 2162, 922160, 5 FROM dual
UNION ALL SELECT 6244, 'NAICS', 1435, 51222, 4, 1434, 512220, 5 FROM dual
UNION ALL SELECT 6245, 'SEC', 2748, 7900, 2, 2751, 7990, 3 FROM dual
UNION ALL SELECT 6246, 'NAICS', 1404, 5111, 3, 1414, 511191, 5 FROM dual
UNION ALL SELECT 6247, 'NAICS', 2, 111, 2, 11, 11114, 4 FROM dual
UNION ALL SELECT 6248, 'NAICS', 235, 238, 2, 256, 238220, 5 FROM dual
UNION ALL SELECT 6249, 'NAICS', 991, 4238, 3, 994, 423820, 5 FROM dual
UNION ALL SELECT 6250, 'NAICS', 235, 238, 2, 250, 23817, 4 FROM dual
UNION ALL SELECT 6251, 'NAICS', 2166, 923, 2, 2175, 92314, 4 FROM dual
UNION ALL SELECT 6252, 'SIC', 4018, 7210, 3, 4021, 7213, 4 FROM dual
UNION ALL SELECT 6253, 'NAICS', 1480, 52, 1, 1545, 524130, 5 FROM dual
UNION ALL SELECT 6254, 'NAICS', 2037, 81, 1, 2080, 8122, 3 FROM dual
UNION ALL SELECT 6255, 'SIC', 3639, 4420, 3, 3640, 4424, 4 FROM dual
UNION ALL SELECT 6256, 'NAICS', 236, 2381, 3, 243, 238140, 5 FROM dual
UNION ALL SELECT 6257, 'SEC', 2221, 1300, 2, 2226, 1382, 4 FROM dual
UNION ALL SELECT 6258, 'NAICS', 1625, 541, 2, 1703, 54187, 4 FROM dual
UNION ALL SELECT 6259, 'NAICS', 1726, 561, 2, 1787, 56191, 4 FROM dual
UNION ALL SELECT 6260, 'NAICS', 2037, 81, 1, 2066, 81142, 4 FROM dual
UNION ALL SELECT 6261, 'SIC', 4192, 8300, 2, 4196, 8331, 4 FROM dual
UNION ALL SELECT 6262, 'SEC', 2605, 5200, 2, 2606, 5210, 3 FROM dual
UNION ALL SELECT 6263, 'NAICS', 1480, 52, 1, 1519, 523130, 5 FROM dual
UNION ALL SELECT 6264, 'NAICS', 2005, 7211, 3, 2008, 721120, 5 FROM dual
UNION ALL SELECT 6265, 'SIC', 3552, 3800, 2, 3573, 3861, 4 FROM dual
UNION ALL SELECT 6266, 'NAICS', 1015, 424, 2, 1056, 42451, 4 FROM dual
UNION ALL SELECT 6267, 'NAICS', 930, 42, 1, 1080, 42492, 4 FROM dual
UNION ALL SELECT 6268, 'SIC', 3480, 3600, 2, 3482, 3612, 4 FROM dual
UNION ALL SELECT 6269, 'SIC', 3978, 6510, 3, 3984, 6519, 4 FROM dual
UNION ALL SELECT 6270, 'NAICS', 1920, 624, 2, 1928, 6242, 3 FROM dual
UNION ALL SELECT 6271, 'NAICS', 2037, 81, 1, 2093, 8129, 3 FROM dual
UNION ALL SELECT 6272, 'NAICS', 1766, 5616, 3, 1770, 561613, 5 FROM dual
UNION ALL SELECT 6273, 'NAICS', 1942, 71, 1, 1959, 711310, 5 FROM dual
UNION ALL SELECT 6274, 'SIC', 4314, 90, 1, 4284, 9600, 2 FROM dual
UNION ALL SELECT 6275, 'SIC', 3050, 2200, 2, 3075, 2282, 4 FROM dual
UNION ALL SELECT 6276, 'SIC', 2813, 170, 3, 2819, 179, 4 FROM dual
UNION ALL SELECT 6277, 'SEC', 2500, 3900, 2, 2505, 3940, 3 FROM dual
UNION ALL SELECT 6278, 'SIC', 3419, 3500, 2, 3467, 3579, 4 FROM dual
UNION ALL SELECT 6279, 'SIC', 4313, 70, 1, 4109, 7690, 3 FROM dual
UNION ALL SELECT 6280, 'SIC', 2918, 1400, 2, 2931, 1470, 3 FROM dual
UNION ALL SELECT 6281, 'NAICS', 1813, 611, 2, 1843, 61169, 4 FROM dual
UNION ALL SELECT 6282, 'SIC', 4311, 52, 1, 3842, 5530, 3 FROM dual
UNION ALL SELECT 6283, 'NAICS', 1536, 524, 2, 1544, 524128, 5 FROM dual
UNION ALL SELECT 6284, 'NAICS', 205, 23, 1, 227, 237210, 5 FROM dual
UNION ALL SELECT 6285, 'SEC', 2791, 20, 1, 2417, 3559, 4 FROM dual
UNION ALL SELECT 6286, 'NAICS', 931, 423, 2, 955, 4234, 3 FROM dual
UNION ALL SELECT 6287, 'SEC', 2355, 3200, 2, 2356, 3210, 3 FROM dual
UNION ALL SELECT 6288, 'SIC', 4308, 20, 1, 3099, 2341, 4 FROM dual
UNION ALL SELECT 6289, 'SIC', 4308, 20, 1, 3514, 3674, 4 FROM dual
UNION ALL SELECT 6290, 'SIC', 4309, 40, 1, 3606, 4110, 3 FROM dual
UNION ALL SELECT 6291, 'SIC', 4313, 70, 1, 4210, 8611, 4 FROM dual
UNION ALL SELECT 6292, 'NAICS', 252, 23819, 4, 251, 238190, 5 FROM dual
UNION ALL SELECT 6293, 'SEC', 2791, 20, 1, 2318, 2790, 3 FROM dual
UNION ALL SELECT 6294, 'SEC', 2500, 3900, 2, 2503, 3930, 3 FROM dual
UNION ALL SELECT 6295, 'SIC', 4310, 50, 1, 3727, 5033, 4 FROM dual
UNION ALL SELECT 6296, 'NAICS', 1969, 712, 2, 1970, 7121, 3 FROM dual
UNION ALL SELECT 6297, 'SIC', 2982, 2000, 2, 3017, 2066, 4 FROM dual
UNION ALL SELECT 6298, 'SIC', 4040, 7300, 2, 4068, 7373, 4 FROM dual
UNION ALL SELECT 6299, 'NAICS', 1480, 52, 1, 1503, 522293, 5 FROM dual
UNION ALL SELECT 6300, 'NAICS', 108, 114, 2, 111, 114111, 5 FROM dual
UNION ALL SELECT 6301, 'SIC', 2798, 100, 2, 2817, 174, 4 FROM dual
UNION ALL SELECT 6302, 'NAICS', 1480, 52, 1, 1540, 524114, 5 FROM dual
UNION ALL SELECT 6303, 'NAICS', 1015, 424, 2, 1020, 42412, 4 FROM dual
UNION ALL SELECT 6304, 'SIC', 4311, 52, 1, 3846, 5550, 3 FROM dual
UNION ALL SELECT 6305, 'NAICS', 1569, 53, 1, 1598, 5322, 3 FROM dual
UNION ALL SELECT 6306, 'NAICS', 2150, 922, 2, 2154, 922120, 5 FROM dual
UNION ALL SELECT 6307, 'SIC', 2919, 1410, 3, 2920, 1411, 4 FROM dual
UNION ALL SELECT 6308, 'SIC', 4312, 60, 1, 3963, 6324, 4 FROM dual
UNION ALL SELECT 6309, 'SIC', 4313, 70, 1, 4149, 8020, 3 FROM dual
UNION ALL SELECT 6310, 'SIC', 2868, 800, 2, 2869, 810, 3 FROM dual
UNION ALL SELECT 6311, 'SEC', 2753, 8000, 2, 2763, 8082, 4 FROM dual
UNION ALL SELECT 6312, 'NAICS', 2021, 7223, 3, 2026, 722330, 5 FROM dual
UNION ALL SELECT 6313, 'NAICS', 2135, 92, 1, 2146, 921150, 5 FROM dual
UNION ALL SELECT 6314, 'SIC', 3337, 3300, 2, 3354, 3341, 4 FROM dual
UNION ALL SELECT 6315, 'SIC', 4313, 70, 1, 4237, 8742, 4 FROM dual
UNION ALL SELECT 6316, 'NAICS', 1817, 6112, 3, 1818, 611210, 5 FROM dual
UNION ALL SELECT 6317, 'NAICS', 79, 1124, 3, 82, 112420, 5 FROM dual
UNION ALL SELECT 6318, 'SIC', 4305, 1, 1, 2838, 254, 4 FROM dual
UNION ALL SELECT 6319, 'NAICS', 1035, 4244, 3, 1041, 42443, 4 FROM dual
UNION ALL SELECT 6320, 'NAICS', 2021, 7223, 3, 2024, 722320, 5 FROM dual
UNION ALL SELECT 6321, 'SIC', 3086, 2320, 3, 3088, 2322, 4 FROM dual
UNION ALL SELECT 6322, 'NAICS', 138, 212, 2, 139, 2121, 3 FROM dual
UNION ALL SELECT 6323, 'NAICS', 2028, 7224, 3, 2029, 722410, 5 FROM dual
UNION ALL SELECT 6324, 'SEC', 2791, 20, 1, 2342, 3000, 2 FROM dual
UNION ALL SELECT 6325, 'SIC', 4256, 9200, 2, 4261, 9222, 4 FROM dual
UNION ALL SELECT 6326, 'SIC', 3190, 2700, 2, 3206, 2770, 3 FROM dual
UNION ALL SELECT 6327, 'NAICS', 1569, 53, 1, 1584, 53131, 4 FROM dual
UNION ALL SELECT 6328, 'NAICS', 1004, 4239, 3, 1014, 42399, 4 FROM dual
UNION ALL SELECT 6329, 'NAICS', 1443, 5151, 3, 1445, 515111, 5 FROM dual
UNION ALL SELECT 6330, 'NAICS', 1812, 61, 1, 1820, 6113, 3 FROM dual
UNION ALL SELECT 6331, 'SIC', 4308, 20, 1, 3538, 3731, 4 FROM dual
UNION ALL SELECT 6332, 'SIC', 3468, 3580, 3, 3473, 3589, 4 FROM dual
UNION ALL SELECT 6333, 'SIC', 4223, 8700, 2, 4224, 8710, 3 FROM dual
UNION ALL SELECT 6334, 'SIC', 3805, 5210, 3, 3806, 5211, 4 FROM dual
UNION ALL SELECT 6335, 'NAICS', 1015, 424, 2, 1036, 424410, 5 FROM dual
UNION ALL SELECT 6336, 'NAICS', 941, 4232, 3, 942, 423210, 5 FROM dual
UNION ALL SELECT 6337, 'NAICS', 1624, 54, 1, 1703, 54187, 4 FROM dual
UNION ALL SELECT 6338, 'SIC', 4308, 20, 1, 3366, 3366, 4 FROM dual
UNION ALL SELECT 6339, 'SIC', 3693, 4900, 2, 3701, 4930, 3 FROM dual
UNION ALL SELECT 6340, 'NAICS', 2003, 72, 1, 2030, 72241, 4 FROM dual
UNION ALL SELECT 6341, 'SIC', 2975, 1790, 3, 2981, 1799, 4 FROM dual
UNION ALL SELECT 6342, 'NAICS', 1625, 541, 2, 1708, 54191, 4 FROM dual
UNION ALL SELECT 6343, 'SEC', 2791, 20, 1, 2295, 2620, 3 FROM dual
UNION ALL SELECT 6344, 'SIC', 4040, 7300, 2, 4065, 7370, 3 FROM dual
UNION ALL SELECT 6345, 'NAICS', 1798, 5622, 3, 1802, 562213, 5 FROM dual
UNION ALL SELECT 6346, 'SIC', 4308, 20, 1, 3252, 2900, 2 FROM dual
UNION ALL SELECT 6347, 'NAICS', 1569, 53, 1, 1573, 53111, 4 FROM dual
UNION ALL SELECT 6348, 'NAICS', 931, 423, 2, 953, 423390, 5 FROM dual
UNION ALL SELECT 6349, 'SIC', 4171, 8090, 3, 4172, 8092, 4 FROM dual
UNION ALL SELECT 6350, 'NAICS', 132, 21, 1, 178, 213114, 5 FROM dual
UNION ALL SELECT 6351, 'SIC', 4308, 20, 1, 3090, 2325, 4 FROM dual
UNION ALL SELECT 6352, 'NAICS', 931, 423, 2, 962, 423440, 5 FROM dual
UNION ALL SELECT 6353, 'NAICS', 2037, 81, 1, 2046, 811121, 5 FROM dual
UNION ALL SELECT 6354, 'NAICS', 930, 42, 1, 943, 42321, 4 FROM dual
UNION ALL SELECT 6355, 'SIC', 4309, 40, 1, 3604, 4013, 4 FROM dual
UNION ALL SELECT 6356, 'NAICS', 1859, 6213, 3, 1866, 621340, 5 FROM dual
UNION ALL SELECT 6357, 'SIC', 4308, 20, 1, 3246, 2890, 3 FROM dual
UNION ALL SELECT 6358, 'NAICS', 1625, 541, 2, 1715, 54194, 4 FROM dual
UNION ALL SELECT 6359, 'NAICS', 1402, 51, 1, 1431, 5122, 3 FROM dual
UNION ALL SELECT 6360, 'SIC', 4308, 20, 1, 3281, 3089, 4 FROM dual
UNION ALL SELECT 6361, 'SEC', 2403, 3500, 2, 2413, 3540, 3 FROM dual
UNION ALL SELECT 6362, 'NAICS', 1015, 424, 2, 1077, 424910, 5 FROM dual
UNION ALL SELECT 6363, 'NAICS', 181, 221, 2, 196, 221210, 5 FROM dual
UNION ALL SELECT 6364, 'NAICS', 1836, 6116, 3, 1837, 611610, 5 FROM dual
UNION ALL SELECT 6365, 'SIC', 4081, 7500, 2, 4084, 7514, 4 FROM dual
UNION ALL SELECT 6366, 'SIC', 4125, 7900, 2, 4142, 7993, 4 FROM dual
UNION ALL SELECT 6367, 'SEC', 2552, 4900, 2, 2567, 4961, 4 FROM dual
UNION ALL SELECT 6368, 'NAICS', 930, 42, 1, 982, 4237, 3 FROM dual
UNION ALL SELECT 6369, 'SEC', 2793, 50, 1, 2581, 5064, 4 FROM dual
UNION ALL SELECT 6370, 'SIC', 3311, 3250, 3, 3314, 3255, 4 FROM dual
UNION ALL SELECT 6371, 'SIC', 4244, 8900, 2, 4246, 8999, 4 FROM dual
UNION ALL SELECT 6372, 'SEC', 2440, 3630, 3, 2441, 3634, 4 FROM dual
UNION ALL SELECT 6373, 'SIC', 3822, 5400, 2, 3833, 5460, 3 FROM dual
UNION ALL SELECT 6374, 'SEC', 2791, 20, 1, 2287, 2522, 4 FROM dual
UNION ALL SELECT 6375, 'NAICS', 1891, 62199, 4, 1892, 621991, 5 FROM dual
UNION ALL SELECT 6376, 'NAICS', 2052, 8112, 3, 2057, 811219, 5 FROM dual
UNION ALL SELECT 6377, 'SIC', 4308, 20, 1, 3114, 2389, 4 FROM dual
UNION ALL SELECT 6378, 'NAICS', 1812, 61, 1, 1828, 611430, 5 FROM dual
UNION ALL SELECT 6379, 'SIC', 4307, 15, 1, 2955, 1700, 2 FROM dual
UNION ALL SELECT 6380, 'SIC', 4313, 70, 1, 4239, 8744, 4 FROM dual
UNION ALL SELECT 6381, 'NAICS', 1758, 5615, 3, 1762, 56152, 4 FROM dual
UNION ALL SELECT 6382, 'SIC', 2983, 2010, 3, 2985, 2013, 4 FROM dual
UNION ALL SELECT 6383, 'SEC', 2585, 5080, 3, 2587, 5084, 4 FROM dual
UNION ALL SELECT 6384, 'SIC', 3527, 3710, 3, 3530, 3714, 4 FROM dual
UNION ALL SELECT 6385, 'SEC', 2319, 2800, 2, 2322, 2821, 4 FROM dual
UNION ALL SELECT 6386, 'NAICS', 1793, 5621, 3, 1794, 56211, 4 FROM dual
UNION ALL SELECT 6387, 'SIC', 3307, 3230, 3, 3308, 3231, 4 FROM dual
UNION ALL SELECT 6388, 'NAICS', 2037, 81, 1, 2094, 812910, 5 FROM dual
UNION ALL SELECT 6389, 'SIC', 4308, 20, 1, 3206, 2770, 3 FROM dual
UNION ALL SELECT 6390, 'SIC', 4313, 70, 1, 4012, 7030, 3 FROM dual
UNION ALL SELECT 6391, 'SEC', 2328, 2840, 3, 2330, 2844, 4 FROM dual
UNION ALL SELECT 6392, 'SIC', 3371, 3400, 2, 3395, 3460, 3 FROM dual
UNION ALL SELECT 6393, 'NAICS', 2189, 9261, 3, 2190, 926110, 5 FROM dual
UNION ALL SELECT 6394, 'SIC', 4311, 52, 1, 3838, 5510, 3 FROM dual
UNION ALL SELECT 6395, 'SIC', 3882, 5800, 2, 3884, 5812, 4 FROM dual
UNION ALL SELECT 6396, 'NAICS', 930, 42, 1, 1034, 42434, 4 FROM dual
UNION ALL SELECT 6397, 'NAICS', 1726, 561, 2, 1728, 561110, 5 FROM dual
UNION ALL SELECT 6398, 'NAICS', 2136, 921, 2, 2140, 921120, 5 FROM dual
UNION ALL SELECT 6399, 'SEC', 2791, 20, 1, 2451, 3672, 4 FROM dual
UNION ALL SELECT 6400, 'SEC', 2552, 4900, 2, 2555, 4920, 3 FROM dual
UNION ALL SELECT 6401, 'SIC', 4311, 52, 1, 3834, 5461, 4 FROM dual
UNION ALL SELECT 6402, 'NAICS', 1942, 71, 1, 1956, 711212, 5 FROM dual
UNION ALL SELECT 6403, 'SIC', 2834, 250, 3, 2835, 251, 4 FROM dual
UNION ALL SELECT 6404, 'NAICS', 1682, 54169, 4, 1681, 541690, 5 FROM dual
UNION ALL SELECT 6405, 'NAICS', 2038, 811, 2, 2064, 811412, 5 FROM dual
UNION ALL SELECT 6406, 'SIC', 4306, 10, 1, 2938, 1499, 4 FROM dual
UNION ALL SELECT 6407, 'SIC', 4133, 7940, 3, 4134, 7941, 4 FROM dual
UNION ALL SELECT 6408, 'NAICS', 108, 114, 2, 115, 114210, 5 FROM dual
UNION ALL SELECT 6409, 'NAICS', 1480, 52, 1, 1514, 5231, 3 FROM dual
UNION ALL SELECT 6410, 'NAICS', 1, 11, 1, 79, 1124, 3 FROM dual
UNION ALL SELECT 6411, 'SIC', 4223, 8700, 2, 4226, 8712, 4 FROM dual
UNION ALL SELECT 6412, 'SEC', 2403, 3500, 2, 2417, 3559, 4 FROM dual
UNION ALL SELECT 6413, 'NAICS', 1554, 525, 2, 1566, 52592, 4 FROM dual
UNION ALL SELECT 6414, 'SEC', 2713, 7300, 2, 2715, 7311, 4 FROM dual
UNION ALL SELECT 6415, 'SEC', 2795, 60, 1, 2656, 6036, 4 FROM dual
UNION ALL SELECT 6416, 'SIC', 3480, 3600, 2, 3486, 3624, 4 FROM dual
UNION ALL SELECT 6417, 'NAICS', 273, 2389, 3, 277, 23899, 4 FROM dual
UNION ALL SELECT 6418, 'SIC', 3511, 3670, 3, 3514, 3674, 4 FROM dual
UNION ALL SELECT 6419, 'NAICS', 1485, 522, 2, 1500, 52229, 4 FROM dual
UNION ALL SELECT 6420, 'NAICS', 1648, 54134, 4, 1647, 541340, 5 FROM dual
UNION ALL SELECT 6421, 'NAICS', 1591, 532, 2, 1599, 532210, 5 FROM dual
UNION ALL SELECT 6422, 'SIC', 3583, 3940, 3, 3584, 3942, 4 FROM dual
UNION ALL SELECT 6423, 'NAICS', 157, 21231, 4, 160, 212313, 5 FROM dual
UNION ALL SELECT 6424, 'NAICS', 1942, 71, 1, 1983, 713120, 5 FROM dual
UNION ALL SELECT 6425, 'SIC', 4305, 1, 1, 2816, 173, 4 FROM dual
UNION ALL SELECT 6426, 'SIC', 2813, 170, 3, 2818, 175, 4 FROM dual
UNION ALL SELECT 6427, 'NAICS', 1920, 624, 2, 1922, 624110, 5 FROM dual
UNION ALL SELECT 6428, 'SIC', 4259, 9220, 3, 4261, 9222, 4 FROM dual
UNION ALL SELECT 6429, 'NAICS', 948, 42331, 4, 947, 423310, 5 FROM dual
UNION ALL SELECT 6430, 'NAICS', 1920, 624, 2, 1933, 624229, 5 FROM dual
UNION ALL SELECT 6431, 'NAICS', 1598, 5322, 3, 1606, 532291, 5 FROM dual
UNION ALL SELECT 6432, 'NAICS', 109, 1141, 3, 113, 114119, 5 FROM dual
UNION ALL SELECT 6433, 'SIC', 2955, 1700, 2, 2971, 1770, 3 FROM dual
UNION ALL SELECT 6434, 'SIC', 3148, 2500, 2, 3160, 2531, 4 FROM dual
UNION ALL SELECT 6435, 'SIC', 4017, 7200, 2, 4037, 7290, 3 FROM dual
UNION ALL SELECT 6436, 'SIC', 3008, 2050, 3, 3011, 2053, 4 FROM dual
UNION ALL SELECT 6437, 'SEC', 2434, 3600, 2, 2451, 3672, 4 FROM dual
UNION ALL SELECT 6438, 'NAICS', 1726, 561, 2, 1729, 56111, 4 FROM dual
UNION ALL SELECT 6439, 'NAICS', 218, 237, 2, 232, 2379, 3 FROM dual
UNION ALL SELECT 6440, 'NAICS', 1569, 53, 1, 1592, 5321, 3 FROM dual
UNION ALL SELECT 6441, 'NAICS', 2003, 72, 1, 2017, 7213, 3 FROM dual
UNION ALL SELECT 6442, 'NAICS', 2135, 92, 1, 2171, 92312, 4 FROM dual
UNION ALL SELECT 6443, 'SIC', 4311, 52, 1, 3875, 5720, 3 FROM dual
UNION ALL SELECT 6444, 'SEC', 4336, 99, 1, 4339, 9995, 2 FROM dual
UNION ALL SELECT 6445, 'NAICS', 1, 11, 1, 14, 111160, 5 FROM dual
UNION ALL SELECT 6446, 'SEC', 2794, 52, 1, 2641, 5910, 3 FROM dual
UNION ALL SELECT 6447, 'SIC', 4308, 20, 1, 3299, 3190, 3 FROM dual
UNION ALL SELECT 6448, 'SIC', 2825, 200, 2, 2833, 241, 4 FROM dual
UNION ALL SELECT 6449, 'NAICS', 108, 114, 2, 114, 1142, 3 FROM dual
UNION ALL SELECT 6450, 'SIC', 3419, 3500, 2, 3440, 3546, 4 FROM dual
UNION ALL SELECT 6451, 'NAICS', 2037, 81, 1, 2129, 813990, 5 FROM dual
UNION ALL SELECT 6452, 'SIC', 3083, 2300, 2, 3097, 2339, 4 FROM dual
UNION ALL SELECT 6453, 'SIC', 3605, 4100, 2, 3614, 4141, 4 FROM dual
UNION ALL SELECT 6454, 'NAICS', 132, 21, 1, 163, 212321, 5 FROM dual
UNION ALL SELECT 6455, 'NAICS', 1625, 541, 2, 1678, 541618, 5 FROM dual
UNION ALL SELECT 6456, 'NAICS', 135, 21111, 4, 136, 211111, 5 FROM dual
UNION ALL SELECT 6457, 'NAICS', 28, 11133, 4, 29, 111331, 5 FROM dual
UNION ALL SELECT 6458, 'NAICS', 1480, 52, 1, 1558, 525120, 5 FROM dual
UNION ALL SELECT 6459, 'SIC', 3774, 5140, 3, 3778, 5144, 4 FROM dual
UNION ALL SELECT 6460, 'SEC', 2791, 20, 1, 2495, 3851, 4 FROM dual
UNION ALL SELECT 6461, 'SIC', 3992, 6700, 2, 4000, 6732, 4 FROM dual
UNION ALL SELECT 6462, 'NAICS', 57, 1121, 3, 64, 11213, 4 FROM dual
UNION ALL SELECT 6463, 'NAICS', 991, 4238, 3, 1002, 423860, 5 FROM dual
UNION ALL SELECT 6464, 'SEC', 2794, 52, 1, 2618, 5410, 3 FROM dual
UNION ALL SELECT 6465, 'SIC', 4308, 20, 1, 3168, 2610, 3 FROM dual
UNION ALL SELECT 6466, 'SEC', 2791, 20, 1, 2357, 3211, 4 FROM dual
UNION ALL SELECT 6467, 'SIC', 4169, 8080, 3, 4170, 8082, 4 FROM dual
UNION ALL SELECT 6468, 'NAICS', 1537, 5241, 3, 1539, 524113, 5 FROM dual
UNION ALL SELECT 6469, 'NAICS', 2150, 922, 2, 2153, 92211, 4 FROM dual
UNION ALL SELECT 6470, 'SIC', 4284, 9600, 2, 4289, 9630, 3 FROM dual
UNION ALL SELECT 6471, 'SIC', 4312, 60, 1, 3950, 6211, 4 FROM dual
UNION ALL SELECT 6472, 'NAICS', 1979, 713, 2, 1997, 713940, 5 FROM dual
UNION ALL SELECT 6473, 'NAICS', 1480, 52, 1, 1515, 523110, 5 FROM dual
UNION ALL SELECT 6474, 'NAICS', 1624, 54, 1, 1666, 5415, 3 FROM dual
UNION ALL SELECT 6475, 'NAICS', 132, 21, 1, 145, 212210, 5 FROM dual
UNION ALL SELECT 6476, 'SIC', 4307, 15, 1, 2967, 1751, 4 FROM dual
UNION ALL SELECT 6477, 'SIC', 2962, 1740, 3, 2964, 1742, 4 FROM dual
UNION ALL SELECT 6478, 'SIC', 4310, 50, 1, 3791, 5170, 3 FROM dual
UNION ALL SELECT 6479, 'NAICS', 1624, 54, 1, 1696, 541840, 5 FROM dual
UNION ALL SELECT 6480, 'SIC', 3434, 3540, 3, 3435, 3541, 4 FROM dual
UNION ALL SELECT 6481, 'NAICS', 2037, 81, 1, 2073, 81211, 4 FROM dual
UNION ALL SELECT 6482, 'NAICS', 1480, 52, 1, 1523, 5232, 3 FROM dual
UNION ALL SELECT 6483, 'SIC', 4312, 60, 1, 3970, 6370, 3 FROM dual
UNION ALL SELECT 6484, 'SIC', 4312, 60, 1, 3969, 6361, 4 FROM dual
UNION ALL SELECT 6485, 'NAICS', 56, 112, 2, 77, 112390, 5 FROM dual
UNION ALL SELECT 6486, 'SEC', 2791, 20, 1, 2368, 3280, 3 FROM dual
UNION ALL SELECT 6487, 'NAICS', 1015, 424, 2, 1025, 42421, 4 FROM dual
UNION ALL SELECT 6488, 'NAICS', 1, 11, 1, 76, 11234, 4 FROM dual
UNION ALL SELECT 6489, 'SIC', 3602, 4010, 3, 3604, 4013, 4 FROM dual
UNION ALL SELECT 6490, 'NAICS', 218, 237, 2, 221, 23711, 4 FROM dual
UNION ALL SELECT 6491, 'SEC', 2791, 20, 1, 2297, 2630, 3 FROM dual
UNION ALL SELECT 6492, 'SEC', 2337, 2900, 2, 2340, 2950, 3 FROM dual
UNION ALL SELECT 6493, 'SIC', 3680, 4800, 2, 3692, 4899, 4 FROM dual
UNION ALL SELECT 6494, 'NAICS', 1953, 7112, 3, 1956, 711212, 5 FROM dual
UNION ALL SELECT 6495, 'SIC', 4308, 20, 1, 3595, 3990, 3 FROM dual
UNION ALL SELECT 6496, 'SIC', 3762, 5100, 2, 3797, 5190, 3 FROM dual
UNION ALL SELECT 6497, 'SEC', 2739, 7810, 3, 2741, 7819, 4 FROM dual
UNION ALL SELECT 6498, 'NAICS', 1980, 7131, 3, 1981, 713110, 5 FROM dual
UNION ALL SELECT 6499, 'NAICS', 1725, 56, 1, 1800, 562211, 5 FROM dual
UNION ALL SELECT 6500, 'SIC', 3587, 3950, 3, 3589, 3952, 4 FROM dual
UNION ALL SELECT 6501, 'SEC', 2790, 15, 1, 2233, 1540, 3 FROM dual
UNION ALL SELECT 6502, 'SEC', 2342, 3000, 2, 2351, 3086, 4 FROM dual
UNION ALL SELECT 6503, 'SEC', 2240, 2000, 2, 2252, 2060, 3 FROM dual
UNION ALL SELECT 6504, 'SEC', 2791, 20, 1, 2500, 3900, 2 FROM dual
UNION ALL SELECT 6505, 'SEC', 2371, 3300, 2, 2382, 3360, 3 FROM dual
UNION ALL SELECT 6506, 'SIC', 3621, 4210, 3, 3624, 4214, 4 FROM dual
UNION ALL SELECT 6507, 'SEC', 2725, 7370, 3, 2727, 7372, 4 FROM dual
UNION ALL SELECT 6508, 'SEC', 2791, 20, 1, 2413, 3540, 3 FROM dual
UNION ALL SELECT 6509, 'NAICS', 43, 1119, 3, 46, 111920, 5 FROM dual
UNION ALL SELECT 6510, 'SIC', 4308, 20, 1, 3134, 2435, 4 FROM dual
UNION ALL SELECT 6511, 'SIC', 3033, 2090, 3, 3036, 2095, 4 FROM dual
UNION ALL SELECT 6512, 'SEC', 2620, 5500, 2, 2621, 5530, 3 FROM dual
UNION ALL SELECT 6513, 'SIC', 4223, 8700, 2, 4227, 8713, 4 FROM dual
UNION ALL SELECT 6514, 'NAICS', 2, 111, 2, 48, 111930, 5 FROM dual
UNION ALL SELECT 6515, 'SEC', 2640, 5900, 2, 2642, 5912, 4 FROM dual
UNION ALL SELECT 6516, 'SIC', 4309, 40, 1, 3700, 4925, 4 FROM dual
UNION ALL SELECT 6517, 'SEC', 2302, 2700, 2, 2309, 2732, 4 FROM dual
UNION ALL SELECT 6518, 'SIC', 3419, 3500, 2, 3438, 3544, 4 FROM dual
UNION ALL SELECT 6519, 'SIC', 4310, 50, 1, 3736, 5049, 4 FROM dual
UNION ALL SELECT 6520, 'SIC', 4313, 70, 1, 4115, 7812, 4 FROM dual
UNION ALL SELECT 6521, 'SIC', 4306, 10, 1, 2889, 1030, 3 FROM dual
UNION ALL SELECT 6522, 'SIC', 4308, 20, 1, 3576, 3900, 2 FROM dual
UNION ALL SELECT 6523, 'SIC', 4308, 20, 1, 3187, 2677, 4 FROM dual
UNION ALL SELECT 6524, 'SIC', 4211, 8620, 3, 4212, 8621, 4 FROM dual
UNION ALL SELECT 6525, 'NAICS', 229, 2373, 3, 231, 23731, 4 FROM dual
UNION ALL SELECT 6526, 'SIC', 3301, 3200, 2, 3325, 3273, 4 FROM dual
UNION ALL SELECT 6527, 'SEC', 2791, 20, 1, 2380, 3350, 3 FROM dual
UNION ALL SELECT 6528, 'SIC', 4307, 15, 1, 2953, 1623, 4 FROM dual
UNION ALL SELECT 6529, 'SIC', 3774, 5140, 3, 3783, 5149, 4 FROM dual
UNION ALL SELECT 6530, 'NAICS', 930, 42, 1, 1055, 424510, 5 FROM dual
UNION ALL SELECT 6531, 'NAICS', 260, 2383, 3, 265, 238330, 5 FROM dual
UNION ALL SELECT 6532, 'NAICS', 180, 22, 1, 185, 221112, 5 FROM dual
UNION ALL SELECT 6533, 'SIC', 3451, 3560, 3, 3456, 3565, 4 FROM dual
UNION ALL SELECT 6534, 'NAICS', 1732, 56121, 4, 1731, 561210, 5 FROM dual
UNION ALL SELECT 6535, 'SIC', 4123, 7840, 3, 4124, 7841, 4 FROM dual
UNION ALL SELECT 6536, 'SIC', 3337, 3300, 2, 3368, 3390, 3 FROM dual
UNION ALL SELECT 6537, 'NAICS', 2037, 81, 1, 2102, 81299, 4 FROM dual
UNION ALL SELECT 6538, 'NAICS', 2151, 9221, 3, 2152, 922110, 5 FROM dual
UNION ALL SELECT 6539, 'SEC', 2241, 2010, 3, 2242, 2011, 4 FROM dual
UNION ALL SELECT 6540, 'SIC', 3301, 3200, 2, 3321, 3269, 4 FROM dual
UNION ALL SELECT 6541, 'NAICS', 2176, 924, 2, 2181, 92412, 4 FROM dual
UNION ALL SELECT 6542, 'SIC', 2960, 1730, 3, 2961, 1731, 4 FROM dual
UNION ALL SELECT 6543, 'NAICS', 1948, 71112, 4, 1947, 711120, 5 FROM dual
UNION ALL SELECT 6544, 'NAICS', 1942, 71, 1, 1965, 71141, 4 FROM dual
UNION ALL SELECT 6545, 'SIC', 4313, 70, 1, 4072, 7377, 4 FROM dual
UNION ALL SELECT 6546, 'NAICS', 1536, 524, 2, 1550, 52429, 4 FROM dual
UNION ALL SELECT 6547, 'NAICS', 1570, 531, 2, 1577, 53113, 4 FROM dual
UNION ALL SELECT 6548, 'SIC', 3148, 2500, 2, 3152, 2514, 4 FROM dual
UNION ALL SELECT 6549, 'SIC', 3715, 5000, 2, 3729, 5040, 3 FROM dual
UNION ALL SELECT 6550, 'SIC', 3337, 3300, 2, 3363, 3363, 4 FROM dual
UNION ALL SELECT 6551, 'NAICS', 2061, 8114, 3, 2067, 811430, 5 FROM dual
UNION ALL SELECT 6552, 'SIC', 4308, 20, 1, 3414, 3495, 4 FROM dual
UNION ALL SELECT 6553, 'SIC', 4308, 20, 1, 3493, 3634, 4 FROM dual
UNION ALL SELECT 6554, 'NAICS', 56, 112, 2, 64, 11213, 4 FROM dual
UNION ALL SELECT 6555, 'NAICS', 1477, 51913, 4, 1476, 519130, 5 FROM dual
UNION ALL SELECT 6556, 'SIC', 2880, 920, 3, 2881, 921, 4 FROM dual
UNION ALL SELECT 6557, 'NAICS', 1799, 56221, 4, 1802, 562213, 5 FROM dual
UNION ALL SELECT 6558, 'SEC', 2640, 5900, 2, 2646, 5960, 3 FROM dual
UNION ALL SELECT 6559, 'NAICS', 2037, 81, 1, 2042, 811112, 5 FROM dual
UNION ALL SELECT 6560, 'NAICS', 1693, 54182, 4, 1692, 541820, 5 FROM dual
UNION ALL SELECT 6561, 'NAICS', 1808, 56292, 4, 1807, 562920, 5 FROM dual
UNION ALL SELECT 6562, 'SIC', 3564, 3840, 3, 3566, 3842, 4 FROM dual
UNION ALL SELECT 6563, 'NAICS', 28, 11133, 4, 30, 111332, 5 FROM dual
UNION ALL SELECT 6564, 'SEC', 2355, 3200, 2, 2367, 3272, 4 FROM dual
UNION ALL SELECT 6565, 'NAICS', 1891, 62199, 4, 1893, 621999, 5 FROM dual
UNION ALL SELECT 6566, 'SIC', 3067, 2260, 3, 3070, 2269, 4 FROM dual
UNION ALL SELECT 6567, 'SIC', 4308, 20, 1, 3105, 2369, 4 FROM dual
UNION ALL SELECT 6568, 'NAICS', 56, 112, 2, 57, 1121, 3 FROM dual
UNION ALL SELECT 6569, 'NAICS', 1867, 62134, 4, 1866, 621340, 5 FROM dual
UNION ALL SELECT 6570, 'NAICS', 1936, 6243, 3, 1937, 624310, 5 FROM dual
UNION ALL SELECT 6571, 'SEC', 2694, 6500, 2, 4326, 6532, 3 FROM dual
UNION ALL SELECT 6572, 'SIC', 4308, 20, 1, 3064, 2257, 4 FROM dual
UNION ALL SELECT 6573, 'SIC', 4310, 50, 1, 3743, 5065, 4 FROM dual
UNION ALL SELECT 6574, 'NAICS', 1075, 42482, 4, 1074, 424820, 5 FROM dual
UNION ALL SELECT 6575, 'SIC', 3282, 3100, 2, 3284, 3111, 4 FROM dual
UNION ALL SELECT 6576, 'SIC', 2982, 2000, 2, 3023, 2076, 4 FROM dual
UNION ALL SELECT 6577, 'SIC', 4309, 40, 1, 3612, 4131, 4 FROM dual
UNION ALL SELECT 6578, 'NAICS', 235, 238, 2, 249, 238170, 5 FROM dual
UNION ALL SELECT 6579, 'NAICS', 1624, 54, 1, 1693, 54182, 4 FROM dual
UNION ALL SELECT 6580, 'SEC', 2630, 5700, 2, 2633, 5730, 3 FROM dual
UNION ALL SELECT 6581, 'NAICS', 1726, 561, 2, 1776, 56171, 4 FROM dual
UNION ALL SELECT 6582, 'SEC', 2791, 20, 1, 2256, 2086, 4 FROM dual
UNION ALL SELECT 6583, 'SIC', 3419, 3500, 2, 3455, 3564, 4 FROM dual
UNION ALL SELECT 6584, 'NAICS', 132, 21, 1, 151, 212231, 5 FROM dual
UNION ALL SELECT 6585, 'NAICS', 1726, 561, 2, 1733, 5613, 3 FROM dual
UNION ALL SELECT 6586, 'SIC', 4178, 8200, 2, 4189, 8249, 4 FROM dual
UNION ALL SELECT 6587, 'SIC', 4309, 40, 1, 3652, 4493, 4 FROM dual
UNION ALL SELECT 6588, 'SIC', 4008, 7010, 3, 4009, 7011, 4 FROM dual
UNION ALL SELECT 6589, 'NAICS', 1485, 522, 2, 1489, 522120, 5 FROM dual
UNION ALL SELECT 6590, 'NAICS', 2030, 72241, 4, 2029, 722410, 5 FROM dual
UNION ALL SELECT 6591, 'SIC', 3762, 5100, 2, 3769, 5130, 3 FROM dual
UNION ALL SELECT 6592, 'NAICS', 955, 4234, 3, 957, 42341, 4 FROM dual
UNION ALL SELECT 6593, 'NAICS', 1849, 61171, 4, 1848, 611710, 5 FROM dual
UNION ALL SELECT 6594, 'NAICS', 205, 23, 1, 207, 2361, 3 FROM dual
UNION ALL SELECT 6595, 'NAICS', 1410, 51113, 4, 1409, 511130, 5 FROM dual
UNION ALL SELECT 6596, 'SIC', 3214, 2800, 2, 3231, 2841, 4 FROM dual
UNION ALL SELECT 6597, 'NAICS', 1979, 713, 2, 1989, 71329, 4 FROM dual
UNION ALL SELECT 6598, 'NAICS', 1482, 5211, 3, 1484, 52111, 4 FROM dual
UNION ALL SELECT 6599, 'SIC', 4314, 90, 1, 4265, 9300, 2 FROM dual
UNION ALL SELECT 6600, 'NAICS', 1982, 71311, 4, 1981, 713110, 5 FROM dual
UNION ALL SELECT 6601, 'NAICS', 2077, 81219, 4, 2078, 812191, 5 FROM dual
UNION ALL SELECT 6602, 'SIC', 3141, 2450, 3, 3143, 2452, 4 FROM dual
UNION ALL SELECT 6603, 'NAICS', 1914, 62331, 4, 1915, 623311, 5 FROM dual
UNION ALL SELECT 6604, 'NAICS', 1726, 561, 2, 1777, 561720, 5 FROM dual
UNION ALL SELECT 6605, 'NAICS', 2072, 8121, 3, 2073, 81211, 4 FROM dual
UNION ALL SELECT 6606, 'SIC', 4040, 7300, 2, 4045, 7319, 4 FROM dual
UNION ALL SELECT 6607, 'SIC', 4223, 8700, 2, 4228, 8720, 3 FROM dual
UNION ALL SELECT 6608, 'SIC', 2834, 250, 3, 2837, 253, 4 FROM dual
UNION ALL SELECT 6609, 'SEC', 2568, 5000, 2, 2584, 5072, 4 FROM dual
UNION ALL SELECT 6610, 'NAICS', 931, 423, 2, 986, 42372, 4 FROM dual
UNION ALL SELECT 6611, 'SEC', 2539, 4800, 2, 2540, 4810, 3 FROM dual
UNION ALL SELECT 6612, 'SIC', 3026, 2080, 3, 3030, 2085, 4 FROM dual
UNION ALL SELECT 6613, 'NAICS', 1508, 52231, 4, 1507, 522310, 5 FROM dual
UNION ALL SELECT 6614, 'SIC', 3496, 3640, 3, 3502, 3647, 4 FROM dual
UNION ALL SELECT 6615, 'SIC', 4313, 70, 1, 4221, 8690, 3 FROM dual
UNION ALL SELECT 6616, 'SIC', 4313, 70, 1, 4246, 8999, 4 FROM dual
UNION ALL SELECT 6617, 'NAICS', 1512, 52239, 4, 1511, 522390, 5 FROM dual
UNION ALL SELECT 6618, 'NAICS', 1850, 62, 1, 1923, 62411, 4 FROM dual
UNION ALL SELECT 6619, 'SIC', 4314, 90, 1, 4264, 9229, 4 FROM dual
UNION ALL SELECT 6620, 'SIC', 3715, 5000, 2, 3739, 5052, 4 FROM dual
UNION ALL SELECT 6621, 'NAICS', 961, 42343, 4, 960, 423430, 5 FROM dual
UNION ALL SELECT 6622, 'SIC', 3917, 6000, 2, 3934, 6090, 3 FROM dual
UNION ALL SELECT 6623, 'NAICS', 23, 1113, 3, 28, 11133, 4 FROM dual
UNION ALL SELECT 6624, 'SIC', 4256, 9200, 2, 4263, 9224, 4 FROM dual
UNION ALL SELECT 6625, 'NAICS', 1817, 6112, 3, 1819, 61121, 4 FROM dual
UNION ALL SELECT 6626, 'SEC', 2366, 3270, 3, 2367, 3272, 4 FROM dual
UNION ALL SELECT 6627, 'SIC', 3301, 3200, 2, 3334, 3296, 4 FROM dual
UNION ALL SELECT 6628, 'SEC', 2302, 2700, 2, 2317, 2780, 3 FROM dual
UNION ALL SELECT 6629, 'NAICS', 174, 21311, 4, 179, 213115, 5 FROM dual
UNION ALL SELECT 6630, 'NAICS', 1480, 52, 1, 1568, 52599, 4 FROM dual
UNION ALL SELECT 6631, 'NAICS', 1812, 61, 1, 1824, 611410, 5 FROM dual
UNION ALL SELECT 6632, 'SEC', 2677, 6300, 2, 2679, 6311, 4 FROM dual
UNION ALL SELECT 6633, 'NAICS', 1689, 5418, 3, 1702, 541870, 5 FROM dual
UNION ALL SELECT 6634, 'NAICS', 2038, 811, 2, 2063, 811411, 5 FROM dual
UNION ALL SELECT 6635, 'SIC', 4313, 70, 1, 4097, 7540, 3 FROM dual
UNION ALL SELECT 6636, 'SIC', 2982, 2000, 2, 3018, 2067, 4 FROM dual
UNION ALL SELECT 6637, 'NAICS', 930, 42, 1, 1090, 4251, 3 FROM dual
UNION ALL SELECT 6638, 'NAICS', 1625, 541, 2, 1712, 541930, 5 FROM dual
UNION ALL SELECT 6639, 'SIC', 3595, 3990, 3, 3598, 3995, 4 FROM dual
UNION ALL SELECT 6640, 'SIC', 4308, 20, 1, 3221, 2821, 4 FROM dual
UNION ALL SELECT 6641, 'NAICS', 1471, 5191, 3, 1473, 51911, 4 FROM dual
UNION ALL SELECT 6642, 'SIC', 2884, 1000, 2, 2887, 1020, 3 FROM dual
UNION ALL SELECT 6643, 'SIC', 3220, 2820, 3, 3221, 2821, 4 FROM dual
UNION ALL SELECT 6644, 'NAICS', 2135, 92, 1, 2188, 926, 2 FROM dual
UNION ALL SELECT 6645, 'NAICS', 235, 238, 2, 260, 2383, 3 FROM dual
UNION ALL SELECT 6646, 'SEC', 2450, 3670, 3, 2453, 3677, 4 FROM dual
UNION ALL SELECT 6647, 'SEC', 2793, 50, 1, 2599, 5160, 3 FROM dual
UNION ALL SELECT 6648, 'NAICS', 1, 11, 1, 97, 11299, 4 FROM dual
UNION ALL SELECT 6649, 'NAICS', 108, 114, 2, 116, 11421, 4 FROM dual
UNION ALL SELECT 6650, 'SEC', 2795, 60, 1, 4340, 6172, 4 FROM dual
UNION ALL SELECT 6651, 'SIC', 4308, 20, 1, 3296, 3170, 3 FROM dual
UNION ALL SELECT 6652, 'SEC', 2791, 20, 1, 2359, 3221, 4 FROM dual
UNION ALL SELECT 6653, 'NAICS', 1939, 6244, 3, 1941, 62441, 4 FROM dual
UNION ALL SELECT 6654, 'NAICS', 2103, 813, 2, 2106, 81311, 4 FROM dual
UNION ALL SELECT 6655, 'SIC', 4312, 60, 1, 3933, 6082, 4 FROM dual
UNION ALL SELECT 6656, 'SIC', 4308, 20, 1, 3037, 2096, 4 FROM dual
UNION ALL SELECT 6657, 'NAICS', 1725, 56, 1, 1771, 56162, 4 FROM dual
UNION ALL SELECT 6658, 'SEC', 2362, 3240, 3, 2363, 3241, 4 FROM dual
UNION ALL SELECT 6659, 'SIC', 3083, 2300, 2, 3101, 2350, 3 FROM dual
UNION ALL SELECT 6660, 'SIC', 4308, 20, 1, 3030, 2085, 4 FROM dual
UNION ALL SELECT 6661, 'SIC', 3434, 3540, 3, 3441, 3547, 4 FROM dual
UNION ALL SELECT 6662, 'SIC', 4309, 40, 1, 3647, 4482, 4 FROM dual
UNION ALL SELECT 6663, 'SIC', 3552, 3800, 2, 3553, 3810, 3 FROM dual
UNION ALL SELECT 6664, 'SIC', 4308, 20, 1, 3265, 3021, 4 FROM dual
UNION ALL SELECT 6665, 'SIC', 3977, 6500, 2, 3981, 6514, 4 FROM dual
UNION ALL SELECT 6666, 'SEC', 2789, 10, 1, 2218, 1200, 2 FROM dual
UNION ALL SELECT 6667, 'SIC', 4308, 20, 1, 3563, 3829, 4 FROM dual
UNION ALL SELECT 6668, 'SIC', 3804, 5200, 2, 3809, 5250, 3 FROM dual
UNION ALL SELECT 6669, 'NAICS', 2037, 81, 1, 2085, 8123, 3 FROM dual
UNION ALL SELECT 6670, 'SEC', 2343, 3010, 3, 2344, 3011, 4 FROM dual
UNION ALL SELECT 6671, 'NAICS', 2077, 81219, 4, 2079, 812199, 5 FROM dual
UNION ALL SELECT 6672, 'SIC', 4235, 8740, 3, 4236, 8741, 4 FROM dual
UNION ALL SELECT 6673, 'NAICS', 1461, 51741, 4, 1460, 517410, 5 FROM dual
UNION ALL SELECT 6674, 'SIC', 4313, 70, 1, 4172, 8092, 4 FROM dual
UNION ALL SELECT 6675, 'NAICS', 1481, 521, 2, 1482, 5211, 3 FROM dual
UNION ALL SELECT 6676, 'SEC', 2791, 20, 1, 2426, 3572, 4 FROM dual
UNION ALL SELECT 6677, 'SIC', 3813, 5270, 3, 3814, 5271, 4 FROM dual
UNION ALL SELECT 6678, 'SEC', 2403, 3500, 2, 2424, 3570, 3 FROM dual
UNION ALL SELECT 6679, 'NAICS', 2085, 8123, 3, 2090, 81233, 4 FROM dual
UNION ALL SELECT 6680, 'SEC', 2791, 20, 1, 2449, 3669, 4 FROM dual
UNION ALL SELECT 6681, 'SIC', 4313, 70, 1, 4218, 8651, 4 FROM dual
UNION ALL SELECT 6682, 'SIC', 3355, 3350, 3, 3360, 3356, 4 FROM dual
UNION ALL SELECT 6683, 'NAICS', 1402, 51, 1, 1410, 51113, 4 FROM dual
UNION ALL SELECT 6684, 'SEC', 2793, 50, 1, 2583, 5070, 3 FROM dual
UNION ALL SELECT 6685, 'SIC', 2982, 2000, 2, 2999, 2038, 4 FROM dual
UNION ALL SELECT 6686, 'NAICS', 117, 115, 2, 120, 115111, 5 FROM dual
UNION ALL SELECT 6687, 'NAICS', 1725, 56, 1, 1809, 56299, 4 FROM dual
UNION ALL SELECT 6688, 'SIC', 4300, 9720, 3, 4301, 9721, 4 FROM dual
UNION ALL SELECT 6689, 'SIC', 4308, 20, 1, 3174, 2650, 3 FROM dual
UNION ALL SELECT 6690, 'SIC', 3959, 6310, 3, 3960, 6311, 4 FROM dual
UNION ALL SELECT 6691, 'SIC', 3620, 4200, 2, 3621, 4210, 3 FROM dual
UNION ALL SELECT 6692, 'SIC', 4017, 7200, 2, 4032, 7241, 4 FROM dual
UNION ALL SELECT 6693, 'NAICS', 107, 11331, 4, 106, 113310, 5 FROM dual
UNION ALL SELECT 6694, 'SIC', 3641, 4430, 3, 3642, 4432, 4 FROM dual
UNION ALL SELECT 6695, 'SIC', 3869, 5700, 2, 3876, 5722, 4 FROM dual
UNION ALL SELECT 6696, 'NAICS', 1624, 54, 1, 1667, 54151, 4 FROM dual
UNION ALL SELECT 6697, 'SIC', 4114, 7810, 3, 4115, 7812, 4 FROM dual
UNION ALL SELECT 6698, 'SIC', 2850, 720, 3, 2854, 724, 4 FROM dual
UNION ALL SELECT 6699, 'NAICS', 1486, 5221, 3, 1488, 52211, 4 FROM dual
UNION ALL SELECT 6700, 'NAICS', 1943, 711, 2, 1945, 711110, 5 FROM dual
UNION ALL SELECT 6701, 'SIC', 3083, 2300, 2, 3098, 2340, 3 FROM dual
UNION ALL SELECT 6702, 'NAICS', 2135, 92, 1, 2200, 927, 2 FROM dual
UNION ALL SELECT 6703, 'NAICS', 1, 11, 1, 110, 11411, 4 FROM dual
UNION ALL SELECT 6704, 'NAICS', 1851, 621, 2, 1890, 62191, 4 FROM dual
UNION ALL SELECT 6705, 'NAICS', 1850, 62, 1, 1860, 621310, 5 FROM dual
UNION ALL SELECT 6706, 'NAICS', 2003, 72, 1, 2007, 72111, 4 FROM dual
UNION ALL SELECT 6707, 'SIC', 2918, 1400, 2, 2927, 1446, 4 FROM dual
UNION ALL SELECT 6708, 'NAICS', 1944, 7111, 3, 1949, 711130, 5 FROM dual
UNION ALL SELECT 6709, 'SIC', 4305, 1, 1, 2848, 710, 3 FROM dual
UNION ALL SELECT 6710, 'SIC', 3371, 3400, 2, 3387, 3443, 4 FROM dual
UNION ALL SELECT 6711, 'SEC', 2791, 20, 1, 2240, 2000, 2 FROM dual
UNION ALL SELECT 6712, 'SIC', 4313, 70, 1, 4007, 7000, 2 FROM dual
UNION ALL SELECT 6713, 'SIC', 4313, 70, 1, 4133, 7940, 3 FROM dual
UNION ALL SELECT 6714, 'SIC', 4313, 70, 1, 4154, 8041, 4 FROM dual
UNION ALL SELECT 6715, 'SIC', 4312, 60, 1, 3928, 6060, 3 FROM dual
UNION ALL SELECT 6716, 'NAICS', 235, 238, 2, 272, 23839, 4 FROM dual
UNION ALL SELECT 6717, 'SIC', 4312, 60, 1, 3927, 6036, 4 FROM dual
UNION ALL SELECT 6718, 'NAICS', 1624, 54, 1, 1711, 541922, 5 FROM dual
UNION ALL SELECT 6719, 'SEC', 2434, 3600, 2, 2449, 3669, 4 FROM dual
UNION ALL SELECT 6720, 'NAICS', 205, 23, 1, 248, 23816, 4 FROM dual
UNION ALL SELECT 6721, 'NAICS', 2037, 81, 1, 2081, 812210, 5 FROM dual
UNION ALL SELECT 6722, 'SEC', 2319, 2800, 2, 2333, 2860, 3 FROM dual
UNION ALL SELECT 6723, 'SIC', 4314, 90, 1, 4301, 9721, 4 FROM dual
UNION ALL SELECT 6724, 'SIC', 4308, 20, 1, 3583, 3940, 3 FROM dual
UNION ALL SELECT 6725, 'SIC', 2826, 210, 3, 2830, 214, 4 FROM dual
UNION ALL SELECT 6726, 'NAICS', 930, 42, 1, 1061, 4246, 3 FROM dual
UNION ALL SELECT 6727, 'NAICS', 1850, 62, 1, 1862, 621320, 5 FROM dual
UNION ALL SELECT 6728, 'SIC', 2798, 100, 2, 2820, 180, 3 FROM dual
UNION ALL SELECT 6729, 'NAICS', 1921, 6241, 3, 1926, 624190, 5 FROM dual
UNION ALL SELECT 6730, 'NAICS', 930, 42, 1, 936, 42312, 4 FROM dual
UNION ALL SELECT 6731, 'NAICS', 1612, 5324, 3, 1615, 532412, 5 FROM dual
UNION ALL SELECT 6732, 'NAICS', 2120, 8139, 3, 2128, 81394, 4 FROM dual
UNION ALL SELECT 6733, 'SIC', 4309, 40, 1, 3666, 4619, 4 FROM dual
UNION ALL SELECT 6734, 'SIC', 4310, 50, 1, 3748, 5078, 4 FROM dual
UNION ALL SELECT 6735, 'NAICS', 1904, 623, 2, 1911, 623220, 5 FROM dual
UNION ALL SELECT 6736, 'NAICS', 1439, 51224, 4, 1438, 512240, 5 FROM dual
UNION ALL SELECT 6737, 'NAICS', 1625, 541, 2, 1681, 541690, 5 FROM dual
UNION ALL SELECT 6738, 'NAICS', 1812, 61, 1, 1834, 611513, 5 FROM dual
UNION ALL SELECT 6739, 'SIC', 2909, 1300, 2, 2916, 1382, 4 FROM dual
UNION ALL SELECT 6740, 'NAICS', 1625, 541, 2, 1640, 5413, 3 FROM dual
UNION ALL SELECT 6741, 'SIC', 4309, 40, 1, 3685, 4822, 4 FROM dual
UNION ALL SELECT 6742, 'NAICS', 1514, 5231, 3, 1522, 52314, 4 FROM dual
UNION ALL SELECT 6743, 'NAICS', 1624, 54, 1, 1709, 54192, 4 FROM dual
UNION ALL SELECT 6744, 'SIC', 2805, 130, 3, 2807, 132, 4 FROM dual
UNION ALL SELECT 6745, 'NAICS', 1569, 53, 1, 1597, 53212, 4 FROM dual
UNION ALL SELECT 6746, 'NAICS', 206, 236, 2, 209, 236115, 5 FROM dual
UNION ALL SELECT 6747, 'SEC', 2458, 3700, 2, 2459, 3710, 3 FROM dual
UNION ALL SELECT 6748, 'NAICS', 1830, 6115, 3, 1834, 611513, 5 FROM dual
UNION ALL SELECT 6749, 'NAICS', 93, 11292, 4, 92, 112920, 5 FROM dual
UNION ALL SELECT 6750, 'SIC', 3190, 2700, 2, 3212, 2791, 4 FROM dual
UNION ALL SELECT 6751, 'SIC', 3261, 3000, 2, 3264, 3020, 3 FROM dual
UNION ALL SELECT 6752, 'SIC', 4308, 20, 1, 3188, 2678, 4 FROM dual
UNION ALL SELECT 6753, 'SIC', 3662, 4600, 2, 3663, 4610, 3 FROM dual
UNION ALL SELECT 6754, 'NAICS', 139, 2121, 3, 142, 212112, 5 FROM dual
UNION ALL SELECT 6755, 'SIC', 4306, 10, 1, 2937, 1490, 3 FROM dual
UNION ALL SELECT 6756, 'SIC', 3214, 2800, 2, 3251, 2899, 4 FROM dual
UNION ALL SELECT 6757, 'NAICS', 2103, 813, 2, 2105, 813110, 5 FROM dual
UNION ALL SELECT 6758, 'SEC', 2566, 4960, 3, 2567, 4961, 4 FROM dual
UNION ALL SELECT 6759, 'SIC', 4308, 20, 1, 3361, 3357, 4 FROM dual
UNION ALL SELECT 6760, 'SIC', 4313, 70, 1, 4030, 7231, 4 FROM dual
UNION ALL SELECT 6761, 'NAICS', 1942, 71, 1, 1977, 712190, 5 FROM dual
UNION ALL SELECT 6762, 'SIC', 4308, 20, 1, 3468, 3580, 3 FROM dual
UNION ALL SELECT 6763, 'NAICS', 118, 1151, 3, 120, 115111, 5 FROM dual
UNION ALL SELECT 6764, 'NAICS', 1, 11, 1, 4, 111110, 5 FROM dual
UNION ALL SELECT 6765, 'SIC', 4313, 70, 1, 4222, 8699, 4 FROM dual
UNION ALL SELECT 6766, 'NAICS', 1725, 56, 1, 1744, 56142, 4 FROM dual
UNION ALL SELECT 6767, 'SIC', 4308, 20, 1, 3397, 3463, 4 FROM dual
UNION ALL SELECT 6768, 'SEC', 2530, 4520, 3, 2531, 4522, 4 FROM dual
UNION ALL SELECT 6769, 'NAICS', 215, 23621, 4, 214, 236210, 5 FROM dual
UNION ALL SELECT 6770, 'SIC', 4308, 20, 1, 3113, 2387, 4 FROM dual
UNION ALL SELECT 6771, 'NAICS', 1420, 5121, 3, 1421, 512110, 5 FROM dual
UNION ALL SELECT 6772, 'NAICS', 2150, 922, 2, 2165, 92219, 4 FROM dual
UNION ALL SELECT 6773, 'SIC', 4308, 20, 1, 3208, 2780, 3 FROM dual
UNION ALL SELECT 6774, 'SIC', 3156, 2520, 3, 3157, 2521, 4 FROM dual
UNION ALL SELECT 6775, 'NAICS', 930, 42, 1, 1054, 4245, 3 FROM dual
UNION ALL SELECT 6776, 'NAICS', 1792, 562, 2, 1799, 56221, 4 FROM dual
UNION ALL SELECT 6777, 'NAICS', 1624, 54, 1, 1716, 541990, 5 FROM dual
UNION ALL SELECT 6778, 'NAICS', 1850, 62, 1, 1869, 621391, 5 FROM dual
UNION ALL SELECT 6779, 'NAICS', 1876, 62149, 4, 1878, 621492, 5 FROM dual
UNION ALL SELECT 6780, 'SIC', 4313, 70, 1, 4127, 7911, 4 FROM dual
UNION ALL SELECT 6781, 'SIC', 4312, 60, 1, 3920, 6019, 4 FROM dual
UNION ALL SELECT 6782, 'NAICS', 1624, 54, 1, 1682, 54169, 4 FROM dual
UNION ALL SELECT 6783, 'SIC', 3167, 2600, 2, 3182, 2672, 4 FROM dual
UNION ALL SELECT 6784, 'SIC', 3214, 2800, 2, 3224, 2824, 4 FROM dual
UNION ALL SELECT 6785, 'SIC', 3958, 6300, 2, 3973, 6399, 4 FROM dual
UNION ALL SELECT 6786, 'NAICS', 238, 23811, 4, 237, 238110, 5 FROM dual
UNION ALL SELECT 6787, 'SEC', 2796, 70, 1, 2780, 8740, 3 FROM dual
UNION ALL SELECT 6788, 'SEC', 2735, 7500, 2, 2736, 7510, 3 FROM dual
UNION ALL SELECT 6789, 'SEC', 2574, 5040, 3, 2576, 5047, 4 FROM dual
UNION ALL SELECT 6790, 'SIC', 4308, 20, 1, 3015, 2063, 4 FROM dual
UNION ALL SELECT 6791, 'SIC', 3526, 3700, 2, 3545, 3761, 4 FROM dual
UNION ALL SELECT 6792, 'SEC', 2358, 3220, 3, 2359, 3221, 4 FROM dual
UNION ALL SELECT 6793, 'SIC', 3520, 3690, 3, 3524, 3695, 4 FROM dual
UNION ALL SELECT 6794, 'NAICS', 172, 213, 2, 175, 213111, 5 FROM dual
UNION ALL SELECT 6795, 'SIC', 2939, 1500, 2, 2946, 1541, 4 FROM dual
UNION ALL SELECT 6796, 'SIC', 3576, 3900, 2, 3598, 3995, 4 FROM dual
UNION ALL SELECT 6797, 'NAICS', 2037, 81, 1, 2067, 811430, 5 FROM dual
UNION ALL SELECT 6798, 'SEC', 2277, 2420, 3, 2278, 2421, 4 FROM dual
UNION ALL SELECT 6799, 'SIC', 3419, 3500, 2, 3431, 3535, 4 FROM dual
UNION ALL SELECT 6800, 'SIC', 4309, 40, 1, 3704, 4939, 4 FROM dual
UNION ALL SELECT 6801, 'SIC', 3917, 6000, 2, 3921, 6020, 3 FROM dual
UNION ALL SELECT 6802, 'SEC', 2791, 20, 1, 2277, 2420, 3 FROM dual
UNION ALL SELECT 6803, 'SEC', 2659, 6100, 2, 4323, 6189, 4 FROM dual
UNION ALL SELECT 6804, 'SIC', 3115, 2390, 3, 3123, 2399, 4 FROM dual
UNION ALL SELECT 6805, 'SEC', 2384, 3400, 2, 2386, 3411, 4 FROM dual
UNION ALL SELECT 6806, 'SEC', 2791, 20, 1, 2498, 3870, 3 FROM dual
UNION ALL SELECT 6807, 'SIC', 4313, 70, 1, 4181, 8220, 3 FROM dual
UNION ALL SELECT 6808, 'NAICS', 1577, 53113, 4, 1576, 531130, 5 FROM dual
UNION ALL SELECT 6809, 'SIC', 4040, 7300, 2, 4052, 7335, 4 FROM dual
UNION ALL SELECT 6810, 'NAICS', 1506, 5223, 3, 1509, 522320, 5 FROM dual
UNION ALL SELECT 6811, 'NAICS', 2189, 9261, 3, 2198, 926150, 5 FROM dual
UNION ALL SELECT 6812, 'NAICS', 1592, 5321, 3, 1597, 53212, 4 FROM dual
UNION ALL SELECT 6813, 'NAICS', 1624, 54, 1, 1641, 541310, 5 FROM dual
UNION ALL SELECT 6814, 'NAICS', 2071, 812, 2, 2083, 812220, 5 FROM dual
UNION ALL SELECT 6815, 'SIC', 3526, 3700, 2, 3530, 3714, 4 FROM dual
UNION ALL SELECT 6816, 'NAICS', 1570, 531, 2, 1581, 531210, 5 FROM dual
UNION ALL SELECT 6817, 'SIC', 3190, 2700, 2, 3208, 2780, 3 FROM dual
UNION ALL SELECT 6818, 'NAICS', 1485, 522, 2, 1506, 5223, 3 FROM dual
UNION ALL SELECT 6819, 'SIC', 4305, 1, 1, 2813, 170, 3 FROM dual
UNION ALL SELECT 6820, 'NAICS', 207, 2361, 3, 209, 236115, 5 FROM dual
UNION ALL SELECT 6821, 'SEC', 2691, 6400, 2, 2693, 6411, 4 FROM dual
UNION ALL SELECT 6822, 'SEC', 2458, 3700, 2, 2466, 3721, 4 FROM dual
UNION ALL SELECT 6823, 'NAICS', 930, 42, 1, 1050, 424480, 5 FROM dual
UNION ALL SELECT 6824, 'SEC', 2790, 15, 1, 2234, 1600, 2 FROM dual
UNION ALL SELECT 6825, 'SIC', 3149, 2510, 3, 3153, 2515, 4 FROM dual
UNION ALL SELECT 6826, 'SIC', 4308, 20, 1, 3151, 2512, 4 FROM dual
UNION ALL SELECT 6827, 'SEC', 2788, 1, 1, 2210, 100, 2 FROM dual
UNION ALL SELECT 6828, 'NAICS', 1625, 541, 2, 1702, 541870, 5 FROM dual
UNION ALL SELECT 6829, 'SIC', 4312, 60, 1, 3957, 6289, 4 FROM dual
UNION ALL SELECT 6830, 'SIC', 3948, 6200, 2, 3954, 6231, 4 FROM dual
UNION ALL SELECT 6831, 'SIC', 4308, 20, 1, 3119, 2394, 4 FROM dual
UNION ALL SELECT 6832, 'SIC', 4310, 50, 1, 3772, 5137, 4 FROM dual
UNION ALL SELECT 6833, 'SIC', 4308, 20, 1, 2985, 2013, 4 FROM dual
UNION ALL SELECT 6834, 'SIC', 4308, 20, 1, 3580, 3915, 4 FROM dual
UNION ALL SELECT 6835, 'SIC', 3811, 5260, 3, 3812, 5261, 4 FROM dual
UNION ALL SELECT 6836, 'NAICS', 1, 11, 1, 28, 11133, 4 FROM dual
UNION ALL SELECT 6837, 'SIC', 4308, 20, 1, 3212, 2791, 4 FROM dual
UNION ALL SELECT 6838, 'SIC', 3978, 6510, 3, 3982, 6515, 4 FROM dual
UNION ALL SELECT 6839, 'NAICS', 205, 23, 1, 263, 238320, 5 FROM dual
UNION ALL SELECT 6840, 'NAICS', 1624, 54, 1, 1714, 541940, 5 FROM dual
UNION ALL SELECT 6841, 'SEC', 2791, 20, 1, 2424, 3570, 3 FROM dual
UNION ALL SELECT 6842, 'SIC', 3468, 3580, 3, 3469, 3581, 4 FROM dual
UNION ALL SELECT 6843, 'SEC', 2215, 1000, 2, 2216, 1040, 3 FROM dual
UNION ALL SELECT 6844, 'SEC', 2403, 3500, 2, 2426, 3572, 4 FROM dual
UNION ALL SELECT 6845, 'SIC', 4259, 9220, 3, 4263, 9224, 4 FROM dual
UNION ALL SELECT 6846, 'NAICS', 930, 42, 1, 1002, 423860, 5 FROM dual
UNION ALL SELECT 6847, 'NAICS', 2157, 92213, 4, 2156, 922130, 5 FROM dual
UNION ALL SELECT 6848, 'SEC', 2371, 3300, 2, 2378, 3340, 3 FROM dual
UNION ALL SELECT 6849, 'NAICS', 1634, 5412, 3, 1639, 541219, 5 FROM dual
UNION ALL SELECT 6850, 'NAICS', 205, 23, 1, 257, 23822, 4 FROM dual
UNION ALL SELECT 6851, 'NAICS', 1442, 515, 2, 1448, 51512, 4 FROM dual
UNION ALL SELECT 6852, 'SIC', 3371, 3400, 2, 3382, 3432, 4 FROM dual
UNION ALL SELECT 6853, 'SIC', 4041, 7310, 3, 4045, 7319, 4 FROM dual
UNION ALL SELECT 6854, 'NAICS', 260, 2383, 3, 261, 238310, 5 FROM dual
UNION ALL SELECT 6855, 'SIC', 4308, 20, 1, 3462, 3571, 4 FROM dual
UNION ALL SELECT 6856, 'NAICS', 132, 21, 1, 139, 2121, 3 FROM dual
UNION ALL SELECT 6857, 'NAICS', 1470, 519, 2, 1473, 51911, 4 FROM dual
UNION ALL SELECT 6858, 'NAICS', 1402, 51, 1, 1454, 517110, 5 FROM dual
UNION ALL SELECT 6859, 'SEC', 2692, 6410, 3, 2693, 6411, 4 FROM dual
UNION ALL SELECT 6860, 'NAICS', 1591, 532, 2, 1614, 532411, 5 FROM dual
UNION ALL SELECT 6861, 'SEC', 2790, 15, 1, 2231, 1530, 3 FROM dual
UNION ALL SELECT 6862, 'SIC', 4305, 1, 1, 2814, 171, 4 FROM dual
UNION ALL SELECT 6863, 'SEC', 2670, 6200, 2, 2673, 6220, 3 FROM dual
UNION ALL SELECT 6864, 'NAICS', 1, 11, 1, 43, 1119, 3 FROM dual
UNION ALL SELECT 6865, 'NAICS', 1090, 4251, 3, 1094, 42512, 4 FROM dual
UNION ALL SELECT 6866, 'NAICS', 1721, 55111, 4, 1724, 551114, 5 FROM dual
UNION ALL SELECT 6867, 'SIC', 2955, 1700, 2, 2964, 1742, 4 FROM dual
UNION ALL SELECT 6868, 'NAICS', 2052, 8112, 3, 2053, 81121, 4 FROM dual
UNION ALL SELECT 6869, 'NAICS', 1726, 561, 2, 1741, 5614, 3 FROM dual
UNION ALL SELECT 6870, 'NAICS', 70, 11231, 4, 69, 112310, 5 FROM dual
UNION ALL SELECT 6871, 'SIC', 4311, 52, 1, 3851, 5571, 4 FROM dual
UNION ALL SELECT 6872, 'NAICS', 235, 238, 2, 262, 23831, 4 FROM dual
UNION ALL SELECT 6873, 'NAICS', 1979, 713, 2, 1987, 71321, 4 FROM dual
UNION ALL SELECT 6874, 'NAICS', 1851, 621, 2, 1852, 6211, 3 FROM dual
UNION ALL SELECT 6875, 'NAICS', 2, 111, 2, 22, 111219, 5 FROM dual
UNION ALL SELECT 6876, 'SIC', 4308, 20, 1, 2994, 2032, 4 FROM dual
UNION ALL SELECT 6877, 'NAICS', 1569, 53, 1, 1613, 53241, 4 FROM dual
UNION ALL SELECT 6878, 'SIC', 4311, 52, 1, 3828, 5431, 4 FROM dual
UNION ALL SELECT 6879, 'SIC', 3655, 4510, 3, 3656, 4512, 4 FROM dual
UNION ALL SELECT 6880, 'NAICS', 1015, 424, 2, 1055, 424510, 5 FROM dual
UNION ALL SELECT 6881, 'NAICS', 1900, 62221, 4, 1899, 622210, 5 FROM dual
UNION ALL SELECT 6882, 'NAICS', 1725, 56, 1, 1780, 56173, 4 FROM dual
UNION ALL SELECT 6883, 'SIC', 4314, 90, 1, 4283, 9532, 4 FROM dual
UNION ALL SELECT 6884, 'NAICS', 946, 4233, 3, 951, 423330, 5 FROM dual
UNION ALL SELECT 6885, 'NAICS', 1726, 561, 2, 1789, 56192, 4 FROM dual
UNION ALL SELECT 6886, 'SIC', 4308, 20, 1, 3183, 2673, 4 FROM dual
UNION ALL SELECT 6887, 'SIC', 3977, 6500, 2, 3991, 6553, 4 FROM dual
UNION ALL SELECT 6888, 'SIC', 4284, 9600, 2, 4288, 9621, 4 FROM dual
UNION ALL SELECT 6889, 'NAICS', 1942, 71, 1, 1953, 7112, 3 FROM dual
UNION ALL SELECT 6890, 'NAICS', 36, 1114, 3, 42, 111422, 5 FROM dual
UNION ALL SELECT 6891, 'NAICS', 1689, 5418, 3, 1701, 54186, 4 FROM dual
UNION ALL SELECT 6892, 'SIC', 4313, 70, 1, 4060, 7353, 4 FROM dual
UNION ALL SELECT 6893, 'NAICS', 930, 42, 1, 933, 423110, 5 FROM dual
UNION ALL SELECT 6894, 'SEC', 2355, 3200, 2, 2365, 3260, 3 FROM dual
UNION ALL SELECT 6895, 'NAICS', 1479, 51919, 4, 1478, 519190, 5 FROM dual
UNION ALL SELECT 6896, 'NAICS', 225, 23713, 4, 224, 237130, 5 FROM dual
UNION ALL SELECT 6897, 'NAICS', 1812, 61, 1, 1818, 611210, 5 FROM dual
UNION ALL SELECT 6898, 'NAICS', 955, 4234, 3, 956, 423410, 5 FROM dual
UNION ALL SELECT 6899, 'NAICS', 930, 42, 1, 1025, 42421, 4 FROM dual
UNION ALL SELECT 6900, 'SIC', 3419, 3500, 2, 3444, 3550, 3 FROM dual
UNION ALL SELECT 6901, 'NAICS', 1624, 54, 1, 1636, 541211, 5 FROM dual
UNION ALL SELECT 6902, 'SIC', 2987, 2020, 3, 2991, 2024, 4 FROM dual
UNION ALL SELECT 6903, 'NAICS', 1530, 52392, 4, 1529, 523920, 5 FROM dual
UNION ALL SELECT 6904, 'SIC', 3484, 3620, 3, 3487, 3625, 4 FROM dual
UNION ALL SELECT 6905, 'NAICS', 28, 11133, 4, 35, 111339, 5 FROM dual
UNION ALL SELECT 6906, 'SIC', 3190, 2700, 2, 3198, 2740, 3 FROM dual
UNION ALL SELECT 6907, 'SIC', 3083, 2300, 2, 3100, 2342, 4 FROM dual
UNION ALL SELECT 6908, 'NAICS', 132, 21, 1, 135, 21111, 4 FROM dual
UNION ALL SELECT 6909, 'SEC', 2791, 20, 1, 2474, 3760, 3 FROM dual
UNION ALL SELECT 6910, 'SIC', 3693, 4900, 2, 3712, 4961, 4 FROM dual
UNION ALL SELECT 6911, 'SEC', 2470, 3740, 3, 2471, 3743, 4 FROM dual
UNION ALL SELECT 6912, 'SIC', 3620, 4200, 2, 3624, 4214, 4 FROM dual
UNION ALL SELECT 6913, 'NAICS', 72, 11232, 4, 71, 112320, 5 FROM dual
UNION ALL SELECT 6914, 'SIC', 3050, 2200, 2, 3057, 2240, 3 FROM dual
UNION ALL SELECT 6915, 'NAICS', 1467, 5182, 3, 1469, 51821, 4 FROM dual
UNION ALL SELECT 6916, 'SIC', 2982, 2000, 2, 3019, 2068, 4 FROM dual
UNION ALL SELECT 6917, 'SIC', 4312, 60, 1, 3960, 6311, 4 FROM dual
UNION ALL SELECT 6918, 'NAICS', 2163, 92216, 4, 2162, 922160, 5 FROM dual
UNION ALL SELECT 6919, 'NAICS', 1452, 517, 2, 1464, 517911, 5 FROM dual
UNION ALL SELECT 6920, 'SEC', 2643, 5940, 3, 2645, 5945, 4 FROM dual
UNION ALL SELECT 6921, 'SIC', 4308, 20, 1, 3175, 2652, 4 FROM dual
UNION ALL SELECT 6922, 'NAICS', 1571, 5311, 3, 1572, 531110, 5 FROM dual
UNION ALL SELECT 6923, 'SIC', 4313, 70, 1, 4038, 7291, 4 FROM dual
UNION ALL SELECT 6924, 'SIC', 2847, 700, 2, 2858, 750, 3 FROM dual
UNION ALL SELECT 6925, 'SIC', 3337, 3300, 2, 3338, 3310, 3 FROM dual
UNION ALL SELECT 6926, 'NAICS', 1624, 54, 1, 1668, 541511, 5 FROM dual
UNION ALL SELECT 6927, 'NAICS', 2135, 92, 1, 2173, 92313, 4 FROM dual
UNION ALL SELECT 6928, 'NAICS', 1881, 6215, 3, 1882, 62151, 4 FROM dual
UNION ALL SELECT 6929, 'NAICS', 1943, 711, 2, 1954, 71121, 4 FROM dual
UNION ALL SELECT 6930, 'SIC', 3214, 2800, 2, 3241, 2870, 3 FROM dual
UNION ALL SELECT 6931, 'SEC', 2371, 3300, 2, 2374, 3317, 4 FROM dual
UNION ALL SELECT 6932, 'NAICS', 205, 23, 1, 242, 23813, 4 FROM dual
UNION ALL SELECT 6933, 'NAICS', 37, 11141, 4, 38, 111411, 5 FROM dual
UNION ALL SELECT 6934, 'SIC', 3159, 2530, 3, 3160, 2531, 4 FROM dual
UNION ALL SELECT 6935, 'SIC', 3050, 2200, 2, 3069, 2262, 4 FROM dual
UNION ALL SELECT 6936, 'SIC', 3526, 3700, 2, 3535, 3724, 4 FROM dual
UNION ALL SELECT 6937, 'NAICS', 1, 11, 1, 7, 11112, 4 FROM dual
UNION ALL SELECT 6938, 'NAICS', 931, 423, 2, 972, 42351, 4 FROM dual
UNION ALL SELECT 6939, 'SIC', 3404, 3480, 3, 3405, 3482, 4 FROM dual
UNION ALL SELECT 6940, 'SEC', 2649, 6000, 2, 2658, 6099, 4 FROM dual
UNION ALL SELECT 6941, 'NAICS', 1570, 531, 2, 1589, 531390, 5 FROM dual
UNION ALL SELECT 6942, 'NAICS', 1054, 4245, 3, 1059, 424590, 5 FROM dual
UNION ALL SELECT 6943, 'SIC', 4313, 70, 1, 4015, 7040, 3 FROM dual
UNION ALL SELECT 6944, 'NAICS', 1035, 4244, 3, 1037, 42441, 4 FROM dual
UNION ALL SELECT 6945, 'NAICS', 1480, 52, 1, 1560, 525190, 5 FROM dual
UNION ALL SELECT 6946, 'SIC', 3419, 3500, 2, 3453, 3562, 4 FROM dual
UNION ALL SELECT 6947, 'NAICS', 1719, 551, 2, 1723, 551112, 5 FROM dual
UNION ALL SELECT 6948, 'SIC', 3958, 6300, 2, 3959, 6310, 3 FROM dual
UNION ALL SELECT 6949, 'NAICS', 1942, 71, 1, 1975, 712130, 5 FROM dual
UNION ALL SELECT 6950, 'SIC', 3316, 3260, 3, 3317, 3261, 4 FROM dual
UNION ALL SELECT 6951, 'NAICS', 1819, 61121, 4, 1818, 611210, 5 FROM dual
UNION ALL SELECT 6952, 'SIC', 2982, 2000, 2, 3031, 2086, 4 FROM dual
UNION ALL SELECT 6953, 'NAICS', 2120, 8139, 3, 2123, 813920, 5 FROM dual
UNION ALL SELECT 6954, 'NAICS', 1942, 71, 1, 1968, 71151, 4 FROM dual
UNION ALL SELECT 6955, 'SIC', 3662, 4600, 2, 3666, 4619, 4 FROM dual
UNION ALL SELECT 6956, 'SIC', 4313, 70, 1, 4209, 8610, 3 FROM dual
UNION ALL SELECT 6957, 'NAICS', 2131, 814, 2, 2132, 8141, 3 FROM dual
UNION ALL SELECT 6958, 'SIC', 4308, 20, 1, 3591, 3955, 4 FROM dual
UNION ALL SELECT 6959, 'NAICS', 1425, 51213, 4, 1427, 512132, 5 FROM dual
UNION ALL SELECT 6960, 'SEC', 2396, 3450, 3, 2398, 3452, 4 FROM dual
UNION ALL SELECT 6961, 'NAICS', 260, 2383, 3, 270, 23835, 4 FROM dual
UNION ALL SELECT 6962, 'NAICS', 1420, 5121, 3, 1426, 512131, 5 FROM dual
UNION ALL SELECT 6963, 'SIC', 4313, 70, 1, 4214, 8631, 4 FROM dual
UNION ALL SELECT 6964, 'SEC', 2302, 2700, 2, 2303, 2710, 3 FROM dual
UNION ALL SELECT 6965, 'SIC', 4313, 70, 1, 4142, 7993, 4 FROM dual
UNION ALL SELECT 6966, 'SIC', 3474, 3590, 3, 3477, 3594, 4 FROM dual
UNION ALL SELECT 6967, 'SIC', 3371, 3400, 2, 3409, 3490, 3 FROM dual
UNION ALL SELECT 6968, 'NAICS', 969, 42349, 4, 968, 423490, 5 FROM dual
UNION ALL SELECT 6969, 'NAICS', 1403, 511, 2, 1407, 511120, 5 FROM dual
UNION ALL SELECT 6970, 'NAICS', 1941, 62441, 4, 1940, 624410, 5 FROM dual
UNION ALL SELECT 6971, 'SEC', 2791, 20, 1, 2296, 2621, 4 FROM dual
UNION ALL SELECT 6972, 'SIC', 4308, 20, 1, 3198, 2740, 3 FROM dual
UNION ALL SELECT 6973, 'SIC', 3620, 4200, 2, 3630, 4226, 4 FROM dual
UNION ALL SELECT 6974, 'SIC', 4311, 52, 1, 3871, 5712, 4 FROM dual
UNION ALL SELECT 6975, 'SEC', 2229, 1500, 2, 2230, 1520, 3 FROM dual
UNION ALL SELECT 6976, 'SIC', 3230, 2840, 3, 3233, 2843, 4 FROM dual
UNION ALL SELECT 6977, 'NAICS', 1026, 4243, 3, 1029, 424320, 5 FROM dual
UNION ALL SELECT 6978, 'NAICS', 1888, 6219, 3, 1889, 621910, 5 FROM dual
UNION ALL SELECT 6979, 'SIC', 4309, 40, 1, 3663, 4610, 3 FROM dual
UNION ALL SELECT 6980, 'NAICS', 1485, 522, 2, 1512, 52239, 4 FROM dual
UNION ALL SELECT 6981, 'NAICS', 1624, 54, 1, 1679, 541620, 5 FROM dual
UNION ALL SELECT 6982, 'SIC', 4062, 7360, 3, 4064, 7363, 4 FROM dual
UNION ALL SELECT 6983, 'NAICS', 52, 11199, 4, 55, 111998, 5 FROM dual
UNION ALL SELECT 6984, 'SIC', 4308, 20, 1, 3256, 2951, 4 FROM dual
UNION ALL SELECT 6985, 'NAICS', 1485, 522, 2, 1490, 52212, 4 FROM dual
UNION ALL SELECT 6986, 'NAICS', 1713, 54193, 4, 1712, 541930, 5 FROM dual
UNION ALL SELECT 6987, 'NAICS', 1, 11, 1, 90, 112910, 5 FROM dual
UNION ALL SELECT 6988, 'NAICS', 1733, 5613, 3, 1740, 56133, 4 FROM dual
UNION ALL SELECT 6989, 'SEC', 2796, 70, 1, 2776, 8711, 4 FROM dual
UNION ALL SELECT 6990, 'SIC', 3246, 2890, 3, 3249, 2893, 4 FROM dual
UNION ALL SELECT 6991, 'SIC', 3886, 5900, 2, 3894, 5941, 4 FROM dual
UNION ALL SELECT 6992, 'NAICS', 157, 21231, 4, 158, 212311, 5 FROM dual
UNION ALL SELECT 6993, 'SIC', 4313, 70, 1, 4018, 7210, 3 FROM dual
UNION ALL SELECT 6994, 'NAICS', 1076, 4249, 3, 1084, 42494, 4 FROM dual
UNION ALL SELECT 6995, 'SIC', 4308, 20, 1, 3474, 3590, 3 FROM dual
UNION ALL SELECT 6996, 'NAICS', 1850, 62, 1, 1856, 6212, 3 FROM dual
UNION ALL SELECT 6997, 'NAICS', 2147, 92115, 4, 2146, 921150, 5 FROM dual
UNION ALL SELECT 6998, 'SIC', 3137, 2440, 3, 3138, 2441, 4 FROM dual
UNION ALL SELECT 6999, 'NAICS', 1, 11, 1, 42, 111422, 5 FROM dual
UNION ALL SELECT 7000, 'NAICS', 206, 236, 2, 216, 236220, 5 FROM dual
UNION ALL SELECT 7001, 'NAICS', 1625, 541, 2, 1706, 5419, 3 FROM dual
UNION ALL SELECT 7002, 'SIC', 4308, 20, 1, 3539, 3732, 4 FROM dual
UNION ALL SELECT 7003, 'SIC', 3533, 3720, 3, 3534, 3721, 4 FROM dual
UNION ALL SELECT 7004, 'NAICS', 181, 221, 2, 189, 221116, 5 FROM dual
UNION ALL SELECT 7005, 'NAICS', 1, 11, 1, 59, 112111, 5 FROM dual
UNION ALL SELECT 7006, 'NAICS', 2061, 8114, 3, 2064, 811412, 5 FROM dual
UNION ALL SELECT 7007, 'NAICS', 2085, 8123, 3, 2091, 812331, 5 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 7008, 'NAICS', 2103, 813, 2, 2117, 8134, 3 FROM dual
UNION ALL SELECT 7009, 'SEC', 2677, 6300, 2, 2690, 6399, 4 FROM dual
UNION ALL SELECT 7010, 'SIC', 3190, 2700, 2, 3211, 2790, 3 FROM dual
UNION ALL SELECT 7011, 'SEC', 2788, 1, 1, 2211, 200, 2 FROM dual
UNION ALL SELECT 7012, 'NAICS', 138, 212, 2, 163, 212321, 5 FROM dual
UNION ALL SELECT 7013, 'NAICS', 1784, 56179, 4, 1783, 561790, 5 FROM dual
UNION ALL SELECT 7014, 'NAICS', 1851, 621, 2, 1855, 621112, 5 FROM dual
UNION ALL SELECT 7015, 'SIC', 4293, 9650, 3, 4294, 9651, 4 FROM dual
UNION ALL SELECT 7016, 'SIC', 4308, 20, 1, 3040, 2099, 4 FROM dual
UNION ALL SELECT 7017, 'NAICS', 253, 2382, 3, 259, 23829, 4 FROM dual
UNION ALL SELECT 7018, 'NAICS', 1625, 541, 2, 1701, 54186, 4 FROM dual
UNION ALL SELECT 7019, 'NAICS', 56, 112, 2, 83, 11242, 4 FROM dual
UNION ALL SELECT 7020, 'SIC', 4311, 52, 1, 3805, 5210, 3 FROM dual
UNION ALL SELECT 7021, 'NAICS', 1, 11, 1, 69, 112310, 5 FROM dual
UNION ALL SELECT 7022, 'NAICS', 138, 212, 2, 145, 212210, 5 FROM dual
UNION ALL SELECT 7023, 'NAICS', 2200, 927, 2, 2202, 927110, 5 FROM dual
UNION ALL SELECT 7024, 'SEC', 2794, 52, 1, 2607, 5211, 4 FROM dual
UNION ALL SELECT 7025, 'SIC', 3215, 2810, 3, 3219, 2819, 4 FROM dual
UNION ALL SELECT 7026, 'SIC', 4313, 70, 1, 4036, 7261, 4 FROM dual
UNION ALL SELECT 7027, 'SIC', 3012, 2060, 3, 3015, 2063, 4 FROM dual
UNION ALL SELECT 7028, 'NAICS', 1015, 424, 2, 1034, 42434, 4 FROM dual
UNION ALL SELECT 7029, 'SIC', 4308, 20, 1, 3334, 3296, 4 FROM dual
UNION ALL SELECT 7030, 'SIC', 4309, 40, 1, 3673, 4731, 4 FROM dual
UNION ALL SELECT 7031, 'SEC', 2791, 20, 1, 2392, 3442, 4 FROM dual
UNION ALL SELECT 7032, 'SIC', 4314, 90, 1, 4304, 9999, 4 FROM dual
UNION ALL SELECT 7033, 'NAICS', 931, 423, 2, 968, 423490, 5 FROM dual
UNION ALL SELECT 7034, 'NAICS', 1485, 522, 2, 1495, 5222, 3 FROM dual
UNION ALL SELECT 7035, 'SIC', 3287, 3140, 3, 3291, 3149, 4 FROM dual
UNION ALL SELECT 7036, 'SIC', 3693, 4900, 2, 3711, 4960, 3 FROM dual
UNION ALL SELECT 7037, 'NAICS', 2038, 811, 2, 2046, 811121, 5 FROM dual
UNION ALL SELECT 7038, 'SIC', 4309, 40, 1, 3653, 4499, 4 FROM dual
UNION ALL SELECT 7039, 'NAICS', 1635, 54121, 4, 1639, 541219, 5 FROM dual
UNION ALL SELECT 7040, 'NAICS', 1569, 53, 1, 1621, 5331, 3 FROM dual
UNION ALL SELECT 7041, 'SIC', 4109, 7690, 3, 4112, 7699, 4 FROM dual
UNION ALL SELECT 7042, 'SEC', 2623, 5600, 2, 2628, 5660, 3 FROM dual
UNION ALL SELECT 7043, 'NAICS', 1404, 5111, 3, 1406, 51111, 4 FROM dual
UNION ALL SELECT 7044, 'SIC', 4308, 20, 1, 3578, 3911, 4 FROM dual
UNION ALL SELECT 7045, 'NAICS', 205, 23, 1, 225, 23713, 4 FROM dual
UNION ALL SELECT 7046, 'SIC', 3170, 2620, 3, 3171, 2621, 4 FROM dual
UNION ALL SELECT 7047, 'SEC', 2555, 4920, 3, 2557, 4923, 4 FROM dual
UNION ALL SELECT 7048, 'SIC', 3144, 2490, 3, 3147, 2499, 4 FROM dual
UNION ALL SELECT 7049, 'SIC', 4313, 70, 1, 4047, 7322, 4 FROM dual
UNION ALL SELECT 7050, 'NAICS', 1480, 52, 1, 1483, 521110, 5 FROM dual
UNION ALL SELECT 7051, 'NAICS', 2068, 81143, 4, 2067, 811430, 5 FROM dual
UNION ALL SELECT 7052, 'NAICS', 1466, 518, 2, 1467, 5182, 3 FROM dual
UNION ALL SELECT 7053, 'SIC', 3620, 4200, 2, 3631, 4230, 3 FROM dual
UNION ALL SELECT 7054, 'SIC', 3282, 3100, 2, 3299, 3190, 3 FROM dual
UNION ALL SELECT 7055, 'SIC', 3362, 3360, 3, 3364, 3364, 4 FROM dual
UNION ALL SELECT 7056, 'NAICS', 1514, 5231, 3, 1517, 523120, 5 FROM dual
UNION ALL SELECT 7057, 'SIC', 4310, 50, 1, 3742, 5064, 4 FROM dual
UNION ALL SELECT 7058, 'SIC', 4312, 60, 1, 3987, 6540, 3 FROM dual
UNION ALL SELECT 7059, 'NAICS', 2020, 722, 2, 2022, 722310, 5 FROM dual
UNION ALL SELECT 7060, 'SEC', 2577, 5050, 3, 2578, 5051, 4 FROM dual
UNION ALL SELECT 7061, 'SIC', 4235, 8740, 3, 4238, 8743, 4 FROM dual
UNION ALL SELECT 7062, 'NAICS', 2037, 81, 1, 2064, 811412, 5 FROM dual
UNION ALL SELECT 7063, 'NAICS', 144, 2122, 3, 146, 21221, 4 FROM dual
UNION ALL SELECT 7064, 'SIC', 4306, 10, 1, 2897, 1081, 4 FROM dual
UNION ALL SELECT 7065, 'SIC', 4033, 7250, 3, 4034, 7251, 4 FROM dual
UNION ALL SELECT 7066, 'SIC', 4306, 10, 1, 2891, 1040, 3 FROM dual
UNION ALL SELECT 7067, 'NAICS', 1979, 713, 2, 1984, 71312, 4 FROM dual
UNION ALL SELECT 7068, 'SIC', 3992, 6700, 2, 4001, 6733, 4 FROM dual
UNION ALL SELECT 7069, 'SEC', 2791, 20, 1, 2259, 2100, 2 FROM dual
UNION ALL SELECT 7070, 'SIC', 4312, 60, 1, 3954, 6231, 4 FROM dual
UNION ALL SELECT 7071, 'SIC', 3948, 6200, 2, 3957, 6289, 4 FROM dual
UNION ALL SELECT 7072, 'NAICS', 1624, 54, 1, 1697, 54184, 4 FROM dual
UNION ALL SELECT 7073, 'SEC', 2476, 3800, 2, 2496, 3860, 3 FROM dual
UNION ALL SELECT 7074, 'SIC', 4046, 7320, 3, 4047, 7322, 4 FROM dual
UNION ALL SELECT 7075, 'SEC', 2793, 50, 1, 2580, 5063, 4 FROM dual
UNION ALL SELECT 7076, 'NAICS', 1, 11, 1, 50, 111940, 5 FROM dual
UNION ALL SELECT 7077, 'NAICS', 1570, 531, 2, 1585, 531311, 5 FROM dual
UNION ALL SELECT 7078, 'NAICS', 930, 42, 1, 1077, 424910, 5 FROM dual
UNION ALL SELECT 7079, 'SIC', 4310, 50, 1, 3717, 5012, 4 FROM dual
UNION ALL SELECT 7080, 'NAICS', 1590, 53139, 4, 1589, 531390, 5 FROM dual
UNION ALL SELECT 7081, 'SIC', 4308, 20, 1, 3585, 3944, 4 FROM dual
UNION ALL SELECT 7082, 'NAICS', 2020, 722, 2, 2028, 7224, 3 FROM dual
UNION ALL SELECT 7083, 'SIC', 4313, 70, 1, 4196, 8331, 4 FROM dual
UNION ALL SELECT 7084, 'NAICS', 1850, 62, 1, 1938, 62431, 4 FROM dual
UNION ALL SELECT 7085, 'SIC', 4307, 15, 1, 2950, 1611, 4 FROM dual
UNION ALL SELECT 7086, 'SIC', 4040, 7300, 2, 4049, 7330, 3 FROM dual
UNION ALL SELECT 7087, 'NAICS', 1, 11, 1, 118, 1151, 3 FROM dual
UNION ALL SELECT 7088, 'NAICS', 172, 213, 2, 176, 213112, 5 FROM dual
UNION ALL SELECT 7089, 'SEC', 2709, 7000, 2, 2710, 7010, 3 FROM dual
UNION ALL SELECT 7090, 'SIC', 3716, 5010, 3, 3718, 5013, 4 FROM dual
UNION ALL SELECT 7091, 'SIC', 4256, 9200, 2, 4257, 9210, 3 FROM dual
UNION ALL SELECT 7092, 'SIC', 4308, 20, 1, 3460, 3569, 4 FROM dual
UNION ALL SELECT 7093, 'SEC', 2794, 52, 1, 2616, 5399, 4 FROM dual
UNION ALL SELECT 7094, 'SIC', 4305, 1, 1, 2802, 115, 4 FROM dual
UNION ALL SELECT 7095, 'SIC', 3877, 5730, 3, 3881, 5736, 4 FROM dual
UNION ALL SELECT 7096, 'NAICS', 1812, 61, 1, 1845, 611692, 5 FROM dual
UNION ALL SELECT 7097, 'NAICS', 2093, 8129, 3, 2101, 812990, 5 FROM dual
UNION ALL SELECT 7098, 'SEC', 2792, 40, 1, 2558, 4924, 4 FROM dual
UNION ALL SELECT 7099, 'SEC', 2753, 8000, 2, 2759, 8062, 4 FROM dual
UNION ALL SELECT 7100, 'SIC', 3618, 4170, 3, 3619, 4173, 4 FROM dual
UNION ALL SELECT 7101, 'SIC', 3955, 6280, 3, 3957, 6289, 4 FROM dual
UNION ALL SELECT 7102, 'SEC', 2349, 3080, 3, 2352, 3089, 4 FROM dual
UNION ALL SELECT 7103, 'NAICS', 930, 42, 1, 944, 423220, 5 FROM dual
UNION ALL SELECT 7104, 'NAICS', 1624, 54, 1, 1631, 54119, 4 FROM dual
UNION ALL SELECT 7105, 'NAICS', 1792, 562, 2, 1800, 562211, 5 FROM dual
UNION ALL SELECT 7106, 'SIC', 3083, 2300, 2, 3111, 2385, 4 FROM dual
UNION ALL SELECT 7107, 'NAICS', 205, 23, 1, 236, 2381, 3 FROM dual
UNION ALL SELECT 7108, 'SIC', 2847, 700, 2, 2859, 751, 4 FROM dual
UNION ALL SELECT 7109, 'NAICS', 930, 42, 1, 1036, 424410, 5 FROM dual
UNION ALL SELECT 7110, 'SIC', 4314, 90, 1, 4290, 9631, 4 FROM dual
UNION ALL SELECT 7111, 'SEC', 2791, 20, 1, 2248, 2033, 4 FROM dual
UNION ALL SELECT 7112, 'SIC', 4308, 20, 1, 3461, 3570, 3 FROM dual
UNION ALL SELECT 7113, 'SIC', 4312, 60, 1, 3936, 6099, 4 FROM dual
UNION ALL SELECT 7114, 'NAICS', 1480, 52, 1, 1565, 525920, 5 FROM dual
UNION ALL SELECT 7115, 'SIC', 3737, 5050, 3, 3738, 5051, 4 FROM dual
UNION ALL SELECT 7116, 'SIC', 4308, 20, 1, 3214, 2800, 2 FROM dual
UNION ALL SELECT 7117, 'SIC', 3649, 4490, 3, 3651, 4492, 4 FROM dual
UNION ALL SELECT 7118, 'NAICS', 1402, 51, 1, 1467, 5182, 3 FROM dual
UNION ALL SELECT 7119, 'SEC', 2670, 6200, 2, 2671, 6210, 3 FROM dual
UNION ALL SELECT 7120, 'NAICS', 1640, 5413, 3, 1643, 541320, 5 FROM dual
UNION ALL SELECT 7121, 'NAICS', 931, 423, 2, 973, 423520, 5 FROM dual
UNION ALL SELECT 7122, 'NAICS', 2020, 722, 2, 2023, 72231, 4 FROM dual
UNION ALL SELECT 7123, 'SIC', 4308, 20, 1, 3211, 2790, 3 FROM dual
UNION ALL SELECT 7124, 'SIC', 4311, 52, 1, 3895, 5942, 4 FROM dual
UNION ALL SELECT 7125, 'NAICS', 1868, 62139, 4, 1869, 621391, 5 FROM dual
UNION ALL SELECT 7126, 'SEC', 2713, 7300, 2, 2722, 7360, 3 FROM dual
UNION ALL SELECT 7127, 'SIC', 4314, 90, 1, 4279, 9511, 4 FROM dual
UNION ALL SELECT 7128, 'SIC', 4312, 60, 1, 3935, 6091, 4 FROM dual
UNION ALL SELECT 7129, 'SIC', 4162, 8060, 3, 4163, 8062, 4 FROM dual
UNION ALL SELECT 7130, 'SIC', 4192, 8300, 2, 4201, 8390, 3 FROM dual
UNION ALL SELECT 7131, 'SIC', 2798, 100, 2, 2810, 139, 4 FROM dual
UNION ALL SELECT 7132, 'NAICS', 1404, 5111, 3, 1413, 51119, 4 FROM dual
UNION ALL SELECT 7133, 'NAICS', 1569, 53, 1, 1606, 532291, 5 FROM dual
UNION ALL SELECT 7134, 'SIC', 2847, 700, 2, 2863, 762, 4 FROM dual
UNION ALL SELECT 7135, 'SEC', 2331, 2850, 3, 2332, 2851, 4 FROM dual
UNION ALL SELECT 7136, 'SIC', 2955, 1700, 2, 2969, 1760, 3 FROM dual
UNION ALL SELECT 7137, 'SEC', 2591, 5100, 2, 2594, 5122, 4 FROM dual
UNION ALL SELECT 7138, 'SIC', 4120, 7830, 3, 4121, 7832, 4 FROM dual
UNION ALL SELECT 7139, 'SIC', 3124, 2400, 2, 3134, 2435, 4 FROM dual
UNION ALL SELECT 7140, 'NAICS', 1850, 62, 1, 1866, 621340, 5 FROM dual
UNION ALL SELECT 7141, 'NAICS', 2048, 81119, 4, 2049, 811191, 5 FROM dual
UNION ALL SELECT 7142, 'SEC', 2536, 4700, 2, 2538, 4731, 4 FROM dual
UNION ALL SELECT 7143, 'SEC', 2568, 5000, 2, 2589, 5094, 4 FROM dual
UNION ALL SELECT 7144, 'NAICS', 1413, 51119, 4, 1414, 511191, 5 FROM dual
UNION ALL SELECT 7145, 'NAICS', 1942, 71, 1, 1974, 71212, 4 FROM dual
UNION ALL SELECT 7146, 'SIC', 4314, 90, 1, 4270, 9411, 4 FROM dual
UNION ALL SELECT 7147, 'NAICS', 930, 42, 1, 934, 42311, 4 FROM dual
UNION ALL SELECT 7148, 'SIC', 3804, 5200, 2, 3813, 5270, 3 FROM dual
UNION ALL SELECT 7149, 'NAICS', 993, 42381, 4, 992, 423810, 5 FROM dual
UNION ALL SELECT 7150, 'SIC', 3115, 2390, 3, 3116, 2391, 4 FROM dual
UNION ALL SELECT 7151, 'SIC', 4241, 8800, 2, 4242, 8810, 3 FROM dual
UNION ALL SELECT 7152, 'NAICS', 930, 42, 1, 1020, 42412, 4 FROM dual
UNION ALL SELECT 7153, 'SIC', 3724, 5030, 3, 3727, 5033, 4 FROM dual
UNION ALL SELECT 7154, 'NAICS', 1758, 5615, 3, 1765, 561599, 5 FROM dual
UNION ALL SELECT 7155, 'SIC', 4308, 20, 1, 3597, 3993, 4 FROM dual
UNION ALL SELECT 7156, 'SIC', 3636, 4400, 2, 3644, 4449, 4 FROM dual
UNION ALL SELECT 7157, 'NAICS', 1666, 5415, 3, 1670, 541513, 5 FROM dual
UNION ALL SELECT 7158, 'SIC', 4307, 15, 1, 2956, 1710, 3 FROM dual
UNION ALL SELECT 7159, 'SIC', 3804, 5200, 2, 3807, 5230, 3 FROM dual
UNION ALL SELECT 7160, 'SIC', 3371, 3400, 2, 3375, 3420, 3 FROM dual
UNION ALL SELECT 7161, 'NAICS', 182, 2211, 3, 183, 22111, 4 FROM dual
UNION ALL SELECT 7162, 'NAICS', 1554, 525, 2, 1564, 52591, 4 FROM dual
UNION ALL SELECT 7163, 'SEC', 2795, 60, 1, 2677, 6300, 2 FROM dual
UNION ALL SELECT 7164, 'SIC', 4314, 90, 1, 4280, 9512, 4 FROM dual
UNION ALL SELECT 7165, 'SEC', 2795, 60, 1, 2672, 6211, 4 FROM dual
UNION ALL SELECT 7166, 'SIC', 2983, 2010, 3, 2984, 2011, 4 FROM dual
UNION ALL SELECT 7167, 'SIC', 4305, 1, 1, 2856, 741, 4 FROM dual
UNION ALL SELECT 7168, 'NAICS', 1015, 424, 2, 1050, 424480, 5 FROM dual
UNION ALL SELECT 7169, 'NAICS', 1942, 71, 1, 1962, 71132, 4 FROM dual
UNION ALL SELECT 7170, 'SIC', 3977, 6500, 2, 3978, 6510, 3 FROM dual
UNION ALL SELECT 7171, 'SEC', 2617, 5400, 2, 2618, 5410, 3 FROM dual
UNION ALL SELECT 7172, 'SIC', 4049, 7330, 3, 4050, 7331, 4 FROM dual
UNION ALL SELECT 7173, 'SEC', 2670, 6200, 2, 2672, 6211, 4 FROM dual
UNION ALL SELECT 7174, 'NAICS', 2038, 811, 2, 2067, 811430, 5 FROM dual
UNION ALL SELECT 7175, 'SEC', 2240, 2000, 2, 2253, 2070, 3 FROM dual
UNION ALL SELECT 7176, 'NAICS', 2032, 72251, 4, 2035, 722514, 5 FROM dual
UNION ALL SELECT 7177, 'SEC', 2539, 4800, 2, 2542, 4813, 4 FROM dual
UNION ALL SELECT 7178, 'SIC', 4308, 20, 1, 3551, 3799, 4 FROM dual
UNION ALL SELECT 7179, 'SIC', 4029, 7230, 3, 4030, 7231, 4 FROM dual
UNION ALL SELECT 7180, 'SIC', 4281, 9530, 3, 4282, 9531, 4 FROM dual
UNION ALL SELECT 7181, 'NAICS', 138, 212, 2, 151, 212231, 5 FROM dual
UNION ALL SELECT 7182, 'NAICS', 2166, 923, 2, 2168, 923110, 5 FROM dual
UNION ALL SELECT 7183, 'SEC', 2791, 20, 1, 2276, 2400, 2 FROM dual
UNION ALL SELECT 7184, 'NAICS', 1485, 522, 2, 1491, 522130, 5 FROM dual
UNION ALL SELECT 7185, 'NAICS', 1928, 6242, 3, 1935, 62423, 4 FROM dual
UNION ALL SELECT 7186, 'SIC', 3426, 3530, 3, 3428, 3532, 4 FROM dual
UNION ALL SELECT 7187, 'SIC', 4297, 9700, 2, 4299, 9711, 4 FROM dual
UNION ALL SELECT 7188, 'NAICS', 1403, 511, 2, 1417, 511210, 5 FROM dual
UNION ALL SELECT 7189, 'SIC', 4309, 40, 1, 3707, 4950, 3 FROM dual
UNION ALL SELECT 7190, 'SEC', 2713, 7300, 2, 2733, 7384, 4 FROM dual
UNION ALL SELECT 7191, 'SIC', 4308, 20, 1, 3284, 3111, 4 FROM dual
UNION ALL SELECT 7192, 'NAICS', 1570, 531, 2, 1586, 531312, 5 FROM dual
UNION ALL SELECT 7193, 'NAICS', 1726, 561, 2, 1740, 56133, 4 FROM dual
UNION ALL SELECT 7194, 'NAICS', 1942, 71, 1, 1988, 713290, 5 FROM dual
UNION ALL SELECT 7195, 'SEC', 2626, 5650, 3, 2627, 5651, 4 FROM dual
UNION ALL SELECT 7196, 'SIC', 4089, 7530, 3, 4092, 7534, 4 FROM dual
UNION ALL SELECT 7197, 'SIC', 4310, 50, 1, 3757, 5091, 4 FROM dual
UNION ALL SELECT 7198, 'SIC', 4224, 8710, 3, 4225, 8711, 4 FROM dual
UNION ALL SELECT 7199, 'NAICS', 1495, 5222, 3, 1504, 522294, 5 FROM dual
UNION ALL SELECT 7200, 'NAICS', 1513, 523, 2, 1528, 52391, 4 FROM dual
UNION ALL SELECT 7201, 'SEC', 2796, 70, 1, 2745, 7830, 3 FROM dual
UNION ALL SELECT 7202, 'SIC', 3296, 3170, 3, 3298, 3172, 4 FROM dual
UNION ALL SELECT 7203, 'SEC', 2795, 60, 1, 2671, 6210, 3 FROM dual
UNION ALL SELECT 7204, 'SIC', 4308, 20, 1, 3321, 3269, 4 FROM dual
UNION ALL SELECT 7205, 'SEC', 2500, 3900, 2, 2506, 3942, 4 FROM dual
UNION ALL SELECT 7206, 'NAICS', 1689, 5418, 3, 1692, 541820, 5 FROM dual
UNION ALL SELECT 7207, 'SEC', 2738, 7800, 2, 2739, 7810, 3 FROM dual
UNION ALL SELECT 7208, 'NAICS', 1402, 51, 1, 1443, 5151, 3 FROM dual
UNION ALL SELECT 7209, 'NAICS', 1420, 5121, 3, 1423, 512120, 5 FROM dual
UNION ALL SELECT 7210, 'SIC', 4312, 60, 1, 4003, 6792, 4 FROM dual
UNION ALL SELECT 7211, 'SIC', 4308, 20, 1, 2996, 2034, 4 FROM dual
UNION ALL SELECT 7212, 'NAICS', 1859, 6213, 3, 1860, 621310, 5 FROM dual
UNION ALL SELECT 7213, 'NAICS', 930, 42, 1, 1015, 424, 2 FROM dual
UNION ALL SELECT 7214, 'NAICS', 1612, 5324, 3, 1619, 53249, 4 FROM dual
UNION ALL SELECT 7215, 'SIC', 4308, 20, 1, 3325, 3273, 4 FROM dual
UNION ALL SELECT 7216, 'NAICS', 1625, 541, 2, 1699, 54185, 4 FROM dual
UNION ALL SELECT 7217, 'NAICS', 1942, 71, 1, 2001, 713990, 5 FROM dual
UNION ALL SELECT 7218, 'SIC', 4314, 90, 1, 4298, 9710, 3 FROM dual
UNION ALL SELECT 7219, 'NAICS', 1547, 5242, 3, 1552, 524292, 5 FROM dual
UNION ALL SELECT 7220, 'NAICS', 2182, 925, 2, 2183, 9251, 3 FROM dual
UNION ALL SELECT 7221, 'NAICS', 98, 113, 2, 99, 1131, 3 FROM dual
UNION ALL SELECT 7222, 'NAICS', 930, 42, 1, 960, 423430, 5 FROM dual
UNION ALL SELECT 7223, 'SIC', 4305, 1, 1, 2823, 190, 3 FROM dual
UNION ALL SELECT 7224, 'NAICS', 1536, 524, 2, 1547, 5242, 3 FROM dual
UNION ALL SELECT 7225, 'NAICS', 1859, 6213, 3, 1862, 621320, 5 FROM dual
UNION ALL SELECT 7226, 'SIC', 4307, 15, 1, 2977, 1793, 4 FROM dual
UNION ALL SELECT 7227, 'SIC', 2982, 2000, 2, 2997, 2035, 4 FROM dual
UNION ALL SELECT 7228, 'SIC', 3837, 5500, 2, 3852, 5590, 3 FROM dual
UNION ALL SELECT 7229, 'NAICS', 1402, 51, 1, 1422, 51211, 4 FROM dual
UNION ALL SELECT 7230, 'SEC', 2793, 50, 1, 2571, 5020, 3 FROM dual
UNION ALL SELECT 7231, 'SIC', 4308, 20, 1, 3348, 3325, 4 FROM dual
UNION ALL SELECT 7232, 'SIC', 3769, 5130, 3, 3770, 5131, 4 FROM dual
UNION ALL SELECT 7233, 'NAICS', 2167, 9231, 3, 2174, 923140, 5 FROM dual
UNION ALL SELECT 7234, 'SIC', 4308, 20, 1, 3504, 3650, 3 FROM dual
UNION ALL SELECT 7235, 'NAICS', 930, 42, 1, 950, 42332, 4 FROM dual
UNION ALL SELECT 7236, 'SIC', 3371, 3400, 2, 3394, 3452, 4 FROM dual
UNION ALL SELECT 7237, 'NAICS', 1942, 71, 1, 1943, 711, 2 FROM dual
UNION ALL SELECT 7238, 'SIC', 4311, 52, 1, 3858, 5621, 4 FROM dual
UNION ALL SELECT 7239, 'NAICS', 2003, 72, 1, 2005, 7211, 3 FROM dual
UNION ALL SELECT 7240, 'SIC', 4305, 1, 1, 2868, 800, 2 FROM dual
UNION ALL SELECT 7241, 'SIC', 3555, 3820, 3, 3558, 3823, 4 FROM dual
UNION ALL SELECT 7242, 'NAICS', 132, 21, 1, 172, 213, 2 FROM dual
UNION ALL SELECT 7243, 'NAICS', 1812, 61, 1, 1819, 61121, 4 FROM dual
UNION ALL SELECT 7244, 'NAICS', 1, 11, 1, 36, 1114, 3 FROM dual
UNION ALL SELECT 7245, 'NAICS', 259, 23829, 4, 258, 238290, 5 FROM dual
UNION ALL SELECT 7246, 'SIC', 4308, 20, 1, 3425, 3524, 4 FROM dual
UNION ALL SELECT 7247, 'NAICS', 1403, 511, 2, 1408, 51112, 4 FROM dual
UNION ALL SELECT 7248, 'SEC', 2791, 20, 1, 2510, 3960, 3 FROM dual
UNION ALL SELECT 7249, 'NAICS', 1827, 61142, 4, 1826, 611420, 5 FROM dual
UNION ALL SELECT 7250, 'NAICS', 2039, 8111, 3, 2045, 81112, 4 FROM dual
UNION ALL SELECT 7251, 'SEC', 2791, 20, 1, 2491, 3843, 4 FROM dual
UNION ALL SELECT 7252, 'SIC', 3208, 2780, 3, 3210, 2789, 4 FROM dual
UNION ALL SELECT 7253, 'NAICS', 1943, 711, 2, 1946, 71111, 4 FROM dual
UNION ALL SELECT 7254, 'SEC', 2791, 20, 1, 2301, 2673, 4 FROM dual
UNION ALL SELECT 7255, 'SIC', 4308, 20, 1, 3276, 3084, 4 FROM dual
UNION ALL SELECT 7256, 'SIC', 3480, 3600, 2, 3513, 3672, 4 FROM dual
UNION ALL SELECT 7257, 'NAICS', 1812, 61, 1, 1841, 611630, 5 FROM dual
UNION ALL SELECT 7258, 'NAICS', 1721, 55111, 4, 1722, 551111, 5 FROM dual
UNION ALL SELECT 7259, 'SIC', 3093, 2330, 3, 3095, 2335, 4 FROM dual
UNION ALL SELECT 7260, 'NAICS', 1015, 424, 2, 1054, 4245, 3 FROM dual
UNION ALL SELECT 7261, 'NAICS', 1403, 511, 2, 1416, 5112, 3 FROM dual
UNION ALL SELECT 7262, 'NAICS', 1697, 54184, 4, 1696, 541840, 5 FROM dual
UNION ALL SELECT 7263, 'NAICS', 2061, 8114, 3, 2063, 811411, 5 FROM dual
UNION ALL SELECT 7264, 'SIC', 4089, 7530, 3, 4093, 7536, 4 FROM dual
UNION ALL SELECT 7265, 'NAICS', 1459, 5174, 3, 1460, 517410, 5 FROM dual
UNION ALL SELECT 7266, 'SEC', 2794, 52, 1, 2615, 5390, 3 FROM dual
UNION ALL SELECT 7267, 'SIC', 4308, 20, 1, 3399, 3466, 4 FROM dual
UNION ALL SELECT 7268, 'SIC', 3883, 5810, 3, 3885, 5813, 4 FROM dual
UNION ALL SELECT 7269, 'NAICS', 2135, 92, 1, 2157, 92213, 4 FROM dual
UNION ALL SELECT 7270, 'NAICS', 236, 2381, 3, 249, 238170, 5 FROM dual
UNION ALL SELECT 7271, 'NAICS', 132, 21, 1, 133, 211, 2 FROM dual
UNION ALL SELECT 7272, 'SIC', 3084, 2310, 3, 3085, 2311, 4 FROM dual
UNION ALL SELECT 7273, 'NAICS', 1034, 42434, 4, 1033, 424340, 5 FROM dual
UNION ALL SELECT 7274, 'NAICS', 1726, 561, 2, 1779, 561730, 5 FROM dual
UNION ALL SELECT 7275, 'NAICS', 1873, 62141, 4, 1872, 621410, 5 FROM dual
UNION ALL SELECT 7276, 'SIC', 4308, 20, 1, 3357, 3353, 4 FROM dual
UNION ALL SELECT 7277, 'NAICS', 1851, 621, 2, 1867, 62134, 4 FROM dual
UNION ALL SELECT 7278, 'SIC', 4146, 8000, 2, 4169, 8080, 3 FROM dual
UNION ALL SELECT 7279, 'SIC', 3148, 2500, 2, 3166, 2599, 4 FROM dual
UNION ALL SELECT 7280, 'NAICS', 1480, 52, 1, 1520, 52313, 4 FROM dual
UNION ALL SELECT 7281, 'SIC', 4308, 20, 1, 3365, 3365, 4 FROM dual
UNION ALL SELECT 7282, 'SIC', 3282, 3100, 2, 3296, 3170, 3 FROM dual
UNION ALL SELECT 7283, 'NAICS', 1859, 6213, 3, 1869, 621391, 5 FROM dual
UNION ALL SELECT 7284, 'SIC', 4306, 10, 1, 2899, 1094, 4 FROM dual
UNION ALL SELECT 7285, 'SIC', 4308, 20, 1, 3476, 3593, 4 FROM dual
UNION ALL SELECT 7286, 'SIC', 4314, 90, 1, 4259, 9220, 3 FROM dual
UNION ALL SELECT 7287, 'NAICS', 1792, 562, 2, 1809, 56299, 4 FROM dual
UNION ALL SELECT 7288, 'NAICS', 156, 2123, 3, 163, 212321, 5 FROM dual
UNION ALL SELECT 7289, 'SIC', 4308, 20, 1, 2984, 2011, 4 FROM dual
UNION ALL SELECT 7290, 'SEC', 2792, 40, 1, 2519, 4213, 4 FROM dual
UNION ALL SELECT 7291, 'NAICS', 1625, 541, 2, 1707, 541910, 5 FROM dual
UNION ALL SELECT 7292, 'SIC', 3533, 3720, 3, 3536, 3728, 4 FROM dual
UNION ALL SELECT 7293, 'SIC', 4208, 8600, 2, 4212, 8621, 4 FROM dual
UNION ALL SELECT 7294, 'NAICS', 235, 238, 2, 273, 2389, 3 FROM dual
UNION ALL SELECT 7295, 'NAICS', 1625, 541, 2, 1630, 54112, 4 FROM dual
UNION ALL SELECT 7296, 'NAICS', 1657, 5414, 3, 1665, 54149, 4 FROM dual
UNION ALL SELECT 7297, 'NAICS', 2137, 9211, 3, 2138, 921110, 5 FROM dual
UNION ALL SELECT 7298, 'SIC', 4308, 20, 1, 3575, 3873, 4 FROM dual
UNION ALL SELECT 7299, 'SIC', 4311, 52, 1, 3810, 5251, 4 FROM dual
UNION ALL SELECT 7300, 'NAICS', 1725, 56, 1, 1766, 5616, 3 FROM dual
UNION ALL SELECT 7301, 'NAICS', 117, 115, 2, 119, 11511, 4 FROM dual
UNION ALL SELECT 7302, 'SEC', 2659, 6100, 2, 2668, 6162, 4 FROM dual
UNION ALL SELECT 7303, 'SIC', 4313, 70, 1, 4066, 7371, 4 FROM dual
UNION ALL SELECT 7304, 'SIC', 4037, 7290, 3, 4038, 7291, 4 FROM dual
UNION ALL SELECT 7305, 'SIC', 3744, 5070, 3, 3747, 5075, 4 FROM dual
UNION ALL SELECT 7306, 'SIC', 4313, 70, 1, 4213, 8630, 3 FROM dual
UNION ALL SELECT 7307, 'NAICS', 56, 112, 2, 71, 112320, 5 FROM dual
UNION ALL SELECT 7308, 'NAICS', 1015, 424, 2, 1061, 4246, 3 FROM dual
UNION ALL SELECT 7309, 'SEC', 2791, 20, 1, 2448, 3663, 4 FROM dual
UNION ALL SELECT 7310, 'NAICS', 1, 11, 1, 23, 1113, 3 FROM dual
UNION ALL SELECT 7311, 'SIC', 2901, 1200, 2, 2902, 1220, 3 FROM dual
UNION ALL SELECT 7312, 'NAICS', 1751, 56144, 4, 1750, 561440, 5 FROM dual
UNION ALL SELECT 7313, 'SIC', 4310, 50, 1, 3788, 5160, 3 FROM dual
UNION ALL SELECT 7314, 'SIC', 3167, 2600, 2, 3184, 2674, 4 FROM dual
UNION ALL SELECT 7315, 'NAICS', 1741, 5614, 3, 1752, 561450, 5 FROM dual
UNION ALL SELECT 7316, 'NAICS', 1850, 62, 1, 1881, 6215, 3 FROM dual
UNION ALL SELECT 7317, 'SIC', 4309, 40, 1, 3683, 4813, 4 FROM dual
UNION ALL SELECT 7318, 'SEC', 2677, 6300, 2, 2684, 6331, 4 FROM dual
UNION ALL SELECT 7319, 'SIC', 4308, 20, 1, 3349, 3330, 3 FROM dual
UNION ALL SELECT 7320, 'NAICS', 134, 2111, 3, 135, 21111, 4 FROM dual
UNION ALL SELECT 7321, 'NAICS', 1624, 54, 1, 1687, 541720, 5 FROM dual
UNION ALL SELECT 7322, 'SIC', 4308, 20, 1, 3596, 3991, 4 FROM dual
UNION ALL SELECT 7323, 'SIC', 4305, 1, 1, 2830, 214, 4 FROM dual
UNION ALL SELECT 7324, 'NAICS', 931, 423, 2, 948, 42331, 4 FROM dual
UNION ALL SELECT 7325, 'NAICS', 2037, 81, 1, 2063, 811411, 5 FROM dual
UNION ALL SELECT 7326, 'NAICS', 205, 23, 1, 217, 23622, 4 FROM dual
UNION ALL SELECT 7327, 'NAICS', 1625, 541, 2, 1692, 541820, 5 FROM dual
UNION ALL SELECT 7328, 'SIC', 4308, 20, 1, 3577, 3910, 3 FROM dual
UNION ALL SELECT 7329, 'SIC', 3822, 5400, 2, 3827, 5430, 3 FROM dual
UNION ALL SELECT 7330, 'NAICS', 205, 23, 1, 240, 23812, 4 FROM dual
UNION ALL SELECT 7331, 'SIC', 4308, 20, 1, 3022, 2075, 4 FROM dual
UNION ALL SELECT 7332, 'SIC', 2798, 100, 2, 2809, 134, 4 FROM dual
UNION ALL SELECT 7333, 'SIC', 2847, 700, 2, 2864, 780, 3 FROM dual
UNION ALL SELECT 7334, 'NAICS', 1689, 5418, 3, 1699, 54185, 4 FROM dual
UNION ALL SELECT 7335, 'SIC', 4312, 60, 1, 3947, 6163, 4 FROM dual
UNION ALL SELECT 7336, 'SIC', 2840, 270, 3, 2841, 271, 4 FROM dual
UNION ALL SELECT 7337, 'SIC', 3167, 2600, 2, 3181, 2671, 4 FROM dual
UNION ALL SELECT 7338, 'NAICS', 931, 423, 2, 990, 42374, 4 FROM dual
UNION ALL SELECT 7339, 'NAICS', 1920, 624, 2, 1940, 624410, 5 FROM dual
UNION ALL SELECT 7340, 'NAICS', 1, 11, 1, 127, 115210, 5 FROM dual
UNION ALL SELECT 7341, 'NAICS', 1016, 4241, 3, 1019, 424120, 5 FROM dual
UNION ALL SELECT 7342, 'NAICS', 1495, 5222, 3, 1497, 52221, 4 FROM dual
UNION ALL SELECT 7343, 'NAICS', 1591, 532, 2, 1602, 53222, 4 FROM dual
UNION ALL SELECT 7344, 'NAICS', 1571, 5311, 3, 1574, 531120, 5 FROM dual
UNION ALL SELECT 7345, 'SIC', 4308, 20, 1, 3058, 2241, 4 FROM dual
UNION ALL SELECT 7346, 'SEC', 2791, 20, 1, 2316, 2771, 4 FROM dual
UNION ALL SELECT 7347, 'SEC', 2434, 3600, 2, 2448, 3663, 4 FROM dual
UNION ALL SELECT 7348, 'NAICS', 183, 22111, 4, 191, 221118, 5 FROM dual
UNION ALL SELECT 7349, 'NAICS', 1850, 62, 1, 1882, 62151, 4 FROM dual
UNION ALL SELECT 7350, 'NAICS', 1480, 52, 1, 1501, 522291, 5 FROM dual
UNION ALL SELECT 7351, 'SIC', 2896, 1080, 3, 2897, 1081, 4 FROM dual
UNION ALL SELECT 7352, 'SIC', 4065, 7370, 3, 4068, 7373, 4 FROM dual
UNION ALL SELECT 7353, 'SEC', 2458, 3700, 2, 2468, 3728, 4 FROM dual
UNION ALL SELECT 7354, 'NAICS', 1625, 541, 2, 1627, 541110, 5 FROM dual
UNION ALL SELECT 7355, 'NAICS', 970, 4235, 3, 971, 423510, 5 FROM dual
UNION ALL SELECT 7356, 'NAICS', 1725, 56, 1, 1799, 56221, 4 FROM dual
UNION ALL SELECT 7357, 'SIC', 3083, 2300, 2, 3108, 2380, 3 FROM dual
UNION ALL SELECT 7358, 'NAICS', 1725, 56, 1, 1757, 561499, 5 FROM dual
UNION ALL SELECT 7359, 'NAICS', 118, 1151, 3, 119, 11511, 4 FROM dual
UNION ALL SELECT 7360, 'NAICS', 2135, 92, 1, 2195, 92613, 4 FROM dual
UNION ALL SELECT 7361, 'SIC', 4311, 52, 1, 3892, 5932, 4 FROM dual
UNION ALL SELECT 7362, 'SIC', 3633, 4300, 2, 3635, 4311, 4 FROM dual
UNION ALL SELECT 7363, 'NAICS', 205, 23, 1, 271, 238390, 5 FROM dual
UNION ALL SELECT 7364, 'SEC', 2630, 5700, 2, 2632, 5712, 4 FROM dual
UNION ALL SELECT 7365, 'SIC', 4314, 90, 1, 4281, 9530, 3 FROM dual
UNION ALL SELECT 7366, 'NAICS', 1488, 52211, 4, 1487, 522110, 5 FROM dual
UNION ALL SELECT 7367, 'SEC', 2792, 40, 1, 2533, 4581, 4 FROM dual
UNION ALL SELECT 7368, 'NAICS', 2048, 81119, 4, 2050, 811192, 5 FROM dual
UNION ALL SELECT 7369, 'NAICS', 930, 42, 1, 967, 42346, 4 FROM dual
UNION ALL SELECT 7370, 'NAICS', 975, 4236, 3, 978, 423620, 5 FROM dual
UNION ALL SELECT 7371, 'NAICS', 2122, 81391, 4, 2121, 813910, 5 FROM dual
UNION ALL SELECT 7372, 'SEC', 2790, 15, 1, 2229, 1500, 2 FROM dual
UNION ALL SELECT 7373, 'NAICS', 2176, 924, 2, 2180, 924120, 5 FROM dual
UNION ALL SELECT 7374, 'NAICS', 1462, 5179, 3, 1464, 517911, 5 FROM dual
UNION ALL SELECT 7375, 'SIC', 4125, 7900, 2, 4133, 7940, 3 FROM dual
UNION ALL SELECT 7376, 'SIC', 3337, 3300, 2, 3369, 3398, 4 FROM dual
UNION ALL SELECT 7377, 'NAICS', 2071, 812, 2, 2086, 812310, 5 FROM dual
UNION ALL SELECT 7378, 'SEC', 2302, 2700, 2, 2313, 2760, 3 FROM dual
UNION ALL SELECT 7379, 'SIC', 4308, 20, 1, 3419, 3500, 2 FROM dual
UNION ALL SELECT 7380, 'SIC', 4308, 20, 1, 3449, 3556, 4 FROM dual
UNION ALL SELECT 7381, 'SIC', 4310, 50, 1, 3753, 5085, 4 FROM dual
UNION ALL SELECT 7382, 'SEC', 2424, 3570, 3, 2425, 3571, 4 FROM dual
UNION ALL SELECT 7383, 'SIC', 2850, 720, 3, 2852, 722, 4 FROM dual
UNION ALL SELECT 7384, 'SEC', 2791, 20, 1, 2324, 2833, 4 FROM dual
UNION ALL SELECT 7385, 'NAICS', 192, 22112, 4, 193, 221121, 5 FROM dual
UNION ALL SELECT 7386, 'NAICS', 2037, 81, 1, 2127, 813940, 5 FROM dual
UNION ALL SELECT 7387, 'NAICS', 43, 1119, 3, 52, 11199, 4 FROM dual
UNION ALL SELECT 7388, 'SIC', 4306, 10, 1, 2932, 1474, 4 FROM dual
UNION ALL SELECT 7389, 'SIC', 4311, 52, 1, 3804, 5200, 2 FROM dual
UNION ALL SELECT 7390, 'SEC', 2677, 6300, 2, 2688, 6361, 4 FROM dual
UNION ALL SELECT 7391, 'NAICS', 1541, 52412, 4, 1544, 524128, 5 FROM dual
UNION ALL SELECT 7392, 'NAICS', 3, 1111, 3, 18, 111199, 5 FROM dual
UNION ALL SELECT 7393, 'NAICS', 930, 42, 1, 991, 4238, 3 FROM dual
UNION ALL SELECT 7394, 'NAICS', 2031, 7225, 3, 2032, 72251, 4 FROM dual
UNION ALL SELECT 7395, 'SIC', 3167, 2600, 2, 3180, 2670, 3 FROM dual
UNION ALL SELECT 7396, 'SIC', 4178, 8200, 2, 4191, 8299, 4 FROM dual
UNION ALL SELECT 7397, 'NAICS', 1402, 51, 1, 1466, 518, 2 FROM dual
UNION ALL SELECT 7398, 'NAICS', 1485, 522, 2, 1505, 522298, 5 FROM dual
UNION ALL SELECT 7399, 'NAICS', 1726, 561, 2, 1781, 561740, 5 FROM dual
UNION ALL SELECT 7400, 'SIC', 4308, 20, 1, 3532, 3716, 4 FROM dual
UNION ALL SELECT 7401, 'NAICS', 1625, 541, 2, 1674, 541611, 5 FROM dual
UNION ALL SELECT 7402, 'NAICS', 1402, 51, 1, 1445, 515111, 5 FROM dual
UNION ALL SELECT 7403, 'SIC', 3837, 5500, 2, 3850, 5570, 3 FROM dual
UNION ALL SELECT 7404, 'SEC', 2465, 3720, 3, 2467, 3724, 4 FROM dual
UNION ALL SELECT 7405, 'NAICS', 1625, 541, 2, 1672, 5416, 3 FROM dual
UNION ALL SELECT 7406, 'SEC', 2262, 2200, 2, 2263, 2210, 3 FROM dual
UNION ALL SELECT 7407, 'SIC', 3041, 2100, 2, 3042, 2110, 3 FROM dual
UNION ALL SELECT 7408, 'NAICS', 930, 42, 1, 959, 42342, 4 FROM dual
UNION ALL SELECT 7409, 'NAICS', 1850, 62, 1, 1894, 622, 2 FROM dual
UNION ALL SELECT 7410, 'NAICS', 955, 4234, 3, 965, 42345, 4 FROM dual
UNION ALL SELECT 7411, 'NAICS', 2032, 72251, 4, 2034, 722513, 5 FROM dual
UNION ALL SELECT 7412, 'SIC', 3480, 3600, 2, 3488, 3629, 4 FROM dual
UNION ALL SELECT 7413, 'SIC', 4310, 50, 1, 3734, 5047, 4 FROM dual
UNION ALL SELECT 7414, 'SIC', 3715, 5000, 2, 3720, 5015, 4 FROM dual
UNION ALL SELECT 7415, 'NAICS', 58, 11211, 4, 59, 112111, 5 FROM dual
UNION ALL SELECT 7416, 'NAICS', 1612, 5324, 3, 1616, 532420, 5 FROM dual
UNION ALL SELECT 7417, 'NAICS', 1836, 6116, 3, 1844, 611691, 5 FROM dual
UNION ALL SELECT 7418, 'NAICS', 2038, 811, 2, 2042, 811112, 5 FROM dual
UNION ALL SELECT 7419, 'SIC', 3520, 3690, 3, 3525, 3699, 4 FROM dual
UNION ALL SELECT 7420, 'SIC', 3977, 6500, 2, 3990, 6552, 4 FROM dual
UNION ALL SELECT 7421, 'SIC', 4081, 7500, 2, 4085, 7515, 4 FROM dual
UNION ALL SELECT 7422, 'NAICS', 208, 23611, 4, 210, 236116, 5 FROM dual
UNION ALL SELECT 7423, 'NAICS', 2037, 81, 1, 2088, 812320, 5 FROM dual
UNION ALL SELECT 7424, 'SEC', 2791, 20, 1, 2399, 3460, 3 FROM dual
UNION ALL SELECT 7425, 'SIC', 3977, 6500, 2, 3985, 6530, 3 FROM dual
UNION ALL SELECT 7426, 'SIC', 4139, 7990, 3, 4142, 7993, 4 FROM dual
UNION ALL SELECT 7427, 'SIC', 4308, 20, 1, 3162, 2541, 4 FROM dual
UNION ALL SELECT 7428, 'SEC', 2424, 3570, 3, 2430, 3579, 4 FROM dual
UNION ALL SELECT 7429, 'SEC', 2691, 6400, 2, 2692, 6410, 3 FROM dual
UNION ALL SELECT 7430, 'SIC', 4310, 50, 1, 3767, 5120, 3 FROM dual
UNION ALL SELECT 7431, 'NAICS', 180, 22, 1, 188, 221115, 5 FROM dual
UNION ALL SELECT 7432, 'NAICS', 1404, 5111, 3, 1409, 511130, 5 FROM dual
UNION ALL SELECT 7433, 'NAICS', 1625, 541, 2, 1628, 54111, 4 FROM dual
UNION ALL SELECT 7434, 'SIC', 4308, 20, 1, 3522, 3692, 4 FROM dual
UNION ALL SELECT 7435, 'SIC', 3200, 2750, 3, 3201, 2752, 4 FROM dual
UNION ALL SELECT 7436, 'SEC', 2795, 60, 1, 2652, 6022, 4 FROM dual
UNION ALL SELECT 7437, 'SIC', 3299, 3190, 3, 3300, 3199, 4 FROM dual
UNION ALL SELECT 7438, 'SIC', 3552, 3800, 2, 3556, 3821, 4 FROM dual
UNION ALL SELECT 7439, 'SIC', 3631, 4230, 3, 3632, 4231, 4 FROM dual
UNION ALL SELECT 7440, 'SIC', 3854, 5600, 2, 3858, 5621, 4 FROM dual
UNION ALL SELECT 7441, 'NAICS', 1942, 71, 1, 1991, 713910, 5 FROM dual
UNION ALL SELECT 7442, 'NAICS', 2103, 813, 2, 2107, 8132, 3 FROM dual
UNION ALL SELECT 7443, 'SIC', 4081, 7500, 2, 4094, 7537, 4 FROM dual
UNION ALL SELECT 7444, 'SIC', 3086, 2320, 3, 3087, 2321, 4 FROM dual
UNION ALL SELECT 7445, 'SIC', 3214, 2800, 2, 3244, 2875, 4 FROM dual
UNION ALL SELECT 7446, 'SEC', 2610, 5300, 2, 2614, 5331, 4 FROM dual
UNION ALL SELECT 7447, 'NAICS', 952, 42333, 4, 951, 423330, 5 FROM dual
UNION ALL SELECT 7448, 'SIC', 4308, 20, 1, 2990, 2023, 4 FROM dual
UNION ALL SELECT 7449, 'NAICS', 2085, 8123, 3, 2092, 812332, 5 FROM dual
UNION ALL SELECT 7450, 'SIC', 4308, 20, 1, 3102, 2353, 4 FROM dual
UNION ALL SELECT 7451, 'SIC', 3371, 3400, 2, 3416, 3497, 4 FROM dual
UNION ALL SELECT 7452, 'SIC', 3680, 4800, 2, 3681, 4810, 3 FROM dual
UNION ALL SELECT 7453, 'NAICS', 1820, 6113, 3, 1822, 61131, 4 FROM dual
UNION ALL SELECT 7454, 'SIC', 3261, 3000, 2, 3271, 3069, 4 FROM dual
UNION ALL SELECT 7455, 'SIC', 3636, 4400, 2, 3650, 4491, 4 FROM dual
UNION ALL SELECT 7456, 'SIC', 4311, 52, 1, 3849, 5561, 4 FROM dual
UNION ALL SELECT 7457, 'SIC', 3917, 6000, 2, 3919, 6011, 4 FROM dual
UNION ALL SELECT 7458, 'NAICS', 119, 11511, 4, 124, 115115, 5 FROM dual
UNION ALL SELECT 7459, 'NAICS', 2113, 81331, 4, 2116, 813319, 5 FROM dual
UNION ALL SELECT 7460, 'NAICS', 2188, 926, 2, 2196, 926140, 5 FROM dual
UNION ALL SELECT 7461, 'SEC', 2795, 60, 1, 2673, 6220, 3 FROM dual
UNION ALL SELECT 7462, 'SIC', 3948, 6200, 2, 3950, 6211, 4 FROM dual
UNION ALL SELECT 7463, 'NAICS', 1725, 56, 1, 1783, 561790, 5 FROM dual
UNION ALL SELECT 7464, 'SEC', 2792, 40, 1, 2547, 4833, 4 FROM dual
UNION ALL SELECT 7465, 'SIC', 4314, 90, 1, 4293, 9650, 3 FROM dual
UNION ALL SELECT 7466, 'SIC', 3667, 4700, 2, 3677, 4783, 4 FROM dual
UNION ALL SELECT 7467, 'SIC', 4313, 70, 1, 4074, 7379, 4 FROM dual
UNION ALL SELECT 7468, 'NAICS', 931, 423, 2, 1005, 423910, 5 FROM dual
UNION ALL SELECT 7469, 'SIC', 4305, 1, 1, 2843, 273, 4 FROM dual
UNION ALL SELECT 7470, 'SEC', 2659, 6100, 2, 4324, 6190, 3 FROM dual
UNION ALL SELECT 7471, 'SIC', 3337, 3300, 2, 3362, 3360, 3 FROM dual
UNION ALL SELECT 7472, 'NAICS', 1706, 5419, 3, 1717, 54199, 4 FROM dual
UNION ALL SELECT 7473, 'SIC', 4125, 7900, 2, 4127, 7911, 4 FROM dual
UNION ALL SELECT 7474, 'SEC', 2791, 20, 1, 2366, 3270, 3 FROM dual
UNION ALL SELECT 7475, 'SEC', 2742, 7820, 3, 2744, 7829, 4 FROM dual
UNION ALL SELECT 7476, 'NAICS', 1657, 5414, 3, 1664, 541490, 5 FROM dual
UNION ALL SELECT 7477, 'SIC', 4313, 70, 1, 4070, 7375, 4 FROM dual
UNION ALL SELECT 7478, 'SEC', 2459, 3710, 3, 2464, 3716, 4 FROM dual
UNION ALL SELECT 7479, 'NAICS', 206, 236, 2, 217, 23622, 4 FROM dual
UNION ALL SELECT 7480, 'NAICS', 1785, 5619, 3, 1791, 56199, 4 FROM dual
UNION ALL SELECT 7481, 'NAICS', 2150, 922, 2, 2163, 92216, 4 FROM dual
UNION ALL SELECT 7482, 'SIC', 3934, 6090, 3, 3935, 6091, 4 FROM dual
UNION ALL SELECT 7483, 'SEC', 2795, 60, 1, 4327, 6770, 3 FROM dual
UNION ALL SELECT 7484, 'SIC', 4313, 70, 1, 4230, 8730, 3 FROM dual
UNION ALL SELECT 7485, 'SIC', 4100, 7600, 2, 4101, 7620, 3 FROM dual
UNION ALL SELECT 7486, 'NAICS', 2052, 8112, 3, 2054, 811211, 5 FROM dual
UNION ALL SELECT 7487, 'SIC', 3098, 2340, 3, 3100, 2342, 4 FROM dual
UNION ALL SELECT 7488, 'SIC', 4308, 20, 1, 3502, 3647, 4 FROM dual
UNION ALL SELECT 7489, 'SIC', 3854, 5600, 2, 3868, 5699, 4 FROM dual
UNION ALL SELECT 7490, 'SIC', 4308, 20, 1, 3285, 3130, 3 FROM dual
UNION ALL SELECT 7491, 'NAICS', 1, 11, 1, 56, 112, 2 FROM dual
UNION ALL SELECT 7492, 'SIC', 3696, 4920, 3, 3697, 4922, 4 FROM dual
UNION ALL SELECT 7493, 'NAICS', 930, 42, 1, 1078, 42491, 4 FROM dual
UNION ALL SELECT 7494, 'SIC', 4314, 90, 1, 4262, 9223, 4 FROM dual
UNION ALL SELECT 7495, 'NAICS', 181, 221, 2, 187, 221114, 5 FROM dual
UNION ALL SELECT 7496, 'SEC', 2476, 3800, 2, 2481, 3822, 4 FROM dual
UNION ALL SELECT 7497, 'NAICS', 219, 2371, 3, 225, 23713, 4 FROM dual
UNION ALL SELECT 7498, 'NAICS', 2037, 81, 1, 2101, 812990, 5 FROM dual
UNION ALL SELECT 7499, 'NAICS', 2071, 812, 2, 2090, 81233, 4 FROM dual
UNION ALL SELECT 7500, 'NAICS', 2131, 814, 2, 2134, 81411, 4 FROM dual
UNION ALL SELECT 7501, 'SIC', 4309, 40, 1, 3631, 4230, 3 FROM dual
UNION ALL SELECT 7502, 'NAICS', 28, 11133, 4, 33, 111335, 5 FROM dual
UNION ALL SELECT 7503, 'NAICS', 1766, 5616, 3, 1773, 561622, 5 FROM dual
UNION ALL SELECT 7504, 'SEC', 2699, 6530, 3, 2700, 6531, 4 FROM dual
UNION ALL SELECT 7505, 'SIC', 3301, 3200, 2, 3318, 3262, 4 FROM dual
UNION ALL SELECT 7506, 'SIC', 3214, 2800, 2, 3243, 2874, 4 FROM dual
UNION ALL SELECT 7507, 'NAICS', 132, 21, 1, 173, 2131, 3 FROM dual
UNION ALL SELECT 7508, 'NAICS', 1480, 52, 1, 1505, 522298, 5 FROM dual
UNION ALL SELECT 7509, 'SIC', 4268, 9400, 2, 4272, 9431, 4 FROM dual
UNION ALL SELECT 7510, 'SIC', 4125, 7900, 2, 4140, 7991, 4 FROM dual
UNION ALL SELECT 7511, 'SIC', 3774, 5140, 3, 3777, 5143, 4 FROM dual
UNION ALL SELECT 7512, 'NAICS', 1726, 561, 2, 1751, 56144, 4 FROM dual
UNION ALL SELECT 7513, 'NAICS', 1851, 621, 2, 1873, 62141, 4 FROM dual
UNION ALL SELECT 7514, 'NAICS', 1985, 7132, 3, 1986, 713210, 5 FROM dual
UNION ALL SELECT 7515, 'SIC', 4306, 10, 1, 2906, 1231, 4 FROM dual
UNION ALL SELECT 7516, 'NAICS', 1689, 5418, 3, 1694, 541830, 5 FROM dual
UNION ALL SELECT 7517, 'SIC', 2948, 1600, 2, 2952, 1622, 4 FROM dual
UNION ALL SELECT 7518, 'SEC', 2630, 5700, 2, 2636, 5735, 4 FROM dual
UNION ALL SELECT 7519, 'SEC', 2725, 7370, 3, 2730, 7377, 4 FROM dual
UNION ALL SELECT 7520, 'SEC', 2791, 20, 1, 2460, 3711, 4 FROM dual
UNION ALL SELECT 7521, 'SIC', 2955, 1700, 2, 2981, 1799, 4 FROM dual
UNION ALL SELECT 7522, 'SEC', 2797, 90, 1, 2787, 9721, 4 FROM dual
UNION ALL SELECT 7523, 'NAICS', 1015, 424, 2, 1047, 42446, 4 FROM dual
UNION ALL SELECT 7524, 'SEC', 2458, 3700, 2, 2465, 3720, 3 FROM dual
UNION ALL SELECT 7525, 'SIC', 4308, 20, 1, 3078, 2295, 4 FROM dual
UNION ALL SELECT 7526, 'NAICS', 1402, 51, 1, 1448, 51512, 4 FROM dual
UNION ALL SELECT 7527, 'NAICS', 1526, 5239, 3, 1532, 52393, 4 FROM dual
UNION ALL SELECT 7528, 'SEC', 2677, 6300, 2, 2680, 6320, 3 FROM dual
UNION ALL SELECT 7529, 'SIC', 4308, 20, 1, 3036, 2095, 4 FROM dual
UNION ALL SELECT 7530, 'SIC', 4311, 52, 1, 3860, 5632, 4 FROM dual
UNION ALL SELECT 7531, 'SIC', 4308, 20, 1, 3067, 2260, 3 FROM dual
UNION ALL SELECT 7532, 'SIC', 3338, 3310, 3, 3340, 3313, 4 FROM dual
UNION ALL SELECT 7533, 'NAICS', 205, 23, 1, 268, 23834, 4 FROM dual
UNION ALL SELECT 7534, 'SEC', 2791, 20, 1, 2472, 3750, 3 FROM dual
UNION ALL SELECT 7535, 'SIC', 3190, 2700, 2, 3192, 2711, 4 FROM dual
UNION ALL SELECT 7536, 'SIC', 3934, 6090, 3, 3936, 6099, 4 FROM dual
UNION ALL SELECT 7537, 'NAICS', 1480, 52, 1, 1517, 523120, 5 FROM dual
UNION ALL SELECT 7538, 'SEC', 2568, 5000, 2, 2588, 5090, 3 FROM dual
UNION ALL SELECT 7539, 'SIC', 3511, 3670, 3, 3517, 3677, 4 FROM dual
UNION ALL SELECT 7540, 'NAICS', 138, 212, 2, 154, 212291, 5 FROM dual
UNION ALL SELECT 7541, 'NAICS', 1621, 5331, 3, 1623, 53311, 4 FROM dual
UNION ALL SELECT 7542, 'SEC', 2791, 20, 1, 2502, 3911, 4 FROM dual
UNION ALL SELECT 7543, 'SEC', 2415, 3550, 3, 2416, 3555, 4 FROM dual
UNION ALL SELECT 7544, 'NAICS', 2103, 813, 2, 2104, 8131, 3 FROM dual
UNION ALL SELECT 7545, 'SIC', 4311, 52, 1, 3856, 5611, 4 FROM dual
UNION ALL SELECT 7546, 'SEC', 2591, 5100, 2, 2597, 5141, 4 FROM dual
UNION ALL SELECT 7547, 'SIC', 3948, 6200, 2, 3955, 6280, 3 FROM dual
UNION ALL SELECT 7548, 'NAICS', 1569, 53, 1, 1620, 533, 2 FROM dual
UNION ALL SELECT 7549, 'NAICS', 1631, 54119, 4, 1632, 541191, 5 FROM dual
UNION ALL SELECT 7550, 'SIC', 4308, 20, 1, 3200, 2750, 3 FROM dual
UNION ALL SELECT 7551, 'NAICS', 2005, 7211, 3, 2009, 72112, 4 FROM dual
UNION ALL SELECT 7552, 'NAICS', 2052, 8112, 3, 2056, 811213, 5 FROM dual
UNION ALL SELECT 7553, 'SIC', 3261, 3000, 2, 3269, 3060, 3 FROM dual
UNION ALL SELECT 7554, 'SIC', 4306, 10, 1, 2887, 1020, 3 FROM dual
UNION ALL SELECT 7555, 'SIC', 4308, 20, 1, 3042, 2110, 3 FROM dual
UNION ALL SELECT 7556, 'SIC', 3552, 3800, 2, 3575, 3873, 4 FROM dual
UNION ALL SELECT 7557, 'NAICS', 931, 423, 2, 987, 423730, 5 FROM dual
UNION ALL SELECT 7558, 'NAICS', 1485, 522, 2, 1501, 522291, 5 FROM dual
UNION ALL SELECT 7559, 'SIC', 3645, 4480, 3, 3647, 4482, 4 FROM dual
UNION ALL SELECT 7560, 'NAICS', 2155, 92212, 4, 2154, 922120, 5 FROM dual
UNION ALL SELECT 7561, 'NAICS', 1725, 56, 1, 1731, 561210, 5 FROM dual
UNION ALL SELECT 7562, 'SIC', 4308, 20, 1, 3218, 2816, 4 FROM dual
UNION ALL SELECT 7563, 'SIC', 2799, 110, 3, 2802, 115, 4 FROM dual
UNION ALL SELECT 7564, 'NAICS', 1, 11, 1, 100, 113110, 5 FROM dual
UNION ALL SELECT 7565, 'NAICS', 1719, 551, 2, 1720, 5511, 3 FROM dual
UNION ALL SELECT 7566, 'SEC', 2791, 20, 1, 2381, 3357, 4 FROM dual
UNION ALL SELECT 7567, 'SIC', 2918, 1400, 2, 2935, 1480, 3 FROM dual
UNION ALL SELECT 7568, 'SIC', 3762, 5100, 2, 3784, 5150, 3 FROM dual
UNION ALL SELECT 7569, 'NAICS', 1536, 524, 2, 1541, 52412, 4 FROM dual
UNION ALL SELECT 7570, 'SIC', 3214, 2800, 2, 3240, 2869, 4 FROM dual
UNION ALL SELECT 7571, 'SEC', 2796, 70, 1, 2715, 7311, 4 FROM dual
UNION ALL SELECT 7572, 'SEC', 2791, 20, 1, 2268, 2253, 4 FROM dual
UNION ALL SELECT 7573, 'SIC', 4310, 50, 1, 3796, 5182, 4 FROM dual
UNION ALL SELECT 7574, 'SIC', 3715, 5000, 2, 3727, 5033, 4 FROM dual
UNION ALL SELECT 7575, 'SIC', 2955, 1700, 2, 2975, 1790, 3 FROM dual
UNION ALL SELECT 7576, 'SIC', 4311, 52, 1, 3877, 5730, 3 FROM dual
UNION ALL SELECT 7577, 'SIC', 4308, 20, 1, 3277, 3085, 4 FROM dual
UNION ALL SELECT 7578, 'SIC', 3649, 4490, 3, 3652, 4493, 4 FROM dual
UNION ALL SELECT 7579, 'SIC', 3729, 5040, 3, 3734, 5047, 4 FROM dual
UNION ALL SELECT 7580, 'NAICS', 117, 115, 2, 118, 1151, 3 FROM dual
UNION ALL SELECT 7581, 'NAICS', 2112, 8133, 3, 2116, 813319, 5 FROM dual
UNION ALL SELECT 7582, 'SIC', 4308, 20, 1, 3062, 2253, 4 FROM dual
UNION ALL SELECT 7583, 'NAICS', 1921, 6241, 3, 1923, 62411, 4 FROM dual
UNION ALL SELECT 7584, 'SIC', 2868, 800, 2, 2873, 850, 3 FROM dual
UNION ALL SELECT 7585, 'SIC', 3086, 2320, 3, 3092, 2329, 4 FROM dual
UNION ALL SELECT 7586, 'SIC', 4308, 20, 1, 3339, 3312, 4 FROM dual
UNION ALL SELECT 7587, 'NAICS', 2072, 8121, 3, 2076, 812113, 5 FROM dual
UNION ALL SELECT 7588, 'SIC', 3190, 2700, 2, 3200, 2750, 3 FROM dual
UNION ALL SELECT 7589, 'NAICS', 138, 212, 2, 169, 212392, 5 FROM dual
UNION ALL SELECT 7590, 'SIC', 4309, 40, 1, 3601, 4000, 2 FROM dual
UNION ALL SELECT 7591, 'SIC', 4117, 7820, 3, 4118, 7822, 4 FROM dual
UNION ALL SELECT 7592, 'NAICS', 56, 112, 2, 75, 112340, 5 FROM dual
UNION ALL SELECT 7593, 'SIC', 4313, 70, 1, 4169, 8080, 3 FROM dual
UNION ALL SELECT 7594, 'NAICS', 1850, 62, 1, 1876, 62149, 4 FROM dual
UNION ALL SELECT 7595, 'SEC', 2613, 5330, 3, 2614, 5331, 4 FROM dual
UNION ALL SELECT 7596, 'SEC', 2795, 60, 1, 2689, 6390, 3 FROM dual
UNION ALL SELECT 7597, 'NAICS', 941, 4232, 3, 945, 42322, 4 FROM dual
UNION ALL SELECT 7598, 'NAICS', 2135, 92, 1, 2206, 928110, 5 FROM dual
UNION ALL SELECT 7599, 'NAICS', 2204, 928, 2, 2207, 92811, 4 FROM dual
UNION ALL SELECT 7600, 'SEC', 2342, 3000, 2, 2352, 3089, 4 FROM dual
UNION ALL SELECT 7601, 'NAICS', 1725, 56, 1, 1752, 561450, 5 FROM dual
UNION ALL SELECT 7602, 'NAICS', 2071, 812, 2, 2098, 812922, 5 FROM dual
UNION ALL SELECT 7603, 'SIC', 2798, 100, 2, 2823, 190, 3 FROM dual
UNION ALL SELECT 7604, 'NAICS', 930, 42, 1, 1081, 424930, 5 FROM dual
UNION ALL SELECT 7605, 'NAICS', 1851, 621, 2, 1877, 621491, 5 FROM dual
UNION ALL SELECT 7606, 'SIC', 4308, 20, 1, 3192, 2711, 4 FROM dual
UNION ALL SELECT 7607, 'SIC', 3301, 3200, 2, 3312, 3251, 4 FROM dual
UNION ALL SELECT 7608, 'NAICS', 1, 11, 1, 104, 11321, 4 FROM dual
UNION ALL SELECT 7609, 'SIC', 2925, 1440, 3, 2927, 1446, 4 FROM dual
UNION ALL SELECT 7610, 'SIC', 4247, 9100, 2, 4252, 9130, 3 FROM dual
UNION ALL SELECT 7611, 'SIC', 4307, 15, 1, 2945, 1540, 3 FROM dual
UNION ALL SELECT 7612, 'SIC', 4308, 20, 1, 3373, 3411, 4 FROM dual
UNION ALL SELECT 7613, 'SIC', 4313, 70, 1, 4132, 7933, 4 FROM dual
UNION ALL SELECT 7614, 'NAICS', 1526, 5239, 3, 1531, 523930, 5 FROM dual
UNION ALL SELECT 7615, 'SIC', 4153, 8040, 3, 4155, 8042, 4 FROM dual
UNION ALL SELECT 7616, 'NAICS', 1850, 62, 1, 1911, 623220, 5 FROM dual
UNION ALL SELECT 7617, 'SEC', 2791, 20, 1, 2488, 3840, 3 FROM dual
UNION ALL SELECT 7618, 'SEC', 2791, 20, 1, 2508, 3949, 4 FROM dual
UNION ALL SELECT 7619, 'NAICS', 930, 42, 1, 956, 423410, 5 FROM dual
UNION ALL SELECT 7620, 'NAICS', 2189, 9261, 3, 2193, 92612, 4 FROM dual
UNION ALL SELECT 7621, 'SIC', 3020, 2070, 3, 3022, 2075, 4 FROM dual
UNION ALL SELECT 7622, 'SEC', 2527, 4510, 3, 2528, 4512, 4 FROM dual
UNION ALL SELECT 7623, 'SIC', 3214, 2800, 2, 3223, 2823, 4 FROM dual
UNION ALL SELECT 7624, 'NAICS', 1657, 5414, 3, 1661, 54142, 4 FROM dual
UNION ALL SELECT 7625, 'SIC', 4302, 9900, 2, 4304, 9999, 4 FROM dual
UNION ALL SELECT 7626, 'SIC', 4139, 7990, 3, 4143, 7996, 4 FROM dual
UNION ALL SELECT 7627, 'NAICS', 3, 1111, 3, 4, 111110, 5 FROM dual
UNION ALL SELECT 7628, 'SEC', 2794, 52, 1, 2635, 5734, 4 FROM dual
UNION ALL SELECT 7629, 'SIC', 4308, 20, 1, 3170, 2620, 3 FROM dual
UNION ALL SELECT 7630, 'NAICS', 2071, 812, 2, 2075, 812112, 5 FROM dual
UNION ALL SELECT 7631, 'SIC', 3749, 5080, 3, 3753, 5085, 4 FROM dual
UNION ALL SELECT 7632, 'SIC', 4284, 9600, 2, 4287, 9620, 3 FROM dual
UNION ALL SELECT 7633, 'SIC', 3907, 5980, 3, 3910, 5989, 4 FROM dual
UNION ALL SELECT 7634, 'NAICS', 930, 42, 1, 1001, 42385, 4 FROM dual
UNION ALL SELECT 7635, 'SIC', 3854, 5600, 2, 3860, 5632, 4 FROM dual
UNION ALL SELECT 7636, 'SIC', 4308, 20, 1, 3085, 2311, 4 FROM dual
UNION ALL SELECT 7637, 'NAICS', 132, 21, 1, 150, 21223, 4 FROM dual
UNION ALL SELECT 7638, 'SEC', 2539, 4800, 2, 2545, 4830, 3 FROM dual
UNION ALL SELECT 7639, 'SIC', 4309, 40, 1, 3624, 4214, 4 FROM dual
UNION ALL SELECT 7640, 'SIC', 3993, 6710, 3, 3994, 6712, 4 FROM dual
UNION ALL SELECT 7641, 'NAICS', 1536, 524, 2, 1546, 52413, 4 FROM dual
UNION ALL SELECT 7642, 'SIC', 2982, 2000, 2, 3025, 2079, 4 FROM dual
UNION ALL SELECT 7643, 'NAICS', 1706, 5419, 3, 1710, 541921, 5 FROM dual
UNION ALL SELECT 7644, 'NAICS', 2040, 81111, 4, 2042, 811112, 5 FROM dual
UNION ALL SELECT 7645, 'NAICS', 173, 2131, 3, 179, 213115, 5 FROM dual
UNION ALL SELECT 7646, 'SEC', 2552, 4900, 2, 2553, 4910, 3 FROM dual
UNION ALL SELECT 7647, 'NAICS', 1850, 62, 1, 1926, 624190, 5 FROM dual
UNION ALL SELECT 7648, 'SEC', 2795, 60, 1, 2683, 6330, 3 FROM dual
UNION ALL SELECT 7649, 'SIC', 4308, 20, 1, 3171, 2621, 4 FROM dual
UNION ALL SELECT 7650, 'NAICS', 2071, 812, 2, 2074, 812111, 5 FROM dual
UNION ALL SELECT 7651, 'SIC', 2875, 900, 2, 2883, 971, 4 FROM dual
UNION ALL SELECT 7652, 'NAICS', 183, 22111, 4, 190, 221117, 5 FROM dual
UNION ALL SELECT 7653, 'SIC', 3854, 5600, 2, 3856, 5611, 4 FROM dual
UNION ALL SELECT 7654, 'SIC', 2850, 720, 3, 2851, 721, 4 FROM dual
UNION ALL SELECT 7655, 'NAICS', 1942, 71, 1, 1945, 711110, 5 FROM dual
UNION ALL SELECT 7656, 'SIC', 4313, 70, 1, 4058, 7350, 3 FROM dual
UNION ALL SELECT 7657, 'NAICS', 235, 238, 2, 265, 238330, 5 FROM dual
UNION ALL SELECT 7658, 'NAICS', 1575, 53112, 4, 1574, 531120, 5 FROM dual
UNION ALL SELECT 7659, 'NAICS', 56, 112, 2, 95, 11293, 4 FROM dual
UNION ALL SELECT 7660, 'NAICS', 2135, 92, 1, 2199, 92615, 4 FROM dual
UNION ALL SELECT 7661, 'NAICS', 205, 23, 1, 238, 23811, 4 FROM dual
UNION ALL SELECT 7662, 'NAICS', 1625, 541, 2, 1649, 541350, 5 FROM dual
UNION ALL SELECT 7663, 'NAICS', 1726, 561, 2, 1737, 561320, 5 FROM dual
UNION ALL SELECT 7664, 'SEC', 2539, 4800, 2, 2548, 4840, 3 FROM dual
UNION ALL SELECT 7665, 'SIC', 4311, 52, 1, 3884, 5812, 4 FROM dual
UNION ALL SELECT 7666, 'SIC', 4307, 15, 1, 2944, 1531, 4 FROM dual
UNION ALL SELECT 7667, 'SIC', 3301, 3200, 2, 3311, 3250, 3 FROM dual
UNION ALL SELECT 7668, 'SIC', 3869, 5700, 2, 3873, 5714, 4 FROM dual
UNION ALL SELECT 7669, 'SIC', 4125, 7900, 2, 4141, 7992, 4 FROM dual
UNION ALL SELECT 7670, 'SIC', 3301, 3200, 2, 3310, 3241, 4 FROM dual
UNION ALL SELECT 7671, 'NAICS', 1420, 5121, 3, 1427, 512132, 5 FROM dual
UNION ALL SELECT 7672, 'NAICS', 1536, 524, 2, 1542, 524126, 5 FROM dual
UNION ALL SELECT 7673, 'SEC', 2789, 10, 1, 2228, 1400, 2 FROM dual
UNION ALL SELECT 7674, 'SIC', 4311, 52, 1, 3868, 5699, 4 FROM dual
UNION ALL SELECT 7675, 'SEC', 2731, 7380, 3, 2734, 7389, 4 FROM dual
UNION ALL SELECT 7676, 'NAICS', 1402, 51, 1, 1435, 51222, 4 FROM dual
UNION ALL SELECT 7677, 'SIC', 4308, 20, 1, 3155, 2519, 4 FROM dual
UNION ALL SELECT 7678, 'NAICS', 1726, 561, 2, 1765, 561599, 5 FROM dual
UNION ALL SELECT 7679, 'SIC', 3480, 3600, 2, 3508, 3661, 4 FROM dual
UNION ALL SELECT 7680, 'NAICS', 1780, 56173, 4, 1779, 561730, 5 FROM dual
UNION ALL SELECT 7681, 'NAICS', 1425, 51213, 4, 1426, 512131, 5 FROM dual
UNION ALL SELECT 7682, 'SIC', 4308, 20, 1, 3120, 2395, 4 FROM dual
UNION ALL SELECT 7683, 'SIC', 4308, 20, 1, 2989, 2022, 4 FROM dual
UNION ALL SELECT 7684, 'SIC', 4100, 7600, 2, 4108, 7641, 4 FROM dual
UNION ALL SELECT 7685, 'NAICS', 2038, 811, 2, 2061, 8114, 3 FROM dual
UNION ALL SELECT 7686, 'SIC', 3190, 2700, 2, 3201, 2752, 4 FROM dual
UNION ALL SELECT 7687, 'SEC', 2559, 4930, 3, 2560, 4931, 4 FROM dual
UNION ALL SELECT 7688, 'SIC', 3161, 2540, 3, 3162, 2541, 4 FROM dual
UNION ALL SELECT 7689, 'SIC', 4308, 20, 1, 3471, 3585, 4 FROM dual
UNION ALL SELECT 7690, 'SIC', 4308, 20, 1, 3340, 3313, 4 FROM dual
UNION ALL SELECT 7691, 'NAICS', 1470, 519, 2, 1471, 5191, 3 FROM dual
UNION ALL SELECT 7692, 'SIC', 4305, 1, 1, 2845, 290, 3 FROM dual
UNION ALL SELECT 7693, 'SIC', 2840, 270, 3, 2843, 273, 4 FROM dual
UNION ALL SELECT 7694, 'SEC', 2792, 40, 1, 2554, 4911, 4 FROM dual
UNION ALL SELECT 7695, 'NAICS', 1624, 54, 1, 1691, 54181, 4 FROM dual
UNION ALL SELECT 7696, 'SEC', 2640, 5900, 2, 2645, 5945, 4 FROM dual
UNION ALL SELECT 7697, 'SIC', 4308, 20, 1, 3072, 2273, 4 FROM dual
UNION ALL SELECT 7698, 'SIC', 4308, 20, 1, 3517, 3677, 4 FROM dual
UNION ALL SELECT 7699, 'SEC', 2342, 3000, 2, 2346, 3021, 4 FROM dual
UNION ALL SELECT 7700, 'NAICS', 1741, 5614, 3, 1757, 561499, 5 FROM dual
UNION ALL SELECT 7701, 'SEC', 2791, 20, 1, 2477, 3810, 3 FROM dual
UNION ALL SELECT 7702, 'NAICS', 2, 111, 2, 15, 11116, 4 FROM dual
UNION ALL SELECT 7703, 'NAICS', 1015, 424, 2, 1052, 424490, 5 FROM dual
UNION ALL SELECT 7704, 'SIC', 4223, 8700, 2, 4238, 8743, 4 FROM dual
UNION ALL SELECT 7705, 'NAICS', 1536, 524, 2, 1551, 524291, 5 FROM dual
UNION ALL SELECT 7706, 'SIC', 3689, 4840, 3, 3690, 4841, 4 FROM dual
UNION ALL SELECT 7707, 'NAICS', 156, 2123, 3, 171, 212399, 5 FROM dual
UNION ALL SELECT 7708, 'NAICS', 138, 212, 2, 156, 2123, 3 FROM dual
UNION ALL SELECT 7709, 'NAICS', 1625, 541, 2, 1694, 541830, 5 FROM dual
UNION ALL SELECT 7710, 'SIC', 4305, 1, 1, 2880, 920, 3 FROM dual
UNION ALL SELECT 7711, 'NAICS', 1642, 54131, 4, 1641, 541310, 5 FROM dual
UNION ALL SELECT 7712, 'SIC', 3715, 5000, 2, 3736, 5049, 4 FROM dual
UNION ALL SELECT 7713, 'NAICS', 1858, 62121, 4, 1857, 621210, 5 FROM dual
UNION ALL SELECT 7714, 'NAICS', 2135, 92, 1, 2148, 921190, 5 FROM dual
UNION ALL SELECT 7715, 'SIC', 4247, 9100, 2, 4251, 9121, 4 FROM dual
UNION ALL SELECT 7716, 'SIC', 4308, 20, 1, 3117, 2392, 4 FROM dual
UNION ALL SELECT 7717, 'SIC', 3338, 3310, 3, 3339, 3312, 4 FROM dual
UNION ALL SELECT 7718, 'SIC', 4308, 20, 1, 3103, 2360, 3 FROM dual
UNION ALL SELECT 7719, 'SIC', 4308, 20, 1, 3364, 3364, 4 FROM dual
UNION ALL SELECT 7720, 'SIC', 4040, 7300, 2, 4079, 7384, 4 FROM dual
UNION ALL SELECT 7721, 'SIC', 4297, 9700, 2, 4301, 9721, 4 FROM dual
UNION ALL SELECT 7722, 'NAICS', 1625, 541, 2, 1642, 54131, 4 FROM dual
UNION ALL SELECT 7723, 'SEC', 2659, 6100, 2, 2665, 6153, 4 FROM dual
UNION ALL SELECT 7724, 'SIC', 4305, 1, 1, 2841, 271, 4 FROM dual
UNION ALL SELECT 7725, 'NAICS', 1625, 541, 2, 1652, 54136, 4 FROM dual
UNION ALL SELECT 7726, 'SIC', 3050, 2200, 2, 3073, 2280, 3 FROM dual
UNION ALL SELECT 7727, 'SIC', 4308, 20, 1, 3556, 3821, 4 FROM dual
UNION ALL SELECT 7728, 'NAICS', 1480, 52, 1, 1491, 522130, 5 FROM dual
UNION ALL SELECT 7729, 'SEC', 2795, 60, 1, 2697, 6513, 4 FROM dual
UNION ALL SELECT 7730, 'SIC', 4308, 20, 1, 3400, 3469, 4 FROM dual
UNION ALL SELECT 7731, 'SIC', 3246, 2890, 3, 3250, 2895, 4 FROM dual
UNION ALL SELECT 7732, 'SEC', 2276, 2400, 2, 2279, 2430, 3 FROM dual
UNION ALL SELECT 7733, 'NAICS', 1624, 54, 1, 1673, 54161, 4 FROM dual
UNION ALL SELECT 7734, 'NAICS', 1, 11, 1, 96, 112990, 5 FROM dual
UNION ALL SELECT 7735, 'SIC', 4305, 1, 1, 2809, 134, 4 FROM dual
UNION ALL SELECT 7736, 'NAICS', 2167, 9231, 3, 2171, 92312, 4 FROM dual
UNION ALL SELECT 7737, 'NAICS', 1404, 5111, 3, 1412, 51114, 4 FROM dual
UNION ALL SELECT 7738, 'SIC', 4309, 40, 1, 3630, 4226, 4 FROM dual
UNION ALL SELECT 7739, 'NAICS', 2, 111, 2, 47, 11192, 4 FROM dual
UNION ALL SELECT 7740, 'SIC', 4308, 20, 1, 3273, 3081, 4 FROM dual
UNION ALL SELECT 7741, 'SIC', 4193, 8320, 3, 4194, 8322, 4 FROM dual
UNION ALL SELECT 7742, 'NAICS', 2004, 721, 2, 2015, 721211, 5 FROM dual
UNION ALL SELECT 7743, 'NAICS', 970, 4235, 3, 974, 42352, 4 FROM dual
UNION ALL SELECT 7744, 'SEC', 2796, 70, 1, 2759, 8062, 4 FROM dual
UNION ALL SELECT 7745, 'SIC', 4312, 60, 1, 3925, 6030, 3 FROM dual
UNION ALL SELECT 7746, 'SIC', 4313, 70, 1, 4100, 7600, 2 FROM dual
UNION ALL SELECT 7747, 'NAICS', 180, 22, 1, 182, 2211, 3 FROM dual
UNION ALL SELECT 7748, 'NAICS', 930, 42, 1, 980, 423690, 5 FROM dual
UNION ALL SELECT 7749, 'SEC', 2458, 3700, 2, 2475, 3790, 3 FROM dual
UNION ALL SELECT 7750, 'SIC', 3636, 4400, 2, 3643, 4440, 3 FROM dual
UNION ALL SELECT 7751, 'NAICS', 1486, 5221, 3, 1492, 52213, 4 FROM dual
UNION ALL SELECT 7752, 'NAICS', 181, 221, 2, 200, 22131, 4 FROM dual
UNION ALL SELECT 7753, 'SIC', 4041, 7310, 3, 4042, 7311, 4 FROM dual
UNION ALL SELECT 7754, 'NAICS', 1688, 54172, 4, 1687, 541720, 5 FROM dual
UNION ALL SELECT 7755, 'SIC', 2943, 1530, 3, 2944, 1531, 4 FROM dual
UNION ALL SELECT 7756, 'SIC', 3794, 5180, 3, 3796, 5182, 4 FROM dual
UNION ALL SELECT 7757, 'SIC', 4241, 8800, 2, 4243, 8811, 4 FROM dual
UNION ALL SELECT 7758, 'NAICS', 1944, 7111, 3, 1952, 71119, 4 FROM dual
UNION ALL SELECT 7759, 'NAICS', 162, 21232, 4, 163, 212321, 5 FROM dual
UNION ALL SELECT 7760, 'SEC', 2794, 52, 1, 2610, 5300, 2 FROM dual
UNION ALL SELECT 7761, 'SEC', 2796, 70, 1, 2773, 8600, 2 FROM dual
UNION ALL SELECT 7762, 'SIC', 3050, 2200, 2, 3055, 2230, 3 FROM dual
UNION ALL SELECT 7763, 'NAICS', 1942, 71, 1, 1985, 7132, 3 FROM dual
UNION ALL SELECT 7764, 'SEC', 2791, 20, 1, 2478, 3812, 4 FROM dual
UNION ALL SELECT 7765, 'SIC', 4308, 20, 1, 3201, 2752, 4 FROM dual
UNION ALL SELECT 7766, 'SIC', 4247, 9100, 2, 4254, 9190, 3 FROM dual
UNION ALL SELECT 7767, 'SIC', 4312, 60, 1, 3973, 6399, 4 FROM dual
UNION ALL SELECT 7768, 'SEC', 2694, 6500, 2, 2696, 6512, 4 FROM dual
UNION ALL SELECT 7769, 'SIC', 4307, 15, 1, 2976, 1791, 4 FROM dual
UNION ALL SELECT 7770, 'SIC', 3804, 5200, 2, 3806, 5211, 4 FROM dual
UNION ALL SELECT 7771, 'NAICS', 1887, 62161, 4, 1886, 621610, 5 FROM dual
UNION ALL SELECT 7772, 'SIC', 3050, 2200, 2, 3052, 2211, 4 FROM dual
UNION ALL SELECT 7773, 'SIC', 3587, 3950, 3, 3591, 3955, 4 FROM dual
UNION ALL SELECT 7774, 'SIC', 4311, 52, 1, 3889, 5920, 3 FROM dual
UNION ALL SELECT 7775, 'NAICS', 56, 112, 2, 92, 112920, 5 FROM dual
UNION ALL SELECT 7776, 'SIC', 4113, 7800, 2, 4122, 7833, 4 FROM dual
UNION ALL SELECT 7777, 'SIC', 3931, 6080, 3, 3933, 6082, 4 FROM dual
UNION ALL SELECT 7778, 'NAICS', 1920, 624, 2, 1939, 6244, 3 FROM dual
UNION ALL SELECT 7779, 'SIC', 3526, 3700, 2, 3548, 3790, 3 FROM dual
UNION ALL SELECT 7780, 'NAICS', 1, 11, 1, 108, 114, 2 FROM dual
UNION ALL SELECT 7781, 'NAICS', 1672, 5416, 3, 1676, 541613, 5 FROM dual
UNION ALL SELECT 7782, 'NAICS', 108, 114, 2, 113, 114119, 5 FROM dual
UNION ALL SELECT 7783, 'SEC', 2738, 7800, 2, 2740, 7812, 4 FROM dual
UNION ALL SELECT 7784, 'NAICS', 1442, 515, 2, 1443, 5151, 3 FROM dual
UNION ALL SELECT 7785, 'NAICS', 117, 115, 2, 127, 115210, 5 FROM dual
UNION ALL SELECT 7786, 'NAICS', 1850, 62, 1, 1864, 621330, 5 FROM dual
UNION ALL SELECT 7787, 'SEC', 2793, 50, 1, 2575, 5045, 4 FROM dual
UNION ALL SELECT 7788, 'SIC', 4308, 20, 1, 3301, 3200, 2 FROM dual
UNION ALL SELECT 7789, 'NAICS', 1012, 42394, 4, 1011, 423940, 5 FROM dual
UNION ALL SELECT 7790, 'SIC', 4201, 8390, 3, 4202, 8399, 4 FROM dual
UNION ALL SELECT 7791, 'SEC', 2791, 20, 1, 2440, 3630, 3 FROM dual
UNION ALL SELECT 7792, 'SIC', 3526, 3700, 2, 3543, 3751, 4 FROM dual
UNION ALL SELECT 7793, 'SIC', 4310, 50, 1, 3739, 5052, 4 FROM dual
UNION ALL SELECT 7794, 'SIC', 3958, 6300, 2, 3963, 6324, 4 FROM dual
UNION ALL SELECT 7795, 'SIC', 3252, 2900, 2, 3253, 2910, 3 FROM dual
UNION ALL SELECT 7796, 'SIC', 4313, 70, 1, 4190, 8290, 3 FROM dual
UNION ALL SELECT 7797, 'SIC', 3667, 4700, 2, 3674, 4740, 3 FROM dual
UNION ALL SELECT 7798, 'NAICS', 2135, 92, 1, 2149, 92119, 4 FROM dual
UNION ALL SELECT 7799, 'SIC', 3886, 5900, 2, 3896, 5943, 4 FROM dual
UNION ALL SELECT 7800, 'SIC', 2834, 250, 3, 2836, 252, 4 FROM dual
UNION ALL SELECT 7801, 'SIC', 4308, 20, 1, 3196, 2731, 4 FROM dual
UNION ALL SELECT 7802, 'NAICS', 205, 23, 1, 255, 23821, 4 FROM dual
UNION ALL SELECT 7803, 'NAICS', 1726, 561, 2, 1785, 5619, 3 FROM dual
UNION ALL SELECT 7804, 'SIC', 4308, 20, 1, 3209, 2782, 4 FROM dual
UNION ALL SELECT 7805, 'SIC', 4313, 70, 1, 4033, 7250, 3 FROM dual
UNION ALL SELECT 7806, 'NAICS', 1569, 53, 1, 1590, 53139, 4 FROM dual
UNION ALL SELECT 7807, 'NAICS', 1842, 61163, 4, 1841, 611630, 5 FROM dual
UNION ALL SELECT 7808, 'SIC', 3480, 3600, 2, 3490, 3631, 4 FROM dual
UNION ALL SELECT 7809, 'SIC', 4305, 1, 1, 2876, 910, 3 FROM dual
UNION ALL SELECT 7810, 'SIC', 3893, 5940, 3, 3901, 5948, 4 FROM dual
UNION ALL SELECT 7811, 'SIC', 4308, 20, 1, 3594, 3965, 4 FROM dual
UNION ALL SELECT 7812, 'NAICS', 1625, 541, 2, 1660, 541420, 5 FROM dual
UNION ALL SELECT 7813, 'SIC', 4305, 1, 1, 2875, 900, 2 FROM dual
UNION ALL SELECT 7814, 'SIC', 2909, 1300, 2, 2917, 1389, 4 FROM dual
UNION ALL SELECT 7815, 'NAICS', 133, 211, 2, 136, 211111, 5 FROM dual
UNION ALL SELECT 7816, 'NAICS', 2, 111, 2, 12, 111150, 5 FROM dual
UNION ALL SELECT 7817, 'SIC', 4308, 20, 1, 3454, 3563, 4 FROM dual
UNION ALL SELECT 7818, 'SIC', 2921, 1420, 3, 2923, 1423, 4 FROM dual
UNION ALL SELECT 7819, 'SIC', 3180, 2670, 3, 3182, 2672, 4 FROM dual
UNION ALL SELECT 7820, 'SIC', 2918, 1400, 2, 2919, 1410, 3 FROM dual
UNION ALL SELECT 7821, 'NAICS', 1625, 541, 2, 1638, 541214, 5 FROM dual
UNION ALL SELECT 7822, 'SEC', 2371, 3300, 2, 2372, 3310, 3 FROM dual
UNION ALL SELECT 7823, 'SIC', 3371, 3400, 2, 3398, 3465, 4 FROM dual
UNION ALL SELECT 7824, 'NAICS', 1850, 62, 1, 1863, 62132, 4 FROM dual
UNION ALL SELECT 7825, 'NAICS', 930, 42, 1, 1048, 424470, 5 FROM dual
UNION ALL SELECT 7826, 'NAICS', 1569, 53, 1, 1618, 532490, 5 FROM dual
UNION ALL SELECT 7827, 'SIC', 3480, 3600, 2, 3511, 3670, 3 FROM dual
UNION ALL SELECT 7828, 'SIC', 3362, 3360, 3, 3365, 3365, 4 FROM dual
UNION ALL SELECT 7829, 'SIC', 4310, 50, 1, 3776, 5142, 4 FROM dual
UNION ALL SELECT 7830, 'SIC', 4081, 7500, 2, 4082, 7510, 3 FROM dual
UNION ALL SELECT 7831, 'SIC', 4308, 20, 1, 3232, 2842, 4 FROM dual
UNION ALL SELECT 7832, 'NAICS', 1624, 54, 1, 1657, 5414, 3 FROM dual
UNION ALL SELECT 7833, 'NAICS', 2150, 922, 2, 2160, 922150, 5 FROM dual
UNION ALL SELECT 7834, 'SIC', 3380, 3430, 3, 3382, 3432, 4 FROM dual
UNION ALL SELECT 7835, 'NAICS', 205, 23, 1, 228, 23721, 4 FROM dual
UNION ALL SELECT 7836, 'NAICS', 1480, 52, 1, 1495, 5222, 3 FROM dual
UNION ALL SELECT 7837, 'SIC', 3715, 5000, 2, 3743, 5065, 4 FROM dual
UNION ALL SELECT 7838, 'NAICS', 56, 112, 2, 81, 11241, 4 FROM dual
UNION ALL SELECT 7839, 'SIC', 3190, 2700, 2, 3209, 2782, 4 FROM dual
UNION ALL SELECT 7840, 'NAICS', 1481, 521, 2, 1484, 52111, 4 FROM dual
UNION ALL SELECT 7841, 'NAICS', 1570, 531, 2, 1578, 531190, 5 FROM dual
UNION ALL SELECT 7842, 'NAICS', 1850, 62, 1, 1879, 621493, 5 FROM dual
UNION ALL SELECT 7843, 'NAICS', 1851, 621, 2, 1865, 62133, 4 FROM dual
UNION ALL SELECT 7844, 'SIC', 3190, 2700, 2, 3196, 2731, 4 FROM dual
UNION ALL SELECT 7845, 'SEC', 2790, 15, 1, 2237, 1700, 2 FROM dual
UNION ALL SELECT 7846, 'SIC', 3480, 3600, 2, 3503, 3648, 4 FROM dual
UNION ALL SELECT 7847, 'SIC', 4311, 52, 1, 3898, 5945, 4 FROM dual
UNION ALL SELECT 7848, 'NAICS', 1541, 52412, 4, 1543, 524127, 5 FROM dual
UNION ALL SELECT 7849, 'SEC', 2498, 3870, 3, 2499, 3873, 4 FROM dual
UNION ALL SELECT 7850, 'NAICS', 1725, 56, 1, 1787, 56191, 4 FROM dual
UNION ALL SELECT 7851, 'SIC', 3917, 6000, 2, 3918, 6010, 3 FROM dual
UNION ALL SELECT 7852, 'SIC', 4308, 20, 1, 2988, 2021, 4 FROM dual
UNION ALL SELECT 7853, 'SIC', 3301, 3200, 2, 3326, 3274, 4 FROM dual
UNION ALL SELECT 7854, 'NAICS', 1570, 531, 2, 1571, 5311, 3 FROM dual
UNION ALL SELECT 7855, 'SEC', 2792, 40, 1, 2550, 4890, 3 FROM dual
UNION ALL SELECT 7856, 'SIC', 3272, 3080, 3, 3274, 3082, 4 FROM dual
UNION ALL SELECT 7857, 'SIC', 4309, 40, 1, 3619, 4173, 4 FROM dual
UNION ALL SELECT 7858, 'SIC', 4035, 7260, 3, 4036, 7261, 4 FROM dual
UNION ALL SELECT 7859, 'SIC', 2798, 100, 2, 2814, 171, 4 FROM dual
UNION ALL SELECT 7860, 'NAICS', 1035, 4244, 3, 1043, 42444, 4 FROM dual
UNION ALL SELECT 7861, 'NAICS', 2031, 7225, 3, 2034, 722513, 5 FROM dual
UNION ALL SELECT 7862, 'SIC', 3526, 3700, 2, 3527, 3710, 3 FROM dual
UNION ALL SELECT 7863, 'SEC', 2458, 3700, 2, 2463, 3715, 4 FROM dual
UNION ALL SELECT 7864, 'SIC', 4308, 20, 1, 3581, 3930, 3 FROM dual
UNION ALL SELECT 7865, 'NAICS', 138, 212, 2, 171, 212399, 5 FROM dual
UNION ALL SELECT 7866, 'NAICS', 1763, 56159, 4, 1764, 561591, 5 FROM dual
UNION ALL SELECT 7867, 'SIC', 3507, 3660, 3, 3509, 3663, 4 FROM dual
UNION ALL SELECT 7868, 'SIC', 4310, 50, 1, 3729, 5040, 3 FROM dual
UNION ALL SELECT 7869, 'NAICS', 1, 11, 1, 93, 11292, 4 FROM dual
UNION ALL SELECT 7870, 'NAICS', 1570, 531, 2, 1587, 531320, 5 FROM dual
UNION ALL SELECT 7871, 'NAICS', 2020, 722, 2, 2029, 722410, 5 FROM dual
UNION ALL SELECT 7872, 'NAICS', 1894, 622, 2, 1901, 6223, 3 FROM dual
UNION ALL SELECT 7873, 'NAICS', 2159, 92214, 4, 2158, 922140, 5 FROM dual
UNION ALL SELECT 7874, 'NAICS', 2053, 81121, 4, 2055, 811212, 5 FROM dual
UNION ALL SELECT 7875, 'NAICS', 1026, 4243, 3, 1028, 42431, 4 FROM dual
UNION ALL SELECT 7876, 'NAICS', 1661, 54142, 4, 1660, 541420, 5 FROM dual
UNION ALL SELECT 7877, 'SEC', 2418, 3560, 3, 2422, 3567, 4 FROM dual
UNION ALL SELECT 7878, 'NAICS', 1569, 53, 1, 1612, 5324, 3 FROM dual
UNION ALL SELECT 7879, 'SIC', 3636, 4400, 2, 3642, 4432, 4 FROM dual
UNION ALL SELECT 7880, 'NAICS', 1624, 54, 1, 1626, 5411, 3 FROM dual
UNION ALL SELECT 7881, 'SIC', 3555, 3820, 3, 3561, 3826, 4 FROM dual
UNION ALL SELECT 7882, 'SIC', 3409, 3490, 3, 3414, 3495, 4 FROM dual
UNION ALL SELECT 7883, 'NAICS', 1850, 62, 1, 1935, 62423, 4 FROM dual
UNION ALL SELECT 7884, 'SIC', 4312, 60, 1, 3995, 6719, 4 FROM dual
UNION ALL SELECT 7885, 'SIC', 4178, 8200, 2, 4187, 8243, 4 FROM dual
UNION ALL SELECT 7886, 'NAICS', 1625, 541, 2, 1646, 54133, 4 FROM dual
UNION ALL SELECT 7887, 'NAICS', 955, 4234, 3, 960, 423430, 5 FROM dual
UNION ALL SELECT 7888, 'NAICS', 1402, 51, 1, 1465, 517919, 5 FROM dual
UNION ALL SELECT 7889, 'SIC', 3663, 4610, 3, 3665, 4613, 4 FROM dual
UNION ALL SELECT 7890, 'SIC', 4313, 70, 1, 4188, 8244, 4 FROM dual
UNION ALL SELECT 7891, 'NAICS', 3, 1111, 3, 14, 111160, 5 FROM dual
UNION ALL SELECT 7892, 'SIC', 3419, 3500, 2, 3477, 3594, 4 FROM dual
UNION ALL SELECT 7893, 'NAICS', 1990, 7139, 3, 1992, 71391, 4 FROM dual
UNION ALL SELECT 7894, 'SIC', 4308, 20, 1, 3226, 2833, 4 FROM dual
UNION ALL SELECT 7895, 'NAICS', 1919, 62399, 4, 1918, 623990, 5 FROM dual
UNION ALL SELECT 7896, 'NAICS', 1943, 711, 2, 1956, 711212, 5 FROM dual
UNION ALL SELECT 7897, 'NAICS', 1089, 425, 2, 1091, 425110, 5 FROM dual
UNION ALL SELECT 7898, 'SIC', 4308, 20, 1, 3507, 3660, 3 FROM dual
UNION ALL SELECT 7899, 'SEC', 2791, 20, 1, 2443, 3650, 3 FROM dual
UNION ALL SELECT 7900, 'SIC', 4081, 7500, 2, 4096, 7539, 4 FROM dual
UNION ALL SELECT 7901, 'NAICS', 1080, 42492, 4, 1079, 424920, 5 FROM dual
UNION ALL SELECT 7902, 'NAICS', 1726, 561, 2, 1786, 561910, 5 FROM dual
UNION ALL SELECT 7903, 'SEC', 2418, 3560, 3, 2419, 3561, 4 FROM dual
UNION ALL SELECT 7904, 'NAICS', 1554, 525, 2, 1567, 525990, 5 FROM dual
UNION ALL SELECT 7905, 'NAICS', 2136, 921, 2, 2139, 92111, 4 FROM dual
UNION ALL SELECT 7906, 'SIC', 3576, 3900, 2, 3584, 3942, 4 FROM dual
UNION ALL SELECT 7907, 'NAICS', 1536, 524, 2, 1553, 524298, 5 FROM dual
UNION ALL SELECT 7908, 'SIC', 2826, 210, 3, 2829, 213, 4 FROM dual
UNION ALL SELECT 7909, 'SIC', 4308, 20, 1, 3124, 2400, 2 FROM dual
UNION ALL SELECT 7910, 'SIC', 2982, 2000, 2, 3006, 2047, 4 FROM dual
UNION ALL SELECT 7911, 'SIC', 4310, 50, 1, 3782, 5148, 4 FROM dual
UNION ALL SELECT 7912, 'NAICS', 1853, 62111, 4, 1854, 621111, 5 FROM dual
UNION ALL SELECT 7913, 'SIC', 4308, 20, 1, 3282, 3100, 2 FROM dual
UNION ALL SELECT 7914, 'NAICS', 930, 42, 1, 1023, 4242, 3 FROM dual
UNION ALL SELECT 7915, 'SEC', 2319, 2800, 2, 2321, 2820, 3 FROM dual
UNION ALL SELECT 7916, 'SIC', 2982, 2000, 2, 3013, 2061, 4 FROM dual
UNION ALL SELECT 7917, 'SEC', 2608, 5270, 3, 2609, 5271, 4 FROM dual
UNION ALL SELECT 7918, 'SEC', 2319, 2800, 2, 2332, 2851, 4 FROM dual
UNION ALL SELECT 7919, 'SIC', 4305, 1, 1, 2857, 742, 4 FROM dual
UNION ALL SELECT 7920, 'SEC', 2792, 40, 1, 2515, 4013, 4 FROM dual
UNION ALL SELECT 7921, 'SIC', 3419, 3500, 2, 3429, 3533, 4 FROM dual
UNION ALL SELECT 7922, 'NAICS', 109, 1141, 3, 111, 114111, 5 FROM dual
UNION ALL SELECT 7923, 'SIC', 3444, 3550, 3, 3446, 3553, 4 FROM dual
UNION ALL SELECT 7924, 'NAICS', 1792, 562, 2, 1794, 56211, 4 FROM dual
UNION ALL SELECT 7925, 'SIC', 3626, 4220, 3, 3628, 4222, 4 FROM dual
UNION ALL SELECT 7926, 'NAICS', 1402, 51, 1, 1428, 51219, 4 FROM dual
UNION ALL SELECT 7927, 'NAICS', 156, 2123, 3, 169, 212392, 5 FROM dual
UNION ALL SELECT 7928, 'SIC', 4305, 1, 1, 2865, 781, 4 FROM dual
UNION ALL SELECT 7929, 'NAICS', 19, 1112, 3, 22, 111219, 5 FROM dual
UNION ALL SELECT 7930, 'NAICS', 932, 4231, 3, 936, 42312, 4 FROM dual
UNION ALL SELECT 7931, 'SEC', 2235, 1620, 3, 2236, 1623, 4 FROM dual
UNION ALL SELECT 7932, 'SIC', 3461, 3570, 3, 3467, 3579, 4 FROM dual
UNION ALL SELECT 7933, 'SIC', 3958, 6300, 2, 3970, 6370, 3 FROM dual
UNION ALL SELECT 7934, 'NAICS', 16, 11119, 4, 18, 111199, 5 FROM dual
UNION ALL SELECT 7935, 'SEC', 2434, 3600, 2, 2443, 3650, 3 FROM dual
UNION ALL SELECT 7936, 'SIC', 3958, 6300, 2, 3969, 6361, 4 FROM dual
UNION ALL SELECT 7937, 'NAICS', 2031, 7225, 3, 2035, 722514, 5 FROM dual
UNION ALL SELECT 7938, 'SIC', 3214, 2800, 2, 3216, 2812, 4 FROM dual
UNION ALL SELECT 7939, 'SIC', 3371, 3400, 2, 3392, 3450, 3 FROM dual
UNION ALL SELECT 7940, 'SIC', 4308, 20, 1, 3436, 3542, 4 FROM dual
UNION ALL SELECT 7941, 'SIC', 2847, 700, 2, 2855, 740, 3 FROM dual
UNION ALL SELECT 7942, 'NAICS', 931, 423, 2, 1000, 423850, 5 FROM dual
UNION ALL SELECT 7943, 'SIC', 3033, 2090, 3, 3037, 2096, 4 FROM dual
UNION ALL SELECT 7944, 'NAICS', 1437, 51223, 4, 1436, 512230, 5 FROM dual
UNION ALL SELECT 7945, 'NAICS', 2071, 812, 2, 2078, 812191, 5 FROM dual
UNION ALL SELECT 7946, 'SIC', 3977, 6500, 2, 3984, 6519, 4 FROM dual
UNION ALL SELECT 7947, 'NAICS', 2071, 812, 2, 2082, 81221, 4 FROM dual
UNION ALL SELECT 7948, 'SIC', 3693, 4900, 2, 3703, 4932, 4 FROM dual
UNION ALL SELECT 7949, 'NAICS', 1480, 52, 1, 1527, 523910, 5 FROM dual
UNION ALL SELECT 7950, 'NAICS', 1569, 53, 1, 1601, 532220, 5 FROM dual
UNION ALL SELECT 7951, 'NAICS', 180, 22, 1, 186, 221113, 5 FROM dual
UNION ALL SELECT 7952, 'NAICS', 930, 42, 1, 965, 42345, 4 FROM dual
UNION ALL SELECT 7953, 'NAICS', 930, 42, 1, 1087, 424990, 5 FROM dual
UNION ALL SELECT 7954, 'SEC', 2791, 20, 1, 2471, 3743, 4 FROM dual
UNION ALL SELECT 7955, 'NAICS', 955, 4234, 3, 959, 42342, 4 FROM dual
UNION ALL SELECT 7956, 'NAICS', 1444, 51511, 4, 1446, 515112, 5 FROM dual
UNION ALL SELECT 7957, 'SIC', 4308, 20, 1, 2983, 2010, 3 FROM dual
UNION ALL SELECT 7958, 'NAICS', 98, 113, 2, 106, 113310, 5 FROM dual
UNION ALL SELECT 7959, 'SIC', 3774, 5140, 3, 3775, 5141, 4 FROM dual
UNION ALL SELECT 7960, 'SEC', 2794, 52, 1, 2608, 5270, 3 FROM dual
UNION ALL SELECT 7961, 'NAICS', 1725, 56, 1, 1763, 56159, 4 FROM dual
UNION ALL SELECT 7962, 'SIC', 2798, 100, 2, 2802, 115, 4 FROM dual
UNION ALL SELECT 7963, 'NAICS', 132, 21, 1, 155, 212299, 5 FROM dual
UNION ALL SELECT 7964, 'SEC', 2795, 60, 1, 2660, 6110, 3 FROM dual
UNION ALL SELECT 7965, 'NAICS', 2188, 926, 2, 2197, 92614, 4 FROM dual
UNION ALL SELECT 7966, 'SEC', 2633, 5730, 3, 2635, 5734, 4 FROM dual
UNION ALL SELECT 7967, 'SIC', 4308, 20, 1, 3132, 2431, 4 FROM dual
UNION ALL SELECT 7968, 'SIC', 4040, 7300, 2, 4071, 7376, 4 FROM dual
UNION ALL SELECT 7969, 'NAICS', 1, 11, 1, 46, 111920, 5 FROM dual
UNION ALL SELECT 7970, 'SIC', 4313, 70, 1, 4202, 8399, 4 FROM dual
UNION ALL SELECT 7971, 'SEC', 2791, 20, 1, 2493, 3845, 4 FROM dual
UNION ALL SELECT 7972, 'SIC', 4308, 20, 1, 3118, 2393, 4 FROM dual
UNION ALL SELECT 7973, 'NAICS', 89, 1129, 3, 92, 112920, 5 FROM dual
UNION ALL SELECT 7974, 'NAICS', 1054, 4245, 3, 1056, 42451, 4 FROM dual
UNION ALL SELECT 7975, 'NAICS', 1492, 52213, 4, 1491, 522130, 5 FROM dual
UNION ALL SELECT 7976, 'SIC', 3230, 2840, 3, 3234, 2844, 4 FROM dual
UNION ALL SELECT 7977, 'SIC', 4310, 50, 1, 3715, 5000, 2 FROM dual
UNION ALL SELECT 7978, 'SIC', 3797, 5190, 3, 3800, 5193, 4 FROM dual
UNION ALL SELECT 7979, 'NAICS', 1480, 52, 1, 1490, 52212, 4 FROM dual
UNION ALL SELECT 7980, 'NAICS', 1691, 54181, 4, 1690, 541810, 5 FROM dual
UNION ALL SELECT 7981, 'SIC', 3190, 2700, 2, 3193, 2720, 3 FROM dual
UNION ALL SELECT 7982, 'NAICS', 85, 11251, 4, 88, 112519, 5 FROM dual
UNION ALL SELECT 7983, 'SIC', 3948, 6200, 2, 3956, 6282, 4 FROM dual
UNION ALL SELECT 7984, 'SIC', 3073, 2280, 3, 3075, 2282, 4 FROM dual
UNION ALL SELECT 7985, 'SIC', 4309, 40, 1, 3681, 4810, 3 FROM dual
UNION ALL SELECT 7986, 'SIC', 4075, 7380, 3, 4079, 7384, 4 FROM dual
UNION ALL SELECT 7987, 'NAICS', 930, 42, 1, 1074, 424820, 5 FROM dual
UNION ALL SELECT 7988, 'SIC', 4308, 20, 1, 3393, 3451, 4 FROM dual
UNION ALL SELECT 7989, 'SIC', 3211, 2790, 3, 3213, 2796, 4 FROM dual
UNION ALL SELECT 7990, 'NAICS', 1480, 52, 1, 1512, 52239, 4 FROM dual
UNION ALL SELECT 7991, 'NAICS', 1, 11, 1, 73, 112330, 5 FROM dual
UNION ALL SELECT 7992, 'SIC', 4305, 1, 1, 2810, 139, 4 FROM dual
UNION ALL SELECT 7993, 'NAICS', 1850, 62, 1, 1901, 6223, 3 FROM dual
UNION ALL SELECT 7994, 'SIC', 3869, 5700, 2, 3880, 5735, 4 FROM dual
UNION ALL SELECT 7995, 'SIC', 4306, 10, 1, 2905, 1230, 3 FROM dual
UNION ALL SELECT 7996, 'NAICS', 89, 1129, 3, 95, 11293, 4 FROM dual
UNION ALL SELECT 7997, 'SIC', 4305, 1, 1, 2867, 783, 4 FROM dual
UNION ALL SELECT 7998, 'SIC', 3680, 4800, 2, 3683, 4813, 4 FROM dual
UNION ALL SELECT 7999, 'SIC', 3715, 5000, 2, 3748, 5078, 4 FROM dual
UNION ALL SELECT 8000, 'NAICS', 1, 11, 1, 101, 11311, 4 FROM dual
UNION ALL SELECT 8001, 'SIC', 4313, 70, 1, 4143, 7996, 4 FROM dual
UNION ALL SELECT 8002, 'NAICS', 1812, 61, 1, 1823, 6114, 3 FROM dual
UNION ALL SELECT 8003, 'SIC', 3480, 3600, 2, 3525, 3699, 4 FROM dual
UNION ALL SELECT 8004, 'SIC', 3426, 3530, 3, 3430, 3534, 4 FROM dual
UNION ALL SELECT 8005, 'NAICS', 991, 4238, 3, 1001, 42385, 4 FROM dual
UNION ALL SELECT 8006, 'NAICS', 1514, 5231, 3, 1520, 52313, 4 FROM dual
UNION ALL SELECT 8007, 'NAICS', 2071, 812, 2, 2079, 812199, 5 FROM dual
UNION ALL SELECT 8008, 'SEC', 2356, 3210, 3, 2357, 3211, 4 FROM dual;

INSERT INTO "industry_level" (industry_level_id, industry_classification, ancestor_id, ancestor_code, ancestor_depth, descendant_id, descendant_code, descendant_depth) 
          SELECT 8009, 'NAICS', 1812, 61, 1, 1837, 611610, 5 FROM dual
UNION ALL SELECT 8010, 'NAICS', 180, 22, 1, 193, 221121, 5 FROM dual
UNION ALL SELECT 8011, 'NAICS', 2, 111, 2, 31, 111333, 5 FROM dual
UNION ALL SELECT 8012, 'SEC', 2793, 50, 1, 2569, 5010, 3 FROM dual
UNION ALL SELECT 8013, 'SIC', 3955, 6280, 3, 3956, 6282, 4 FROM dual
UNION ALL SELECT 8014, 'NAICS', 218, 237, 2, 229, 2373, 3 FROM dual
UNION ALL SELECT 8015, 'NAICS', 1726, 561, 2, 1788, 561920, 5 FROM dual
UNION ALL SELECT 8016, 'NAICS', 1813, 611, 2, 1847, 6117, 3 FROM dual
UNION ALL SELECT 8017, 'SIC', 4149, 8020, 3, 4150, 8021, 4 FROM dual
UNION ALL SELECT 8018, 'SEC', 2240, 2000, 2, 2247, 2030, 3 FROM dual
UNION ALL SELECT 8019, 'NAICS', 1966, 7115, 3, 1967, 711510, 5 FROM dual
UNION ALL SELECT 8020, 'NAICS', 1625, 541, 2, 1684, 54171, 4 FROM dual
UNION ALL SELECT 8021, 'SIC', 3077, 2290, 3, 3081, 2298, 4 FROM dual
UNION ALL SELECT 8022, 'SEC', 2659, 6100, 2, 2669, 6163, 4 FROM dual
UNION ALL SELECT 8023, 'SIC', 2902, 1220, 3, 2903, 1221, 4 FROM dual
UNION ALL SELECT 8024, 'NAICS', 1813, 611, 2, 1829, 61143, 4 FROM dual
UNION ALL SELECT 8025, 'SIC', 4311, 52, 1, 3916, 5999, 4 FROM dual
UNION ALL SELECT 8026, 'SEC', 2791, 20, 1, 2331, 2850, 3 FROM dual
UNION ALL SELECT 8027, 'SIC', 3282, 3100, 2, 3290, 3144, 4 FROM dual
UNION ALL SELECT 8028, 'SIC', 4113, 7800, 2, 4117, 7820, 3 FROM dual
UNION ALL SELECT 8029, 'NAICS', 205, 23, 1, 256, 238220, 5 FROM dual
UNION ALL SELECT 8030, 'SIC', 3337, 3300, 2, 3360, 3356, 4 FROM dual
UNION ALL SELECT 8031, 'SIC', 3044, 2120, 3, 3045, 2121, 4 FROM dual
UNION ALL SELECT 8032, 'SIC', 3762, 5100, 2, 3794, 5180, 3 FROM dual
UNION ALL SELECT 8033, 'NAICS', 1695, 54183, 4, 1694, 541830, 5 FROM dual
UNION ALL SELECT 8034, 'NAICS', 2037, 81, 1, 2076, 812113, 5 FROM dual
UNION ALL SELECT 8035, 'NAICS', 1442, 515, 2, 1445, 515111, 5 FROM dual
UNION ALL SELECT 8036, 'NAICS', 1588, 53132, 4, 1587, 531320, 5 FROM dual
UNION ALL SELECT 8037, 'SEC', 2790, 15, 1, 2236, 1623, 4 FROM dual
UNION ALL SELECT 8038, 'SEC', 2791, 20, 1, 2484, 3825, 4 FROM dual
UNION ALL SELECT 8039, 'SEC', 2368, 3280, 3, 2369, 3281, 4 FROM dual
UNION ALL SELECT 8040, 'SIC', 3124, 2400, 2, 3145, 2491, 4 FROM dual
UNION ALL SELECT 8041, 'NAICS', 1, 11, 1, 119, 11511, 4 FROM dual
UNION ALL SELECT 8042, 'NAICS', 213, 2362, 3, 215, 23621, 4 FROM dual
UNION ALL SELECT 8043, 'NAICS', 2004, 721, 2, 2009, 72112, 4 FROM dual
UNION ALL SELECT 8044, 'SIC', 2858, 750, 3, 2859, 751, 4 FROM dual
UNION ALL SELECT 8045, 'NAICS', 260, 2383, 3, 272, 23839, 4 FROM dual
UNION ALL SELECT 8046, 'NAICS', 1943, 711, 2, 1965, 71141, 4 FROM dual
UNION ALL SELECT 8047, 'SIC', 3000, 2040, 3, 3004, 2045, 4 FROM dual
UNION ALL SELECT 8048, 'NAICS', 974, 42352, 4, 973, 423520, 5 FROM dual
UNION ALL SELECT 8049, 'SEC', 2434, 3600, 2, 2440, 3630, 3 FROM dual
UNION ALL SELECT 8050, 'SIC', 3978, 6510, 3, 3979, 6512, 4 FROM dual
UNION ALL SELECT 8051, 'NAICS', 205, 23, 1, 250, 23817, 4 FROM dual
UNION ALL SELECT 8052, 'SIC', 3050, 2200, 2, 3074, 2281, 4 FROM dual
UNION ALL SELECT 8053, 'SIC', 4306, 10, 1, 2915, 1381, 4 FROM dual
UNION ALL SELECT 8054, 'SIC', 4305, 1, 1, 2878, 913, 4 FROM dual
UNION ALL SELECT 8055, 'NAICS', 198, 2213, 3, 200, 22131, 4 FROM dual
UNION ALL SELECT 8056, 'NAICS', 1733, 5613, 3, 1737, 561320, 5 FROM dual
UNION ALL SELECT 8057, 'SIC', 4308, 20, 1, 3193, 2720, 3 FROM dual
UNION ALL SELECT 8058, 'SEC', 2793, 50, 1, 2578, 5051, 4 FROM dual
UNION ALL SELECT 8059, 'NAICS', 1701, 54186, 4, 1700, 541860, 5 FROM dual
UNION ALL SELECT 8060, 'SEC', 2240, 2000, 2, 2254, 2080, 3 FROM dual
UNION ALL SELECT 8061, 'SEC', 2793, 50, 1, 2596, 5140, 3 FROM dual
UNION ALL SELECT 8062, 'NAICS', 1624, 54, 1, 1644, 54132, 4 FROM dual
UNION ALL SELECT 8063, 'SIC', 3103, 2360, 3, 3105, 2369, 4 FROM dual
UNION ALL SELECT 8064, 'SIC', 4310, 50, 1, 3793, 5172, 4 FROM dual
UNION ALL SELECT 8065, 'SEC', 2796, 70, 1, 2748, 7900, 2 FROM dual
UNION ALL SELECT 8066, 'SEC', 2796, 70, 1, 2710, 7010, 3 FROM dual
UNION ALL SELECT 8067, 'NAICS', 1888, 6219, 3, 1890, 62191, 4 FROM dual
UNION ALL SELECT 8068, 'NAICS', 56, 112, 2, 67, 11221, 4 FROM dual
UNION ALL SELECT 8069, 'NAICS', 955, 4234, 3, 967, 42346, 4 FROM dual
UNION ALL SELECT 8070, 'NAICS', 205, 23, 1, 216, 236220, 5 FROM dual
UNION ALL SELECT 8071, 'SIC', 2955, 1700, 2, 2962, 1740, 3 FROM dual
UNION ALL SELECT 8072, 'SIC', 4007, 7000, 2, 4016, 7041, 4 FROM dual
UNION ALL SELECT 8073, 'SIC', 2884, 1000, 2, 2889, 1030, 3 FROM dual
UNION ALL SELECT 8074, 'SIC', 3261, 3000, 2, 3266, 3050, 3 FROM dual
UNION ALL SELECT 8075, 'NAICS', 235, 238, 2, 270, 23835, 4 FROM dual
UNION ALL SELECT 8076, 'SIC', 4308, 20, 1, 3157, 2521, 4 FROM dual
UNION ALL SELECT 8077, 'NAICS', 181, 221, 2, 195, 2212, 3 FROM dual
UNION ALL SELECT 8078, 'NAICS', 2151, 9221, 3, 2164, 922190, 5 FROM dual
UNION ALL SELECT 8079, 'NAICS', 1486, 5221, 3, 1494, 52219, 4 FROM dual
UNION ALL SELECT 8080, 'SIC', 4308, 20, 1, 3541, 3743, 4 FROM dual
UNION ALL SELECT 8081, 'SIC', 3148, 2500, 2, 3158, 2522, 4 FROM dual
UNION ALL SELECT 8082, 'SIC', 3419, 3500, 2, 3433, 3537, 4 FROM dual
UNION ALL SELECT 8083, 'SIC', 4313, 70, 1, 4124, 7841, 4 FROM dual
UNION ALL SELECT 8084, 'SIC', 3977, 6500, 2, 3980, 6513, 4 FROM dual
UNION ALL SELECT 8085, 'SEC', 4336, 99, 1, 4338, 8888, 2 FROM dual
UNION ALL SELECT 8086, 'SIC', 3282, 3100, 2, 3285, 3130, 3 FROM dual
UNION ALL SELECT 8087, 'SEC', 2791, 20, 1, 2483, 3824, 4 FROM dual
UNION ALL SELECT 8088, 'SEC', 2795, 60, 1, 2655, 6035, 4 FROM dual
UNION ALL SELECT 8089, 'SEC', 2659, 6100, 2, 2663, 6141, 4 FROM dual
UNION ALL SELECT 8090, 'SIC', 4100, 7600, 2, 4111, 7694, 4 FROM dual
UNION ALL SELECT 8091, 'NAICS', 1721, 55111, 4, 1723, 551112, 5 FROM dual
UNION ALL SELECT 8092, 'NAICS', 1726, 561, 2, 1775, 561710, 5 FROM dual
UNION ALL SELECT 8093, 'NAICS', 930, 42, 1, 1014, 42399, 4 FROM dual
UNION ALL SELECT 8094, 'SEC', 2792, 40, 1, 2527, 4510, 3 FROM dual
UNION ALL SELECT 8095, 'SEC', 2793, 50, 1, 2601, 5171, 4 FROM dual
UNION ALL SELECT 8096, 'SIC', 3083, 2300, 2, 3110, 2384, 4 FROM dual
UNION ALL SELECT 8097, 'NAICS', 1943, 711, 2, 1962, 71132, 4 FROM dual
UNION ALL SELECT 8098, 'SIC', 4313, 70, 1, 4022, 7215, 4 FROM dual
UNION ALL SELECT 8099, 'SIC', 4307, 15, 1, 2957, 1711, 4 FROM dual
UNION ALL SELECT 8100, 'NAICS', 1851, 621, 2, 1889, 621910, 5 FROM dual
UNION ALL SELECT 8101, 'NAICS', 1850, 62, 1, 1893, 621999, 5 FROM dual
UNION ALL SELECT 8102, 'SEC', 2791, 20, 1, 2327, 2836, 4 FROM dual
UNION ALL SELECT 8103, 'SIC', 3083, 2300, 2, 3112, 2386, 4 FROM dual
UNION ALL SELECT 8104, 'NAICS', 1520, 52313, 4, 1519, 523130, 5 FROM dual
UNION ALL SELECT 8105, 'SEC', 2505, 3940, 3, 2507, 3944, 4 FROM dual
UNION ALL SELECT 8106, 'SEC', 2791, 20, 1, 2358, 3220, 3 FROM dual
UNION ALL SELECT 8107, 'NAICS', 132, 21, 1, 154, 212291, 5 FROM dual
UNION ALL SELECT 8108, 'NAICS', 205, 23, 1, 209, 236115, 5 FROM dual
UNION ALL SELECT 8109, 'NAICS', 2093, 8129, 3, 2102, 81299, 4 FROM dual
UNION ALL SELECT 8110, 'SEC', 2403, 3500, 2, 2418, 3560, 3 FROM dual
UNION ALL SELECT 8111, 'SIC', 4311, 52, 1, 3814, 5271, 4 FROM dual
UNION ALL SELECT 8112, 'SIC', 4308, 20, 1, 3475, 3592, 4 FROM dual
UNION ALL SELECT 8113, 'SIC', 4313, 70, 1, 4140, 7991, 4 FROM dual
UNION ALL SELECT 8114, 'NAICS', 1963, 7114, 3, 1964, 711410, 5 FROM dual
UNION ALL SELECT 8115, 'SEC', 2472, 3750, 3, 2473, 3751, 4 FROM dual
UNION ALL SELECT 8116, 'NAICS', 1514, 5231, 3, 1515, 523110, 5 FROM dual
UNION ALL SELECT 8117, 'SIC', 3794, 5180, 3, 3795, 5181, 4 FROM dual
UNION ALL SELECT 8118, 'NAICS', 1061, 4246, 3, 1062, 424610, 5 FROM dual
UNION ALL SELECT 8119, 'NAICS', 1813, 611, 2, 1836, 6116, 3 FROM dual
UNION ALL SELECT 8120, 'NAICS', 1851, 621, 2, 1858, 62121, 4 FROM dual
UNION ALL SELECT 8121, 'SIC', 4065, 7370, 3, 4071, 7376, 4 FROM dual
UNION ALL SELECT 8122, 'NAICS', 1667, 54151, 4, 1670, 541513, 5 FROM dual
UNION ALL SELECT 8123, 'NAICS', 205, 23, 1, 277, 23899, 4 FROM dual
UNION ALL SELECT 8124, 'SEC', 2796, 70, 1, 2763, 8082, 4 FROM dual
UNION ALL SELECT 8125, 'NAICS', 1, 11, 1, 85, 11251, 4 FROM dual
UNION ALL SELECT 8126, 'NAICS', 1480, 52, 1, 1506, 5223, 3 FROM dual
UNION ALL SELECT 8127, 'NAICS', 1725, 56, 1, 1761, 561520, 5 FROM dual
UNION ALL SELECT 8128, 'NAICS', 2020, 722, 2, 2036, 722515, 5 FROM dual
UNION ALL SELECT 8129, 'SIC', 4308, 20, 1, 3342, 3316, 4 FROM dual
UNION ALL SELECT 8130, 'NAICS', 1402, 51, 1, 1469, 51821, 4 FROM dual
UNION ALL SELECT 8131, 'NAICS', 1683, 5417, 3, 1687, 541720, 5 FROM dual
UNION ALL SELECT 8132, 'NAICS', 2167, 9231, 3, 2173, 92313, 4 FROM dual
UNION ALL SELECT 8133, 'NAICS', 932, 4231, 3, 933, 423110, 5 FROM dual
UNION ALL SELECT 8134, 'NAICS', 1799, 56221, 4, 1801, 562212, 5 FROM dual
UNION ALL SELECT 8135, 'NAICS', 1726, 561, 2, 1762, 56152, 4 FROM dual
UNION ALL SELECT 8136, 'SIC', 4308, 20, 1, 3070, 2269, 4 FROM dual
UNION ALL SELECT 8137, 'SIC', 4313, 70, 1, 4194, 8322, 4 FROM dual
UNION ALL SELECT 8138, 'SEC', 2796, 70, 1, 2722, 7360, 3 FROM dual
UNION ALL SELECT 8139, 'NAICS', 1419, 512, 2, 1441, 51229, 4 FROM dual
UNION ALL SELECT 8140, 'NAICS', 68, 1123, 3, 74, 11233, 4 FROM dual
UNION ALL SELECT 8141, 'NAICS', 181, 221, 2, 197, 22121, 4 FROM dual
UNION ALL SELECT 8142, 'SEC', 2791, 20, 1, 2489, 3841, 4 FROM dual
UNION ALL SELECT 8143, 'SIC', 4314, 90, 1, 4299, 9711, 4 FROM dual
UNION ALL SELECT 8144, 'SIC', 4306, 10, 1, 2926, 1442, 4 FROM dual
UNION ALL SELECT 8145, 'NAICS', 2037, 81, 1, 2116, 813319, 5 FROM dual
UNION ALL SELECT 8146, 'SEC', 2796, 70, 1, 2770, 8300, 2 FROM dual
UNION ALL SELECT 8147, 'SIC', 3552, 3800, 2, 3563, 3829, 4 FROM dual
UNION ALL SELECT 8148, 'SIC', 4308, 20, 1, 3553, 3810, 3 FROM dual
UNION ALL SELECT 8149, 'NAICS', 132, 21, 1, 169, 212392, 5 FROM dual
UNION ALL SELECT 8150, 'NAICS', 1466, 518, 2, 1469, 51821, 4 FROM dual
UNION ALL SELECT 8151, 'NAICS', 1705, 54189, 4, 1704, 541890, 5 FROM dual
UNION ALL SELECT 8152, 'SIC', 4146, 8000, 2, 4172, 8092, 4 FROM dual
UNION ALL SELECT 8153, 'SEC', 2796, 70, 1, 2751, 7990, 3 FROM dual
UNION ALL SELECT 8154, 'NAICS', 56, 112, 2, 58, 11211, 4 FROM dual
UNION ALL SELECT 8155, 'NAICS', 2135, 92, 1, 2147, 92115, 4 FROM dual
UNION ALL SELECT 8156, 'SIC', 3127, 2420, 3, 3128, 2421, 4 FROM dual
UNION ALL SELECT 8157, 'NAICS', 1812, 61, 1, 1844, 611691, 5 FROM dual
UNION ALL SELECT 8158, 'NAICS', 1813, 611, 2, 1835, 611519, 5 FROM dual
UNION ALL SELECT 8159, 'SIC', 4308, 20, 1, 3589, 3952, 4 FROM dual
UNION ALL SELECT 8160, 'SIC', 4308, 20, 1, 3463, 3572, 4 FROM dual
UNION ALL SELECT 8161, 'NAICS', 1673, 54161, 4, 1676, 541613, 5 FROM dual
UNION ALL SELECT 8162, 'SEC', 2791, 20, 1, 2509, 3950, 3 FROM dual
UNION ALL SELECT 8163, 'SIC', 4125, 7900, 2, 4132, 7933, 4 FROM dual
UNION ALL SELECT 8164, 'SIC', 4308, 20, 1, 3450, 3559, 4 FROM dual
UNION ALL SELECT 8165, 'NAICS', 930, 42, 1, 1052, 424490, 5 FROM dual
UNION ALL SELECT 8166, 'NAICS', 1719, 551, 2, 1724, 551114, 5 FROM dual
UNION ALL SELECT 8167, 'SEC', 2319, 2800, 2, 2335, 2890, 3 FROM dual
UNION ALL SELECT 8168, 'NAICS', 205, 23, 1, 226, 2372, 3 FROM dual
UNION ALL SELECT 8169, 'NAICS', 1908, 6232, 3, 1911, 623220, 5 FROM dual
UNION ALL SELECT 8170, 'NAICS', 2073, 81211, 4, 2075, 812112, 5 FROM dual
UNION ALL SELECT 8171, 'SIC', 3966, 6350, 3, 3967, 6351, 4 FROM dual
UNION ALL SELECT 8172, 'NAICS', 1071, 4248, 3, 1073, 42481, 4 FROM dual
UNION ALL SELECT 8173, 'SIC', 3576, 3900, 2, 3599, 3996, 4 FROM dual
UNION ALL SELECT 8174, 'SIC', 3368, 3390, 3, 3369, 3398, 4 FROM dual
UNION ALL SELECT 8175, 'NAICS', 1624, 54, 1, 1632, 541191, 5 FROM dual
UNION ALL SELECT 8176, 'NAICS', 1480, 52, 1, 1533, 52399, 4 FROM dual
UNION ALL SELECT 8177, 'NAICS', 1942, 71, 1, 1946, 71111, 4 FROM dual
UNION ALL SELECT 8178, 'SEC', 2795, 60, 1, 2664, 6150, 3 FROM dual
UNION ALL SELECT 8179, 'SIC', 3362, 3360, 3, 3366, 3366, 4 FROM dual
UNION ALL SELECT 8180, 'SIC', 3489, 3630, 3, 3490, 3631, 4 FROM dual
UNION ALL SELECT 8181, 'NAICS', 1403, 511, 2, 1418, 51121, 4 FROM dual
UNION ALL SELECT 8182, 'NAICS', 235, 238, 2, 261, 238310, 5 FROM dual
UNION ALL SELECT 8183, 'SIC', 4309, 40, 1, 3709, 4953, 4 FROM dual
UNION ALL SELECT 8184, 'SEC', 2384, 3400, 2, 2388, 3420, 3 FROM dual
UNION ALL SELECT 8185, 'NAICS', 1, 11, 1, 34, 111336, 5 FROM dual
UNION ALL SELECT 8186, 'SEC', 2450, 3670, 3, 2451, 3672, 4 FROM dual
UNION ALL SELECT 8187, 'NAICS', 1624, 54, 1, 1647, 541340, 5 FROM dual
UNION ALL SELECT 8188, 'NAICS', 1851, 621, 2, 1871, 6214, 3 FROM dual
UNION ALL SELECT 8189, 'SIC', 4313, 70, 1, 4009, 7011, 4 FROM dual
UNION ALL SELECT 8190, 'SIC', 2832, 240, 3, 2833, 241, 4 FROM dual
UNION ALL SELECT 8191, 'SEC', 2371, 3300, 2, 2376, 3330, 3 FROM dual
UNION ALL SELECT 8192, 'NAICS', 1625, 541, 2, 1677, 541614, 5 FROM dual
UNION ALL SELECT 8193, 'SIC', 4117, 7820, 3, 4119, 7829, 4 FROM dual
UNION ALL SELECT 8194, 'NAICS', 1831, 61151, 4, 1832, 611511, 5 FROM dual
UNION ALL SELECT 8195, 'NAICS', 2071, 812, 2, 2084, 81222, 4 FROM dual
UNION ALL SELECT 8196, 'NAICS', 1, 11, 1, 18, 111199, 5 FROM dual
UNION ALL SELECT 8197, 'NAICS', 260, 2383, 3, 262, 23831, 4 FROM dual
UNION ALL SELECT 8198, 'NAICS', 2, 111, 2, 25, 11131, 4 FROM dual
UNION ALL SELECT 8199, 'NAICS', 1092, 42511, 4, 1091, 425110, 5 FROM dual
UNION ALL SELECT 8200, 'SEC', 2349, 3080, 3, 2351, 3086, 4 FROM dual
UNION ALL SELECT 8201, 'SEC', 2795, 60, 1, 2651, 6021, 4 FROM dual
UNION ALL SELECT 8202, 'SIC', 2884, 1000, 2, 2899, 1094, 4 FROM dual
UNION ALL SELECT 8203, 'SIC', 3033, 2090, 3, 3040, 2099, 4 FROM dual
UNION ALL SELECT 8204, 'SIC', 2921, 1420, 3, 2922, 1422, 4 FROM dual
UNION ALL SELECT 8205, 'SIC', 4308, 20, 1, 3478, 3596, 4 FROM dual
UNION ALL SELECT 8206, 'SIC', 4162, 8060, 3, 4164, 8063, 4 FROM dual
UNION ALL SELECT 8207, 'SIC', 4297, 9700, 2, 4298, 9710, 3 FROM dual
UNION ALL SELECT 8208, 'NAICS', 2037, 81, 1, 2061, 8114, 3 FROM dual
UNION ALL SELECT 8209, 'NAICS', 2039, 8111, 3, 2051, 811198, 5 FROM dual
UNION ALL SELECT 8210, 'NAICS', 2135, 92, 1, 2187, 92512, 4 FROM dual
UNION ALL SELECT 8211, 'NAICS', 982, 4237, 3, 985, 423720, 5 FROM dual
UNION ALL SELECT 8212, 'NAICS', 1015, 424, 2, 1081, 424930, 5 FROM dual
UNION ALL SELECT 8213, 'SIC', 4308, 20, 1, 3402, 3471, 4 FROM dual
UNION ALL SELECT 8214, 'NAICS', 56, 112, 2, 80, 112410, 5 FROM dual
UNION ALL SELECT 8215, 'SIC', 4313, 70, 1, 4024, 7217, 4 FROM dual
UNION ALL SELECT 8216, 'SIC', 4309, 40, 1, 3692, 4899, 4 FROM dual
UNION ALL SELECT 8217, 'SEC', 2796, 70, 1, 2782, 8742, 4 FROM dual
UNION ALL SELECT 8218, 'SEC', 2323, 2830, 3, 2326, 2835, 4 FROM dual
UNION ALL SELECT 8219, 'NAICS', 1526, 5239, 3, 1530, 52392, 4 FROM dual
UNION ALL SELECT 8220, 'SIC', 4285, 9610, 3, 4286, 9611, 4 FROM dual
UNION ALL SELECT 8221, 'NAICS', 1451, 51521, 4, 1450, 515210, 5 FROM dual
UNION ALL SELECT 8222, 'NAICS', 2188, 926, 2, 2194, 926130, 5 FROM dual
UNION ALL SELECT 8223, 'SIC', 3261, 3000, 2, 3274, 3082, 4 FROM dual
UNION ALL SELECT 8224, 'SEC', 2435, 3610, 3, 2436, 3612, 4 FROM dual
UNION ALL SELECT 8225, 'SIC', 4308, 20, 1, 3326, 3274, 4 FROM dual
UNION ALL SELECT 8226, 'NAICS', 2071, 812, 2, 2092, 812332, 5 FROM dual
UNION ALL SELECT 8227, 'NAICS', 132, 21, 1, 156, 2123, 3 FROM dual
UNION ALL SELECT 8228, 'SIC', 3167, 2600, 2, 3178, 2656, 4 FROM dual
UNION ALL SELECT 8229, 'SIC', 4310, 50, 1, 3795, 5181, 4 FROM dual
UNION ALL SELECT 8230, 'NAICS', 1076, 4249, 3, 1079, 424920, 5 FROM dual
UNION ALL SELECT 8231, 'NAICS', 1851, 621, 2, 1874, 621420, 5 FROM dual
UNION ALL SELECT 8232, 'NAICS', 1813, 611, 2, 1848, 611710, 5 FROM dual
UNION ALL SELECT 8233, 'SIC', 3083, 2300, 2, 3091, 2326, 4 FROM dual
UNION ALL SELECT 8234, 'NAICS', 84, 1125, 3, 87, 112512, 5 FROM dual
UNION ALL SELECT 8235, 'NAICS', 1814, 6111, 3, 1815, 611110, 5 FROM dual
UNION ALL SELECT 8236, 'NAICS', 1500, 52229, 4, 1502, 522292, 5 FROM dual
UNION ALL SELECT 8237, 'SIC', 4311, 52, 1, 3822, 5400, 2 FROM dual
UNION ALL SELECT 8238, 'NAICS', 1625, 541, 2, 1713, 54193, 4 FROM dual
UNION ALL SELECT 8239, 'SEC', 2792, 40, 1, 2535, 4610, 3 FROM dual
UNION ALL SELECT 8240, 'SEC', 2630, 5700, 2, 2631, 5710, 3 FROM dual
UNION ALL SELECT 8241, 'NAICS', 138, 212, 2, 150, 21223, 4 FROM dual
UNION ALL SELECT 8242, 'SIC', 4313, 70, 1, 4141, 7992, 4 FROM dual
UNION ALL SELECT 8243, 'NAICS', 1741, 5614, 3, 1744, 56142, 4 FROM dual
UNION ALL SELECT 8244, 'SIC', 4313, 70, 1, 4103, 7623, 4 FROM dual
UNION ALL SELECT 8245, 'NAICS', 36, 1114, 3, 39, 111419, 5 FROM dual
UNION ALL SELECT 8246, 'NAICS', 930, 42, 1, 983, 423710, 5 FROM dual
UNION ALL SELECT 8247, 'SIC', 3337, 3300, 2, 3353, 3340, 3 FROM dual
UNION ALL SELECT 8248, 'SIC', 4223, 8700, 2, 4234, 8734, 4 FROM dual
UNION ALL SELECT 8249, 'SIC', 4313, 70, 1, 4225, 8711, 4 FROM dual
UNION ALL SELECT 8250, 'NAICS', 2073, 81211, 4, 2074, 812111, 5 FROM dual
UNION ALL SELECT 8251, 'SIC', 4308, 20, 1, 3262, 3010, 3 FROM dual
UNION ALL SELECT 8252, 'NAICS', 2093, 8129, 3, 2094, 812910, 5 FROM dual
UNION ALL SELECT 8253, 'SIC', 4308, 20, 1, 3533, 3720, 3 FROM dual
UNION ALL SELECT 8254, 'SIC', 4313, 70, 1, 4026, 7219, 4 FROM dual
UNION ALL SELECT 8255, 'SIC', 3338, 3310, 3, 3342, 3316, 4 FROM dual
UNION ALL SELECT 8256, 'SEC', 2791, 20, 1, 2244, 2015, 4 FROM dual
UNION ALL SELECT 8257, 'SEC', 2791, 20, 1, 2418, 3560, 3 FROM dual
UNION ALL SELECT 8258, 'SIC', 4312, 60, 1, 3931, 6080, 3 FROM dual
UNION ALL SELECT 8259, 'SEC', 2646, 5960, 3, 2647, 5961, 4 FROM dual
UNION ALL SELECT 8260, 'SIC', 2798, 100, 2, 2813, 170, 3 FROM dual
UNION ALL SELECT 8261, 'NAICS', 206, 236, 2, 207, 2361, 3 FROM dual
UNION ALL SELECT 8262, 'NAICS', 235, 238, 2, 253, 2382, 3 FROM dual
UNION ALL SELECT 8263, 'SIC', 4146, 8000, 2, 4154, 8041, 4 FROM dual
UNION ALL SELECT 8264, 'NAICS', 1, 11, 1, 66, 112210, 5 FROM dual
UNION ALL SELECT 8265, 'SIC', 3917, 6000, 2, 3922, 6021, 4 FROM dual
UNION ALL SELECT 8266, 'SIC', 3115, 2390, 3, 3121, 2396, 4 FROM dual
UNION ALL SELECT 8267, 'SIC', 4313, 70, 1, 4199, 8360, 3 FROM dual
UNION ALL SELECT 8268, 'NAICS', 1026, 4243, 3, 1027, 424310, 5 FROM dual
UNION ALL SELECT 8269, 'SIC', 3822, 5400, 2, 3826, 5421, 4 FROM dual
UNION ALL SELECT 8270, 'SEC', 2384, 3400, 2, 2398, 3452, 4 FROM dual
UNION ALL SELECT 8271, 'SEC', 2623, 5600, 2, 2626, 5650, 3 FROM dual
UNION ALL SELECT 8272, 'SEC', 2695, 6510, 3, 2696, 6512, 4 FROM dual
UNION ALL SELECT 8273, 'NAICS', 28, 11133, 4, 32, 111334, 5 FROM dual
UNION ALL SELECT 8274, 'NAICS', 205, 23, 1, 230, 237310, 5 FROM dual
UNION ALL SELECT 8275, 'NAICS', 2037, 81, 1, 2121, 813910, 5 FROM dual
UNION ALL SELECT 8276, 'NAICS', 1, 11, 1, 52, 11199, 4 FROM dual
UNION ALL SELECT 8277, 'NAICS', 930, 42, 1, 1047, 42446, 4 FROM dual
UNION ALL SELECT 8278, 'SIC', 4313, 70, 1, 4069, 7374, 4 FROM dual
UNION ALL SELECT 8279, 'SIC', 4307, 15, 1, 2947, 1542, 4 FROM dual
UNION ALL SELECT 8280, 'SIC', 3564, 3840, 3, 3567, 3843, 4 FROM dual
UNION ALL SELECT 8281, 'NAICS', 1624, 54, 1, 1698, 541850, 5 FROM dual
UNION ALL SELECT 8282, 'SIC', 4313, 70, 1, 4116, 7819, 4 FROM dual
UNION ALL SELECT 8283, 'SEC', 2754, 8010, 3, 2755, 8011, 4 FROM dual
UNION ALL SELECT 8284, 'SIC', 4308, 20, 1, 3452, 3561, 4 FROM dual
UNION ALL SELECT 8285, 'SIC', 4308, 20, 1, 3254, 2911, 4 FROM dual
UNION ALL SELECT 8286, 'SEC', 2307, 2730, 3, 2308, 2731, 4 FROM dual
UNION ALL SELECT 8287, 'SIC', 4206, 8420, 3, 4207, 8422, 4 FROM dual
UNION ALL SELECT 8288, 'SIC', 4219, 8660, 3, 4220, 8661, 4 FROM dual
UNION ALL SELECT 8289, 'NAICS', 2037, 81, 1, 2112, 8133, 3 FROM dual
UNION ALL SELECT 8290, 'SIC', 4305, 1, 1, 2820, 180, 3 FROM dual
UNION ALL SELECT 8291, 'NAICS', 1591, 532, 2, 1593, 53211, 4 FROM dual
UNION ALL SELECT 8292, 'SEC', 2791, 20, 1, 2264, 2211, 4 FROM dual
UNION ALL SELECT 8293, 'SEC', 2637, 5800, 2, 2639, 5812, 4 FROM dual
UNION ALL SELECT 8294, 'NAICS', 2107, 8132, 3, 2108, 81321, 4 FROM dual
UNION ALL SELECT 8295, 'SEC', 2503, 3930, 3, 2504, 3931, 4 FROM dual
UNION ALL SELECT 8296, 'NAICS', 1402, 51, 1, 1460, 517410, 5 FROM dual
UNION ALL SELECT 8297, 'SIC', 4248, 9110, 3, 4249, 9111, 4 FROM dual
UNION ALL SELECT 8298, 'NAICS', 1890, 62191, 4, 1889, 621910, 5 FROM dual
UNION ALL SELECT 8299, 'NAICS', 2085, 8123, 3, 2086, 812310, 5 FROM dual
UNION ALL SELECT 8300, 'SEC', 2624, 5620, 3, 2625, 5621, 4 FROM dual
UNION ALL SELECT 8301, 'NAICS', 23, 1113, 3, 34, 111336, 5 FROM dual
UNION ALL SELECT 8302, 'SEC', 2321, 2820, 3, 2322, 2821, 4 FROM dual
UNION ALL SELECT 8303, 'NAICS', 1850, 62, 1, 1875, 62142, 4 FROM dual
UNION ALL SELECT 8304, 'NAICS', 56, 112, 2, 94, 112930, 5 FROM dual
UNION ALL SELECT 8305, 'NAICS', 1726, 561, 2, 1773, 561622, 5 FROM dual
UNION ALL SELECT 8306, 'NAICS', 1836, 6116, 3, 1841, 611630, 5 FROM dual
UNION ALL SELECT 8307, 'SIC', 3694, 4910, 3, 3695, 4911, 4 FROM dual
UNION ALL SELECT 8308, 'SIC', 3595, 3990, 3, 3599, 3996, 4 FROM dual
UNION ALL SELECT 8309, 'NAICS', 2150, 922, 2, 2161, 92215, 4 FROM dual
UNION ALL SELECT 8310, 'NAICS', 1625, 541, 2, 1670, 541513, 5 FROM dual
UNION ALL SELECT 8311, 'SIC', 4307, 15, 1, 2973, 1780, 3 FROM dual
UNION ALL SELECT 8312, 'SIC', 3605, 4100, 2, 3610, 4121, 4 FROM dual
UNION ALL SELECT 8313, 'SIC', 3715, 5000, 2, 3717, 5012, 4 FROM dual
UNION ALL SELECT 8314, 'NAICS', 932, 4231, 3, 934, 42311, 4 FROM dual
UNION ALL SELECT 8315, 'SIC', 2939, 1500, 2, 2941, 1521, 4 FROM dual
UNION ALL SELECT 8316, 'SIC', 2868, 800, 2, 2870, 811, 4 FROM dual
UNION ALL SELECT 8317, 'SIC', 4308, 20, 1, 3434, 3540, 3 FROM dual
UNION ALL SELECT 8318, 'SIC', 3337, 3300, 2, 3347, 3324, 4 FROM dual
UNION ALL SELECT 8319, 'SIC', 4311, 52, 1, 3897, 5944, 4 FROM dual
UNION ALL SELECT 8320, 'SIC', 3371, 3400, 2, 3405, 3482, 4 FROM dual
UNION ALL SELECT 8321, 'SIC', 3214, 2800, 2, 3248, 2892, 4 FROM dual
UNION ALL SELECT 8322, 'SIC', 3083, 2300, 2, 3089, 2323, 4 FROM dual
UNION ALL SELECT 8323, 'SIC', 3198, 2740, 3, 3199, 2741, 4 FROM dual
UNION ALL SELECT 8324, 'NAICS', 132, 21, 1, 138, 212, 2 FROM dual
UNION ALL SELECT 8325, 'SEC', 2764, 8090, 3, 2765, 8093, 4 FROM dual
UNION ALL SELECT 8326, 'SIC', 3605, 4100, 2, 3607, 4111, 4 FROM dual
UNION ALL SELECT 8327, 'SIC', 3822, 5400, 2, 3832, 5451, 4 FROM dual
UNION ALL SELECT 8328, 'SIC', 4312, 60, 1, 3956, 6282, 4 FROM dual
UNION ALL SELECT 8329, 'NAICS', 931, 423, 2, 979, 42362, 4 FROM dual
UNION ALL SELECT 8330, 'NAICS', 1480, 52, 1, 1489, 522120, 5 FROM dual
UNION ALL SELECT 8331, 'NAICS', 181, 221, 2, 204, 22133, 4 FROM dual
UNION ALL SELECT 8332, 'NAICS', 2136, 921, 2, 2144, 921140, 5 FROM dual
UNION ALL SELECT 8333, 'SEC', 2795, 60, 1, 2682, 6324, 4 FROM dual
UNION ALL SELECT 8334, 'SEC', 2384, 3400, 2, 2401, 3480, 3 FROM dual
UNION ALL SELECT 8335, 'NAICS', 1, 11, 1, 39, 111419, 5 FROM dual
UNION ALL SELECT 8336, 'SEC', 2793, 50, 1, 2568, 5000, 2 FROM dual
UNION ALL SELECT 8337, 'NAICS', 931, 423, 2, 958, 423420, 5 FROM dual
UNION ALL SELECT 8338, 'NAICS', 1931, 62422, 4, 1932, 624221, 5 FROM dual
UNION ALL SELECT 8339, 'SIC', 4309, 40, 1, 3602, 4010, 3 FROM dual
UNION ALL SELECT 8340, 'NAICS', 1015, 424, 2, 1078, 42491, 4 FROM dual
UNION ALL SELECT 8341, 'NAICS', 1536, 524, 2, 1543, 524127, 5 FROM dual
UNION ALL SELECT 8342, 'SIC', 3715, 5000, 2, 3742, 5064, 4 FROM dual
UNION ALL SELECT 8343, 'SIC', 3083, 2300, 2, 3109, 2381, 4 FROM dual
UNION ALL SELECT 8344, 'SEC', 2338, 2910, 3, 2339, 2911, 4 FROM dual
UNION ALL SELECT 8345, 'NAICS', 147, 21222, 4, 148, 212221, 5 FROM dual
UNION ALL SELECT 8346, 'NAICS', 1076, 4249, 3, 1082, 42493, 4 FROM dual
UNION ALL SELECT 8347, 'SEC', 2623, 5600, 2, 2624, 5620, 3 FROM dual
UNION ALL SELECT 8348, 'SEC', 2794, 52, 1, 2612, 5311, 4 FROM dual
UNION ALL SELECT 8349, 'SIC', 4308, 20, 1, 3509, 3663, 4 FROM dual
UNION ALL SELECT 8350, 'NAICS', 2080, 8122, 3, 2081, 812210, 5 FROM dual
UNION ALL SELECT 8351, 'SIC', 3977, 6500, 2, 3988, 6541, 4 FROM dual
UNION ALL SELECT 8352, 'NAICS', 131, 11531, 4, 130, 115310, 5 FROM dual
UNION ALL SELECT 8353, 'SIC', 3958, 6300, 2, 3960, 6311, 4 FROM dual
UNION ALL SELECT 8354, 'SIC', 3371, 3400, 2, 3374, 3412, 4 FROM dual
UNION ALL SELECT 8355, 'SIC', 4308, 20, 1, 3351, 3334, 4 FROM dual
UNION ALL SELECT 8356, 'NAICS', 13, 11115, 4, 12, 111150, 5 FROM dual
UNION ALL SELECT 8357, 'SEC', 2791, 20, 1, 2419, 3561, 4 FROM dual
UNION ALL SELECT 8358, 'SEC', 2766, 8100, 2, 2768, 8111, 4 FROM dual
UNION ALL SELECT 8359, 'SIC', 4314, 90, 1, 4247, 9100, 2 FROM dual
UNION ALL SELECT 8360, 'SIC', 3050, 2200, 2, 3053, 2220, 3 FROM dual
UNION ALL SELECT 8361, 'SIC', 3867, 5690, 3, 3868, 5699, 4 FROM dual
UNION ALL SELECT 8362, 'NAICS', 2037, 81, 1, 2122, 81391, 4 FROM dual
UNION ALL SELECT 8363, 'SIC', 3484, 3620, 3, 3485, 3621, 4 FROM dual
UNION ALL SELECT 8364, 'SIC', 3804, 5200, 2, 3811, 5260, 3 FROM dual
UNION ALL SELECT 8365, 'SIC', 3961, 6320, 3, 3962, 6321, 4 FROM dual
UNION ALL SELECT 8366, 'SIC', 4313, 70, 1, 4076, 7381, 4 FROM dual
UNION ALL SELECT 8367, 'NAICS', 84, 1125, 3, 86, 112511, 5 FROM dual
UNION ALL SELECT 8368, 'NAICS', 1569, 53, 1, 1595, 532112, 5 FROM dual
UNION ALL SELECT 8369, 'NAICS', 1449, 5152, 3, 1450, 515210, 5 FROM dual
UNION ALL SELECT 8370, 'NAICS', 1541, 52412, 4, 1542, 524126, 5 FROM dual
UNION ALL SELECT 8371, 'NAICS', 1719, 551, 2, 1722, 551111, 5 FROM dual
UNION ALL SELECT 8372, 'SIC', 4146, 8000, 2, 4149, 8020, 3 FROM dual
UNION ALL SELECT 8373, 'NAICS', 2071, 812, 2, 2091, 812331, 5 FROM dual
UNION ALL SELECT 8374, 'SIC', 3762, 5100, 2, 3803, 5199, 4 FROM dual
UNION ALL SELECT 8375, 'NAICS', 2037, 81, 1, 2133, 814110, 5 FROM dual
UNION ALL SELECT 8376, 'SIC', 3992, 6700, 2, 4002, 6790, 3 FROM dual
UNION ALL SELECT 8377, 'SIC', 4139, 7990, 3, 4141, 7992, 4 FROM dual
UNION ALL SELECT 8378, 'SEC', 2791, 20, 1, 2405, 3520, 3 FROM dual
UNION ALL SELECT 8379, 'SEC', 2302, 2700, 2, 2310, 2740, 3 FROM dual
UNION ALL SELECT 8380, 'SIC', 2982, 2000, 2, 2998, 2037, 4 FROM dual
UNION ALL SELECT 8381, 'SEC', 2791, 20, 1, 2422, 3567, 4 FROM dual
UNION ALL SELECT 8382, 'SIC', 4208, 8600, 2, 4219, 8660, 3 FROM dual
UNION ALL SELECT 8383, 'NAICS', 2037, 81, 1, 2118, 813410, 5 FROM dual
UNION ALL SELECT 8384, 'SEC', 2552, 4900, 2, 2560, 4931, 4 FROM dual
UNION ALL SELECT 8385, 'NAICS', 2019, 72131, 4, 2018, 721310, 5 FROM dual
UNION ALL SELECT 8386, 'SIC', 3480, 3600, 2, 3484, 3620, 3 FROM dual
UNION ALL SELECT 8387, 'SEC', 2521, 4230, 3, 2522, 4231, 4 FROM dual
UNION ALL SELECT 8388, 'SIC', 3823, 5410, 3, 3824, 5411, 4 FROM dual
UNION ALL SELECT 8389, 'SIC', 2951, 1620, 3, 2953, 1623, 4 FROM dual
UNION ALL SELECT 8390, 'SEC', 2795, 60, 1, 4322, 6180, 3 FROM dual
UNION ALL SELECT 8391, 'NAICS', 1480, 52, 1, 1522, 52314, 4 FROM dual
UNION ALL SELECT 8392, 'SIC', 3349, 3330, 3, 3350, 3331, 4 FROM dual
UNION ALL SELECT 8393, 'SIC', 4308, 20, 1, 3195, 2730, 3 FROM dual
UNION ALL SELECT 8394, 'SEC', 2792, 40, 1, 2531, 4522, 4 FROM dual
UNION ALL SELECT 8395, 'SEC', 2793, 50, 1, 2591, 5100, 2 FROM dual
UNION ALL SELECT 8396, 'SIC', 3149, 2510, 3, 3152, 2514, 4 FROM dual
UNION ALL SELECT 8397, 'NAICS', 1943, 711, 2, 1953, 7112, 3 FROM dual
UNION ALL SELECT 8398, 'NAICS', 1894, 622, 2, 1898, 6222, 3 FROM dual
UNION ALL SELECT 8399, 'SIC', 4308, 20, 1, 2987, 2020, 3 FROM dual
UNION ALL SELECT 8400, 'SIC', 4040, 7300, 2, 4061, 7359, 4 FROM dual
UNION ALL SELECT 8401, 'NAICS', 2135, 92, 1, 2145, 92114, 4 FROM dual
UNION ALL SELECT 8402, 'SIC', 3269, 3060, 3, 3270, 3061, 4 FROM dual
UNION ALL SELECT 8403, 'NAICS', 2151, 9221, 3, 2156, 922130, 5 FROM dual
UNION ALL SELECT 8404, 'NAICS', 1513, 523, 2, 1529, 523920, 5 FROM dual
UNION ALL SELECT 8405, 'NAICS', 931, 423, 2, 1009, 423930, 5 FROM dual
UNION ALL SELECT 8406, 'SIC', 4311, 52, 1, 3888, 5912, 4 FROM dual
UNION ALL SELECT 8407, 'SIC', 4313, 70, 1, 4193, 8320, 3 FROM dual
UNION ALL SELECT 8408, 'SIC', 4305, 1, 1, 2817, 174, 4 FROM dual
UNION ALL SELECT 8409, 'NAICS', 1813, 611, 2, 1830, 6115, 3 FROM dual
UNION ALL SELECT 8410, 'SEC', 2360, 3230, 3, 2361, 3231, 4 FROM dual
UNION ALL SELECT 8411, 'SEC', 2791, 20, 1, 2294, 2611, 4 FROM dual
UNION ALL SELECT 8412, 'SIC', 4308, 20, 1, 3586, 3949, 4 FROM dual
UNION ALL SELECT 8413, 'SIC', 4308, 20, 1, 3310, 3241, 4 FROM dual
UNION ALL SELECT 8414, 'SIC', 3715, 5000, 2, 3757, 5091, 4 FROM dual
UNION ALL SELECT 8415, 'SIC', 4308, 20, 1, 3311, 3250, 3 FROM dual
UNION ALL SELECT 8416, 'SIC', 4308, 20, 1, 3189, 2679, 4 FROM dual
UNION ALL SELECT 8417, 'NAICS', 1958, 7113, 3, 1961, 711320, 5 FROM dual
UNION ALL SELECT 8418, 'SEC', 2791, 20, 1, 2453, 3677, 4 FROM dual
UNION ALL SELECT 8419, 'SIC', 4308, 20, 1, 3079, 2296, 4 FROM dual
UNION ALL SELECT 8420, 'SIC', 3419, 3500, 2, 3439, 3545, 4 FROM dual
UNION ALL SELECT 8421, 'NAICS', 2176, 924, 2, 2178, 924110, 5 FROM dual
UNION ALL SELECT 8422, 'SEC', 2488, 3840, 3, 2490, 3842, 4 FROM dual
UNION ALL SELECT 8423, 'NAICS', 1943, 711, 2, 1968, 71151, 4 FROM dual
UNION ALL SELECT 8424, 'SIC', 3548, 3790, 3, 3549, 3792, 4 FROM dual
UNION ALL SELECT 8425, 'NAICS', 2135, 92, 1, 2174, 923140, 5 FROM dual
UNION ALL SELECT 8426, 'NAICS', 1571, 5311, 3, 1575, 53112, 4 FROM dual
UNION ALL SELECT 8427, 'NAICS', 1452, 517, 2, 1462, 5179, 3 FROM dual
UNION ALL SELECT 8428, 'NAICS', 1569, 53, 1, 1580, 5312, 3 FROM dual
UNION ALL SELECT 8429, 'SEC', 2793, 50, 1, 2582, 5065, 4 FROM dual
UNION ALL SELECT 8430, 'SIC', 4308, 20, 1, 3270, 3061, 4 FROM dual
UNION ALL SELECT 8431, 'NAICS', 119, 11511, 4, 125, 115116, 5 FROM dual
UNION ALL SELECT 8432, 'NAICS', 144, 2122, 3, 148, 212221, 5 FROM dual
UNION ALL SELECT 8433, 'NAICS', 1812, 61, 1, 1827, 61142, 4 FROM dual
UNION ALL SELECT 8434, 'SIC', 3419, 3500, 2, 3446, 3553, 4 FROM dual
UNION ALL SELECT 8435, 'NAICS', 1480, 52, 1, 1500, 52229, 4 FROM dual
UNION ALL SELECT 8436, 'SIC', 4309, 40, 1, 3695, 4911, 4 FROM dual
UNION ALL SELECT 8437, 'NAICS', 20, 11121, 4, 21, 111211, 5 FROM dual
UNION ALL SELECT 8438, 'NAICS', 1942, 71, 1, 1954, 71121, 4 FROM dual
UNION ALL SELECT 8439, 'NAICS', 938, 42313, 4, 937, 423130, 5 FROM dual
UNION ALL SELECT 8440, 'SIC', 2982, 2000, 2, 3011, 2053, 4 FROM dual
UNION ALL SELECT 8441, 'NAICS', 1836, 6116, 3, 1845, 611692, 5 FROM dual
UNION ALL SELECT 8442, 'NAICS', 1015, 424, 2, 1074, 424820, 5 FROM dual
UNION ALL SELECT 8443, 'SEC', 2513, 4010, 3, 2515, 4013, 4 FROM dual
UNION ALL SELECT 8444, 'SIC', 3077, 2290, 3, 3082, 2299, 4 FROM dual
UNION ALL SELECT 8445, 'NAICS', 1850, 62, 1, 1892, 621991, 5 FROM dual
UNION ALL SELECT 8446, 'SEC', 2434, 3600, 2, 2456, 3690, 3 FROM dual
UNION ALL SELECT 8447, 'SIC', 4308, 20, 1, 3272, 3080, 3 FROM dual
UNION ALL SELECT 8448, 'SIC', 4305, 1, 1, 2829, 213, 4 FROM dual
UNION ALL SELECT 8449, 'NAICS', 2, 111, 2, 55, 111998, 5 FROM dual
UNION ALL SELECT 8450, 'NAICS', 1402, 51, 1, 1429, 512191, 5 FROM dual
UNION ALL SELECT 8451, 'SIC', 2982, 2000, 2, 3035, 2092, 4 FROM dual
UNION ALL SELECT 8452, 'SIC', 4312, 60, 1, 3959, 6310, 3 FROM dual
UNION ALL SELECT 8453, 'SIC', 3480, 3600, 2, 3500, 3645, 4 FROM dual
UNION ALL SELECT 8454, 'SIC', 4308, 20, 1, 3401, 3470, 3 FROM dual
UNION ALL SELECT 8455, 'NAICS', 1625, 541, 2, 1651, 541360, 5 FROM dual
UNION ALL SELECT 8456, 'NAICS', 1592, 5321, 3, 1595, 532112, 5 FROM dual
UNION ALL SELECT 8457, 'NAICS', 1635, 54121, 4, 1636, 541211, 5 FROM dual
UNION ALL SELECT 8458, 'SEC', 2731, 7380, 3, 4329, 7385, 4 FROM dual
UNION ALL SELECT 8459, 'SIC', 3190, 2700, 2, 3195, 2730, 3 FROM dual
UNION ALL SELECT 8460, 'NAICS', 235, 238, 2, 258, 238290, 5 FROM dual
UNION ALL SELECT 8461, 'SIC', 2875, 900, 2, 2881, 921, 4 FROM dual
UNION ALL SELECT 8462, 'SIC', 3886, 5900, 2, 3902, 5949, 4 FROM dual
UNION ALL SELECT 8463, 'SIC', 3480, 3600, 2, 3516, 3676, 4 FROM dual
UNION ALL SELECT 8464, 'NAICS', 2, 111, 2, 38, 111411, 5 FROM dual
UNION ALL SELECT 8465, 'SIC', 3328, 3280, 3, 3329, 3281, 4 FROM dual
UNION ALL SELECT 8466, 'NAICS', 1798, 5622, 3, 1801, 562212, 5 FROM dual
UNION ALL SELECT 8467, 'NAICS', 1813, 611, 2, 1826, 611420, 5 FROM dual
UNION ALL SELECT 8468, 'SEC', 2793, 50, 1, 2602, 5172, 4 FROM dual
UNION ALL SELECT 8469, 'SEC', 2796, 70, 1, 2743, 7822, 4 FROM dual
UNION ALL SELECT 8470, 'NAICS', 132, 21, 1, 171, 212399, 5 FROM dual
UNION ALL SELECT 8471, 'NAICS', 1804, 5629, 3, 1806, 56291, 4 FROM dual
UNION ALL SELECT 8472, 'SIC', 3729, 5040, 3, 3736, 5049, 4 FROM dual
UNION ALL SELECT 8473, 'SIC', 3937, 6100, 2, 3947, 6163, 4 FROM dual
UNION ALL SELECT 8474, 'NAICS', 1015, 424, 2, 1087, 424990, 5 FROM dual
UNION ALL SELECT 8475, 'SIC', 3451, 3560, 3, 3458, 3567, 4 FROM dual
UNION ALL SELECT 8476, 'SIC', 3774, 5140, 3, 3781, 5147, 4 FROM dual
UNION ALL SELECT 8477, 'SIC', 3893, 5940, 3, 3899, 5946, 4 FROM dual
UNION ALL SELECT 8478, 'NAICS', 930, 42, 1, 957, 42341, 4 FROM dual
UNION ALL SELECT 8479, 'SIC', 4312, 60, 1, 3955, 6280, 3 FROM dual
UNION ALL SELECT 8480, 'SEC', 2659, 6100, 2, 2667, 6160, 3 FROM dual
UNION ALL SELECT 8481, 'SIC', 3180, 2670, 3, 3181, 2671, 4 FROM dual
UNION ALL SELECT 8482, 'SEC', 2588, 5090, 3, 2590, 5099, 4 FROM dual
UNION ALL SELECT 8483, 'SIC', 4040, 7300, 2, 4062, 7360, 3 FROM dual
UNION ALL SELECT 8484, 'NAICS', 1776, 56171, 4, 1775, 561710, 5 FROM dual
UNION ALL SELECT 8485, 'NAICS', 2045, 81112, 4, 2046, 811121, 5 FROM dual
UNION ALL SELECT 8486, 'SIC', 4131, 7930, 3, 4132, 7933, 4 FROM dual
UNION ALL SELECT 8487, 'SEC', 2283, 2500, 2, 2287, 2522, 4 FROM dual
UNION ALL SELECT 8488, 'SIC', 4223, 8700, 2, 4236, 8741, 4 FROM dual
UNION ALL SELECT 8489, 'SEC', 2403, 3500, 2, 2419, 3561, 4 FROM dual
UNION ALL SELECT 8490, 'NAICS', 3, 1111, 3, 7, 11112, 4 FROM dual
UNION ALL SELECT 8491, 'NAICS', 1402, 51, 1, 1414, 511191, 5 FROM dual
UNION ALL SELECT 8492, 'SIC', 3395, 3460, 3, 3398, 3465, 4 FROM dual
UNION ALL SELECT 8493, 'NAICS', 89, 1129, 3, 94, 112930, 5 FROM dual
UNION ALL SELECT 8494, 'SEC', 2342, 3000, 2, 2344, 3011, 4 FROM dual
UNION ALL SELECT 8495, 'SEC', 2791, 20, 1, 2456, 3690, 3 FROM dual
UNION ALL SELECT 8496, 'SIC', 4309, 40, 1, 3649, 4490, 3 FROM dual
UNION ALL SELECT 8497, 'NAICS', 1822, 61131, 4, 1821, 611310, 5 FROM dual
UNION ALL SELECT 8498, 'SEC', 2512, 4000, 2, 2515, 4013, 4 FROM dual
UNION ALL SELECT 8499, 'SIC', 3869, 5700, 2, 3872, 5713, 4 FROM dual
UNION ALL SELECT 8500, 'NAICS', 1480, 52, 1, 1548, 524210, 5 FROM dual
UNION ALL SELECT 8501, 'NAICS', 1813, 611, 2, 1821, 611310, 5 FROM dual
UNION ALL SELECT 8502, 'NAICS', 138, 212, 2, 155, 212299, 5 FROM dual
UNION ALL SELECT 8503, 'SIC', 3992, 6700, 2, 3996, 6720, 3 FROM dual
UNION ALL SELECT 8504, 'NAICS', 1555, 5251, 3, 1557, 52511, 4 FROM dual
UNION ALL SELECT 8505, 'SEC', 2791, 20, 1, 2257, 2090, 3 FROM dual
UNION ALL SELECT 8506, 'SIC', 2955, 1700, 2, 2974, 1781, 4 FROM dual
UNION ALL SELECT 8507, 'SIC', 4310, 50, 1, 3720, 5015, 4 FROM dual
UNION ALL SELECT 8508, 'NAICS', 930, 42, 1, 1092, 42511, 4 FROM dual
UNION ALL SELECT 8509, 'SIC', 3526, 3700, 2, 3549, 3792, 4 FROM dual
UNION ALL SELECT 8510, 'SEC', 2791, 20, 1, 2364, 3250, 3 FROM dual
UNION ALL SELECT 8511, 'SEC', 2403, 3500, 2, 2422, 3567, 4 FROM dual
UNION ALL SELECT 8512, 'NAICS', 1015, 424, 2, 1023, 4242, 3 FROM dual
UNION ALL SELECT 8513, 'NAICS', 2166, 923, 2, 2170, 923120, 5 FROM dual
UNION ALL SELECT 8514, 'SIC', 3164, 2590, 3, 3165, 2591, 4 FROM dual
UNION ALL SELECT 8515, 'SEC', 2403, 3500, 2, 2405, 3520, 3 FROM dual
UNION ALL SELECT 8516, 'SEC', 2791, 20, 1, 2243, 2013, 4 FROM dual
UNION ALL SELECT 8517, 'SIC', 3337, 3300, 2, 3345, 3321, 4 FROM dual
UNION ALL SELECT 8518, 'SIC', 4310, 50, 1, 3801, 5194, 4 FROM dual
UNION ALL SELECT 8519, 'NAICS', 110, 11411, 4, 112, 114112, 5 FROM dual
UNION ALL SELECT 8520, 'SIC', 4308, 20, 1, 3495, 3639, 4 FROM dual
UNION ALL SELECT 8521, 'SIC', 3180, 2670, 3, 3184, 2674, 4 FROM dual
UNION ALL SELECT 8522, 'NAICS', 1859, 6213, 3, 1863, 62132, 4 FROM dual
UNION ALL SELECT 8523, 'SIC', 4308, 20, 1, 3205, 2761, 4 FROM dual
UNION ALL SELECT 8524, 'SIC', 3124, 2400, 2, 3132, 2431, 4 FROM dual
UNION ALL SELECT 8525, 'SIC', 4308, 20, 1, 3380, 3430, 3 FROM dual
UNION ALL SELECT 8526, 'NAICS', 1767, 56161, 4, 1768, 561611, 5 FROM dual
UNION ALL SELECT 8527, 'SIC', 3715, 5000, 2, 3734, 5047, 4 FROM dual
UNION ALL SELECT 8528, 'SIC', 3869, 5700, 2, 3870, 5710, 3 FROM dual
UNION ALL SELECT 8529, 'NAICS', 205, 23, 1, 244, 23814, 4 FROM dual
UNION ALL SELECT 8530, 'SIC', 4308, 20, 1, 300